% (C)1992 Institute for New Generation Computer Technology
% (Read COPYRIGHT for detailed information.)

%%  mgpl4.pl: Ground-MGTP/pl Translator

:- public mgtp_version/1.
mgtp_version(mg4).

%%  > prolog
%%  ?- compile(mgpl4).
%%  ?- save_mgpl(mgpl).
%%  ...
%%  > mgpl name
%%         name.mg4 --> name_mg4.pl
%%  > mgtp name

:- public save_mgpl/1.
save_mgpl(Save) :-
        save(Save),
        unix(argv(Argv)), 
        pls(Argv),
        halt.

pls([F|Fs]) :- pl(F),!,pls(Fs).
pls([]).

concat([X],X) :- !.
concat([H|L],X) :- append(H,Y,X),!, concat(L,Y).

name_concat(NameList,NewName) :-
	name_concat1(NameList,String),
	name(NewName,String).

name_concat1([H|L],String) :-
	name(H,StringH),
	append(StringH,StringL,String),
	name_concat1(L,StringL).
name_concat1([],[]).

append([H|X],Y,[H|Z]) :- !, append(X,Y,Z).
append([],Y,Y).

%%%%

%%  MGTP4 SYNTAX
%%  ============
%%  
%%  MGTP_Program ::= { MGTP_Clause }+ [ Auxiliary_Clauses ]
%%  MGTP_Clause ::= Negative_Clause | 
%%                  Positive_Horn_Clause | Positive_NonHorn_Clause |
%%                  Neutral_Horn_Clause | Neutral_NonHorn_Clause
%%
%%  Negative_Clause ::= Antecedent `-->' `false' `.' |
%%                      `false' `:-' Simple_Conjunction `.' |
%%  Positive_Horn_Clause ::= `true' `-->' Atom_Group `.' |
%%                           MGTP_Atom `:-' `true' `.'
%%  Positive_NonHorn_Clause ::= `true' `-->' Consequent `.' |
%%                              Simple_Disjunction `:-' `true' `.'
%%  Neutral_Horn_Clause ::= Antecedent `-->' Atom_Group `.'
%%                          MGTP_Atom `:-' Simple_Conjunction `.'
%%  Neutral_NonHorn_Clause ::= Antecedent `-->' Consequent `.' |
%%                             Simple_Disjunction `:-' Simple_Conjunction `.'
%%  
%%  Antecedent ::= Antecedent_Unit { `;' Antecedent_Unit }*
%%  Antecedent_Unit ::= Antecedent_Atom { `,' Antecedent_Atom }*
%%  Antecedent_Atom ::= MGTP_Atom | `{{' [ Aux_Predicates ] `}}'
%%
%%  Simple_Conjunction ::= MGTP_Atom { `,' MGTP_Atom }*
%%
%%  Consequent ::= Atom_Group { `;' Atom_Group }+
%%  Atom_Group ::= MGTP_Atom { `,' MGTP_Atom }*
%%
%%  Simple_Disjunction ::= MGTP_Atom { `;' MGTP_Atom }*
%%
%%  MGTP_Atom ::= /* represented by a Prolog term */
%%  
%%  Aux_Predicates ::= Prolog_Call { `,' Prolog_Call }* 
%%  Auxiliary_Clauses ::= { Prolog_Clause }+

%%%%

:- op( 800, xfx, (:)).
:- dynamic mgpl_option/1, problem_type/1.
:- dynamic t_call/2, h_call/2, nh_call/2,
	   t_call_number/1, h_call_number/1, nh_call_number/1.

:- public pl/1, pld/1.
pl(Name) :-
        assert(mgpl_option(language,pl)),
        assert(mgpl_option(delta_duplication,no)),
        do(Name).
pld(Name) :-
        assert(mgpl_option(language,pl)),
        assert(mgpl_option(delta_duplication,yes)),
        do(Name).

do(Name) :-
        output_file(Name,OutF), tell(OutF),
        output_banner(OutF),
        nofileerrors, input_file(Name,InF), see(InF), fileerrors,
        read_cls(InF,Pcls,Tcls,Gcls,Preds,Fncts,AuxCode),!,
        multi_entries(Tcls,Tcls1,[]),
        multi_entries(Gcls,Gcls1,[]),
        remove_taut(Gcls1,Gcls2),
        review(Pcls,Tcls1,Gcls2),
	newpage,
        output_hdr,
        output_tm(Preds,Fncts),
        newpage,
        firing_atoms(Pcls,Pcls1,Tcls1,Gcls2),
        firing_atoms(Gcls2,Gcls3,Tcls1,Gcls2),
        output_positive_cls(Pcls1),
        output_firing_cls,
        newpage,
        output_cls(Tcls1),
        output_cls(Gcls3),
	nl,
        transfer_verbatim(AuxCode),
        output_eof(OutF),
        told.

space :- write(' ').
comma :- write(',').
semicolon :- write(';').
period :- write('.').
kakko :- write('(').
kokka :- write(')').
if :- write(' :- ').

newpage :- nl,write('%%%%'),nl.

input_file(Name,InF) :-
        mgtp_version(Ver),
	name_concat([Name,'.',Ver],InF).
input_file(Name,InF) :-
	name_concat([Name,'.mgtp'],InF).

output_file(Name,OutF) :-
        mgpl_option(delta_duplication,Dup),!,
        mgpl_option(language,pl),
        mgtp_version(Ver),
        ( Dup=yes ->
            name_concat([Name,'_',Ver,'d.pl'],OutF) ;
          name_concat([Name,'_',Ver,'.pl'],OutF) ).

output_banner(OutF) :-
        write('
%%  '),write(OutF),write('  Ground-MGTP/pl User-clauses
').

output_hdr :-
        problem_type(Type),
        write('
:- op( 800, xfx, (:)).

:- public problem_type/1.
'),
        write(problem_type(Type)),period,nl.

output_eof(OutF) :-
        write('%%  '),write(OutF),write('  EOF'),nl.

%%%%

%%  read_cls  BEGIN

read_cls(InF,Pcls,Tcls,Gcls,Preds,Fncts,AuxCode) :-
        write('
%%  Given clauses:
%%
'),
        scan([InF],0,Pos_cls,Neg_cls,Gen_cls,Type,AuxCode),
	( var(Type) -> Type=horn ; true ),
        assert(problem_type(Type)),
        symbols(Pos_cls,[],P2,[],F2),
        symbols(Neg_cls,P2,P3,F2,F3),
        symbols(Gen_cls,P3,Preds,F3,Fncts),
        numbering(Pos_cls,Pcls),
        numbering(Neg_cls,Tcls),
        numbering(Gen_cls,Gcls).

numbering([mg_cl(CN,(A->G --> C))|Rest],
          [mg_cl(CN,(NewA->G --> NewC))|Nrest]) :-
        numbering1(A,ante,NewA,0,N),
        numbering1(C,cnsq,NewC,N,_),
        numbering(Rest,Nrest).
numbering([],[]).

numbering1((A,B),AC,(NewA,NewB),Ni,No) :- !,
        numbering1(A,AC,NewA,Ni,N1),
        numbering1(B,AC,NewB,N1,No).
numbering1((A;B),AC,(NewA;NewB),Ni,No) :- !,
        numbering1(A,AC,NewA,Ni,N1),
        numbering1(B,AC,NewB,N1,No).
numbering1(false,cnsq,No:false,Ni,No) :- !, No is Ni+1.
numbering1(Lit,cnsq,No:Lit,Ni,No) :- !, No is Ni+1.
numbering1(Lit,ante,No:Lit,Ni,No) :- !, No is Ni+1.

%%%%

scan(InF,CN,Pos_cls,Neg_cls,Gen_cls,Type,AuxCode) :-
        read(Cls),!,
        scan1(InF,CN,Cls,Pos_cls,Neg_cls,Gen_cls,Type,AuxCode).
scan(_,_,_,_,_,_,_) :-
        display('!! Illeagal input !!'),nl, told,!, fail.

scan1(InF,CN,(true --> CNS),
        [mg_cl(Cid,(true->{{}}-->SCNS))|Pcls],
        Ncls,Gcls,Type,AuxCode) :- !,
        CN1 is CN+1, write('%%   c'), write(CN1), write(')	'),
        \+(\+(( numbervars(CNS,0,_),
                write((true --> CNS)),period,nl ))),
        standard_cnsq(CNS,SCNS,Type),
	name_concat(['c',CN1],Cid),
        scan(InF,CN1,Pcls,Ncls,Gcls,Type,AuxCode).

scan1(InF,CN,(CNS :- true),
        [mg_cl(Cid,(true->{{}}-->SCNS))|Pcls],
        Ncls,Gcls,Type,AuxCode) :- !,
        CN1 is CN+1, write('%%   c'), write(CN1), write(')	'),
        \+(\+(( numbervars(CNS,0,_),
                write((true --> CNS)),period,nl ))),
        standard_cnsq(CNS,SCNS,Type),
	name_concat(['c',CN1],Cid),
        scan(InF,CN1,Pcls,Ncls,Gcls,Type,AuxCode).

scan1(InF,CN,(ANT --> false),Pcls,Ncls,Gcls,Type,AuxCode) :- !,
        CN1 is CN+1, write('%%   c'), write(CN1), write(')	'),
        \+(\+(( numbervars(ANT,0,_),
                write((ANT --> false)),period,nl ))),
        multiple_ante(CN1,0,_,ANT,false,Ncls,Ncls1),
        scan(InF,CN1,Pcls,Ncls1,Gcls,Type,AuxCode).

scan1(InF,CN,(false :- ANT),Pcls,Ncls,Gcls,Type,AuxCode) :- !,
        CN1 is CN+1, write('%%   c'), write(CN1), write(')	'),
        \+(\+(( numbervars(ANT,0,_),
                write((ANT --> false)),period,nl ))),
        multiple_ante(CN1,0,_,ANT,false,Ncls,Ncls1),
        scan(InF,CN1,Pcls,Ncls1,Gcls,Type,AuxCode).

scan1(InF,CN,(ANT --> CNS),Pcls,Ncls,Gcls,Type,AuxCode) :- !,
        CN1 is CN+1, write('%%   c'), write(CN1), write(')	'),
        \+(\+(( numbervars((ANT,CNS),0,_),
                write(ANT),
                write(' --> '),write(CNS),period,nl ))),
        standard_cnsq(CNS,SCNS,Type),
        multiple_ante(CN1,0,_,ANT,SCNS,Gcls,Gcls1),
        scan(InF,CN1,Pcls,Ncls,Gcls1,Type,AuxCode).

scan1(InF,CN,(CNS :- ANT),Pcls,Ncls,Gcls,Type,AuxCode) :- !,
        CN1 is CN+1, write('%%   c'), write(CN1), write(')	'),
        \+(\+(( numbervars((ANT,CNS),0,_),
                write(ANT),
                write(' --> '),write(CNS),period,nl ))),
        standard_cnsq(CNS,SCNS,Type),
        multiple_ante(CN1,0,_,ANT,SCNS,Gcls,Gcls1),
        scan(InF,CN1,Pcls,Ncls,Gcls1,Type,AuxCode).

scan1(InF,CN,(:- see(File)),Pcls,Ncls,Gcls,Type,AuxCode) :- !,
        see(File),
        scan([File|InF],CN,Pcls,Ncls,Gcls,Type,AuxCode).
scan1([_,File|InF],CN,end_of_file,Pcls,Ncls,Gcls,Type,AuxCode) :- !,
        seen, see(File),
        scan([File|InF],CN,Pcls,Ncls,Gcls,Type,AuxCode).

scan1([_],_,end_of_file,[],[],[],_,no) :- !, seen.
scan1([_],_,(:- prolog),[],[],[],_,pl) :- !.

scan1(InF,CN,(:-X),Pcls,Ncls,Gcls,Type) :- !,
        \+(\+(( numbervars(X,0,_),
                write('%%  '),write((:-X)),nl ))),
        scan(InF,CN,Pcls,Ncls,Gcls,Type).

scan1(InF,CN,(H:-G;B),Pcls,Ncls,Gcls,Type) :- !,
        \+(\+(( numbervars((H:-G;B),0,_),
                write(H),if,
                output_guard({{G}}),
                output_body(B),period,nl ))),
        scan(InF,CN,Pcls,Ncls,Gcls,Type).

scan1(InF,CN,UserProg,Pcls,Ncls,Gcls,Type) :- !,
        \+(\+(( numbervars(UserProg,0,_),
                write(UserProg),period,nl ))),
        scan(InF,CN,Pcls,Ncls,Gcls,Type).

%%%%

transfer_verbatim(no).
transfer_verbatim(pl) :-
        write('%%  Auxiliary Prolog clauses:'),nl,
	transfer_chars,nl.

transfer_chars :- get0(Char),!,
        ( Char= -1 -> true ; put(Char), transfer_chars ).

%%%%

symbols([mg_cl(_,(A --> C))|R],Pi,Po,Fi,Fo) :- !,
        symbols(A,Pi,P1,Fi,F1),
        symbols(C,P1,P2,F1,F2),
        symbols(R,P2,Po,F2,Fo).
symbols([],Pi,Pi,Fi,Fi) :- !.

symbols(true,Pi,Pi,Fi,Fi) :- !.
symbols(false,Pi,Pi,Fi,Fi) :- !.
symbols({{}},Pi,Pi,Fi,Fi) :- !.
symbols({{_}},Pi,Pi,Fi,Fi) :- !.

symbols((A->B),Pi,Po,Fi,Fo) :- !,
        symbols(A,Pi,P1,Fi,F1),
        symbols(B,P1,Po,F1,Fo).
symbols((A,B),Pi,Po,Fi,Fo) :- !,
        symbols(A,Pi,P1,Fi,F1),
        symbols(B,P1,Po,F1,Fo).
symbols((A;B),Pi,Po,Fi,Fo) :- !,
        symbols(A,Pi,P1,Fi,F1),
        symbols(B,P1,Po,F1,Fo).
symbols(Atom,Pi,Po,Fi,Fo) :-
        functor(Atom,Pred,Arity),
        functor(Atom1,Pred,Arity),
        enter(Atom1,Pi,Po),
        functors(1,Arity,Atom,Fi,Fo).

functors(Jth,Last,Term,Fi,Fo) :-
        Jth=<Last, arg(Jth,Term,Arg), var(Arg),!,
        J1th is Jth+1,
        functors(J1th,Last,Term,Fi,Fo).
functors(Jth,Last,Term,Fi,Fo) :-
        Jth=<Last, arg(Jth,Term,Arg), !,
        functor(Arg,Fnct,Arity),
        functor(Arg1,Fnct,Arity),
        enter(Arg1,Fi,F1),
        functors(1,Arity,Arg,F1,F2),
        J1th is Jth+1,
        functors(J1th,Last,Term,F2,Fo).
functors(Jth,Last,_,Fi,Fi) :- Jth>Last.

enter(Term,[Term|Rest],NewRest) :- !, NewRest=[Term|Rest].
enter(Term,[Term1|Rest],[Term1|NewRest]) :- enter(Term,Rest,NewRest).
enter(Term,[],[Term]).

anonymous_vars(P) :- P=..[_|Vars], anonymous_vars1(Vars).

anonymous_vars1(['_'|V]) :- anonymous_vars1(V).
anonymous_vars1([]).

%%%%

vars_in(V,Vi,Vo) :- var(V),!, add_v(V,Vi,Vo).
vars_in({{P}},Vi,Vo) :- !, vars_in(P,Vi,Vo).
vars_in({{}},Vi,Vi) :- !.
vars_in([A|B],Vi,Vo) :- vars_in(A,Vi,Vm), vars_in(B,Vm,Vo).
vars_in([],Vi,Vi).
vars_in(Atom,Vi,Vi) :- atom(Atom),!.
vars_in(Int,Vi,Vi) :- integer(Int),!.
vars_in(Term,Vi,Vo) :- functor(Term,_,Arity),
        vars_in_args(1,Arity,Term,Vi,Vo).

vars_in_args(Jth,Last,Term,Vi,Vo) :-
        Jth<Last,
        arg(Jth,Term,Arg),
        vars_in(Arg,Vi,Vm),
        J1th is Jth+1,
        vars_in_args(J1th,Last,Term,Vm,Vo).
vars_in_args(Last,Last,Term,Vi,Vo) :-
        arg(Last,Term,Arg),
        vars_in(Arg,Vi,Vo).

add_v(V,[V1|Vi],[V1|Vi]) :- V==V1,!.
add_v(V,[V1|Vi],[V1|Vo]) :- add_v(V,Vi,Vo).
add_v(V,[],[V]).

%%%%

multiple_ante(CN,Ni,No,(A;B),C,Hd,Tl) :- !,
        multiple_ante1(CN,Ni,No,(A;B),C,Hd,Tl).
multiple_ante(CN,Ni,Ni,ANT,C,[mg_cl(Cid,(A->G --> C))|Tl],Tl) :-
        split_ante(ANT,A,G),
	name_concat(['c',CN],Cid).

multiple_ante1(CN,Ni,No,(A;B),C,Hd,Tl) :- !,
        multiple_ante1(CN,Ni,Nm,A,C,Hd,Tl1),
        multiple_ante1(CN,Nm,No,B,C,Tl1,Tl).
multiple_ante1(CN,Ni,No,ANT,C,[mg_cl(Cid,(A->G --> C))|Tl],Tl) :-
        split_ante(ANT,A,G),
        No is Ni+1,
	name_concat(['c',CN,'a',No],Cid).

split_ante((P,Q),A,G) :- !,
        split_ante(P,PA,PG),
        split_ante(Q,QA,QG),
        mk_conj(PA,QA,A),
        mk_conj(PG,QG,G).
split_ante({{}},true,{{}}) :- !.
split_ante({{G}},true,{{G}}) :- !.
split_ante(P,P,{{}}) :- !.

mk_conj({{}},{{}},{{}}) :- !.
mk_conj({{P}},{{}},{{P}}) :- !.
mk_conj({{}},{{Q}},{{Q}}) :- !.
mk_conj({{P}},{{Q}},{{P,Q}}) :- !.
mk_conj(true,true,true) :- !.
mk_conj(P,true,P) :- !.
mk_conj(true,Q,Q) :- !.
mk_conj(P,Q,(P,Q)) :- !.

standard_cnsq(C,NewC,Type) :- check_cnsq(C,NewC,Type).

check_cnsq((A;B),(A;B),nonHorn) :- !.
check_cnsq(A,A,_).

%%%%

review(Pcls,Tcls,Gcls) :-
        write('
%%  After conversion...

%%  Positive clauses:
%%
'),
        review_cls(Pcls),
        write('
%%  Negative clauses:
%%
'),
        review_cls(Tcls),
        write('
%%  Neutral clauses:
%%
'),
        review_cls(Gcls).

review_cls([G1|Gs]) :- review_cls_1([G1|Gs]).
review_cls([]).

write_ante(fire(Entry,OL,_)) :- write(Entry), write_OL(OL).

write_OL([{{}}]) :- !.
write_OL([{{}}|OL]) :- !, write_OL(OL).
write_OL([{{G}}]) :- !, comma, write({{G}}).
write_OL([{{G}}|OL]) :- !, comma, write({{G}}), write_OL(OL).
write_OL([[ID:A+_+_+_|B]]) :- !, write_OL([ID:A|B]).
write_OL([A]) :- !, comma, write(A).
write_OL([A1,A2|B]) :- !, comma, write(A1), write_OL([A2|B]).
write_OL([]).

%%%%

review_cls_1([mg_cl(N,(1:true->{{}} --> C))|Gs]) :- !,
        \+(\+(( numbervars(C,0,_),
        write('%%   '),write(N),
        write(')	'), write('1:true -->'),
        write(' '),write(C),period,nl ))),
        review_cls_1(Gs).
review_cls_1([mg_cl(N,(A->{{}} --> C))|Gs]) :- !, A=fire((Ent:_),_,A0),
        \+(\+(( numbervars((A0,C),0,_),
        write('%%   '),write(N),write('_'),write(Ent),
        write(')	'),write_ante(A), write(' -->'),
        write(' '),write(C),period,nl ))),
        review_cls_1(Gs).
review_cls_1([mg_cl(N,(A->G --> C))|Gs]) :- !, A=fire((Ent:_),_,A0),
        \+(\+(( numbervars((A0,G,C),0,_),
        write('%%   '),write(N),write('_'),write(Ent),
        write(')	'),write_ante(A), write(' -->'),nl,
        write('%%      '),write(C),period,nl ))),
        review_cls_1(Gs).
review_cls_1([]).

%%%%

firing_atoms([mg_cl(N,(A->G-->C))|Gs],
             [mg_cl(N,(A->G-->NewC))|NewGs],Tcls,Gcls) :-
        firing_atoms1(C,NewC,Tcls,Gcls),
        firing_atoms(Gs,NewGs,Tcls,Gcls).
firing_atoms([],[],_,_).

firing_atoms1((A,B),(NewA,NewB),Tcls,Gcls) :- !,
        firing_atoms1(A,NewA,Tcls,Gcls),
        firing_atoms1(B,NewB,Tcls,Gcls).
firing_atoms1((A;B),(NewA;NewB),Tcls,Gcls) :- !,
        firing_atoms1(A,NewA,Tcls,Gcls),
        firing_atoms1(B,NewB,Tcls,Gcls).
firing_atoms1(AtomNbr:Atom,'ATOM'(Atom,AtomNbr,Tcall,Hcall,NHcall),
	      Tcls,Gcls) :-
        copy_term(Atom,TA),
	entry_atoms(Tcls,false,TA,Tentries,[]),
	( t_call(Tcall,Tentries) ;
          new_t_call_number(TN),
	  name_concat(['t_',TN],Tcall),
          assert(t_call(Tcall,Tentries)) ),
        copy_term(Atom,HA),
	entry_atoms(Gcls,horn,HA,Hentries,[]),
	( h_call(Hcall,Hentries) ;
          new_h_call_number(HN),
	  name_concat(['h_',HN],Hcall),
          assert(h_call(Hcall,Hentries)) ),
        copy_term(Atom,NHA),
	entry_atoms(Gcls,nonhorn,NHA,NHentries,[]),
	( nh_call(NHcall,NHentries) ;
          new_nh_call_number(NHN),
	  name_concat(['nh_',NHN],NHcall),
          assert(nh_call(NHcall,NHentries)) ).

entry_atoms([mg_cl(CN,(A->_ --> _))|Rest],false,Atom,Hd,Tl) :- !,
        entry_atoms1(CN,A,Atom,Hd,Tl1),
        entry_atoms(Rest,false,Atom,Tl1,Tl).
entry_atoms([mg_cl(CN,(A->_ --> C))|Rest],horn,Atom,Hd,Tl) :- 
	\+nonhorn(C),!,
        entry_atoms1(CN,A,Atom,Hd,Tl1),
        entry_atoms(Rest,horn,Atom,Tl1,Tl).
entry_atoms([mg_cl(CN,(A->_ --> C))|Rest],nonhorn,Atom,Hd,Tl) :-
	nonhorn(C),!,
        entry_atoms1(CN,A,Atom,Hd,Tl1),
        entry_atoms(Rest,nonhorn,Atom,Tl1,Tl).
entry_atoms([_|Rest],HNH,Atom,Hd,Tl) :- 
        entry_atoms(Rest,HNH,Atom,Hd,Tl).
entry_atoms([],_,_,Tl,Tl).

entry_atoms1(CN,fire(LN:Lit,_,_),Atom,Hd,Tl) :-
        \+(\+(( unify(Lit,Atom) ))),!,
        Hd=[(CN-LN)|Tl].
entry_atoms1(CN,LN:Lit,Atom,Hd,Tl) :-
        \+(\+(( unify(Lit,Atom) ))),!,
        Hd=[(CN-LN)|Tl].
entry_atoms1(_,_,_,Tl,Tl).

new_t_call_number(N1) :-
        ( retract(t_call_number(N)) -> N1 is N+1 ; N1=1 ),
        assert(t_call_number(N1)).

new_h_call_number(N1) :-
        ( retract(h_call_number(N)) -> N1 is N+1 ; N1=1 ),
        assert(h_call_number(N1)).

new_nh_call_number(N1) :-
        ( retract(nh_call_number(N)) -> N1 is N+1 ; N1=1 ),
        assert(nh_call_number(N1)).

%nonhorn((_;_)).
%nonhorn(((_;_),_)).

nonhorn((_;_)).
nonhorn((A,B)) :- nonhorn(A); nonhorn(B).

%%%%

output_firing_cls :-
        write('
:- public fire_testers/6.
fire_testers(Tcall,Atom,Nbr,TM,FatomInfo) :-
    Call=..[Tcall,(Atom,Nbr,TM),FatomInfo],!, call(Call).
'),write('
:- public fire_horn_cls/6.
fire_horn_cls(Hcall,Atom,Nbr,TM,NewCls) :-
    findall(New,(Call=..[Hcall,(Atom,Nbr,TM),New], call(Call)),
            NewCls),!.
'),write('
:- public fire_nonhorn_cls/6.
fire_nonhorn_cls(NHcall,Atom,Nbr,TM,NewCls) :-
    findall(New,(Call=..[NHcall,(Atom,Nbr,TM),New], call(Call)),
            NewCls),!.
'),nl,
	output_t_calls,nl,
	output_h_calls,nl,
	output_nh_calls.

output_t_calls :-
	t_call(Tcall,Tentries),
        ( Tentries = [] ->
                Call=..[Tcall,'_','_'],
                write((Call:-!,fail)),period,nl ;
          Head=..[Tcall,'X','Y'],
          mk_fire1(Tentries,Head) ),fail.
output_t_calls.

output_h_calls :-
	h_call(Hcall,Hentries),
        ( Hentries = [] ->
                Call=..[Hcall,'_','_'],
                write((Call:-!,fail)),period,nl ;
          Head=..[Hcall,'X','Y'],
          mk_fire1(Hentries,Head) ),fail.
output_h_calls.

output_nh_calls :-
	nh_call(NHcall,NHentries),
        ( NHentries = [] ->
                Call=..[NHcall,'_','_'],
                write((Call:-!,fail)),period,nl ;
          Head=..[NHcall,'X','Y'],
          mk_fire1(NHentries,Head) ),fail.
output_nh_calls.

mk_fire1([CN-LN|Rest],Head) :-
	name_concat([CN,'_',LN],Pname),
	Body=..[Pname,'X','Y'],
        write((Head:-Body)),period,nl,
	mk_fire1(Rest,Head).
mk_fire1([],_).

%%%%

multi_entries(Cls,Hd,Tl) :-
        mgpl_option(delta_duplication,YN),
        (YN=yes -> Opt=dup ; YN=no -> Opt=nodup ),
        multi_entries(Cls,Hd,Tl,Opt).

multi_entries([mg_cl(CN,(A->G-->C))|Cls],Hd,Tl,Opt) :-
        conj2list(A,AL,[]),
        setof(Entry,entry(CN,AL,A,G,{{}},C,Entry,Opt),Entries),
        merge_variants(Entries,Entries1),
        multi_entries1(Entries1,Hd,Tl1),
        multi_entries(Cls,Tl1,Tl,Opt).
multi_entries([],Tl,Tl,_).

merge_variants([E1|Entries1],NewEntries) :-
        merge_variants1(Entries1,E1,E2,Entries2),
        NewEntries=[E2|NewEntries1],
        merge_variants(Entries2,NewEntries1).
merge_variants([],[]).

merge_variants1([E|Entries1],E1,E2,Entries2) :-
        ( variant_entry(E,E1,NewE1) ->
                merge_variants1(Entries1,NewE1,E2,Entries2) ;
          Entries2=[E|Entries3],
          merge_variants1(Entries1,E1,E2,Entries3) ).
merge_variants1([],E1,E1,[]).

variant_entry(  mg_cl(CN,(fire(LN1:Lit1,OL1,_)->{{}}-->CSQ1)),
                mg_cl(CN,(fire(LN2:Lit2,OL2,A2)->{{}}-->CSQ2)),
                NewE1) :-
        ordered_ante(OL1,Ante1),
        ordered_ante(OL2,Ante2),
        copy_term((Lit1,Ante1,OL1,CSQ1),(Lit3,Ante3,OL3,CSQ3)),
        variant([Lit3|Ante3],[Lit2|Ante2]),
        (Lit3,Ante3)=(Lit2,Ante2),
        name(LN1,LN1S), name(LN2,LN2S),
        concat(["e",LN2S,"e",LN1S],LN3S),
	name(LN3,LN3S),
        merge_OL(OL3,OL2,NewOL),
        merge_cnsq(CSQ2,CSQ3,NewCSQ),
        NewE1=  mg_cl(CN,(fire(LN3:Lit2,NewOL,A2)->{{}}-->NewCSQ)).

merge_cnsq(CSQ1,CSQ2,CSQ1) :-
        \+(\+((numbervars(CSQ1,0,_),CSQ1=CSQ2))),!.
merge_cnsq(CSQ1,CSQ2,(CSQ1,CSQ2)).

ordered_ante([{{}}],[]) :- !.
ordered_ante([{{}}|OL],A) :- !, ordered_ante(OL,A).
ordered_ante([{{_}}],[]) :- !.
ordered_ante([{{_}}|OL],A) :- !, ordered_ante(OL,A).
ordered_ante([[_:A+_+_+_|B]],[A|Ante]) :- !, ordered_ante(B,Ante).
ordered_ante([_:A],[A]) :- !.
ordered_ante([_:A1,A2|B],[A1|Ante]) :- !, ordered_ante([A2|B],Ante).

%%%%

merge_OL([{{}}],[{{}}],[{{}}]) :- !.
merge_OL([{{}}|OL1],[{{}}|OL2],[{{}}|OL3]) :- !, merge_OL(OL1,OL2,OL3).
merge_OL([[LN1:A1+A2+A3+A4|B1]],[[LN2:_+_+_+_|B2]],
         [[LN3:A1+A2+A3+A4|B3]]):- !,
        name(LN1,LN1S), name(LN2,LN2S),
        concat(["e",LN2S,"e",LN1S],LN3S), name(LN3,LN3S),
        merge_OL(B1,B2,B3).
merge_OL([LN1:A],[LN2:_],[LN3:A]) :- !,
        name(LN1,LN1S), name(LN2,LN2S),
        concat(["e",LN2S,"e",LN1S],LN3S), name(LN3,LN3S).
merge_OL([LN1:A1,A2|B1],[LN2:_,A3|B2],[LN3:A1|Ante]) :- !,
        name(LN1,LN1S), name(LN2,LN2S),
        concat(["e",LN2S,"e",LN1S],LN3S), name(LN3,LN3S),
        merge_OL([A2|B1],[A3|B2],Ante).

variant(X,Y) :-
        \+(\+(( numbervars(X,0,_), numbervars(Y,0,_), X=Y ))).

multi_entries1([X|Rest],[X|Hd],Tl) :- multi_entries1(Rest,Hd,Tl).
multi_entries1([],Tl,Tl).

entry(CN,AL,A,GD,Proc,CSQ,Entry,_) :-
        entry1(CN,[],AL,A,GD,Proc,CSQ,Entry).
entry(CN,AL,A,GD,Proc,CSQ,Entry,nodup) :-
        entry2(CN,[],AL,A,GD,Proc,CSQ,Entry).

entry1(CN,Lits,[LN:Lit|Rest],A0,GD,Proc,CSQ,Entry) :-
        rev_append(Lits,Rest,OL),
        continuation(Lit,OL,GD,Proc,CSQ,NewLit+NewOL),
        Entry= mg_cl(CN,(fire(LN:NewLit,NewOL,A0)->GD-->CSQ)).
entry1(CN,Lits,[LN:Lit|Rest],A0,GD,Proc,CSQ,Entry) :-
        entry1(CN,[LN:Lit|Lits],Rest,A0,GD,Proc,CSQ,Entry).

entry2(CN,Lits,[LN:Lit|Rest],A0,GD,Proc,CSQ,Entry) :-
        factoring(LN,Lit,Rest,Lits,(A0,GD,Proc,CSQ),
                  LN1,Lit1,Rest1,Lits1,(AL1,GD1,Proc1,CSQ1)),
        name(LN1,LN1S), name(NewLN,[102|LN1S]),
        rev_append(Lits1,Rest1,OL),
        continuation(Lit1,OL,GD1,Proc1,CSQ1,NewLit+NewOL),
        Entry= mg_cl(CN,(fire(NewLN:NewLit,NewOL,AL1)->GD1-->CSQ1)).
entry2(CN,Lits,[LN:Lit|Rest],A0,GD,Proc,CSQ,Entry) :-
        entry2(CN,[LN:Lit|Lits],Rest,A0,GD,Proc,CSQ,Entry).

rev_append([X|L1],L2,L3) :- !, rev_append(L1,[X|L2],L3).
rev_append([],L2,L2).

%%%%

continuation(Lit,OL,GD,Proc,CSQ,Entry) :-
        vars_in(Lit,[],Lvars),
        vars_in((GD,Proc,CSQ),[],Qvars),
        reorder_lit(Lvars,Qvars,OL,OL1,GD),
        Entry= Lit+OL1,!.

reorder_lit(Lvars,Cvars,LL,[FixG|OL],GD) :-
        extract_tmpG(Lvars,GD,TmpG,RestG),
        check_lit(Lvars,LL,OL,TL,[],Ilit),
        reorder_lit1(Lvars,Cvars,Ilit,TL,RestG,TmpG,FixG).

reorder_lit1(_,_,[],[],{{}},LastG,LastG) :- !.
reorder_lit1(_,_,[],[],{{RestG}},{{}},{{RestG}}) :- !.
reorder_lit1(_,_,[],[],{{RestG}},{{TmpG}},{{TmpG,RestG}}) :- !.

reorder_lit1(Lvars,Cvars,[LN:Lit|Ilit],[[LN:Lit+LitA+LitB+Vars|OL]],
        RestG,TmpG,TmpG) :-
        vars_in(Lit,[],Tvars),
        vars_in(Ilit,[],Ivars),
        union_vars(Cvars,Ivars,Nvars),
        intersection_vars(Lvars,Nvars,SVars),
        Vars=..[env|SVars],
        intersection_vars(Lvars,Tvars,Avars),
        mk_pattern(Lit,Avars,(*),LitA),
        intersection_vars(Nvars,Tvars,Bvars),
        vars_count(Lit,[],Tvars1),
        dup_vars(Tvars1,DupVars),
        union_vars(DupVars,Bvars,Bvars1),
        mk_pattern(Lit,Bvars1,'_',LitB),
        union_vars(Lvars,Tvars,NewLvars),
        reorder_lit(NewLvars,Cvars,Ilit,OL,RestG).

extract_tmpG(_,{{}},{{}},{{}}) :- !.
extract_tmpG(Lvars,{{G1,G2}},TmpG,RestG) :- !,
        extract_tmpG(Lvars,{{G1}},TmpG1,RestG1),
        extract_tmpG(Lvars,{{G2}},TmpG2,RestG2),
        add_G(TmpG1,TmpG2,TmpG),
        add_G(RestG1,RestG2,RestG).
extract_tmpG(Lvars,{{G}},{{G}},{{}}) :-
        vars_in(G,[],Gvars),
        subset_vars(Gvars,Lvars),!.
extract_tmpG(_,{{G}},{{}},{{G}}).

add_G({{}},TmpG,TmpG) :- !.
add_G(TmpG,{{}},TmpG) :- !.
add_G({{A}},{{B}},{{A,B}}).

mk_pattern(V,Dvars,_,V) :- var(V), member_var(V,Dvars),!.
mk_pattern(V,_,M,M) :- var(V),!.
mk_pattern(A,_,_,A) :- atom(A),!.
mk_pattern(T,Dvars,M,NewT) :- T=..[F|Args],
        mk_pattern_args(Args,Dvars,M,NewArgs),
        NewT=..[F|NewArgs].

mk_pattern_args([A1|Args],Dvars,M,[N1|NewArgs]) :-
        mk_pattern(A1,Dvars,M,N1),
        mk_pattern_args(Args,Dvars,M,NewArgs).
mk_pattern_args([],_,_,[]).

%%%%

check_lit(Lvars,[LN:Lit|LL],[LN:Lit|OL],TL,Li,Lo) :-
        vars_in(Lit,[],Tvars),
        subset_vars(Tvars,Lvars),!,
        check_lit(Lvars,LL,OL,TL,Li,Lo).
check_lit(Lvars,[LN:Lit|LL],OL,TL,Li,Lo) :-
        vars_in(Lit,[],Tvars),
        intersection_vars(Lvars,Tvars,Ivars),
        length(Ivars,IV),
        insert_lit(Lvars,LN,Lit,IV,Li,Li1),
        check_lit(Lvars,LL,OL,TL,Li1,Lo).
check_lit(_,[],TL,TL,Li,Li).

insert_lit(_,LN,Lit,_,[],[LN:Lit]) :- !.
insert_lit(Lvars,LN,Lit,IV,[LN1:Lit1|Li],Lo) :-
        vars_in(Lit1,[],Tvars1),
        intersection_vars(Tvars1,Lvars,Ivars1),
        length(Ivars1,IV1),
        ( IV1 >= IV -> Lo=[LN1:Lit1|Lo1],
                      insert_lit(Lvars,LN,Lit,IV,Li,Lo1) ;
          IV1 < IV -> Lo=[LN:Lit,LN1:Lit1|Li] ). 

%%%%

member_var(V,[VV|_]) :- V==VV,!.
member_var(V,[_|VV]) :- member_var(V,VV).

subset_vars([V|VV1],VV2) :- member_var(V,VV2),!,
        subset_vars(VV1,VV2).
subset_vars([],_).

union_vars(VV1,[],VV1) :- !.
union_vars([],VV2,VV2) :- !.
union_vars([V|VV1],VV2,UVV) :- member_var(V,VV2),!,
        union_vars(VV1,VV2,UVV).
union_vars([V|VV1],VV2,[V|UVV]) :- union_vars(VV1,VV2,UVV).

intersection_vars(_,[],[]) :- !.
intersection_vars([],_,[]) :- !.
intersection_vars([V|VV1],VV2,[V|UVV]) :- member_var(V,VV2),!,
         intersection_vars(VV1,VV2,UVV).
intersection_vars([_|VV1],VV2,UVV) :- intersection_vars(VV1,VV2,UVV).

vars_count(V,Vi,Vo) :- var(V),!, add_vc(V,Vi,Vo).
vars_count({{P}},Vi,Vo) :- !, vars_count(P,Vi,Vo).
vars_count([A|B],Vi,Vo) :- vars_count(A,Vi,Vm), vars_count(B,Vm,Vo).
vars_count([],Vi,Vi).
vars_count(Atom,Vi,Vi) :- atom(Atom),!.
vars_count(Int,Vi,Vi) :- integer(Int),!.
vars_count(Term,Vi,Vo) :- functor(Term,_,Arity),
        vars_count_args(1,Arity,Term,Vi,Vo).

vars_count_args(Jth,Last,Term,Vi,Vo) :-
        arg(Jth,Term,Arg),
        ( Jth < Last ->
	        J1th is Jth+1,
	        vars_count(Arg,Vi,Vm),
	        vars_count_args(J1th,Last,Term,Vm,Vo) ;
	  Jth= Last ->
	        vars_count(Arg,Vi,Vo) ).

add_vc(V,[V1=C|Vi],[V1=C1|Vi]) :- V==V1,!, C1 is C+1.
add_vc(V,[V1|Vi],[V1|Vo]) :- add_vc(V,Vi,Vo).
add_vc(V,[],[V=1]).

get_vc(V,[V1=C|_],C) :- V==V1,!.
get_vc(V,[_|Vi],C) :- get_vc(V,Vi,C).

dup_vars([_=1|Vars],DupVars) :- !, dup_vars(Vars,DupVars).
dup_vars([V=_|Vars],[V|DupVars]) :- !, dup_vars(Vars,DupVars).
dup_vars([],[]).

%%%%

factoring(LN,Lit,Rest,Lits,PP,LN1,Lit1,Rest1,Lits1,PP1) :-
        factoring1(LN,Lit,Rest,Lits,PP,LN2,Lit2,Rest2,Lits2,PP2),
        ( LN1=LN2, Lit1=Lit2, Rest1=Rest2, Lits1=Lits2, PP1=PP2 ;
          factoring(LN2,Lit2,Rest2,Lits2,PP2,LN1,Lit1,Rest1,Lits1,PP1) ).

factoring1(LN,Lit,[LN2:Lit2|Rest],Lits,PP,LN1,FLit,FRest,FLits,FPP) :-
        copy_term((Lit,Lit2,Rest,Lits,PP),(FLit,FLit2,FRest,FLits,FPP)),
        unify_once(FLit,FLit2),
        name(LN,LNS), name(LN2,LN2S), append(LNS,[102|LN2S],LN1S),
        name(LN1,LN1S).
factoring1(LN,Lit,[LN2:Lit2|Rest],Lits,PP,LN1,Lit1,Rest1,Lits1,PP1) :-
        factoring1(LN,Lit,Rest,[LN2:Lit2|Lits],PP,LN1,Lit1,Rest1,Lits1,PP1).

unify_once(X,Y) :- unify(X,Y),!.

%%%%

unify(X,Y) :-
        var(X) ->
                (var(Y) ->
                        X = Y;
                %true ->
                        functor(Y,_,N),
                        (N = 0 -> true;
                        N = 1 ->
                                arg(1,Y,Y1),
                                not_occurs_in(X,Y1);
                        %true ->
                                not_occurs_in_args(X,Y,N)),
                        X = Y);
        var(Y) ->
                functor(X,_,N),
                (N = 0 ->
                        true;
                N = 1 ->
                        arg(1,X,X1),
                        not_occurs_in(Y,X1);
                %true ->
                        not_occurs_in_args(Y,X,N)),
                X = Y;
        %true ->
                functor(X,F,N),
                functor(Y,F,N),
                (N = 0 ->
                        true;
                N = 1 ->
                        arg(1,X,X1),
                        arg(1,Y,Y1),
                        unify(X1,Y1);
                %true ->
                        unify_args(X,Y,N)).

unify_args(X,Y,N) :-
        N = 1 ->
                arg(1,X,X1),
                arg(1,Y,Y1),
                unify(X1,Y1);
        %true ->
                arg(N,X,Xn),
                arg(N,Y,Yn),
                unify(Xn,Yn),
                N1 is N - 1,
                unify_args(X,Y,N1).

%%%%

not_occurs_in(Var,Term) :-
        Var == Term ->
                fail;
        var(Term) ->
                true;
        %true ->
                functor(Term,_,N),
                (N = 0 ->
                        true;
                N = 1 ->
                        arg(1,Term,Arg1),
                        not_occurs_in(Var,Arg1);
                %true ->
                        not_occurs_in_args(Var,Term,N)).

not_occurs_in_args(Var,Term,N) :-
        N = 1 ->
                arg(1,Term,Arg1),
                not_occurs_in(Var,Arg1);
        %true ->
                arg(N,Term,Argn),
                not_occurs_in(Var,Argn),
                N1 is N - 1,
                not_occurs_in_args(Var,Term,N1).

%%%%

remove_taut([Cls|Cls1],Cls2) :- 
        \+(\+((numbervars(Cls,0,_),
               Cls=mg_cl(_,(fire(_:Atom,_,_)->_-->_:Atom))))),!,
        remove_taut(Cls1,Cls2).
remove_taut([Cls|Cls1],[Cls|Cls2]) :- !,
        remove_taut(Cls1,Cls2).
remove_taut([],[]).

%%  read_cls  END

%%%%

%%  output_tm  BEGIN

output_tm(Preds,Fncts) :-
        length(Preds,PL),
        mk_new_vec(PL,PList),
        Root=..['''ROOT'''|PList],
        write('
:- public tm_new/1.
'),
        write(tm_new(Root)),period,nl,
        output_tm_put(Preds,Fncts),
        newpage,
        output_tm_member(Preds,Fncts).

mk_new_vec(0,[]) :- !.
mk_new_vec(N,['_'|L]) :- N1 is N-1, mk_new_vec(N1,L).

%%%%

output_tm_member(Preds,Fncts) :-
        write('
:- public tm_member/2.
tm_member(X,TM) :- tm_member0(X,TM,Leaf),!, nonvar(Leaf).

'),
        output_tm_member1(preds,Preds,1),write('
tm_member1(_,Node,_) :- var(Node),!.
'),
        output_tm_member1(fncts,Fncts,2),
	write('
tm_member1(X,Node,NextNode) :- arg(1,Node,NodeList),
    tm_member_list(NodeList,X,NextNode).
'),write('
tm_member_list(NodeList,_,_) :- var(NodeList),!.
tm_member_list([(X->NextNode)|_],X,NextNode).
tm_member_list([_|NodeList],X,NextNode) :-
    tm_member_list(NodeList,X,NextNode).
').

output_tm_member1(preds,[Term|Rest],Key) :- !,
        functor(Term,_,Arity),
        ( Arity = 0 -> \+(\+(( numbervars(Term,0,_),
             write(tm_member0(Term,'Root','Leaf')),write(' :- '),
             write(arg(Key,'Root','Leaf')),period,nl ))) ;
          Arity > 0 -> \+(\+(( numbervars(Term,0,_),
             write(tm_member0(Term,'Root','Leaf')),write(' :- '),
             write(arg(Key,'Root','Node1')),comma,nl,
             output_tm_member2(Term,1,Arity,'Leaf') ))) ),
        Key1 is Key+1,
        output_tm_member1(preds,Rest,Key1).
output_tm_member1(fncts,[Term|Rest],Key) :- !,
        functor(Term,_,Arity),
        ( Arity = 0 -> \+(\+(( numbervars(Term,0,_),
             write(tm_member1(Term,'Node','NextNode')),write(' :- '),
             write(arg(Key,'Node','NextNode')),period,nl ))) ;
          Arity > 0 -> \+(\+(( numbervars(Term,0,_),
             write(tm_member1(Term,'Node','NextNode')),write(' :- '),
             write(arg(Key,'Node','Node1')),comma,nl,
             output_tm_member2(Term,1,Arity,'NextNode') ))) ),
        Key1 is Key+1,
        output_tm_member1(fncts,Rest,Key1).
output_tm_member1(_,[],_).

output_tm_member2(Term,Jth,Last,LastNode) :-
        arg(Jth,Term,Arg),
        tab(4),write('tm_member1('),write(Arg),write(',Node'),
        ( Jth < Last -> J1th is Jth+1,
                write(Jth),write(',Node'),write(J1th),write('),'),nl,
                output_tm_member2(Term,J1th,Last,LastNode) ;
          Jth = Last ->
                write(Jth),write(','),write(LastNode),write(').'),nl ).

%%%%

output_tm_put(Preds,Fncts) :-
        length(Fncts,FL),
        mk_new_vec(FL,FList),
        FreshNode=..['''NODE''','_'|FList],
        write('
:- public tm_put/3.
'),nl,
        output_tm_put1(preds,Preds,1),
	write('
tm_put1(X,Node,NextNode) :- var(Node),!,
    Node='),write(FreshNode),write(',
    tm_put1(X,Node,NextNode).
'),
        output_tm_put1(fncts,Fncts,2),write('
tm_put1(X,Node,NextNode) :- arg(1,Node,NodeList),
    tm_put_list(NodeList,X,NextNode).
'),write('
tm_put_list(NodeList,X,NextNode) :- var(NodeList),!,
    NodeList=[(X->NextNode)|_].
tm_put_list([(X->NextNode)|_],X,NextNode) :- !.
tm_put_list([_|NodeList],X,NextNode) :-
    tm_put_list(NodeList,X,NextNode).
').

%%%%

output_tm_put1(preds,[Term|Rest],Key) :-
        write('tm_put('),
        functor(Term,_,Arity),
        ( Arity = 0 -> \+(\+(( numbervars(Term,0,_),
             write(Term),
             write(',Root,Leaf) :- !, '),
             write(arg(Key,'Root','Leaf')),period,nl ))) ;
          Arity > 0 -> \+(\+(( numbervars(Term,0,_),
             write(Term),
             write(',Root,Leaf) :- !, '),
             write(arg(Key,'Root','Node1')),comma,nl,
             output_tm_put2(preds,Term,1,Arity,'Leaf') ))) ),
        Key1 is Key+1,
        output_tm_put1(preds,Rest,Key1).
output_tm_put1(fncts,[Term|Rest],Key) :-
        write('tm_put1('),
        functor(Term,_,Arity),
        ( Arity = 0 -> \+(\+(( numbervars(Term,0,_),
             write(Term),write(',Node,NextNode) :- !, '),
             write(arg(Key,'Node','NextNode')),period,nl ))) ;
          Arity > 0 -> \+(\+(( numbervars(Term,0,_),
             write(Term),write(',Node,NextNode) :- !, '),
             write(arg(Key,'Node','Node1')),comma,nl,
             output_tm_put2(fncts,Term,1,Arity,'NextNode') ))) ),
        Key1 is Key+1,
        output_tm_put1(fncts,Rest,Key1).
output_tm_put1(_,[],_).

output_tm_put2(preds,Term,Jth,Last,LastNode) :-
        arg(Jth,Term,Arg),
        tab(4),write('tm_put1('),write(Arg),
        write(',Node'),write(Jth),
        ( Jth < Last -> J1th is Jth+1,
                write(',Node'),write(J1th),kokka,comma,nl,
                output_tm_put2(preds,Term,J1th,Last,LastNode) ;
          Jth = Last ->
                comma,write(LastNode),write(').'),nl ).

output_tm_put2(fncts,Term,Jth,Last,LastNode) :-
        arg(Jth,Term,Arg),
        tab(4),write('tm_put1('),write(Arg),
        write(',Node'),write(Jth),
        ( Jth < Last -> J1th is Jth+1,
                write(',Node'),write(J1th),kokka,comma,nl,
                output_tm_put2(fncts,Term,J1th,Last,LastNode) ;
          Jth = Last ->
                comma,write(LastNode),write(').'),nl ).

%%  output_tm  END

%%%%

%%  output_cls  BEGIN

output_positive_cls([]) :- !,
        write('!! THERE IS NO POSITIVE CLAUSE.  !!
').
output_positive_cls(Pcls) :-
        write('
:- public positive_horn_cls/1.
positive_horn_cls(Cls) :- '),
	write(findall('X',pos_horn('X'),'Cls')),write(',!'),period,nl,
        output_positive_cls1(Pcls,horn),
        write('
:- public positive_nonhorn_cls/1.
positive_nonhorn_cls(Cls) :- '),
	write(findall('X',pos_nonhorn('X'),'Cls')),write(',!'),period,nl,
        output_positive_cls1(Pcls,nonhorn).

output_positive_cls1([mg_cl(CN,(_->{{}}-->CNSQ))|Ps],horn) :-
	\+nonhorn(CNSQ),!,
	nl,write(pos_horn('Res')),if,nl,
        output_cnsq(CN,1:true,CNSQ),period,nl,
        output_positive_cls1(Ps,horn).
output_positive_cls1([mg_cl(CN,(_->{{}}-->CNSQ))|Ps],nonhorn) :-
	nonhorn(CNSQ),!,
	nl,write(pos_nonhorn('Res')),if,nl,
        output_cnsq(CN,1:true,CNSQ),period,nl,
        output_positive_cls1(Ps,nonhorn).
output_positive_cls1([_|Ps],HNH) :-
        output_positive_cls1(Ps,HNH).
output_positive_cls1([],_).

%%%%

output_cls([C1|Cls]) :- output_cls1(C1),!, output_cls(Cls).
output_cls([]).

output_cls1(mg_cl(CN,(fire(LN:Lit,OL,ANT) -> GD --> CSQ))) :-
	name_concat([CN,'_',LN],Pname),
	refer_tm(OL,TM),
        vars_in(CSQ,[],Cvars),
        \+(\+(( numbervars((ANT,GD,CSQ),0,_),
	        Call=..[Pname,(Lit,'Nbr',TM),'Res'],
	        nl,write(Call),if,nl,
	        output_ante(CN,LN,{LN:'Nbr'},1,OL,Cvars,{{}},CSQ) ))).

refer_tm([],'_') :- !.
refer_tm([{{}}],'_') :- !.
refer_tm([{{_}}],'_') :- !.
refer_tm(_,'TM').

output_ante(CN,_,{Info},_,[],_,{{}},N:false) :- !,
	tab(4),write('Res'={CN:(Info-->N)}),period,nl.

output_ante(CN,_,{Info},_,[],_,{{}},CNSQ) :- !,
	output_cnsq(CN,Info,CNSQ),period,nl.

output_ante(CN,LN,{Info},NN,[[LN1:Lit+_+_+_|OL]],CSQvars,Proc,CSQ) :- !,
	name_concat(['Nbr',NN],Nbr1), NN1 is NN+1,
	name_concat(['Leaf',NN],LeafN),
	tab(4),write(tm_member0(Lit,'TM',LeafN)),comma,
	write((nonvar(LeafN),LeafN='''LEAF'''(Nbr1))),comma,nl,
    output_ante(CN,LN,{Info,LN1:Nbr1},NN1,OL,CSQvars,Proc,CSQ).

output_ante(CN,LN,{Info},NN,[LN1:Lit|OL],CSQvars,Proc,CSQ) :- !,
	name_concat(['Nbr',NN],Nbr1), NN1 is NN+1,
	name_concat(['Leaf',NN],LeafN),
	tab(4),write(tm_member0(Lit,'TM',LeafN)),comma,
	write((nonvar(LeafN),LeafN='''LEAF'''(Nbr1))),comma,nl,
    output_ante(CN,LN,{Info,LN1:Nbr1},NN1,OL,CSQvars,Proc,CSQ).

output_ante(CN,LN,Info,NN,[{{}}|OL],CSQvars,Proc,CSQ) :- !,
    output_ante(CN,LN,Info,NN,OL,CSQvars,Proc,CSQ).

output_ante(CN,LN,Info,NN,[{{G}}|OL],CSQvars,Proc,CSQ) :- !,
	tab(4),output_guard({{G}}),
    output_ante(CN,LN,Info,NN,OL,CSQvars,Proc,CSQ).

%%%%

output_cnsq(CN,Ainfo,(A,B)) :-
        unit_disj_in((A,B),Ulist,[],Dlist,[]),
	tab(2),kakko,space,
        output_cnsq_u(Ulist,CN,Ainfo),
        output_cnsq_d(Dlist,CN,Ainfo),
	space,kokka.
output_cnsq(CN,Ainfo,(A;B)) :-
        disj2list((A;B),DL,[]),
	tab(4),write('Res={[ '), output_disj2(DL,CN,Ainfo), write(' ]}').
output_cnsq(CN,Ainfo,'ATOM'(U1,ID1,Tcall,Hcall,NHcall)) :-
	tab(4),
	write('Res'='''ATOM'''(U1,{CN:(Ainfo-->ID1)},Tcall,Hcall,NHcall)).

unit_disj_in((A,B),Uhd,Utl,Dhd,Dtl) :- !,
        unit_disj_in(A,Uhd,Umd,Dhd,Dmd),
        unit_disj_in(B,Umd,Utl,Dmd,Dtl).
unit_disj_in((A;B),Utl,Utl,[(A;B)|Dtl],Dtl) :- !.
unit_disj_in(A,[A|Utl],Utl,Dtl,Dtl).

output_cnsq_u(['ATOM'(U1,ID1,Tcall,Hcall,NHcall)],CN,Ainfo) :- !,
	write('Res'='''ATOM'''(U1,{CN:(Ainfo-->ID1)},Tcall,Hcall,NHcall)).
output_cnsq_u(['ATOM'(U1,ID1,Tcall,Hcall,NHcall)|Us],CN,Ainfo) :-
	write('Res'='''ATOM'''(U1,{CN:(Ainfo-->ID1)},Tcall,Hcall,NHcall)),
	space,semicolon,nl,tab(4),
        output_cnsq_u(Us,CN,Ainfo).
output_cnsq_u([],_,_).

output_cnsq_d([D1],CN,Ainfo) :- !,
        disj2list(D1,DL,[]),
	write('Res={[ '), output_disj2(DL,CN,Ainfo), write(' ]}').
output_cnsq_d([D1|Ds],CN,Ainfo) :-
        disj2list(D1,DL,[]),
	write('Res={[ '), output_disj2(DL,CN,Ainfo), write(' ]}'),
	space,semicolon,nl,tab(4),
        output_cnsq_d(Ds,CN,Ainfo).
output_cnsq_d([],_,_).

output_disj2([D1],CN,Ainfo) :- !,
        write('[ '), output_disj3(D1,CN,Ainfo), write(' ]').
output_disj2([D1|Ds],CN,Ainfo) :- !,
        write('[ '), output_disj3(D1,CN,Ainfo), write(' ]'),
	comma,nl,tab(11),
        output_disj2(Ds,CN,Ainfo).

output_disj3(['ATOM'(U1,ID1,Tcall,Hcall,NHcall)],CN,Ainfo) :- !,
        write('''ATOM'''(U1,{CN:(Ainfo-->ID1)},Tcall,Hcall,NHcall)).
output_disj3(['ATOM'(U1,ID1,Tcall,Hcall,NHcall)|Us],CN,Ainfo) :-
        write('''ATOM'''(U1,{CN:(Ainfo-->ID1)},Tcall,Hcall,NHcall)),
        comma,nl,tab(13),
        output_disj3(Us,CN,Ainfo).

conj2list((A,B),Hd,Tl) :- !, conj2list(A,Hd,Md), conj2list(B,Md,Tl).
conj2list(A,[A|Tl],Tl).

disj2list((A;B),Hd,Tl) :- !, disj2list(A,Hd,Md), disj2list(B,Md,Tl).
disj2list(A,[AL|Tl],Tl) :- conj2list(A,AL,[]).

%%%%

output_guard({{}}) :- !.
output_guard(G) :- output_guard1(G),comma,nl.

output_guard1({{G1,G2}}) :- !,
        output_guard1({{G1}}),comma,nl,
        output_guard1({{G2}}).
output_guard1({{G}}) :- write(G).

output_body((P1,P2)) :- !,
        output_body(P1),comma,nl,
        output_body(P2).
output_body(P) :- write(P).

%%  output_cls  END

%%  mgpl4.pl  EOF

