/* ---------------------------------------------------------- 
%   (C)1992 Institute for New Generation Computer Technology 
%       (Read COPYRIGHT for detailed information.) 
----------------------------------------------------------- */
/*
 *	mktrie.c
 *			(C) Copyright 1991, All rights reserved by ICOT
 */

#include <stdio.h>
#include <ctype.h>
#include <strings.h>
#include "define.h"
#include "mktrie.h"
#include "kio.h"

static FILE *input_fp, *root_fp, *path_fp, *info_fp;
static long path_offset;

static bool input_flag;
static int line_num;
static w_char_t line_buf[MAX_LINE];

static w_char_t midasigo[NUM_MIDASIGO][MAX_MIDASIGO];
static int midasigo_max;

static struct path_r path_records[MAX_PATH_RECORDS];
static path_t path_p;
static struct info_r info_records[NUM_MIDASIGO];

static long write_path_buffer[MAX_PATH_TBL/sizeof(long)+1];
static long write_info_buffer[MAX_INFO_TBL/sizeof(long)+1];

static char *write_path_buf = (char *)write_path_buffer;
static char *write_info_buf = (char *)write_info_buffer;

#ifdef DEBUG
int	_max_m = 0;
int	_max_b = 0;
#endif

/*
 *	read functions
 */

void error_record()
{
    (void)fprintf(stderr, "illegal record [%d]\n", line_num);
    write_line(stderr, line_buf);
}

w_char_t *read_string(p1, p2, max)
    register w_char_t *p1, *p2;
    int max;
{
    register int i;

    while (ISspace(*p1))
        p1++;
    for (i = 0; i < max; i++) {
	if (ISspace(*p1) || *p1 == NULL)
	    break;
	*p2++ = *p1++;
    }
    if (i == 0){
	error_record();
	return NULL;
    }
    *p2++ = NULL;
    return p1;
}

w_char_t *read_ln()
{
    if (input_flag == TRUE) {
	input_flag = FALSE;
	return line_buf;
    }
    ++line_num;
    switch (read_line(input_fp, line_buf)) {
    case -1:
	(void)fprintf(stderr, "input error.\n");
	exit(1);
    case 0:
	return NULL;
    default:
	break;
    }
    return line_buf;
}

void unread()
{
    input_flag = TRUE;
}

bool read_record(n)
    int n;
{
    register w_char_t *bp;
    register int i;
    static w_char_t sbuf[5];
    static char str[5];
    int code;

 retry:
    if ((bp = read_ln()) == NULL) 
	return FALSE;
    if ((bp = read_string(bp, &(midasigo[n][0]), MAX_MIDASIGO)) == NULL)
        goto retry;
    if ((bp = read_string(bp, &(info_records[n].yomi[0]), MAX_YOMI)) == NULL)
        goto retry;
    for (i = 0; i < MAX_YOMI; i++)
        if (info_records[n].yomi[i] == 0)
	    break;
    info_records[n].length = i;
    if (read_string(bp, sbuf, 4) == NULL)
        goto retry;
    for (i = 0; i < 4; i++) {
	if (sbuf[i] & 0xFF00) {
	    error_record();
	    goto retry;
	}
	str[i] = (char)sbuf[i];
    }
    str[i] = '\0';
    (void)sscanf(str, "%x", &code);
    info_records[n].hinsi = (unsigned short)code;
    info_records[n].next = NULL;
#ifdef DEBUG
    {
	int i;
	for (i = 0; i < MAX_MIDASIGO; i++)
	    if (midasigo[n][i] == 0)
	        break;
	if (_max_m < i)
	    _max_m = i;
    }
#endif
    return TRUE;
}

bool read_file()
{
    register w_char_t wch;
    register int n;

    bzero((char *)midasigo, sizeof(w_char_t) * MAX_MIDASIGO * NUM_MIDASIGO);
    n = 0;
    if (read_record(n) == FALSE)
        return FALSE;
    wch = midasigo[0][0];
    do {
	if (read_record(++n) == FALSE) {
	    midasigo_max = n;
	    return TRUE;
	}
    } while (midasigo[n][0] == wch);
    midasigo_max = n;
    unread();
    return TRUE;
}
	
/*
 *	write
 */

long write_info(info)
    register info_t info;
{
    register char *wp;
    int i, ch1, ch2;
    u_short size;
    long offset;

    wp = write_info_buf + INFO_HEAD_SIZE;
    while (info != NULL) {
	put_short(wp, info->hinsi);
	forward_short(wp);
	put_short(wp, info->length);
	forward_short(wp);
	for (i = 0; i < info->length; i++) {
	    ch1 = ch2 = info->yomi[i];
	    *wp++ = ch1 >> 8;
	    *wp++ = ch2 & 0xFF;
	}
	info = info->next;
    }
    size = wp - write_info_buf;
    put_short(write_info_buf, size);
    offset = ftell(info_fp);
    (void)fwrite(write_info_buf, sizeof(char), (int)size, info_fp);
    return offset;
}

long write_node(path, num, offset)
    register path_t path;
    int num;
    long offset;
{
    register char *wp;

    wp = write_path_buf;
    put_long(wp, num);
    forward_long(wp);
    while (path != NULL) {
	put_long(wp, path->offset);
	forward_long(wp);
	put_short(wp, path->code);
	forward_short(wp);
	path = path->next;
    }
    (void)fseek(path_fp, offset, 0);
    (void)fwrite(write_path_buf, sizeof(char), wp - write_path_buf, path_fp);
}

long write_path(parent, offset)
    register path_t parent;
    long offset;
{
    register path_t path;
    register long tail;
    int size;

    size = PATH_HEAD_SIZE + PATH_REC_SIZE * parent->count;
    tail = offset + size;
    for (path = parent->child; path != NULL; path = path->next) {
	if (path->code == 0)
	    path->offset = write_info((info_t)(path->child));
	else
	    tail = write_path(path, tail);
    }
    parent->offset = offset;
    write_node(parent->child, parent->count, offset);
    return tail;
}

void write_file(path)
    path_t path;
{
    register long offset;
    long n;

    set_long(n, path_offset);
    path_offset = write_path(path, path_offset);
    offset = (long)path->code - ROOT_BASE;
    offset <<= 2;
    (void)fseek(root_fp, offset, 0);
    (void)fwrite((char *)&n, sizeof(n), 1, root_fp);
}

/*
 *	make path
 */

void init_records()
{
    path_p = path_records;
}

path_t get_path()
{
    if (path_p >= &(path_records[MAX_PATH_RECORDS])) {
	(void)fprintf(stderr, "path area shortage.\n");
	exit(1);
    }
    return path_p++;
}

path_t make_info(row, num)
    int row, num;
{
    register int i, j, n;

    n = row + num;
    for (i = row, j = row + 1; j < n; i++, j++)
	info_records[i].next = &(info_records[j]);
    info_records[i].next = NULL;
    return (path_t)&(info_records[row]);
}

path_t make_path(row, col, num, cnt)
    int row, col, num, *cnt;
{
    register path_t path;
    register int n, rn;
    path_t prev;
    int end, count, i;
    w_char_t wch;
    struct path_r dummy;

    path = &dummy;
    rn = row;
    end = row + num;
    i = 0;
    do {
	wch = midasigo[rn][col];
	for (n = rn + 1; midasigo[n][col] == wch && n < end; n++)
	    ;
	count = n - rn;
	prev = path;
	prev->next = path = get_path();
	path->code = wch;
	path->child =
	  wch != NULL ?
	    make_path(rn, col+1, count, &(path->count)): make_info(rn, count);
	++i;
    } while ((rn += count) < end);
    path->next = NULL;
    *cnt = i;
    return dummy.next;
}

/*
 *	initialize routines
 */

void create_file(name, ext, mode, fp)
    char *name, *ext, *mode;
    FILE **fp;
{
    char file[FILE_NAME_SIZE];

    (void)strcpy(file, name);
    (void)strcat(file, ext);

    if ((*fp = fopen(file, mode)) == NULL) {
	(void)fprintf(stderr, "can't open file: %s\n", file);
	exit(1);
    }
}

void initialize(argc, argv)
    int argc;
    char **argv;
{
    char *input_file, *mode;

    if (argc != 2) {
	(void)fprintf(stderr, "Usage: mktrie file_name\n");
	exit(1);
    }
    input_file = argv[1];
    mode = "w";
    create_file(input_file, ".dic", "r", &input_fp);
    create_file(input_file, ".rot", mode, &root_fp);
    create_file(input_file, ".pat", mode, &path_fp);
    create_file(input_file, ".inf", mode, &info_fp);
    path_offset = PATH_BASE;
    input_flag = FALSE;
    line_num = 0;
}

void terminate()
{
    (void)fclose(input_fp);
    (void)fclose(root_fp);
    (void)fclose(path_fp);
    (void)fclose(info_fp);
}

/*
 *	main
 */

void main(argc, argv)
    int argc;
    char **argv;
{
    register path_t path;
    int count;

    initialize(argc, argv);
    while (read_file() != FALSE) {
	init_records();
	path = make_path(0, 0, midasigo_max, &count);
#ifdef DEBUG
	if (_max_b < midasigo_max)
	    _max_b = midasigo_max;
#endif
	write_file(path);
    }
#ifdef DEBUG
    (void)fprintf(stderr, "[%4d:%2d]\n", _max_b, _max_m);
#endif
    terminate();
    exit(0);
}
