/*
 *	(C)1993 Institute for New Generation Computer Technology
 *	Read COPYRIGHT for detailed information.
 *
 *
 *	spread.c	---	Spreading-activation manager.
 *
 */

#include	<stdio.h>

#define	PROTO_SPREAD_C
#include	"config.h"
#include	"define.h"
#include	"typedef.h"
#include	"global.h"
#include	"proto.h"
#include	"debug.h"
#undef	PROTO_SPREAD_C

#pragma segment	spread


void spread_activation(itrmax)
     int itrmax;
{
  int i, j;
  litrlrec *literal;
  linkrec *link;
  double disj, excl, assm, cmpl, eqlz, trns, dpnd, imin, imax;
  double disjD, exclD, assmD, cmplD, eqlzD, trnsD, dpndD;

#ifndef PROFILE
  printf("Spreading activations");
#endif
  detect_loops();
  for (i = 0; i < itrmax && !Gparams.cnvrgdp; i++) {
    /**** calculate depency-energy by spreading pressure ****/
    calc_dependency_energy();
    /**** calculate other sorts of energy ****/
    Gparams.cnvrgdp = TRUE;
    Gparams.ampl = 0.0;
    /* update activation values of literals */
    for (literal = Gcontrol.unsigned_preds;
	 literal != NULL; literal = literal->ctrl.nxt) {
      if (literal->tag == PSTERM ||
	  literal->tag == NUMBER || literal->tag == SYMBOL) {
	literal->act = 1.0;
	continue;
      } else if (literal->pphndl != NULL)
	calc_clausal_forces(literal->pphndl->begin, literal->pphndl,
			    &disj, &excl, &disjD, &exclD);
      calc_literal_forces(literal, &assm, &cmpl, &assmD, &cmplD, &imin, &imax);
      update_literal_activation(literal,
				disj, excl, assm, cmpl,
				disjD, exclD, assmD, cmplD, imin, imax);
    }
    for (literal = Gcontrol.signed_preds;
	 literal != NULL; literal = literal->ctrl.nxt) {
      if (literal->tag == PSTERM ||
	  literal->tag == NUMBER || literal->tag == SYMBOL) {
	literal->act = 1.0;
	continue;
      } else if (literal->pphndl != NULL)
	calc_clausal_forces(literal->pphndl->begin, literal->pphndl,
			    &disj, &excl, &disjD, &exclD);
      calc_literal_forces(literal, &assm, &cmpl, &assmD, &cmplD, &imin, &imax);
      update_literal_activation(literal,
				disj, excl, assm, cmpl,
				disjD, exclD, assmD, cmplD, imin, imax);
    }
    /* update activation values of links */
    for (link = Gcontrol.links; link != NULL; link = link->ctrl.nxt) {
      if (link->ptr[0]->joint->ltrl->tag == CONSTRAINT &&
	  strcmp(link->ptr[0]->joint->ltrl->body.afm.name, "true") == 0) {
	link->sub = 1.0;
	continue;
      }
      for (j = 0; j < link->n; j++)
	if (link->coeff[j].body.act < 1.0) {
	  calc_link_forces(link, j,
			   &assm, &eqlz, &trns, &dpnd,
			   &assmD, &eqlzD, &trnsD, &dpndD);
	  update_link_activation(link, j,
				 assm, eqlz, trns, dpnd,
				 assmD, eqlzD, trnsD, dpndD);
	}
    }
#ifndef PROFILE
    putchar('.');
#endif
  }
#ifndef PROFILE
  if (Gparams.cnvrgdp)
    printf("converged (%d)\n", i);
  else
    printf("not converged (%d, %lf)\n", i, Gparams.ampl);
#endif
}


void calc_dependency_energy()
{
  litrlrec *ltrl;
  jointrec *jnt0, *jnt1, *ljnt;
  dependrec *dpnd;

  /* First, dispose old pressures */
  for (ltrl = Gcontrol.signed_preds; ltrl != NULL; ltrl = ltrl->ctrl.nxt) {
    dispose_pressure_list(ltrl->press);
    ltrl->dfrc = 0.0;
    ltrl->press = NULL;
    ltrl->path = NULL;
  }
  /* Second, spread new pressures and mark all literals */
  for (ltrl = Gcontrol.signed_preds; ltrl != NULL; ltrl = ltrl->ctrl.nxt) {
    if (ltrl->tag == FEATURE)
      for (jnt0 = ltrl->body.pst.root->lefth;
	   jnt0 != NULL; jnt0 = jnt0->nxt)
	for (dpnd = jnt0->depend; dpnd != NULL; dpnd = dpnd->nxt) {
	  jnt1 = dpnd->link->ptr[dpnd->dir]->joint;
	  put_pressure(ltrl,
		       dpnd->link, dpnd->dir, jnt1->nth, jnt1->ftr,
		       nAct(ltrl->act), 0, 0, NULL);
	}
    else if (ltrl->tag != PSTERM)
      for (jnt0 = ltrl->lefth; jnt0 != NULL; jnt0 = jnt0->nxt)
	for (dpnd = jnt0->depend; dpnd != NULL; dpnd = dpnd->nxt) {
	  jnt1 = dpnd->link->ptr[dpnd->dir]->joint;
	  put_pressure(ltrl,
		       dpnd->link, dpnd->dir, jnt1->nth, jnt1->ftr,
		       nAct(ltrl->act), 0, 0, NULL);
	}
    ltrl->mark = FALSE;
  }
}


void calc_clausal_forces(clause, literal, disj, excl, disjD, exclD)
     pphandle *clause;
     pphandle *literal;
     double *disj, *excl, *disjD, *exclD;
{
  if (Gparams.disjSW)
    disjunction_force(disj, disjD, clause, literal);
  else {
    *disj = 0.0, *disjD = 0.0;
  }
  if (Gparams.exclSW)
    exclusion_force(excl, exclD, clause, literal);
  else {
    *excl = 0.0, *exclD = 0.0;
  }
}


void calc_literal_forces(literal, assm, cmpl, assmD, cmplD, imin, imax)
     litrlrec *literal;
     double *assm, *cmpl, *assmD, *cmplD;
     double *imin, *imax;
{
  if (Gparams.cmplSW)
    completion_force(cmpl, cmplD, literal);
  else {
    *cmpl = 0.0, *cmplD = 0.0;
  }
  if (Gparams.assmSW)
    aform_assimilation_force(assm, assmD, literal, imin, imax);
  else {
    *assm = 0.0, *assmD = 0.0;
    *imin = Gparams.epsilon, *imax = 1.0-Gparams.epsilon;
  }
}


void update_literal_activation(literal,
			       disj, excl, assm, cmpl,
			       disjD, exclD, assmD, cmplD,
			       imin, imax)
     litrlrec *literal;
     double disj, excl, assm, cmpl;
     double disjD, exclD, assmD, cmplD;
     double imin, imax;
{
  double dUdx, dfdx, temp;

  dUdx =
    - Gparams.tmprtr * (log(literal->act) - log(1.0-literal->act))
      - Gparams.disjWT * disj
	- Gparams.exclWT * excl
	  - Gparams.assmWT * assm
	    - Gparams.cmplWT * cmpl
	      - Gparams.psdoWT * literal->dfrc * nDXDx(literal->act);
  dfdx =
    - Gparams.tmprtr / (literal->act * (1.0-literal->act))
      - Gparams.disjWT * disjD
	- Gparams.exclWT * exclD
	  - Gparams.assmWT * assmD
	    - Gparams.cmplWT * cmplD
	      - Gparams.psdoWT * literal->dfrc * nD2XDx2(literal->act);

  temp = literal->act - dUdx / dfdx;
  if (temp >= 1.0)
    temp = (literal->act + 1.0) / 2.0;
  else if (temp <= 0.0)
    temp = (0.0 + literal->act) / 2.0;

  if (temp > imax)
    temp = imax;
  else if (temp < imin)
    temp = imin;

  if (fabs(literal->act-temp) > Gparams.epsilon) {
    Gparams.ampl = fabs(literal->act - temp);
    Gparams.cnvrgdp = FALSE;
  }
  literal->act = temp;
}


void calc_link_forces(link, n,
		      assm, eqlz, trns, dpnd,
		      assmD, eqlzD, trnsD, dpndD)
     linkrec *link;
     int n;
     double *assm, *eqlz, *trns, *dpnd;
     double *assmD, *eqlzD, *trnsD, *dpndD;
{
  if (Gparams.assmSW)
    link_assimilation_force(assm, assmD, link, n);
  else {
    *assm = 0.0, *assmD = 0.0;
  }
  if (Gparams.eqlzSW)
    equalization_force(eqlz, eqlzD, link, n);
  else {
    *eqlz = 0.0, *eqlzD = 0.0;
  }
  if (Gparams.trnsSW)
    transitivity_force(trns, trnsD, link, n);
  else {
    *trns = 0.0, *trnsD = 0.0;
  }
  link_dependency_force(dpnd, dpndD, link, n);
}


void update_link_activation(link, n,
			    assm, eqlz, trns, dpnd,
			    assmD, eqlzD, trnsD, dpndD)
     linkrec *link;
     int n;
     double assm, eqlz, trns, dpnd;
     double assmD, eqlzD, trnsD, dpndD;
{
  double dUdx, dfdx, temp;

  dUdx =
    - Gparams.tmprtr * (log(link->coeff[n].body.act)
			- log(1.0-link->coeff[n].body.act))
      - Gparams.assmWT * assm
	- Gparams.eqlzWT * eqlz
	  - Gparams.trnsWT * trns
	    -Gparams.psdoWT * dpnd;
  dfdx =
    - Gparams.tmprtr / (link->coeff[n].body.act*(1.0-link->coeff[n].body.act))
      - Gparams.assmWT * assmD
	- Gparams.eqlzWT * eqlzD
	  - Gparams.trnsWT * trnsD
	    - Gparams.psdoWT * dpndD;

  temp = link->coeff[n].body.act - dUdx / dfdx;
  if (temp >= 1.0)
    temp = (link->coeff[n].body.act + 1.0) / 2.0;
  else if (temp <= 0.0)
    temp = (0.0 + link->coeff[n].body.act) / 2.0;

  if (fabs(link->coeff[n].body.act-temp) > Gparams.epsilon) {
    Gparams.ampl = fabs(link->coeff[n].body.act - temp);
    Gparams.cnvrgdp = FALSE;
  }
  link->coeff[n].body.act = temp;
}


/*
 *   E_disj = D Pi (1 - r_i Y_i)
 *   --> d   E_disj / dx_k   = - SIGN(k) D r_k (nDXDx_k)   Pi_{i!=k} (1 - r_i Y_i)
 *   ==> d^2 E_disj / dx_k^2 = - SIGN(k) D r_k (nD2XDx2_k) Pi_{i!=k} (1 - r_i Y_i)
 *
 *   where y_i = x_i     (literal i is positive)
 *               1-x_i   (literal i is negative)
 *
 *     SIGN(i) = 1       (literal i is positive)
 *               -1      (literal i is negative)
 *
 *     nDXDx   = d   X / dx   = 1 - T'{log(x) + log(1-x) + 2}
 *     nD2XDx2 = d^2 X / dx^2 = - T'(1-2x) / x(1-x)
 */
void disjunction_force(dE, d2E, clause, literal)
     double *dE, *d2E;
     pphandle *clause, *literal;
{
  double subtotal;
  pphandle *ltrl;

  subtotal = 1.0;
  for (ltrl = clause; ltrl != NULL; ltrl = ltrl->next)
    if (ltrl != literal &&
	ltrl->body->tag != SYMBOL &&
	ltrl->body->tag != NUMBER &&
	ltrl->body->tag != PSTERM)
      subtotal *= 1.0 - ltrl->body->rel * nActV(ltrl->body);
  if (literal->body->pol == NEGATIVE ||
      literal->body->pol == COST_NEGATIVE)
    subtotal  *= - literal->body->rel;
  else
    subtotal  *= literal->body->rel;
  subtotal *= - clause->body->coeff.disj * clause->body->coeff.nOmega;

  *dE  = nDXDx(literal->body->act) * subtotal;
  *d2E = nD2XDx2(literal->body->act) * subtotal;
}


/*
 *   E_excl = E Sigma_{i!=j} (r_i Y_i r_j Y_j)
 *   --> d   E_excl / dx_k   = SIGN(k) E r_k (nDXDx_k)   Sigma_{i!=k} (r_i Y_i)
 *   ==> d^2 E_excl / dx_k^2 = SIGN(k) E r_k (nD2XDx2_k) Sigma_{i!=k} (r_i Y_i)
 *
 *   where y_i = x_i     (literal i is positive)
 *               1-x_i   (literal i is negative)
 *
 *     SIGN(i) = 1       (literal i is positive)
 *               -1      (literal i is negative)
 *
 *     nDXDx   = d   X / dx   = 1 - T'{log(x) + log(1-x) + 2}
 *     nD2XDx2 = d^2 X / dx^2 = - T'(1-2x) / x(1-x)
 */
void exclusion_force(dE, d2E, clause, literal)
     double *dE, *d2E;
     pphandle *clause, *literal;
{
  double subtotal;
  pphandle *ltrl;

  subtotal = 0.0;
  for (ltrl = clause; ltrl != NULL; ltrl = ltrl->next)
    if (ltrl != literal &&
	ltrl->body->tag != SYMBOL &&
	ltrl->body->tag != NUMBER &&
	ltrl->body->tag != PSTERM)
      subtotal += ltrl->body->rel * nActV(ltrl->body);
  if (literal->body->pol == NEGATIVE ||
      literal->body->pol == COST_NEGATIVE)
    subtotal *= - literal->body->rel;
  else
    subtotal *= literal->body->rel;
  subtotal *= clause->body->coeff.excl * clause->body->coeff.omegaN;

  *dE  = nDXDx(literal->body->act) * subtotal;
  *d2E = nD2XDx2(literal->body->act) * subtotal;
}


/*
 *   E_cmpl = C X_i Pi_{i@j} {1 - s_ij X_j}
 *   --> (1) d   E_cmpl / dx_i   = C (nDXDx_i) Pi_{i@j} {1 - s_ij X_j}
 *       (2) d   E_cmpl / dx_k
 *               = - C X_i s_ik (nDXDx_k) Pi_{i@j,j!=k} {1 - s_ij X_j}
 *   ==> (1) d^2 E_cmpl / dx_i^2 = C (nD2XDx2_i) Pi_{i@j} {1 - s_ij X_j}
 *       (2) d^2 E_cmpl / dx_k^2 =
 *            - C X_i s_ik (nD2XDx2_k) Pi_{i@j,j!=k} {1 - s_ij X_j}
 *
 *   where
 *     nDXDx   = d   X / dx   = 1 - T'{log(x) + log(1-x) + 2}
 *     nD2XDx2 = d^2 X / dx^2 = - T'(1-2x) / x(1-x)
 *
 * *NOTICE*
 *    x_i and x_j are not activation values of LITERALS.
 *    This routine calculates positive completion force.
 */
void completion_force(dE, d2E, literal)
     double *dE, *d2E;
     litrlrec *literal;
{
  double subtotal;
  litrlrec *ltrl;
  jointrec *jnt;
  dependrec *dpnd;
  double df;

  if (literal->tag != CONSTRAINT) {
    *dE = 0.0, *d2E = 0.0;
    return;
  }
  subtotal = 0.0;
  if (literal->pol != UNSIGNED) {
    if (literal->pol == COST_POSITIVE || literal->pol == COST_NEGATIVE ||
	((jnt=literal->joint) != NULL && (dpnd=jnt->depend) != NULL &&
	 (ltrl=dpnd->link->ptr[dpnd->dir]->joint->ltrl) != NULL &&
	 ltrl->pol == UNSIGNED))
      /*=== case (1): `literal' is a cost or a defined literal ===*/
      completion_force_aux1(&subtotal, literal, NULL);
    else if (jnt != NULL && dpnd != NULL && ltrl != NULL &&
	     (ltrl->pol == COST_POSITIVE || ltrl->pol == COST_NEGATIVE)) {
      /*=== case (2): `literal' is a cost literal ===*/
      for (jnt = literal->joint; jnt != NULL; jnt = jnt->nxt)
	for (dpnd = jnt->depend; dpnd != NULL; dpnd = dpnd->nxt) {
	  ltrl = dpnd->link->ptr[dpnd->dir]->joint->ltrl;
	  completion_force_aux1(&df, ltrl, literal);
	  subtotal -= nAct(ltrl->act) * df;
	}
    }
  } else
    /*=== case (2): `literal' is a defined literal ===*/
    for (jnt = literal->joint; jnt != NULL; jnt = jnt->nxt)
      for (dpnd = jnt->depend; dpnd != NULL; dpnd = dpnd->nxt) {
	ltrl = dpnd->link->ptr[dpnd->dir]->joint->ltrl;
	completion_force_aux1(&df, ltrl, literal);
	subtotal -= nAct(ltrl->act) * df;
      }
  *dE  = literal->coeff.cmpl * nDXDx(literal->act) * subtotal;
  *d2E = literal->coeff.cmpl * nD2XDx2(literal->act) * subtotal;
}


/*
 *   `sliteral' specifies i and `uliteral' specifies k.
 *   Result = s_ik Pi_{i@j,j!=k} {1 - s_ij X_j}
 *
 * *NOTICE*
 *    x_j is not activation value of LITERAL.
 *    This routine calculates a part of the positive completion force.
 */
void completion_force_aux1(res, sliteral, uliteral)
     double *res;
     litrlrec *sliteral, *uliteral;
{
  litrlrec *ltrl;
  jointrec *jnt;
  dependrec *dpnd;
  linkrec *lnk;

  *res = 1.0;
  for (jnt = sliteral->joint; jnt != NULL; jnt = jnt->nxt)
    for (dpnd = jnt->depend; dpnd != NULL; dpnd = dpnd->nxt) {
      lnk = dpnd->link;
      ltrl = lnk->ptr[dpnd->dir]->joint->ltrl;
      if (ltrl == uliteral)
	*res *= lnk->sub * lnk->confuse;
      else
	*res *= 1.0 - lnk->sub*lnk->confuse*nAct(ltrl->act);
    }
}


/*
 *   E_assm = - A s_ab (X_a - S(dP/ds_a))(X_b - S(dP/ds_b)) Pi X_e_ab
 *   --> d   E_assm / dx_a   = - A s_ab (nDXDx_a)   (X_b - S(dP/ds_b)) Pi X_e_ab
 *   ==> d^2 E_assm / dx_a^2 = - A s_ab (nD2XDx2_a) (X_b - S(dP/ds_b)) Pi X_e_ab
 *
 *   where
 *     nDXDx   = d   X / dx   = 1 - T'{log(x) + log(1-x) + 2}
 *     nD2XDx2 = d^2 X / dx^2 = - T'(1-2x) / x(1-x)
 */
void aform_assimilation_force(dE, d2E, literal, imin, imax)
     double *dE, *d2E;
     litrlrec *literal;
     double *imin, *imax;
{
  int i;
  double subtotal, delta;
  litrlrec *ltrl;
  jointrec *jnt;
  dependrec *dpnd;
  linkrec *lnk;

  subtotal = 0.0;
  *imin = Gparams.epsilon;
  *imax = 1.0-Gparams.epsilon;
  for (jnt = literal->joint; jnt != NULL; jnt = jnt->nxt)
    for (dpnd = jnt->depend; dpnd != NULL; dpnd = dpnd->nxt) {
      lnk = dpnd->link;
      ltrl = lnk->ptr[dpnd->dir]->joint->ltrl;
      for (delta = 1.0, i = 0; i < lnk->n; i++)
	delta *= nAct(lnk->coeff[i].body.act);
      subtotal += lnk->sub * lnk->confuse *
	(nAct(ltrl->act) - negP(ltrl->imp)) * delta;
    }
  *dE  = -literal->coeff.assm * nDXDx(literal->act) * subtotal;
  *d2E = -literal->coeff.assm * nD2XDx2(literal->act) * subtotal;
}


/*
 *   E_assm = - A s_ab (X_a - S(dP/ds_a))(X_b - S(dP/ds_b)) Pi X_e_ab
 *   --> d   E_assm / de_k   =
 *       - A s_ab (nDXDx_k) (X_a - S(dP/ds_a)) (X_b - S(dP/ds_b)) Pi_{d!=k} X_e_d
 *   ==> d^2 E_assm / de_k^2 =
 *       - A s_ab (nD2XDx2_a) (X_a - S(dP/ds_a)) (X_b - S(dP/ds_b)) Pi_{d!=k} X_e_d
 *
 *   where
 *     nDXDx   = d   X / dx   = 1 - T'{log(x) + log(1-x) + 2}
 *     nD2XDx2 = d^2 X / dx^2 = - T'(1-2x) / x(1-x)
 */
void link_assimilation_force(dE, d2E, link, nth)
     double *dE, *d2E;
     linkrec *link;
     int nth;
{
  int i;
  double subtotal, delta;
  litrlrec *ltrl0, *ltrl1;

  if (link->tag != INFERENCE) {
    *dE = 0.0, *d2E = 0.0;
    return;
  }
  ltrl0 = link->ptr[0]->joint->ltrl;
  ltrl1 = link->ptr[1]->joint->ltrl;
  for (delta = 1.0, i = 0; i < link->n; i++)
    if (i != nth)
      delta *= link->coeff[i].body.act;
  subtotal = - ltrl0->coeff.assm
    * link->sub * link->confuse
      * (nAct(ltrl0->act)-negP(ltrl0->imp))
	* (nAct(ltrl1->act)-negP(ltrl1->imp)) * delta;
  *dE  = nDXDx(link->coeff[nth].body.act) * subtotal;
  *d2E = nD2XDx2(link->coeff[nth].body.act) * subtotal;
}


/*
 *   E_eqlz = - X_e_0 Sigma_{i>0} X_e_i
 *   --> (1) dE_eqlz / dx_e_0 = - 1/n (nDXDx_e_0) Sigma_{i>0} X_e_i
 *       (2) dE_eqlz / dx_e_j = - 1/n X_e_0 (nDXDx_e_j)
 *
 *   ==> (1) d^2 E_eqlz / dx_e_0^2 = - 1/n (nD2XDx2_e_0) Sigma_{i>0} X_e_i
 *       (2) d^2 E_eqlz / dx_e_j^2 = - 1/n X_e_0 (nD2XDx2_e_j)
 *
 *   where
 *     nDXDx   = d   X / dx   = 1 - T'{log(x) + log(1-x) + 2}
 *     nD2XDx2 = d^2 X / dx^2 = - T'(1-2x) / x(1-x)
 */
void equalization_force(dE, d2E, link, nth)
     double *dE;
     double *d2E;
     linkrec *link;
     int nth;
{
  double subtotal;
  int i;

  if (link->tag != INFERENCE ||
      link->ptr[0]->joint->ltrl->tag == CONSTRAINT) {
    *dE = 0.0, *d2E = 0.0;
    return;
  }
  if (link->ptr[0]->joint->ltrl->tag == FEATURE) {
    /*=== case (1),(2) ===*/
    *dE  = - nDXDx(link->coeff[nth].body.act)
      * nAct(link->coeff[1-nth].body.act);
    *d2E = - nD2XDx2(link->coeff[nth].body.act)
      * nAct(link->coeff[1-nth].body.act);
  } else if (nth == link->n-1) {
    /*=== case (2) ===*/
    for (subtotal = 0.0, i = 0; i < link->n-1; i++)
      subtotal -= nAct(link->coeff[i].body.act);
    *dE  = nDXDx(link->coeff[nth].body.act) * subtotal / (link->n-1);
    *d2E = nD2XDx2(link->coeff[nth].body.act) * subtotal / (link->n-1);
  } else {
    /*=== case (1) ===*/
    *dE  = - nDXDx(link->coeff[nth].body.act)
      * nAct(link->coeff[link->n-1].body.act) / (link->n-1);
    *d2E = - nD2XDx2(link->coeff[nth].body.act)
      * nAct(link->coeff[link->n-1].body.act) / (link->n-1);
  }
}


/*
 *   E_trns = - Pi (X_e_i - theta)
 *   --> d   E_trns / dx_e_j   = - (nDXDx_e_j) Pi_{i!=j} (X_e_i - theta)
 *   ==> d^2 E_trns / dx_e_j^2 = - (nD2XDx2_e_j) Pi_{i!=j} (X_e_i - theta)
 *
 *   where
 *     nDXDx   = d   X / dx   = 1 - T'{log(x) + log(1-x) + 2}
 *     nD2XDx2 = d^2 X / dx^2 = - T'(1-2x) / x(1-x)
 */
void transitivity_force(dE, d2E, link, nth)
     double *dE;
     double *d2E;
     linkrec *link;
     int nth;
{
  double subtotal, df;
  index *loop;

  for (subtotal = 0.0, loop = Gcontrol.loops;
       loop != NULL; loop = loop->others)
    if (transitivity_force_aux(&df, loop, link, nth))
      subtotal += df;

  if (link->tag == EQUATION) {
    *dE  = - nDXDx(link->coeff[0].body.act) * subtotal;
    *d2E = - nD2XDx2(link->coeff[0].body.act) * subtotal;
  } else if (nth == PSTENTRY) {
    *dE  = - nDXDx(link->coeff[0].body.act) * subtotal;
    *d2E = - nD2XDx2(link->coeff[0].body.act) * subtotal;
  } else if (nth == LEFTHAND) {
    *dE  = nDXDx(link->coeff[link->n-1].body.act) * subtotal;
    *d2E = nD2XDx2(link->coeff[link->n-1].body.act) * subtotal;
  } else {
    *dE  = - nDXDx(link->coeff[nth].body.act) * subtotal;
    *d2E = - nD2XDx2(link->coeff[nth].body.act) * subtotal;
  }
}


/*
 *   `link', `nth' specify j.
 *   Result = Pi_{i!=j} (X_e_i - theta)
 *
 *   Notes:
 *	This routine is depend on routine detect_loops(). (loop.c)
 *	Though detect_loops() can detect loops including weak links,
 *	it cannot detect all possible loops.
 */
Boolean transitivity_force_aux(res, loop, link, nth)
     double *res;
     index *loop;
     linkrec *link;
     int nth;
{
  double pact;
  linkact_cell *az;
  int count;
  Boolean memberp;
  index *elem;

  count = 0;
  memberp = FALSE;
  *res = 1.0;
  for (elem = loop; elem != NULL; elem = elem->next)
    if (elem->link == link) {
      if (link->tag == EQUATION ||
	  (elem->nth == PSTENTRY && nth == 0) ||
	  (elem->nth == LEFTHAND && nth == link->n-1) ||
	  (elem->nth == nth))
	memberp = TRUE;
    } else {
      az = nth_act_z(elem->link, elem->nth);
      pact = nAct(az->body.act);
      if (pact - Gparams.theta <= Gparams.epsilon) {
	if (++count > 1)
	  break;
      }
      *res *= pact - Gparams.theta;
    }
  return (memberp && elem == NULL);
}


/*
 *   E_dpnd = - X_p X_q Pi s_i X_e_i
 *   --> d   E_dpnd / dx_e_j   = - X_p X_q s_j (nDXDx_e_j) Pi_{i!=j} s_i X_e_i
 *   ==> d^2 E_dpnd / dx_e_j^2 = - X_p X_q s_j (nD2XDx2_e_j) Pi_{i!=j} s_i X_e_i
 *
 *   dEs  = Sigma_j d   E_dpnd / dx_e_j
 *   d2Es = Sigma_j d^2 E_dpnd / dx_e_j^2
 */
void link_dependency_force(dEs, d2Es, link, nth)
     double *dEs;
     double *d2Es;
     linkrec *link;
     int nth;
{
  double subtotal;
  litrlrec *binding;
  press *prss;

  subtotal = 0.0;
  for (binding = Gcontrol.unsigned_preds;
       binding != NULL;
       binding = binding->ctrl.nxt) {
    for (prss = binding->path; prss != NULL; prss = prss->path.nxt)
      if (prss->link == link && prss->nth == nth)
	break;
    if (prss != NULL)
      subtotal -= binding->dfrc
	* nAct(binding->act) / nAct(link->coeff[nth].body.act);
  }
  *dEs  = nDXDx(link->coeff[nth].body.act) * subtotal;
  *d2Es = nD2XDx2(link->coeff[nth].body.act) * subtotal;
}
