/* Copyright	Massachusetts Institute of Technology	1987, 1988
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of M.I.T. not be used in advertising or
 * publicity pertaining to distribution of the software without specific,
 * written prior permission.  M.I.T. makes no representations about the
 * suitability of this software for any purpose.  It is provided "as is"
 * without express or implied warranty.
 *
 * M.I.T. DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL M.I.T.
 * BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION
 * OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#ifdef DEBUG
#include <stdio.h>
#endif
#include <X11/Intrinsic.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/CoreP.h>
#include <X11/CompositeP.h>
#include <X11/ConstrainP.h>
#include "LatticeP.h"

#define MAX(a, b)       (((a) > (b)) ? (a): (b))
#define IsVertical(lw) ((lw)->lattice.gravity == NorthGravity || \
			  (lw)->lattice.gravity == SouthGravity)
#define IsMirror(lw)   ((lw)->lattice.gravity == SouthGravity || \
			  (lw)->lattice.gravity == EastGravity)

					/* widget class method */
static void		ClassInitialize();
static void             Initialize();
static void             ConstraintInitialize();
static Boolean          ConstraintSetValues();
static void             Destroy();
static Boolean          SetValues();
static XtGeometryResult GeometryManager();
static void             ChangeManaged();
static void             Redisplay();

					/* utility routines */
static void             insert_node();
static void             delete_node();
static void             layout_lattice();

static LatticeOffsetPtr	create_offset();
static void		reset_positions();
static void		reset_offset();
static Position		current_position();
static void		set_current_position();
static Position		sum_of_positions();
static int		sort_list();
static int		compute_positions();
static void		shift_subtree();
static void		set_positions();
static void		mirror_positions();
static void		set_point();
static int		get_depth();

/*
 * resources of the lattice itself
 */
static XtResource resources[] = {
    { XtNhPad, XtCHPad, XtRDimension, sizeof (Dimension),
	XtOffsetOf(QxtLatticeRec, lattice.hpad), XtRImmediate, (XtPointer) 0 },
    { XtNvPad, XtCVPad, XtRDimension, sizeof (Dimension),
	XtOffsetOf(QxtLatticeRec, lattice.vpad), XtRImmediate, (XtPointer) 0 },
    { XtNlineColor, XtCLineColor, XtRPixel, sizeof (Pixel),
	XtOffsetOf(QxtLatticeRec, lattice.line_pixel), XtRString,
	XtDefaultForeground},
    { XtNlineWidth, XtCLineWidth, XtRDimension, sizeof (Dimension),
	XtOffsetOf(QxtLatticeRec,lattice.line_width),XtRImmediate,(XtPointer)1},
    { XtNgravity, XtCGravity, XtRGravity, sizeof (XtGravity),
	XtOffsetOf(QxtLatticeRec, lattice.gravity), XtRImmediate,
	(XtPointer) NorthGravity },
};

/*
 * resources that are attached to all children of the lattice
 */
static XtResource latticeConstraintResources[] = {
  { XtNparents, XtCParents, XtRWidgetList, sizeof(XtPointer),
	XtOffsetOf(LatticeConstraintsRec,lattice.parents),XtRWidgetList,NULL },
  { XtNnumParents, XtCNumParents, XtRInt, sizeof(int),
	XtOffsetOf(LatticeConstraintsRec,lattice.n_parents), XtRImmediate, 0 },
  { XtNlatticeGC, XtCLatticeGC, XtRGC, sizeof(GC),
	XtOffsetOf(LatticeConstraintsRec, lattice.gc), XtRImmediate, NULL },
};

QxtLatticeClassRec QxtlatticeClassRec = {
  {
					/* core_class fields  */
    (WidgetClass) &constraintClassRec,	/* superclass         */
    "Lattice",				/* class_name         */
    sizeof(QxtLatticeRec),		/* widget_size        */
    ClassInitialize,			/* class_init         */
    NULL,				/* class_part_init    */
    FALSE,				/* class_inited       */	
    Initialize,				/* initialize         */
    NULL,				/* initialize_hook    */	
    XtInheritRealize,			/* realize            */
    NULL,				/* actions            */
    0,					/* num_actions        */	
    resources,				/* resources          */
    XtNumber(resources),		/* num_resources      */
    NULLQUARK,				/* xrm_class          */
    TRUE,				/* compress_motion    */	
    TRUE,				/* compress_exposure  */	
    TRUE,				/* compress_enterleave*/	
    TRUE,				/* visible_interest   */
    Destroy,				/* destroy            */
    NULL,				/* resize             */
    Redisplay,				/* expose             */
    SetValues,				/* set_values         */
    NULL,				/* set_values_hook    */	
    XtInheritSetValuesAlmost,		/* set_values_almost  */
    NULL,				/* get_values_hook    */	
    NULL,				/* accept_focus       */
    XtVersion,				/* version            */	
    NULL,				/* callback_private   */
    NULL,				/* tm_table           */
    NULL,				/* query_geometry     */	
    NULL,				/* display_accelerator*/
    NULL,				/* extension          */
  },
  {
					/* composite_class fields */
    GeometryManager,			/* geometry_manager    */
    ChangeManaged,			/* change_managed      */
    XtInheritInsertChild,		/* insert_child        */	
    XtInheritDeleteChild,		/* delete_child        */	
    NULL,				/* extension           */
  },
  { 
					/* constraint_class fields */
   latticeConstraintResources,		/* subresources        */
   XtNumber(latticeConstraintResources),/* subresource_count   */
   sizeof(LatticeConstraintsRec),	/* constraint_size     */
   ConstraintInitialize,		/* initialize          */
   NULL,				/* destroy             */
   ConstraintSetValues,			/* set_values          */
   NULL,				/* extension           */
   },
  {
					/* Lattice class fields */
    0,					/* ignore              */	
  }
};

WidgetClass QxtlatticeWidgetClass = (WidgetClass)&QxtlatticeClassRec;

/*----------------------------------------------------------------
 * lattice utility routines 
 *----------------------------------------------------------------*/

static GC get_gc(w)
QxtLatticeWidget w;
{
    XtGCMask valuemask = GCBackground | GCForeground | GCCapStyle;
    XGCValues values;

    values.background = w->core.background_pixel;
    values.foreground = w->lattice.line_pixel;
    values.cap_style  = CapProjecting;
    if (w->lattice.line_width != 0) {
	valuemask |= GCLineWidth;
	values.line_width = w->lattice.line_width;
    }

    return XtGetGC ((Widget) w, valuemask, &values);    
}

static void insert_node (pindex, parent, node)
     Widget parent, node;
{
    LatticeConstraints pc;
    LatticeConstraints lc = LATTICE_CONSTRAINT(node);
    int nindex;
  
    if (pindex >= lc->lattice.n_parents) {
	lc->lattice.parents = (WidgetList)
	    XtRealloc(lc->lattice.parents, sizeof(Widget) * (pindex+1));
    }
    lc->lattice.parents[pindex] = parent;

    if (parent == NULL) return;

    /*
     * If there isn't more room in the children array, 
     * allocate additional space.
     */  
    pc = LATTICE_CONSTRAINT(parent);
    nindex = pc->lattice.n_children;
  
    if (pc->lattice.n_children == pc->lattice.max_children) {
	pc->lattice.max_children += (pc->lattice.max_children / 2) + 2;
	pc->lattice.children = 
	    (WidgetList) XtRealloc ((char *)pc->lattice.children, 
				    (unsigned int)
				    ((pc->lattice.max_children) *
				     sizeof(Widget)));
    } 

    /*
     * Add the sub_node in the next available slot and 
     * increment the counter.
     */
    pc->lattice.children[nindex] = node;
    pc->lattice.n_children++;
}

static void delete_node (parent, node)
    Widget parent, node;
{
    LatticeConstraints pc;
    int pos, i;

    /*
     * Make sure the parent exists.
     */
    if (!parent) return;  
  
    pc = LATTICE_CONSTRAINT(parent);

    /*
     * Find the sub_node on its parent's list.
     */
    for (pos = 0; pos < pc->lattice.n_children; pos++)
      if (pc->lattice.children[pos] == node) break;

    if (pos == pc->lattice.n_children) return;

    /*
     * Decrement the number of children
     */  
    pc->lattice.n_children--;

    /*
     * Fill in the gap left by the sub_node.
     * Zero the last slot for good luck.
     */
    for (i = pos; i < pc->lattice.n_children; i++) 
      pc->lattice.children[i] = pc->lattice.children[i+1];

    pc->lattice.children[pc->lattice.n_children]=0;
}

static void check_gravity (lw, grav)
QxtLatticeWidget lw;
XtGravity grav;
{
    switch (lw->lattice.gravity) {
      case WestGravity: case NorthGravity: case EastGravity: case SouthGravity:
	break;
      default:
	lw->lattice.gravity = grav;
	break;
    }
}

/*----------------------------------------------------------------
 * lattice class methods 
 *----------------------------------------------------------------*/

static void ClassInitialize ()
{
    XawInitializeWidgetSet();
    XtAddConverter (XtRString, XtRGravity, XmuCvtStringToGravity,
		    (XtConvertArgList) NULL, (Cardinal) 0);
}

static void Initialize(grequest, gnew)
Widget grequest, gnew;
{
    QxtLatticeWidget request = (QxtLatticeWidget) grequest, 
    		  new = (QxtLatticeWidget) gnew;

    /*
     * Make sure the widget's width and height are 
     * greater than zero.
     */
    if (request->core.width <= 0) new->core.width = 5;
    if (request->core.height <= 0) new->core.height = 5;

    /*
     * Set the padding according to the orientation
     */
    if (request->lattice.hpad == 0 && request->lattice.vpad == 0) {
	new->lattice.hpad = LATTICE_HORIZONTAL_DEFAULT_SPACING;
	new->lattice.vpad = LATTICE_VERTICAL_DEFAULT_SPACING;
    }

    /*
     * Create a graphics context for the connecting lines.
     */
    new->lattice.gc = get_gc (new);

    /*
     * Create the hidden top and bottom widget.
     */
    new->lattice.lattice_root = (Widget) NULL;
    new->lattice.lattice_root = XtVaCreateWidget ("root", widgetClass, gnew, 
					    XtNwidth, 1, 
					    XtNheight, 1, 
					    XtNlabel, "root",
					    NULL);
    /*
     * Allocate the tables used by the layout algorithm
     */
    new->lattice.horizontal = create_offset(10);
    new->lattice.vertical = create_offset(10);  
  
    /*
     * make sure that our gravity is one of the acceptable values
     */
    check_gravity (new, NorthGravity);
}

static void ConstraintInitialize (request, new)
Widget request, new;
{
    LatticeConstraints lc = LATTICE_CONSTRAINT(new);
    QxtLatticeWidget lw = (QxtLatticeWidget)new->core.parent;
    register int i;

    /* 
     * Initialize the widget to have no subnodes.
     */
    lc->lattice.n_parents = 0;
    lc->lattice.parents = NULL;
    lc->lattice.n_children = lc->lattice.max_children = 0;
    lc->lattice.children = (WidgetList)NULL;
    lc->lattice.x = lc->lattice.y = 0;

    /*
     * If this widget has a parent_list, add it to that 
     * widget' child list. Otherwise make it a child of 
     * the lattice_root widget.
     */
    for (i = 0; i < lc->lattice.n_parents; i++) { 
	insert_node (i, lc->lattice.parents[i], new);
    }
    if (i == 0 && lw->lattice.lattice_root) {
	insert_node (i, lw->lattice.lattice_root, new);
	lc->lattice.n_parents = 1;
    }
}

static Boolean SetValues (gcurrent, grequest, gnew)
Widget gcurrent, grequest, gnew;
{
    QxtLatticeWidget current = (QxtLatticeWidget)gcurrent, 
    		  new = (QxtLatticeWidget)gnew;
    Boolean redraw = FALSE;

    /*
     * If the foreground color has changed, redo the GC's
     * and indicate a redraw.
     */
    if (new->lattice.line_pixel != current->lattice.line_pixel ||
	new->core.background_pixel != current->core.background_pixel ||
	new->lattice.line_width != current->lattice.line_width) {
	XtReleaseGC (gnew, new->lattice.gc);
	new->lattice.gc = get_gc (new);
	redraw = TRUE;     
    }
    /*
     * If the minimum spacing has changed, recalculate the
     * lattice layout. layout_lattice() does a redraw, so we don't
     * need SetValues to do another one.
     */
    if (new->lattice.gravity != current->lattice.gravity) {
	check_gravity (new, current->lattice.gravity);
    }

    if (IsVertical(new) != IsVertical(current)) {
	if (new->lattice.vpad == current->lattice.vpad &&
	    new->lattice.hpad == current->lattice.hpad) {
	    new->lattice.vpad = current->lattice.hpad;
	    new->lattice.hpad = current->lattice.vpad;
	}
    }

    if (new->lattice.vpad != current->lattice.vpad ||
	new->lattice.hpad != current->lattice.hpad) {
	layout_lattice (new);
	redraw = FALSE;
    }
    return redraw;
}

static Boolean ConstraintSetValues (current, request, new, args, num_args)
Widget current, request, new;
ArgList args;
Cardinal *num_args;
{
    LatticeConstraints newc = LATTICE_CONSTRAINT(new);
    LatticeConstraints curc = LATTICE_CONSTRAINT(current);
    QxtLatticeWidget lw = (QxtLatticeWidget) new->core.parent;
    Boolean layout = FALSE;
    register int i;

    if (curc->lattice.n_parents != newc->lattice.n_parents ||
	curc->lattice.parents != newc->lattice.parents) {
	if (curc->lattice.parents) {
	    for (i = 0; i < curc->lattice.n_parents; i++) {
		delete_node(curc->lattice.parents[i], new);
	    }
	    XtFree(curc->lattice.parents);
	}
	for (i = 0; i < newc->lattice.n_parents; i++) {
	    insert_node(i, newc->lattice.parents[i], new);
	}
	layout = TRUE;
    }
    /*
     * If the Lattice widget has been realized, 
     * compute new layout.
     */
    if (layout && XtIsRealized((Widget)lw))
	layout_lattice(lw);
    return False;
}

/* ARGSUSED */
static XtGeometryResult GeometryManager (w, request, reply)
Widget w;
XtWidgetGeometry *request;
XtWidgetGeometry *reply;
{

    QxtLatticeWidget lw = (QxtLatticeWidget) w->core.parent;

    /*
     * No position changes allowed!.
     */
    if ((request->request_mode & CWX && request->x!=w->core.x)
	||(request->request_mode & CWY && request->y!=w->core.y))
      return (XtGeometryNo);

    /*
     * Allow all resize requests.
     */

    if (request->request_mode & CWWidth)
      w->core.width = request->width;
    if (request->request_mode & CWHeight)
      w->core.height = request->height;
    if (request->request_mode & CWBorderWidth)
      w->core.border_width = request->border_width;

    /*
     * Compute the new layout based on the new widget sizes.
     */
    layout_lattice (lw);
    return (XtGeometryYes);
}

static void ChangeManaged (gw)
Widget gw;
{
    layout_lattice ((QxtLatticeWidget) gw);
}

static void Destroy (gw)
Widget gw;
{
    QxtLatticeWidget w = (QxtLatticeWidget) gw;
    XtReleaseGC (gw, w->lattice.gc);
}

static void Redisplay (lw, event, region)
QxtLatticeWidget lw;
XEvent *event;
Region region;
{
    register int i, j, srcx, srcy;

    /*
     * If the Lattice widget is visible, visit each managed child.
     */
    if (lw->core.visible) {
	for (i = 0; i < lw->composite.num_children; i++) {
	    Widget child = lw->composite.children[i];
	    LatticeConstraints lc = LATTICE_CONSTRAINT(child);

	    if (child != lw->lattice.lattice_root && 
		XtIsManaged(child) && lc->lattice.n_children)
	    {
		switch (lw->lattice.gravity) {
		  case NorthGravity:
		    srcx = child->core.x + child->core.width/2 
			+ child->core.border_width;
		    srcy = child->core.y + child->core.height
			+ child->core.border_width*2;
		    break;
		  case SouthGravity:
		    srcx = child->core.x + child->core.width/2 
			+ child->core.border_width;
		    srcy = child->core.y;
		    break;
		  case WestGravity:
		    srcx = child->core.x + child->core.width
			+ child->core.border_width*2;
		    srcy = child->core.y + child->core.height/2
			+ child->core.border_width;
		    break;
		  case EastGravity:
		    srcx = child->core.x;
		    srcy = child->core.y + child->core.height/2
			+ child->core.border_width;
		    break;
		}

		for (j = 0; j < lc->lattice.n_children; j++) {
		    register Widget k = lc->lattice.children[j];
		    LatticeConstraints klc = LATTICE_CONSTRAINT(k);
		    XPoint points[3];
		    register int npoint = 0;

		    if (!XtIsManaged(k))
			continue;
		    set_point(&points[npoint], srcx, srcy); npoint++;

		    switch (lw->lattice.gravity) {
		      case NorthGravity:
/*			if (k->core.y - srcy > lw->lattice.vpad) {	*/
			if (klc->lattice.level > lc->lattice.level + 1) {
			    set_point(&points[npoint], 
				      srcx, k->core.y-lw->lattice.vpad+1); 
			    npoint++;
			}
			set_point(&points[npoint], 
				  k->core.x + k->core.border_width
				     + k->core.width/2,
				  k->core.y); 
			npoint++;
			break;
		      case SouthGravity:
			if (srcy - (k->core.y + k->core.height
				    + k->core.border_width*2)
			    > lw->lattice.vpad) {
/*			if (klc->lattice.level > lc->lattice.level + 1) { */
			    set_point(&points[npoint], 
				      srcx, 
				      k->core.y + k->core.height 
				       + k->core.border_width*2
				       + lw->lattice.vpad+1); 
			    npoint++;
			}
			set_point(&points[npoint], 
				  k->core.x + k->core.border_width
				     + k->core.width/2,
				  k->core.y + k->core.height 
				     + k->core.border_width*2); 
			npoint++;
			break;
		      case WestGravity:
/*			if (k->core.x - srcx > lw->lattice.hpad) {	*/
			if (klc->lattice.level > lc->lattice.level + 1) {
			    set_point(&points[npoint], 
				      k->core.x-lw->lattice.hpad+1, srcy); 
			    npoint++;
			}
			set_point(&points[npoint], 
				  k->core.x,
				  k->core.y + k->core.border_width
				     + k->core.height/2);
			npoint++;
			break;
		      case EastGravity:
			if (srcx - (k->core.x + k->core.width 
					+ k->core.border_width*2) 
			    > lw->lattice.hpad) {	
/*			if (klc->lattice.level > lc->lattice.level + 1) {*/
			    set_point(&points[npoint], 
				      k->core.x + k->core.width 
					+ k->core.border_width*2
					+ lw->lattice.hpad+1, srcy); 
			    npoint++;
			}
			set_point(&points[npoint], 
				  k->core.x + k->core.width 
				     + k->core.border_width*2,
				  k->core.y + k->core.border_width 
				     + k->core.height / 2);
			npoint++;
			break;
		    }
		    XDrawLines(XtDisplay(lw), XtWindow(lw), lw->lattice.gc,
			       points, npoint, CoordModeOrigin);
		}
	    }
	}
    }
}

/* lattice layout algorithm */

static void layout_lattice(lw)
QxtLatticeWidget lw;
{
    sort_list(lw);

    /*
     * reset the positions;
     */
    reset_positions(lw);
    reset_offset(lw->lattice.vertical);
    reset_offset(lw->lattice.horizontal);

    /*
     * compute each widget's x, y position.
     */
    compute_positions(lw, lw->lattice.lattice_root, 0);

    /*
     * Move each widget into place.
     */
    set_positions(lw, lw->lattice.lattice_root, 0);
    if (IsMirror(lw))
	mirror_positions(lw);

    /*
     * Redisplay the lines connecting nodes.
     */
    if (XtIsRealized(lw)) {
	XClearWindow(XtDisplay(lw), XtWindow(lw));
	Redisplay(lw, NULL, NULL);
    }
}

static int comp_parents(w1, w2)
Widget *w1, *w2;
{
    return((int)*w1 - (int)*w2);
}

static int comp_children(w1, w2)
Widget *w1, *w2;
{
    LatticeConstraints lc1 = LATTICE_CONSTRAINT(*w1);
    LatticeConstraints lc2 = LATTICE_CONSTRAINT(*w2);

    if (lc1->lattice.n_children != lc2->lattice.n_children)
	return(lc1->lattice.n_children - lc2->lattice.n_children);
    return((int)*w1 - (int)*w2);
}

static int sort_list(lw)
QxtLatticeWidget lw;
{
    int i;

    for (i = 0; i < lw->composite.num_children; i++) {
	Widget w = lw->composite.children[i];
	LatticeConstraints lc = LATTICE_CONSTRAINT(w);

	qsort((char *)lc->lattice.parents, lc->lattice.n_parents,
	      sizeof(Widget), comp_parents);
	qsort((char *)lc->lattice.children, lc->lattice.n_children,
	      sizeof(Widget), comp_children);
    }
}

static int compute_positions(lw, w, level)
QxtLatticeWidget lw;
Widget w;
int level;
{
    Position cur_hpos, cur_vpos;
    int i, depth = 0;
    LatticeConstraints lc = LATTICE_CONSTRAINT(w);
    Boolean isvert = IsVertical(lw);

    /*
     * set level
     */
    lc->lattice.level = MAX(lc->lattice.level, level);

    /*
     * get the current positions for this level.
     */
    cur_hpos = current_position(lw->lattice.horizontal, level);
    cur_vpos = current_position(lw->lattice.vertical, level);

    /* 
     * set the current vertical height to the max heights of all widgets
     * at this level.
     */
    if (isvert) 
	set_current_position(lw->lattice.vertical, level,
			     MAX(cur_vpos, w->core.height));
    else {
	if (XtIsManaged(w))
	    set_current_position(lw->lattice.horizontal, level,
				 MAX(cur_hpos, w->core.width));
	else
	    set_current_position(lw->lattice.horizontal, level, cur_hpos);
    }

    /*
     * if the node has no children, just as the horizontal position
     * to the next available space.
     */
    if (lc->lattice.n_children == 0) {
	if (isvert)
	    lc->lattice.x = cur_hpos;
	else
	    lc->lattice.y = cur_vpos;
    } else if (IsLattice(lw, w)) {
	Widget child = lc->lattice.children[0];
	LatticeConstraints clc = LATTICE_CONSTRAINT(child),
			   plc = LATTICE_CONSTRAINT(clc->lattice.parents[0]);
	Position left, right;
	Dimension offset;

	if (isvert)
 	    lc->lattice.x = cur_hpos;	
	else
	    lc->lattice.y = cur_vpos;	

	if (lc->lattice.level+1 < clc->lattice.level) {
	    for (i = lc->lattice.level+1; i < clc->lattice.level; i++) {
		if (isvert) {
		    lc->lattice.x = 
			MAX(lc->lattice.x, 
			    current_position(lw->lattice.horizontal,i));
		} else {
		    lc->lattice.y = 
			MAX(lc->lattice.y, 
			    current_position(lw->lattice.vertical,i));
		}
	    }
	}

	if (isvert) {
	    left = plc->lattice.x + (clc->lattice.parents[0])->core.width/2;
	    right = lc->lattice.x + w->core.width/2;
	    offset = ((left + right)/2 - child->core.width/2) - clc->lattice.x;
	} else {
	    left = plc->lattice.y + (clc->lattice.parents[0])->core.height/2;
	    right = lc->lattice.y + w->core.height/2;
	    offset = ((left + right)/2 - child->core.height/2)-clc->lattice.y;
	}

	shift_subtree(lw, child, offset, level+1);

	/*
	 * adjust the next available space at all levels below 
	 * the current level
	 */
	depth = get_depth(child, level+1);
	for (i = level/* + 1*/; i <= depth; i++) {
	    if (isvert) {
		set_current_position(lw->lattice.horizontal, i, 
			     lc->lattice.x+w->core.width+lw->lattice.hpad);
	    } else {
		set_current_position(lw->lattice.vertical, i, 
			     lc->lattice.y+w->core.height+lw->lattice.vpad);
	    }
	}
    } else {
	Widget first_kid, last_kid;
	LatticeConstraints const1, const2;
	Position left, right;

	/*
	 * if the node has sub_nodes, recursively figure the positions
	 * of each child.
	 */
	for (i = 0; i < lc->lattice.n_children; i++) {
	    depth = compute_positions(lw, lc->lattice.children[i],level+1);
	}
	/*
	 * Note that the horizontal positions of all children are known,
	 * find the horizontal extent of all children.
	 */
	first_kid = lc->lattice.children[0];
	last_kid = lc->lattice.children[lc->lattice.n_children-1];
	const1 = LATTICE_CONSTRAINT(first_kid);
	const2 = LATTICE_CONSTRAINT(last_kid);

	if (isvert) {
	    left = const1->lattice.x + first_kid->core.width/2;
	    right = const2->lattice.x + last_kid->core.width/2;
	    /* 
	     * set the node's position to the center of its children. 
	     */
	    lc->lattice.x = (left + right)/2 - (w->core.width/2);
	} else {
	    left = const1->lattice.y + first_kid->core.height/2;
	    right = const2->lattice.y + last_kid->core.height/2;
	    /* 
	     * set the node's position to the center of its children. 
	     */
	    lc->lattice.y = (left + right)/2 - (w->core.height/2);
	}

	/*
	 * if this position is less than the next available position, 
	 * current it to be the next available position, calculate 
	 * the amount by which all sub_nodes must be shifted, and shift 
	 * the entire sub_tree.
	 */
	if (isvert) {
	    if (lc->lattice.x < cur_hpos) {
		Dimension offset = cur_hpos - lc->lattice.x;

		for (i = 0; i < lc->lattice.n_children; i++) {
		    shift_subtree(lw,lc->lattice.children[i],offset,level+1);
		}

		/*
		 * adjust the next available space at all levels below 
		 * the current level
		 */
		for (i = level+1; i <= depth; i++) {
		    Position pos = current_position(lw->lattice.horizontal, i);
		    set_current_position(lw->lattice.horizontal,i, pos+offset);
		}
		lc->lattice.x = cur_hpos;
	    }
	} else {
	    if (lc->lattice.y < cur_vpos) {
		Dimension offset = cur_vpos - lc->lattice.y;

		for (i = 0; i < lc->lattice.n_children; i++) {
		    shift_subtree(lw, lc->lattice.children[i], offset, level+1);
		}

		/*
		 * adjust the next available space at all levels below 
		 * the current level
		 */
		for (i = level+1; i <= depth; i++) {
		    Position pos = current_position(lw->lattice.vertical, i);
		    set_current_position(lw->lattice.vertical,i, pos+offset);
		}
		lc->lattice.y = cur_vpos;
	    }
	}
    }
    /*
     * record the current horizontal position at this level.
     */
    if (isvert)
	set_current_position(lw->lattice.horizontal, level, 
		lw->lattice.hpad + lc->lattice.x + w->core.width);
    else
	set_current_position(lw->lattice.vertical, level, 
		lw->lattice.vpad + lc->lattice.y + w->core.height);
    return(MAX(depth, level));
}

static void shift_subtree(lw, w, offset, level)
QxtLatticeWidget lw;
Widget w;
int offset;
int level;
{
    register int i;
    LatticeConstraints lc = LATTICE_CONSTRAINT(w);
    Boolean isvert = IsVertical(lw);

    /*
     * set level
     */
    lc->lattice.level = MAX(lc->lattice.level, level);

    /*
     * shift the node by the offset.
     */
    if (isvert)
	lc->lattice.x += offset;
    else
	lc->lattice.y += offset;

    /*
     * shift each subnode into place.
     */
    for (i = 0; i < lc->lattice.n_children; i++) {
	shift_subtree(lw, lc->lattice.children[i], offset, level+1);
    }
}

static void set_positions(lw, w, level)
QxtLatticeWidget lw;
Widget w;
int level;
{
    register int i;
    Dimension	rw = 0, rh = 0;
    Position	pos = 0;
    XtGeometryResult result;
    LatticeConstraints lc = LATTICE_CONSTRAINT(w);
    Boolean isvert = IsVertical(lw);
    static int top;

    lw->lattice.width = lw->core.width;
    lw->lattice.height = lw->core.height;

    if (w) {
	if (isvert) {
	    if (lc->lattice.level == 1) 		/* TOP */
		top = lc->lattice.x + w->core.width/2;
	    if (lc->lattice.n_children == 0 && lc->lattice.y < 0) /* BOTTOM */
		lc->lattice.x = top - w->core.width/2;
	    /* 
	     * Add up the sum of the height's of all nodes to this depth,
	     * and use it as the y position.
	     */
	    pos = (level * lw->lattice.vpad) 
	    	+ sum_of_positions(lw->lattice.vertical, level);
	    lc->lattice.y = MAX(pos, lc->lattice.y);
	} else {
	    if (lc->lattice.level == 1) 		/* TOP */
		top = lc->lattice.y + w->core.height/2;
	    if (lc->lattice.n_children == 0 && lc->lattice.x < 0) /* BOTTOM */
		lc->lattice.y = top - w->core.height/2;
	    /* 
	     * Add up the sum of the height's of all nodes to this depth,
	     * and use it as the y position.
	     */
	    pos = (level * lw->lattice.hpad) 
	    	+ sum_of_positions(lw->lattice.horizontal, level);
	    lc->lattice.x = MAX(pos, lc->lattice.x);
	}

	/* 
	 * move the widget into position.
	 */
	XtMoveWidget(w, lc->lattice.x, lc->lattice.y);

	/*
	 * if the widget position plus its width or height doesn't fit in
	 * the lattice, ask if the lattice can be resized.
	 */
	if (lw->core.width < lc->lattice.x+w->core.width+lw->lattice.hpad ||
	    lw->core.height < lc->lattice.y+w->core.height+lw->lattice.vpad) {
	    result = XtMakeResizeRequest(lw, 
		MAX(lw->core.width, 
		    lc->lattice.x + w->core.width + lw->lattice.hpad),
		MAX(lw->core.height,
		    lc->lattice.y + w->core.height + lw->lattice.hpad),
		&rw, &rh);
	    /*
	     * accept any comprmise.
	     */
	    if (result == XtGeometryAlmost)
		XtMakeResizeRequest(lw, rw, rh, NULL, NULL);

	    lw->lattice.width = rw;
	    lw->lattice.width = rh;
	}
	/* 
	 * set the positions of all children.
	 */
	for (i = 0; i < lc->lattice.n_children; i++) {
	    set_positions(lw, lc->lattice.children[i], level+1);
	}
    }
}

static void mirror_positions(lw)
QxtLatticeWidget lw;
{
    register int i;

    for (i = 0; i < lw->composite.num_children; i++) {
	Widget w = lw->composite.children[i];
	LatticeConstraints lc = LATTICE_CONSTRAINT(w);

	if (w != lw->lattice.lattice_root && XtIsManaged(w) &&
	    lc->lattice.level > 0) {
	    switch (lw->lattice.gravity) {
	      case EastGravity:
		lc->lattice.x = (((Position) lw->lattice.width) -
			      ((Position) w->core.width) - lc->lattice.x);
		break;

	      case SouthGravity:
		lc->lattice.y = (((Position) lw->lattice.height) -
			      ((Position) w->core.height) - lc->lattice.y);
		break;
	    }
	    /*
	     * Move the widget into position.
	     */
	    XtMoveWidget (w, lc->lattice.x, lc->lattice.y);
	}
    }
}

static LatticeOffsetPtr create_offset(size)
int size;
{
    LatticeOffsetPtr offset=(LatticeOffsetPtr)XtMalloc(sizeof(LatticeOffset));
    offset->size = size;
    offset->array = (Dimension *)XtMalloc(size * sizeof(Dimension));
    return(offset);
}

static void
reset_positions(lw)
QxtLatticeWidget lw;
{
    register int i;

    for (i = 0; i < lw->composite.num_children; i++) {
	Widget w = lw->composite.children[i];
	LatticeConstraints lc = LATTICE_CONSTRAINT(w);

	lc->lattice.x = lc->lattice.y = -1;
	lc->lattice.level = 0;
    }
}

static void reset_offset(offset)
LatticeOffsetPtr offset;
{
    register int i;
    for (i = 0; i < offset->size; i++) {
	offset->array[i] = 0;
    }
}

static Position current_position(offset, position)
LatticeOffsetPtr offset;
int position;
{
    if (position >= offset->size)
	return(0);
    return(offset->array[position]);
}

static void set_current_position(offset, index, value)
LatticeOffsetPtr offset;
int index;
Dimension value;
{
    if (index >= offset->size) {
	offset->size = index + index/2;
	offset->array = (Dimension *)XtRealloc(offset->array, 
					       offset->size*sizeof(Dimension));
    }
    offset->array[index] = value;
}

static Position sum_of_positions(offset, index)
LatticeOffsetPtr offset;
int index;
{
    register int i;
    Position sum = 0;
    int stop = index;

    if (index > offset->size)
	stop = offset->size;
    for (i = 0; i < stop; i++) {
	sum += offset->array[i];
    }
    return(sum);
}

IsLattice(lw, w)
QxtLatticeWidget lw;
Widget w;
{
    LatticeConstraints lc = LATTICE_CONSTRAINT(w);
    Boolean isvert = IsVertical(lw);

    if (lc->lattice.n_children == 1) {
	LatticeConstraints clc = LATTICE_CONSTRAINT(lc->lattice.children[0]);
	if (isvert) {
	    if (clc->lattice.n_parents > 1 && clc->lattice.x >= 0)
		return(1);
	} else {
	    if (clc->lattice.n_parents > 1 && clc->lattice.y >= 0)
		return(1);
	}
    }
    return(0);
}

static void
set_point(p, x, y)
XPoint *p;
short x, y;
{
    p->x = x;
    p->y = y;
}

static int
get_depth(w, level)
Widget w;
int level;
{
    int i, depth = 0;
    LatticeConstraints lc = LATTICE_CONSTRAINT(w);

    for (i = 0; i < lc->lattice.n_children; i++) {
	depth = get_depth(lc->lattice.children[i],level+1);
    }
    return(MAX(level,depth));
}
