(defun qxt-interaction-mode ()
  "Major mode for interaction to Quixote.
Quixote $B$H$N%$%s%?%i%/%7%g%s$N<g%b!<%I$G$9!#(B

$B8=:_(B, $B;H$($k%3%^%s%I$O(B M-q $B$N(B qxt-command $B$H(B
C-j $B$N(B qxt-execute-this-line $B$G$9!#(BM-q $B$O(B M-x qxt- $B$HBG$D$N$,(B
$B$a$s$I$/$5$$;~$KJXMx$G$9!#(BM-q $B$O(B session $B$r46CN$7$F(B, $B$=$N;~$K(B
$B;H$($k%3%^%s%I$@$1$rA*Br$G$-$k$h$&$K$7$^$9!#(B

\\{qxt-interaction-mode-map}
"
  (interactive)
  (setq major-mode 'qxt-interaction-mode)
  (setq mode-name "Quixote Interaction")
  (use-local-map qxt-interaction-mode-map)
  (setq mode-line-format
	(list ""
	      'mode-line-modified
	      'mode-line-buffer-identification
	      "   "
	      'global-mode-string
	      "   %[("
	      'mode-name 'minor-mode-alist "%n" '(qxt-client-state "(Running)")
	      ")%]----"
	      '(-3 . "%p")
	      "-%-"))
  (run-hooks 'qxt-interaction-mode-hook))

(defvar qxt-interaction-mode-map nil
  "Keymap for Quixote Interaction mode")

(if qxt-interaction-mode-map
    ()
  (setq qxt-interaction-mode-map (make-sparse-keymap))
  (define-key qxt-interaction-mode-map "\t"  'qxt-interact-complete)
  (define-key qxt-interaction-mode-map "\eq"  'qxt-command)
  (define-key qxt-interaction-mode-map "\C-j" 'qxt-change-input-state)
  (define-key qxt-interaction-mode-map "\C-m" 'qxt-execute-this-line)
  )

(defun qxt-valid-for-current-session (entry)
  (and entry
       (or (eq (cdr entry) qxt-current-session)
	   (eq (cdr entry) 'meta))))

(defun qxt-interact-complete ()
  "$B%3%s%W%j!<%7%g%s$G$9!#$&$^$/%3%s%W%j!<%7%g%s$,=PMh$?$i(B t $B$rJV$9!#(B"
  (interactive)
  (if qxt-input-state
      (qxt-complete-for qxt-lex-keywords 'qxt-exclude-nil "\\s-")
    (qxt-complete-for qxt-interact-command-table
		      'qxt-valid-for-current-session "^.*> ")))

(defun qxt-exclude-nil (entry)
  (not (string= (car entry) "nil")))

(defvar qxt-complete-did-something-complete nil)

(defun qxt-complete-for (table predicate from-where-regexp)
  (let* ((line
	  (buffer-substring
	   (save-excursion
	     (re-search-backward from-where-regexp)
	     (match-end 0))
	   (point)))
	 (completion
	  (try-completion line table predicate)))
    (cond ((eq completion t)
	   (message "[Sole Completion]")
	   (setq qxt-complete-did-something-complete nil)
	   t)
	  ((null completion)
	   (ding)
	   (message "[No match]")
	   (setq qxt-complete-did-something-complete nil)
	   nil)
	  ((string= completion line)
	   (let* ((completion-list
		   (all-completions line table predicate))
		  (entry
		    (assoc line table))
		  (exact-match
		   (if predicate
		       (funcall predicate entry)
		     entry)))
	     (if (> (length completion-list) 1)
		 (if exact-match
		     (progn
		       (if (and (eq last-command 'qxt-interact-complete)
				(not qxt-complete-did-something-complete))
			   (with-output-to-temp-buffer " *Completion*"
			     (display-completion-list
			      (sort completion-list 'string<)))
			 (message "[Complete, but not unique]"))
		       (setq qxt-complete-did-something-complete nil)
		       t)
		   (with-output-to-temp-buffer " *Completion*"
		     (display-completion-list
		      (sort completion-list 'string<)))
		   (setq qxt-complete-did-something-complete nil)
		   nil))))
	  (t
	   (delete-region
	    (save-excursion
	      (re-search-backward from-where-regexp)
	      (match-end 0)) (point))
	   (insert completion)
	   (setq qxt-complete-did-something-complete t)
	   nil))))

(defun qxt-command (prefix)
  "Read Quixote function name, then read its arguments and call it.
Many say that it is quite unconfortable to type M-x qxt-...
So, I present you this command.  How do you feel?

Should support prefix argument.  Not yet supported."
  (interactive "P")
  (let* ((command-name-sans-qxt
	  (completing-read "M-x qxt-" qxt-interact-command-table
			   'qxt-valid-for-current-session t nil)))
    (qxt-command-execute command-name-sans-qxt)))

(defun qxt-command-execute (command-name-sans-qxt)
  (let* ((command-name
	  (concat "qxt-" command-name-sans-qxt))
	 (command (intern command-name))
	 (qxt-meta-command nil))
    (if (fboundp command)
	(progn
	  (command-execute command t)
	  (let ((command-and-args (car command-history)))
	    ;; delete "qxt-"
	    (insert (substring (format "%s" (car command-and-args)) 4))
	    (setq command-and-args (cdr command-and-args))
	    (while command-and-args
	      (insert " ")
	      (insert (prin1-to-string (car command-and-args)))
	      (setq command-and-args (cdr command-and-args))))
	  (insert "\n")
	  (if qxt-meta-command ;; dynamic binding
	      (qxt-insert-prompt qxt-current-session)))
      (ding)
      (message "No such command"))))

(defun qxt-query-this-text (&optional insert-here)
  (interactive)
  (let ((end (point-marker))
	start
	line)
    (re-search-backward "^?-")
    (setq start (point))
    (narrow-to-region start end)
    (replace-regexp "^> " "")
    (setq line (buffer-substring start end))
    (widen)
    (if insert-here
	(if (= (marker-position end) (point-max))
	    (goto-char (point-max))
	  (goto-char (point-max))
	  (setq start (point))
	  (delete-region (point)
			 (save-excursion
			   (forward-line 0)
			   (point)))
	  (insert line)))
    (set-marker end nil) ; free the marker
    (qxt-query-string line)
    (insert "\n")))

(defun qxt-create-database-this-text (&optional insert-here)
  (interactive)
  (let ((end (point-marker))
	start
	line)
    (re-search-backward "^&program")
    (setq start (point))
    (narrow-to-region start end)
    (replace-regexp "^> " "")
    (setq line (buffer-substring start end))
    (widen)
    (if insert-here
	(if (= (marker-position end) (point-max))
	    (goto-char (point-max))
	  (goto-char (point-max))
	  (setq start (point))
	  (delete-region (point)
			 (save-excursion
			   (forward-line 0)
			   (point)))
	  (setq start (point))
	  (insert line)))
    (set-marker end nil) ; free the marker
  (let* ((text line)
	 (database-name (read-string "Database Name: "))
	 program-text program
	 (index 0)
	 (len  (length text))
	 text-of-one-line)
      (while (and (< index len)
		  (string-match "\n" text index))
	(setq text-of-one-line (substring text index (match-beginning 0)))
	(setq program-text (cons text-of-one-line program-text))
	(setq index (match-end 0)))
      (setq program-text (reverse program-text))
      (condition-case err
	  (setq program 
		(if qxt-qmacs-i-mode
		    (qxt-qxt2src-I text)
		  (qxt-qxt2src-D text)))
	(syntax (goto-char start)
		(forward-char qxt-lex-pos)
		(error (car (cdr err))))
	(lexical (goto-char start)
		 (forward-char (1- qxt-lex-pos))
		 (error (car (cdr err)))))
      (qxt-timer-start)
      (qxt-send-command
       (qxt-lisp2term-string
	(vector 'create_database
		(intern database-name) program-text program)))
      (setq qxt-input-state nil))
    (insert "\n")))

(defvar qxt-input-state nil
  "Valid input state is one of:
PROGRAM-BEGIN(program)
QUERY-BEGIN(query)
PROGRAM(in middle of program)
QUERY(in middle of query)
COMMAND(nil)
")

(defun qxt-change-input-state ()
  (interactive)
  (cond ((eq qxt-input-state 'program-begin)
	 (if (<
	      (save-excursion
	       (re-search-backward "^&program;;")
	       (match-end 0))
	      (point))
	     (if (y-or-n-p "discard input?")
		 (progn
		   (message "input discarded")
		   (setq qxt-input-state nil)
		   (insert "\n")
		   (qxt-insert-prompt qxt-current-session))
	       (ding))
	   (setq qxt-input-state nil)
	   (insert "\n")
	   (qxt-insert-prompt qxt-current-session)))
	((eq qxt-input-state 'query-begin)
	 (if (<
	      (save-excursion
	       (re-search-backward "^\\?-")
	       (match-end 0))
	      (point))
	     (if (y-or-n-p "discard input?")
		 (progn
		   (message "input discarded")
		   (setq qxt-input-state nil)
		   (insert "\n")
		   (qxt-insert-prompt qxt-current-session))
	       (ding))
	   (setq qxt-input-state nil)
	   (insert "\n")
	   (qxt-insert-prompt qxt-current-session)))
	((or (eq qxt-input-state 'program)
	     (eq qxt-input-state 'query))
	 (if (y-or-n-p "discard input?")
	     (progn
	       (message "input discarded")
	       (setq qxt-input-state nil)
	       (insert "\n")
	       (qxt-insert-prompt qxt-current-session))
	   (ding)))
	(t
	 (cond ((eq qxt-current-session 'server)
		(setq qxt-input-state 'program-begin)
		(insert "\n")
		(insert "&program;;"))
	       ((eq qxt-current-session 'database)
		(setq qxt-input-state 'query-begin)
		(insert "\n")
		(insert "?-"))
	       (t
		(error "Not supported in current session"))))))

(defun qxt-execute-this-line ()
  (interactive)
  (cond ((eq qxt-input-state 'program-begin)
	 (if (eq (preceding-char) ?.)
	     (qxt-create-database-this-text t)
	   (if (/= (point-max) (point))
	       (let ((end (point))
		     line)
		 (forward-line 0)
		 (setq line (buffer-substring start end))
		 (goto-char (point-max))
		 (setq end (point))
		 (forward-line 0)
		 (delete-region (point) end)
		 (insert line)))
	   (setq qxt-input-state 'program)
	   (insert "\n")
	   (qxt-insert-prompt "")))
	((eq qxt-input-state 'query-begin)
	 (if (eq (preceding-char) ?.)
	     (qxt-query-this-text t)
	   (if (/= (point-max) (point))
	       (let ((end (point))
		     line)
		 (forward-line 0)
		 (setq line (buffer-substring start end))
		 (goto-char (point-max))
		 (setq end (point))
		 (forward-line 0)
		 (delete-region (point) end)
		 (insert line)))
	   (setq qxt-input-state 'query)
	   (insert "\n")
	   (qxt-insert-prompt "")))
	((eq qxt-input-state 'program)
	 (if (/= (point-max) (point))
	     (let ((end (point))
		   line)
	       (forward-line 0)
	       (setq line (buffer-substring start end))
	       (goto-char (point-max))
	       (insert line)))
	 (if (eq (preceding-char) ?.)
	     (qxt-create-database-this-text nil)
	   (insert "\n")
	   (qxt-insert-prompt "")))
	((eq qxt-input-state 'query)
	 (if (/= (point-max) (point))
	     (let ((end (point))
		   line)
	       (forward-line 0)
	       (setq line (buffer-substring start end))
	       (goto-char (point-max))
	       (insert line)))
	 (if (eq (preceding-char) ?.)
	     (qxt-query-this-text nil)
	   (insert "\n")
	   (qxt-insert-prompt "")))
	(t
	 (let ((end (point))
	       line)
	   (re-search-backward "^.*> ")
	   (goto-char (match-end 0))
	   (if (eq (point) end)
	       (progn ; null command
		 (insert "\n")
		 (qxt-insert-prompt qxt-current-session))
	     (setq line (buffer-substring (point) end))
	     (if (eq end (point-max))
		 (kill-line nil)
	       (goto-char (point-max)))
	     (if (string-match "[ \t]" line)
		 (let ((expression (concat "(qxt-" line ")")))
		   (insert (concat line "\n"))
		   (let ((exp (read expression)))
		     (condition-case err
			 (eval exp)
		       (error
			(let ((end (point)))
			  (re-search-backward "^.*> ")
			  (goto-char (match-end 0))
			  (delete-region (point) end)
			  (message (format "No such command: %s" line))
			  (ding))))))
	       (qxt-command-execute line)))))))

(defvar qxt-ss-mode-map nil
  "Keymap for Quixote Server session")

(defvar qxt-db-mode-map nil
  "Keymap for Quixote Database session")

(if qxt-ss-mode-map
    ()
  (setq qxt-ss-mode-map (make-sparse-keymap))
  (define-key qxt-ss-mode-map "\C-c\C-s" 'qxt-session-start)
  (define-key qxt-ss-mode-map "\C-c\C-z" 'qxt-quit)
  (define-key qxt-ss-mode-map "\C-c\C-c" 'qxt-close-database)
  (define-key qxt-ss-mode-map "\C-c\C-q" 'qxt-query)
  (define-key qxt-ss-mode-map "\C-c\C-r" 'qxt-create-database))

(defvar qxt-syntax-table nil
  "Syntax table for Quixote Program or Query.")

(if qxt-syntax-table
    ()
  (setq qxt-syntax-table (copy-syntax-table (standard-syntax-table)))
  (modify-syntax-entry ?$ "." qxt-syntax-table)
  (modify-syntax-entry ?% "<" qxt-syntax-table)
  (modify-syntax-entry ?& "_" qxt-syntax-table)
  (modify-syntax-entry ?* "." qxt-syntax-table)
  (modify-syntax-entry ?+ "." qxt-syntax-table)
  (modify-syntax-entry ?- "." qxt-syntax-table)
  (modify-syntax-entry ?/ "." qxt-syntax-table)
  (modify-syntax-entry ?< "." qxt-syntax-table)
  (modify-syntax-entry ?> "." qxt-syntax-table)
  (modify-syntax-entry ?\\ "." qxt-syntax-table)
  (modify-syntax-entry ?| "." qxt-syntax-table)
  (modify-syntax-entry ?\M-\C-? " " qxt-syntax-table)
  (let ((i 128))
    (while (<= i ?\M-\ )
      (modify-syntax-entry i " " qxt-syntax-table)
      (setq i (1+ i)))
    (while (<= i 254)
      (modify-syntax-entry i "w" qxt-syntax-table)
      (setq i (1+ i)))))

;; C-@ .. SPC        	which means: whitespace
;; DEL               	which means: whitespace
;; !               . 	which means: punctuation
;; "               " 	which means: string
;; #               . 	which means: punctuation
;; $               .    which means: punctuation
;; %               <    which means: comment starter
;; &               _ 	which means: symbol
;; '               . 	which means: punctuation
;; (               ()	which means: open, matches )
;; )               )(	which means: close, matches (
;; * .. +          . 	which means: punctuation
;; ,               . 	which means: punctuation
;; -               . 	which means: punctuation
;; .               . 	which means: punctuation
;; /               . 	which means: punctuation
;; 0 .. 9          w 	which means: word
;; : .. ;          . 	which means: punctuation
;; < .. >          . 	which means: punctuation
;; ? .. @          . 	which means: punctuation
;; A .. Z          w 	which means: word
;; [               (]	which means: open, matches ]
;; \               \ 	which means: punctuation
;; ]               )[	which means: close, matches [
;; ^               . 	which means: punctuation
;; _               _ 	which means: symbol
;; `               . 	which means: punctuation
;; a .. z          w 	which means: word
;; {               (}	which means: open, matches }
;; |               . 	which means: punctuation
;; }               ){	which means: close, matches {
;; ~               . 	which means: punctuation
;; M-C-@ .. M-SPC    	which means: whitespace
;; M-! .. M-~      w 	which means: word
;; M-DEL             	which means: whitespace
