;;
;; query
;;	C: query(Query)
;;		Query = Query form of SRC
;;	S: result_query(Status, Answer)
;;		Answer = Answer form of SRC
;;
(defun qxt-query (start end)
  "start$B$H(Bend$B$GI=$9%j!<%8%g%sFb$N<ALd$rAw?.$9$k!#(B"
  (interactive "r")
  (let* ((text (buffer-substring start end)))
    (qxt-query-string text)))

(defun qxt-query-string (text)
  "$BJ8;zNs(Btext$BCf$N<ALd$rAw?.$9$k!#(B"
  (qxt-check-status 'database)
  (setq qxt-input-string text)
  (setq qxt-input-string-len (length text))
  (setq qxt-lex-pos 0)
  (setq qxt-line-no 1)
  (let* (query)
    (advance)
    (condition-case err
	(setq query (query))
      (syntax (goto-char start)
	      (forward-char qxt-lex-pos)
	      (error (car (cdr err)))))
    (qxt-timer-start)
    (qxt-send-command
     (qxt-lisp2term-string
      (vector 'query query)))))

(defun qxt-query-file (FileName)
  "$B%U%!%$%kL>$,(BFileName$B$N%U%!%$%k$NFbMF$N<ALd$rAw?.$9$k!#(B
FileName = String"
  (interactive "fFileName: ")
  (save-excursion
    (find-file FileName)
    (qxt-query (point-min) (point-max))))

;;
(defun qxt-result-query(Result)
  "$B<ALd$N<B9T7k2L$G$"$k(BAnswer$B$rI=<($9$k!#(B
Result = result_query(Status, AnswerOrError)
        AnswerOrError = Answer | UserError
        Answer    = Answer form of SRC
        UserError = runtime_errors(Term) | runtime_errors(Atom,Term)
"
  (let ((Answer (aref Result 2)))
    (setq qxt-current-session 'database)
    (qxt-timer-end)
    (if (not (eq qxt-conv-mode 'conv))
	(progn
	  (insert "** Raw Data from Server **\n")
	  (insert (qxt-lisp2term-string Answer) "\n"))
      ;; normal mode
      (if (eq 'answer (aref Answer 0))
	  (let ((AnswerElements (aref Answer 1))
		AnswerElement
		(Count 1))
	    (if (eq qxt-window-mode t)
		(qxt-answer-window (qxt-src2str-answer Answer)))
	    (if (eq AnswerElements nil)
		(insert " NO\n")
	      (if (= (length AnswerElements) 1)
		  (insert (concat "** 1 answer exists **\n"))
		(insert (concat "** "
				(length AnswerElements)
				" answers exist **\n")))
	      (while AnswerElements
		(setq AnswerElement (car AnswerElements))
		(insert (concat "** Answer " Count " **\n"))
		(qxt-display-answer AnswerElement)
		(setq AnswerElements (cdr AnswerElements))
		(setq Count (+ 1 Count))))
	    (setq AnswerElements (aref Answer 1))
	    (if (boundp 'qxt-answer-hook)
		;; use dynamic binding
		(run-hooks 'qxt-answer-hook)))
	;; runtime_errors
	(insert (format "** runtime error **\n%s\n" Answer))))))
;;
(defun qxt-display-answer (AnswerElement)
  "AnswerElement$B$rI=<($9$k!#(B
AnswerElement = answer_element(DotCnstrs VarCnstrs, ExpRules)
"
  (let ((DotCnstrs (aref AnswerElement 1)) DotCnstr
	(VarCnstrs (aref AnswerElement 2)) VarCnstr
	(ExpRules (aref AnswerElement 3)))
    (cond
     ((and (eq DotCnstrs nil) (eq VarCnstrs nil))
      (insert " YES\n"))
     ((and (eq DotCnstrs nil) (not (eq VarCnstrs nil)))
      (qxt-display-VarCnstrs VarCnstrs "    "))
     ((and (not (eq DotCnstrs nil)) (eq VarCnstrs nil))
      (progn
	(qxt-display-DotCnstrs DotCnstrs)
	(insert " THEN YES\n")))
     (t
      (progn
	(qxt-display-DotCnstrs DotCnstrs)
	(insert " THEN\n")
	(qxt-display-VarCnstrs VarCnstrs "    ")))) ;; end of cond
    (qxt-display-ExpRules ExpRules)))

;;
(defun qxt-display-DotCnstrs(DotCnstrs)
  "DotCnstrs$B$rI=<($9$k!#(B
DotConstrs = [DotCnstr, ...]
"
  (let (DotCnstr)
    (setq DotCnstr (car DotCnstrs))
    (setq DotCnstrs (cdr DotCnstrs))
    (insert (concat " IF " (qxt-lisp2qxt DotCnstr)))
    (while (setq DotCnstr (car DotCnstrs))
      (insert (concat "    " (qxt-lisp2qxt DotCnstr)))
      (newline)
      (setq DotCnstrs (cdr DotCnstrs)))))

;;
(defun qxt-display-VarCnstrs(VarCnstrs Indent)
  "VarCnstrs$B$r(BIndent$B$K%"%Z%s%I$7$FI=<($9$k!#(B
VarCnstrs = [VarCnstr, ...]
"
  (let (VarCnstr)
    (insert Indent)
    (while (setq VarCnstr (car VarCnstrs))
      (insert (qxt-lisp2qxt VarCnstr))
      (setq VarCnstrs (cdr VarCnstrs))
      (if VarCnstrs
	  (insert ", ")
	(newline)))))
;;
(defun qxt-display-ExpRules(ExpRules)
  "Explanation$B$rI=<($9$k!#(B
ExpRule = {Explanation,[{Rule_id,Rule},...]}  | &void 
"
  (if (eq ExpRules '&void) ()
    (let ((Explanation (aref ExpRules 0))
	  (UsingRules  (aref ExpRules 1)))
      (insert "** Explanation **\n")
      (if (eq qxt-conv-mode 'conv)
	  (insert (qxt-lisp2term-string (qxt-src2str-explanation Explanation)))
	(insert (qxt-lisp2term-string Explanation)))
      (newline)
      (if (eq UsingRules nil)
	  ()
	(insert "** Using Rules **\n")
	(if (eq qxt-conv-mode 'conv)
	    (insert (qxt-lisp2term-string 
		     (qxt-src2str-ruleid-rule-pairs UsingRules)))
	  (insert (qxt-lisp2term-string UsingRules)))
	(newline)))))
;;
(defun qxt-answer-window (answer)
  "X$B%&%$%s%I%&$K(B ANSWER $B$rI=<($9$k!#(B
xanswer$B$r5/F0$7!"%G!<%?JQ49$7$?(B ANSWER $B$rI8=`F~NO$KAw$k!#(B
"
  (let* ((answer-string (qxt-lisp2term-string answer))
	 ;; use pipe for channel
	 (process-connection-type nil)
	 (proc (qxt-stuff-start-process "xanswer")))
    (set-process-filter proc 'qxt-process-message-filter)
    (process-send-string proc answer-string)
    (process-send-string proc "\n\004\n")))

;;
;; close_database  
;;	C: close_database(End)
;;        	End = '&end'|'&abort'
;;	S: result_close_database(Status)
;;
(defun qxt-close-database (End)
  "$B%G!<%?%Y!<%9$N%/%m!<%:$r%Q%i%a!<%?(BEnd$B$GAw?.$9$k!#(B
End = \"end\" | \"abort\""
  (interactive 
   (list (completing-read "CloseMode: " '( ("end" 1) ("abort" 2) ) nil t )))
  (qxt-check-status 'database)
  (qxt-timer-start)
  (if (string= End "")
      (setq End "\'&end\'")
    (setq End (concat "\'&" End "\'")))
  (qxt-send-command (concat "{close_database," End "}")))

;;
(defun qxt-result-close-database(Result)
  "$B%G!<%?%Y!<%9$N%/%m!<%:$r<u?.8e$N=hM}$r9T$J$&!#(B"
  (qxt-timer-end)
  (setq qxt-current-session 'server))

;;
;; begin_transaction
;;	C: begin_transaction
;;	S: result_begin_transaction(Status)
;;
(defun qxt-begin-transaction()
  "$B%H%i%s%6%/%7%g%s$N3+;O$rAw?.$9$k!#(B"
  (interactive)
  (qxt-check-status 'database)
  (qxt-send-command "begin_transaction" ))

;;
;; end_transaction
;;	C: end_transaction
;;	S: result_end_transaction(Status)
;;
(defun qxt-end-transaction()
  "$B%H%i%s%6%/%7%g%s$N=*N;$rAw?.$9$k!#(B"
  (interactive)
  (qxt-check-status 'database)
  (qxt-send-command "end_transaction"))

;;
;; abort_transaction
;;	C: abort_transaction
;;	S: result_abort_transaction(Status)
;;
(defun qxt-abort-transaction()
  "$B%H%i%s%6%/%7%g%s$N%"%\!<%H$rAw?.$9$k!#(B"
  (interactive)
  (qxt-check-status 'database)
  (qxt-send-command "abort_transaction"))

;;
;; show_module
;;	C: show_module
;;	S: result_show_module(Status, ModuleIdPairs)
;;		ModulIdPairs = [ModuleIdPair,  ...]
;;		ModuleIdPair = {ModuleId,ModuleId}
;;		ModuleId = Module_id form of SRC
;;
(defun qxt-show-module()
  "$B%b%8%e!<%k4X78$N<hF@$rAw?.$9$k!#(B"
  (interactive)
  (qxt-check-status 'database)
  (qxt-send-command "show_module"))

;;
(defun qxt-result-show-module(Result)
  "$B%b%8%e!<%k4X78$rI=<($9$k!#(B"
  (let ((ModuleIdPairs (aref Result 2))
	ModuleIdPair ModuleId ModuleDesc)
    (if (eq qxt-window-mode nil)
	(progn
	  (insert "** Module **\n")
	  (if (eq ModuleIdPairs nil)
	      ()
	    (insert "[")
	    (setq ModuleIdPair (car ModuleIdPairs))
	    (setq ModuleId1    (aref ModuleIdPair 0))
	    (setq ModuleId2    (aref ModuleIdPair 1))
	    (insert "{")
	    (insert (qxt-lisp2qxt ModuleId1))
	    (insert ", ")
	    (insert (qxt-lisp2qxt ModuleId2))
	    (insert "}")
	    (setq ModuleIdPairs (cdr ModuleIdPairs))
	    (while  ModuleIdPairs
	      (setq ModuleIdPair (car ModuleIdPairs))
	      (setq ModuleId1    (aref ModuleIdPair 0))
	      (setq ModuleId2    (aref ModuleIdPair 1))
	      (insert ", {")
	      (insert (qxt-lisp2qxt ModuleId1))
	      (insert ", ")
	      (insert (qxt-lisp2qxt ModuleId2))
	      (insert "}")
	      (setq ModuleIdPairs (cdr ModuleIdPairs)))
	    (insert "]\n")))
      (let* ((xdata (qxt-lisp2term-string (qxt-change-module-hierarchy
					   (vector ModuleIdPairs nil))))
	     ;; use pipe for channel
	     (process-connection-type nil)
	     (proc (qxt-stuff-start-process "xmodule")))
	(set-process-filter proc 'qxt-process-message-filter)
	(process-send-string proc xdata)
	(process-send-string proc "\n\004\n")))))


;;
;; show_module_nodes
;;	C: show_module_nodes(ModuleId, Neighbour)
;;		Neighbour = Number | {=, Number}
;;		Number = Integer | + Interger | - Interger
;;	S: result_show_module_nodes(Status, ModuleNodes)
;;		ModuleNodes = [ModuleNode, ...]
;;		ModuleNode = ModuleId | ModuleDesc
;;		ModuleDesc = M_desc form of SRC
;;
(defun qxt-show-module-nodes(ModuleId Neighbour)
  "$B%b%8%e!<%k(BModuleId$B$N(BNeighbour$B$NHO0O$N%N!<%I%G!<%?$N<hF@$rAw?.$9$k!#(B
ModuleId = String
Neighbour = String 
"
  (interactive "sModuleId: \nsNeighbour: ")
  (qxt-check-status 'database)
  (qxt-send-command (concat "{show_module_nodes," 
			    (qxt-oterm2src ModuleId) "," Neighbour "}")))

;;
(defun qxt-result-show-module-nodes(Result)
  "Result$B$N%b%8%e!<%k%N!<%I%G!<%?$rI=<($9$k!#(B"
  (let ((ModuleNodes (aref Result 2)))
    (insert "** Module node **\n")
    (if (eq ModuleNodes nil) ()
      (insert "[")
      (insert (qxt-lisp2qxt (car ModuleNodes)))
      (setq ModuleNodes (cdr ModuleNodes))
      (while  ModuleNodes
	(insert ", ")
	(insert (qxt-lisp2qxt (car ModuleNodes)))
	(setq ModuleNodes (cdr ModuleNodes)))
      (insert "]\n"))))

;;
;; show_lattice  
;;	C: show_lattice
;;	S: result_show_lattice(Status, Lattice)
;;		Lattice = [BasicObjectPair, ...]
;;		BasicObjectPair = {BasicObject, BasicObjectList}
;;                	%               small -- large 
;;	        BasicObjectList=[BasicObject,....]
;;		BasicObject = Atom
;;
(defun qxt-show-lattice()
  "$B%i%F%#%9$N<hF@$rAw?.$9$k!#(B"
  (interactive)
  (qxt-check-status 'database)
  (qxt-send-command "show_lattice" ))

;;
(defun qxt-result-show-lattice(Result)
  "$B%i%F%#%9$rI=<($9$k4X?t$G$"$k!#(B
$B%&%$%s%I%&%b!<%I$,(Buse$B$N>l9g!"(Bxlattice$B$r5/F0$7!"(B
X$B%&%$%s%I%&>e$KI=<($9$k!#(B
"
  (let ((Lattice (aref Result 2)))
    (if (eq qxt-window-mode nil)
	(progn
	  (insert "** Lattice ** \n")
	  (insert (qxt-lisp2term-string Lattice) "\n"))
      (let* ((lattice (qxt-lisp2term-string Lattice))
	     ;; use pipe for channel
	     (process-connection-type nil) 
	     (proc (qxt-stuff-start-process "xlattice")))
	(set-process-filter proc 'qxt-process-message-filter)
	(process-send-string proc lattice)
	(process-send-string proc "\n\004\n")))))

(defun qxt-process-message-filter(p s)
  (message s))

;;
;; show_lattice_nodes
;;	C: show_lattice_nodes(BasicNode, Neighbour)
;;	S: result_show_lattice_nodes(Status, LatticeNodes)
;;		LatticeNodes = [BasicObject, ...]
;;
(defun qxt-show-lattice-nodes(BasicNode Neighbour)
  "BasicNode$B$N(BNeighbour$B$NHO0O$N%i%F%#%9%N!<%I$N<hF@$rAw?.$9$k!#(B
BasicNode = String
Neighbour = String 
"
  (interactive "sBasicNode: \nsNeighbour: ")
  (progn
    (qxt-check-status 'database)
    (qxt-send-command
     (concat "{show_lattice_nodes," BasicNode "," Neighbour "}"))))

;;
(defun qxt-result-show-lattice-nodes(Result)
  "Result$B$N%i%F%#%9%N!<%I$rI=<($9$k!#(B"
  (let ((LatticeNodes (aref Result 2)))
    (insert "** Lattice node ** \n")
    (insert (qxt-lisp2term-string LatticeNodes) "\n")))

;;
;; compress_set
;;	C: compress_set(Set, CompressOption)
;;		Set = [BasicObject, ...]
;;		Oterm = Oterm form of SRC
;;		Compress_option = minimal | maximal
;;	S: result_compress_set(Status, CompressSet)
;;		CompressSet =[BasicObject, ...]
;;
(defun qxt-compress-set(Start End CompressOption)
  "Start$B$H(BEnd$B$N%j!<%8%g%s$K0O$^$l$?%*%V%8%'%/%H%j%9%H$N%3%s%W%l%9%;%C%H$N<hF@$r(B
CompressOption$BIU$-$GAw?.$9$k!#(B
Start = Integer (region Start point)
End   = Integer (region End point)
CompressOption = \"minimal\" | \"maximal\"
"
  (interactive
   (let (Strart End CompressOption InputCompressOption)
     (setq Start (region-beginning))
     (setq End (region-end))
     (setq InputCompressOption (completing-read "CompressOption: " 
		  '( ("minimal" 1) ("maximal" 2)) nil t ))
     (if (string= InputCompressOption "")
	 (setq CompressOption "minimal")
       (setq CompressOption InputCompressOption))
     (list Start End CompressOption)
     ))
  (let ((Set (buffer-substring Start End)))
    (qxt-check-status 'database)
    (let ((index 0)
	  (len (length Set)))
      (while (and (< index len)
		  (string-match "^.*$" Set index))
	(if (> len (match-end 0))  ;; last new line
	    (aset Set (match-end 0) ? )
	  (aset Set (- len 1) ? ))
	(setq index (1+ (match-end 0)))))
    (qxt-send-command (concat "{compress_set," Set "," CompressOption "}"))))

;;
(defun qxt-result-compress-set(Result)
  "Result$B$N%3%s%W%l%9%;%C%H$rI=<($9$k!#(B"
  (let ((CompressSet (aref Result 2)))
    (insert "** Compress set ** \n")
    (insert (qxt-lisp2term-string CompressSet) "\n")))

;;
;; delete_database
;;	C: delete_database
;;	S: result_delete_database(Status)
;;
(defun qxt-delete-database()
  "$B%G!<%?%Y!<%9$N:o=|$rAw?.$9$k!#(B"
  (interactive)
  (progn
    (qxt-check-status 'database)
    (qxt-timer-start)
    (qxt-send-command "delete_database" )))

(defun qxt-result-delete-database(Result)
  "$B%G!<%?%Y!<%9$N:o=|$r<u?.8e$N=hM}$r$9$k!#(B"
  (qxt-timer-end)
  (setq qxt-current-session 'server))
;;
;; show_rules
;;	C: show_rules(ModuleId)
;;	S: result_show_rules(Status, Rules)
;;		Rules = [Rule,...]
;;		Rule = Rule form of SRC
;;
(defun qxt-show-rules(ModuleId)
  "ModuleId$B$N%k!<%k$N<hF@$rAw?.$9$k!#(B
ModuleId = String"
  (interactive "sModuleID: ")
  (qxt-check-status 'database)
  (qxt-send-command
   (concat "{show_rules," (qxt-oterm2src ModuleId) "}" )))

;;
(defun qxt-result-show-rules (Result)
  "Result$B$N%k!<%k$rI=<($9$k!#(B"
  (let ((Rules (aref Result 2)))
    (insert "** Rules **\n")
    (while Rules
      (insert (qxt-lisp2qxt (car Rules)))
      (newline)
      (setq Rules (cdr Rules)))))
;;
;;	C: show_normalize_rules(ModuleId)
;;	S: result_show_normalize_rules(Status, Rules)
;;		Rules = [Rule,...]
;;		Rule = Rule form of SRC
;;
(defun qxt-show-normalize-rules(ModuleId)
  "ModuleId$B$N%N!<%^%i%$%:%k!<%k$N<hF@$rAw?.$9$k!#(B
ModuleId = String
"
  (interactive "sModuleId: ")
  (progn
    (qxt-check-status 'database)
    (qxt-send-command
     (concat "{show_normalize_rules," (qxt-oterm2src ModuleId) "}" ))))

;;
(defun qxt-result-show-normalize-rules(Result)
  "Result$B$N%N!<%^%i%$%:%k!<%k$rI=<($9$k!#(B"
  (let ((Rules (aref Result 2)))
    (insert "** Normalize Rules **\n")
    (while Rules
      (insert (qxt-lisp2qxt (car Rules)))
      (newline)
      (setq Rules (cdr Rules)))))

;;
;; get_id_rule
;;	C: get_id_rule(RuleIds)
;;		RuleIds = [RuleId, ...]
;;		RuleId = String
;;	S: result_get_id_rule(Status, Rules)
;;
(defun qxt-get-id-rule(Start End)
  "Start$B$H(BEnd$B$N%j!<%8%g%s$K0O$^$l$?%k!<%k(BID$B%j%9%H$N%k!<%k$N<hF@$rAw?.$9$k!#(B
Start = Integer (region Start point)
End   = Integer (region End point)
"
  (interactive "r")
  (let ((RuleIds (buffer-substring Start End)))
    (qxt-check-status 'database)
    (let ((index 0)
	  (len (length RuleIds))
	  line)
      (while (and (< index len)
		  (string-match "^.*$" RuleIds index))
	(if (> len (match-end 0))  ;; last new line
	    (aset RuleIds (match-end 0) ? )
	  (aset RuleIds (- len 1) ? ))
	(setq index (1+ (match-end 0)))))
     (qxt-send-command (concat "{get_id_rule," RuleIds "}"))))

;;
(defun qxt-result-get-id-rule(Result)
  "Result$B$N%k!<%k%j%9%H$rI=<($9$k!#(B"
  (let ((Rules (aref Result 2)))
    (insert "** Rules **\n")
    (while Rules
      (insert (qxt-lisp2qxt (car Rules)))
      (newline)
      (setq Rules (cdr Rules)))))

;;
;; get_default_mode
;;	C: get_default_mode
;;	S: result_get_default_mode(Status, DefaultModes)
;;		DefaultModes = [DefaultMode, ...]
;;	        DefaultMode_id = proc | answer | inheritance | merge | explanation
;;                DefaultModeValue = '&multi' | '&single'	%proc
;;			|	'&normal' | '&minimal'		%answer
;;			|	'&all' | '&down'|'&up'|'&no'	%inheritance
;;			|	'&yes' | '&no'			%merge
;;			|	'&on' | '&off'			%explanation
;;
(defun qxt-get-default-mode()
  "$B%G%U%)%k%H%b!<%I$N<hF@$rAw?.$9$k!#(B"
  (interactive)
  (progn
    (qxt-check-status 'database)
    (qxt-send-command "get_default_mode")))

;;
(defun qxt-result-get-default-mode(Result)
  "Result$B$N%G%U%)%k%H%b!<%I$rI=<($9$k!#(B"
  (let ((DefaultModes (aref Result 2)))
    (insert "** Default mode **\n")
    (while DefaultModes
      (setq DefaultMode (car DefaultModes))
      (insert (symbol-name (aref DefaultMode 0)))
      (insert " == ")
      (insert (symbol-name (aref DefaultMode 1)))
      (newline)
      (setq DefaultModes (cdr DefaultModes)))))

;;
;; set_default_mode
;;	C: set_default_mode(DefaultModeId, DefaultModeValue)
;;	S: result_set_default_mode(Status)
;;
(defun qxt-set-default-mode(ModeId ModeValue)
  "ModeId$B$H(BModeValue$B$G;XDj$7$?%G%U%)%k%H%b!<%I$N@_Dj$rAw?.$9$k!#(B
ModeId = String 
ModeValue = String
"
  (interactive
   (let (ModeId ModeValue InputModeValue)
     (setq ModeId (completing-read "ModeId: " 
	  '( ("proc" 1) ("answer" 2) ("inheritance" 3) ("merge" 4) ("explanation" 5)) nil t ))
     (setq InputModeValue 
	   (cond
	    ((string= ModeId "proc")
	     (completing-read "ProcMode: " '( ("multi" 1) ("single" 2)) nil  t))
	    ((string= ModeId "answer")
	     (completing-read "AnswerMode: " '( ("normal" 1) ("minimal" 2)) nil  t))
	    ((string= ModeId "inheritance")
	     (completing-read "InheritanceMode: " '( ("all" 1) ("down" 2) ("up" 3) ("no" 4)) nil  t))
	    ((string= ModeId "merge")
	     (completing-read "MergeMode: " '( ("yes" 1) ("no" 2)) nil  t))
	    ((string= ModeId "explanation")
	     (completing-read "ExplanationMode: " '( ("on" 1) ("off" 2)) nil  t))
	    ))
     (setq ModeValue (concat "\'&" InputModeValue "\'"))
     (list ModeId ModeValue))
     )
  (progn
    (qxt-check-status 'database)
    (qxt-send-command (concat "{set_default_mode," ModeId "," ModeValue "}"))))

;;
;; show_basic_objects
;;	C: show_basic_objects(Kind)
;;        	Kind = mid | label | value
;;	S: result_show_basic_objects(Status,BasicObjects)
;;
(defun qxt-show-basic-objects(Kind)
  "Kind$B$G;XDj$7$?<oN`$N4pK\%*%V%8%'%/%H$N<hF@$rAw?.$9$k!#(B
Kind = \"mid\" | \"label\" | \"value\"
"
  (interactive
   (let ( Kind)
     (setq Kind (completing-read "Kind: " 
		  '( ("mid" 1) ("label" 2) ("value" 3) ) nil t ))
     (list Kind) ))
  (progn
    (qxt-check-status 'database)
    (qxt-send-command (concat "{show_basic_objects," Kind "}"))))

;;
(defun qxt-result-show-basic-objects(Result)
  "Result$B$N4pK\%*%V%8%'%/%H$rI=<($9$k!#(B"
  (let ((BasicObjects (aref Result 2)))
    (insert "** Basic object ** \n")
    (insert (qxt-lisp2term-string BasicObjects) "\n")))

;;
;; show_objects
;;	C: show_objects(ModuleId)
;;	S: result_show_objects(Status, Oterms)
;;		Oterms = [Oterm, ...]
;;
(defun qxt-show-objects(ModuleId)
  "ModuleId$B$N%*%V%8%'%/%H$N<hF@$rAw?.$9$k!#(B
ModuleId = String"
  (interactive "sModuleID: ")
  (progn
    (qxt-check-status 'database)
    (qxt-send-command
     (concat "{show_objects," (qxt-oterm2src ModuleId) "}"))))

;;
(defun qxt-result-show-objects(Result)
  "Result$B$N%*%V%8%'%/%H$rI=<($9$k!#(B"
  (let ((Objects (aref Result 2)))
    (insert "** Objects **\n")
    (if (eq Objects nil) ()
      (insert "[")
      (insert (qxt-lisp2qxt (car Objects)))
      (setq Objects (cdr Objects))
      (while Objects
	(insert ", ")
	(insert (qxt-lisp2qxt (car Objects)))
	(setq Objects (cdr Objects)))
      (insert "]\n"))))

;;
;; change_conv_mode
;;	C: change_conv_mode(ConvMode)
;;		Conv_mode = conv_QD | conv | no_conv
;;	S: result_change_conv_mode(Status)
;;
(defvar qxt-conv-mode 'conv)
(defun qxt-change-conv-mode(ConvMode)
  "ConvMode$B$X$NJQ99$rAw?.$9$k!#(B
ConvMode = \"conv_QD\" | \"conv\" | \"no_conv\"
"
  (interactive
   (let ( ConvMode)
     (setq ConvMode (completing-read "ConvMode: " 
	  '( ("conv_QD" 1) ("conv" 2) ("no_conv" 3) ) nil t ))
     (list ConvMode)))
  (progn
    (qxt-check-status 'database)
    (cond ((string= ConvMode "conv_QD") (setq qxt-conv-mode 'conv_QD))
	  ((string= ConvMode "conv") (setq qxt-conv-mode 'conv))
	  ((string= ConvMode "no_conv") (setq qxt-conv-mode 'no_conv)))
    (qxt-send-command (concat "{change_conv_mode," ConvMode "}"))))

;;
;; show_dot_label
;;	C:show_dot_labels(ModuleId,Oterm)
;;                Module_id is form of SRC
;;                Oterm is form of SRC
;;      S:result_show_dot_labels(Status,Labels)
;;		Labels=[Label,....]
;;
(defun qxt-show-dot-labels(ModuleId Oterm)
  "ModuleId$B$N(BOterm$B$N%I%C%H%i%Y%k$N<hF@$rAw?.$9$k!#(B
ModuleId = String
Oterm = String of Quixte source
"
  (interactive "sModuleId: \nsOterm: ")
  (progn
    (qxt-check-status 'database)
    (qxt-send-command
     (concat "{show_dot_labels," (qxt-oterm2src ModuleId) "," 
	     (qxt-oterm2src Oterm) "}"))))

;;
(defun qxt-result-show-dot-labels(Result)
  "Result$B$N%I%C%H%i%Y%k$rI=<($9$k!#(B"
  (let ((Objects (aref Result 2))
	Id Oterm)
    (insert "** Dot Labels **\n")
    (if (eq Objects nil) ()
      (insert "[")
      (setq Id    (aref (car Objects) 0))
      (setq Oterm (aref (car Objects) 1))
      (insert (qxt-lisp2qxt Oterm))
      (setq Objects (cdr Objects))
      (while Objects
	(insert ", ")
	(setq Id    (aref (car Objects) 0))
	(setq Oterm (aref (car Objects) 1))
	(insert (qxt-lisp2qxt Oterm))
	(setq Objects (cdr Objects)))
      (insert "]\n"))))

;;
;; show_module_hierarchy
;;   C: show_module_hierarchy
;;   S: result_show_module_hierarchy(Status, ModuleHierarchyAndModuleRules)
;;        ModuleHierarchyAndModuleRules={ModuleHierarchy, ModuleRules}
;;	  ModuleHierarchy =[ModuleIdPiar, ...]
;;        ModuleRules = [ModuleRule, ...]
;;        ModuleRule = {ModuleId, UserRules, NormalizeRules}
;;        UserRules = Rules
;;        NormalizeRules = Rules
;;
(defun qxt-show-module-hierarchy()
  "$B%b%8%e!<%k%O%$%"%i%-$N<hF@$rAw?.$9$k!#(B"
  (interactive)
  (progn
    (qxt-check-status 'database)
    (qxt-send-command "show_module_hierarchy")))

;;
(defun qxt-result-show-module-hierarchy(Result)
  "Result$B$N%b%8%e!<%k%O%$%"%i%-$rI=<($9$k4X?t$G$"$k!#(B
$B%&%$%s%I%&%b!<%I$,(Buse$B$N>l9g!"(B
xmodule$B$r5/F0$7!"(BX$B%&%$%s%I%&>e$KI=<($9$k!#(B
"
  (let ((HierarchyAndRules (aref Result 2)) XData)
    (setq XData (qxt-change-module-hierarchy HierarchyAndRules))
    (if (eq qxt-window-mode nil)
	(progn
	  (insert "** Module hierarchy ** \n")
	  (insert (qxt-lisp2term-string XData) "\n"))
      (let* ((xdata (qxt-lisp2term-string XData))
	     ;; use pipe for channel
	     (process-connection-type nil)
	     (proc (qxt-stuff-start-process "xmodule")))
	(set-process-filter proc 'qxt-process-message-filter)
	(process-send-string proc xdata)
	(process-send-string proc "\n\004\n")))))

;;
;;   XData = {XHierarchy, XRules}
;;
(defun qxt-change-module-hierarchy (HierarchyAndRules)
  "\.src$B7A<0%G!<%?$G$"$k(BHierarchyAndRules$B$r(B
xmodule$BMQ$N%G!<%?7A<0$KJQ49$9$k4X?t$G$"$k!#(B
"
  (let ((Hierarchy (aref HierarchyAndRules 0))
	(ModuleRules    (aref HierarchyAndRules 1)))
    (vector (qxt-S2X-hierarchy Hierarchy)
	    (qxt-S2X-module-rules  ModuleRules))))

;;
;;	  XHierarchy =[XIdPiar, ...]
;;        XIdPair = {XModuleId, XModuleId}
;;        XModuleId = String
;;
(defun qxt-S2X-hierarchy (Hierarchy)
  (let ((XHierarchy nil)
	IdPair XMid1 XMid2)
    (while Hierarchy
      (setq IdPair (car Hierarchy))
      (setq Hierarchy (cdr Hierarchy))
      (setq XMid1 (qxt-lisp2qxt (aref IdPair 0)))
      (setq XMid2 (qxt-lisp2qxt (aref IdPair 1)))
      (setq XHierarchy (append XHierarchy (list (vector XMid1 XMid2)))))
    (setq XHierarchy XHierarchy)))

;;
;;        XRules = [XRule, ...]
;;
(defun qxt-S2X-module-rules  (Rules)
  (let ((XRules nil))
    (while Rules
      (setq XRule (qxt-S2X-rule (car Rules)))
      (setq XRules (append XRules (list XRule)))
      (setq Rules (cdr Rules)))
    (setq XRules XRules)))

;;
;;        XRule = {XModuleId, XUserRules, XNormalizeRules}
;;        XUserRules = RuleStrings
;;        XNormalizeRules = RuleStrings
;;        RuleStrings = [String, ...]
;;
(defun qxt-S2X-rule (Rule)
  (let ((ModuleId (aref Rule 0))
	(UserRules (aref Rule 1)) 
	(NormalizeRules (aref Rule 2)))
    (vector (qxt-lisp2qxt ModuleId)
	    (qxt-S2X-rule-strings UserRules)
	    (qxt-S2X-rule-strings NormalizeRules))))

;;
(defun qxt-S2X-rule-strings (Rules)
  (let ((RuleStrings nil) RuleString)
    (while Rules
      (setq RuleString (qxt-lisp2qxt (car Rules)))
      (setq Rules (cdr Rules))
      (setq RuleStrings (append RuleStrings (list RuleString))))
    (setq RuleStrings RuleStrings)))

;;
;; tracer commands
;;

;;
;; set_trace_mode
;;	C:set_trace_mode(TraceMode)
;;          TraceMode = notrtace|trace|spy
;;	S:result_set_trace_mode(Status)
;;
(defun qxt-set-trace-mode(TraceMode)
  "TraceMode$B$G%H%l!<%9%b!<%I$N@_Dj$rAw?.$9$k!#(B
TraceMode = \"notrace\" | \"trace\" | \"spy\"
"
  (interactive
   (let (TraceMode)
     (setq TraceMode (completing-read "TraceMode: " 
		  '( ("notrace" 1) ("trace" 2) ("spy" 3)) nil t ))
     (list TraceMode)))
  (qxt-check-status 'database)
  (qxt-send-command (concat "{set_trace_mode," TraceMode "}") ))

;;
;; get_trace_mode
;;	C:get_trace_mode
;;	S:result_get_trace_mode(Status,TraceMode)
;;
(defun qxt-get-trace-mode()
  "$B%H%l!<%9%b!<%I$N<hF@$rAw?.$9$k!#(B"
  (interactive)
  (qxt-check-status 'database)
  (qxt-send-command "get_trace_mode" ))

;;
(defun qxt-result-get-trace-mode(Result)
  "$B%H%l!<%9%b!<%I$rI=<($9$k!#(B"
  (let ((TraceMode (aref Result 2)))
    (insert (concat "** Trace Mode **\n" (symbol-name TraceMode) "\n"))))

;;
;; set_gate
;;	C: set_gate(GateID,Switch)  
;;		GateID = call | exit | fail | lookup
;;			bt | et | at | ic | update
;;		Switch = on | off 
;;	S: result_set_gate(Status)
;;
(defun qxt-set-gate(GateID Switch)  
  "GateId$B$H(BSwitch$B$G%2!<%H$N@_Dj$rAw?.$9$k!#(B
GateID = \"call\" | \"exit\" | \"fail\" | \"lookup\" | 
         \"bt\" | \"et\" | \"at\" | \"ic\" | \"update\"
Switch = \"on\" | \"off\"
"
  (interactive
   (let (GateId Switch)
     (setq GateId (completing-read "GateId: " 
	   '( ("call" 1) ("exit" 2) ("fail" 3) ("lookup" 4) ("bt" 5)
	      ("et" 6) ("at" 7) ("ic" 8) ("update"  9)) nil t ))
     (setq Switch (completing-read "Switch: " 
		  '( ("on" 1) ("off" 2) ) nil t ))
     (list GateId Switch)))
  (qxt-check-status 'database)
  (qxt-send-command (concat "{set_gate," GateID "," Switch "}" )))

;;
;; set_gate_all
;;	C: set_gate_all(Gates)
;;		Gates = {Call, Exit, Fail, Lookup, Bt, Et, At, It, Update}
;;		Call = Switch 
;;		Exit = Switch 
;;		Fail = Switch 
;;		Lookup = Switch
;;		Bt = Switch
;;		Et = Switch
;;		At = Switch
;;		Ic = Switch
;;		Update = Switch
;;	S: result_set_gate_all(Status)
;;
(defun qxt-set-gate-all(Call Exit Fail Lookup Bt Et At Ic Update)
  "Call Exit Fail Lookup Bt Et At Ic Update$B$G$9$Y$F$N%2!<%H$N@_Dj$rAw?.$9$k!#(B
Call Exit Fail Lookup Bt Et At Ic Update = \"on\" | \"off\"
"
  (interactive
   (let (Call Exit Fail Lookup Bt Et At Ic Update)
     (setq Call (completing-read "Call: " 
		  '( ("on" 1) ("off" 2) ) nil t ))
     (if (string= Call "") (setq Call "off"))
     (setq Exit (completing-read "Exit: " 
		  '( ("on" 1) ("off" 2) ) nil t ))
     (if (string= Exit "") (setq Exit "off"))
     (setq Fail (completing-read "Fail: " 
		  '( ("on" 1) ("off" 2) ) nil t ))
     (if (string= Fail "") (setq Fail "off"))
     (setq Lookup (completing-read "Lookup: " 
		  '( ("on" 1) ("off" 2) ) nil t ))
     (if (string= Lookup "") (setq Lookup "off"))
     (setq Bt (completing-read "Bt: " 
		  '( ("on" 1) ("off" 2) ) nil t ))
     (if (string= Bt "") (setq Bt "off"))
     (setq Et (completing-read "Et: " 
		  '( ("on" 1) ("off" 2) ) nil t ))
     (if (string= Et "") (setq Et "off"))
     (setq At (completing-read "At: " 
		  '( ("on" 1) ("off" 2) ) nil t ))
     (if (string= At "") (setq At "off"))
     (setq Ic (completing-read "Ic: " 
		  '( ("on" 1) ("off" 2) ) nil t ))
     (if (string= Ic "") (setq Ic "off"))
     (setq Update (completing-read "Update: "
		  '( ("on" 1) ("off" 2) ) nil t ))
     (if (string= Update "") (setq Update "off"))
     (list Call Exit Fail Lookup Bt Et At Ic Update)))
  (qxt-check-status 'database)
  (qxt-send-command
   (concat "{set_gate_all,{" Call "," Exit "," Fail ","
	   Lookup "," Bt "," Et "," At "," Ic "," Update "}}" )))

;;
;; get_gate
;;	C: get_gate(GateID)  
;;	S: result_get_gate(Status,Switch)
;;
(defun qxt-get-gate(GateID)  
  "GateID$B$G;XDj$7$?%2!<%H$N>uBV$N<hF@$rAw?.$9$k!#(B
GateID = \"call\" | \"exit\" | \"fail\" | \"lookup\" | 
         \"bt\" | \"et\" | \"at\" | \"ic\" | \"update\"
"
  (interactive
   (let (GateId)
     (setq GateId (completing-read "GateId: " 
	   '( ("call" 1) ("exit" 2) ("fail" 3) ("lookup" 4) ("bt" 5)
	      ("et" 6) ("at" 7) ("ic" 8) ("update"  9)) nil t ))
     (list GateId)))
  (qxt-check-status 'database)
  (qxt-send-command (concat "{get_gate," GateID "}" )))

;;
(defun qxt-result-get-gate(Result)
  "Result$B$N%2!<%H$N>uBV$rI=<($9$k!#(B"
  (let ((Switch (aref Result 2)))
    (insert (concat "** gate ** " (symbol-name Switch) "\n"))))

;;
;; get_gate_all
;;	C: get_gate_all
;;	S: result_get_gate_all(Status,Gates)
;;
(defun qxt-get-gate-all()
  "$B$9$Y$F$N%2!<%H$N>uBV$N<hF@$rAw?.$9$k!#(B"
  (interactive)
  (qxt-check-status 'database)
  (qxt-send-command "get_gate_all" ))
;;
;;		Call = Switch 
;;		Exit = Switch 
;;		Fail = Switch 
;;		Lookup = Switch
;;		Bt = Switch
;;		Et = Switch
;;		At = Switch
;;		Ic = Switch
;;		Update = Switch
(defun qxt-result-get-gate-all(Result)
  "$B$9$Y$F$N%2!<%H$N>uBV$rI=<($9$k!#(B"
  (let ((Gates (aref Result 2)))
    (insert "** Gate **\n")
    (insert (concat "Call: " (symbol-name (aref Gates 0))))
    (newline)
    (insert (concat "Exit: " (symbol-name (aref Gates 1))))
    (newline)
    (insert (concat "Fail: " (symbol-name (aref Gates 2))))
    (newline)
    (insert (concat "Lookup: " (symbol-name (aref Gates 3))))
    (newline)
    (insert (concat "Bt: " (symbol-name (aref Gates 4))))
    (newline)
    (insert (concat "Et: " (symbol-name (aref Gates 5))))
    (newline)
    (insert (concat "At: " (symbol-name (aref Gates 6))))
    (newline)
    (insert (concat "Ic: " (symbol-name (aref Gates 7))))
    (newline)
    (insert (concat "Update: " (symbol-name (aref Gates 8)) "\n"))))

;;
;; spy_at_subgoals
;;	C:spy_at_subgoals(SpySubgoals)
;;		SpySubgoals = [SpySubgoal, ... ]
;;		SpySubgoal = {ModuleIdVoid, Oterm}
;;		ModuleIdVoid = ModuleId | '&void'
;;	S:result_spy_at_subgoals(Status)
;;
(defun qxt-spy-at-subgoals (SubgoalVectors)
  "SubgoalVectors$B$G%5%V%4!<%k$KBP$9$k%9%Q%$@_Dj$rAw?.$9$k!#(B
SubgoalVectors = List of (vector ModuleIdString SubgoalString)
[$BNc(B]  (list [\"m1\" \"g1\"] [\"m2\" \"g2\"])
"
  (interactive
   (let ((SubgoalVectors nil)
	 (SubgoalVector  nil))
     (while  (setq SubgoalVector (qxt-read-spy-subgoal))
       (setq SubgoalVectors (append SubgoalVectors (list SubgoalVector))))
     (list SubgoalVectors)))
    (qxt-check-status 'database)
    (qxt-send-command 
     (concat "{spy_at_subgoals,"
	     (qxt-change-spy-subgoal-vectors SubgoalVectors) "}" )))

(defun qxt-read-spy-subgoal ()
  (let ((ModuleId (read-string "ModuleId: "))
	(Oterm    (read-string "Oterm: ")))
    (if (and (string= ModuleId "")
	     (string= Oterm ""))
	()
      (vector ModuleId Oterm))))

;; SubgoalVectors = List of (vector ModuleIdString SubgoalString)
(defun qxt-change-spy-subgoal-vectors (SubgoalVectors)
  (let ((SendingForm "["))
    (if (eq SubgoalVectors nil)
	()
      (setq SendingForm
	    (concat SendingForm
		    (qxt-change-spy-subgoal-vector (car SubgoalVectors))))
      (setq SubgoalVectors (cdr SubgoalVectors))
      (while SubgoalVectors
	(setq SendingForm 
	      (concat SendingForm ","
		      (qxt-change-spy-subgoal-vector (car SubgoalVectors))))
	(setq SubgoalVectors (cdr SubgoalVectors))))
    (concat SendingForm "]")))

(defun qxt-change-spy-subgoal-vector (SubgoalVector)
  (let ((ModuleIdString (aref SubgoalVector 0))
	(SubgoalString  (aref SubgoalVector 1)))
    (concat "{"
	    (if (string= "" ModuleIdString)  "\'&void\'"
	      (qxt-oterm2src ModuleIdString))
	    ","
	    (qxt-oterm2src SubgoalString)
	    "}")))
;;
;; spy_at_rules
;;	C:spy_at_rules(SpyRules)
;;		SpyRules = [SpyRule, ... ]
;;		SpyRule = {RuleID, ModuleIdVoid}
;;	S:result_spy_at_rules(Status)
;;
(defun qxt-spy-at-rules(RuleVectors)
  "RuleVectors$B$G%k!<%k$KBP$9$k%9%Q%$@_Dj$rAw?.$9$k!#(B
RuleVectors = list of (vector RuleIdString ModuleIdString)
[$BNc(B]  (list [\"r1\" \"m1\"] [\"r2\" \"m2\"])
"
  (interactive
   (let ((RuleVectors nil)
	 (RuleVector  nil))
     (while (setq RuleVector (qxt-read-spy-rule))
       (setq RuleVectors (append RuleVectors (list RuleVector))))
     (list RuleVectors)))
    (qxt-check-status 'database)
    (qxt-send-command 
     (concat "{spy_at_rules,"
	     (qxt-change-spy-rule-vectors RuleVectors) "}" )))

(defun qxt-read-spy-rule ()
  (let ((ModuleId (read-string "ModuleId: "))
	(RuleId   (read-string "RuleId: ")))
    (if (and (string= ModuleId "") (string= RuleId "")) nil
      (vector RuleId ModuleId))))

;; RuleVectors = list of (vector RuleIdString ModuleIdString)
(defun qxt-change-spy-rule-vectors (RuleVectors)
  (let ((SendingForm "["))
    (if (eq RuleVectors nil)
	()
      (setq SendingForm 
	    (concat SendingForm
		    (qxt-change-spy-rule-vector (car RuleVectors))))
      (setq RuleVectors (cdr RuleVectors))
      (while RuleVectors
	(setq SendingForm 
	      (concat SendingForm ","
		      (qxt-change-spy-rule-vector (car RuleVectors))))
	(setq RuleVectors (cdr RuleVectors))))
    (concat SendingForm "]")))

(defun qxt-change-spy-rule-vector (RuleVector)
  (let ((RuleIdString    (aref RuleVector 0))
	(ModuleIdString  (aref RuleVector 1)))
    (concat "{"
	    "\"" RuleIdString "\""
	    ","
	    (if (string= "" ModuleIdString)  "\'&void\'"
	      (qxt-oterm2src ModuleIdString))
	    "}")))
;;
;; spy_at_modules
;;	C:spy_at_modules(ModuleIds)
;;		ModuleIds = [ModuleId, ... ]
;;	S:result_spy_at_modules(Status)
;;
(defun qxt-spy-at-modules(ModuleIdStrings)
  "ModuleIdStrings$B$G%b%8%e!<%k$KBP$9$k%9%Q%$@_Dj$rAw?.$9$k!#(B
MouleIdStrings = list of ModuleIdString
[$BNc(B]  (list \"m1\" \"m2\")
"
  (interactive
   (let ((ModuleIdStrings nil)
	 (ModuleIdString  nil))
     (while  (setq ModuleIdString (qxt-read-spy-module-id))
       (setq ModuleIdStrings (append ModuleIdStrings (list ModuleIdString))))
     (list ModuleIdStrings)))
    (qxt-check-status 'database)
    (qxt-send-command 
     (concat "{spy_at_modules,"
	     (qxt-change-spy-module-ids ModuleIdStrings) "}" )))

(defun qxt-read-spy-module-id ()
  (let ((InputString (read-string "ModuleId: ")))
    (if (string= InputString "") nil
      InputString)))

;; MouleIdStrings = list of ModuleIdString
(defun qxt-change-spy-module-ids (ModuleIdStrings)
  (let ((SendingForm "["))
    (if (eq ModuleIdStrings nil) ()
      (setq SendingForm 
	    (concat SendingForm
		    (qxt-change-spy-module-id (car ModuleIdStrings))))
      (setq ModuleIdStrings (cdr ModuleIdStrings))
      (while ModuleIdStrings
	(setq SendingForm 
	      (concat SendingForm ","
		      (qxt-change-spy-module-id (car ModuleIdStrings))))
	(setq ModuleIdStrings (cdr ModuleIdStrings))))
    (concat SendingForm "]")))

(defun qxt-change-spy-module-id (ModuleIdString)
  (if (string= "" ModuleIdString)  "\'&void\'"
    (qxt-oterm2src ModuleIdString)))


;; list_spy
;;	C:list_spy
;;	S:result_list_spy(Status,{SubgoalSpyPoints,RuleSpyPoints,ModuleSpyPoints})
;;		SubgoalSpyPoints = [{{SpySubgoal,SpyInfo},SpyPointId}, ... ]
;;		RuleSpyPoints = [{{SpyRule,SpyInfo},SpyPointId}, ... ]
;;		ModuleSpyPoints = [{{ModuleId,SpyInfo},SpyPointId}, ... ]
;;		SpyInfo = enable | disable
;;		SpyPoint = Integer
;;
(defun qxt-list-spy()
  "$B8=:_@_Dj$7$F$$$k%9%Q%$$N<hF@$rAw?.$9$k!#(B"
  (interactive)
  (qxt-check-status 'database)
  (qxt-send-command "list_spy" ))

;;
(defun qxt-result-list-spy (Result)
  "$B8=:_@_Dj$7$F$$$k%9%Q%$$rI=<($9$k!#(B"
  (let ((SubgoalSpyPoints  (aref (aref Result 2) 0))
	(RuleSpyPoints     (aref (aref Result 2) 1))
	(ModuleSpyPoints   (aref (aref Result 2) 2)))
    (insert "** Subgoal spy point **\n")
    (while SubgoalSpyPoints
      (let ((ModuleId (aref (aref (aref (car SubgoalSpyPoints) 0) 0) 0))
	    (Oterm    (aref (aref (aref (car SubgoalSpyPoints) 0) 0) 1))
	    (SpyInfo        (aref (aref (car SubgoalSpyPoints) 0) 1))
	    (SpyPointId           (aref (car SubgoalSpyPoints) 1)))
	(setq SubgoalSpyPoints (cdr SubgoalSpyPoints))
	(insert (concat "(Id:" SpyPointId ", " (symbol-name SpyInfo) ") "
			(qxt-lisp2qxt ModuleId) ":" (qxt-lisp2qxt Oterm)))
	(newline)))
    (insert "** Rule spy point **\n")
    (while RuleSpyPoints
      (let ((RuleId    (aref (aref (aref (car RuleSpyPoints) 0) 0) 0))
	    (ModuleId  (aref (aref (aref (car RuleSpyPoints) 0) 0) 1))
	    (SpyInfo         (aref (aref (car RuleSpyPoints) 0) 1))
	    (SpyPointId            (aref (car RuleSpyPoints) 1)))
	(setq RuleSpyPoints (cdr RuleSpyPoints))
	(insert (concat "(Id:" SpyPointId ", " (symbol-name SpyInfo) ") "
			(qxt-lisp2qxt ModuleId) ":" RuleId))
	(newline)))
    (insert "** Module spy point **\n")
    (while ModuleSpyPoints
      (let ((ModuleId  (aref (aref (car ModuleSpyPoints) 0) 0))
	    (SpyInfo   (aref (aref (car ModuleSpyPoints) 0) 1))
	    (SpyPointId      (aref (car ModuleSpyPoints) 1)))
	(newline)
	(setq ModuleSpyPoints (cdr ModuleSpyPoints))
	(insert (concat "(Id:" SpyPointId ", " (symbol-name SpyInfo) ") "
			(qxt-lisp2qxt ModuleId) "\n"))))))

;;
;; unspy_at_subgoals
;;	C:unspy_at_subgoals(SpySubgoals)
;;	S:result_unspy_at_subgoals(Status)
;;
(defun qxt-unspy-at-subgoals(SubgoalVectors)
  "SubgoalVectors$B$G%5%V%4!<%k$KBP$9$k%9%Q%$$N2r=|$rAw?.$9$k!#(B
SubgoalVectors = List of (vector ModuleIdString SubgoalString)
[$BNc(B]  (list [\"m1\" \"g1\"] [\"m2\" \"g2\"])
"
  (interactive
   (let ((SubgoalVectors nil)
	 (SubgoalVector  nil))
     (while  (setq SubgoalVector (qxt-read-spy-subgoal))
       (setq SubgoalVectors (append SubgoalVectors (list SubgoalVector))))
     (list SubgoalVectors)))
    (qxt-check-status 'database)
    (qxt-send-command 
     (concat "{unspy_at_subgoals,"
	     (qxt-change-spy-subgoal-vectors SubgoalVectors) "}" )))
;;
;; unspy_at_rules
;;	C:unspy_at_rules(SpyRules)
;;	S:result_unspy_at_rules(Status)
;;
(defun qxt-unspy-at-rules(RuleVectors)
  "RuleVectors$B$G%k!<%k$KBP$9$k%9%Q%$2r=|$rAw?.$9$k!#(B
RuleVectors = list of (vector RuleIdString ModuleIdString)
[$BNc(B]  (list [\"r1\" \"m1\"] [\"r2\" \"m2\"])
"
  (interactive
   (let ((RuleVectors nil)
	 (RuleVector  nil))
     (while  (setq RuleVector (qxt-read-spy-rule))
       (setq RuleVectors (append RuleVectors (list RuleVector))))
     (list RuleVectors)))
    (qxt-check-status 'database)
    (qxt-send-command 
     (concat "{unspy_at_rules,"
	     (qxt-change-spy-rule-vectors RuleVectors) "}" )))

;;
;; unspy_at_modules
;;	C:unspy_at_modules(ModuleIds)
;;	S:result_unspy_at_modules(Status)
;;
(defun qxt-unspy-at-modules(ModuleIdStrings)
  "ModuleVectors$B$G%b%8%e!<%k$KBP$9$k%9%Q%$2r=|$rAw?.$9$k!#(B
MouleIdStrings = list of ModuleIdString
[$BNc(B]  (list \"m1\" \"m2\")
"
  (interactive
   (let ((ModuleIdStrings nil)
	 (ModuleIdString  nil))
     (while  (setq ModuleIdString (qxt-read-spy-module-id))
       (setq ModuleIdStrings (append ModuleIdStrings (list ModuleIdString))))
     (list ModuleIdStrings)))
    (qxt-check-status 'database)
    (qxt-send-command 
     (concat "{unspy_at_modules,"
	     (qxt-change-spy-module-ids ModuleIdStrings) "}" )))

;;
;; unspy_at_spypoints
;;	C:unspy_at_spypoints(SpyPoints)
;;		SpyPoints = [{Indication,SpyPointId}, ... ]
;;		Indication = subgoal | rule | module
;;	S:result_unspy_at_spypoints(Status)
;;
(defun qxt-unspy-at-spypoints(SpyPointVectors)
  "SpyPointVectors$B$G%9%Q%$@_Dj$N2r=|$rAw?.$9$k!#(B
SpyPointVectors = list of (vector Indication SpyPointId)
Indication = \"subgoal\" | \"rule\" | \"module\"
SpyPointId = Integer
[$BNc(B] (list [ \"subgoal\" 1] [\"rule\" 2])
"
  (interactive
   (let ((SpyPointVectors nil)
	 (SpyPointVector  nil))
     (while  (setq SpyPointVector (qxt-read-spypoint))
       (setq SpyPointVectors (append SpyPointVectors (list SpyPointVector))))
     (list SpyPointVectors)))
    (qxt-check-status 'database)
    (qxt-send-command 
     (concat "{unspy_at_spypoints,"
	     (qxt-change-spy-point-vectors SpyPointVectors) "}" )))
;;
(defun qxt-read-spypoint()
  (let (Indication SpyPointId)
    (setq Indication (completing-read "Indication: " 
		  '( ("subgoal" 1) ("rule" 2) ("module" 3) ) nil t ))
    (if (string= Indication "") nil
      (setq SpyPointId (read-string "SpyPointId: "))
      (vector Indication  SpyPointId ))))

;; SpyPointVectors = list of (vector Indication SpyPointId)
;; Indication = "subgoal" | "rule" | "module"
;; SpyPointId = Integer
(defun qxt-change-spy-point-vectors (SpyPointVectors)
  (let ((SendingForm "["))
    (if (eq SpyPointVectors nil) ()
      (setq SendingForm 
	    (concat SendingForm
		    (qxt-change-spy-point-vector (car SpyPointVectors))))
      (setq SpyPointVectors (cdr SpyPointVectors))
      (while SpyPointVectors
	(setq SendingForm 
	      (concat SendingForm ","
		      (qxt-change-spy-point-vector (car SpyPointVectors))))
	(setq SpyPointVectors (cdr SpyPointVectors))))
    (concat SendingForm "]")))

(defun qxt-change-spy-point-vector (SpyPointVector)
  (let ((Indication  (aref SpyPointVector 0))
	(SpyPointId  (aref SpyPointVector 1)))
    (concat "{"
	    Indication
	    ","
	    SpyPointId
	    "}")))

;;
;; enable_all
;;	C:enable_all
;;	S:result_enable_all(Status)
;;
(defun qxt-enable-all()
  "$B$9$Y$F$N%9%Q%$$N%$%M!<%V%k$rAw?.$9$k!#(B"
  (interactive)
  (progn
    (qxt-check-status 'database)
    (qxt-send-command "enable_all" )))

;;
;; enable_at_spypoints
;;	C:enable_at_spypoints(SpyPoints)
;;	S:result_enable_at_spypoints(Status)
;;
(defun qxt-enable-at-spypoints(SpyPointVectors)
  "SpyPointVectors$B$G;XDj$7$?%9%Q%$%]%$%s%H$N%$%M!<%V%k$rAw?.$9$k!#(B
SpyPointVectors = list of (vector Indication SpyPointId)
Indication = \"subgoal\" | \"rule\" | \"module\"
SpyPointId = Integer
[$BNc(B] (list [ \"subgoal\" 1] [\"rule\" 2])
"
  (interactive
   (let ((SpyPointVectors nil)
	 (SpyPointVector  nil))
     (while  (setq SpyPointVector (qxt-read-spypoint))
       (setq SpyPointVectors (append SpyPointVectors (list SpyPointVector))))
     (list SpyPointVectors)))
    (qxt-check-status 'database)
    (qxt-send-command 
     (concat "{enable_at_spypoints,"
	     (qxt-change-spy-point-vectors SpyPointVectors) "}" )))

;;
;; disable_all
;;	C:disable_all
;;	S:result_disable_all(Status)
;;
(defun qxt-disable-all()
  "$B$9$Y$F$N%9%Q%$$N%G%#%;!<%V%k$rAw?.$9$k!#(B"
  (interactive)
  (qxt-check-status 'database)
  (qxt-send-command "disable_all" ))

;;
;; disable_at_spypoints(SpyPoints)
;;	C:disable_at_spypoints(SpyPoints)
;;	S:result_disable_at_spypoints(Status)
;;
(defun qxt-disable-at-spypoints(SpyPointVectors)
  "SpyPointVectors$B$G;XDj$7$?%9%Q%$%]%$%s$N%G%#%;!<%V%k$rAw?.$9$k!#(B
Spypoints = list of (vector Indication SpyPointId)
Indication = \"subgoal\" | \"rule\" | \"module\"
SpyPointId = Integer
[$BNc(B] (list [ \"subgoal\" 1] [\"rule\" 2])
"
  (interactive
   (let ((SpyPointVectors nil)
	 (SpyPointVector  nil))
     (while  (setq SpyPointVector (qxt-read-spypoint))
       (setq SpyPointVectors (append SpyPointVectors (list SpyPointVector))))
     (list SpyPointVectors)))
    (qxt-check-status 'database)
    (qxt-send-command 
     (concat "{disable_at_spypoints,"
	     (qxt-change-spy-point-vectors SpyPointVectors) "}" )))
;;
;;
;; save_spy
;;	C:get_all_state
;;	S:result_get_all_state(Status,SubgoalInfos,RuleInfos,ModuleInfos,
;;			Gates,TraceMode)
;;		SubgoalInfos = [{SpySubgoal,SpyInfo}, ... ]
;;		RuleInfos = [{SpyRule,SpyInfo}, ... ]
;;		ModuleInfos = [{ModuleId,SpyInfo}, ... ]
;;
(defvar qxt-spy-file-name
"qxt-save-spy$B$G;XDj$7$?(BFileName$B$rJ]B8$7$F$*$/JQ?t!"(B
qxt-result-get-all-spy$B$GMxMQ$9$k!#(B")

(defun qxt-save-spy (FileName)
  "$B8=:_$N%H%l!<%9$N>uBV$N<hF@$rAw?.$9$k!#(B
FileName$B$O(Bqxt-spy-file-name$B$KJ]B8$9$k!#(B
FileName = String
"
  (interactive "FSaveSpyFile: ")
  (qxt-check-status 'database)
  (setq qxt-spy-file-name FileName)
  (qxt-send-command "get_all_state" ))
;;
(defun qxt-result-get-all-state(Result)
  "$B<u?.$7$?%H%l!<%9$N>uBV$r%U%!%$%k$KJ]B8$9$k!#(B
"
  (let ((SubgoalInfos (qxt-lisp2term-string (aref Result 2)))
	(RuleInfos (qxt-lisp2term-string (aref Result 3)))
	(ModuleInfos (qxt-lisp2term-string (aref Result 4)))
	(Gates (qxt-lisp2term-string (aref Result 5)))
	(TraceMode (qxt-lisp2term-string (aref Result 6)))
	Buffer)
    (setq CommandString (concat "set_all_state("
	   SubgoalInfos "," RuleInfos "," ModuleInfos ","
	   Gates "," TraceMode ")" ))
    (save-excursion
      (setq Buffer (get-buffer-create "*save-spy*"))
      (set-buffer Buffer)
      (erase-buffer)
      (print CommandString Buffer)
      (write-file qxt-spy-file-name) )))

;;
;; set-all-state
;;	C:set_all_state(SubgoaInfos,RuleInfos,ModuleInfos,Gates,TraceMode)
;;	S:result_set_all_state(Status)
;;
(defun qxt-set-all-state (FileName)
  "$B%H%l!<%9$N>uBV$r(BFileName$B$N%U%!%$%k$+$iFI$_!"%9%Q%$$N@_Dj$rAw?.$9$k!#(B
FileName = String
"
  (interactive "FLoadSpyFile: ")
  (let (CommandString
	(Buffer (find-file-noselect FileName)))
    (qxt-check-status 'database)
    (save-excursion
      (set-buffer Buffer)
      ;; point needs at beginning for second read
      (beginning-of-buffer)
      (setq CommandString (read Buffer))
      (qxt-send-command CommandString))))

;;
;; reset_trace
;;	C: reset_trace
;;	S: result_reset_trace(Status)
;;
(defun qxt-reset-trace ()
  "$B%H%l!<%9>uBV$N=i4|2=$rAw?.$9$k!#(B"
  (interactive)
  (qxt-check-status 'database)
  (qxt-send-command "reset_trace" ))

