//
// HiRiseDemo: A Demo Applet for the HiRise Constraint Solver
//
// Copyright (C) 1998-1999 Hiroshi HOSOBE
//
////////////////////////////////////////////////////////////////

import java.util.*;
import java.awt.*;
import java.awt.event.*;

public final class TreeDemo extends Panel
implements ActionListener, ItemListener, MouseListener, MouseMotionListener {
	
	private static final int DEFAULT_HEIGHT = 4;
	private static final int DEFAULT_MAX_CHILDREN = 2;
	private static final long DEFAULT_SRAND = 44;

	private static final int IM_NORMAL = 0;
	private static final int IM_DRAG_OBJECT = 1;
	private static final int IM_SPECIFY_LINE = 2;

	private final class TreeNode extends Object {

		public TreeNode m_parent;
		public Vector m_children = new Vector();

		public Variable m_x = new Variable();
		public Variable m_y = new Variable();
		public Variable m_left = new Variable();
		public Variable m_right = new Variable();

		public Linear m_midAlign = new Linear(0);
		
		// for a leaf node
		public Linear m_xInterval;

		// for an internal node
		public Linear m_leftAlign;
		public Linear m_rightAlign;
		public Vector m_childrenGlues = new Vector();
		public Vector m_yIntervals = new Vector();

		TreeNode(Variable xUnit, Variable yUnit,
				 TreeNode parent, int height, int nChildren,
				 Random rand)
		{
			int i;

			m_parent = parent;

			int nChildren1 = 0;
			if (height > 1) {
				nChildren1 = (int) (rand.nextDouble() * (nChildren + 1));
				if (nChildren1 == nChildren + 1)
					nChildren1 = nChildren;
			}

			if (nChildren1 == 0 || height == 1) { // leaf node
				m_midAlign.center(m_left, m_right, m_x);
				m_solver.add(m_midAlign);

				m_xInterval = new Linear(1);
				m_xInterval.sum(m_left, xUnit, m_right);
				m_solver.add(m_xInterval);
			}
			else { // internal node
				TreeNode lastChild = null;
				for (i = 0; i < nChildren1; i++) {
					TreeNode child = new TreeNode(xUnit, yUnit,
												  this, height - 1, nChildren,
												  rand);
					m_children.addElement(child);

					if (i > 0) {
						Linear childrenGlue = new Linear(0);
						m_childrenGlues.addElement(childrenGlue);
				
						childrenGlue.equal(lastChild.m_right, child.m_left);
						m_solver.add(childrenGlue);
					}

					Linear yInterval = new Linear(1);
					m_yIntervals.addElement(yInterval);

					yInterval.sum(m_y, yUnit, child.m_y);
					m_solver.add(yInterval);

					lastChild = child;
				}

				TreeNode firstChild = (TreeNode) m_children.elementAt(0);
				m_leftAlign = new Linear(0);
				m_leftAlign.equal(m_left, firstChild.m_left);
				m_solver.add(m_leftAlign);

				m_midAlign.center(m_left, m_right, m_x);
				m_solver.add(m_midAlign);

				m_rightAlign = new Linear(0);
				m_rightAlign.equal(m_right, lastChild.m_right);
				m_solver.add(m_rightAlign);
			}
		}
		
		void dispose()
		{
			int nChildren = m_children.size();
			int n, i;

			m_solver.remove(m_midAlign);

			if (nChildren == 0) {
				m_solver.remove(m_xInterval);
				return;
			}

			m_solver.remove(m_leftAlign);
			m_solver.remove(m_rightAlign);

			for (i = 0; i < nChildren; i++) {
				if (i > 0) {
					Linear childrenGlue = (Linear)
						m_childrenGlues.elementAt(i - 1);
					m_solver.remove(childrenGlue);
				}

				Linear yInterval = (Linear) m_yIntervals.elementAt(i);
				m_solver.remove(yInterval);
			}

			for (i = 0; i < nChildren; i++) {
				TreeNode child = (TreeNode) m_children.elementAt(i);
				child.dispose();
			}
		}

	}
	
	private final class TreeView extends Panel {

		void calcEndPoints(Dimension dim, 
						   Point point0, Point point1, 
						   Point end0, Point end1)
		{
 			double a, b, c;
 			double x, y;
 			int i;

			a = point1.y - point0.y;
			b = -(point1.x - point0.x);
			c = point0.x * a + point0.y * b;

 			Point[] end = new Point[4];
			for (i = 0; i < 4; i++)
				end[i] = new Point();

			int n = 0;

			if (b != 0) {
				// left
				y = c / b;
				if (y >= 0 && y <= dim.height) {
					end[n].x = 0;
					end[n].y = (int) y;
					n++;
				}
		
				// right
				y = (c - a * dim.width) / b;
				if (y >= 0 && y <= dim.height) {
					end[n].x = dim.width;
					end[n].y = (int) y;
					n++;
				}
			}

			if (a != 0) {
				// top
				x = c / a;
				if (x >= 0 && x <= dim.width) {
					end[n].x = (int) x;
					end[n].y = 0;
					n++;
				}

				// bottom
				x = (c - b * dim.height) / a;
				if (x >= 0 && x <= dim.width) {
					end[n].x = (int) x;
					end[n].y = dim.height;
					n++;
				}
			}

			if (n == 0) {
				end0.setLocation(0, 0);
				end1.setLocation(0, 0);
			}
			else if (n == 1) {
				end0.setLocation(end[0]);
				end1.setLocation(end[0]);
			}
			else if (n == 2) {
				end0.setLocation(end[0]);
				end1.setLocation(end[1]);
			}
			else if (n > 2) {
				end0.setLocation(end[0]);
				for (i = 1; i < n; i++) {
					if (end[i].x != end0.x || end[i].y != end0.y) {
						end1.setLocation(end[i]);
						break;
					}
				}
				if (i == n)
					end1.setLocation(end0);
			}
 		}

		private void drawTree(TreeNode tree, Graphics g)
		{
			int n, i;

			Point p = new Point((int) tree.m_x.get(), (int) tree.m_y.get());

			n = tree.m_children.size();
 			for (i = 0; i < n; i++) {
 				TreeNode child = (TreeNode) tree.m_children.elementAt(i);
 				if (child != null) {
					g.setColor(Color.black);
 					GraphicsUtility.drawThickLine(
						g, p.x, p.y,
						(int) child.m_x.get(), (int) child.m_y.get(), 2);
 				}
 			}

			Point rectTL = new Point(p.x - m_nodeSize / 2,
									 p.y - m_nodeSize / 2);

 			if (tree == m_draggedTreeNode) {
				g.setColor(Color.green);
				g.fillRect(rectTL.x, rectTL.y, m_nodeSize, m_nodeSize);
			}
 			else {
 				boolean done = false;

				n = m_stayTreeData.size();
 				for (i = 0; i < n; i++) {
 					StayTreeData data = (StayTreeData)
 						m_stayTreeData.elementAt(i);
 					if (data.m_node == tree) {
						g.setColor(Color.red);
						g.fillRect(rectTL.x, rectTL.y, m_nodeSize, m_nodeSize);
 						done = true;
 						break;
 					}
 				}

				if (!done) {
					n = m_lineTreeData.size();
					for (i = 0; i < n; i++) {
						LineTreeData data = (LineTreeData)
							m_lineTreeData.elementAt(i);
						if (data.m_node == tree) {
							g.setColor(Color.red);
							g.fillRect(rectTL.x, rectTL.y,
									   m_nodeSize, m_nodeSize);
							done = true;
							break;
						}
					}
				}

 				if (!done) {
					g.setColor(Color.blue);
 					g.fillRect(rectTL.x, rectTL.y, m_nodeSize, m_nodeSize);
				}
 			}

			n = tree.m_children.size();
 			for (i = 0; i < n; i++) {
 				TreeNode child = (TreeNode) tree.m_children.elementAt(i);
 				if (child != null)
 					drawTree(child, g);
 			}
		}

		public void paint(Graphics g)
		{
			int n, i;

			if (m_tree == null)
				return;
			
			// prepare off-screen for double buffering
			Dimension d = getSize();
			Graphics og;
			if (m_offScreen == null) {
				m_offScreen = createImage(d.width, d.height);
				og = m_offScreen.getGraphics();
				og.setClip(0, 0, d.width, d.height);
			}
			else {
				og = m_offScreen.getGraphics();
				og.clearRect(0, 0, d.width, d.height);
			}

			super.paint(og);

			// draw objects on off-screen

			// draw specifyed line
			if (m_inputMode == IM_SPECIFY_LINE) {
				Point end0 = new Point();
				Point end1 = new Point();
				calcEndPoints(d, m_lineEnd0, m_lineEnd1, end0, end1);

				og.setColor(Color.green);
				GraphicsUtility.drawThickLine(og, end0.x, end0.y,
											  end1.x, end1.y, 2);
			}

			// draw constraining lines
			og.setColor(Color.red);
			n = m_lineTreeData.size();
			for (i = 0; i < n; i++) {
				LineTreeData data = (LineTreeData)
					m_lineTreeData.elementAt(i);

				Point end0 = new Point();
				Point end1 = new Point();
				calcEndPoints(d, data.m_lineEnd0, data.m_lineEnd1,
							  end0, end1);

				GraphicsUtility.drawThickLine(og, end0.x, end0.y,
											  end1.x, end1.y, 2);
			}

			// draw tree
			drawTree(m_tree, og);
		
			// transfer off-screen image
			g.drawImage(m_offScreen, 0, 0, null);
			og.dispose();
		}	
	
		public void invalidate()
		{
			super.invalidate();
			m_offScreen = null;
		}

		public void update(Graphics g)
		{
			paint(g);
		}

	}
	
	private HiRiseDemo m_applet;
	
	private int m_height = DEFAULT_HEIGHT;
	private int m_maxChildren = DEFAULT_MAX_CHILDREN;
	private long m_srand = DEFAULT_SRAND;
	
	private Solver m_solver;

	private Variable m_xUnit;
	private Variable m_yUnit;
	private Stay m_stayXUnit;
	private Stay m_stayYUnit;
	private Stay m_stayTopX;
	private Stay m_stayTopY;
	private TreeNode m_tree;

	private Point m_lastMousePosition;
	private int m_inputMode;

	private TreeNode m_draggedTreeNode;
	private Edit m_editX;
	private Edit m_editY;
	private ConstraintSet m_editXnY;

	private Point m_popupPoint;
	private TreeNode m_selectedNode;

	private class StayTreeData {
		public TreeNode m_node;
		public Stay m_stayX;
		public Stay m_stayY;
	}

	private Vector m_stayTreeData;

	private TreeNode m_linedTreeNode;
	private Point m_lineEnd0;
	private Point m_lineEnd1;

	private class LineTreeData {
		public Point m_lineEnd0;
		public Point m_lineEnd1;
		public TreeNode m_node;
		public Linear m_line;
	};

	private Vector m_lineTreeData;

	private int m_nodeSize = 20;

	private ExtTextField m_heightField;
	private ExtTextField m_maxChildrenField;
	private ExtTextField m_srandField;

	private TreeView m_treeView;
	private Image m_offScreen;

	private PopupMenu m_popupMenu;
	private MenuItem m_addChildItem;
	private MenuItem m_removeItem;
	private CheckboxMenuItem m_stayItem;
	private CheckboxMenuItem m_followLineItem;
	
	TreeDemo(HiRiseDemo applet)
	{
		m_applet = applet;
		
		setLayout(new BorderLayout());
		
		Panel p = new Panel();
		add(p, BorderLayout.NORTH);
		p.setLayout(new FlowLayout(FlowLayout.LEFT));

		p.add(new Label("Height:"));
		
		m_heightField = new ExtTextField(Integer.toString(m_height), 2);
		p.add(m_heightField);
		
		p.add(new Label("Max children:"));

		m_maxChildrenField = new ExtTextField(Integer.toString(m_maxChildren), 2);
		p.add(m_maxChildrenField);
		
		p.add(new Label("Seed of random numbers:"));

		m_srandField = new ExtTextField(Long.toString(m_srand), 5);
		p.add(m_srandField);

		m_treeView = new TreeView();
		add(m_treeView, BorderLayout.CENTER);
		m_treeView.addMouseListener(this);
		m_treeView.addMouseMotionListener(this);
		
		m_popupMenu = new PopupMenu();
		m_treeView.add(m_popupMenu);
		m_addChildItem = new MenuItem("Add child");
		m_popupMenu.add(m_addChildItem);
		m_addChildItem.addActionListener(this);
		m_removeItem = new MenuItem("Remove");
		m_popupMenu.add(m_removeItem);
		m_removeItem.addActionListener(this);
		m_popupMenu.addSeparator();
		m_stayItem = new CheckboxMenuItem("Stay");
		m_popupMenu.add(m_stayItem);
		m_stayItem.addItemListener(this);
		m_followLineItem = new CheckboxMenuItem("Follow line");
		m_popupMenu.add(m_followLineItem);
		m_followLineItem.addItemListener(this);
	}
	

	public void start()
	{
		try {
			m_height = new Integer(m_heightField.getText()).intValue();
			if (m_height < 1)
				m_height = 1;
			else if (m_height > 32)
				m_height = 32;
		}
		catch (NumberFormatException ex) {
			m_height = DEFAULT_HEIGHT;
		}
		m_heightField.setText(Integer.toString(m_height));

		try {
			m_maxChildren = new
				Integer(m_maxChildrenField.getText()).intValue();
			if (m_maxChildren < 1)
				m_maxChildren = 1;
			else if (m_maxChildren > 32)
				m_maxChildren = 32;
		}
		catch (NumberFormatException ex) {
			m_maxChildren = DEFAULT_MAX_CHILDREN;
		}
		m_maxChildrenField.setText(Integer.toString(m_maxChildren));

		try {
			m_srand = new Long(m_srandField.getText()).longValue();
		}
		catch (NumberFormatException ex) {
			m_srand = DEFAULT_SRAND;
		}
		m_srandField.setText(Long.toString(m_srand));

		m_solver = new Solver();

		m_xUnit = new Variable();
		m_yUnit = new Variable();

		m_stayXUnit = new Stay(4);
		m_stayXUnit.add(m_xUnit);
		m_solver.add(m_stayXUnit);

		m_stayYUnit = new Stay(4);
		m_stayYUnit.add(m_yUnit);
		m_solver.add(m_stayYUnit);

		m_tree = new TreeNode(m_xUnit, m_yUnit,
							  null, m_height, m_maxChildren,
							  new Random(m_srand));

		m_stayTopX = new Stay(5);
		m_stayTopX.add(m_tree.m_x);
		m_solver.add(m_stayTopX);

		m_stayTopY = new Stay(5);
		m_stayTopY.add(m_tree.m_y);
		m_solver.add(m_stayTopY);

		Edit editXUnit = new Edit(2);
		editXUnit.add(m_xUnit, 30);
		m_solver.add(editXUnit);

		Edit editYUnit = new Edit(2);
		editYUnit.add(m_yUnit, 60);
		m_solver.add(editYUnit);

		Edit editTopX = new Edit(2);
		editTopX.add(m_tree.m_x, 240);
		m_solver.add(editTopX);

		Edit editTopY = new Edit(2);
		editTopY.add(m_tree.m_y, 20);
		m_solver.add(editTopY);

		planAndExec();
		
		m_solver.remove(editXUnit);
		m_solver.remove(editYUnit);
		m_solver.remove(editTopX);
		m_solver.remove(editTopY);

		m_inputMode = IM_NORMAL;
		m_draggedTreeNode = null;

		m_stayTreeData = new Vector();
		m_lineTreeData = new Vector();

		m_treeView.invalidate();
		m_treeView.repaint();
	}

	void planAndExec()
	{
		m_applet.setWaitCursor();

		long t0 = System.currentTimeMillis();
		m_solver.plan();
		long t1 = System.currentTimeMillis();
		m_solver.execute();
		long t2 = System.currentTimeMillis();

		m_applet.setDefaultCursor();
		
		long planTime = t1 - t0;
		long execTime = t2 - t1;

		String stat = "plan: " + planTime + " ms, exec: " + execTime
			+ " ms (#vars: " + m_solver.countVars()
			+ ", #cons: " + m_solver.countCons() + ")";
		m_applet.showStatus(stat);
	}

	private void setPopupMenuState(boolean addChildEnabled,
								   boolean removeEnabled,
								   boolean stayChecked,
								   boolean followLineChecked)
	{
		m_addChildItem.setEnabled(addChildEnabled);
		m_removeItem.setEnabled(removeEnabled);
		m_stayItem.setState(stayChecked);
		m_followLineItem.setState(followLineChecked);
	}
	
	TreeNode PickTreeNode(TreeNode tree, int nodeSize, Point point)
	{
		int n, i;

		int x = (int) tree.m_x.get();
		int y = (int) tree.m_y.get();
		int h = nodeSize / 2;
		if (point.x >= x - h && point.x <= x + h &&
			point.y >= y - h && point.y <= y + h)
			return tree;

		n = tree.m_children.size();
		for (i = 0; i < n; i++) {
			TreeNode child = (TreeNode) tree.m_children.elementAt(i);
			if (child != null) {
				TreeNode pickedNode = PickTreeNode(child, nodeSize, point);
				if (pickedNode != null)
					return pickedNode;
			}
		}

		return null;
	}

	public void mouseClicked(MouseEvent ev)
	{
		// no operation
	}

	public void processPopupMenu(MouseEvent ev)
	{
		int n, i;

		if (!ev.isPopupTrigger())
			return;

		m_popupPoint = ev.getPoint();

		m_selectedNode = PickTreeNode(m_tree, m_nodeSize, m_popupPoint);
		if (m_selectedNode != null) {

			boolean enableRemove = m_selectedNode.m_parent != null;
		
			n = m_stayTreeData.size();
			for (i = 0; i < n; i++) {
				StayTreeData data = (StayTreeData)
					m_stayTreeData.elementAt(i);
				if (data.m_node == m_selectedNode)
					break;
			}
			boolean checkStay = i < n;

			n = m_lineTreeData.size();
			for (i = 0; i < n; i++) {
				LineTreeData data = (LineTreeData)
					m_lineTreeData.elementAt(i);
				if (data.m_node == m_selectedNode)
					break;
			}
			boolean checkFollowLine = i < n;

			setPopupMenuState(true, enableRemove, checkStay, checkFollowLine);
			m_popupMenu.show(m_treeView, m_popupPoint.x, m_popupPoint.y);
		}
	}

	public void mousePressed(MouseEvent ev)
	{
		if (m_tree == null)
			return;

		Object src = ev.getSource();
		if (src == m_treeView) {
			if (ev.isMetaDown()) {
				processPopupMenu(ev);
				return;
			}

			Point point;
			switch (m_inputMode) {
			case IM_NORMAL:
				point = ev.getPoint();
				m_draggedTreeNode = PickTreeNode(m_tree, m_nodeSize, point);
				if (m_draggedTreeNode != null) {
					m_inputMode = IM_DRAG_OBJECT;

					m_editX = new Edit(3);
					m_editX.add(m_draggedTreeNode.m_x, point.x);
		
					m_editY = new Edit(3);
					m_editY.add(m_draggedTreeNode.m_y, point.y);
		
					//m_editXnY = new ConstraintSet(ConstraintSet.NORMAL);
					m_editXnY = new ConstraintSet(ConstraintSet.OPTIMIZING);
					m_editXnY.add(m_editX);
					m_editXnY.add(m_editY);
					m_solver.add(m_editXnY);

					m_treeView.repaint();

					planAndExec();
				}
				break;
			case IM_DRAG_OBJECT:
				m_inputMode = IM_NORMAL;
				break;
			case IM_SPECIFY_LINE:
				break;
			}
		}
	}

	public void mouseReleased(MouseEvent ev)
	{
		if (m_tree == null)
			return;

		Object src = ev.getSource();
		if (src == m_treeView) {
			if (ev.isMetaDown()) {
				processPopupMenu(ev);
				return;
			}

			switch (m_inputMode) {
			case IM_NORMAL:
				break;
			case IM_DRAG_OBJECT:
				m_inputMode = IM_NORMAL;

				m_solver.remove(m_editXnY);		

				planAndExec();
				
				m_draggedTreeNode = null;

				m_treeView.repaint();
				break;
			case IM_SPECIFY_LINE:
				m_inputMode = IM_NORMAL;

				LineTreeData data = new LineTreeData();
				data.m_node = m_linedTreeNode;
				data.m_lineEnd0 = m_lineEnd0;
				data.m_lineEnd1 = m_lineEnd1;

				data.m_line = new Linear(2);
				data.m_line.add(m_linedTreeNode.m_x,
								(double) m_lineEnd1.y - m_lineEnd0.y);
				data.m_line.add(m_linedTreeNode.m_y,
								(double) -(m_lineEnd1.x - m_lineEnd0.x));
				data.m_line.setConst((double) m_lineEnd0.x
									 * (m_lineEnd1.y - m_lineEnd0.y)
								     - m_lineEnd0.y
								     * (m_lineEnd1.x - m_lineEnd0.x));
				m_solver.add(data.m_line);

				planAndExec();

				m_lineTreeData.addElement(data);

				m_treeView.repaint();
				break;
			}
		}
	}

	public void mouseEntered(MouseEvent ev)
	{
		// no operation
	}

	public void mouseExited(MouseEvent ev)
	{
		// no operation
	}

	public void mouseDragged(MouseEvent ev)
	{
		mouseMoved(ev);
	}
		
    public void mouseMoved(MouseEvent ev)
	{
		m_lastMousePosition = ev.getPoint();

		if (m_tree == null)
			return;

		if (!m_treeView.contains(m_lastMousePosition))
			return;

		Object src = ev.getSource();
		if (src == m_treeView) {
			switch (m_inputMode) {
			case IM_NORMAL:
				break;
			case IM_DRAG_OBJECT:
				m_editX.set((int) m_lastMousePosition.x);
				m_editY.set((int) m_lastMousePosition.y);
				m_solver.execute();

				m_treeView.repaint();
				break;
			case IM_SPECIFY_LINE:
				m_lineEnd1.setLocation(m_lastMousePosition);

				m_treeView.repaint();
				break;
			}
		}
	}
	
	public void actionPerformed(ActionEvent ev)
	{
		if (m_tree == null)
			return;

		Object src = ev.getSource();
		if (src == m_addChildItem)
			addChildToNode();
		else if (src == m_removeItem)
			removeNode();
	}

	public void itemStateChanged(ItemEvent ev)
	{
		if (m_tree == null)
			return;

		Object src = ev.getSource();
		if (src == m_stayItem)
			stayNode();
		else if (src == m_followLineItem)
			makeNodeFollowLine();
	}

	private void addChildToNode()
	{
		if (m_selectedNode == null)
			return;

		TreeNode child = new TreeNode(m_xUnit, m_yUnit,
									  m_selectedNode, 1, 0, (Random) null);
		m_selectedNode.m_children.addElement(child);
		
		Linear yInterval = new Linear(1);
		m_selectedNode.m_yIntervals.addElement(yInterval);

		yInterval.sum(m_selectedNode.m_y, m_yUnit, child.m_y);
		m_solver.add(yInterval);

		int nChildren = m_selectedNode.m_children.size();
		if (nChildren == 1) {
			m_solver.remove(m_selectedNode.m_xInterval);
			m_selectedNode.m_xInterval = null; // dispose m_xInterval

			m_selectedNode.m_leftAlign = new Linear(0);
			m_selectedNode.m_leftAlign.equal(m_selectedNode.m_left,
											 child.m_left);
			m_solver.add(m_selectedNode.m_leftAlign);
		}
		else {
			Linear childrenGlue = new Linear(0);
			m_selectedNode.m_childrenGlues.addElement(childrenGlue);
				
			TreeNode lastChild = (TreeNode)
				m_selectedNode.m_children.elementAt(nChildren - 2);
			childrenGlue.equal(lastChild.m_right, child.m_left);
			m_solver.add(childrenGlue);

			// replace m_rightAlign
			m_solver.remove(m_selectedNode.m_rightAlign);
		}

		m_selectedNode.m_rightAlign = new Linear(0);
		m_selectedNode.m_rightAlign.equal(m_selectedNode.m_right,
										  child.m_right);
		m_solver.add(m_selectedNode.m_rightAlign);

		planAndExec();

		m_treeView.repaint();
	}

	private void removeNode()
	{
		int i;
		
		if (m_selectedNode == null || 
			m_selectedNode.m_parent == null)
			return;

		TreeNode parent = m_selectedNode.m_parent; 

		int nBros = parent.m_children.size();

		int idx = 0;
		for (i = 0; i < nBros; i++) {
			if (parent.m_children.elementAt(i) == m_selectedNode) {
				parent.m_children.removeElementAt(i);
				idx = i;
				nBros--;
				break;
			}
		}

		disposeStaysAndLines(m_selectedNode);
		m_selectedNode.dispose();

		m_solver.remove((Linear) parent.m_yIntervals.elementAt(idx));
		parent.m_yIntervals.removeElementAt(idx);

		if (nBros == 0) {
			m_solver.remove(parent.m_leftAlign);
			m_solver.remove(parent.m_rightAlign);

			parent.m_xInterval = new Linear(1);
			parent.m_xInterval.sum(parent.m_left, m_xUnit, parent.m_right);
			m_solver.add(parent.m_xInterval);
		}
		else {
			if (idx == 0) { // leftmost
				m_solver.remove((Linear)
								parent.m_childrenGlues.elementAt(0));
				parent.m_childrenGlues.removeElementAt(0);

				TreeNode firstBro = (TreeNode) parent.m_children.elementAt(0);

				// replace m_leftAlign
				m_solver.remove(parent.m_leftAlign);
				
				parent.m_leftAlign = new Linear(0);
				parent.m_leftAlign.equal(parent.m_left, firstBro.m_left);
				m_solver.add(parent.m_leftAlign);
			}
			else if (idx < nBros) { // inbetween
				m_solver.remove((Linear)
								parent.m_childrenGlues.elementAt(idx));
				parent.m_childrenGlues.removeElementAt(idx);
				m_solver.remove((Linear)
								parent.m_childrenGlues.elementAt(idx - 1));
				parent.m_childrenGlues.removeElementAt(idx - 1);

				TreeNode prevBro = (TreeNode)
					parent.m_children.elementAt(idx - 1);
				TreeNode nextBro = (TreeNode)
					parent.m_children.elementAt(idx);

				Linear childrenGlue = new Linear(0);
				parent.m_childrenGlues.addElement(childrenGlue);
				
				childrenGlue.equal(prevBro.m_right, nextBro.m_left);
				m_solver.add(childrenGlue);
			}
			else { // rightmost
				m_solver.remove((Linear)
								parent.m_childrenGlues.elementAt(nBros - 1));
				parent.m_childrenGlues.removeElementAt(nBros - 1);

				TreeNode lastBro = (TreeNode)
					parent.m_children.elementAt(nBros - 1);

				// replace m_rightAlign
				m_solver.remove(parent.m_rightAlign);

				parent.m_rightAlign = new Linear(0);
				parent.m_rightAlign.equal(parent.m_right, lastBro.m_right);
				m_solver.add(parent.m_rightAlign);
			}
		}

		planAndExec();

		m_treeView.repaint();
	}

	void disposeStaysAndLines(TreeNode node)
	{
		int n, i;

		n = node.m_children.size();
		for (i = 0; i < n; i++) {
			TreeNode child = (TreeNode) node.m_children.elementAt(i);
			disposeStaysAndLines(child);
		}

		n = m_stayTreeData.size();
		for (i = 0; i < n; i++) {
			StayTreeData data = (StayTreeData) m_stayTreeData.elementAt(i);
			if (data.m_node == node) {
				m_stayTreeData.removeElementAt(i);
				m_solver.remove(data.m_stayX);
				m_solver.remove(data.m_stayY);
				break;
			}
		}
	
		n = m_lineTreeData.size();
		for (i = 0; i < n; i++) {
			LineTreeData data = (LineTreeData) m_lineTreeData.elementAt(i);
			if (data.m_node == node) {
				m_lineTreeData.removeElementAt(i);
				m_solver.remove(data.m_line);
				break;
			}
		}
	}

	private void stayNode()
	{
		int n, i;

		if (m_selectedNode == null)
			return;

		StayTreeData data = null;
		n = m_stayTreeData.size();
		for (i = 0; i < n; i++) {
			data = (StayTreeData) m_stayTreeData.elementAt(i);
			if (data.m_node == m_selectedNode)
				break;
		}
		if (i < n) {
			m_stayTreeData.removeElementAt(i);
			m_solver.remove(data.m_stayX);
			m_solver.remove(data.m_stayY);
		}
		else {
			data = new StayTreeData();
	
			data.m_node = m_selectedNode;

			data.m_stayX = new Stay(2);
			data.m_stayX.add(m_selectedNode.m_x);
			m_solver.add(data.m_stayX);

			data.m_stayY = new Stay(2);
			data.m_stayY.add(m_selectedNode.m_y);
			m_solver.add(data.m_stayY);

			m_stayTreeData.addElement(data);
		}

		planAndExec();

		m_treeView.repaint();
	}

	private void makeNodeFollowLine()
	{
		int n, i;

		if (m_selectedNode == null)
			return;

		LineTreeData data = null;
		n = m_lineTreeData.size();
		for (i = 0; i < n; i++) {
			data = (LineTreeData) m_lineTreeData.elementAt(i);
			if (data.m_node == m_selectedNode)
				break;
		}
		if (i < n) {
			m_lineTreeData.removeElementAt(i);
			m_solver.remove(data.m_line);

			planAndExec();
		}
		else {
			m_inputMode = IM_SPECIFY_LINE;
	
			m_linedTreeNode = m_selectedNode;
			m_lineEnd0 = new Point((int) m_selectedNode.m_x.get(),
								   (int) m_selectedNode.m_y.get());
			m_lineEnd1 = new Point(m_lastMousePosition);
		}

		m_treeView.repaint();
	}

}
