//
// The HiRise Constraint Solver for Java, Ver. 2.0
//
// Copyright (C) 1998-1999 Hiroshi HOSOBE
//
////////////////////////////////////////////////////////////////

//package jp.ac.u_tokyo.s.is.hirise;

import java.util.*;

public final class Solver extends HiRiseObject {

	public static final int N_LEVELS = 16;
	public static final double EPS = 1.0e-4;

	private final class ConIndex extends Object {

		int strength;
		int index;

		ConIndex(int s, int i)
		{
			strength = s;
			index = i;
		}

		ConIndex(ConIndex ci)
		{
			strength = ci.strength;
			index = ci.index;
		}

		public final boolean stronger(ConIndex ci)
		{
			return strength < ci.strength ||
				(strength == ci.strength && index < ci.index);
		}

		public final boolean equals(Object obj)
		{
			ConIndex ci = (ConIndex) obj;
			return strength == ci.strength && index == ci.index;
		}

		public final boolean weaker(ConIndex ci)
		{
			return strength > ci.strength ||
				(strength == ci.strength && index > ci.index);
		}

	}

	private final class LowerVec extends Object {

		ConIndex cIdx;
		Vector elems = new Vector(8, 8);
		double cst;

	}

	private final class UpperVec extends Object {

		boolean shift;
		int srcCol;
		int dstCol;
		Vector elems = new Vector(8, 8);

	}

	private final class BackupInfo {

		Vector lowerVecs = new Vector(8, 8);
		int upperVecsSize;
		
	}

	private final class IncrementalFactorizationFailedException
		extends Exception {

		IncrementalFactorizationFailedException()
		{
		}

	}

	private static final Class m_setClass = (new ConstraintSet(0)).getClass();
	private static final Class m_stayClass = (new Stay(0)).getClass();
	private static final Class m_editClass = (new Edit(0)).getClass();
	private static final Class m_linearClass = (new Linear(0)).getClass();

	public static final boolean nearEq(double val0, double val1)
	{
		double diff = val0 - val1;
		return diff > -EPS && diff < EPS;
	}

	private Vector m_vars = new Vector(8, 8);
	private Vector[] m_level = new Vector[N_LEVELS];
	private Vector m_edits = new Vector(8, 8);
	private Vector m_optimizingSets = new Vector(8, 8);
	private int m_nRealVars = 0;
	private int m_nRealCons = 0;

	private Vector m_lowerVecs = new Vector(8, 8);
	private Vector m_upperVecs = new Vector(8, 8);
	private ConIndex[] m_waStr;
	private double[] m_row;
	private double[] m_row2;
	private double[] m_fsValue;
	private double[] m_bsValue;

	private Vector m_inactCons = new Vector(8, 8);
	private Vector m_addedCons = new Vector(8, 8);
	private Vector m_deletedRows = new Vector(8, 8);
	private ConIndex m_strongestDeletedRowStr
	    = new ConIndex(N_LEVELS + 1, 0);
	private Vector m_removedVars = new Vector(8, 8);
	private int m_firstPrefConIndex = 0;

	Solver()
	{
		int i;

		for (i = 0; i < N_LEVELS; i++)
			m_level[i] = new Vector(8, 8);
	}

	private final Constraint getCon(ConIndex ci)
	{
		return (Constraint) m_level[ci.strength].elementAt(ci.index);
	}

	private final void add(int str, Constraint con)
	{
		m_level[str].addElement(con);
	}

	private final void set(ConIndex ci, Constraint con)
	{
		m_level[ci.strength].setElementAt(con, ci.index);
	}

	private final int countCons(int str)
	{
		return m_level[str].size();
	}

	private final int index(int str, Constraint con)
	{
		return m_level[str].indexOf(con);
	}

	public final int countVars()
	{
		return m_nRealVars;
	}

	public final int countCons()
	{
		return m_nRealCons;
	}

	public final void add(Constraint con)
	{
		int n, i;

		if (con == null)
			throw new IllegalArgumentException(
				"null pointer given as constraint");

		if (con.getClass() == m_setClass) {
			ConstraintSet set = (ConstraintSet) con;

			set.setSolver(this);

			if (set.getStrategy() == ConstraintSet.OPTIMIZING)
				m_optimizingSets.addElement(set);

			n = set.countCons();
			for (i = 0; i < n; i++) {
				Constraint c = set.getCon(i);
				add1(c);
			}

			return;
		}
		
		add1(con);
	}

	private final void add1(Constraint con)
	{
		int n, i;

		Class cls = con.getClass();
		if (cls != m_stayClass &&
			cls != m_editClass &&
			cls != m_linearClass)
			throw new IllegalArgumentException(
				"constraint of invalid type: " + con.getClassAndName());

		int str = con.getStrength();
		if (str < 0 || str >= N_LEVELS)
			throw new IllegalArgumentException(
				"constraint with invalid strength: "
				+ con.getClassAndName());

		if (index(str, con) != -1)
			throw new IllegalArgumentException(
				"constraint added twice: " + con.getClassAndName());

		int nVars = con.countVars();
		if ((cls == m_editClass ||
			 cls == m_stayClass) &&
			nVars != 1)
			throw new IllegalArgumentException(
				"stay or edit constraint not with one variable: "
                + con.getClassAndName());

		con.setSolver(this);
		add(str, con);
		m_nRealCons++;

		con.setRow(-1);

		for (i = 0; i < nVars; i++) {
			Variable var = con.getVar(i);
			var.add(con);
			if (m_vars.indexOf(var) == -1) {
				m_vars.addElement(var);
				m_nRealVars++;
			}
		}
		
		if (cls == m_editClass)
			m_edits.addElement(con);
		else if (cls == m_linearClass) {
			Linear lin = (Linear) con;
			Vector coefs = lin.getCoefs();
			n = lin.countVars();
			for (i = 0; i < n; i++) {
				Variable var = lin.getVar(i);
				VecElem coef = (VecElem) coefs.elementAt(i);
				coef.index = m_vars.indexOf(var);
			}
		}

		n = m_addedCons.size();
		for (i = 0; i < n; i++) {
			Constraint c = (Constraint) m_addedCons.elementAt(i);
			if (c.getStrength() > str) {
				m_addedCons.insertElementAt(con, i);
				break;
			}
		}
		if (i == n)
			m_addedCons.addElement(con);
	}

	public final void remove(Constraint con)
	{
		int n, i;

		if (con == null)
			throw new IllegalArgumentException(
				"null pointer given as constraint");

		if (con.getClass() == m_setClass) {
			ConstraintSet set = (ConstraintSet) con;
			
			n = set.countCons();
			for (i = 0; i < n; i++) {
				Constraint c = set.getCon(i);
				remove1(c);
			}

			if (set.getStrategy() == ConstraintSet.OPTIMIZING)
				m_optimizingSets.removeElement(set);

			set.setSolver(null);

			return;
		}

		remove1(con);
	}

	private final void remove1(Constraint con)
	{
 		int n, i;

		int str0 = con.getStrength();
 		ConIndex ci = new ConIndex(str0, index(str0, con));

 		if (ci.index == -1)
 			throw new IllegalArgumentException(
				"constraint not found: " + con.getClassAndName());

		int row = con.getRow();
		if (row != -1) {
			LowerVec lVec = (LowerVec) m_lowerVecs.elementAt(row);
			lVec.cIdx.index = -1;

			n = m_deletedRows.size();
			for (i = 0; i < n; i++) {
				Integer rowi = (Integer) m_deletedRows.elementAt(i);
				if (rowi.intValue() > row) {
					m_deletedRows.insertElementAt(new Integer(row), i);
					break;
				}
			}
			if (i == n)
				m_deletedRows.addElement(new Integer(row));

			if (m_strongestDeletedRowStr.weaker(ci))
				m_strongestDeletedRowStr = ci;
		}
		else 
			m_inactCons.removeElement(con);

		m_addedCons.removeElement(con);

 		if (con.getClass() == m_editClass)
 			m_edits.removeElement(con);

 		con.setSolver(null);
 		set(ci, null);
 		m_nRealCons--;

		n = con.countVars();
		for (i = 0; i < n; i++) {
			Variable var = con.getVar(i);
			var.remove(con);

			if (var.countCons() == 0) {
				int idx = m_vars.indexOf(var);

				m_vars.setElementAt(null, idx);
				m_removedVars.addElement(new Integer(idx));
				m_nRealVars--;
			}
		}
	}

	public final void refresh()
	{
		int nVars0 = m_vars.size();
		int i, j, k;

		int nLowerVecs = m_lowerVecs.size();
		for (i = 0; i < nLowerVecs; i++) {
			LowerVec lowerVec = (LowerVec) m_lowerVecs.elementAt(i);
			if (lowerVec.cIdx.strength < N_LEVELS &&
				lowerVec.cIdx.index != -1)
				getCon(lowerVec.cIdx).setRow(-1);
			// dispose the vector
		}
		m_lowerVecs.removeAllElements();

		int[] varIndex = new int[nVars0];

		int idx = 0;
		for (i = 0; i < nVars0; i++) {
			Variable var = (Variable) m_vars.elementAt(i);
			if (var == null)
				varIndex[i] = -1;
			else {
				m_vars.setElementAt(var, idx);
				varIndex[i] = idx;
				idx++;
			}
		}
		for (i = nVars0 - 1; i >= idx; i--)
			m_vars.removeElementAt(i);
	
		for (i = 0; i < N_LEVELS; i++) {
			Vector lev = m_level[i];
			int nCons0 = lev.size();

			idx = 0;
			for (j = 0; j < nCons0; j++) {
				Constraint con = (Constraint) lev.elementAt(j);
				if (con == null)
					continue;

				lev.setElementAt(con, idx);
				idx++;

				if (con.getClass() == m_linearClass) {
					Linear lin = (Linear) con;
					Vector coefs = lin.getCoefs();
					int nCoefs = coefs.size();
					for (k = 0; k < nCoefs; k++) {
						VecElem coef = (VecElem) coefs.elementAt(k);
						coef.index = varIndex[coef.index];
					}
				}
			}

			for (j = nCons0 - 1; j >= idx; j--)
				lev.removeElementAt(j);
		}

		// dispose varIndex

		int nVars = m_vars.size();

		if (nLowerVecs == 0) {
			int len = (nVars / 256 + 1) * 256;

			m_row = new double[len];
			m_row2 = new double[len];
			m_fsValue = new double[len];
			m_bsValue = new double[len];
			m_waStr = new ConIndex[len];
		}
		else {
			int len = m_row.length;

			if (nVars > len) {
				len = (nVars / 256 + 1) * 256;

				m_row = new double[len];
				m_row2 = new double[len];
				m_fsValue = new double[len];
				m_bsValue = new double[len];
				m_waStr = new ConIndex[len];
			}
		}

		refactorize();
	}
	
    public final void plan()
	{
		int nVars = m_vars.size();
		int i;

		int len = 0;
		if (m_row != null)
			len = m_row.length;
		if (nVars > len) {
			len = (nVars / 256 + 1) * 256;

			m_row = new double[len];
			m_row2 = new double[len];
			m_fsValue = new double[len];
			m_bsValue = new double[len];
			m_waStr = new ConIndex[len];
		}

		if (m_lowerVecs.size() == 0)
			refactorize();
		else
			factorize();

		reoptimize();
	}

    public final void execute()
	{
		changeConstants();

		forwardSubstitution();
		backwardSubstitution();
	}

	//
	// refactorization
	// 

	private final void refactorize()
	{
		int nVars = m_vars.size();
		int n, i, j;

		n = m_lowerVecs.size();
		for (i = 0; i < n; i++) {
			LowerVec lowerVec = (LowerVec) m_lowerVecs.elementAt(i);
			if (lowerVec.cIdx.strength < N_LEVELS &&
				lowerVec.cIdx.index != -1) {
				Constraint con = getCon(lowerVec.cIdx);
				con.setRow(-1);
			}
		}
		m_lowerVecs.removeAllElements();
		// previous lowerVec's are now garbage

		m_upperVecs.removeAllElements();
		// previous upperVec's are now garbage

		m_inactCons.removeAllElements();
		m_addedCons.removeAllElements();
		m_deletedRows.removeAllElements();
		m_strongestDeletedRowStr.strength = N_LEVELS + 1;
		m_strongestDeletedRowStr.index = 0;
		m_removedVars.removeAllElements();
		m_firstPrefConIndex = 0;

		//
		// enforce constraints
		//

		boolean firstOptConSatisfied = false;

 		i = 0;
		ConIndex ci = new ConIndex(0, 0);
 		for (getFirst(ci); ; getNext(ci)) {
			Constraint con = null;
			ConstraintSet set = null;
			if (ci.strength > 0 &&
				ci.strength < N_LEVELS) {
				con = getCon(ci);
				set = con.getConstraintSet();
			}

			if (i == nVars &&
				(set == null ||
				 set.getStrategy() != ConstraintSet.OPTIMIZING ||
				 set.countCons() != 2 ||
				 set.index(con) == 0 ||
				 !firstOptConSatisfied))
				break;

			if (set != null) {
				if (set.getStrategy() == ConstraintSet.CONJUNCTIVE) {
					if (set.countCons() == 2 &&
						set.index(con) == 0 &&
						i + 1 == nVars) {
						getNext(ci);
						continue;
					}
				}
			}

			// set elements for the current constraint
			double[] cst = new double[1];
			setRow(m_row, cst, ci);
			
			// determine the i-th row of Ak; the first to (i-1)-th entries of
			// the i-th row of Ak is those of the i-th row of L
			multUpperVecs(m_row, m_upperVecs);

			// find column to swap
			for (j = i; j < nVars; j++)
				if (!nearEq(m_row[j], 0))
					break;

			if (j == nVars) {
				if (ci.strength < N_LEVELS) {
					boolean toAddConToInactCons = true;
					if (set != null) {
						if (set.getStrategy() == ConstraintSet.CONJUNCTIVE &&
							set.countCons() == 2) {
							if (set.index(con) == 0) {
								m_inactCons.addElement(con);
								getNext(ci);

								Constraint nextCon = set.getCon(1);
								m_inactCons.addElement(nextCon);

								toAddConToInactCons = false;
							}
							else { // set.index(con) == 1
								// recover m_upperVecs, m_lowerVecs, i
								m_upperVecs.removeElementAt(
									m_upperVecs.size() - 1);
								removeLowerVec(m_lowerVecs.size() - 1);
								i--;

								Constraint prevCon = set.getCon(0);
								m_inactCons.addElement(prevCon);
							}
						}
						else if (set.getStrategy()
								 == ConstraintSet.OPTIMIZING &&
								 set.countCons() == 2) {
							if (set.index(con) == 0)
								firstOptConSatisfied = false;
							else { // set.index(con) == 1
								if (firstOptConSatisfied) {
									// try to enforce the optimal constraint
									if (optimizeConstraintSet0(i, con)) {
										// enforced con
										Constraint prevCon = set.getCon(0);
										m_inactCons.addElement(prevCon);

										toAddConToInactCons = false;
									}
								}
							}
						}
					}					
					if (toAddConToInactCons)
						m_inactCons.addElement(con);
				}

				continue;
			}

			// the current constraint is hierarchically independent

			if (set != null &&
				set.getStrategy() == ConstraintSet.OPTIMIZING &&
				set.countCons() == 2 &&
				set.index(con) == 0)
				firstOptConSatisfied = true;

			if (ci.strength == 0)
				m_firstPrefConIndex = i + 1;

			eliminateRow(i, j, m_row, cst[0], ci);

			i++;
 		}

 		for ( ; ci.strength < N_LEVELS; getNext(ci)) {
 			Constraint con = getCon(ci);
 			m_inactCons.addElement(con);
 		}
	}
	
	final boolean optimizeConstraintSet0(int ii, Constraint con)
	{
		int nVars = m_vars.size();
		int n, m, i, j, k;

		updateWalkaboutStrengths();
							
		int str = con.getStrength();
		ConIndex cIdx = new ConIndex(str, index(str, con));

		// set elements for the current constraint
		double[] cst = new double[1];
		setRow(m_row, cst, con);
		
		ConIndex waStr = getWeakestWalkaboutStrength(nVars, m_row);
		if (waStr.strength < cIdx.strength ||
			waStr.index < cIdx.index - 1)
			// cannot enforce con
			return false;
		
		m = m_lowerVecs.size();
		for (i = 0; i < m - 1; i++)
			m_bsValue[i] = 0;
		m_bsValue[m - 1] = 1;

		for (i = m_upperVecs.size() - 1; i >= 0; i--) {
			UpperVec upperVec = (UpperVec) m_upperVecs.elementAt(i);
			int src = upperVec.srcCol;
			int dst = upperVec.dstCol;
			if (upperVec.shift) {
				double val = m_bsValue[dst];
				for (k = dst; k > src; k--)
					m_bsValue[k] = m_bsValue[k - 1];
				m_bsValue[src] = val;
			}
			else {
				double val = 0;
				n = upperVec.elems.size();
				for (k = 0; k < n; k++) {
					VecElem elem = (VecElem) upperVec.elems.elementAt(k);
					val += elem.value * m_bsValue[elem.index];
				}
				m_bsValue[upperVec.srcCol] = val;
		
				if (dst != src) {
					val = m_bsValue[src];
					m_bsValue[src] = m_bsValue[dst];
					m_bsValue[dst] = val;
				}
			}
		}

		double delta = 0;
		for (i = 0; i < nVars; i++) 
			delta += m_row[i] * m_bsValue[i];

		if (delta >= -1 &&
			delta <= 1)
			return false;

		// recover m_upperVecs, m_lowerVecs
		m_upperVecs.removeElementAt(m_upperVecs.size() - 1);
		removeLowerVec(m_lowerVecs.size() - 1);

		// enforce con by revoking the pair

		// determine the i-th row of Ak; the first to (i-1)-th entries of
		// the i-th row of Ak is those of the i-th row of L
		multUpperVecs(m_row, m_upperVecs);

		// find column to swap
		for (j = ii - 1; j < nVars; j++)
			if (!nearEq(m_row[j], 0))
				break;

		if (j == nVars)
			throw new InternalError("fatal error");

		eliminateRow(ii - 1, j, m_row, cst[0], cIdx);

		return true;
	}

	//
	// refactorization of preferential constraints
	//

	private final void refactorizePrefCons()
	{
		refactorize();
	}
	
	//
	// incremental factorization
	//

    private final void factorize()
	{
		int nVars = m_vars.size();
		int n, i, j, i0;
		
		try {
			// revoke removed constraints
			revokeRemovedCons();

			// enforce added constraints by revoking weaker ones
			enforceAddedCons();

			// enforce the weakest stay constraints
			enforceWeakestStay();
		}
		catch (IncrementalFactorizationFailedException ex) {
			// cannot factorize incrementally
			refactorizePrefCons();
		}
	}
	
	private final void revokeRemovedCons()
		throws IncrementalFactorizationFailedException
	{
		int nVars = m_vars.size();
		int n, i, j, i0;

		//
		// revoke removed constraints
		//

		boolean removed = (m_deletedRows.size() > 0);
		if (removed) {
			int nActCons = m_lowerVecs.size() - m_deletedRows.size();

			for (i = m_deletedRows.size() - 1; i >= 0; i--) {
				int row = ((Integer) m_deletedRows.elementAt(i)).intValue();
				if (row < m_firstPrefConIndex)
					m_firstPrefConIndex = row;
				m_lowerVecs.removeElementAt(row);
				// dispose the row
			}

			modifyFactorization(m_deletedRows, m_row);
		}

		m_deletedRows.removeAllElements();

		//
		// eliminate removed variables
		//

		i = m_lowerVecs.size();

		n = m_removedVars.size();
		for (i0 = 0; i0 < n; i0++) {
			int idx = ((Integer) m_removedVars.elementAt(i0)).intValue();

			// set elements for the current constraint
			double[] cst = new double[1];
			setRowForWeakestStay(m_row, cst, idx);
			
			// determine the i-th row of Ak; the first to (i-1)-th entries of
			// the i-th row of Ak is those of the i-th row of L
			multUpperVecs(m_row, m_upperVecs);
			
			// find column to swap
			for (j = i; j < nVars; j++)
				if (!nearEq(m_row[j], 0))
					break;

			ConIndex ci = new ConIndex(N_LEVELS, idx);
			eliminateRow(i, j, m_row, cst[0], ci);

			i++;
		}

		m_removedVars.removeAllElements();

		if (removed) 
			// enforce inactive constraints
			enforceInactiveCons();
	}

	private final boolean enforceAddedCons()
		throws IncrementalFactorizationFailedException
	{
		int nVars = m_vars.size();
		int n, i, j, i0;

		BackupInfo backupInfo = null;

		updateWalkaboutStrengths();

		boolean firstOptConSatisfied = false;

		for (i0 = 0; i0 < m_addedCons.size(); i0++) {
			Constraint con = (Constraint) m_addedCons.elementAt(i0);
			int str = con.getStrength();
			ConIndex cIdx = new ConIndex(str, index(str, con));

			// set elements for the current constraint
			double[] cst = new double[1];
			setRow(m_row, cst, con);

			ConIndex waStr = getWeakestWalkaboutStrength(nVars, m_row);
			if (waStr.stronger(cIdx)) {
				// cannot enforce con

				ConstraintSet set = con.getConstraintSet();

				boolean toInsertConToInactCons = true;
				if (set != null) {
					if (set.getStrategy() == ConstraintSet.CONJUNCTIVE &&
						set.countCons() == 2) {
						if (set.index(con) == 0)
							i0++;
						else { // set.index(con) == 1
							// recover previous info
							recoverLowerAndUpperVecs(backupInfo);

							updateWalkaboutStrengths();
						}

						insertToInactCons(cIdx, set);
						toInsertConToInactCons = false;
					}
					else if (set.getStrategy() == ConstraintSet.OPTIMIZING &&
							 set.countCons() == 2) {
						if (set.index(con) == 0)
							firstOptConSatisfied = false;
						else { // set.index(con) == 1
							if (firstOptConSatisfied &&
								waStr.strength == cIdx.strength &&
								waStr.index == cIdx.index - 1) {
								// try to enforce the optimal constraint
								if (optimizeConstraintSet(set)) {
									// recover previous info
									recoverLowerAndUpperVecs(backupInfo);

									updateWalkaboutStrengths();
				
									Constraint prevCon = set.getCon(0);
									insertToInactCons(cIdx, prevCon);
									toInsertConToInactCons = false;

									i0--;
								}
							}
						}
					}
				}
				if (toInsertConToInactCons)
					insertToInactCons(cIdx, con);

				continue;
			}

			ConstraintSet set = con.getConstraintSet();
			int removedRow = -1;

			if (waStr.strength <= N_LEVELS) {
				// delete victim row

				if (set != null &&
					(set.getStrategy() == ConstraintSet.CONJUNCTIVE ||
					 set.getStrategy() == ConstraintSet.OPTIMIZING) &&
					set.countCons() == 2 &&
					set.index(con) == 0) {
					// backup current info for recovery
					backupInfo = backupLowerAndUpperVecs();

					if (set.getStrategy() == ConstraintSet.CONJUNCTIVE) {
						m_strongestDeletedRowStr.strength = N_LEVELS + 1;
						m_strongestDeletedRowStr.index = 0;
					}
				}

				Vector removedRows = new Vector(8, 8);

				// examine walkabout strength
				
				if (waStr.strength < N_LEVELS) {
					Constraint revokedCon = getCon(waStr);
					removedRow = revokedCon.getRow();

					removeLowerVec(removedRow);
					// dispose the row

					ConstraintSet set1 = revokedCon.getConstraintSet();
					if (set1 != null &&
						set1.getStrategy() == ConstraintSet.CONJUNCTIVE &&
						set1.countCons() == 2) {
						m_strongestDeletedRowStr.strength = waStr.strength;
						if (set1.index(revokedCon) == 0) {
							Constraint nextCon = set1.getCon(1);
							int removedRow2 = nextCon.getRow();

							removeLowerVec(removedRow2);
							// dispose the row

							removedRows.addElement(new Integer(removedRow));
							removedRows.addElement(new Integer(removedRow2));

							m_strongestDeletedRowStr.index = waStr.index + 1;
						}
						else { // set.index(revokedCon) == 1
							Constraint prevCon = set1.getCon(0);
							int removedRow2 = prevCon.getRow();

							removeLowerVec(removedRow2);
							// dispose the row

							removedRows.addElement(new Integer(removedRow2));
							removedRows.addElement(new Integer(removedRow));

							m_strongestDeletedRowStr.index = waStr.index;
						}

						insertToInactCons(waStr, set1);
					}
					else {
						removedRows.addElement(new Integer(removedRow));

						insertToInactCons(waStr, revokedCon);
					}
				}
				else { // waStr == N_LEVELS
					// we cannot obtain removedRow because there is 
					// no corresponding constraint.

					n = m_lowerVecs.size();
					for (i = 0; i < n; i++) {
						LowerVec lowerVec = (LowerVec) m_lowerVecs.elementAt(i);
						if (lowerVec.cIdx.strength == N_LEVELS &&
							lowerVec.cIdx.index == waStr.index) {
							removedRow = i;
							break;
						}
					}

					removeLowerVec(removedRow);
					// dispose the row

					removedRows.addElement(new Integer(removedRow));
				}

				modifyFactorization(removedRows, m_row2);
			}

			// determine the i-th row of Ak; the first to (i-1)-th entries of
			// the i-th row of Ak is those of the i-th row of L
			multUpperVecs(m_row, m_upperVecs);

			// find column to swap
			int nActCons = m_lowerVecs.size();
			for (j = nActCons; j < nVars; j++)
				if (!nearEq(m_row[j], 0))
					break;

			if (j == nVars) {
				if (set != null &&
					set.getStrategy() == ConstraintSet.OPTIMIZING &&
					set.countCons() == 2 &&
					set.index(con) == 1 &&
					firstOptConSatisfied == true) {
					// try to enforce the optimal constraint
					if (optimizeConstraintSet(set)) {
						// recover previous info
						recoverLowerAndUpperVecs(backupInfo);

						Constraint revokedCon = getCon(waStr);
						m_inactCons.removeElement(revokedCon);

						Constraint prevCon = set.getCon(0);
						insertToInactCons(cIdx, prevCon);

						i0--;

						updateWalkaboutStrengths();
					}

					continue;
				}

				// cannot factorize incrementally
				throw new IncrementalFactorizationFailedException();
			}

			// the current constraint is hierarchically independent
			
			ConIndex ci = new ConIndex(str, index(str, con));
			eliminateRow(nActCons, j, m_row, cst[0], ci);
				
			updateWalkaboutStrengths();

			boolean toEnforceInactiveCons = true;
			if (set != null) {
				if (set.getStrategy() == ConstraintSet.CONJUNCTIVE &&
					set.countCons() == 2) {
					if (set.index(con) == 0)
						toEnforceInactiveCons = false;
					else // set.index(con) == 1
						// dispose bakcupInfo
						backupInfo = null;
				}
				else if (set.getStrategy() == ConstraintSet.OPTIMIZING &&
						 set.countCons() == 2) {
					if (set.index(con) == 0)
						firstOptConSatisfied = true;
					else // set.index(con) == 1
						// dispose bakcupInfo
						backupInfo = null;
				}
			}
			if (toEnforceInactiveCons &&
				m_strongestDeletedRowStr.strength <= N_LEVELS)
				// enforce inactive constraints
				enforceInactiveCons();
		}

		m_addedCons.removeAllElements();

		return false;
	}

	final boolean optimizeConstraintSet(ConstraintSet set)
	{
		int nVars = m_vars.size();
		int n, m, i, j, k;

		Constraint prevCon = set.getCon(0);
		Constraint con = set.getCon(1);

		// set elements for the current constraint
		double[] cst = new double[1];
		setRow(m_row, cst, con);
		
		int prevConRow = prevCon.getRow();

		int nActCons = m_lowerVecs.size();
		for (i = 0; i < nActCons; i++) {
			LowerVec lowerVec = (LowerVec) m_lowerVecs.elementAt(i);

			double val = 0;
			if (i == prevConRow)
				val = 1;

			n = lowerVec.elems.size();
			for (k = 0; k < n; k++) {
				VecElem elem = (VecElem) lowerVec.elems.elementAt(k);
				val -= elem.value * m_fsValue[elem.index];
			}
			m_fsValue[i] = val;
		}

		for (i = 0; i < nVars; i++)
			m_bsValue[i] = m_fsValue[i];

		for (i = m_upperVecs.size() - 1; i >= 0; i--) {
			UpperVec upperVec = (UpperVec) m_upperVecs.elementAt(i);
			int src = upperVec.srcCol;
			int dst = upperVec.dstCol;
			if (upperVec.shift) {
				double val = m_bsValue[dst];
				for (k = dst; k > src; k--)
					m_bsValue[k] = m_bsValue[k - 1];
				m_bsValue[src] = val;
			}
			else {
				double val = 0;
				n = upperVec.elems.size();
				for (k = 0; k < n; k++) {
					VecElem elem = (VecElem) upperVec.elems.elementAt(k);
					val += elem.value * m_bsValue[elem.index];
				}
				m_bsValue[upperVec.srcCol] = val;
		
				if (dst != src) {
					val = m_bsValue[src];
					m_bsValue[src] = m_bsValue[dst];
					m_bsValue[dst] = val;
				}
			}
		}

		double delta = 0;
		for (i = 0; i < nVars; i++) 
			delta += m_row[i] * m_bsValue[i];

		return delta < -1 || delta > 1;
	}

	private final void enforceInactiveCons()
		throws IncrementalFactorizationFailedException
	{
		int nVars = m_vars.size();
		int i, j, i0;

		Vector newInactCons = new Vector(8, 8);

		// progress until m_strongestDeletedRowStr
		int nInactCons = m_inactCons.size();
		for (i0 = 0; i0 < nInactCons; i0++) {
			Constraint con = (Constraint) m_inactCons.elementAt(i0);
			int str = con.getStrength();
			if (str < m_strongestDeletedRowStr.strength) {
				newInactCons.addElement(con);
				continue;
			}
			if (str == m_strongestDeletedRowStr.strength) {
				int idx = index(str, con);
				if (idx <= m_strongestDeletedRowStr.index) {
					newInactCons.addElement(con);
					continue;
				}
			}
			break;
		}
		
		m_strongestDeletedRowStr.strength = N_LEVELS + 1;
		m_strongestDeletedRowStr.index = 0;

		boolean firstConjConSatisfied = false;
		BackupInfo backupInfo = null;
		
		i = m_lowerVecs.size();
		for ( ; i0 < m_inactCons.size(); i0++) {
			Constraint con = (Constraint) m_inactCons.elementAt(i0);
			ConstraintSet set = con.getConstraintSet();

			if (i == nVars &&
				(set == null ||
				 set.getStrategy() != ConstraintSet.CONJUNCTIVE ||
				 set.countCons() != 2 ||
				 set.index(con) == 0 ||
				 !firstConjConSatisfied))
				break;

			int str = con.getStrength();

			// set elements for the current constraint
			double[] cst = new double[1];
			setRow(m_row, cst, con);
			
			// determine the i-th row of Ak; the first to (i-1)-th entries of
			// the i-th row of Ak is those of the i-th row of L
			multUpperVecs(m_row, m_upperVecs);

			// find column to swap
			for (j = i; j < nVars; j++)
				if (!nearEq(m_row[j], 0))
					break;
			
			if (j == nVars) {
				if (set != null &&
					set.getStrategy() == ConstraintSet.CONJUNCTIVE &&
					set.countCons() == 2) {
					if (set.index(con) == 0) {
						firstConjConSatisfied = false;
						newInactCons.addElement(con);
					}
					else { // set.index(con) == 1
						if (firstConjConSatisfied) {
							// try to enforce con by revoking weaker ones
							if (enforceConjunctiveCon(con)) {
								// can enforce con
							}
							else {
								// cannot enforce con
								newInactCons.addElement(con);

								// recover previous info
								recoverLowerAndUpperVecs(backupInfo);
							}
						}
						else
							newInactCons.addElement(con);
					}
				}
				else
					newInactCons.addElement(con);
			}
			else {
				// the current constraint is hierarchically independent

				boolean conjunctive = 
					(set != null &&
					 set.getStrategy() == ConstraintSet.CONJUNCTIVE &&
					 set.countCons() == 2);
				if (conjunctive) {
					if (set.index(con) == 0)
						firstConjConSatisfied = true;

					// backup current info for recovery
					backupInfo = backupLowerAndUpperVecs();
				}

				ConIndex ci = new ConIndex(str, index(str, con));
				eliminateRow(i, j, m_row, cst[0], ci);

				i++;

				if (conjunctive &&
					set.index(con) == 1) {
					if (!firstConjConSatisfied) {
						// try to enforce the pair by revoking weaker ones
						Constraint prevCon = set.getCon(0);
						if (enforceConjunctiveCon(prevCon))
							// can enforce prevCon
							newInactCons.removeElementAt(
								newInactCons.size() - 1);
						else {
							// cannot enforce prevCon
							newInactCons.addElement(con);

							// recover previous info
							recoverLowerAndUpperVecs(backupInfo);
						}
					}
				}
			}
		}

		for ( ; i0 < m_inactCons.size(); i0++)
			newInactCons.addElement(m_inactCons.elementAt(i0));

		m_inactCons = newInactCons;
	}

	final boolean enforceConjunctiveCon(Constraint con)
		throws IncrementalFactorizationFailedException
	{
		int nVars = m_vars.size();
		int n, i, j;

		updateWalkaboutStrengths();
							
		int str = con.getStrength();
		ConIndex cIdx = new ConIndex(str, index(str, con));

		// set elements for the current constraint
		double[] cst = new double[1];
		setRow(m_row, cst, con);
		
		ConIndex waStr = getWeakestWalkaboutStrength(nVars, m_row);
		if (waStr.stronger(cIdx))
			// cannot enforce con
			return false;
		
		int removedRow = -1;

		if (waStr.strength <= N_LEVELS) {
			// delete victim row

			Vector removedRows = new Vector(8, 8);

			// examine walkabout strength

			if (waStr.strength < N_LEVELS) {
				Constraint revokedCon = getCon(waStr);
				removedRow = revokedCon.getRow();

				removeLowerVec(removedRow);
				// dispose the row

				ConstraintSet set1 = revokedCon.getConstraintSet();
				if (set1 != null &&
					set1.getStrategy() == ConstraintSet.CONJUNCTIVE &&
					set1.countCons() == 2) {
					if (set1.index(revokedCon) == 0) {
						Constraint nextCon = set1.getCon(1);
						int removedRow2 = nextCon.getRow();

						removeLowerVec(removedRow2);
						// dispose the row

						removedRows.addElement(new Integer(removedRow));
						removedRows.addElement(new Integer(removedRow2));
					}
					else { // set.index(revokedCon) == 1
						Constraint prevCon = set1.getCon(0);
						int removedRow2 = prevCon.getRow();

						removeLowerVec(removedRow2);
						// dispose the row

						removedRows.addElement(new Integer(removedRow2));
						removedRows.addElement(new Integer(removedRow));
					}

					insertToInactCons(waStr, set1);
				}
				else {
					removedRows.addElement(new Integer(removedRow));

					insertToInactCons(waStr, revokedCon);
				}
			}
			else { // waStr == N_LEVELS
				// we cannot obtain removedRow because there is 
				// no corresponding constraint.

				n = m_lowerVecs.size();
				for (i = 0; i < n; i++) {
					LowerVec lowerVec = (LowerVec) m_lowerVecs.elementAt(i);
					if (lowerVec.cIdx.strength == N_LEVELS &&
						lowerVec.cIdx.index == waStr.index) {
						removedRow = i;
						break;
					}
				}

				removeLowerVec(removedRow);
				// dispose the row

				removedRows.addElement(new Integer(removedRow));
			}

			modifyFactorization(removedRows, m_row2);
		}

		// determine the i-th row of Ak; the first to (i-1)-th entries of
		// the i-th row of Ak is those of the i-th row of L
		multUpperVecs(m_row, m_upperVecs);

		// find column to swap
		int nActCons = m_lowerVecs.size();
		for (j = nActCons; j < nVars; j++)
			if (!nearEq(m_row[j], 0))
				break;

		if (j == nVars)
			// cannot factorize incrementally
			throw new IncrementalFactorizationFailedException();
		else {
			// the current constraint is hierarchically independent

			ConIndex ci = new ConIndex(str, index(str, con));
			eliminateRow(nActCons, j, m_row, cst[0], ci);
		}

		return true;
	}

	private final void enforceWeakestStay()
	{
		int nVars = m_vars.size();
		int n, i, j;

		i = m_lowerVecs.size();
		int idx = 0;
		while (i < nVars) {
			if (m_vars.elementAt(idx) == null) {
				idx++;
				continue;
			}

			n = m_lowerVecs.size();
			for (j = 0; j < n; j++) {
				LowerVec lowerVec = (LowerVec) m_lowerVecs.elementAt(j);
				if (lowerVec.cIdx.strength == N_LEVELS &&
					lowerVec.cIdx.index == idx)
					break;
			}
			if (j < n) {
				idx++;
				continue;
			}
		
			// set elements for the current constraint
			double[] cst = new double[1];
			setRowForWeakestStay(m_row, cst, idx);
			
			// determine the i-th row of Ak; the first to (i-1)-th entries of
			// the i-th row of Ak is those of the i-th row of L
			multUpperVecs(m_row, m_upperVecs);
			
			// find column to swap
			for (j = i; j < nVars; j++)
				if (!nearEq(m_row[j], 0))
					break;

			if (j < nVars) {
				// the current constraint is hierarchically independent
				
				ConIndex ci = new ConIndex(N_LEVELS, idx);
				eliminateRow(i, j, m_row, cst[0], ci);

				i++;
			}
			
			idx++;
		}
	}

	private final void reoptimize()
	{
		int nVars = m_vars.size();
		int n0, n, i0, i, k;

		n0 = m_optimizingSets.size();
		for (i0 = 0; i0 < n0; i0++) {
			ConstraintSet set = (ConstraintSet)
				m_optimizingSets.elementAt(i0);
			
			if (set.countCons() < 2)
				continue;
			
			Constraint con0 = set.getCon(0);
			int row0 = con0.getRow();

			Constraint con1 = set.getCon(1);
			int row1 = con1.getRow();

			Constraint satCon;
			Constraint unsatCon;
			if (row0 >= 0 &&
				row1 == -1) {
				satCon = con0;
				unsatCon = con1;
			}
			else if (row0 == -1 &&
					 row1 >= 0) {
				satCon = con1;
				unsatCon = con0;
			}
			else
				continue;

			// set elements for the current constraint
			double[] cst = new double[1];
			setRow(m_row, cst, unsatCon);
		
			int satConRow = satCon.getRow();

			int nActCons = m_lowerVecs.size();
			for (i = 0; i < nActCons; i++) {
				LowerVec lowerVec = (LowerVec) m_lowerVecs.elementAt(i);

				double val = 0;
				if (i == satConRow)
					val = 1;

				n = lowerVec.elems.size();
				for (k = 0; k < n; k++) {
					VecElem elem = (VecElem) lowerVec.elems.elementAt(k);
					val -= elem.value * m_fsValue[elem.index];
				}
				m_fsValue[i] = val;
			}

			for (i = 0; i < nVars; i++)
				m_bsValue[i] = m_fsValue[i];

			for (i = m_upperVecs.size() - 1; i >= 0; i--) {
				UpperVec upperVec = (UpperVec) m_upperVecs.elementAt(i);
				int src = upperVec.srcCol;
				int dst = upperVec.dstCol;
				if (upperVec.shift) {
					double val = m_bsValue[dst];
					for (k = dst; k > src; k--)
						m_bsValue[k] = m_bsValue[k - 1];
					m_bsValue[src] = val;
				}
				else {
					double val = 0;
					n = upperVec.elems.size();
					for (k = 0; k < n; k++) {
						VecElem elem = (VecElem) upperVec.elems.elementAt(k);
						val += elem.value * m_bsValue[elem.index];
					}
					m_bsValue[upperVec.srcCol] = val;
		
					if (dst != src) {
						val = m_bsValue[src];
						m_bsValue[src] = m_bsValue[dst];
						m_bsValue[dst] = val;
					}
				}
			}

			double delta = 0;
			for (i = 0; i < nVars; i++) 
				delta += m_row[i] * m_bsValue[i];

			if (delta < -1 || delta > 1) {
				refactorize();
				return;
			}
		}
	}

	//
	// primitive functions
	//

	private final void getFirst(ConIndex ci)
	{
		int n;

		ci.index = 0;

		while (ci.strength < N_LEVELS) {
			n = countCons(ci.strength);
			while (ci.index < n) {
				if (getCon(ci) != null)
					break;

				ci.index++;
			}
			if (ci.index < n)
				break;

			ci.strength++;
			ci.index = 0;
		}
	}
	
	private final void getNext(ConIndex ci)
	{
		int n;

		ci.index++;

		while (ci.strength < N_LEVELS) {
			n = countCons(ci.strength);
			while (ci.index < n) {
				if (getCon(ci) != null)
					break;

				ci.index++;
			}
			if (ci.index < n)
				break;

			ci.strength++;
			ci.index = 0;
		}
	}

	private final void modifyFactorization(Vector removedRows, double[] row)
	{
		int nVars = m_vars.size();
		int i, j, k, l;

		int nActCons = m_lowerVecs.size();
		int nUpperVecs = m_upperVecs.size();
		int nRemovedRows = 0;
		for (i = ((Integer) removedRows.elementAt(0)).intValue();
			 i < nActCons; i++) {
			while (nRemovedRows < removedRows.size()) { 
				if (i + nRemovedRows < 
					((Integer) removedRows.elementAt(nRemovedRows)).intValue())
					break;

				UpperVec upperVec = new UpperVec();
				upperVec.shift = true;
				upperVec.srcCol = i;
				upperVec.dstCol = nVars - 1;
				m_upperVecs.addElement(upperVec);

				nRemovedRows++;
			}

			LowerVec lowerVec = (LowerVec) m_lowerVecs.elementAt(i);

			if (lowerVec.elems.size() > 0) {
				k = 0;
				for (j = 0; j < nRemovedRows; j++) {
					row[nVars - nRemovedRows + j] = 0;

					int removedRow
						= ((Integer) removedRows.elementAt(j)).intValue();
					for ( ; k < lowerVec.elems.size(); k++) {
						VecElem elem = (VecElem) lowerVec.elems.elementAt(k);
						if (elem.index > removedRow)
							break;
						else if (elem.index == removedRow) {
							row[nVars - nRemovedRows + j] = elem.value;
							lowerVec.elems.removeElementAt(k);
							break;
						}
						else
							elem.index -= j;
					}
				}
				for ( ; k < lowerVec.elems.size(); k++) {
					VecElem elem = (VecElem) lowerVec.elems.elementAt(k);
					elem.index -= nRemovedRows;
				}

				int n = 0;
				k = 0;
				for (j = nUpperVecs; j < m_upperVecs.size(); j++) {
					UpperVec upperVec = (UpperVec) m_upperVecs.elementAt(j);

					if (upperVec.shift) {
						n++;
						continue;
					}
					
					int src = upperVec.srcCol;
					for ( ; k < lowerVec.elems.size(); k++) {
						VecElem elem = (VecElem) lowerVec.elems.elementAt(k);
						if (elem.index > src)
							break;
						else if (elem.index == src) {
							for (l = 1; l < upperVec.elems.size(); l++) {
								VecElem e = (VecElem)
									upperVec.elems.elementAt(l);
								row[e.index - nRemovedRows + n]
									+= elem.value * e.value;
							}
							k++;
							break;
						}
					}
					if (k == lowerVec.elems.size())
						break;
				}

				UpperVec upperVec = new UpperVec();
				upperVec.shift = false;
				upperVec.srcCol = i;
				upperVec.dstCol = i;

				VecElem elem = new VecElem(i, 1);
				upperVec.elems.addElement(elem);

				for (j = nVars - nRemovedRows; j < nVars; j++) {
					if (!nearEq(row[j], 0)) {
						elem = new VecElem(j, -row[j]);
						upperVec.elems.addElement(elem);
					}
				}

				if (upperVec.elems.size() > 1)
					m_upperVecs.addElement(upperVec);
				// otherwise dispose upperVec
			}				

			if (lowerVec.cIdx.strength < N_LEVELS)
				getCon(lowerVec.cIdx).setRow(i);
		}
	}
	
	private final void multUpperVecs(double[] row, Vector upperVecs)
	{
		int n, m, j, k;

		n = upperVecs.size();
		for (j = 0; j < n; j++) {
			UpperVec upperVec = (UpperVec) upperVecs.elementAt(j);
			int src = upperVec.srcCol;
			int dst = upperVec.dstCol;
			if (upperVec.shift) {
				double val = row[src];
				for (k = src; k < dst; k++)
					row[k] = row[k + 1];
				row[dst] = val;
			}
			else {
				if (dst != src) {
					double val = row[src];
					row[src] = row[dst];
					row[dst] = val;
				}

				double piv = row[src];
				VecElem elem0 = (VecElem) upperVec.elems.elementAt(0);
				row[src] = piv * elem0.value;
				m = upperVec.elems.size();
				for (k = 1; k < m; k++) {
					VecElem elem = (VecElem) upperVec.elems.elementAt(k);
					row[elem.index] += piv * elem.value;
				}
			}
		}
	}
	
	private final void eliminateRow(int i, int j, double[] row, double cst,
									ConIndex ci)
	{
		int nVars = m_vars.size();

		LowerVec lowerVec = new LowerVec();
		UpperVec upperVec = new UpperVec();
		if (i < m_lowerVecs.size())
		    throw new InternalError();
		else 
			m_lowerVecs.addElement(lowerVec);
		m_upperVecs.addElement(upperVec);

		// swap columns
		if (j != i) {
			double val = row[j];
			row[j] = row[i];
			row[i] = val;
		}

		// record column swap
		upperVec.shift = false;
		upperVec.srcCol = i;
		upperVec.dstCol = j;

		// determine the i-th row of Uk
		double rowiInv = 1 / row[i];
		VecElem elem = new VecElem(i, rowiInv);
		upperVec.elems.addElement(elem);
		for (j = i + 1; j < nVars; j++) {
			if (!nearEq(row[j], 0)) {
				elem = new VecElem(j, -row[j] * rowiInv);
				upperVec.elems.addElement(elem);
			}
		}

		// set lower elements
		for (j = 0; j < i; j++) {
			if (!nearEq(row[j], 0)) {
				elem = new VecElem(j, row[j]);
				lowerVec.elems.addElement(elem);
			}
		}

		// set constant
		lowerVec.cst = cst;

		// set other info
		lowerVec.cIdx = new ConIndex(ci);
		if (ci.strength < N_LEVELS) {
			Constraint con = getCon(ci);
			con.setRow(i);
		}
	}
	
	private final void updateWalkaboutStrengths()
	{
		int nVars = m_vars.size();
		int n, m, i, k;

		n = m_lowerVecs.size();
		for (i = 0; i < n; i++) {
			LowerVec lowerVec = (LowerVec) m_lowerVecs.elementAt(i);

			ConIndex waStr = new ConIndex(lowerVec.cIdx);
			if (lowerVec.cIdx.strength == N_LEVELS &&
				m_vars.elementAt(lowerVec.cIdx.index) == null) {
				waStr.strength = -1;
				waStr.index = 0;
			}
			m = lowerVec.elems.size();
			for (k = 0; k < m; k++) {
				VecElem elem = (VecElem) lowerVec.elems.elementAt(k);
				ConIndex waStr1 = m_waStr[elem.index];
				if (waStr1.weaker(waStr))
					waStr = waStr1;
			}
			m_waStr[i] = waStr;
		}
		for ( ; i < nVars; i++)
			m_waStr[i] = new ConIndex(N_LEVELS + 1, 0);
		    //m_waStr[i] = new ConIndex(-1, 0);
	
		for (i = m_upperVecs.size() - 1; i >= 0; i--) {
			UpperVec upperVec = (UpperVec) m_upperVecs.elementAt(i);
			int src = upperVec.srcCol;
			int dst = upperVec.dstCol;
			if (upperVec.shift) {
				ConIndex waStr = m_waStr[dst];
				for (k = dst; k > src; k--)
					m_waStr[k] = m_waStr[k - 1];
				m_waStr[src] = waStr;
			}
			else {
				VecElem elem0 = (VecElem) upperVec.elems.elementAt(0);
				ConIndex waStr = m_waStr[elem0.index];
				n = upperVec.elems.size();
				for (k = 1; k < n; k++) {
					VecElem elem = (VecElem) upperVec.elems.elementAt(k);
					ConIndex waStr1 = m_waStr[elem.index];
					if (waStr1.weaker(waStr))
						waStr = waStr1;
				}
				m_waStr[upperVec.srcCol] = waStr;
		
				if (dst != src) {
					waStr = m_waStr[src];
					m_waStr[src] = m_waStr[dst];
					m_waStr[dst] = waStr;
				}
			}
		}
	}

	private final ConIndex getWeakestWalkaboutStrength(int nVars, double[] row)
	{
		int i;

		ConIndex waStr = new ConIndex(-1, 0);

		for (i = 0; i < nVars; i++) {
			if (nearEq(row[i], 0))
				continue;

			ConIndex waStr1 = m_waStr[i];
			if (waStr1.weaker(waStr))
				waStr = waStr1;
		}

		return waStr;
	}

    private final void changeConstants()
	{
		int n, i;

		n = m_edits.size();
		for (i = 0; i < n; i++) {
			Edit edit = (Edit) m_edits.elementAt(i);
		
			int row = edit.getRow();
			if (row == -1)
				continue;
			
			LowerVec lVec = (LowerVec) m_lowerVecs.elementAt(row);
			lVec.cst = edit.get();
		}
	}
	
    private final void forwardSubstitution()
	{
		int nVars = m_vars.size();
		int n, i, k;

		for (i = 0; i < nVars; i++) {
			LowerVec lowerVec = (LowerVec) m_lowerVecs.elementAt(i);

			double val = lowerVec.cst;
			n = lowerVec.elems.size();
			for (k = 0; k < n; k++) {
				VecElem elem = (VecElem) lowerVec.elems.elementAt(k);
				val -= elem.value * m_fsValue[elem.index];
			}
			m_fsValue[i] = val;
		}
	}
		
    private final void backwardSubstitution()
	{
		int nVars = m_vars.size();
		int n, i, k;

		for (i = 0; i < nVars; i++)
			m_bsValue[i] = m_fsValue[i];

		for (i = m_upperVecs.size() - 1; i >= 0; i--) {
			UpperVec upperVec = (UpperVec) m_upperVecs.elementAt(i);
			int src = upperVec.srcCol;
			int dst = upperVec.dstCol;
			if (upperVec.shift) {
				double val = m_bsValue[dst];
				for (k = dst; k > src; k--)
					m_bsValue[k] = m_bsValue[k - 1];
				m_bsValue[src] = val;
			}
			else {
				double val = 0;
				n = upperVec.elems.size();
				for (k = 0; k < n; k++) {
					VecElem elem = (VecElem) upperVec.elems.elementAt(k);
					val += elem.value * m_bsValue[elem.index];
				}
				m_bsValue[upperVec.srcCol] = val;
		
				if (dst != src) {
					val = m_bsValue[src];
					m_bsValue[src] = m_bsValue[dst];
					m_bsValue[dst] = val;
				}
			}
		}

		for (i = 0; i < nVars; i++) {
			Variable var = (Variable) m_vars.elementAt(i);
			if (var != null)
				var.set(m_bsValue[i]);
		}
	}

    private final void setRow(double[] row, double[] cst, Constraint con)
	{
		Class cls = con.getClass();
		if (cls == m_stayClass)
			setRowForStay(row, cst, (Stay) con);
		else if (cls == m_editClass)
			setRowForEdit(row, cst, (Edit) con);
		else if (cls == m_linearClass)
			setRowForLinear(row, cst, (Linear) con);
		else
			throw new InternalError("fatal error");
	}
	
    private final void setRow(double[] row, double[] cst, ConIndex ci)
	{
		if (ci.strength < N_LEVELS) {
			Constraint con = getCon(ci);
			Class cls = con.getClass();
			if (cls == m_stayClass)
				setRowForStay(row, cst, (Stay) con);
			else if (cls == m_editClass) 
				setRowForEdit(row, cst, (Edit) con);
			else if (cls == m_linearClass)
				setRowForLinear(row, cst, (Linear) con);
			else 
				throw new InternalError("fatal error");
		}
		else
			setRowForWeakestStay(row, cst, ci.index);
	}
	
    private final void setRowForStay(double[] row, double[] cst, Stay stay)
	{
		int nVars = m_vars.size();
		int j;

		Variable var = stay.getVar(0);
		int idx = m_vars.indexOf(var);
                
		for (j = 0; j < nVars; j++)
			row[j] = 0;
		row[idx] = 1;
    
		cst[0] = var.get();
	}
	
    private final void setRowForEdit(double[] row, double[] cst, Edit edit)
	{
		int nVars = m_vars.size();
		int j;

		Variable var = edit.getVar(0);
		int idx = m_vars.indexOf(var);
                
		for (j = 0; j < nVars; j++)
			row[j] = 0;
		row[idx] = 1;
    
		cst[0] = edit.get();
	}
		 
    private final void setRowForLinear(double[] row, double[] cst, Linear lin)
	{
		int nVars = m_vars.size();
		int n, j;

		for (j = 0; j < nVars; j++)
			row[j] = 0;

		Vector coefs = lin.getCoefs();
		n = lin.countVars();
		for (j = 0; j < n; j++) {
			Variable var = lin.getVar(j);
			VecElem coef = (VecElem) coefs.elementAt(j);
			row[coef.index] = coef.value;
		}

		cst[0] = lin.getConst();
	}
	
    private final void setRowForWeakestStay(double[] row, double[] cst,
											int vIdx)
	{
		int nVars = m_vars.size();
		int j;

		for (j = 0; j < nVars; j++)
			row[j] = 0;
		row[vIdx] = 1;

		Variable var = (Variable) m_vars.elementAt(vIdx);
		if (var == null)
			cst[0] = 0;
		else
			cst[0] = var.get();
	}

	private final void insertToInactCons(ConIndex cIdx, Constraint con)
	{
		int n, j;

		n = m_inactCons.size();
		for (j = 0; j < n; j++) {
			Constraint c = (Constraint) m_inactCons.elementAt(j);
			if (c.getStrength() > cIdx.strength) {
				m_inactCons.insertElementAt(con, j);
				break;
			}
		}
		if (j == n)
			m_inactCons.addElement(con);
	}

	private final void insertToInactCons(ConIndex cIdx, ConstraintSet set)
	{
		int n, j, k;

		n = m_inactCons.size();
		for (j = 0; j < n; j++) {
			Constraint c = (Constraint) m_inactCons.elementAt(j);
			if (c.getStrength() > cIdx.strength) {
				for (k = 0; k < set.countCons(); k++)
					m_inactCons.insertElementAt(set.getCon(k), j + k);
				break;
			}
		}
		if (j == n)
			for (k = 0; k < set.countCons(); k++)
				m_inactCons.addElement(set.getCon(k));
	}

	private final BackupInfo backupLowerAndUpperVecs()
	{
		BackupInfo info = new BackupInfo();
	    int n, m, i, j;

		n = m_lowerVecs.size();
		for (i = 0; i < n; i++) {
			LowerVec lowerVec = (LowerVec) m_lowerVecs.elementAt(i);

			LowerVec copy = new LowerVec();
			info.lowerVecs.addElement(copy);

			copy.cIdx = new ConIndex(lowerVec.cIdx);
															 
			m = lowerVec.elems.size();
			for (j = 0; j < m; j++) {
				VecElem elem = (VecElem) lowerVec.elems.elementAt(j);
				copy.elems.addElement(new VecElem(elem));
			}

			copy.cst = lowerVec.cst;
		}
		
		info.upperVecsSize = m_upperVecs.size();

		return info;
	}

	private final void recoverLowerAndUpperVecs(BackupInfo info)
	{
		int n, i;

		n = m_lowerVecs.size();
		for (i = 0; i < n; i++) {
			LowerVec lowerVec = (LowerVec) m_lowerVecs.elementAt(i);
			if (lowerVec.cIdx.strength < N_LEVELS) {
				Constraint con = getCon(lowerVec.cIdx);
				if (con != null)
					con.setRow(-1);
			}
		}

		m_lowerVecs = info.lowerVecs;
		
		n = m_lowerVecs.size();
		for (i = 0; i < n; i++) {
			LowerVec lowerVec = (LowerVec) m_lowerVecs.elementAt(i);
			if (lowerVec.cIdx.strength < N_LEVELS) {
				Constraint con = getCon(lowerVec.cIdx);
				if (con != null)
					con.setRow(i);
			}
		}

		for (i = m_upperVecs.size() - 1; i >= info.upperVecsSize; i--)
			m_upperVecs.removeElementAt(i);
	}
	
	private final void removeLowerVec(int row)
	{
		int n, i;

		LowerVec lowerVec = (LowerVec) m_lowerVecs.elementAt(row);
		if (lowerVec.cIdx.strength < N_LEVELS) {
			Constraint con = getCon(lowerVec.cIdx);
			con.setRow(-1);
		}
		
		m_lowerVecs.removeElementAt(row);

		n = m_lowerVecs.size();
		for (i = row; i < n; i++) {
			lowerVec = (LowerVec) m_lowerVecs.elementAt(i);
			if (lowerVec.cIdx.strength < N_LEVELS) {
				Constraint con = getCon(lowerVec.cIdx);
				con.setRow(i);
			}
		}
	}

}
