//
// HiRiseDemo: A Demo Applet for the HiRise Constraint Solver
//
// Copyright (C) 1998-1999 Hiroshi HOSOBE
//
////////////////////////////////////////////////////////////////

import java.util.*;
import java.awt.*;
import java.awt.event.*;

public final class CSetDemo extends Panel
implements MouseListener, MouseMotionListener {
	
	private static final int IM_NORMAL = 0;
	private static final int IM_DRAG_OBJECT = 1;

	private final class Node extends Object {

		public Variable m_x;
		public Variable m_y;

		Node(double x, double y)
		{
			m_x = new Variable(x);
			m_y = new Variable(y);
		}
		
	}
	
	private final class CSetView extends Panel {

		void calcEndPoints(Dimension dim, 
						   Point point0, Point point1, 
						   Point end0, Point end1)
		{
 			double a, b, c;
 			double x, y;
 			int i;

			a = point1.y - point0.y;
			b = -(point1.x - point0.x);
			c = point0.x * a + point0.y * b;

 			Point[] end = new Point[4];
			for (i = 0; i < 4; i++)
				end[i] = new Point();

			int n = 0;

			if (b != 0) {
				// left
				y = c / b;
				if (y >= 0 && y <= dim.height) {
					end[n].x = 0;
					end[n].y = (int) y;
					n++;
				}
		
				// right
				y = (c - a * dim.width) / b;
				if (y >= 0 && y <= dim.height) {
					end[n].x = dim.width;
					end[n].y = (int) y;
					n++;
				}
			}

			if (a != 0) {
				// top
				x = c / a;
				if (x >= 0 && x <= dim.width) {
					end[n].x = (int) x;
					end[n].y = 0;
					n++;
				}

				// bottom
				x = (c - b * dim.height) / a;
				if (x >= 0 && x <= dim.width) {
					end[n].x = (int) x;
					end[n].y = dim.height;
					n++;
				}
			}

			if (n == 0) {
				end0.setLocation(0, 0);
				end1.setLocation(0, 0);
			}
			else if (n == 1) {
				end0.setLocation(end[0]);
				end1.setLocation(end[0]);
			}
			else if (n == 2) {
				end0.setLocation(end[0]);
				end1.setLocation(end[1]);
			}
			else if (n > 2) {
				end0.setLocation(end[0]);
				for (i = 1; i < n; i++) {
					if (end[i].x != end0.x || end[i].y != end0.y) {
						end1.setLocation(end[i]);
						break;
					}
				}
				if (i == n)
					end1.setLocation(end0);
			}
 		}

		public void paint(Graphics g)
		{
			int n, m, i, j;

			if (m_nodes == null)
				return;

			// prepare off-screen for double buffering
			Dimension d = getSize();
			Graphics og;
			if (m_offScreen == null) {
				m_offScreen = createImage(d.width, d.height);
				og = m_offScreen.getGraphics();
				og.setClip(0, 0, d.width, d.height);
			}
			else {
				og = m_offScreen.getGraphics();
				og.clearRect(0, 0, d.width, d.height);
			}

			super.paint(og);

			// draw objects on off-screen

			// draw constraining lines
			og.setColor(Color.red);
			n = m_lineData.size();
			for (i = 0; i < n; i++) {
				LineData data = (LineData) m_lineData.elementAt(i);

				Point end0 = new Point();
				Point end1 = new Point();
				calcEndPoints(d, data.m_lineEnd0, data.m_lineEnd1,
							  end0, end1);

				GraphicsUtility.drawThickLine(og, end0.x, end0.y,
											  end1.x, end1.y, 2);
			}

			// draw nodes
			n = m_nodes.size();
			for (i = 0; i < n; i++) {
				Node node = (Node) m_nodes.elementAt(i);

				Point p = new Point((int) node.m_x.get(),
									(int) node.m_y.get());
				Point rectTL = new Point(p.x - m_nodeSize / 2,
										 p.y - m_nodeSize / 2);

				if (node == m_draggedNode) {
					og.setColor(Color.green);
					og.fillRect(rectTL.x, rectTL.y, m_nodeSize, m_nodeSize);
				}
				else {
					boolean done = false;

					m = m_lineData.size();
					for (j = 0; j < m; j++) {
						LineData data = (LineData) m_lineData.elementAt(j);
						if (data.m_node == node) {
							og.setColor(Color.red);
							og.fillRect(rectTL.x, rectTL.y,
										m_nodeSize, m_nodeSize);
							done = true;
							break;
						}
					}
					
					if (!done) {
						og.setColor(Color.blue);
						og.fillRect(rectTL.x, rectTL.y,
									m_nodeSize, m_nodeSize);
					}
				}
			}
		
			// transfer off-screen image
			g.drawImage(m_offScreen, 0, 0, null);
			og.dispose();
		}	
	
		public void invalidate()
		{
			super.invalidate();
			m_offScreen = null;
		}

		public void update(Graphics g)
		{
			paint(g);
		}

	}
	
	private HiRiseDemo m_applet;
	
	private Solver m_solver;

	private Vector m_nodes;

	private Point m_lastMousePosition;
	private int m_inputMode;

	private Node m_draggedNode;
	private Edit m_editX;
	private Edit m_editY;
	private ConstraintSet m_editXnY;

	private class LineData {
		public Point m_lineEnd0;
		public Point m_lineEnd1;
		public Node m_node;
		public Linear m_line;
	};

	private Vector m_lineData;

	private int m_nodeSize = 20;

	private CSetView m_cSetView;
	private Image m_offScreen;

	private CheckboxGroup m_modeCheckGroup;
	private Checkbox m_normalCheck;
	private Checkbox m_optimizingCheck;
	private Checkbox m_conjunctiveCheck;

	CSetDemo(HiRiseDemo applet)
	{
		m_applet = applet;
		
		setLayout(new BorderLayout());
		
		Panel p = new Panel();
		add(p, BorderLayout.NORTH);
		p.setLayout(new FlowLayout(FlowLayout.LEFT));

		m_modeCheckGroup = new CheckboxGroup();

		m_normalCheck = new Checkbox("Normal", true, m_modeCheckGroup);
		p.add(m_normalCheck);

		m_optimizingCheck = new Checkbox("Optimizing", false,
										 m_modeCheckGroup);
		p.add(m_optimizingCheck);

		m_conjunctiveCheck = new Checkbox("Conjunctive", false,
										  m_modeCheckGroup);
		p.add(m_conjunctiveCheck);

		m_cSetView = new CSetView();
		add(m_cSetView, BorderLayout.CENTER);
		m_cSetView.addMouseListener(this);
		m_cSetView.addMouseMotionListener(this);
	}
	
	public void start()
	{
		m_solver = new Solver();

		m_nodes = new Vector();

		Node node = new Node(100, 100);
		m_nodes.addElement(node);

		m_lineData = new Vector();

		putNodeOnLine(320, 180, 1, -1);
		putNodeOnLine(200, 270, 8, -1);
		putNodeOnLine(400, 280, 1, 8);

		m_inputMode = IM_NORMAL;
		m_draggedNode = null;

		m_cSetView.invalidate();
		m_cSetView.repaint();
	}

	void putNodeOnLine(double x, double y, double dx, double dy)
	{
		Node node = new Node(x, y);
		m_nodes.addElement(node);

		LineData data = new LineData();
		data.m_lineEnd0 = new Point((int) x, (int) y);
		data.m_lineEnd1 = new Point((int) (x + dx), (int) (y + dy));

		data.m_node = node;
		
		data.m_line = new Linear(0);
		data.m_line.add(node.m_x, dy);
		data.m_line.add(node.m_y, -dx);
		data.m_line.setConst(x * dy - y * dx);
		m_solver.add(data.m_line);

		m_lineData.addElement(data);
	}

	void planAndExec()
	{
		m_applet.setWaitCursor();

		long t0 = System.currentTimeMillis();
		m_solver.plan();
		long t1 = System.currentTimeMillis();
		m_solver.execute();
		long t2 = System.currentTimeMillis();

		m_applet.setDefaultCursor();
		
		long planTime = t1 - t0;
		long execTime = t2 - t1;

		String stat = "plan: " + planTime + " ms, exec: " + execTime
			+ " ms (#vars: " + m_solver.countVars()
			+ ", #cons: " + m_solver.countCons() + ")";
		m_applet.showStatus(stat);
	}

	public void mouseClicked(MouseEvent ev)
	{
		// no operation
	}

	Node PickNode(Point point)
	{
		int n, i;

		n = m_nodes.size();
		for (i = 0; i < n; i++) {
			Node node = (Node) m_nodes.elementAt(i);

			int x = (int) node.m_x.get();
			int y = (int) node.m_y.get();
			int h = m_nodeSize / 2;
			if (point.x >= x - h && point.x <= x + h &&
				point.y >= y - h && point.y <= y + h)
				return node;
		}

		return null;
	}

	public void mousePressed(MouseEvent ev)
	{
		if (m_nodes == null)
			return;

		Object src = ev.getSource();
		if (src == m_cSetView) {
			Point point;
			switch (m_inputMode) {
			case IM_NORMAL:
				point = ev.getPoint();
				m_draggedNode = PickNode(point);
				if (m_draggedNode != null) {
					m_inputMode = IM_DRAG_OBJECT;

					m_editX = new Edit(1);
					m_editX.add(m_draggedNode.m_x, point.x);
		
					m_editY = new Edit(1);
					m_editY.add(m_draggedNode.m_y, point.y);
		
					int mode = ConstraintSet.NORMAL;
					Checkbox sel = m_modeCheckGroup.getSelectedCheckbox();
					if (sel == m_optimizingCheck)
						mode = ConstraintSet.OPTIMIZING;
					else if (sel == m_conjunctiveCheck)
						mode = ConstraintSet.CONJUNCTIVE;

					m_editXnY = new ConstraintSet(mode);
					m_editXnY.add(m_editX);
					m_editXnY.add(m_editY);
					m_solver.add(m_editXnY);

					m_cSetView.repaint();

					planAndExec();
				}
				break;
			case IM_DRAG_OBJECT:
				m_inputMode = IM_NORMAL;
				break;
			}
		}
	}
	
	public void mouseReleased(MouseEvent ev)
	{
		if (m_nodes == null)
			return;

		Object src = ev.getSource();
		if (src == m_cSetView) {
			switch (m_inputMode) {
			case IM_NORMAL:
				break;
			case IM_DRAG_OBJECT:
				m_inputMode = IM_NORMAL;

				m_solver.remove(m_editXnY);		

				planAndExec();
				
				m_draggedNode = null;

				m_cSetView.repaint();
				break;
			}
		}
	}

	public void mouseEntered(MouseEvent ev)
	{
		// no operation
	}

	public void mouseExited(MouseEvent ev)
	{
		// no operation
	}
	
	public void mouseDragged(MouseEvent ev)
	{
		mouseMoved(ev);
	}
		
    public void mouseMoved(MouseEvent ev)
	{
		if (m_nodes == null)
			return;

		m_lastMousePosition = ev.getPoint();

		if (!m_cSetView.contains(m_lastMousePosition))
			return;

		Object src = ev.getSource();
		if (src == m_cSetView) {
			switch (m_inputMode) {
			case IM_NORMAL:
				break;
			case IM_DRAG_OBJECT:
				m_editX.set((int) m_lastMousePosition.x);
				m_editY.set((int) m_lastMousePosition.y);
				m_solver.execute();

				m_cSetView.repaint();
				break;
			}
		}
	}

}
