/* %%%                                      %%%
 * %%%   Vutility.c                         %%%
 * %%%   Copyright (C) 1999 Fumio Mizoguchi %%%
 * %%%                                      %%%
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>

#undef NULL
#include <xil/xil.h>
#include "Vutility.h"
#include <xil/xil.h>
#undef NULL
#include "img.h"

#define NULL 0

/***** FUNCTION PROTOTYPE *****/
void get_std_dither_info(XilSystemState, float*, float*, 
			 XilDitherMask*, XilLookup*, XilLookup*);
void get_dither_info(XilSystemState, float*, float*,
		XilDitherMask*, XilLookup*, XilLookup*);

void init_cmap(XilLookup, Display*, Window, int);


/****** EXTERNALDATA *****/
int dither_version = 1;
double dither_gamma = 2.0;


Atom closedownAtom;
Display *xdisplay;

int grayscale = 0;
XilImage rtvc_image, rtvc_luma, display;
XilImage rescaled_image, scaled_image, single_band;
float scale_factor;
XilMemoryStorage storage[1];
int display_depth;
XilLookup graylookup;
Xil_boolean nodisplay = FALSE;
int doscale = 4;
float rescale_values[3];
float offset_values[3];
XilLookup colorcube;
XilDitherMask mask;

/***** VISION_SETUP FUNCTION ******************************************/
void vision_setup(int argc, char **argv, int grayorcolor, int scale)
{
    /***** INTERNALDATA *****/
    XilSystemState state;
    unsigned int width, height, nbands;
    int child_width = 0;
    int child_height = 0;
    int winx = 0;
    int winy = 0;
    int xyflag = 0;
    XilDataType datatype;
    Window xwindow;
    XEvent event;
    XilLookup yuv_to_rgb;
    XilColorspace cspace_ycc601;
    XilColorspace cspace_rgb709;
    int image_skip = 0;
    int max_buffers = 1;
    double t;
    struct timeval t0, t1;
    int value;
    char *type;
    int port = 1;
    Xil_boolean error_out = FALSE;
    char *devname = "/dev/rtvc0";
    int c,i,j,k,Mode = 1;
    int frames;
    char data[100];
    
    /***** PROCESS *****/
    grayscale = grayorcolor;
    doscale = scale;

    state = xil_open();
    if (state == NULL) {
	fprintf(stderr, "unable to open xil library\n");
	exit(1);
    }
    
    {
	XilDevice device;
	
	if (!(device = xil_device_create(state, "SUNWrtvc"))) {
	    fprintf(stderr, "Unable to create a device object\n");
	    xil_close(state);
	    exit(1);
	}
	
	xil_device_set_value(device, "DEVICE_NAME", (void *) globals.device);
	xil_device_set_value(device, "PORT_V", (void *) globals.sv_port);
	if (image_skip)
	  xil_device_set_value(device, "IMAGE_SKIP", (void *) image_skip);
	if (max_buffers >= 0)
	  xil_device_set_value(device, "MAX_BUFFERS", (void *) max_buffers);
	
	if (!(rtvc_image = xil_create_from_device(state, "SUNWrtvc",
						  device))) {
	    fprintf(stderr, "failed to open SUNWrtvc device\n");
	    xil_close(state);
	    exit(1);
	}

	xil_device_destroy(device);
    }





    {
	int format;
	while(format == 0){
	    xil_get_device_attribute(rtvc_image, "FORMAT_V", (void **) &format);
	}
	if (format == 0) {
	    fprintf(stderr, "Unknown video format, exiting.\n");
	    xil_close(state);
	    exit(1);
	}
    }


    xil_get_info(rtvc_image, &width, &height, &nbands, &datatype);

    /* check for width or height too big */
    if (child_width > width || child_height > height) {
	fprintf(stderr, "Maximum width,height = %d,%d\n", width, height);
	xil_close(state);
	exit(1);
    }


    if ((child_width) || (child_height)) {
	if (!child_width)
	    child_width = width;
	if (!child_height)
	    child_height = height;
	if (!(rtvc_image = xil_create_child(rtvc_image,
					    (width - child_width) / 2,
					    (height - child_height) / 2,
					    child_width, child_height,
					    0, nbands))) {
	    fprintf(stderr, "failed to open child image on SUNWrtvc\n");
	    xil_close(state);
	    exit(1);
	}

	xil_get_info(rtvc_image, &width, &height, &nbands, &datatype);
    }


    /* GRAY */
    rtvc_luma = xil_create_child(rtvc_image, 0, 0, width, height, 0, 1);
    
    /* COLOR */
    rescaled_image = xil_create(state, width, height, nbands, XIL_BYTE);
    
    if (doscale) {
	scale_factor = 1.0 / doscale;
	width = (int) (((float) width * scale_factor) + 0.5);
	height = (int) (((float) height * scale_factor) + 0.5);
	
	/* COLOR */
	scaled_image = xil_create(state, width, height, nbands, XIL_BYTE);
    }
    else {
	scale_factor = 1.0;
    }

    /* GRAY */
    single_band = xil_create(state, width, height, 1, XIL_BYTE);

    if (nodisplay) {
	/* for simplicity's sake, the 'nodisplay' option assumes a 1-banded */
	/* XIL_BYTE destination image */
	display = xil_create(state, width, height, 1, XIL_BYTE);
	display_depth = 8;
    }
    else {
	XSizeHints hints;
	int port;
	char *format;
	char titlebar[1024];

	/* xlib window creation */
	xdisplay = XOpenDisplay(NULL);
	if (!xdisplay) {
	    fprintf(stderr, "Unable to connect to server\n");
	    xil_close(state);
	    exit(1);
	}
	display_depth = DefaultDepth(xdisplay, DefaultScreen(xdisplay));
	xwindow = XCreateSimpleWindow(xdisplay, DefaultRootWindow(xdisplay),
				      winx, winy, width, height, 0, 0, 0);
	if (!xwindow) {
	    fprintf(stderr, "Unable to create window\n");
	    xil_close(state);
	    exit(1);
	}

	XInfo(xdisplay,xwindow);

	if (xyflag) {
	    hints.flags = USPosition;
	    hints.x = winx;
	    hints.y = winy;
	}
	else {
	    hints.flags = 0;
	}
	xil_get_device_attribute(rtvc_image, "FORMAT", (void **) &format);
	xil_get_device_attribute(rtvc_image, "PORT_V", (void **) &port);
	sprintf(titlebar, "%s: Port %d [%s]",
		(devname ? devname : "/dev/rtvc0"), port, format);

	XSetStandardProperties(xdisplay, xwindow, titlebar, "rtvc_display", 0,
			       argv, argc, &hints);

	/* if the user tries to shut down the window using the menu */
	if (closedownAtom = XInternAtom(xdisplay, "WM_DELETE_WINDOW", False))
	    XSetWMProtocols(xdisplay, xwindow, &closedownAtom, 1);

	/* list window events used */
	XSelectInput(xdisplay, xwindow, ExposureMask | ButtonPressMask);

	/* make the window visible */
	XMapWindow(xdisplay, xwindow);

	/* wait for the window to be mapped (an Expose event) */
	do
	    XNextEvent(xdisplay, &event);
	while (event.xany.type != Expose);

	display = xil_create_from_window(state, xdisplay, xwindow);
    }


    /* $B%G%#%9%W%l%$4XO"(B */
    if (display_depth == 8) {
	if (grayscale) {
	    if (!nodisplay) {
		XilLookup grayramp;
		int num_entries = 256;
		int i;

		Xil_unsigned8 *graydata = malloc(3 * num_entries);

		for (i = 0; i < num_entries; i++) {
		    graydata[i * 3 + 2] = graydata[i * 3 + 1] = graydata[i * 3] = i;
		}
		grayramp = xil_lookup_create(state, XIL_BYTE, XIL_BYTE,
					     3, num_entries, 0,
					     graydata);
		init_cmap(grayramp, xdisplay, xwindow, 0);
	    }
	}
	else {
	    get_dither_info(state, rescale_values, offset_values, &mask,
			    &yuv_to_rgb, &colorcube);
	    if (!nodisplay) {
		init_cmap(yuv_to_rgb, xdisplay, xwindow, -1);
	    }
	}
    }
    else if (display_depth == 24) {
	if (grayscale) {
	    int i, t;
	    Xil_unsigned8 *data = malloc(256 * 3);

	    if (data == NULL) {
		fprintf(stderr, "Out of memory\n");
		xil_close(state);
		exit(1);
	    }
	    for (i = 0, t = 0; i < 256; i++, t += 3) {
		data[t] = i;
		data[t + 1] = i;
		data[t + 2] = i;
	    }
	    graylookup = xil_lookup_create(state, XIL_BYTE, XIL_BYTE,
					   3, 256, 0, data);
	}
	else {
	    cspace_ycc601 = xil_colorspace_get_by_name(state, "ycc601");
	    cspace_rgb709 = xil_colorspace_get_by_name(state, "rgb709");
	    if (doscale)
		xil_set_colorspace(scaled_image, cspace_ycc601);
	    else
		xil_set_colorspace(rtvc_image, cspace_ycc601);
	    xil_set_colorspace(display, cspace_rgb709);
	}
    }
    else {
	fprintf(stderr, "visual depth of %d is not supported\n", display_depth);
	xil_close(state);
	exit(1);
    }

    frames = 0;
    

    /* GRAY */
    if(XIL_FAILURE == xil_export(single_band)){
	printf("------XIL_EXPORT ERROR\n");
    }

    if(0 == xil_get_exported(single_band) || 
       -1 == xil_get_exported(single_band)){
	printf("------XIL_GET_EXPORTED ERROR\n");
    }

    /* COLOR */
    if(XIL_FAILURE == xil_export(scaled_image)){
	printf("------XIL_EXPORT ERROR\n");
    }
    
    if(0 == xil_get_exported(scaled_image) || 
       -1 == xil_get_exported(scaled_image)){
	printf("------XIL_GET_EXPORTED ERROR\n");
    }
}
/***** END OF VISION_SETUP FUNCTION ***********************************/



/***** GET_VISION_DATA FUNCTION ***************************************/
unsigned char *get_vision_data(int flg)
{
    /***** INTERNALDATA *****/
    
    /***** PROCESS *****/

    /* GRAY */
    if(flg == GRAY){
	xil_scale(rtvc_luma, single_band, "nearest",
		  scale_factor, scale_factor);
	xil_get_memory_storage(single_band, storage);
    }

    /* COLOR */
    else{
	xil_scale(rtvc_image, scaled_image, "nearest",
		  scale_factor, scale_factor);
	xil_get_memory_storage(scaled_image, storage);
    }
    
    return storage->byte.data;
}
/***** END OF GET_VISION_DATA FUNCTION ********************************/


/***** SET_VISION_DATA FUNCTION ***************************************/
void set_vision_data(unsigned char *v_data)
{
    /***** INTERNALDATA *****/
    
    /***** PROCESS *****/
    /* GRAY */
    if (grayscale) {
	xil_set_memory_storage(single_band, storage);
	if (display_depth == 24) {
	    xil_lookup(single_band, display, graylookup);
	    xil_toss(single_band);
	}
	else {		/* display_depth is 8 */
	    if (!nodisplay){	    
		xil_copy(single_band, display);
	    }
	    else{
		xil_sync(display);
	    }
	}
    }
    
    /* COLOR */
    else{
	if (display_depth == 24) {
	    xil_set_memory_storage(scaled_image, storage);
	    if (doscale) {
		xil_color_convert(scaled_image, display);
		xil_toss(scaled_image);
	    }
	    else {
		xil_color_convert(rtvc_image, display);
	    }
	}
	else {		/* display_depth is 8 */
	    
	    /* $BFC$K$O$$$i$J$$$i$7$$!%(B*/
	    /*xil_rescale(rtvc_image, rescaled_image, rescale_values,
			offset_values);*/
	    if (doscale) {
		/*xil_scale(rescaled_image, scaled_image, "nearest",
			  scale_factor, scale_factor);*/
		xil_ordered_dither(scaled_image, display, colorcube, mask);
		xil_toss(scaled_image);
	    }
	    else {
		xil_ordered_dither(rescaled_image, display,
				   colorcube, mask);
	    }
	    if (nodisplay)
	      xil_sync(display);
	    xil_toss(rescaled_image);
	}
    }
}
/***** END OF SET_VISION_DATA FUNCTION ********************************/



void 
get_std_dither_info(XilSystemState state,
		    float *rescale_values,
		    float *offset_values,
		    XilDitherMask * mask,
		    XilLookup * yuv_to_rgb,
		    XilLookup * colorcube)
{
    rescale_values[0] = 255.0 / (235.0 - 16.0);
    rescale_values[1] = 255.0 / (240.0 - 16.0);
    rescale_values[2] = 255.0 / (240.0 - 16.0);
    offset_values[0] = -16.0 * rescale_values[0];
    offset_values[1] = -16.0 * rescale_values[1];
    offset_values[2] = -16.0 * rescale_values[2];

    /* get a standard dither mask */
    *mask = xil_dithermask_get_by_name(state, "dm443");

    /*
     * Get a standard lookup which is an 8x5x5 colorcube whose values have
     * been colorspace-transformed from YUV to RGB, install it into the X
     * colormap, and adjust the lookup offset based on the starting index of
     * the allocated colors in the X colormap.
     */
    *yuv_to_rgb = xil_lookup_get_by_name(state, "yuv_to_rgb");

    /*
     * Get a standard 8x5x5 colorcube to use with xil_ordered_dither, and
     * copy the lookup offset from yuv_to_rgb to this colorcube.
     */
    *colorcube = xil_lookup_get_by_name(state, "cc855");
    xil_lookup_set_offset(*colorcube, xil_lookup_get_offset(*yuv_to_rgb));

}



void 
get_dither_info(XilSystemState state,
		float *rescale_values,
		float *offset_values,
		XilDitherMask * mask,
		XilLookup * yuv_to_rgb,
		XilLookup * colorcube)
{
    int min[3];
    int max[3];
    int i;

    if (dither_version == 0) {
	get_std_dither_info(state, rescale_values, offset_values, mask,
			    yuv_to_rgb, colorcube);
	return;
    }


    /* get a standard dither mask */
    *mask = xil_dithermask_get_by_name(state, "dm443");

    /*
     * Get a standard 8x5x5 colorcube to use with xil_ordered_dither, and
     * copy the lookup offset from yuv_to_rgb to this colorcube.
     */
    *colorcube = xil_lookup_get_by_name(state, "cc855");

    /*
     * experimentally derived numbers that ignore highest saturation values
     * that are unlikely to occur in order to get a higher density of
     * displayable values in the range that is more likely to occur
     */
    min[0] = 27;
    max[0] = 200;
    min[1] = 72;
    max[1] = 175;
    min[2] = 78;
    max[2] = 168;

    for (i = 0; i < 3; i++) {
	rescale_values[i] = 255.0 / (float) (max[i] - min[i]);
	offset_values[i] = -min[i] * rescale_values[i];
    }

    /*
     * colorconvert the colorcube into a yuv_to_rgb lookup that can be
     * installed as the colormap and used to do the yuv_to_rgb conversion.
     */
    {
	XilColorspace src_cspace = xil_colorspace_get_by_name(state,
							      "ycc601");
	XilColorspace dst_cspace = xil_colorspace_get_by_name(state,
							      "rgb709");
	unsigned int count, buf_size, i;
	unsigned int table_offset;
	Xil_unsigned8 *buffer;
	XilImage image_src;
	XilImage image_dst;
	Xil_unsigned8 *data_ptr;
	XilMemoryStorage storage;
	float scale[3];
	float offset[3];
	double gamma;

	*yuv_to_rgb = xil_lookup_create_copy(*colorcube);

	for (i = 0; i < 3; i++) {
	    scale[i] = (float) (max[i] - min[i]) / 255.0;
	    offset[i] = (float) min[i];
	}

	count = xil_lookup_get_num_entries(*colorcube);
	buf_size = 3 * count * sizeof(Xil_unsigned8);

	/* allocate the values buffer */
	buffer = (Xil_unsigned8 *) malloc(buf_size);

	/* Copy the current values from the look-up table into the buffer */
	table_offset = xil_lookup_get_offset(*colorcube);
	xil_lookup_get_values(*colorcube, table_offset, count, buffer);

	/*
	 * Now create 2 images with width == number of table entries height
	 * == 1 datatype == Xil_unsigned8 nbands == 3 Set the colorspace
	 * attributes of the images.
	 */
	image_src = xil_create(state, count, (unsigned int) 1,
			       (unsigned int) 3, XIL_BYTE);
	image_dst = xil_create(state, count, (unsigned int) 1,
			       (unsigned int) 3, XIL_BYTE);
	xil_set_colorspace(image_src, src_cspace);
	xil_set_colorspace(image_dst, dst_cspace);


	/*
	 * copy the lookup table data into the image.
	 */
	xil_export(image_src);
	xil_get_memory_storage(image_src, &storage);
	data_ptr = storage.byte.data;

	/*
	 * gamma correct the y values
	 */
	gamma = 1.0 / dither_gamma;
	for (i = 0; i < buf_size; i += 3) {
	    double tmp = ((double) buffer[i]) / 255.0;

	    tmp = pow(tmp, gamma) * 255.0;
	    if (tmp > 255)
		buffer[i] = 255;
	    else
		buffer[i] = (Xil_unsigned8) tmp;
	}

	for (i = 0; i < buf_size; i += 3) {
	    data_ptr[0] = buffer[i];
	    data_ptr[1] = buffer[i + 1];
	    data_ptr[2] = buffer[i + 2];
	    data_ptr += storage.byte.pixel_stride;
	}
	xil_import(image_src, TRUE);

	xil_rescale(image_src, image_src, scale, offset);
	xil_color_convert(image_src, image_dst);

	xil_export(image_dst);
	xil_get_memory_storage(image_dst, &storage);
	data_ptr = storage.byte.data;
	for (i = 0; i < buf_size; i += 3) {
	    buffer[i] = data_ptr[0];
	    buffer[i + 1] = data_ptr[1];
	    buffer[i + 2] = data_ptr[2];
	    data_ptr += storage.byte.pixel_stride;
	}
	xil_lookup_set_values(*yuv_to_rgb, table_offset, count, buffer);

	xil_destroy(image_src);
	xil_destroy(image_dst);
	free(buffer);

    }
}

#define CMAPSIZE        256
#define TOP2 		  2	/* reserve the top two entries of the
				 * colormap to reduce colormap flashing */
void
init_cmap(XilLookup xil_cmap, Display * display, Window window, int offset)
{
    unsigned long junk[CMAPSIZE], pixels[CMAPSIZE], mask;
    XColor cdefs[CMAPSIZE];
    Colormap rcmap;
    int cmapsize;
    int i;
    Xil_unsigned8 cmap_data[CMAPSIZE * 3];
    Xil_unsigned8 *ptr;

    rcmap = XCreateColormap(display, window,
			    DefaultVisual(display, DefaultScreen(display)),
			    AllocNone);

    cmapsize = xil_lookup_get_num_entries(xil_cmap);

    /* determine the offset for the colormap */
    if (offset < 0) {
	offset = 256 - cmapsize - TOP2;
	if (offset < 0)
	    offset = 0;		/* in case cmapsize >= 255 */
    }

    if (offset) {
	if (!XAllocColorCells(display, rcmap, 0, &mask, 0, junk, offset)) {
	    fprintf(stderr, "XAlloc1 failed\n");
	}
    }

    if (!XAllocColorCells(display, rcmap, 0, &mask, 0, pixels, cmapsize)) {
	fprintf(stderr, "XAlloc2 failed\n");
    }

    /* free the unused colors in the front */
    if (offset) {
	XFreeColors(display, rcmap, junk, offset, 0);
    }

    for (i = 0; i < cmapsize; i++) {
	cdefs[i].pixel = i + offset;
    }

    xil_lookup_get_values(xil_cmap, xil_lookup_get_offset(xil_cmap),
			  cmapsize, cmap_data);

    ptr = cmap_data;
    for (i = 0; i < cmapsize; i++) {
	cdefs[i].flags = DoRed | DoGreen | DoBlue;

	/*
	 * since 24-bit XIL images are in BGR order, colormaps are also in
	 * BGR order
	 */
	cdefs[i].blue = *ptr++ << 8;
	cdefs[i].green = *ptr++ << 8;
	cdefs[i].red = *ptr++ << 8;
    }
    XStoreColors(display, rcmap, cdefs, cmapsize);

    /*
     * This will cause the colormap to be installed unless the cursor is
     * moved to another window -- any other window; if this happens, then
     * colormap flashing may occur.
     */
    XSetWindowColormap(display, window, rcmap);
    XInstallColormap(display, rcmap);
    XSync(display, False);


    /* set the offset of the XilLookup */
    xil_lookup_set_offset(xil_cmap, offset);
}
