// Copyright (C) 1997  Toyoda Masashi (toyoda@is.titech.ac.jp)

#include <amulet.h>
#include OBJECT_ADVANCED__H
#include INTER_ADVANCED__H

#if defined(SOLARIS)
extern "C" void bzero(void *, int);
#endif

#include <strstream.h>
#include <fstream.h>
#include <string.h>
#include <stdlib.h>
#include <limits.h>
#include <math.h>
#include <unistd.h>

#define _INTERNAL_
#include "ToyWidgets.h"
#undef _INTERNAL_

#include "Arrow.h"
#include "Formulas.h"
#include "PopUpMenu.h"



// ------------------------------------------------------------
// Align and Zoom Group
// ------------------------------------------------------------

// Global Objects
// ----------------------------------------
Am_Object AlignZoomGroup = 0;
Am_Object SequenceAlignZoomGroup = 0;
Am_Object AlignZoomSelectionWidget = 0;

int denom = 100000;

// Public Slots
// ----------------------------------------
DEFINE_SLOT(propagateParent); // used
DEFINE_SLOT(propagateChildren); // used
DEFINE_SLOT(hyperLinks);

DEFINE_SLOT(alignThreshold); // used
DEFINE_SLOT(minimumSpace); // used
DEFINE_SLOT(maximumPercentage); // used
DEFINE_SLOT(detailThreshold); 

DEFINE_SLOT(seqDir); // used
DEFINE_SLOT(partMoved); // used

// Private Slots
// ----------------------------------------
DEFINE_SLOT(xScaleFactor);
DEFINE_SLOT(yScaleFactor);
DEFINE_SLOT(origWidth);
DEFINE_SLOT(origHeight);
DEFINE_SLOT(gridObject);
DEFINE_SLOT(cellObject);

DEFINE_SLOT(editPointer);
DEFINE_SLOT(origPointer);
DEFINE_SLOT(currPointer);
DEFINE_SLOT(sortedHistory);

DEFINE_SLOT(undoButtons);
DEFINE_SLOT(undoFeedback);
DEFINE_SLOT(parentList);
DEFINE_SLOT(undoMode);
DEFINE_SLOT(grown);

DEFINE_SLOT(selectableObjects);

// For changing behavior
DEFINE_SLOT(logActive);
DEFINE_SLOT(predictionActive);

#define PartGrowing	0
#define PartMoving	1
#define NoChangeInside	2
#define NothingHappened	3

#define ORIG_SIZE 0
#define EDIT_SIZE 1

// Utility Functions
// ----------------------------------------

int index(Am_Value_List list, int value)
  // list$B$N$J$+$G(Bvalue$B$,2?HVL\$K$"$k$+D4$Y$k!#(B0$B$+$i!#(B
{
  int idx = 0;
  for (list.Start(); !list.Last(); list.Next(), ++idx)
    if ((int)list.Get() == value) return idx;
  return -1;
}

bool equal_size(Am_Inter_Location s1, Am_Inter_Location s2)
  // $B%5%$%:$,F1$8$+$I$&$+(B
{
    int x1, y1, w1, h1;
    s1.Get_Points(x1, y1, w1, h1);

    int x2, y2, w2, h2;
    s2.Get_Points(x2, y2, w2, h2);

    return (w1 == w2) && (h1 == h2);
}

bool is_smaller_size(Am_Inter_Location s1, Am_Inter_Location s2)
  // s1$B$N%5%$%:$,(Bs2$B$h$j>.$5$$$+$I$&$+(B
{
    int x1, y1, w1, h1;
    s1.Get_Points(x1, y1, w1, h1);

    int x2, y2, w2, h2;
    s2.Get_Points(x2, y2, w2, h2);

    return (w1 * h1) < (w2 * h2);
}

int area(Am_Inter_Location loc)
  // $BLL@Q$r7W;;$9$k(B
{
    int x1, y1, w1, h1;
    loc.Get_Points(x1, y1, w1, h1);

    return w1 * h1;
}

Am_Object Content_Instance_Of(Am_Object group, int x, int y, Am_Object ref, Am_Value_List protos)
  // group$B$r%k!<%H$H$7$F!"(B(x,y,ref)$B$N0LCV$K$"$k%*%V%8%'%/%H$N$&$A(B
  // protos$B$K4^$^$l$k%$%s%9%?%s%9$N$b$N$rC5$9!#(B
{
  // cout << "start content instance of" << endl;
  Am_Object found = Am_No_Object;

  for (Am_Object current = group; current.Get_Slot_Type(propagateChildren); ) {
    Am_Object parts_group = current.Get(propagateChildren);
    current = Am_Point_In_Part (parts_group, x, y, ref, false, true);

    // cout << "found" << current << endl;
    if (current.Valid()) {
      for (protos.Start(); !protos.Last(); protos.Next()) {
	Am_Object proto = protos.Get();
	if (current.Is_Instance_Of(proto)) found = current;
      }
    }
  }
  //  cout << "returning" << found << endl;
  return found;
}


void set_size_by_mode(Am_Object obj, int mode, Am_Inter_Location size)
  // $B%5%$%:$N%;%C%H(B
{
  Am_Inter_Location orig_size = obj.Get(origPointer);
  Am_Inter_Location edit_size = obj.Get(editPointer);

  if (mode == ORIG_SIZE) {
    //    cout << obj << " Set SMALL from " << orig_size << " to " << size; 
    obj.Set(origPointer, size);
    if (is_smaller_size(edit_size, size)) {
      obj.Set(editPointer, size);
      //      cout << ", change also LARGE from " << edit_size; 
    }
  } else {
    //    cout << obj << " Set LARGE from " << edit_size << " to " << size; 
    obj.Set(editPointer, size);
    if (is_smaller_size(size, orig_size)) {
      obj.Set(origPointer, size);
      //      cout << ", change also SMALL from " << orig_size; 
    }
  }
  //  cout << endl;
}

bool is_small_size(Am_Object obj)
{
  Am_Inter_Location orig_size = obj.Get(origPointer);
  Am_Inter_Location curr_size = obj.Get(currPointer);

  return (equal_size(orig_size, curr_size));
}

bool is_large_size(Am_Object obj)
{
  Am_Inter_Location edit_size = obj.Get(editPointer);
  Am_Inter_Location curr_size = obj.Get(currPointer);

  return (equal_size(edit_size, curr_size));
}



// grid, cell, interval objects in C++
//
// Am_Group --> [obj1,  obj2,  obj3]
//   ^            ^      ^      ^
//   |            |      |      |
//   v            v      v      v
// grid     --> [cell1, cell2, cell3]
//          --> [wall,  wall,  wall,  wall]
//              [wall,  cell1, cell2, wall]
//              [wall,  cell3, wall,  wall]
//              [wall,  wall,  wall,  wall]
//          -->     [intx1, intx2, intx3]
//          -->     [inty1, inty2, inty3]


// ----------------------------------------------------------------------
// CLASS cell
// ----------------------------------------------------------------------
class cell {
public:
  Am_Object obj;
  int x;
  int y;
  int w;
  int h;
  int origX;
  int origY;
  int origW;
  int origH;
  int origMaxW; // part$B$,0l$D$N$H$-MQ(B
  int origMaxH; //
  bool isMoved;
  bool isGrown;
  bool visible;
  bool origVisible;

  cell() : obj(Am_No_Object),
    x(0), y(0), w(0), h(0),
    origX(0), origY(0), origW(0), origH(0),
    origMaxW(0), origMaxH(0), isMoved(false), isGrown(false),
    visible(true), origVisible(true) {}  

  cell(Am_Object object, Am_Object group)
  {
    obj = object;

    int xscale = group.Get(xScaleFactor);
    int yscale = group.Get(yScaleFactor);

    int ox = object.Get(Am_LEFT);
    int oy = object.Get(Am_TOP);
    int ow = object.Get(Am_WIDTH);
    int oh = object.Get(Am_HEIGHT);

    x = ox * denom / xscale;
    y = oy * denom / yscale;
    w = ow;
    h = oh;
    origX = x * denom / xscale;
    origY = y * denom / yscale;
    origW = ow;
    origH = oh;
    origMaxW = ow;
    origMaxH = oh;
    isMoved = true;
    isGrown = false;
    visible = (bool)object.Get(Am_VISIBLE);
    origVisible = visible;
  }

  void show()
  {
    cout << "[" << origW << "," << origH << "]";
  }
};


// ----------------------------------------------------------------------
// CLASS interval
// ----------------------------------------------------------------------
class interval {
public:
  int  index;
  int  dir;	// 0: x,  1: y
  bool shrinkable;
  bool shrink;
  int  required;
  int  shrinked;
  int  realspace;

  interval() : index(0), dir(0), shrinkable(false), shrink(false),
               required(0), shrinked(0), realspace(0) {}

  bool operator>(interval& x)
  {
    return (this->required - this->shrinked) > (x.required - x.shrinked);
  }
};


// ----------------------------------------------------------------------
// CLASS space_distributer
// ----------------------------------------------------------------------
class space_distributer {
public:
  void distribute_required_space(interval* intvls, int size)
  {
    for (int i = 0; i < size - 1; ++i) {
      if (intvls[i].shrink)
	intvls[i].realspace = intvls[i].shrinked;
      else
	intvls[i].realspace = intvls[i].required;
    }
  }

  virtual void distribute_space(interval* intvls, int size, int req_length, int length) = 0;
};

class equal_distributer : public space_distributer {
public:
  void distribute_space(interval* intvls, int size, int req_length, int length)
  {
    int space = length - req_length;
    int average = length;
    if (size > 2) average = length / (size - 2);
    int shortage = 0;
    for (int i = 0; i < size - 1; ++i) {
      int required = (intvls[i].shrink) ? intvls[i].shrinked : intvls[i].required;
      if (i == 0 || i == size - 2) {
	if ((average / 2) - required > 0)
	  shortage += (average / 2) - required;
      } else {
	if (average - required > 0)
	  shortage += average - required;
      }
    }
    if (shortage == 0) return;
    for (int i = 0; i < size - 1; ++i) {
      int local_shortage = 0;
      int required = (intvls[i].shrink) ? intvls[i].shrinked : intvls[i].required;
      if (i == 0 || i == size - 2) {
	if ((average / 2) - required > 0)
	local_shortage = (average / 2) - required;
      } else {
	if (average - required > 0)
	  local_shortage = average - required;
      }
      int sp = (space * local_shortage) / shortage;
      intvls[i].realspace += sp;
    }
  }
};


// ----------------------------------------------------------------------
// CLASS grid
// ----------------------------------------------------------------------
class grid {
private:
  space_distributer* _sdist;

  Am_Object _group;

  int _width;
  int _height;
  int _xscale;
  int _yscale;
  int _orig_xscale;
  int _orig_yscale;

  int _orig_width; // allowGrowing$B$,(Btrue$B$N$H$-;H$&M=Dj!%(B
  int _orig_height;

  // parameters
  int _minspace; 
  int _minsize;
  int _maxpercent;
  bool _allow_growing;

public: 
  int _req_width;
  int _req_height;

private: 
  // grid
  int _xsize;
  int _ysize;
  int _psize;
  cell*** _grid;
  cell** _parts;
  cell* _wall;

  // intervals
  interval* _xintervals;
  interval* _yintervals;

public:

  grid(Am_Object group)
  {
    _sdist = new equal_distributer();

    _group = group;
    
    _wall = new cell;

    _width = group.Get(Am_WIDTH);
    _height = group.Get(Am_HEIGHT);
    _xscale = group.Get(xScaleFactor);
    _yscale = group.Get(yScaleFactor);
    _orig_xscale = group.Get(xScaleFactor);
    _orig_yscale = group.Get(yScaleFactor);

    _orig_width = group.Get(origWidth);
    _orig_height = group.Get(origHeight);

    _minspace = group.Get(minimumSpace);
    _maxpercent = group.Get(maximumPercentage);

    // initialize grid
    if (group.Is_Instance_Of(SequenceAlignZoomGroup)) {
      init_sequence_grid();
    } else {
      init_grid();
    }

    // initialize interval
    _xintervals = new interval[_xsize - 1];
    _yintervals = new interval[_ysize - 1];
  }

  ~grid()
  {
    for (int i = 0; i < _xsize; ++i) {
      delete[] _grid[i];
    }
    delete _sdist;
    delete[] _grid;
    delete[] _parts;
    delete _wall;
    delete[] _xintervals;
    delete[] _yintervals;
  } 
    
private: 
  void init_grid()
  {
    Am_Value_List x_list = Am_Value_List();
    Am_Value_List y_list = Am_Value_List();

    Am_Value_List parts = _group.Get(Am_GRAPHICAL_PARTS);
    _psize = parts.Length();
    _parts = new cell*[_psize];

    int p = 0;
    for (parts.Start(); !parts.Last(); parts.Next(), ++p) {
      Am_Object obj = parts.Get();
      cell* c = (cell*)(Am_Ptr)obj.Get(cellObject);
      _parts[p] = c;
      int xcenter = c->x + c->w / 2;
      for (x_list.Start(); ; x_list.Next()) {
	if (x_list.Last()) { x_list.Add(xcenter);  break; }
	int val = x_list.Get();
	if (xcenter == val) break;
	if (xcenter < val) { x_list.Insert(xcenter, Am_BEFORE);  break; }
      }
      int ycenter = c->y + c->h / 2;
      for (y_list.Start(); ; y_list.Next()) {
	if (y_list.Last()) { y_list.Add(ycenter);  break; }
	int val = y_list.Get();
	if (ycenter == val) break;
	if (ycenter < val) { y_list.Insert(ycenter, Am_BEFORE);  break; }
      }
    }
    _xsize = x_list.Length() + 2;
    _ysize = y_list.Length() + 2;

    _grid = new cell**[_xsize];
    for (int i = 0; i < _xsize; ++i) {
      _grid[i] = new cell*[_ysize];
      for (int j = 0; j < _ysize; ++j) {
	if (i == 0 || i == _xsize -1 || j == 0 || j == _ysize -1)
	  _grid[i][j] = _wall;
	else
	  _grid[i][j] = 0;
      }
    }
    
    for(int i = 0; i < _psize; ++i) {
      cell* c = _parts[i];
      int xidx = index(x_list, c->x + c->w / 2);
      int yidx = index(y_list, c->y + c->h / 2);
      _grid[xidx+1][yidx+1] = c;
    }
  }

  void init_sequence_grid()
  {
    int dir = _group.Get(seqDir);
    
    Am_Value_List parts = _group.Get(Am_GRAPHICAL_PARTS);
    _psize = parts.Length();
    _parts = new cell*[_psize];

    int p = 0;
    for (parts.Start(); !parts.Last(); parts.Next(), ++p) {
      Am_Object obj = parts.Get();
      cell* c = (cell*)(Am_Ptr)obj.Get(cellObject);
      _parts[p] = c;
    }

    if (dir == RightLeft || dir == LeftRight) {
      _xsize = 6;
      _ysize = 3;
    } else {
      _xsize = 3;
      _ysize = 6;
    }

    _grid = new cell**[_xsize];
    for (int i = 0; i < _xsize; ++i) {
      _grid[i] = new cell*[_ysize];
      for (int j = 0; j < _ysize; ++j) {
	if (i == 0 || i == _xsize -1 || j == 0 || j == _ysize -1)
	  _grid[i][j] = _wall;
	else
	  _grid[i][j] = 0;
      }
    }
    
    int start = 0;
    int diff = 1;
    if (dir == RightLeft || dir == BottomTop) {
      start = _psize - 1;
      diff = -1;
    }
    for(int i = 0; i < _psize; ++i) {
      cell* c = _parts[i];
      if (dir == LeftRight || dir == RightLeft) {
	_grid[start + 1][1] = c;
      } else {
	_grid[1][start + 1] = c;
      }
      start += diff;
    }
  }

  bool greater_interval(interval* int1, interval* int2)
  {
    int xshrink = _req_width  - _width;
    int yshrink = _req_height - _height;

    if (int1->dir == int2->dir || (xshrink <= 0 && yshrink <= 0)) {
      return int1->required - int1->shrinked > int2->required - int2->shrinked;
    }
    if (int1->dir == 0) {
      if (xshrink > 0 && yshrink <= 0) return true;
      if (xshrink <= 0 && yshrink > 0) return false;
      return (int1->required - int1->shrinked) * _req_width / _width >
	(int2->required - int2->shrinked) * _req_height / _height;
    }
    if (int1->dir == 1) {
      if (xshrink > 0 && yshrink <= 0) return false;
      if (xshrink <= 0 && yshrink > 0) return true;
      return (int1->required - int1->shrinked) * _req_height / _height >
	(int2->required - int2->shrinked) * _req_width / _width;
    }
    return false;
  }
    

public: 
  void show() {
    cout << "==================================================" << endl;
    cout << _xscale << " " << _yscale << " " << _width << " " << _height << endl;
    cout << "--------------------------------------------------" << endl;
    cout << "\t\t";
    for (int x = 0; x < _xsize - 1; ++x) {
      cout << _xintervals[x].required << "\t";
    }
    cout << endl << "\t\t";
    for (int x = 0; x < _xsize - 1; ++x) {
      cout << _xintervals[x].shrinked << ((_xintervals[x].shrink) ? "o" : "x") << "\t";
    }
    cout << endl;

    for (int y = 0; y < _ysize; ++y) {
      if (y < _ysize - 1) {
	cout << _yintervals[y].required << "\t" << _yintervals[y].shrinked
	     << ((_yintervals[y].shrink) ? "o" : "x") << "\t";
      } else cout << "\t\t";
      for (int x = 0; x < _xsize; ++x) {
	if (_grid[x][y] != 0) { _grid[x][y]->show();  cout << "\t"; }
	else cout << ".\t";
      }
      cout << endl;
    }

    cout << "--------------------------------------------------" << endl;
    for (int i = 0; i < _xsize - 1; ++i) {
      interval first = _xintervals[i];
      cout << "[(" << first.dir << "," << first.index << ")" << first.realspace << "," << first.required << "," << first.shrinked << "," << first.shrink << "]";
      cout << endl;
    }
    for (int i = 0; i < _ysize - 1; ++i) {
      interval first = _yintervals[i];
      cout << "[(" << first.dir << "," << first.index << ")" << first.realspace << "," << first.required << "," << first.shrinked << "," << first.shrink << "]";
      cout << endl;
    }
    cout << "==================================================" << endl;

  }

public:
  bool size_changed() {
    if (_width == (int)_group.Get(Am_WIDTH) &&
	_height == (int)_group.Get(Am_HEIGHT)) return false;
    return true;
  }

  void update() {
    _width = _group.Get(Am_WIDTH);
    _height = _group.Get(Am_HEIGHT);
  }

public:
  void calc_intervals()
  {
    calc_x_intervals();
    calc_y_intervals();
  }

private:
  void calc_x_intervals()
  {
    _req_width = 0;
    for (int x = 0; x < _xsize - 1; ++x) {
      int lmax = 0;
      int rmax = 0;
      int shmax = 0;
      for (int y = 1; y < _ysize - 1; ++y) {
	int left = 0;
	int right = 0;
	if (_grid[x][y] != 0) {
	  left = _grid[x][y]->w / 2;
	  lmax = imax(lmax, left);
	}
	if (_grid[x+1][y] != 0) {
	  right = _grid[x+1][y]->w / 2;
	  rmax = imax(rmax, right);
	}
	shmax = imax(shmax, left + right);
      }
      _xintervals[x].index = x;
      _xintervals[x].dir = 0;
      _xintervals[x].required = lmax + rmax + ((lmax + rmax > 0) ? _minspace : 0);
      _xintervals[x].shrinked = shmax + ((shmax > 0) ? (_minspace / 2) : 0);
      _req_width += _xintervals[x].required;
    }
  }

  void calc_y_intervals()
  {
    _req_height = 0;
    for (int y = 0; y < _ysize - 1; ++y) {
      int lmax = 0;
      int rmax = 0;
      int shmax = 0;
      for (int x = 1; x < _xsize - 1; ++x) {
	int left = 0;
	int right = 0;
	if (_grid[x][y] != 0) {
	  left  = (_grid[x][y]->h) / 2;
	  lmax = imax(lmax, left);
	}
	if (_grid[x][y+1] != 0) {
	  right = (_grid[x][y+1]->h) / 2;
	  rmax = imax(rmax, right);
	}
	shmax = imax(shmax, left + right);
      }
      _yintervals[y].index = y;
      _yintervals[y].dir = 1;
      _yintervals[y].required = lmax + rmax + ((lmax + rmax > 0) ? _minspace : 0);
      _yintervals[y].shrinked = shmax + ((shmax > 0) ? (_minspace / 2) : 0);
      _req_height += _yintervals[y].required;
    }
  }

public:
  void shrink(int type)
  {
    int req_w = calc_required(Am_LEFT);
    int req_h = calc_required(Am_TOP);

    if ((req_w > _width && /* || */  req_h > _height) && type != PartMoving) return;

    for (int y = 1; y < _ysize - 2; ++y) _yintervals[y].shrink = false;
    for (int x = 1; x < _xsize - 2; ++x) _xintervals[x].shrink = false;
      
    Am_Value_List sorted_intervals = Am_Value_List();

    for (int y = 1; y < _ysize - 2; ++y) {
      for (sorted_intervals.Start(); !sorted_intervals.Last()
	     && greater_interval((interval*)(Am_Ptr)sorted_intervals.Get(), &_yintervals[y]);
	   sorted_intervals.Next());
      sorted_intervals.Insert(&_yintervals[y], Am_BEFORE);
    }
    for (int x = 1; x < _xsize - 2; ++x) {
      for (sorted_intervals.Start(); !sorted_intervals.Last()
	     && greater_interval((interval*)(Am_Ptr)sorted_intervals.Get(), &_xintervals[x]);
	   sorted_intervals.Next());
      sorted_intervals.Insert(&_xintervals[x], Am_BEFORE);
    }
    for (sorted_intervals.Start(); !sorted_intervals.Last(); sorted_intervals.Next()) {
      bool shrinkable = false;
      interval* i = (interval*)(Am_Ptr)sorted_intervals.Get();
      if (i->dir == 0) shrinkable = is_x_shrinkable(i->index);
      if (i->dir == 1) shrinkable = is_y_shrinkable(i->index);
      //      cout << "(" << (*first)->dir << "," << (*first)->index << "," << shrinkable << ")";
      if (shrinkable) i->shrink = true;
    }
    //    cout << endl;
  }

  void arrange(int type)
  {
    _req_width = calc_shrinked_required(Am_LEFT);
    _req_height = calc_shrinked_required(Am_TOP);

    _sdist->distribute_required_space(_xintervals, _xsize);
    _sdist->distribute_required_space(_yintervals, _ysize);

    _orig_xscale = _xscale;
    _orig_yscale = _yscale;
    _xscale = denom;
    _yscale = denom;

    if (_req_width <= _width) {
      _sdist->distribute_space(_xintervals, _xsize, _req_width, _width);
    } else {
      _xscale = (denom * _width) / _req_width;
    }
    if (_req_height <= _height) {
      _sdist->distribute_space(_yintervals, _ysize, _req_height, _height);
    } else {
      _yscale = (denom * _height) / _req_height;
    }

    if (_xscale <= 1 || _yscale <= 1) {
      cerr << "scale(s) too small. aborting..." << endl;
      abort();
      return;
    }

    bool local = check_local4(type);
    if (local) {
      _group.Set(xScaleFactor, _xscale);
      _group.Set(yScaleFactor, _yscale);
      set_lefts(_xscale);
      set_tops(_yscale);
    } else {
      calc_intervals();
      arrange(PartMoving);
    }
  }

  // For shrink()

  bool is_x_shrinkable(int x)
  {
    for (int y = 1; y < _ysize-1; ++y) {
      if (_grid[x][y] != 0)
	if((_grid[x+1][y-1] != 0 && _yintervals[y-1].shrink) ||
	   (_grid[x+1][y+1] != 0 && _yintervals[y].shrink))
	  return false;
      if (_grid[x+1][y] != 0)
	if((_grid[x][y-1] != 0 && _yintervals[y-1].shrink) ||
	   (_grid[x][y+1] != 0 && _yintervals[y].shrink))
	  return false;
    }
    return true;
  }

  bool is_y_shrinkable(int y)
  {
    for (int x = 1; x < _xsize-1; ++x) {
      if (_grid[x][y] != 0)
	if((_grid[x-1][y+1] != 0 && _xintervals[x-1].shrink) ||
	   (_grid[x+1][y+1] != 0 && _xintervals[x].shrink))
	  return false;
      if (_grid[x][y+1] != 0)
	if((_grid[x-1][y] != 0 && _xintervals[x-1].shrink) ||
	   (_grid[x+1][y] != 0 && _xintervals[x].shrink))
	  return false;
    }
    return true;
  }

  // For arrange()

  bool check_local4(int type)
  {
    if (type == NoChangeInside) return true;
    bool local = true;
    Am_Object parent = _group.Get(propagateParent);

    for (int i = 0; i < _psize; ++i) {
      cell* c = _parts[i];
      c->isMoved = false;

      int new_w = c->w;
      int new_h = c->h;
      int org_w = c->origW;
      int org_h = c->origH;

      if (new_w > org_w && org_w > _width && _xsize == 3 && type == PartGrowing) {
	if (parent.Valid()) {
	  _width = new_w * _width / org_w;
	  _group.Set(Am_WIDTH, _width);
	  parent.Set(grown, true);
	  if (_width < org_w) c->w = org_w;
	} else {
	  c->w = _width - _minspace * 2;
	}
	local = false;
      }
      else if (new_w > _width && _width > org_w && _xsize == 3 && type == PartGrowing) {
	if (parent.Valid()) {
	  _width = new_w + _minspace * 2;
	  _group.Set(Am_WIDTH, _width);
	  parent.Set(grown, true);
	} else {
	  c->w = _width - _minspace * 2;
	}
	local = false;
      }

      if (new_h > org_h && org_h > _height && _ysize == 3 && type == PartGrowing) {
	if (parent.Valid()) {
	  _height = new_h * _height / org_h;
	  _group.Set(Am_HEIGHT, _height);
	  parent.Set(grown, true);
	  if (_height < org_h) c->h = org_h;	  
	} else {
	  c->h = _height - _minspace * 2;
	}
	local = false;
      }
      else if (new_h > _height && _height > org_h && _ysize == 3 && type == PartGrowing) {
	if (parent.Valid()) {
	  _height = new_h + _minspace * 2;
	  _group.Set(Am_HEIGHT, _height);
	  parent.Set(grown, true);
	} else {
	  c->h = _height - _minspace * 2;
	}
	local = false;
      }
      if (c->w > _width * 10 && c->isGrown) {
	c->w = _width * 10;
      }
      if (c->h > _height * 10 && c->isGrown) c->h = _height * 10;
    }

    // 100$B8D$N%*%V%8%'%/%H$,0lNs$KJB$s$GA4It$,?F$N(B10$BG\$K$J$i$J$$8B$jBg>fIW(B
    if (_xscale < denom / 10 && type == PartGrowing && _xsize > 3) {
      if (parent.Valid()) {
	_width = _req_width / 10;
	_group.Set(Am_WIDTH, _width);
	parent.Set(grown, true);
      } else {
	abort();
      }
      local = false;
    }

    if (_yscale < denom / 10 && type == PartGrowing && _ysize > 3) {
      if (parent.Valid()) {
	_height = _req_height / 10;
	_group.Set(Am_HEIGHT, _height);
	parent.Set(grown, true);
      } else {
	abort();
      }
      local = false;
    }
    return local;
  }

  void abort()
  {
    for (int i = 0; i < _psize; ++i) {
      cell* c = _parts[i];
      //      if (!c->visible) continue;
      c->x = c->origX;
      c->y = c->origY;
      c->w = c->origW;
      c->h = c->origH;
    }
  }

  void set_lefts(int scale)
  {
    int left = 0;
    for (int x = 0; x < _xsize - 2; ++x) {
      left += _xintervals[x].realspace;
      for (int y = 1; y < _ysize - 1; ++y) {
	cell* c = _grid[x+1][y];
	if (c != 0) {
	  c->origX = left - c->w / 2;
	  c->x = c->origX * scale / denom;
	  c->obj.Set(Am_LEFT, c->x);
	  if (c->isGrown) { c->origW = c->w; }
	  //	  if (c->visible)
	  c->w = imax(2, c->w * scale / denom);
	  c->obj.Set(Am_WIDTH, c->w);
	}
      }
    }
  }

  void set_tops(int scale)
  {
    int top = 0;
    for (int y = 0; y < _ysize - 2; ++y) {
      top += _yintervals[y].realspace;
      for (int x = 1; x < _xsize - 1; ++x) {
	cell* c = _grid[x][y+1];
	if (c != 0) {
	  c->origY = top - c->h / 2;
	  c->y = c->origY * scale / denom;
	  c->obj.Set(Am_TOP, c->y);
	  if (c->isGrown) {c->origH = c->h; c->isGrown = false;}
	  //	  if (c->visible)
	  c->h = imax(2, c->h * scale / denom);
	  c->obj.Set(Am_HEIGHT, c->h);
	}
      }
    }
  }

  int calc_shrinked_required(Am_Slot_Key key)
  {
    interval* intervals = 0; 
    int size;
    if (key == Am_LEFT) { intervals = _xintervals;  size = _xsize - 1; }
    else                { intervals = _yintervals;  size = _ysize - 1; }   

    int length = 0;
    for (int i = 0; i < size; ++i) {
      if (intervals[i].shrink)
	length += intervals[i].shrinked;
      else 
	length += intervals[i].required;
    }
    return length;
  }

  int calc_required(Am_Slot_Key key)
  {
    interval* intervals = 0; 
    int size;
    if (key == Am_LEFT) { intervals = _xintervals;  size = _xsize - 1; }
    else                { intervals = _yintervals;  size = _ysize - 1; }   

    int length = 0;
    for (int i = 0; i < size; ++i) {
      length += intervals[i].required;
    }
    return length;
  }
};


// Prediction Strategy Class
// ----------------------------------------------------------------------

DEFINE_SLOT(sizeStatus);
DEFINE_SLOT(tmpSize);

#define SMALL 0
#define LARGE 1
#define SMALLEx 2

class prediction_strategy {
public:
  virtual void flush_grow() = 0;
  
  virtual void Grow  (Am_Object obj,Am_Inter_Location pre,Am_Inter_Location post) = 0;
  virtual void Shrink(Am_Object obj,Am_Inter_Location pre,Am_Inter_Location post) = 0;
  virtual void Expand(Am_Object obj,Am_Inter_Location pre,Am_Inter_Location post) = 0;
  virtual void Small (Am_Object obj,Am_Inter_Location pre,Am_Inter_Location post) = 0;
  virtual void Large (Am_Object obj,Am_Inter_Location pre,Am_Inter_Location post) = 0;
  virtual void Move  (Am_Object obj,Am_Inter_Location pre,Am_Inter_Location post) = 0;
};

prediction_strategy *PredictionStrategy = 0;

class prediction1 : public prediction_strategy
{
  Am_Object last_growed;
  Am_Inter_Location pre_size;
  Am_Inter_Location post_size;


  int get_state(Am_Object obj);

public: 
  // constructor
  prediction1();
  // actions
  void flush_grow();
  
  void Grow  (Am_Object obj,Am_Inter_Location pre,Am_Inter_Location post);
  void Shrink(Am_Object obj,Am_Inter_Location pre,Am_Inter_Location post);
  void Expand(Am_Object obj,Am_Inter_Location pre,Am_Inter_Location post);
  void Small (Am_Object obj,Am_Inter_Location pre,Am_Inter_Location post);
  void Large (Am_Object obj,Am_Inter_Location pre,Am_Inter_Location post);
  void Move  (Am_Object obj,Am_Inter_Location pre,Am_Inter_Location post);
};

prediction1::prediction1()
{
  last_growed = 0;
  pre_size = 0;
  post_size = 0;
}

void prediction1::flush_grow()
{
  if (last_growed.Valid()) {
    if (is_smaller_size(pre_size, post_size)) {
      Expand(last_growed, pre_size, post_size);
    } else {
      Shrink(last_growed, pre_size, post_size);
    }
    last_growed = 0;
  }
}

int prediction1::get_state(Am_Object obj)
{
  if (obj.Get_Slot_Type(sizeStatus) == Am_NONE) {
    obj.Set(sizeStatus, SMALL);
  }
  return (int)obj.Get(sizeStatus);
}

void prediction1::Grow(Am_Object obj,Am_Inter_Location pre,Am_Inter_Location post)
{
  if (obj == last_growed) {
    post_size = post;
  } else {
    flush_grow();
    last_growed = obj;
    pre_size = pre;
    post_size = post;
  }
}

void prediction1::Shrink(Am_Object obj,Am_Inter_Location pre,Am_Inter_Location post)
{
  int curr_state = get_state(obj);
  //  cout << obj << " Shrink from " << pre << " to " << post << endl;

  switch(curr_state) {
  case SMALL:
    set_size_by_mode(obj, ORIG_SIZE, post);
    break;
  case SMALLEx:
    set_size_by_mode(obj, EDIT_SIZE, pre);
    set_size_by_mode(obj, ORIG_SIZE, post);
    obj.Set(sizeStatus, SMALL);
    break;
  case LARGE:
    set_size_by_mode(obj, EDIT_SIZE, post);
    break;
  }
}

void prediction1::Expand(Am_Object obj,Am_Inter_Location pre,Am_Inter_Location post)
{
  //  cout << obj << " Expand from " << pre << " to " << post << endl;
  int curr_state = get_state(obj);
  //  cout << "State:" << curr_state << ", Expand:" << obj << "*******************" << endl;
  Am_Inter_Location tmp = 0;
  switch(curr_state) {
  case SMALL:
    obj.Set(tmpSize, post);
    obj.Set(sizeStatus, SMALLEx);
    break;
  case SMALLEx:
    tmp = obj.Get(tmpSize);
    set_size_by_mode(obj, ORIG_SIZE, tmp);
    break;
  case LARGE:
    set_size_by_mode(obj, EDIT_SIZE, post);
  }
}

void prediction1::Small(Am_Object obj,Am_Inter_Location pre,Am_Inter_Location post)
{
  int curr_state = get_state(obj);
  //  cout << "State:" << curr_state << ", Small:" << obj << "*******************" << endl;
  Am_Inter_Location tmp = 0;
  switch(curr_state) {
  case SMALL:
    break;
  case SMALLEx:
    set_size_by_mode(obj, EDIT_SIZE, pre);
    obj.Set(sizeStatus, SMALL);
    break;
  case LARGE:
    obj.Set(sizeStatus, SMALL);
  }
}

void prediction1::Large(Am_Object obj,Am_Inter_Location pre,Am_Inter_Location post)
{
  int curr_state = get_state(obj);
  //  cout << "State:" << curr_state << ", LARGE:" << obj << "*******************" << endl;
  Am_Inter_Location tmp = 0;
  switch(curr_state) {
  case SMALL:
    obj.Set(sizeStatus, LARGE);
    break;
  case SMALLEx:
    tmp = obj.Get(tmpSize);
    set_size_by_mode(obj, ORIG_SIZE, tmp);
    obj.Set(sizeStatus, LARGE);
    break;
  case LARGE:
    break;
  }
}

void prediction1::Move(Am_Object obj,Am_Inter_Location pre,Am_Inter_Location post)
{
  flush_grow();
}


// ----------------------------------------------------------------------
// Public Functions
// ----------------------------------------------------------------------

// get required width and height
// -----------------------------
int Get_Required_Width(Am_Object group)
{
  if (!group.Is_Instance_Of(AlignZoomGroup)) return group.Get(Am_WIDTH);
  if (group.Get(gridObject).Valid()) {
    grid* g = (grid*)(Am_Ptr)group.Get(gridObject);
    return g->_req_width;
  }
  else group.Get(Am_WIDTH);
}

int Get_Required_Height(Am_Object group)
{
  if (!group.Is_Instance_Of(AlignZoomGroup)) return group.Get(Am_HEIGHT);
  if (group.Get(gridObject).Valid()) {
    grid* g = (grid*)(Am_Ptr)group.Get(gridObject);
    return g->_req_height;
  }
  else group.Get(Am_HEIGHT);
}

// get original x, y, width, height
// --------------------------------
int origX(Am_Object obj)
{
  if (obj.Get_Slot_Type(cellObject) == Am_NONE) return obj.Get(Am_LEFT);
  cell* c = (cell*)(Am_Ptr)obj.Get(cellObject);
  return c->origX;
}

int origY(Am_Object obj)
{
  if (obj.Get_Slot_Type(cellObject) == Am_NONE) return obj.Get(Am_TOP);
  cell* c = (cell*)(Am_Ptr)obj.Get(cellObject);
  return c->origY;
}

int origW(Am_Object obj)
{
  if (obj.Get_Slot_Type(cellObject) == Am_NONE) return obj.Get(Am_WIDTH);
  cell* c = (cell*)(Am_Ptr)obj.Get(cellObject);
  return c->origW;
}

int origH(Am_Object obj)
{
  if (obj.Get_Slot_Type(cellObject) == Am_NONE) return obj.Get(Am_HEIGHT);
  cell* c = (cell*)(Am_Ptr)obj.Get(cellObject);
  return c->origH;
}

void OutputCoords(Am_Object obj, ofstream& ofile)
{
  ofile << origX(obj) << " " << origY(obj) << " ";

  if (obj.Get_Slot_Type(currPointer) != Am_NONE) {
    Am_Inter_Location cur_size = obj.Get(currPointer);
    Am_Inter_Location orig_size = obj.Get(origPointer);
    Am_Inter_Location edit_size = obj.Get(editPointer);

    int x, y, cw, ch, sw, sh, lw, lh;
    cur_size.Get_Points(x, y, cw, ch);
    orig_size.Get_Points(x, y, sw, sh);  
    edit_size.Get_Points(x, y, lw, lh);

    ofile << cw << " " << ch << " " << sw << " " << sh << " " << lw << " " << lh << " ";
  } else {
    ofile << origW(obj) << " " << origH(obj) << " " << 0 << " " << 0 << " " << 0 << " " << 0;
  }
  return;
}

void Add_History(Am_Object target, Am_Inter_Location new_loc);

void SetOrigXYWH(Am_Object obj, int x, int y, int w, int h)
{
  if (obj.Get_Slot_Type(cellObject) == Am_NONE) return;
  cell* c = (cell*)(Am_Ptr)obj.Get(cellObject);
  c->origX = x;
  c->origY = y;
  c->origW = w;
  c->origH = h;
}

void SetPointers(Am_Object obj, int cw, int ch, int sw, int sh, int lw, int lh)
{
  Am_Inter_Location curr_loc = Am_Inter_Location(false, 0, 0, 0, cw, ch);
  Am_Inter_Location small_loc = Am_Inter_Location(false, 0, 0, 0, sw, sh);
  Am_Inter_Location large_loc = Am_Inter_Location(false, 0, 0, 0, lw, lh);

  Am_Value_List sorted_history = Am_Value_List();
  if (is_smaller_size(curr_loc, small_loc))
    sorted_history.Add(curr_loc).Add(small_loc).Add(large_loc);
  if (is_smaller_size(large_loc, curr_loc))
    sorted_history.Add(small_loc).Add(large_loc).Add(curr_loc);
  else
    sorted_history.Add(small_loc).Add(curr_loc).Add(large_loc);

  obj.Set(currPointer, curr_loc)
    .Set(origPointer, small_loc)
    .Set(editPointer, large_loc)
    .Set(sortedHistory, sorted_history)
    ;

  Add_History(obj, small_loc);
  Add_History(obj, large_loc);
}

// Create and Copy parts of AlignZoomGroup
// ---------------------------------------
void remove_private_slots(Am_Object obj)
{
  obj.Remove_Slot(cellObject);
  obj.Remove_Slot(editPointer);
  obj.Remove_Slot(origPointer);
  obj.Remove_Slot(currPointer);  
  obj.Remove_Slot(sortedHistory);
  obj.Remove_Slot(grown);
}

Am_Object Align_Zoom_Create_Part(Am_Object part)
{
  cell* c = (cell*)(Am_Ptr)part.Get(cellObject);
  Am_Object new_part = part.Create()
    .Set(Am_LEFT,   c->origX)
    .Set(Am_TOP,    c->origY)
    .Set(Am_WIDTH,  c->origW)
    .Set(Am_HEIGHT, c->origH)
    ;
  remove_private_slots(new_part);
      
  return new_part;
}

Am_Object Align_Zoom_Copy_Part(Am_Object part)
{
  Am_Object new_part = part.Copy();
  if (new_part.Get_Slot_Type(cellObject) != Am_NONE) {
    cell* c = (cell*)(Am_Ptr)part.Get(cellObject);
    new_part
      .Set(Am_LEFT,   c->origX)
      .Set(Am_TOP,    c->origY)
      .Set(Am_WIDTH,  c->origW)
      .Set(Am_HEIGHT, c->origH)
      ;
    remove_private_slots(new_part);
  }
  return new_part;
}


// ----------------------------------------------------------------------
// Create Copy Demon procedures for AlignZoomGroup
// ----------------------------------------------------------------------
void alignzoom_create_copy_demon_proc(Am_Object self)
{
  // grid $B$O%3%T!<$7$J$$!#(B
  self.Set(gridObject, 0);
  // $B3F%Q!<%H$N(Bcell$B$r%3%T!<(B
  Am_Value_List parts = self.Get(Am_GRAPHICAL_PARTS);
  for (parts.Start(); !parts.Last(); parts.Next()) {
    Am_Object p = parts.Get();
    cell* orig_cell = (cell*)(Am_Ptr)p.Get(cellObject);
    cell* new_cell = new cell(*orig_cell);
    new_cell->obj = p;
    p.Set(cellObject, (Am_Ptr)new_cell);
  }
}

void alignzoom_create_demon(Am_Object self)
{
  // $B%*%j%8%J%k$N%G!<%b%s%W%m%7!<%8%c$r8F$V(B
  Am_Object_Advanced proto_adv = (Am_Object_Advanced&)Am_Group;
  Am_Object_Demon* proto_create = proto_adv.Get_Demons().Get_Object_Demon(Am_CREATE_OBJ);
  if (proto_create) proto_create(self);
  alignzoom_create_copy_demon_proc(self);
}

void alignzoom_copy_demon(Am_Object self)
{
  // $B%*%j%8%J%k$N%G!<%b%s%W%m%7!<%8%c$r8F$V(B
  Am_Object_Advanced proto_adv = (Am_Object_Advanced&)Am_Group;
  Am_Object_Demon* proto_create = proto_adv.Get_Demons().Get_Object_Demon(Am_COPY_OBJ);
  if (proto_create) proto_create(self);
  alignzoom_create_copy_demon_proc(self);
}

// ----------------------------------------------------------------------
// Add Part Demon for AlignZoomGroup
// ----------------------------------------------------------------------
void alignzoom_part_demon(Am_Object self, Am_Object removed, Am_Object added)
{
  // $B%*%j%8%J%k$N%G!<%b%s%W%m%7!<%8%c$r8F$V(B
  Am_Object_Advanced proto_adv = (Am_Object_Advanced&)Am_Group;
  Am_Part_Demon* proto_part = proto_adv.Get_Demons().Get_Part_Demon(Am_ADD_PART);
  if (proto_part) proto_part(self, removed, added);
  // $B%0%i%U%#%+%k%Q!<%H$NJQ99$+(B?
  if (!added.Is_Instance_Of(Am_Graphical_Object)
      && !removed.Is_Instance_Of(Am_Graphical_Object)) return;

  if (removed.Valid()) {
    if (removed.Get_Slot_Type(cellObject) == Am_NONE) return;
    cell* c = (cell*)(Am_Ptr)removed.Get(cellObject);
    removed
      .Set(Am_LEFT, c->origX)
      .Set(Am_TOP, c->origY)
      .Set(Am_WIDTH, c->origW)
      .Set(Am_HEIGHT, c->origH)
      ;
    delete c;
    remove_private_slots(removed);
    self.Set(partMoved, true);
  }
  if (added.Valid()) {
    cell* c = new cell(added, self);
    Am_Inter_Location new_loc = Am_Inter_Location(false, self, 0, 0,
						  c->origW, c->origH);
    Am_Inter_Location edit_loc = Am_Inter_Location(false, self, 0, 0,
						   c->origW, c->origH);
    added.Set(cellObject, (Am_Ptr)c);

    if (added.Get_Slot_Type(currPointer) == Am_NONE) {
      added.Set(editPointer, edit_loc);
      added.Set(origPointer, new_loc);
      added.Set(currPointer, new_loc);
      added.Set(sortedHistory, Am_Value_List().Add(new_loc));
    }
    added.Set(grown, false);
    self.Set(partMoved, true);
  }
}


// ----------------------------------------------------------------------
// Formulas
// ----------------------------------------------------------------------

// prppagate
// ---------
Am_Define_Object_Formula(PropagateForm)
{
  Am_Object owner = self.GV_Owner();
  if (!owner.Valid()) return 0;
  return owner;
}

// #define DEBUGCHECK

// Check Changes
// -------------
void set_parent_editing_size(Am_Object parent);

Am_Define_Formula(int, CheckChangesForm)
{
#ifdef DEBUGCHECK
  cout << self << endl;
#endif
  // grid $B%*%V%8%'%/%H$,$J$$>l9g$O:n$k!#(B
  grid* g = 0;
  if ((grid*)(Am_Ptr)self.Get(gridObject) != 0) {
    g = (grid*)(Am_Ptr)self.Get(gridObject);
  } else {
    g = new grid(self);
    self.Set(gridObject, (Am_Ptr)g);
  }

  int xscale = self.Get(xScaleFactor);
  int yscale = self.Get(yScaleFactor);

  bool moving  = false;
  bool growing = false;
  bool allgrowing = g->size_changed();

  if ((bool)self.Get(partMoved) == true) {
#ifdef DEBUGCHECK
    cout << "\tmoving: by partMoved flag" << endl;
#endif
    moving = true;
    self.Set(partMoved, false);
  }

  Am_Value_List parts = self.GV(Am_GRAPHICAL_PARTS);
  for (parts.Start(); !parts.Last(); parts.Next()) {
    Am_Object obj = parts.Get();
    // $B8=:_$N:BI8!&Bg$-$5!&2D;k(B
    int nx = obj.GV(Am_LEFT);
    int ny = obj.GV(Am_TOP);
    int nw = obj.GV(Am_WIDTH);
    int nh = obj.GV(Am_HEIGHT);

    bool temp_growing = false;

    cell* c = (cell*)(Am_Ptr)obj.Get(cellObject);
    c->visible = (bool)obj.GV(Am_VISIBLE);
    //    if (!c->visible) continue; // $BIT2D;k$OL5;k(B
    if (c->visible != c->origVisible) {
#ifdef DEBUGCHECK
      cout << "\tmoving: change visibility of " << obj << endl;
#endif
      moving = true;
    }
    c->origVisible = c->visible;

    int ow = c->origW;
    int oh = c->origH;
    
    if (c->isMoved) { // $B%*%V%8%'%/%H$,DI2C$5$l$?$H$-!"(Btrue$B$K$J$C$F$$$k(B
      c->x = c->origX;
      c->y = c->origY;
      c->w = c->origW;
      c->h = c->origH;
#ifdef DEBUGCHECK
      cout << "\tmoving: " << obj << "is added" << endl;
#endif
      moving = true;
      //      set_parent_editing_size(self); // ***
    } else {
      if (nw != c->w) { // $BI}$,JQ2=$7$F$$$k$H$-(B
	c->w = nw * denom / xscale;
	c->x = (c->origX + ow / 2) - c->w / 2; // $BCf1{$OF1$80LCV(B
	c->isGrown = true;
#ifdef DEBUGCHECK
	cout << "\tgrowing: " << obj << "is growed" << endl;
#endif
	growing = true;
	temp_growing = true;
	//	set_parent_editing_size(self); // ***
      } else {
	c->w = ow;
      }
      if (nh != c->h) { // $B9b$5$,JQ2=$7$F$$$k$H$-(B
	c->h = nh * denom / yscale;
	c->y = (c->origY + oh / 2) - c->h / 2; // $BCf1{$OF1$80LCV(B
	c->isGrown = true;
#ifdef DEBUGCHECK
	cout << "\tgrowing: " << obj << "is growed" << endl;
#endif
	growing = true;
	temp_growing = true;
	//	set_parent_editing_size(self); // ***
      } else {
	c->h = oh;
      }
      if (!temp_growing) {
	if (nx != c->x) { // x$B:BI8$,JQ2=$7$F$$$k$H$-(B
	  c->x = nx * denom / xscale;
#ifdef DEBUGCHECK
	  cout << "\tmoving: " << obj << "is moved" << endl;
#endif
	  c->isMoved = true;
	  moving = true;
	  //	  set_parent_editing_size(self); // ***
	} else {
	  c->x = c->origX;
	}
	if (ny != c->y) { // y$B:BI8$,JQ2=$7$F$$$k$H$-(B
	  c->y = ny * denom / yscale;
#ifdef DEBUGCHECK
	  cout << "\tmoving: " << obj << "is moved" << endl;
#endif
	  c->isMoved = true;
	  moving = true;
	  //	  set_parent_editing_size(self); // ***
	} else {
	  c->y = c->origY;
	}
      }
    }
  }
  if (moving)  {
#ifdef DEBUGCHECK
    cout << "Part moving" << endl;
#endif
    return PartMoving;
  }  
  if (growing) {
#ifdef DEBUGCHECK
    cout << "Part growing" << endl;
#endif
    return PartGrowing;
  }
  if (allgrowing) {
#ifdef DEBUGCHECK
    cout << "All growing" << endl;
#endif
    return NoChangeInside;
  }
  return NothingHappened;
}


// Avoid Overlap
// -------------
Am_Define_Formula(int, AvoidOverlapLayout)
{
  int threshold = self.GV(alignThreshold);

  Am_Value_List parts = self.Get(Am_GRAPHICAL_PARTS);
  Am_Value_List others = self.Get(Am_GRAPHICAL_PARTS);

  for (parts.Start(); !parts.Last(); parts.Next()) {
    cell* one = (cell*)(Am_Ptr)((Am_Object)parts.Get()).Get(cellObject);
    if (one->isMoved) {
      bool objectMoved;
      int xdir = 0;
      int ydir = 0;
      do {
	objectMoved = false;
	for (others.Start(); !others.Last(); others.Next()) {
	  cell* another = (cell*)(Am_Ptr)((Am_Object)others.Get()).Get(cellObject);
	  if (one == another) continue;
	  
	  int onex = one->x + one->w / 2;
	  int oney = one->y + one->h / 2;
	  int anox = another->x + another->w / 2;
	  int anoy = another->y + another->h / 2;
	  int xdiff = abs(onex - anox);
	  int ydiff = abs(oney - anoy);

	  if (xdiff < threshold && ydiff < threshold) {
	    if (xdir == 0 && ydir == 0) {
	      xdir = ((onex - anox < 0) ? -1 : 1) * ((xdiff >= ydiff) ? 1 : 0);
	      ydir = ((oney - anoy < 0) ? -1 : 1) * ((ydiff >= xdiff) ? 1 : 0);
	    }
	    one->x = one->x + (anox - xdir * (threshold + 2) - onex);
	    one->y = one->y + (anoy - ydir * (threshold + 2) - oney);
	    objectMoved = true;
	  }
	}
      } while (objectMoved);
    }
  }
  return 0;
}

Am_Define_Formula(int, AlignSequenceLayout)
{
  int maxw = 0;
  int maxh = 0;

  Am_Value_List parts = self.GV(Am_GRAPHICAL_PARTS);
  int length = parts.Length();
  if (length <= 0) return 1;

  for (parts.Start(); !parts.Last(); parts.Next()) {
    cell* c = (cell*)(Am_Ptr)((Am_Object)parts.Get()).Get(cellObject);
    int w = c->w;
    int h = c->h;
    maxw = imax(maxw, w);
    maxh = imax(maxh, h);
  }
  int dir = self.GV(seqDir);
  bool inc_left = true;
  bool forward = true;
  if (dir == TopBottom || dir == BottomTop) inc_left = false;
  if (dir == LeftRight || dir == TopBottom) {parts.Start(); forward = true;}
  if (dir == RightLeft || dir == BottomTop) {parts.End(); forward = false;}

  int left = 0;
  int top = 0;

  for ( ; !parts.Last() && !parts.First() ; ) {
    cell* c = (cell*)(Am_Ptr)((Am_Object)parts.Get()).Get(cellObject);
    int w = c->w;
    int h = c->h;
    if (inc_left) top = (maxh - h) / 2;
    else          left = (maxw - w) / 2;
    c->x = left;
    c->origX = left;
    c->y = top;
    c->origY = top;
    if (inc_left) left += w; else top += h; 
    if (forward) parts.Next(); else parts.Prev();
  }
  return 1;
}


// Align
// -----
Am_Define_Formula(int, AlignLayout)
{
  int threshold = self.Get(alignThreshold);

  Am_Value_List parts = self.Get(Am_GRAPHICAL_PARTS);
  Am_Value_List others = self.Get(Am_GRAPHICAL_PARTS);

  for (parts.Start(); !parts.Last(); parts.Next()) {
    cell* one = (cell*)(Am_Ptr)((Am_Object)parts.Get()).Get(cellObject);
    others.Start();
    others.Member(one);
    others.Next();
    for (; !others.Last(); others.Next()) {
      cell* another = (cell*)(Am_Ptr)((Am_Object)others.Get()).Get(cellObject);
      int onex = one->x + one->w / 2;
      int oney = one->y + one->h / 2;
      int onew = one->w;
      int oneh = one->h;
      int anox = another->x + another->w / 2;
      int anoy = another->y + another->h / 2;
      int anow = another->w;
      int anoh = another->h;
      int xdiff = anox - onex;
      int ydiff = anoy - oney;
      if (abs(xdiff) < threshold && abs(xdiff) < (onew + anow) / 2)
	one->x = one->x + xdiff;
      if (abs(ydiff) < threshold && abs(ydiff) < (oneh + anoh) / 2)
	one->y = one->y + ydiff;
    }
  }
  return 0;
}


// Arrange and Zoom
// ----------------

// some utilitiy functions 

Am_Define_Formula(int, ArrangeLayout)
{
  //  int minspace = self.GV(minimumSpace);
  //  int maxpercent = self.GV(maximumPercentage);

  grid* g = new grid(self);
  self.Set(gridObject, (Am_Ptr)g);

  g->calc_intervals();
  g->shrink(PartMoving);
  g->arrange(PartMoving);
  //  g->show();
  return 0;
}

Am_Define_Formula(int, RearrangeLayout)
{
  grid* g = 0;
  if ((grid*)(Am_Ptr)self.Get(gridObject) != 0) {
    g = (grid*)(Am_Ptr)self.Get(gridObject);
  } else {
    ArrangeLayout(cc, self);
    return 0;
  }
  g->update();
  g->calc_intervals();
  g->shrink(PartGrowing);
  g->arrange(PartGrowing);
  return 0;
}

Am_Define_Formula(int, ResizeLayout)
{
  grid* g = 0;
  if ((grid*)(Am_Ptr)self.Get(gridObject) != 0) {
    g = (grid*)(Am_Ptr)self.Get(gridObject);
  } else {
    ArrangeLayout(cc, self);
    return 0;
  }
  g->update();
  g->calc_intervals();
  g->shrink(NoChangeInside);
  g->arrange(NoChangeInside);
  return 0;
}

int CountZooming = 0;
int CountPartMove = 0;
int CountPartGrow = 0;
int CountAllGrow = 0;

Am_Define_Formula(int, AlignZoomLayout)
{
  // ACTIVE$B$G$J$$$H$-$O2?$b$7$J$$!#(B
  if ((bool)self.GV(Am_ACTIVE) == false) return 0;

  // $B6u$N>l9g$J$K$b$7$J$$!#(B
  if (((Am_Value_List)self.GV(Am_GRAPHICAL_PARTS)).Empty()) return 0;

  // detailThreshold$B0J2<$N;~$OI=<($7$J$$!#(B
  int thres = self.GV(detailThreshold);
  if ((int)self.GV(Am_WIDTH) < thres || (int)self.GV(Am_HEIGHT) < thres) {
    self.Set(Am_VISIBLE, false);
    return 0;
  }
  self.Set(Am_VISIBLE, true);

  CountZooming++;

  // $B5/$3$C$?JQ2=$O2?$+(B?
  int type = CheckChangesForm(cc, self);
  switch(type) {
  case PartMoving:
    CountPartMove++;
    AvoidOverlapLayout(cc, self);
    AlignLayout(cc, self);
    ArrangeLayout(cc, self);
    break;
  case PartGrowing:
    CountPartGrow++;
    RearrangeLayout(cc, self);
    break;
  default:
    CountAllGrow++;
    ResizeLayout(cc, self);
    break;
  }
  return 0;
}

Am_Define_Formula(int, SequenceAlignZoomLayoutForm)
{
  // ACTIVE$B$G$J$$$H$-$O2?$b$7$J$$!#(B
  if ((bool)self.GV(Am_ACTIVE) == false) return 0;

  // $B6u$N>l9g$J$K$b$7$J$$!#(B
  if (((Am_Value_List)self.GV(Am_GRAPHICAL_PARTS)).Empty()) return 0;

  // detailThreshhold$B0J2<$N;~$OI=<($7$J$$!#(B
  int thres = self.GV(detailThreshold);
  if ((int)self.GV(Am_WIDTH) <= thres || (int)self.GV(Am_HEIGHT) <= thres) {
    //    self.Set(Am_VISIBLE, false);
    //    return 0;
  }
  self.Set(Am_VISIBLE, true);

  // $B5/$3$C$?JQ2=$O2?$+(B?
  int type = CheckChangesForm(cc, self);
  //  AlignSequenceLayout(cc, self);
  switch(type) {
  case PartMoving:
    ArrangeLayout(cc, self);
    break;
  case PartGrowing:
    RearrangeLayout(cc, self);
    break;
  default:
    ResizeLayout(cc, self);
    break;
  }
  return 0;
}



// ----------------------------------------------------------------------
// $BM=B,$K$h$k%j%5%$%:(B
// ----------------------------------------------------------------------
// $B!&Bg$-$5$G%=!<%H$5$l$?MzNr$r:n$k!#(B*Done*
// $B!&(Bcreate$B;~$K(BorigPointer$B!"(BeditPointer$B$rF1$8$K$9$k!#(B*Done*
// $B!&%/%i%9%?Fb$G$NJT=8;~$K!"(BeditPointer$B$rJQ99$9$k!#(B*Done*
// $B!&(BorigPointer$B$X$NJQ99$,5/$C$?$i;R%N!<%I$N(BorigPointer$B$rJQ99$9$k!#(B*Done*
// $B!&=$@5!"EAGE%$%s%?%U%'!<%9$r:n@.!#(B
// $B!&(Bedit$B%5%$%:$N<+F07W;;!#(B

Am_Define_Object_Formula(OwnerForm)
{
  return self.GV_Owner();
}

Am_Inter_Location Align_Zoom_Get_Location(Am_Object obj)
{
  if (obj.Get_Slot_Type(cellObject) == Am_NONE) return 0;
  cell* c = (cell*)(Am_Ptr)obj.Get(cellObject);
  Am_Object group = obj.Get_Owner();
  Am_Inter_Location loc = Am_Inter_Location(false, group, 0, 0, c->origW, c->origH);
  return loc;
}

void add_sorted_history(Am_Object target)
{
  // history$B%9%m%C%H$O$"$k$+(B
  if (target.Get_Slot_Type(sortedHistory) == Am_NONE) return;

  // cell$B$+$i(BLocation$B%*%V%8%'%/%H$r:n@.(B
  cell* c = (cell*)(Am_Ptr)target.Get(cellObject);
  Am_Inter_Location new_loc = Am_Inter_Location(false, 0, 0, 0, c->origW, c->origH);

  // log
  Am_Inter_Location curr_loc = target.Get(currPointer);
  //  cout << target << " Grow from " << curr_loc << " to " << new_loc << endl;
  PredictionStrategy->Grow(target, curr_loc, new_loc);

  // currPointer$B$NJQ99(B
  target.Set(currPointer, new_loc);

  target.Make_Unique(sortedHistory);
  Am_Value_List history = target.Get(sortedHistory);
  // history$B$,6u$J$i0l$D$NMWAG$N%j%9%H$r:n$j%9%m%C%H$K%;%C%H$9$k(B
  if (history.Empty()) {
    target.Set(sortedHistory, Am_Value_List().Add(new_loc));
    return;
  }
  // insertion$B$r9T$&(B
  for (history.Start(); ; history.Next()) {
    if (history.Last()) {
      history.Add(new_loc, Am_TAIL, false);
      break;
    }
    Am_Inter_Location size = history.Get();
    if (equal_size(new_loc, size)) break;
    if (is_smaller_size(new_loc, size)) {
      history.Insert(new_loc, Am_BEFORE, false);
      break;
    }
  }
  target.Note_Changed(sortedHistory);
}

void Add_History(Am_Object target, Am_Inter_Location new_loc)
{
  // history$B%9%m%C%H$O$"$k$+(B
  if (target.Get_Slot_Type(sortedHistory) == Am_NONE) return;

  target.Make_Unique(sortedHistory);
  Am_Value_List history = target.Get(sortedHistory);
  // history$B$,6u$J$i0l$D$NMWAG$N%j%9%H$r:n$j%9%m%C%H$K%;%C%H$9$k(B
  if (history.Empty()) {
    target.Set(sortedHistory, Am_Value_List().Add(new_loc));
    return;
  }
  // insertion$B$r9T$&(B
  for (history.Start(); ; history.Next()) {
    if (history.Last()) {
      history.Add(new_loc, Am_TAIL, false);
      break;
    }
    Am_Inter_Location size = history.Get();
    if (equal_size(new_loc, size)) break;
    if (is_smaller_size(new_loc, size)) {
      history.Insert(new_loc, Am_BEFORE, false);
      break;
    }
  }
  target.Note_Changed(sortedHistory);
}

Am_Define_Method(Am_Object_Method, void, record_size_history, (Am_Object command_obj))
{
  Am_Do_Events();
  // Am_Object target = command_obj.Get_Owner().Get(Am_OBJECT_MODIFIED);
  // add_history(target);
  Am_Value_List objs = command_obj.Get_Owner().Get_Owner().Get(Am_VALUE);
  for (objs.Start(); !objs.Last(); objs.Next()) {
    Am_Object target = objs.Get();
    add_sorted_history(target);

    Am_Object group = target.Get_Owner();
    if (!group.Valid()) break;
    while (group.Get_Slot_Type(propagateParent)) {
      Am_Object parent = group.Get(propagateParent);
      if (parent.Get_Slot_Type(grown)) {
	if (parent.Get(grown) == true) {
	  parent.Set(grown, false);
	  add_sorted_history(parent);
	  group = parent.Get_Owner();
	  continue;
	}
      }
      break;
    }
  }
}

Am_Define_Method(Am_Object_Method, void, grow_all_selected_objs, (Am_Object command_obj))
{
  Am_Object target = command_obj.Get_Owner().Get(Am_OBJECT_MODIFIED);
  int width = target.Get(Am_WIDTH);
  int height = target.Get(Am_HEIGHT);

  Am_Value_List objs = command_obj.Get_Owner().Get_Owner().Get(Am_VALUE);
  for (objs.Start(); !objs.Last(); objs.Next()) {
    Am_Object obj = objs.Get();
    obj.Set(Am_WIDTH, width);
    obj.Set(Am_HEIGHT, height);
  }
}

// $BJT=80U;V$N%A%'%C%/(B
// ----------------------------------------------------------------------
bool intend_to_edit(Am_Object obj)
{
  if (!obj.Valid()) return false;
  Am_Object child_group = obj.Get(propagateChildren);
  if (!child_group.Valid()) return false;
  if (!child_group.Is_Instance_Of(AlignZoomGroup)) return false;

  Am_Value_List children = child_group.Get(Am_GRAPHICAL_PARTS);
  for (children.Start(); !children.Last(); children.Next()) {
    Am_Object child = children.Get();

    Am_Inter_Location cur_size = child.Get(currPointer);
    Am_Inter_Location orig_size = child.Get(origPointer);
    Am_Inter_Location edit_size = child.Get(editPointer);

    if (!equal_size(cur_size, orig_size)) return true;    
    if (equal_size(cur_size, edit_size)
	&& !equal_size(orig_size, edit_size)) return true;

  }
  return false;
}

void Align_Zoom_Animate_Obj(Am_Object target, Am_Inter_Location loc)
  // $B%5%$%:@Z$jBX$($N%"%K%a!<%7%g%s(B
{
  Am_Object group = target.Get_Owner();

  int x, y, w, h;
  loc.Get_Points(x, y, w, h);
  
  for (int i = 0; i < 4; ++i) {
    int startw = target.Get(Am_WIDTH);
    int starth = target.Get(Am_HEIGHT);

    int xscale = group.Get(xScaleFactor);
    int yscale = group.Get(yScaleFactor);

    int endw = w * xscale / denom;
    int endh = h * yscale / denom;

    int diffw = (endw - startw) / (5 - i);
    int diffh = (endh - starth) / (5 - i);

    target.Set(Am_WIDTH, startw + diffw).Set(Am_HEIGHT, starth + diffh);
    Am_Do_Events();
  }
  int xscale = group.Get(xScaleFactor);
  int yscale = group.Get(yScaleFactor);
  int endw = w * xscale / denom;
  int endh = h * yscale / denom;

  target
    .Set(Am_WIDTH, endw)
    .Set(Am_HEIGHT, endh)
    ;
  Am_Do_Events();
}

void Align_Zoom_Animate_Objs(Am_Value_List objs, Am_Value_List locs)
  // $B%5%$%:@Z$jBX$($N%"%K%a!<%7%g%s(B
{
  Am_Screen.Set(Am_ACTIVE, false);
#ifdef DEBUGCHECK
  cout << "--------------------------------------------------" << endl;
  cout << "ZoomCount: " << CountZooming << " PartMove: " << CountPartMove << " PartGrow: " << CountPartGrow << " AllGrow " << CountAllGrow << endl;
#endif
  int num = objs.Length();
  int frames = 4;
  if (10 < num) frames = 3;
  if (20 < num) frames = 2;
  if (30 < num) frames = 1;

  for (int i = 0; i < frames; ++i) {
    Am_Value_List ws = Am_Value_List();
    Am_Value_List hs = Am_Value_List();

    for (objs.Start(), locs.Start(); !objs.Last(); objs.Next(), locs.Next()) {
      Am_Object target = objs.Get();
      Am_Object group = target.Get_Owner();
      
      Am_Inter_Location loc = locs.Get();
      int x, y, w, h;
      loc.Get_Points(x, y, w, h);
  
      int startw = target.Get(Am_WIDTH);
      int starth = target.Get(Am_HEIGHT);

      int xscale = group.Get(xScaleFactor);
      int yscale = group.Get(yScaleFactor);

      int endw = w * xscale / denom;
      int endh = h * yscale / denom;

      int diffw = (endw - startw) / (frames - i);
      int diffh = (endh - starth) / (frames - i);

      ws.Add(startw + diffw);
      hs.Add(starth + diffh);
    }
    ws.Start();
    hs.Start();
    for (objs.Start(); !objs.Last(); objs.Next()) {
      Am_Object target = objs.Get();
      target.Set(Am_WIDTH, (int)ws.Get())
	    .Set(Am_HEIGHT, (int)hs.Get());
      ws.Next();
      hs.Next();
    }
    ws.Make_Empty();
    hs.Make_Empty();
    Am_Do_Events();
#ifdef DEBUGCHECK
    cout << "------------------------------" << endl;
    cout << "ZoomCount: " << CountZooming << " PartMove: " << CountPartMove << " PartGrow: " << CountPartGrow << " AllGrow " << CountAllGrow << endl;
#endif
  }
  Am_Screen.Set(Am_ACTIVE, true);
}

void change_size_to_loc(Am_Object obj, Am_Inter_Location loc)
  // obj$B$N%5%$%:@Z$jBX$((B
{
  Align_Zoom_Animate_Obj(obj, loc);
  obj.Set(currPointer, loc);
}

void change_sizes_to_locs(Am_Value_List objs, Am_Value_List locs)
{
  Align_Zoom_Animate_Objs(objs, locs);
  for (objs.Start(), locs.Start(); !objs.Last(); objs.Next(), locs.Next()) {
    Am_Object obj = objs.Get();
    Am_Inter_Location loc = locs.Get();
    obj.Set(currPointer, loc);
  }
}

int play_sound(int mode)
{
  return 0;

  pid_t child_pid;
  char* soundfile;

  child_pid = fork();
  if (child_pid < 0) {
    perror("Can't execute sound play process");
    return 0;
  }
  if (child_pid == 0) {
    if (mode == ORIG_SIZE)
      system("/usr/bin/audioplay /home/toyoda/work/src/amulet/toywidgets/whistldown.au");
    else
      system("/usr/bin/audioplay /home/toyoda/work/src/amulet/toywidgets/whistlup.au");
    exit(1);
  }
}

void change_size_by_mode(Am_Object obj, int mode)
  // obj$B$r(Borig$B%5%$%:$^$?$O(Bedit$B%5%$%:$KJQ99$9$k(B
{
  PredictionStrategy->flush_grow();
  Am_Inter_Location curr_size = obj.Get(currPointer);
  Am_Slot_Key pointer_key;
  if (mode == ORIG_SIZE) {
    pointer_key = origPointer;
    //    cout << obj << " Change to SMALL from " << curr_size ;
  } else {
    pointer_key = editPointer;
    //    cout << obj << " Change to LARGE from " << curr_size ;
  }
  Am_Inter_Location new_size = obj.Get(pointer_key);
  //  cout << " to " << new_size << endl;

  if (mode == ORIG_SIZE)
    PredictionStrategy->Small(obj, curr_size, new_size);
  else
    PredictionStrategy->Large(obj, curr_size, new_size);

  change_size_to_loc(obj, new_size);
}

void change_sizes_by_mode(Am_Value_List objs, int mode)
  // obj$B$r(Borig$B%5%$%:$^$?$O(Bedit$B%5%$%:$KJQ99$9$k(B
{
  PredictionStrategy->flush_grow();
  Am_Value_List locs = Am_Value_List();
  for (objs.Start(); !objs.Last(); objs.Next()) {
    Am_Object obj = objs.Get();
    Am_Inter_Location curr_size = obj.Get(currPointer);
    Am_Slot_Key pointer_key;
    if (mode == ORIG_SIZE) {
      pointer_key = origPointer;
      //      cout << obj << " Change to SMALL from " << curr_size ;
    } else {
      pointer_key = editPointer;
      //      cout << obj << " Change to LARGE from " << curr_size ;
    }
    Am_Inter_Location new_size = obj.Get(pointer_key);
    //    cout << " to " << new_size << endl;

    locs.Add(new_size);

    if (mode == ORIG_SIZE)
      PredictionStrategy->Small(obj, curr_size, new_size);
    else
      PredictionStrategy->Large(obj, curr_size, new_size);
  }
  change_sizes_to_locs(objs, locs);
}

void change_to_large_size(Am_Object obj)
  // large size$B$X$N@Z$jBX$(!#$9$Y$F$N?F$r(Bediting size$B$K$9$k!#(B
  // $B@Z$jBX$($O?F$+$i9T$&!#(B
{
  if (is_large_size(obj)) return;

  // $B%k!<%H$^$G$N%Q%9$r:n@.(B
  Am_Value_List path = Am_Value_List();

  while (obj.Get_Slot_Type(editPointer) != Am_NONE) {
    path.Add(obj);
    //  $B?F$r$9$Y$F(Bediting size$B$KJQ99!#:F5"E*$K(B
    if (obj.Get_Owner().Get_Slot_Type(propagateParent) == Am_NONE) break;
    obj = obj.Get_Owner().Get(propagateParent);
  }

  play_sound(EDIT_SIZE);
  // $B%k!<%H$+$i=g$K3HBg(B
  for (path.End(); !path.First(); ) {
    Am_Object target = path.Get();
    path.Prev();
    if (is_small_size(target) || path.First()) { // $B$h$jBg$-$/$J$i$J$$?F$OL5;k(B
      change_size_by_mode(target, EDIT_SIZE);
    }
  }
}

void change_to_small_size(Am_Object obj)
  // small size $B$X$N@Z$jBX$(!#JT=80U;V$N$J$$O"B3$9$k?F$r(Bsmall$B%5%$%:$K@Z$jBX$($k!#(B
  // $B@Z$jBX$($O;R$+$i9T$&!#(B
{
  if (obj.Get_Slot_Type(origPointer) == Am_NONE) return;
  if (is_small_size(obj)) return;
  play_sound(ORIG_SIZE);

  while (obj.Get_Slot_Type(origPointer) != Am_NONE) {
    change_size_by_mode (obj, ORIG_SIZE);
    if (obj.Get_Owner().Get_Slot_Type(propagateParent) == Am_NONE) break;
    obj = obj.Get_Owner().Get(propagateParent);
    if (!obj.Valid()) break;
    if (intend_to_edit(obj)) break; // $BJT=80U;W$N$"$k%N!<%I$h$j>e$OJQ99$7$J$$(B
  }
}

// $B$$$C$Z$s%"%K%a!<%7%g%s4X78$N4X?t(B

Am_Value_List objs_and_their_parents(Am_Value_List objs)
{
  Am_Value_List all_objs = Am_Value_List();

  for (objs.Start(); !objs.Last(); objs.Next()) {
    Am_Object obj = objs.Get();

    while (obj.Get_Slot_Type(origPointer) != Am_NONE) {
      all_objs.Start();
      if (!all_objs.Member(obj)) all_objs.Add(obj);
      //  $B?F$r$9$Y$F(Bediting size$B$KJQ99!#:F5"E*$K(B
      if (obj.Get_Owner().Get_Slot_Type(propagateParent) == Am_NONE) break;
      obj = obj.Get_Owner().Get(propagateParent);
    }
  }
  return all_objs;
}

Am_Value_List large_list(Am_Value_List objs)
{
  Am_Value_List all_objs = objs_and_their_parents(objs);
  Am_Value_List cands = Am_Value_List();

  for (all_objs.Start(); !all_objs.Last(); all_objs.Next()) {
    Am_Object obj = all_objs.Get();

    if (is_small_size(obj)) cands.Add(obj);
  }
  all_objs.Make_Empty();  
  return cands;
}

Am_Value_List small_list(Am_Value_List objs)
{
  Am_Value_List all_objs = objs_and_their_parents(objs);
  Am_Value_List cands = Am_Value_List();

  for (all_objs.Start(); !all_objs.Last(); all_objs.Next()) {
    Am_Object obj = all_objs.Get();
    if (!is_small_size(obj)) cands.Add(obj);
  }
  all_objs.Make_Empty();
  return cands;
}

void change_to_large_sizes(Am_Value_List objs)
  // large size$B$X$N@Z$jBX$(!#$9$Y$F$N?F$r(Bediting size$B$K$9$k!#(B
  // $B@Z$jBX$($O?F$+$i9T$&!#(B
{
  Am_Value_List cands = large_list(objs);
  if (!cands.Empty()) play_sound(EDIT_SIZE);
  change_sizes_by_mode(cands, EDIT_SIZE);
  cands.Make_Empty();
}

void change_to_small_sizes(Am_Value_List objs)
  // small size $B$X$N@Z$jBX$(!#JT=80U;V$N$J$$O"B3$9$k?F$r(Bsmall$B%5%$%:$K@Z$jBX$($k!#(B
  // $B@Z$jBX$($O;R$+$i9T$&!#(B
{
  Am_Value_List cands = small_list(objs);
  if (!cands.Empty()) play_sound(ORIG_SIZE);
  change_sizes_by_mode(cands, ORIG_SIZE);
  cands.Make_Empty();
}

void change_small_then_large(Am_Value_List smalls, Am_Value_List larges)
{
  //$B=L$a$k$Y$-%*%V%8%'%/%H$N%j%9%H$r:n$k!#(B
  Am_Value_List all_smalls = objs_and_their_parents(smalls);
  //$B9-$2$k$Y$-%*%V%8%'%/%H$N%j%9%H$r:n$k!#(B
  Am_Value_List all_larges = objs_and_their_parents(larges);

  // smalls$B$N$J$+$K$D$.$K9-$2$k$Y$-$b$N$,F~$C$F$$$?$i>C$9!#(B
  Am_Value_List already_list = Am_Value_List();

  //  cout << "smalls before: " << all_smalls << endl;
  for (all_larges.Start(); !all_larges.Last(); all_larges.Next()) {
    Am_Object obj = all_larges.Get();
    all_smalls.Start();
    if (all_smalls.Member(obj)) {
      all_smalls.Delete();
      already_list.Add(obj);
    }
  }
  //  cout << "smalls filtered: " << all_smalls << endl;
  //  cout << "already: " << already_list << endl;

  if (!all_smalls.Empty()) {
    //    play_sound(ORIG_SIZE);
    change_sizes_by_mode(all_smalls, ORIG_SIZE);
  }

  //  cout << "larges before: " << all_larges << endl;
  /*
  for (already_list.Start(); !already_list.Last(); already_list.Next()) {
    Am_Object obj = already_list.Get();
    all_larges.Start();
    if (all_larges.Member(obj)) {
      all_larges.Delete();
    }
  }
  */
  //  cout << "larges filtered: " << all_larges << endl;
  if (!all_larges.Empty()) {
    //    play_sound(EDIT_SIZE);
    change_sizes_by_mode(all_larges, EDIT_SIZE);
  }
  all_smalls.Make_Empty();
  all_larges.Make_Empty();
  already_list.Make_Empty();
}

// Activate/Dismiss Adjust and Repeat Button
// ----------------------------------------------------------------------
void activate_undo_buttons(Am_Object window, Am_Object target, Am_Object selection, int mode, Am_Inter_Location cur_size)
{
  Am_Object undo_buttons = selection.Get_Part(undoButtons);
  Am_Object undo_feedback = selection.Get_Part(undoFeedback);

  int x, y;
  Am_Translate_Coordinates(target, (int)target.Get(Am_WIDTH) / 2,
			   (int)target.Get(Am_HEIGHT) / 2, selection, x, y);

  undo_buttons
    .Set(Am_LEFT, x)
    .Set(Am_TOP, y)
    .Set(TargetObj, target)
    .Set(undoMode, mode)
    .Set(parentList, Am_Value_List().Add(target))
    .Set(sortedHistory, Am_Value_List().Add(cur_size))
    .Set(Am_VISIBLE, true)
    ;

  undo_feedback
    .Set(TargetObj, target)
    .Set(Am_VISIBLE, true)
    ;
}

Am_Define_Method(Am_Object_Method, void, invisible_undo_buttons,
		 (Am_Object command_obj)) {
  command_obj.Get_Owner().Get_Owner()
    .Get_Part(undoButtons)
    .Set(Am_VISIBLE, false)
    .Get_Owner()
    .Get_Part(undoFeedback)
    .Set(Am_VISIBLE, false)
    .Get_Owner();
}

// link$B$rC)$k(B
//----------------------------------------------------------------------

Am_Value_List check_list_for_children_link(Am_Object des)
// $B3,AX9=B$$K$*$1$k7;Do%N!<%I$N%j%9%H$r@8@.(B
{
  Am_Value_List nodes = Am_Value_List();

  Am_Object parent = des.Get_Owner();
  Am_Value_List children = parent.Get(Am_GRAPHICAL_PARTS);

  for (children.Start(); !children.Last(); children.Next()) {
    Am_Object child = children.Get();
    if (!(child == des)) nodes.Add(child);
  }
  return nodes;
}

Am_Value_List check_list_for_hyper_link(Am_Object src, Am_Object des)
// $B%j%s%/9=B$$K$*$1$k7;Do%N!<%I$N%j%9%H$r@8@.(B
{
  Am_Value_List nodes = Am_Value_List().Add(src);

  Am_Value_List links = src.Get(hyperLinks);
  for (links.Start(); !links.Last(); links.Next()) {
    Am_Object link = links.Get();
    if (!(link == des)) nodes.Add(link);
  }
  return nodes;
}

void Retrieve_Child_Link(Am_Object des)
{
  // change_size_by_mode(des, EDIT_SIZE);
  change_to_large_size(des);
  return;

  Am_Value_List list = check_list_for_children_link(des);
  for (list.Start(); !list.Last(); list.Next()) {
    Am_Object obj = list.Get();
    if (!intend_to_edit(obj)) {
      change_to_small_size(obj);
    }
  }
}

void Retrieve_Hyper_Link(Am_Object src, Am_Object des, Am_Object selection)
{
  Am_Object win = des.Get(Am_WINDOW);
  Am_Inter_Location cur_size = des.Get(currPointer);

  //  activate_undo_buttons(win, des, selection, EDIT_SIZE, cur_size);
  change_to_large_size(des);

  Am_Value_List list = check_list_for_hyper_link(src, des);
  for (list.Start(); !list.Last(); list.Next()) {
    Am_Object obj = list.Get();
    if (!intend_to_edit(obj)) {
      change_to_small_size(obj);
    }
  }
}


// Popup Menu Methods
// ----------------------------------------------------------------------
Am_Define_Method(Am_Object_Method, void, Orig_Size_Method, (Am_Object command_obj))
{
  Am_Object menu = command_obj.Get_Owner().Get_Owner(); // $B%a%K%e!<(B
  Am_Object target = menu.Get(TargetObj); // undo$B$9$k?^7A(B
  Am_Object window = menu.Get_Owner();
  Am_Object selection = target.Get_Object(Am_WINDOW).Get(Am_SELECTION_WIDGET);
  
  Am_Finish_Pop_Up_Waiting(window, Am_No_Value);

  // origPointer$B$O$"$k$+(B
  if (target.Get_Slot_Type(origPointer) == Am_NONE) return;

  Am_Inter_Location cur_size = target.Get(currPointer);
  //  activate_undo_buttons(window, target, selection, ORIG_SIZE, cur_size);
  if (!is_small_size(target)) play_sound(ORIG_SIZE);
  change_size_by_mode(target, ORIG_SIZE);
}

Am_Define_Method(Am_Object_Method, void, Edit_Size_Method, (Am_Object command_obj))
{
  Am_Object menu = command_obj.Get_Owner().Get_Owner(); // $B%a%K%e!<(B
  Am_Object target = menu.Get(TargetObj); // undo$B$9$k?^7A(B
  Am_Object window = menu.Get_Owner();
  Am_Object selection = target.Get_Object(Am_WINDOW).Get(Am_SELECTION_WIDGET);

  Am_Finish_Pop_Up_Waiting(window, Am_No_Value);

  Am_Inter_Location cur_size = target.Get(currPointer);
  //  activate_undo_buttons(window, target, selection, EDIT_SIZE, cur_size);

  //  Retrieve_Child_Link(target);
  change_to_large_size(target);
}

Am_Define_Method(Am_Object_Method, void, Set_Orig_Size_Method, (Am_Object command_obj))
{
  Am_Object menu = command_obj.Get_Owner().Get_Owner(); // $B%a%K%e!<(B
  Am_Object target = menu.Get(TargetObj); // set$B$9$k?^7A(B
  Am_Object window = menu.Get_Owner();

  Am_Finish_Pop_Up_Waiting(window, Am_No_Value);

  Am_Inter_Location cur_size = target.Get(currPointer);
  set_size_by_mode(target, ORIG_SIZE, cur_size);
}

Am_Define_Method(Am_Object_Method, void, Set_Edit_Size_Method, (Am_Object command_obj))
{
  Am_Object menu = command_obj.Get_Owner().Get_Owner(); // $B%a%K%e!<(B
  Am_Object target = menu.Get(TargetObj); // set$B$9$k?^7A(B
  Am_Object window = menu.Get_Owner();

  Am_Finish_Pop_Up_Waiting(window, Am_No_Value);

  Am_Inter_Location cur_size = target.Get(currPointer);
  set_size_by_mode(target, EDIT_SIZE, cur_size);
}

Am_Define_Method(Am_Object_Method, void, Toggle_Size_Method, (Am_Object command_obj))
{
  Am_Object target = command_obj.Get_Owner().Get_Owner(); // undo$B$9$k?^7A(B
  Am_Object window = target.Get(Am_WINDOW);
  //  Am_Object selection = command_obj.Get(Am_SELECTION_WIDGET);
  Am_Object selection = target.Get_Object(Am_WINDOW).Get(Am_SELECTION_WIDGET);

  Am_Inter_Location cur_size = target.Get(currPointer);
  Am_Inter_Location small_size = target.Get(origPointer);
  Am_Inter_Location large_size = target.Get(editPointer);

  if (equal_size(cur_size, large_size)) {
    //    activate_undo_buttons(window, target, selection, ORIG_SIZE, cur_size);
    play_sound(ORIG_SIZE);
    change_size_by_mode(target, ORIG_SIZE);
  } else if (equal_size(cur_size, small_size)) {
    //    activate_undo_buttons(window, target, selection, EDIT_SIZE, cur_size);
    //    Retrieve_Child_Link(target);
    change_to_large_size(target);
  }
}

// UndoButtons' Methods
// --------------------

void set_children_orig_sizes(Am_Object target)
  //$BM=B,(B
{
  Am_Object child_group = target.Get(propagateChildren);
  if (!child_group.Valid()) return;
  if (!child_group.Is_Instance_Of(AlignZoomGroup)) return;
  Am_Value_List children = child_group.Get(Am_GRAPHICAL_PARTS);
  for (children.Start(); !children.Last(); children.Next()) {
    Am_Object child = children.Get();

    Am_Inter_Location cur_size = child.Get(currPointer);
    Am_Inter_Location orig_size = child.Get(origPointer);
    Am_Inter_Location edit_size = child.Get(editPointer);

    if (area(cur_size) < (area(orig_size) + area(edit_size)) / 2) {
      set_size_by_mode(child, ORIG_SIZE, cur_size);
    } else {
      set_size_by_mode(child, EDIT_SIZE, cur_size);
    }
  }
}

// $B;R$,Bg$-$/$J$C$F?F$NBg$-$5$rJQ$($k$H$-$J$K$,5/$3$k$+!#(B
// set_parent_editing...$B!!$3$N$H$-?F$O85$N$^$^(B
// $B?F$N%j%5%$%:(B small large $B6&$KJQ$o$i$J$$$7$+$b(BcurrPointer$B$rJQ99$7$J$$!#(B
// $B$7$?$,$C$F<!$N(Bset_parent..$B$G$b(Bsmall large$B$OJQ99$5$l$J$$!#(B
// $B7+$jJV$9!#(B

Am_Define_Method(Am_Object_Method, void, Smaller_Size_Method, (Am_Object command_obj))
{
  // undobuttons -- button -- one_shot_inter -- command_obj
  Am_Object buttons = command_obj.Get_Owner().Get_Owner().Get_Owner();

  Am_Value_List paren_list = buttons.Get(parentList);
  paren_list.Start();
  if (paren_list.Empty()) return;
  Am_Object target = paren_list.Get();
  if (target.Get_Slot_Type(sortedHistory) == Am_NONE) return;

  Am_Inter_Location cur_loc = target.Get(currPointer);

  Am_Value_List history = target.Get(sortedHistory);
  for (history.End(); !history.First(); history.Prev()) {
    Am_Inter_Location loc = history.Get();
    if (is_smaller_size(loc, cur_loc)) {
      change_size_to_loc(target, loc);
      //      cout << "Adjust ";
      set_size_by_mode(target, (int)buttons.Get(undoMode), loc);
      break;
    }
  }
}

Am_Define_Method(Am_Object_Method, void, Larger_Size_Method, (Am_Object command_obj))
{
  // undobuttons -- button -- one_shot_inter -- command_obj
  Am_Object buttons = command_obj.Get_Owner().Get_Owner().Get_Owner();

  Am_Value_List paren_list = buttons.Get(parentList);
  paren_list.Start();
  if (paren_list.Empty()) return;
  Am_Object target = paren_list.Get();
  if (target.Get_Slot_Type(sortedHistory) == Am_NONE) return;

  Am_Inter_Location cur_loc = target.Get(currPointer);

  Am_Value_List history = target.Get(sortedHistory);
  for (history.Start(); !history.Last(); history.Next()) {
    Am_Inter_Location loc = history.Get();
    if (is_smaller_size(cur_loc, loc)) {
      change_size_to_loc(target, loc);
      //      cout << "Adjust ";
      set_size_by_mode(target, (int)buttons.Get(undoMode), loc);      
      break;
    }
  }
}

Am_Define_Method(Am_Object_Method, void, Parent_Size_Method, (Am_Object command_obj))
{
  // undobuttons -- button -- one_shot_inter -- command_obj
  Am_Object buttons = command_obj.Get_Owner().Get_Owner().Get_Owner();
  Am_Object feedback = buttons.Get_Owner().Get_Part(undoFeedback);
  // target$B$H(Bmode$B$r%2%C%H(B
  Am_Object target = buttons.Get(TargetObj);
  int mode = buttons.Get(undoMode);

  buttons.Make_Unique(sortedHistory);
  buttons.Make_Unique(parentList);

  Am_Value_List size_list = buttons.Get(sortedHistory);
  Am_Value_List paren_list = buttons.Get(parentList);

  paren_list.Start();
  if (paren_list.Empty()) return;
  Am_Object top = paren_list.Get();
  // top$B$N?F$O(BAlignZoom$B$=$N(BpropagateParent$B$,??$N?F(B
  if (top.Get_Owner().Get_Slot_Type(propagateParent) == Am_NONE) return;
  Am_Object parent = top.Get_Owner().Get(propagateParent);
  if (!parent.Valid()) return;
  if (parent.Get_Slot_Type(cellObject) == Am_NONE) return;

  feedback.Set(TargetObj, parent);

  Am_Inter_Location cur_size = parent.Get(currPointer);
  paren_list.Add(parent, Am_HEAD, false);
  size_list.Add(cur_size, Am_HEAD, false);
  change_size_by_mode(parent, mode);

  buttons.Note_Changed(sortedHistory);
  buttons.Note_Changed(parentList);
}

Am_Define_Method(Am_Object_Method, void, Child_Size_Method, (Am_Object command_obj))
{
  // undobuttons -- button -- one_shot_inter -- command_obj
  Am_Object buttons = command_obj.Get_Owner().Get_Owner().Get_Owner();

  // target, mode$B$r$2$C$H(B
  Am_Object target = buttons.Get(TargetObj);
  int mode = buttons.Get(undoMode);

  buttons.Make_Unique(sortedHistory);
  buttons.Make_Unique(parentList);

  Am_Value_List size_list = buttons.Get(sortedHistory);
  Am_Value_List paren_list = buttons.Get(parentList);
  size_list.Start();
  paren_list.Start();

  // $B?F$rC)$C$?MzNr$,$J$1$l$P$J$K$b$7$J$$(B
  if (paren_list.Length() <= 1) return;
  Am_Object top = paren_list.Get();
  
  if (!top.Valid()) return;
  if (top.Get_Slot_Type(cellObject) == Am_NONE) return;

  // $B%5%$%:$r85$KLa$7!"MzNr$r:o=|$9$k(B
  Am_Inter_Location undo_size = size_list.Get();
  change_size_to_loc(top, undo_size);

  paren_list.Delete(false);
  size_list.Delete(false);

  buttons.Note_Changed(sortedHistory);
  buttons.Note_Changed(parentList);

  // feedback$B$NJQ99(B
  Am_Object feedback = buttons.Get_Owner().Get_Part(undoFeedback);
  paren_list.Next();
  Am_Object next = paren_list.Get();
  feedback.Set(TargetObj, next);
}

Am_Define_Style_Formula(UndoButtonColorFormula)
{
  Am_Object buttons = self.GV_Owner();
  if (!buttons.Valid()) return Am_Black;
  if ((int)buttons.GV(undoMode) == ORIG_SIZE)
    return Am_Style(0, 0, 1, 10, Am_CAP_BUTT, Am_JOIN_MITER,
				 Am_LINE_SOLID, Am_DEFAULT_DASH_LIST,
				 Am_DEFAULT_DASH_LIST_LENGTH,
				 Am_FILL_STIPPLED, Am_FILL_POLY_EVEN_ODD,
				 Am_Image_Array(20));
  else
    return Am_Style(1, 0, 0, 10, Am_CAP_BUTT, Am_JOIN_MITER,
				 Am_LINE_SOLID, Am_DEFAULT_DASH_LIST,
				 Am_DEFAULT_DASH_LIST_LENGTH,
				 Am_FILL_STIPPLED, Am_FILL_POLY_EVEN_ODD,
				 Am_Image_Array(20));
}

Am_Define_Formula(bool, UndoUpArrowVisibleForm)
{
  Am_Object buttons = self.GV_Owner();
  if (!buttons.Valid()) return false;

  Am_Object target = buttons.GV(TargetObj);
  Am_Value_List paren_list = buttons.GV(parentList);
  paren_list.Start();
  if (paren_list.Empty()) return false;

  Am_Object top = paren_list.Get();
  if (top.GV_Owner().Get_Slot_Type(propagateParent) == Am_NONE) return false;
  Am_Object parent = top.GV_Owner().GV(propagateParent);
  if (!parent.Valid()) return false;

  return true;
}

Am_Define_Formula(bool, UndoDownArrowVisibleForm)
{
  Am_Object bgroup = self.GV_Owner();
  if (!bgroup.Valid()) return false;

  Am_Value_List plist = bgroup.GV(parentList);
  if (plist.Length() <= 1) return false;

  return true;
}

Am_Define_Formula(int, UndoFeedbackLeftForm)
{
  Am_Object target = self.GV(TargetObj);
  if (!target.Valid()) return 0;
  
  Am_Object win = target.Get(Am_WINDOW);
  int w = target.GV(Am_WIDTH);
  int h = target.GV(Am_HEIGHT);
  int x = target.GV(Am_LEFT);
  int y = target.GV(Am_TOP);

  Am_Translate_Coordinates(target.Get_Owner(), x, y, win, x, y);

  self
    .Set(Am_WIDTH, w)
    .Set(Am_HEIGHT, h)
    //    .Set(Am_LEFT, x)
    .Set(Am_TOP, y)
    ;
  return x;
}

// Global Objects

Am_Object OrigSizeCmd = 0;
Am_Object EditSizeCmd = 0;
Am_Object SetOrigCmd = 0;
Am_Object SetEditCmd = 0;
Am_Object ToggleSizeCmd = 0;
Am_Object UndoButtonGroup = 0;
Am_Object UndoFeedback = 0;



void initResizeByExample(void)
{
  PredictionStrategy = new prediction1();

  OrigSizeCmd = MenuCommand.Create()
    .Set(Am_LABEL, "Small Size")
    .Set(Am_DO_METHOD, Orig_Size_Method);

  EditSizeCmd = MenuCommand.Create()
    .Set(Am_LABEL, "Large Size")
    .Set(Am_DO_METHOD, Edit_Size_Method);

  SetOrigCmd = MenuCommand.Create()
    .Set(Am_LABEL, "Set Small Size")
    .Set(Am_DO_METHOD, Set_Orig_Size_Method);

  SetEditCmd = MenuCommand.Create()
    .Set(Am_LABEL, "Set Large Size")
    .Set(Am_DO_METHOD, Set_Edit_Size_Method);

  ToggleSizeCmd = Am_One_Shot_Interactor.Create()
    .Get_Part(Am_COMMAND)
    .Set(Am_DO_METHOD, Toggle_Size_Method)
    .Get_Owner();

  // Undo Widgets

  Am_Object arrow = Am_Arrow.Create("arrow")
    .Set(Am_POINT_LIST, ArrowPointListWithVariableWidthForm)
    ;

  Am_Object upArrow = arrow.Create("UpArrow")
    .Set(Am_POINT_LIST, ArrowPointListWithVariableWidthForm)
    .Set(Am_X1, 45)
    .Set(Am_Y1, 30)
    .Set(Am_X2, 45)
    .Set(Am_Y2, 0)
    .Set(Am_VISIBLE, UndoUpArrowVisibleForm)
    .Set(Am_ARROW_H_WIDTH, 30)
    .Set(Am_ARROW_H_HEIGHT, 15)
    .Set(Am_ARROW_WIDTH, 15)
    .Set(Am_LINE_STYLE, Am_Style(1, 0, 0, 3, Am_CAP_BUTT, Am_JOIN_MITER,
				 Am_LINE_SOLID, Am_DEFAULT_DASH_LIST,
				 Am_DEFAULT_DASH_LIST_LENGTH,
				 Am_FILL_SOLID, Am_FILL_POLY_EVEN_ODD,
				 Am_Image_Array(50)))
    //    .Set(Am_LINE_STYLE, Am_Black)
    .Set(Am_FILL_STYLE, UndoButtonColorFormula)
    ;

  Am_Object downArrow = upArrow.Create("downArrow")
    .Set(Am_X1, 45)
    .Set(Am_Y1, 40)
    .Set(Am_X2, 45)
    .Set(Am_Y2, 70)
    .Set(Am_VISIBLE, UndoDownArrowVisibleForm)
    ;

  Am_Object smallerRect = Am_Rectangle.Create("SmallerRect")
    .Set(Am_LEFT, 10)
    .Set(Am_TOP, 25)
    .Set(Am_WIDTH, 20)
    .Set(Am_HEIGHT, 20)
    .Set(Am_LINE_STYLE, Am_Style(1, 0, 0, 3, Am_CAP_BUTT, Am_JOIN_MITER,
				 Am_LINE_SOLID, Am_DEFAULT_DASH_LIST,
				 Am_DEFAULT_DASH_LIST_LENGTH,
				 Am_FILL_SOLID, Am_FILL_POLY_EVEN_ODD,
				 Am_Image_Array(50)))
    //    .Set(Am_LINE_STYLE, Am_Black)
    .Set(Am_FILL_STYLE, UndoButtonColorFormula)
    ;

  Am_Object largerRect = smallerRect.Create("LargerRect")
    .Set(Am_LEFT, 60)
    .Set(Am_TOP, 20)
    .Set(Am_WIDTH, 30)
    .Set(Am_HEIGHT, 30)
    ;


  Am_Object up_inter = Am_One_Shot_Interactor.Create()
    .Set(Am_PRIORITY, 5)
    .Get_Part(Am_COMMAND)
    .Set(Am_DO_METHOD, Parent_Size_Method)
    .Get_Owner()
    ;

  Am_Object down_inter = Am_One_Shot_Interactor.Create()
    .Set(Am_PRIORITY, 5)
    .Get_Part(Am_COMMAND)
    .Set(Am_DO_METHOD, Child_Size_Method)
    .Get_Owner()
    ;

  Am_Object smaller_inter = Am_One_Shot_Interactor.Create()
    .Set(Am_PRIORITY, 5)
    .Get_Part(Am_COMMAND)
    .Set(Am_DO_METHOD, Smaller_Size_Method)
    .Get_Owner()
    ;

  Am_Object larger_inter = Am_One_Shot_Interactor.Create()
    .Set(Am_PRIORITY, 5)
    .Get_Part(Am_COMMAND)
    .Set(Am_DO_METHOD, Larger_Size_Method)
    .Get_Owner()
    ;

  UndoButtonGroup = Am_Group.Create("UndoButtonGroup")
    .Set(Am_WIDTH, 90)
    .Set(Am_HEIGHT, 70)
    .Set(TargetObj, 0)
    .Set(undoMode, ORIG_SIZE)
    .Set(sortedHistory, Am_Value_List())
    .Set(parentList, Am_Value_List())
    .Set(Am_VISIBLE, false)
    //    .Add_Part(upArrow.Add_Part(up_inter))
    //    .Add_Part(downArrow.Add_Part(down_inter))
    .Add_Part(smallerRect.Add_Part(smaller_inter))
    .Add_Part(largerRect.Add_Part(larger_inter))
    .Add_Part(Am_Line.Create().Set(Am_X1, 10).Set(Am_Y1, 25).Set(Am_X2, 60).Set(Am_Y2, 20))
    .Add_Part(Am_Line.Create().Set(Am_X1, 10).Set(Am_Y1, 45).Set(Am_X2, 60).Set(Am_Y2, 49))
    ;

  UndoFeedback = Am_Rectangle.Create("UndoFeedback")
    .Set(TargetObj, 0)
    .Set(Am_VISIBLE, false)
    .Set(Am_LEFT, UndoFeedbackLeftForm)
    .Set(Am_LINE_STYLE, Am_Style(1, 0, 0, 3, Am_CAP_BUTT, Am_JOIN_MITER,
				 Am_LINE_SOLID, Am_DEFAULT_DASH_LIST,
				 Am_DEFAULT_DASH_LIST_LENGTH, Am_FILL_SOLID,
				 Am_FILL_POLY_EVEN_ODD, Am_No_Image))
    .Set(Am_FILL_STYLE, Am_No_Style)
    ;

}



// ----------------------------------------------------------------------
// Customized Stop Method for grow_inter in the selection widget
// ----------------------------------------------------------------------
Am_Object move_grow_set_impl_command(Am_Object inter,
				     Am_Object object_modified,
				     Am_Inter_Location data);

Am_Inter_Location move_grow_interim_val(Am_Object inter,
			   Am_Object command_obj, Am_Object object_to_move ,
			   int x, int y, Am_Object ref_obj, Am_Input_Char ic);

Am_Define_Method(Am_Inter_Internal_Method, void, my_Am_Move_Grow_Stop_Method,
		 (Am_Object& inter, Am_Object& /* object */,
		  Am_Object& event_window, Am_Input_Event *ev)) {
  int x = ev->x;
  int y = ev->y;
  // ignore object parameter, used saved object
  Am_Object object;
  object = inter.Get(Am_START_OBJECT);
  Am_INTER_TRACE_PRINT(inter, "Move_Grow stopping over " << object);
  //now, call interim_do on last point
  Am_Input_Char ic = ev->input_char;
  Am_Object command_obj;
  command_obj = inter.Get_Part(Am_COMMAND);

  // **********************************************************************
  // modified by toyoda: cancel interim val at last
  Am_Inter_Location data (inter.Get(Am_INTERIM_VALUE));
  // **********************************************************************
  
  // call DO method
  //data is points calculated by move_grow_interim_val
  Am_Call_Final_Do_And_Register(inter, command_obj, x, y, event_window, ic,
				object, data, move_grow_set_impl_command);

  // LAST, call the prototype's method
  Am_Inter_Internal_Method inter_method;
  inter_method = Am_Interactor.Get(Am_INTER_STOP_METHOD);
  inter_method.Call(inter, object, event_window, ev);
}

void Am_Check_And_Fix_Owner_For_Object(Am_Object &obj, Am_Inter_Location &data);
void Am_Copy_Data_Into_Slot(Am_Object dest_obj, Am_Slot_Key dest_slot, Am_Inter_Location src_data);
bool Am_Check_Final_Point_In_Multi(Am_Object inter);

Am_Define_Method(Am_Current_Location_Method, void, my_Am_Move_Grow_Do,
		 (Am_Object inter, Am_Object object_modified,
		  Am_Inter_Location data)) {
  Am_Object feedback;
  Am_Object inter_window = inter.Get(Am_WINDOW);
  feedback = inter.Get(Am_FEEDBACK_OBJECT);
  bool growing = inter.Get(Am_GROWING);
  if (feedback.Valid ()) {
    Am_REPORT_VIS_CONDITION(Am_INTER_TRACE_SETTING, inter, feedback, false);
    feedback.Set(Am_VISIBLE, false);
    if (feedback.Is_Instance_Of(Am_Window)) {
      Am_Add_Remove_Inter_To_Extra_Window(inter, feedback, false, false);
      //special test to see if need to figure out new window for the object
      inter_window =
	Am_Check_And_Fix_Point_In_Feedback_Window(inter_window, feedback);
      if (!inter_window.Valid()) { //release outside all windows
	Am_INTER_TRACE_PRINT(inter, "Aborting " << inter <<
			     " because stopped outside all windows");
	if ((bool)inter.Get(Am_INTER_BEEP_ON_ABORT))
	  Am_Beep();
	Am_Abort_Interactor(inter);
	return;
      }
    }
  }
  //if multi-window, abort if final point is outside of the windows
  // ** Is this always what is desired? **
  if (!Am_Check_Final_Point_In_Multi(inter)) {
	Am_INTER_TRACE_PRINT(inter, "Aborting " << inter <<
		      " because stopped outside windows in Am_MULTI_OWNERS");
	if ((bool)inter.Get(Am_INTER_BEEP_ON_ABORT))
	  Am_Beep();
	Am_Abort_Interactor(inter);
	return;
      }
  if (object_modified.Valid ()) {
    if (!growing) {
      //see if need to move the real object
      if (!Am_Check_And_Fix_Object_Window(object_modified,inter,inter_window))
	return;
    }
    // **********************************************************************
    // modified by toyoda: cancel position modification at last
    //    Am_Modify_Object_Pos(object_modified, data, growing, inter);
    // **********************************************************************
  }
    
  //set VALUE of inter with final location
  Am_Copy_Data_Into_Slot(inter, Am_VALUE, data);
  inter.Set(Am_OBJECT_MODIFIED, object_modified);
  Am_Copy_Values_To_Command(inter);
}


Am_Define_Formula(bool, UndoArrowGroupVisibleForm)
{
  Am_Object owner = self.GV_Owner();  if (!owner.Valid()) return false;
  Am_Value_List selection = owner.GV(Am_VALUE);

  if (selection.Empty()) return false;
  selection.End();
  Am_Object last_selected = selection.Get();
  int x = last_selected.GV(Am_LEFT);
  int y = last_selected.GV(Am_TOP);
  int w = last_selected.GV(Am_WIDTH);
  int h = last_selected.GV(Am_HEIGHT);

  Am_Translate_Coordinates(last_selected.Get_Owner(), x, y, owner, x, y);

  self.Set(Am_LEFT, x + (w / 2) - 35)
      .Set(Am_TOP, y + (h / 2) - 15)
    ;
  return true;
}

// Methods for selection widget
// ----------------------------------------------------------------------
Am_Define_Method(Am_Where_Method, Am_Object, Selection_Where_Test,
		 (Am_Object inter, Am_Object /* object */, Am_Object event_window,
		  Am_Input_Char /* ic */, int x, int y))
{
  Am_Object widget = inter.Get_Owner();
  Am_Value_List items = widget.Get(selectableObjects);

  Am_Object below_root = widget.Get_Object(Am_OPERATES_ON);
  Am_Object obj = Content_Instance_Of(below_root, x, y, event_window, items);
  if (obj.Valid()) return obj;
  return widget; // click on background
}

Am_Define_Method(Am_Where_Method, Am_Object, Move_Where_Test,
		 (Am_Object inter, Am_Object /* object */, Am_Object event_window,
		  Am_Input_Char /* ic */, int x, int y))
{
  Am_Object widget = inter.Get_Owner();
  Am_Value_List items = widget.Get(selectableObjects);

  Am_Object below_root = widget.Get_Object(Am_OPERATES_ON);
  Am_Object obj = Content_Instance_Of(below_root, x, y, event_window, items);
  if (obj.Valid()) return obj;
  return Am_No_Object; // widget $B$rF0$+$5$J$$$h$&$K(B
}

Am_Define_Method(Am_Object_Method, void, Move_Do, (Am_Object command_obj))
{
  PredictionStrategy->Move(0,0,0);
}

// Drag & Drop Support

Am_Object DropPointFeedback = 0;

DEFINE_SLOT(DropPoints);
DEFINE_SLOT(DropMethod);
DEFINE_SLOT(CheckDropMethod);

void set_commands_for_move_(Am_Object widget, Am_Object inter,
			    Am_Value object_modified, Am_Value old_value,
 			    Am_Value new_value, int final_left, int final_top) {
  Am_Object inter_command, widget_command;
  inter_command = inter.Get_Part(Am_COMMAND);
  if (inter_command.Valid()) {
    inter_command.Set(Am_OLD_VALUE, old_value);
    inter_command.Set(Am_VALUE, new_value);
    inter_command.Set(Am_OBJECT_MODIFIED, object_modified);
    inter_command.Set(Am_LEFT, final_left);
    inter_command.Set(Am_TOP, final_top);
  }
  widget_command = widget.Get_Part(Am_MOVE_GROW_COMMAND);
  if (widget_command.Valid()) {
    widget_command.Set(Am_OLD_VALUE, old_value);
    widget_command.Set(Am_VALUE, new_value);
    widget_command.Set(Am_OBJECT_MODIFIED, object_modified);
    widget_command.Set(Am_GROWING, false);
  }
}

Am_Define_Method(Am_Current_Location_Method, void, Drag_And_Drop_Do,
		 (Am_Object inter, Am_Object object_modified,
		  Am_Inter_Location data)) {
  Am_Object window = inter.Get(Am_WINDOW);
  Am_Object widget = inter.Get_Owner();

  // get current x and y values w.r.t. window
  bool as_line;
  Am_Object ref;
  int x, y, w, h;
  data.Get_Location(as_line, ref, x, y, w, h);
  //  cout << x << ", " << y << ", " << w << ", " << h << ", " << ref << endl;
  Am_Translate_Coordinates(ref, x, y, window, x, y);
  //  cout << x << ", " << y << endl;

  // investigate destination object under the cursor
  int xoff = inter.Get(Am_X_OFFSET);
  int yoff = inter.Get(Am_Y_OFFSET);
  Am_Object top = widget.Get(Am_OPERATES_ON);
  Am_Value_List droppts = object_modified.Get(DropPoints);
  Am_Object target = Content_Instance_Of(top, x + xoff, y + yoff, window, droppts);

  bool success = false;
  if (target.Valid()) {
    Am_Object feedback;
    feedback = inter.Get(Am_FEEDBACK_OBJECT);
    if (feedback.Valid ()) 
      feedback.Set(Am_VISIBLE, false);
    else if (object_modified.Valid ()) {
      bool growing = inter.Get(Am_GROWING);
      Am_Modify_Object_Pos(object_modified, data, growing, inter);
    }
    DropPointFeedback.Set(Am_VISIBLE, false);

    Am_Object widget = inter.Get_Owner();
    Am_Value objs_value = (Am_Value)object_modified;
    Am_Value old_values_value = inter.Get(Am_OLD_VALUE);
    Am_Value new_values_value = (Am_Value)data;
    int final_left = object_modified.Get(Am_LEFT);
    int final_top = object_modified.Get(Am_TOP);
    set_commands_for_move_(widget, inter, objs_value, old_values_value,
			  new_values_value, final_left, final_top);

    // drop $B$9$k%*%V%8%'%/%H$N(BCheckDropMethod$B$G%A%'%C%/$7!"(B
    // drop $B$9$k%*%V%8%'%/%H$N(BDropMethod$B$r<B9T$9$k!#(B
    Bin_Check_Method check_method = object_modified.Get(CheckDropMethod);
    Drop_Method drop_method = object_modified.Get(DropMethod);
    if (check_method.Valid() && drop_method.Valid()) {
      success = check_method.Call(object_modified, target);
      if (success) {
	drop_method.Call(object_modified, target, inter.Get(Am_COMMAND));
      }
    }
  }
  if (!success) {
    Am_Current_Location_Method orig_do = Am_Selection_Widget
      .Get_Part(Am_MOVE_INTERACTOR).Get(Am_DO_METHOD);
    orig_do.Call(inter, object_modified, data);
  }
}

Am_Define_Method(Am_Object_Method, void, Drag_Interim_Do, (Am_Object command_obj))
{
  Am_Object inter = command_obj.Get_Owner();
  Am_Object window = inter.Get(Am_WINDOW);
  Am_Object widget = inter.Get_Owner();

  // get current x and y values w.r.t. window
  Am_Inter_Location interim_loc = inter.Get(Am_INTERIM_VALUE);
  bool as_line;
  Am_Object ref;
  int x, y, w, h;
  interim_loc.Get_Location(as_line, ref, x, y, w, h);
  Am_Translate_Coordinates(ref, x, y, window, x, y);

  // check whether the moved object has CheckDropMethod
  Am_Object source = inter.Get(Am_OBJECT_MODIFIED);
  Bin_Check_Method check_method = source.Get(CheckDropMethod);
  if (!check_method.Valid()) return;

  // investigate destination object under the cursor
  int xoff = inter.Get(Am_X_OFFSET);
  int yoff = inter.Get(Am_Y_OFFSET);
  Am_Object top = widget.Get(Am_OPERATES_ON);
  Am_Value_List droppts = source.Get(DropPoints);
  Am_Object target = Content_Instance_Of(top, x + xoff, y + yoff, window, droppts);

  if (target.Valid() && check_method.Call(source, target)) {
    // translate coordinates w.r.t. destination owner into ones w.r.t. window
    Am_Object owner = target.Get_Owner();
    int left = target.Get(Am_LEFT);
    int top = target.Get(Am_TOP);
    int width = target.Get(Am_WIDTH);
    int height = target.Get(Am_HEIGHT);
    Am_Translate_Coordinates(owner, left, top, window, left, top);
    // make feedback object visible
    if (!DropPointFeedback.Get_Owner().Valid())
      window.Add_Part(DropPointFeedback);
    DropPointFeedback.Set(Am_VISIBLE, true)
      .Set(Am_LEFT, left)
      .Set(Am_TOP, top)
      .Set(Am_WIDTH, width)
      .Set(Am_HEIGHT, height);
  } else {
    // make feedback object invisible
    DropPointFeedback.Set(Am_VISIBLE, false);
  }
}

void initAlignZoomGroup(void)
{
  AlignZoomGroup = Am_Group.Create("AlignZoomGroup")
    .Set(gridObject, 0)
    .Set(seqDir, 0)
    .Set(partMoved, false)
    .Set(Am_ACTIVE, true)
    .Set(alignThreshold, 20)
    .Set(minimumSpace, 5)
    .Set(maximumPercentage, 90)
    .Set(detailThreshold, 30)
    .Set(propagateParent, 0)
    .Set(xScaleFactor, denom)
    .Set(yScaleFactor, denom)
    .Set(origWidth, 0)
    .Set(origHeight, 0)

    .Set(Am_WIDTH, 50)
    .Set(Am_HEIGHT, 50)
    .Set(Am_LAYOUT, AlignZoomLayout)
    ;

  Am_Object_Advanced azg_adv = (Am_Object_Advanced&)AlignZoomGroup;
  Am_Demon_Set azg_demon_set(azg_adv.Get_Demons().Copy());
  azg_demon_set.Set_Object_Demon(Am_CREATE_OBJ, alignzoom_create_demon);
  azg_demon_set.Set_Object_Demon(Am_COPY_OBJ, alignzoom_copy_demon);
  azg_demon_set.Set_Part_Demon(Am_ADD_PART, alignzoom_part_demon);
  azg_adv.Set_Demons(azg_demon_set);

  SequenceAlignZoomGroup = AlignZoomGroup.Create("SequenceAlignZoomGroup")
    .Set(Am_LAYOUT, SequenceAlignZoomLayoutForm)
    ;
  
  // Selection Widget
  // ----------------    
  AlignZoomSelectionWidget = Am_Selection_Widget.Create()
    .Set(Am_OPERATES_ON, OwnerForm)
    .Set(selectableObjects, 0)
    .Get_Part(Am_INTERACTOR)
      .Set(Am_START_WHERE_TEST, Selection_Where_Test)
      .Get_Part(Am_COMMAND)
        .Set(Am_START_DO_METHOD, invisible_undo_buttons)
      .Get_Owner()
    .Get_Owner()
    .Get_Part(Am_GROW_INTERACTOR)
      .Set(Am_FEEDBACK_OBJECT, NULL)
      .Set(Am_MINIMUM_WIDTH, 10)
      .Set(Am_MINIMUM_HEIGHT, 10)
      .Set(Am_INTER_STOP_METHOD, my_Am_Move_Grow_Stop_Method)
      .Set(Am_DO_METHOD, my_Am_Move_Grow_Do)
      .Get_Part(Am_COMMAND)
        .Set(Am_INTERIM_DO_METHOD, grow_all_selected_objs)
        .Set(Am_DO_METHOD, record_size_history)
      .Get_Owner()
    .Get_Owner()
    .Get_Part(Am_MOVE_INTERACTOR)
      .Set(Am_PRIORITY, 2)
      .Set(Am_START_WHERE_TEST, Move_Where_Test)
      .Set(Am_DO_METHOD, Drag_And_Drop_Do)
      .Get_Part(Am_COMMAND)
        .Set(Am_INTERIM_DO_METHOD, Drag_Interim_Do)
        .Set(Am_DO_METHOD, Move_Do)
      .Get_Owner()
    .Get_Owner()
    .Add_Part(undoButtons, UndoButtonGroup)
    .Add_Part(undoFeedback, UndoFeedback)
    ;
  Am_Object move_inter = AlignZoomSelectionWidget.Get_Part(Am_MOVE_INTERACTOR);
  AlignZoomSelectionWidget.Remove_Part(Am_MOVE_INTERACTOR);
  AlignZoomSelectionWidget.Add_Part(Am_MOVE_INTERACTOR, move_inter);

  DropPointFeedback = Am_Rectangle.Create()
    .Set(Am_LINE_STYLE, Am_No_Style)
    .Set(Am_FILL_STYLE, Am_Light_Gray_Stipple)
    ;
}

// ------------------------------------------------------------
// Zooming Text
// ------------------------------------------------------------

#define NofFontSizes	8
#define FontSizes	{60, 46, 34, 24, 18, 12, 10, 8}

Am_Font FontTable[NofFontFamilies][2][2][NofFontSizes];

Am_Object ZoomingText = 0;
Am_Object ZoomingMultiLineText = 0;
Am_Object ZoomingTextBrowser = 0;

// public
DEFINE_SLOT(TextPart);
DEFINE_SLOT(FontFamily);
DEFINE_SLOT(FontIsBold);
DEFINE_SLOT(FontIsItalic);
DEFINE_SLOT(FontIsUnderline);


char* make_font_name(int family, bool is_bold, bool is_italic, int size)
{
  char* wild_font_name = new char[50];  // Given hard-coded defaults, max length is 50
  char *family_part, *face_part;

  switch(family) {
  case FontFixed:      family_part = "courier"; break;
  case FontSerif:      family_part = "times"; break;
  case FontSansSerif:  family_part = "helvetica"; break;
  }
  
  if((is_bold == false) && (is_italic == false))
    face_part = "medium-r";
  else if ((is_bold == true) && (is_italic == false))
    face_part = "bold-r";
  else if ((is_bold == false) && (is_italic == true)) {
    if (family == Am_FONT_SERIF) face_part = "medium-i";
    else face_part = "medium-o";
  }
  else if ((is_bold == true) && (is_italic == true)) {
    if (family == Am_FONT_SERIF) face_part = "bold-i";
    else face_part = "bold-o";
  }
  // Guide to fonts.dir format (try xfontsel on an HP):
  // -fndry-fmly-weight-slant-sWidth-adstyl-pixelsize-pointsize-...
  // ...resx-resy-spc-avgWdth-registry-encoding
  sprintf(wild_font_name, "-*-%s-%s-*-*-%i-*-*-*-*-*-*-*", family_part, face_part, size);

  return wild_font_name;
}

Am_Font* update_font_table(Am_Constraint_Context& cc, Am_Object self)
{
  const int sizeList[NofFontSizes] = FontSizes;

  int  family       = self.GV(FontFamily);
  bool is_bold      = self.GV(FontIsBold);
  bool is_italic    = self.GV(FontIsItalic);
  //  bool is_underline = self.GV(FontIsUnderline);

  Am_Font* fontList = FontTable[family][is_bold][is_italic];
  if ((void *)fontList[0] == (void *)0) {
    for (int i = 0; i < NofFontSizes; ++i) {
      fontList[i] = Am_Font(make_font_name(family, is_bold, is_italic, sizeList[i]));
    }
  }
  return fontList;
}

Am_Define_Formula(int, ResizeTextForm) {
  Am_Font* fontList = update_font_table(cc, self);

  Am_Object textObj = self.Get_Part(TextPart);
  //  if (!Am_Screen.Get(Am_ACTIVE)) {
  //    textObj.Set(Am_VISIBLE, false);
  //    return 0;
  //  }
  //  textObj.Set(Am_VISIBLE, true);

  int w = self.GV(Am_WIDTH);
  int h = self.GV(Am_HEIGHT);
  int tw = 0;
  int th = 0;

  for (int i = 0; i < NofFontSizes; ++i) {
    textObj.Set(Am_FONT, fontList[i]);
    tw = textObj.GV(Am_WIDTH);
    th = textObj.GV(Am_HEIGHT);

    if (tw <= w + 4 && th <= h + 4) break;
  }
  if (tw <= w + 4) textObj.Set(Am_LEFT, w / 2 - tw / 2);
  else textObj.Set(Am_LEFT, 0);
  textObj.Set(Am_TOP, h / 2 - th / 2);
  return 0;
}

Am_Define_Formula(int, ResizeMultiTextForm)
{
  Am_Font* fontList = update_font_table(cc, self);
  int w = self.GV(Am_WIDTH);
  int h = self.GV(Am_HEIGHT);

  Am_Value_List texts = self.GV(Am_GRAPHICAL_PARTS);
  int lines = texts.Length();
  int top = 0;
  Am_Font font;
  for (texts.Start(); !texts.Last(); texts.Next(), top += h / lines) {
    Am_Object tx = texts.Get();
    tx.Set(Am_LEFT, 0).Set(Am_TOP, top);

    if (top == 0) {
      for (int i = 0; i < NofFontSizes; ++i) {
	font = fontList[i];
	tx.Set(Am_FONT, fontList[i]);
	int tw = tx.Get(Am_WIDTH);
	int th = tx.Get(Am_HEIGHT);
	if (tw <= w && th <= h / lines) break;
      }
    }
    tx.Set(Am_FONT, font);
  }
  return 1;
}

Am_Define_Formula(bool, InnerTextVisibleForm)
{
  return Am_Screen.GV(Am_ACTIVE);
}

Am_Object Create_Text_Browser(Am_String filename, Am_Object browser)
{
  if (!filename.Valid()) return 0;
  ifstream ifile;
  ifile.open(filename, ios::in);
  if (!ifile) return 0;

  char line[5000];
  while (ifile.getline(line, 5000)) {
    int length = strlen(line);
    Am_Object zline = ZoomingMultiLineText.Create();
    int h = 0;
    for (int i = 0; i < length; i += 80, ++h) {
      char text[81];
      strncpy(text, &line[i], 80);
      text[80] = '\0';
      Am_Object tobj = Am_Text.Create().Set(Am_TEXT, text);
      zline.Add_Part(tobj);
    }
    zline
      .Set(Am_WIDTH, Am_Width_Of_Parts)
      .Set(Am_HEIGHT, h * 8);
    browser.Get_Part(propagateChildren).Add_Part(zline);

    if (ifile.eof()) break;
  }
  ifile.close();
  return browser;
}


void initZoomingText(void)
{
  Am_Screen.Set(Am_ACTIVE, true);

  Am_Object textObj = Am_Text.Create("ZoomingTextPart")
    .Set_Single_Constraint_Mode(Am_TEXT, false);

  ZoomingText = Am_Group.Create("ZoomingText")
    .Set(FontFamily, FontFixed)
    .Set(FontIsBold, false)
    .Set(FontIsItalic, false)
    .Set(FontIsUnderline, false)
    .Set(Am_LAYOUT, ResizeTextForm)
    .Add_Part(TextPart, textObj
	      //	      .Set(Am_VISIBLE, InnerTextVisibleForm)
	      .Set(Am_TEXT, ""))
    ;

  ZoomingMultiLineText = Am_Group.Create("ZoomingMultiLineText")
    .Set(FontFamily, FontSansSerif)
    .Set(FontIsBold, false)
    .Set(FontIsItalic, false)
    .Set(FontIsUnderline, false)
    .Set(Am_LAYOUT, ResizeMultiTextForm)
    ;

  ZoomingTextBrowser = Am_Scrolling_Group.Create("ZoomingTextBrowser")
    .Set(Am_WIDTH, 400)
    .Set(Am_HEIGHT, 400)
    .Set(Am_INNER_FILL_STYLE, Am_White)
    .Set(Am_INNER_WIDTH, Am_Width_Of_Parts)
    .Set(Am_INNER_HEIGHT, Am_Height_Of_Parts)
    .Add_Part(propagateChildren, Am_Group.Create()
	      .Set(Am_WIDTH, Am_Width_Of_Parts)
	      .Set(Am_HEIGHT, Am_Height_Of_Parts)
	      .Set(Am_LAYOUT, Am_Vertical_Layout))
    .Set(DropPoints, 0)
    .Set(CheckDropMethod, 0)
    .Set(DropMethod, 0)
    ;
}

// ------------------------------------------------------------
// Initialize all widgets
// ------------------------------------------------------------
void Initialize_Zooming_Widgets(void) {
  bzero (FontTable, sizeof(FontTable));
  initResizeByExample();
  initAlignZoomGroup();
  initZoomingText();
}
