// Copyright (C) 1997  Toyoda Masashi (toyoda@is.titech.ac.jp)

#include <amulet.h>
#include OBJECT_ADVANCED__H

#include <strstream.h>

#include "ToyWidgets.h"
#include "PovlWidgets.h"
#include "Resources.h"
#include "Port.h"
#include "Pattern.h"
#include "Rule.h"
#include "ProcessMenu.h"
#include "Module.h"
#include "Binder.h"

#define _INTERNAL_
#include "Process.h"

// ------------------------------------------------------------
// Process Prototype
// ------------------------------------------------------------

// Global objects
// --------------------------------------------------
Am_Object ProcessModel = 0;
Am_Object ProcessProto = 0;

Am_Object HoleModel = 0;
Am_Object HoleProto = 0;
Am_Object ReplHolesProto = 0;

// Local Objects
// --------------------------------------------------
Am_Object ImplSelectionCmd = 0;
Am_Object ImplSelectionMenu = 0;
Am_Object ImplDialog = 0;

// Functions
// --------------------------------------------------

Am_Object Process_Find_Port(Am_Object proc, Am_String name)
{
  Am_Value_List port_list = proc.Get(PortList);
  Am_Object port = Object_List_Get_By_Model_Key(port_list, Name, name);

  if (port.Valid()) return port;
  return 0;
}

Am_Object ProcessFindPortForm(Am_Object proc, Am_String name, Am_Constraint_Context& cc)
{
  Am_Value_List port_list = proc.GV(PortList);
  Am_Object port = ObjectListGetByModelKeyForm(port_list, Name, name, cc);

  if (port.Valid()) return port;
  return 0;
}

bool Process_Equal_Interface(Am_Object proc1, Am_Object proc2)
{
  if (proc1.MGet(Name) != proc2.MGet(Name)) return false;
  if (proc1.MGet(ModuleName) != proc2.MGet(ModuleName)) return false;

  Am_Value_List ports1 = proc1.Get(PortList);
  Am_Value_List ports2 = proc2.Get(PortList);
  if (ports1.Length() != ports2.Length()) return false;

  for (ports1.Start(); !ports1.Last(); ports1.Next()) {
    Am_Object pt1 = ports1.Get();
    Am_Object pt2 = Process_Find_Port(proc2, pt1.MGet(Name));
    if (!pt1.Valid() || !pt2.Valid()) return false;
    if (!Port_Equal(pt1, pt2)) return false;
  }
  return true;
}

bool ProcessEqualInterfaceForm(Am_Object proc1, Am_Object proc2, Am_Constraint_Context& cc)
{
  Am_String proc1_name = proc1.MGV(Name);
  Am_String proc2_name = proc2.MGV(Name);
  if (!(proc1_name == proc2_name)) return false;
  if (!(proc1.MGV(ModuleName) == proc2.MGV(ModuleName))) return false;

  Am_Value_List ports1 = proc1.GV(PortList);
  Am_Value_List ports2 = proc2.GV(PortList);
  if (ports1.Length() != ports2.Length()) return false;

  for (ports1.Start(); !ports1.Last(); ports1.Next()) {
    Am_Object pt1 = ports1.Get();
    Am_Object pt2 = ProcessFindPortForm(proc2, pt1.MGV(Name), cc);
    pt2.MGV(Name);
    if (!pt1.Valid() || !pt2.Valid()) return false;
    if (!Port_Equal(pt1, pt2)) return false;
  }
  return true;
}

void Process_Set_Module(Am_Object proc, Am_Object module)
{
  proc.MSet(Module, module);
  proc.MSet(ModuleName, module.MGet(Name));
}

Am_Object Process_Get_Interface(Am_Object proc)
{
  switch ((int)proc.Get(Type)) {
  case ProcessInterface:
    return 0;
  case ProcessGoal:
    return proc.Get_Prototype();
  case ProcessInterGoal:
    Am_String module_name = proc.MGet(ModuleName);
    Am_String proc_name = proc.MGet(Name);

    Am_Object module = Module_Find_Module(module_name);
    if (!module.Valid()) return 0;
    Am_Object interface = Module_Find_Process(module, proc_name);
    return interface;
  }
}

// Goal Creation Functions
// --------------------------------------------------
void disable_name_edit(Am_Object goal)
{
  // $BL>A0$NJT=8$OIT2D(B
  Disable_Name_Edit(goal);
  // $B3F%]!<%H$NL>A0$NJT=8$bIT2D!"$*$h$S%a%K%e!<JQ99!#(B
  Am_Value_List ports = goal.Get(PortList);
  for (ports.Start(); !ports.Last(); ports.Next()) {
    Am_Object p = ports.Get();
    Disable_Name_Edit(p);
    Change_Menu(p, ProcessPortMenuPopper.Create());
  }
}

Am_Object Process_Goal_Create(Am_Object proc)
{
  Am_Object goal = Align_Zoom_Create_Part(proc);
  goal.Set(Type, ProcessGoal);
  // $BL>A0$NJT=8$rIT2D$K$9$k!#(B
  disable_name_edit(goal);
  // $B%k!<%k$N%"%$%3%s2=(B
  goal.Set(Am_VISIBLE, ContentsVisibleForm);
  // Menu$B$NJQ99!#(B
  Change_Menu(goal, GoalMenuPopper.Create());
  return goal;
}

Am_Define_Formula(bool, IsValidInterface)
{
  Am_String module_name = self.MGV(ModuleName);
  Am_String proc_name = self.MGV(Name);

  Am_Object module = ModuleFindModuleForm(module_name, cc);
  if (!module.Valid()) return false;

  Am_Object process = ModuleFindProcessForm(module, proc_name, cc);
  if (!process.Valid()) return false;

  return ProcessEqualInterfaceForm(process ,self, cc);
}

Am_Define_Style_Formula(InterGoalFillStyleForm)
{
  Am_Object process = self.GV_Owner();
  if (!process.Valid()) return 0;
  if ((bool)process.GV(Am_ACTIVE))
    return PovlResources.GV(ProcessFill);
  else
    return PovlResources.GV(ProcessInvalidFill);
}

Am_String inter_module_goal_name(Am_Object goal)
{
  Am_String module_name = goal.MGet(ModuleName);
  Am_String proc_name   = goal.MGet(Name);

  char name[1000];
  ostrstream oname(name, 1000);
  //  oname << module_name << ":" << proc_name << '\0';
  oname << proc_name << '\0';

  return (Am_String)name;
}

Am_Object setup_inter_module_goal(Am_Object goal)
{
  // $BL>A0$NJT=8$rIT2D$K$9$k!#(B
  disable_name_edit(goal);
  // $B>.$5$$%k!<%kFb$N%*%V%8%'%/%H?t$r8:$i$9!#(B
  goal.Set(Am_VISIBLE, ContentsVisibleForm);
  // Model.Module$B$r(Binvalid$B$K$9$k!#(B
  Am_Object module = goal.MGet(Module);
  if (module.Valid()) goal.MSet(ModuleName, module.MGet(Name));
  goal.MSet(Module, 0);
  // Am_ACTIVE$B$G%W%m%;%9$,(BValid$B$+%A%'%C%/$9$k!#(B
  goal.Set(Am_ACTIVE, IsValidInterface)
    .Get_Part(FramePart)
    .Set(Am_FILL_STYLE, InterGoalFillStyleForm);
  // Menu$B$NJQ99!#(B
  Change_Menu(goal, GoalMenuPopper.Create());
  // $BI=<($9$kL>A0$r(Bmodule:proc$B$KJQ99!#(B
  goal.Get_Part(NamePart).Get_Part(TextPart)
    .Set_Single_Constraint_Mode(Am_TEXT, true) // formula$B$NJ#?t$r$5$1$k!#(B
    .Set(Am_TEXT, inter_module_goal_name(goal));

  return goal;
}

Am_Object Process_Inter_Module_Goal_Create(Am_Object proc)
{
  Am_Object goal = Align_Zoom_Copy_Part(proc);
  goal.Set(Type, ProcessInterGoal);

  return setup_inter_module_goal(goal);
}

Am_Object Process_Inter_Module_Goal_New(Am_String name, Am_String mname)
{
  Am_Object goal = CALL1(ProcessProto, Copy_Method, NewMethod);
  goal.Set(Type, ProcessInterGoal);
  goal.MSet(Name, name);
  goal.MSet(ModuleName, mname);

  return setup_inter_module_goal(goal);
}

// Hole Replace Functions
// --------------------------------------------------
// $B!&(BHole$B$N%W%m%;%9$r%j%9%HCf$N$I$l$+$KF~$lBX$($k!#(B
// $B!&(BHole$B$K%W%m%;%9$rDI2C$9$k!#(B
//	$B!&$9$G$KF~$C$F$$$k$H$-$O(B,$B$=$l$r$I$1$FBeF~!#(B
//	$B!&F~$C$F$$$J$$$H$-$O(B,$B$=$N$^$^BeF~!#(B
// $B!&(BHole$B$+$i%W%m%;%9$r:o=|$9$k!#(B
//	$B!&8=:_$N%W%m%;%9$r:o=|$7(B,$B%j%9%H$N<!$N%W%m%;%9$KF~$lBX$(!#(B

// Process Removal
// ----------------------------------------
bool single_hole_remove(Am_Object proc)
// $B#1$D$N%[!<%k$+$i!"8=:_$N%W%m%;%9$r:o=|!#(B
{
  Am_Object hole = proc.Get(Hole);

  Am_Object proc_owner = proc.Get_Owner();
  proc_owner.Set(Am_ACTIVE, false);
  {
    int cx, cy, left, top;
    Center_Of_Object(proc, cx, cy);
    Left_Top_Of_Object_By_Center(hole, cx, cy, left, top);

    proc_owner.Add_Part(hole.Set(Am_LEFT, left).Set(Am_TOP, top));
    Am_Move_Object(hole, proc);
    proc.Remove_From_Owner();

    hole.Set(Process, 0);
  }
  proc_owner.Set(Am_ACTIVE, true);
  return true;
}

bool hole_remove(Am_Object proc)
// $BA4%[!<%k$+$i8=:_$N%W%m%;%9$r:o=|!#$?$@$78uJd%j%9%H$+$i$O>C$5$J$$!#(B
{
  Am_Value_List propagate_list = Change_Propagate_List(proc);
  for (propagate_list.Start(); !propagate_list.Last(); propagate_list.Next()) {
    Am_Object p = propagate_list.Get();

    single_hole_remove(p);
  }
  propagate_list.Make_Empty();
  return true;
}

bool hole_remove_from_list(Am_Object proc)
// $BA4%[!<%k$+$i8=:_$N%W%m%;%9$r:o=|!#8uJd%j%9%H$+$i$b>C$9!#(B
{
  Am_Value_List propagate_list = Change_Propagate_List(proc);
  for (propagate_list.Start(); !propagate_list.Last(); propagate_list.Next()) {
    Am_Object p = propagate_list.Get();

    single_hole_remove(p);
    Value_List_Slot_Remove(p.Get_Object(Hole), ProcessList, p);
    p.Set(Hole, 0);
  }
  propagate_list.Make_Empty();
  return true;
}

// Replace
// ----------------------------------------
Am_Object find_matching_port_by_distance(Am_Object hole_port, Am_Object proc)
{
  int min_dist = 999999;
  Am_Object min_pport = 0;

  Am_Value_List proc_ports = proc.Get(PortList);
  for (proc_ports.Start(); !proc_ports.Last(); proc_ports.Next()) {
    Am_Object pport = proc_ports.Get();
    int hport_mode = hole_port.Get(Mode);
    int pport_mode = Port_Outside_Mode(pport);
    if ((int)hole_port.MGet(Type) == (int)pport.MGet(Type) &&
	hport_mode == pport_mode) {
      int d = Distance_Of_Objects(hole_port, pport);
      if (d < min_dist) {
	min_dist = d;
	min_pport = pport;
      }
    }
  }
  return min_pport;
}

void bind_hole_port(Am_Object hport, Am_Object pport)
{
  Am_Value_List binders = hport.Get(HoleBinders);
  for (binders.Start(); !binders.Last(); binders.Next()) {
    Am_Object bl = binders.Get();
    Am_Object src = bl.Get(SrcPort);
    Am_Object dest = bl.Get(DestPort);
    if (src == hport) {
      Single_Binding_Line_Create(pport, dest);
    } else {
      Single_Binding_Line_Create(src, pport);
    }
  }
}

bool bind_hole_ports(Am_Object hole, Am_Object proc)
{
  Am_Value_List hole_ports = hole.Get(PortList);
  Am_Value_List proc_ports = proc.Get(PortList);

  for (hole_ports.Start(); !hole_ports.Last(); hole_ports.Next()) {
    Am_Object hport = (Am_Object)hole_ports.Get();
    Am_Object found_port = find_matching_port_by_distance(hport, proc);
    if (found_port.Valid()) {
      bind_hole_port(hport, found_port);      // $B7k$V(B
      // $B$$$^7k$P$l$?%]!<%H$r8uJd%j%9%H$+$i:o=|(B
      proc_ports.Start();
      proc_ports.Member(found_port);
      proc_ports.Delete(); // $B$[$s$H$K>C$5$l$k$o$1$G$O$J$$(B
    } else {
      return false;
    }
  }
  return true;
}

Am_Object hole_has_proc(Am_Object hole, Am_Object proc)
{
  // Hole$B$N%W%m%;%9%j%9%H$r%A%'%C%/(B
  Am_Value_List proc_list = hole.Get(ProcessList);
  for (proc_list.Start(); !proc_list.Last(); proc_list.Next()) {
    Am_Object p = proc_list.Get();
    if (Process_Equal_Interface(p, proc)) {
      return p;
    }
  }
  return 0;
}

bool Single_Hole_Replace(Am_Object hole, Am_Object proc)
  // proc$B$O%3%T!<:Q$_$NJ*(B
{
  // $B$9$G$K(Bproc$B$,F~$C$F$$$k$H$-$O%j%9%H$r$=$N$^$^$K$7$F:o=|(B
  Am_Object old_proc = hole.Get(Process);
  if (old_proc.Valid()) {
    if (Process_Equal_Interface(old_proc, proc)) return true;
    single_hole_remove(old_proc);
  }

  bool result = true;
  Am_Object hole_owner = hole.Get_Owner();   // Zoom layout$B$r0l;~L58z2=(B
  hole_owner.Set(Am_ACTIVE, false);  
  {
    // hole$B$HCf?4$r9g$o$;$?$H$-$N(Bproc$B$N:BI8$r5a$a$k!#(B
    int cx, cy, left, top;
    Center_Of_Object(hole, cx, cy); // hole$B$NCf1{$N:BI8(B
    Left_Top_Of_Object_By_Center(proc, cx, cy, left, top);
    proc.Set(Am_LEFT, left).Set(Am_TOP, top);

    // pattern$B$K(Bproc$B$rDI2C!#0LCV!JO@M}!&J*M}!K$r%[!<%k$N$H$3$m$X(B
    Am_Object pattern = hole.Get_Object(Parent);
    CALL2(pattern, Bin_Check_Method, AddMethod, proc);
    Am_Move_Object(proc, hole);

    // port$B$N7k9g(B
    if (!bind_hole_ports(hole, proc)) {
      // process$B$N7?<+BN$,9g$o$J$$$N$G:o=|!#(B
      proc.Remove_From_Owner();
      result = false;
    } else {
      hole.Remove_From_Owner();
      hole.Set(Process, proc);
      proc.Set(Hole, hole);
      // $B%j%9%H$XDI2C(B
      if (!hole_has_proc(hole, proc).Valid()) Value_List_Slot_Add(hole, ProcessList, proc);
      result = true;
    }
  }
  hole_owner.Set(Am_ACTIVE, true);  // Zoom layout $BI|5"(B
  return result;
}

bool Hole_Replace(Am_Object hole, Am_Object proc)
{
  // $B$9$G$K(Bproc$B$,F~$C$F$$$k$H$-$O%j%9%H$r$=$N$^$^$K$7$F:o=|(B
  Am_Object old_proc = hole.Get(Process);
  if (old_proc.Valid()) {
    if (Process_Equal_Interface(old_proc, proc)) return true;
    hole_remove(old_proc);
  }

  Am_Object copy = 0;
  Am_Value_List propagate_list = Change_Propagate_List(hole);
  for (propagate_list.Start(); !propagate_list.Last(); propagate_list.Next()) {
    Am_Object h = propagate_list.Get();

    // Hole$B$N%W%m%;%9%j%9%H$r%A%'%C%/!#%j%9%H$K$"$l$P!"$=$l$rDI2C!#(B
    // $B$J$1$l$P8e$G7?%A%'%C%/8e$K%j%9%H$KDI2C(B
    Am_Object p = hole_has_proc(h, proc);
    if (p.Valid()) {
      copy = p;
    } else {
      if (copy == 0) copy = proc; // $B:G=i$O0z?t$r$=$N$^$^(B
      else           copy = CALL1(proc, Copy_Method, CopyMethod); // $B<!0J9_$O%3%T!<$r(B
    }

    if (!Single_Hole_Replace(h, copy)) {
      Am_Show_Alert_Dialog (Am_Value_List()
			    .Add("Can't Instantiate")
			    .Add(hole.MGet(Name)).Add("with")
			    .Add(proc.MGet(Name)));
      return false;
    }
  }
  propagate_list.Make_Empty();
  return true;
}

// ----------------------------------------


bool Hole_Recover(Am_Object proc)
  // $B%[!<%k$+$i8=:_$N%W%m%;%9$r:o=|$7$?8e!"<!$N8uJd%W%m%;%9$KF~$lBX$((B
{
  Am_Object hole = proc.Get(Hole);
  hole_remove_from_list(proc);

  Am_Value_List proc_list = hole.Get(ProcessList);
  proc_list.Start();
  if (!proc_list.Empty()) {
    Am_Object new_proc = proc_list.Get();
    //    Hole_Replace_In_List(hole, new_proc);
    Hole_Replace(hole, new_proc);
  }

  return true;
}

// Implementation Selection Dialog Functions
// --------------------------------------------------

void Popup_Wait_Impl_Selection_Dialog(Am_Object hole, int x, int y)
{
  // Dialog$B$NL>A0(B
  Am_Object dialog = ImplDialog.Create()
    .Set(Am_ITEMS, Am_Value_List().Add(hole.MGet(Name)));

  cout << x << " " << y << endl;
  dialog
    .Set(Am_LEFT, x)
    .Set(Am_TOP, y);

  // Menu Item$B$N%j%9%H$r@8@.!#(B
  Am_Value_List menu_items = 0;
  Am_Value_List procs = hole.Get(ProcessList);
  for (procs.Start(); !procs.Last(); procs.Next()) {
    Am_Object proc = procs.Get();
    menu_items.Add(ImplSelectionCmd.Create()
		   .Set(Am_LABEL, proc.MGet(Name))
		   .Set(Hole, hole)
		   .Set(Process, proc));
  }
  dialog.Get_Part(Am_DIALOG_GROUP).Get_Part(Am_ITEMS).Get_Part(Am_ITEMS)
    .Set(Am_ITEMS, menu_items);

  Am_Object old_proc = hole.Get(Process);

  Am_String result = Am_Show_Dialog_And_Wait(dialog, false);
  if (!(result == "OK")) {
    //    if (old_proc.Valid()) Hole_Replace_In_List(hole, old_proc);
    if (old_proc.Valid()) Hole_Replace(hole, old_proc);
  }
}

Am_Define_Method(Am_Object_Method, void, Impl_Menu_Do, (Am_Object command_obj))
{
  Am_Object hole = command_obj.Get(Hole);
  Am_Object proc = command_obj.Get(Process);

  //  Hole_Replace_In_List(hole, proc);
  Hole_Replace(hole, proc);
}

// ------------------------------------------------------------
// Methods
// ------------------------------------------------------------

// Process Methods
// --------------------------------------------------

Am_Define_Method(Copy_Method, Am_Object, New_Process, (Am_Object /* proc */))
{
  Am_Object new_model = ProcessModel.Create();
  Am_Object new_proc  = ProcessProto.Create()
    .Set(Model, new_model)
    .Set(Am_WIDTH, 60)
    .Set(Am_HEIGHT, 60)
    ;
  Set_User_Name(new_proc, Creator, "", CreatedTime, 0);
  Set_User_Name(new_proc, Modifier, "", ModifiedTime, 0);
  return new_proc;
}

Am_Define_Method(Copy_Method, Am_Object, Copy_Process, (Am_Object proc))
{
  //  Am_Object new_proc = Process_Inter_Module_Goal_Create(proc);
  Am_Object new_proc = Align_Zoom_Copy_Part(proc);

  Set_User_Name(new_proc, Creator, "", CreatedTime, 0);
  Set_User_Name(new_proc, Modifier, "", ModifiedTime, 0);
  return new_proc;
}

Am_Define_Method(Naming_Method, bool, Process_Rename, (Am_Object process, Am_String name))
{
  Am_Object module = process.Get(Parent);
  if (module.Valid())
    if (Member_Name(module, ProcPatternNameList, name)) return false;
  process.Get_Object(Model).Set(Name, name);

  Set_User_Name(process, Modifier, "", ModifiedTime, 0);
  return true;
}

Am_Define_Method(Bin_Check_Method, bool, Process_Add, (Am_Object process, Am_Object port))
{
  if (!port.Is_Instance_Of(PortProto)) {
    cerr << "Can't add " << port << " to " << process << endl;
    return false;
  }

  if ((int)process.Get(Type) == ProcessInterGoal) {
    Disable_Name_Edit(port);
    Change_Menu(port, ProcessPortMenuPopper.Create());
  }

  Am_String port_name = port.Get_Object(Model).Get(Name); // original name
  Am_String new_name = Next_Name(process, PortNameList, port_name); // change if the same name exists
  port.Get_Object(Model).Set(Name, new_name); // directly rename
  process.Get_Part(ContentsPart).Add_Part(port);

  //  Am_Instance_Iterator goals = process;
  Am_Value_List goals = process.Get(ProcessList);
  for (goals.Start(); !goals.Last(); goals.Next()) {
    Am_Object g = goals.Get();
    Am_Object p = Align_Zoom_Create_Part(port);
    Disable_Name_Edit(p);
    Change_Menu(p, ProcessPortMenuPopper.Create());
    g.Get_Part(ContentsPart).Add_Part(p);
  }

  Am_Value_List rules = (Am_Value_List)process.Get(RuleList);
  for (rules.Start(); !rules.Last(); rules.Next()) {
     Am_Object rule = (Am_Object)rules.Get();
     CALL2(rule, Bin_Check_Method, AddMethod, port);
  }

  Set_User_Name(process, Modifier, "", ModifiedTime, 0);
  return true;
}

Am_Define_Method(Bin_Check_Method, bool, Process_Remove, (Am_Object process, Am_Object port))
{
  //  Remove_All_Binder(port);
  process.Get_Part(ContentsPart).Remove_Part(port);

  //  Am_Instance_Iterator goals = process;
  Am_Value_List goals = process.Get(ProcessList);
  for (goals.Start(); !goals.Last(); goals.Next()) {
    Am_Object gl = goals.Get();
    Am_Object pt = Proc_Get_Port(gl, (Am_String)port.MGet(Name));
    if (pt.Valid())
      CALL2(gl, Bin_Check_Method, RemoveMethod, pt);
  }

  Am_Value_List rules = process.Get(RuleList);
  for (rules.Start(); !rules.Last(); rules.Next()) {
    Am_Object rule = (Am_Object)rules.Get();
    Am_Value_List ports = rule.Get(PortList);
    for (ports.Start(); !ports.Last(); ports.Next()) {
      Am_Object pt = ports.Get();
      if ((Am_String)pt.MGet(Name) == (Am_String)port.MGet(Name))
	CALL2(rule, Bin_Check_Method, RemoveMethod, pt);
    }
  }
  Set_User_Name(process, Modifier, "", ModifiedTime, 0);
  return true;
}

Am_Define_Method(Bin_Check_Method, bool, Process_Check_Drop, (Am_Object source, Am_Object target))
{
  if ((int)source.Get(Type) != ProcessInterface) return false;
  Am_Object group = target.Get_Part(ContentsPart);
  if (source == target) return false; // $BF1$8%W%m%;%9$N>e$K$O%I%m%C%W$G$-$J$$!#(B
  if (source.Get_Owner() == group) return false;
  return true;
}

Am_Define_Method(Drop_Method, bool, Process_Drop, (Am_Object proc, Am_Object target, Am_Object cmd))
  // $B%k!<%k$K%I%m%C%W$7$?>l9g(B
  // $B0[$J$k%b%8%e!<%k$N%k!<%k$K%I%m%C%W$7$?>l9g(B
  // $B%Q%?!<%s$N%[!<%k$K%I%m%C%W$7$?>l9g(B
{
  Am_Object proc_model = proc.Get_Object(Model);
  Am_Object copy = 0;
  
  if (target.Is_Instance_Of(NetworkRuleProto)) {
    Am_Object rule_model = target.Get_Object(Model);
    if (proc_model.Get(Module) == rule_model.Get(Module)) {
      copy = Process_Goal_Create(proc);
    } else {
      copy = Process_Inter_Module_Goal_Create(proc);
    }
    Am_Object group = target.Get_Part(ContentsPart);
    cout << "Process_Drop: dropping " << proc << " in " << proc.Get_Owner() << " into " << group << endl;

    Set_User_Name(copy, Dropper, "", DroppedTime, 0);

    Am_Inter_Location new_loc = cmd.Get(Am_VALUE);
    bool as_line;
    Am_Object nref, oref;
    int nx, ny, w, h;
    new_loc.Get_Location(as_line, nref, nx, ny, w, h);
    Am_Translate_Coordinates(nref, nx, ny, group, nx, ny);

    copy.Set(Am_LEFT, nx).Set(Am_TOP, ny);
    group.Add_Part(copy);
    return true;
  } else if (target.Is_Instance_Of(HoleProto)) {
    // hole$B$NCV$-49$(!#(B
    copy = Process_Inter_Module_Goal_Create(proc);
    if (CALL2(target, Bin_Check_Method, AddMethod, copy)) {
      Set_User_Name(copy, Dropper, "", DroppedTime, 0);
      return true;
    }
    return false;
  } else if (target.Is_Instance_Of(ProcessProto)||
	     target.Is_Instance_Of(PatternProto)) {
    Am_Object hole = target.Get(Hole);
    if (hole.Valid()) {
      copy = Process_Inter_Module_Goal_Create(proc);
      if (CALL2(hole, Bin_Check_Method, AddMethod, copy)) {
	Set_User_Name(copy, Dropper, "", DroppedTime, 0);
	return true;
      }
      return false;
    }
  }
}

// Hole Methods
// --------------------------------------------------

Am_Define_Method(Copy_Method, Am_Object, New_Hole, (Am_Object /* proc */ ))
{
  Am_Object new_model = HoleModel.Create();
  Am_Object new_proc  = HoleProto.Create()
    .Set(Model, new_model)
    .Set(Am_WIDTH, 60)
    .Set(Am_HEIGHT, 60)
    ;
  return new_proc;
}

Am_Define_Method(Naming_Method, bool, Hole_Rename, (Am_Object hole, Am_String name))
{
  Am_Object pattern = hole.Get(Parent);
  if (pattern.Valid())
    if (Member_Name(pattern, HoleNameList, name)) return false;

  Propagate_Rename(hole, name);
  return true;
}

bool hole_add_port(Am_Object hole, Am_Object port)
{
  Am_String port_name = port.Get_Object(Model).Get(Name); // original name
  Am_String new_name = Next_Name(hole, PortNameList, port_name); // change if the same name exists
  port.Get_Object(Model).Set(Name, new_name); // directoly rename
  
  Propagate_Add(hole, port);
  return true;
}

Am_Define_Method(Bin_Check_Method, bool, Hole_Add, (Am_Object hole, Am_Object obj))
{
  if (obj.Is_Instance_Of(PortProto)) {
    return hole_add_port(hole, obj);
  }
  else if (obj.Is_Instance_Of(ProcessProto) || obj.Is_Instance_Of(PatternProto)) {
    return Hole_Replace(hole, obj);
  }
  Am_Show_Alert_Dialog (Am_Value_List().Add("Can't Add")
			.Add(obj.Get_Name()).Add("to")
			.Add(hole.Get_Name()));
  return false;
}

Am_Define_Method(Bin_Check_Method, bool, Hole_Remove, (Am_Object hole, Am_Object port))
{
  if (port.Get_Object(Parent) != hole) return false;
  Propagate_Remove(port);
  return true;
}

// Formulas
// --------------------------------------------------

Am_Define_Formula(int, CenterYOfOwnerForm)
{
  return (int)self.GV_Owner().GV(Am_HEIGHT) / 2;
}

Am_Define_Value_List_Formula(PortNameListForm) {
  Am_Value_List names = Am_Value_List();
  Am_Value_List ports = ContentsListForm(cc, self);
  for (ports.Start(); !ports.Last(); ports.Next()) {
    Am_Object p = ports.Get();
    names.Add((Am_String)p.GV_Object(Model).GV(Name));
  }
  return names;
}

Am_Define_Value_List_Formula(ProcRuleListForm) {
  // ProcessInterface$B$N$_CM$r;}$D(B
  if ((int)self.GV(Type) != ProcessInterface) return 0;

  Am_String name = self.MGV(Name);
  Am_Object module = self.MGV(Module);
  if (!module.Valid()) return 0;

  Am_Value_List rules = Am_Value_List();

  Am_Value_List rule_list = module.GV(RuleList);
  for (rule_list.Start(); !rule_list.Last(); rule_list.Next()) {
    Am_Object r = rule_list.Get();
    if ((Am_String)r.GV_Object(Model).GV(Name) == name) {
      rules.Add(r);
    }
  }
  return rules;
}

Am_Define_Value_List_Formula(ProcGoalListForm) {
  // ProcessInterface$B$N$_CM$r;}$D(B
  if ((int)self.GV(Type) != ProcessInterface) return 0;

  Am_Object module = self.MGV(Module);
  if (!module.Valid()) return 0;

  Am_String name = self.MGV(Name);
  Am_String modname = self.MGV(ModuleName);

  Am_Value_List goals = Am_Value_List();

  Am_Value_List rule_list = module.GV(RuleList);
  for (rule_list.Start(); !rule_list.Last(); rule_list.Next()) {
    Am_Object r = rule_list.Get();
    Am_Value_List goal_list = r.GV(ProcessList);
    for (goal_list.Start(); !goal_list.Last(); goal_list.Next()) {
      Am_Object g = goal_list.Get();
      if ((int)g.GV(Type) == ProcessGoal &&
	  (Am_String)g.MGV(Name) == name &&
	  (Am_String)g.MGV(ModuleName) == modname) {
	goals.Add(g);
      }
    }
  }
  return goals;
}

Am_Define_Object_Formula(HoleParentForm)
{
  Am_Object parent = ParentForm(cc, self);
  if (parent.Valid()) return parent;

  Am_Object process = self.GV(Process);
  if (process.Valid()) return (Am_Object)process.GV(Parent);
  else return 0;
}

void initProcessProto(void) {

  // Process --------------------------------------------------

  Am_Object name = EditableZoomingText.Create("ProcessName")
    .Get_Part(TextPart).Set(Am_TEXT, ModelNameForm).Get_Owner()
    .Set(FontFamily, FontSansSerif)
    .Set(FontIsBold, true)
    .Set(NameHeightMax, GetResourceForm(NameHeightMax))
    .Set(Am_WIDTH, Am_From_Owner(Am_WIDTH))
    .Set(Am_HEIGHT, NameHeightForm)
    ;

  Am_Object frame = Am_Border_Roundtangle.Create("ProcessFrame")
    .Set(Am_TOP, Bellow_Of_Sibling(NamePart))
    .Set(Am_WIDTH, Am_From_Owner(Am_WIDTH))     
    .Set(Am_HEIGHT, Fill_To_Bottom)
    .Set(Am_RADIUS, 10)
    .Set(Am_THICKNESS, 2)
    .Set(Am_FILL_STYLE, GetResourceForm(ProcessFill))
    ;

  Am_Object contents = PovlContents.Create("ProcessContents")
    .Set(alignThreshold, 6)
    .Set(Am_TOP,    Bellow_Of_Sibling(NamePart))
    .Set(Am_WIDTH,  Am_From_Owner(Am_WIDTH))
    .Set(Am_HEIGHT, Fill_To_Bottom)
    ;

  ProcessModel = Am_Root_Object.Create("ProcessModel")
    .Set(Name,	"process")
    .Set(Module, 0)
    .Set(ModuleName, "")
    .Set(RuleVisible, true)
    ;

  ProcessProto = PovlGroup.Create("ProcessProto")
    // Attributes
    .Set(Model, ProcessModel)
    .Set(Parent, ParentForm)
    .Set(Type,   ProcessInterface)
    .Set(Priority, 0)
    // Lists
    .Set(PortList, ContentsListForm)
    .Set(PortNameList, PortNameListForm)
    .Set(RuleList, ProcRuleListForm)
    .Set(ProcessList, ProcGoalListForm)
    .Set(Hole, 0)
    .Set(LayoutNameList, Am_Value_List())
    .Set(LayoutSizeList, Am_Value_List())
    // Methods
    .Set(NewMethod, New_Process)
    .Set(CopyMethod, Copy_Process)
    .Set(NamingMethod, Process_Rename)
    .Set(AddMethod, Process_Add)
    .Set(RemoveMethod, Process_Remove)
    .Set(CheckDropMethod, Process_Check_Drop)
    .Set(DropMethod, Process_Drop)
    // Graphics
    .Set_Single_Constraint_Mode(Am_WIDTH, false)
    .Set_Single_Constraint_Mode(Am_HEIGHT, false)
    .Set(Am_WIDTH,  50)
    .Set(Am_HEIGHT, 40)

    .Add_Part(NamePart, name)
    .Add_Part(FramePart, frame)
    .Add_Part(ContentsPart, contents)

    .Set(Am_WIDTH, Am_From_Part(ContentsPart, Am_WIDTH))
    .Set(Am_HEIGHT, PovlGroupHeightForm)
    ;

  // Hole --------------------------------------------------

  name = EditableZoomingText.Create("HoleName")
    .Get_Part(TextPart).Set(Am_TEXT, ModelNameForm).Get_Owner()
    .Set(FontFamily, FontSansSerif)
    .Set(FontIsBold, true)
    .Set(NameHeightMax, GetResourceForm(NameHeightMax))
    .Set(Am_WIDTH, Am_From_Owner(Am_WIDTH))
    .Set(Am_HEIGHT, NameHeightForm)
    ;

  frame = Am_Border_Rectangle.Create("HoleFrame")
    .Set(Am_TOP, Bellow_Of_Sibling(NamePart))
    .Set(Am_WIDTH, Am_From_Owner(Am_WIDTH))     
    .Set(Am_HEIGHT, Fill_To_Bottom)
    .Set(Am_RAISED, false)
    .Set(Am_THICKNESS, 2)
    .Set(Am_FILL_STYLE, GetResourceForm(ProcessFill))
    ;

  contents = PovlContents.Create("HoleContents")
    .Set(Am_TOP,    Bellow_Of_Sibling(NamePart))
    .Set(Am_WIDTH,  Am_From_Owner(Am_WIDTH))
    .Set(Am_HEIGHT, Fill_To_Bottom)
    ;

  HoleModel = Am_Root_Object.Create("HoleModel")
    .Set(Name, "hole")
    ;

  HoleProto = PovlGroup.Create("HoleProto")
    // Attributes
    .Set(Model, HoleModel)
    .Set(Parent, HoleParentForm)
    // Lists
    .Set(PortList, ContentsListForm)
    .Set(PortNameList, PortNameListForm)
    .Set(RuleList, Am_Value_List())
    .Set(Process, 0)
    .Set(ProcessList, Am_Value_List())
    .Set(LayoutNameList, Am_Value_List())
    .Set(LayoutSizeList, Am_Value_List())
    // Methods
    .Set(NewMethod, New_Hole)
    .Set(NamingMethod, Hole_Rename)
    .Set(AddMethod, Hole_Add)
    .Set(RemoveMethod, Hole_Remove)
    .Set(CheckDropMethod, 0)
    .Set(DropMethod, 0)
    // Graphics
    .Set(Am_WIDTH, 50)
    .Set(Am_HEIGHT, 40)
    .Set(Am_VISIBLE, ContentsVisibleForm)
    .Add_Part(NamePart, name)
    .Add_Part(FramePart, frame)
    .Add_Part(ContentsPart, contents)
    ;

  ReplHolesProto = SequencePovlGroup.Create("ReplHolesProto")
    // Attributes
    .Set(Model, HoleModel.Create())
    .Set(Parent, ParentForm)
    .Set(Dir, LeftRight)
    .Set(seqDir, Am_Same_As(Dir))
    // Methods
    .Set(NewMethod, 0)
    .Set(NamingMethod, 0)
    .Set(AddMethod, 0)
    .Set(RemoveMethod, 0)
    .Set(CheckDropMethod, 0)
    .Set(DropMethod, 0)
    // Graphics
    .Set(Am_WIDTH, 220)
    .Set(Am_HEIGHT, 110)
    .Set(ContentsPart, SelfForm)
    ;

  // Implementations Dialog --------------------------------------------------
  // ImplSelectionMenu.Am_ITEMS: $B8uJd%W%m%;%9$N%j%9%H!#(B
  // ImplDialog.Am_ITEMS: $B%[!<%k$NL>A0!#(B

  ImplSelectionCmd = Am_Command.Create()
    .Set(Am_DO_METHOD, Impl_Menu_Do)
    ;

  ImplSelectionMenu = Am_Menu.Create()
    .Set(Am_FONT, GetResourceForm(ImplDialogFont))
    .Set(Am_WIDTH, 200)
    .Set(Am_HEIGHT, 600)
    .Set(Am_ITEMS, Am_Value_List())
    ;

  Am_Object scrolling_selection = Am_Scrolling_Group.Create()
    .Set(Am_WIDTH, 200)
    .Set(Am_HEIGHT, 150)
    .Set(Am_H_SCROLL_BAR, false)
    .Add_Part(Am_ITEMS, ImplSelectionMenu)
    ;

  ImplDialog = Am_Choice_Dialog.Create()
    .Set(Am_OMIT_TITLE_BAR, false)
    .Set(Am_VISIBLE, false)
    .Set(Am_TITLE, "Implementations")
    .Set(Am_ITEMS, Am_Value_List())
    .Get_Part(Am_DIALOG_GROUP)
      .Add_Part(Am_ITEMS, scrolling_selection)
      .Get_Part(Am_DIALOG_BUTTONS)
        .Set(Am_H_SPACING, 5)
      .Get_Owner()
    .Get_Owner()
    ;

  Am_To_Top(ImplDialog.Get_Part(Am_DIALOG_GROUP).Get_Part(Am_DIALOG_BUTTONS));
  Am_Value_List parts = ImplDialog.Get_Part(Am_DIALOG_GROUP).Get(Am_GRAPHICAL_PARTS);
  parts.Start();
  Am_Object text = parts.Get();
  text.Get_Object(Am_ITEM_PROTOTYPE).Set(Am_FONT, Am_Font(Am_FONT_SANS_SERIF, false, false, false, Am_FONT_LARGE));
}

// Retrieve Hyper Link
// ----------------------------------------------------------------------
void retrieve_definition_link(Am_Object proc)
{
  Am_Object des = Process_Get_Interface(proc);
  if (!des.Valid()) return;

  Am_Object src = proc.Get(Parent);
  Retrieve_Hyper_Link(src, des, PovlSelectionWidget);
  return;
}

Am_Define_Method(Am_Object_Method, void, HyperLinkDo, (Am_Object command_obj))
{
  Am_Object proc = command_obj.Get_Owner().Get_Owner();
  retrieve_definition_link(proc);
}

void initHyperLink(void)
{
  Am_Object link_interactor = Am_One_Shot_Interactor.Create("link_interactor")
    .Set(Am_START_WHEN, "double_left_down")
    .Set(Am_PRIORITY, 2)
    .Get_Part(Am_COMMAND)
    .Set(Am_DO_METHOD, HyperLinkDo)
    .Get_Owner()
    ;

  ProcessProto.Add_Part(Am_INTERACTOR, link_interactor);
}



// ------------------------------------------------------------
// Initialize
// ------------------------------------------------------------

void InitializeProcess(void)
{
  initProcessProto();
  initHyperLink();
}
