// Copyright (C) 1997  Toyoda Masashi (toyoda@is.titech.ac.jp)

#include <fstream.h>
#include <strstream.h>

#include <amulet.h>
#include FORMULA_ADVANCED__H
#include INTER_ADVANCED__H
#include WIDGETS_ADVANCED__H

#include "ToyWidgets.h"
#include "PovlWidgets.h"
#include "Resources.h"
#include "Module.h"
#include "Process.h"
#include "Rule.h"
#include "Pattern.h"
#include "Atom.h"
#include "Message.h"
#include "Guard.h"
#include "Port.h"
#include "Binder.h"
#include "Item.h"

// ---- Tags ----
// a b c d e  g h i  l m n o p q r s t v x y z

#define Beg_Module 'M'
#define End_Module 'm'

#define Beg_Item   'J'

#define Beg_Procs  'P'
#define End_Procs  'p'
#define Beg_Proc   'C'
#define End_Proc   'c'
#define Beg_Patts  'X'
#define End_Patts  'x'
#define Beg_Rules  'R'
#define End_Rules  'r'
#define Beg_Rule   'L'
#define End_Rule   'l'
#define Beg_Patt   'E'
#define End_Patt   'e'
#define Beg_Goals  'O'
#define End_Goals  'o'
#define Beg_Goal   'N'
#define End_Goal   'n'
#define Beg_Holes  'H'
#define End_Holes  'h'
#define Beg_Hole   'I'
#define End_Hole   'i'
#define Beg_Gurds  'G'
#define End_Gurds  'g'
#define Beg_Gurd   'F'
#define End_Gurd   'f'
#define Beg_Bndrs  'B'
#define End_Bndrs  'b'
#define Beg_Bndr   'D'
#define End_Bndr   'd'

#define Beg_Ports  'Q'
#define End_Ports  'q'
#define Beg_Port   'T'
#define End_Port   't'
#define Beg_Vals   'V'
#define End_Vals   'v'
#define Beg_Stru   'S'
#define End_Stru   's'
#define Beg_Atom   'A'
#define End_Atom   'a'

#define Beg_LName  '#'
#define End_LName  '!'
#define Beg_SList  'Z'
#define End_SList  'z'

// ---- Global ----
#define Old_Ver_Num    '3'
#define Ver_Num	       '0'

#define Ver_2_String   "KLIEG Ver. 2"
#define Ver_3_String   "KLIEG Ver. 3"
#define Ver_3_2_String "KLIEG Ver. 3.2"
#define Ver_3_3_String "KLIEG Ver. 3.3"

Am_Object Process_table[1024];
char Version[20];

// ---- Check Version Number ----
void Check_version_number(ifstream& ifile)
{
  strcpy(Version, Ver_2_String); // $B2?$b$J$$$H$-$O(BVer. 2

  char ver;
  ifile >> ver; if (ver != Ver_Num && ver != Old_Ver_Num) {
    ifile.putback(ver); return;
  }
  if (ver == Old_Ver_Num)
    strcpy(Version, Ver_3_String); // Old_Ver_Num$B$O(BVer. 3
  else if (ver == Ver_Num) {
    ifile >> ws; // $B6uGr$H$P$7!#(B
    ifile.getline(Version, 20); // Ver_Num$B$N$H$-$O%U%!%$%k$+$iFI$`!#(B
  }
  cerr << "Version: " << Version << endl;
}

// ---- Utility ----
Am_Object New_obj(Am_Object proto)
{
  Copy_Method new_method = proto.Get(NewMethod);
  if (new_method.Valid()) 
    return new_method.Call(proto);
  else
    return 0;
}

void Add_obj(Am_Object dest, Am_Object src)
{
  Bin_Check_Method add_method = dest.Get(AddMethod);
  if (add_method.Valid())
    add_method.Call(dest, src);
}

int Save_position(Am_Object obj, ofstream& ofile)
{
  ofile << " ";
  OutputCoords(obj, ofile);
  ofile << " ";
  /*
  ofile << " " << origX(obj) << " " << origY(obj) << " "
	<< origW(obj) << " " << origH(obj) << " ";
  */
  return 1;
}

int Load_position(Am_Object obj, ifstream& ifile)
{
  int left, top, height, width;
  if (strcmp(Version, Ver_2_String) == 0) { // Ver. 2
    ifile >> left >> top >> width >> height;
    obj.Get(Am_LEFT); // demon$B$r8F$S=P$9$?$a!)!)!)(B
    obj.Set(Am_LEFT, left)
      .Set(Am_TOP, top)
      .Set(Am_WIDTH, width)
      .Set(Am_HEIGHT, height)
      ;
  } else { // Ver. 3$B0J9_(B
    int id, pid, x, y, cw, ch, sw, sh, lw, lh;
    ifile >> x >> y >> cw >> ch >> sw >> sh >> lw >> lh;
    obj.Get(Am_LEFT); // demon$B$r8F$S=P$9$?$a!)!)!)(B
    obj.Set(Am_LEFT, x)
      .Set(Am_TOP, y)
      .Set(Am_WIDTH, cw)
      .Set(Am_HEIGHT, ch)
      ;
    SetPointers(obj, cw, ch, sw, sh, lw, lh);
  }
  return 1;
}

void Set_zooming(Am_Object obj, bool active)
{
  Am_Do_Events();
  obj.Get_Part(ContentsPart).Set(Am_ACTIVE, active);
}

// -------- Error --------
void Err(char *string, char* name)
{
  Am_Show_Alert_Dialog(Am_Value_List().Add("Error:").Add(string).Add(name));
  //  cerr << "Error: " << string << endl;
}

// -----------------------------------
int Save_module_to_file(Am_Object module, ofstream& ofile);
int Save_module(Am_Object Window, ofstream& ofile);
int Load_module(Am_String file_name);
int   Save_proc_part(Am_Value_List procs, ofstream& ofile);
int   Load_proc_part(Am_Object module, ifstream& ifile);
int     Save_process(Am_Object process, ofstream& ofile);
int     Load_process(Am_Object proc_agg, ifstream& ifile);
int   Save_patterns_part(Am_Value_List patterns, ofstream& ofile);
int   Load_patterns_part(Am_Object module, ifstream& ifile);
int   Save_rules_part(Am_Value_List rules, ofstream& ofile);
int   Load_rules_part(Am_Object module, ifstream& ifile);
int     Save_rule(Am_Object rule, ofstream& ofile);
int     Load_rule(Am_Object module, ifstream& ifile);
int     Load_really_rule(Am_Object module, ifstream& ifile);
int       Save_goals_part(Am_Object rule, ofstream& ofile);
int       Load_goals_part(Am_Object rule, ifstream& ifile);
int         Save_goal(Am_Object goal, ofstream& ofile);
int         Load_goal(Am_Object rule, ifstream& ifile);
int         Load_really_goal(Am_Object rule, ifstream& ifile);
int         Save_binder_part(Am_Object rule, ofstream& ofile);
int       Load_binder_part(Am_Object rule, ifstream& ifile);
int       Save_binder(Am_Object rule, Am_Object binder, ofstream& ofile);
int       Load_binder(Am_Object rule, ifstream& ifile);
int       Save_guard_part(Am_Object rule, ofstream& ofile);
int       Load_guard_part(Am_Object rule, ifstream& ifile);
int       Save_guard(Am_Object guard, ofstream& ofile);
int       Load_guard(Am_Object rule, ifstream& ifile);
int     Save_pattern(Am_Object pattern, ofstream& ofile);
int     Load_pattern(Am_Object patt_agg, ifstream& ifile);
int       Save_holes_part(Am_Value_List holes, ofstream& ofile);
int       Load_holes_part(Am_Object patt, ifstream& ifile);
int       Save_hole(Am_Object hole, ofstream& ofile);
int       Load_hole(Am_Object patt, int i, ifstream& ifile);

int Save_port_part(Am_Value_List ports, ofstream& ofile);
int Load_port_part(Am_Object proc_agg, ifstream& ifile);
int Load_exist_port_part(Am_Object process, ifstream& ifile);
int Save_port(Am_Object port, ofstream& ofile);
int Load_port(Am_Object port, ifstream& ifile);
int Load_exist_port(Am_Object port, ifstream& ifile);

int Save_value_part(Am_Value_List values, ofstream& ofile);
int Load_value_part(Am_Object port, ifstream& ifile);
int Save_value(Am_Object value, ofstream& ofile);
int Load_value(Am_Object port, ifstream& ifile);

int Save_atom(Am_Object value, ofstream& ofile);
int Load_atom(Am_Object port, ifstream& ifile);
int Save_structure(Am_Object value, ofstream& ofile);
int Load_structure(Am_Object port, ifstream& ifile);

// -------- Module --------
// Beg_Module Name
// ... processes ...
// ... rules ...
// End_Module

int Save_module_to_file(Am_Object module, ofstream& ofile)
{
  // Version Number
  //  ofile << Ver_Num << endl; // Ver. 3
  //  ofile << Ver_Num << " " << Ver_3_2_String << endl; // Ver. 3.2
  ofile << Ver_Num << " " << Ver_3_3_String << endl; // Ver. 3.3

  // Save module
  //  ofile << Beg_Module << " " << module.MGet(Name) << endl; // Ver. 3
  ofile << Beg_Module << " " << module.MGet(Name) << " "; // Ver. 3.2
  Save_position(module, ofile); // Ver. 3.2
  ofile << endl; // Ver. 3.2
  
  Save_proc_part(module.Get(ProcessList), ofile); 
  Save_patterns_part(module.Get(PatternList), ofile);
  Save_rules_part(module.Get(RuleList), ofile); 
  ofile << End_Module << endl;
  return 1;
}

int Save_module(Am_Object module)
{
  char name[1000];
  ostrstream oname(name, 1000);

  //  oname << module.MGet(Name) << ".klg" << ends;
  oname << module.MGet(Name) << ends;
  ofstream ofile;
  ofile.open(name, ios::out);
  if (!ofile) {
    cerr << "can't save to file: " << name << endl;
    return 0;
  }
  Save_module_to_file(module, ofile);
  return 1;
}

Am_Object load_single_module_from_file(Am_Object top_module, ifstream& ifile)
{
  char tag;
  char name[1000];

  Check_version_number(ifile);

  ifile >> tag; if (tag != Beg_Module) {/* Err("Beg_Module", "");*/ ifile.putback(tag); return 0;}
  ifile >> name;

  Am_Object module = ((Copy_Method)ModuleProto.Get(NewMethod)).Call(ModuleProto);
  module.MSet(Name, "");
  SetPointers(module, 600, 550, 200, 40, 600, 550);
  // Ver 4 $B$+$i$O(Bposition$B$b%;!<%V$7$F$$$k!#(B
  //  cout << Version << " " << Ver_3_2_String << " " << strcmp(Version, Ver_3_2_String) << endl;
  if (strcmp(Version, Ver_3_2_String) >= 0) {
    cout << "load position" << endl;
    Load_position(module, ifile);
  }
  ((Bin_Check_Method)top_module.Get(AddMethod)).Call(top_module, module);

  if (!CALL2(module, Naming_Method, NamingMethod, name)) {
    module.Remove_From_Owner();
    Err(name, "Already exists.");
    // $B$G$b%3%T!<$H$+$5$l$F$k$HBgJQ$J$s$@$h$J!#%3%T!<$H$O$J$s$@!)(B
    return 0;
  }
  Set_zooming(module, false);
  Load_proc_part (module, ifile);
  Load_patterns_part(module, ifile);
  Load_rules_part(module, ifile);
  Set_zooming(module, true);
  ifile >> tag; if (tag != End_Module) {Err("End_Module", name); return 0;}
  return module;
}

Am_Object load_single_module(Am_Object top_module, Am_String file_name)
{
  if (!file_name.Valid()) return 0;

  ifstream ifile;
  ifile.open(file_name, ios::in);
  if (!ifile) {
    cerr << "can't open " << file_name << endl;
    return 0;
  }
  return load_single_module_from_file(top_module, ifile);
}

void check_depend_pattern_iter(Am_Object pattern, Am_Value_List& depend);

void check_depend_pattern_hole(Am_Object hole, Am_Value_List& depend)
{
  Am_Value_List procs = hole.Get(ProcessList);
  for (procs.Start(); !procs.Last(); procs.Next()) {
    Am_Object p = procs.Get();
    if (p.Is_Instance_Of(PatternProto)) {
      check_depend_pattern_iter(p, depend);
    } else if ((int)p.Get(Type) == ProcessInterGoal) {
      depend.Add((Am_String)p.MGet(ModuleName));
    }
  }
}

void check_depend_pattern_iter(Am_Object pattern, Am_Value_List& depend)
{
  Am_Value_List holes = pattern.Get(HoleList);
  if (pattern.Is_Instance_Of(ReplPatternProto)) {
    holes.Start();
    Am_Object h = holes.Get();
    check_depend_pattern_hole(h, depend);
  } else {
    for (holes.Start(); !holes.Last(); holes.Next()) {
      Am_Object h = holes.Get();
      check_depend_pattern_hole(h, depend);
    }
  }
}

void check_depend(Am_Object module, Am_Value_List& depend)
{
  // $B%k!<%kFb$N%4!<%k$N0MB84X78$r%A%'%C%/(B
  Am_Value_List rules = module.Get(RuleList);
  for (rules.Start(); !rules.Last(); rules.Next()) {
    Am_Value_List goals = ((Am_Object)rules.Get()).Get(ProcessList);
    for (goals.Start(); !goals.Last(); goals.Next()) {
      Am_Object g = goals.Get();
      if (g.Is_Instance_Of(PatternProto)) {
	check_depend_pattern_iter(g, depend);
      } else if ((int)g.Get(Type) == ProcessInterGoal) {
	depend.Add((Am_String)g.MGet(ModuleName));
      }
    }
  }
  Am_Value_List patterns = module.Get(PatternList);
  for (patterns.Start(); !patterns.Last(); patterns.Next()) {
    Am_Object pat = patterns.Get();
    check_depend_pattern_iter(pat, depend);
  }
}

int Load_module(Am_Object top_module, Am_String file_name)
{
  Set_zooming(top_module, false);

  Am_Value_List depend = Am_Value_List().Add(file_name);  depend.Start();
  while (depend.Start(), !depend.Empty()) {
    Am_String name = depend.Get();  depend.Delete();
    if (!Member_Name(top_module, ModuleNameList, name)) {
      Am_Object module = load_single_module(top_module, name);
      if (module.Valid()) check_depend(module, depend);
      cerr << depend << endl;
    }
  }

  Set_zooming(top_module, true);
  return 1;
}

// -------- Snapshot --------
// Ver_Num Ver_3_2_String
// Beg_Module Name Position
// ...
// End_Module
// ......
// Ver_Num Ver_3_2_String
// Beg_Module Name Position
// ...
// End_Module
// Beg_Item
// ....
// ....

int Save_Snapshot(Am_Object top_module, Am_String file_name)
{
  if (!file_name.Valid()) return 0;

  ofstream ofile;
  ofile.open(file_name, ios::out);
  if (!ofile) {
    cerr << "can't save Snapshot: " << file_name << endl;
    return 0;
  }

  int id = 500;

  Am_Value_List modules = top_module.Get(ModuleList);
  for (modules.Start(); !modules.Last(); modules.Next()) {
    Am_Object mod = modules.Get();
    Save_module_to_file(mod, ofile);
    mod.Set(MId, id);
    ++id;
  }

  Item_Save_to_file(top_module, ofile);
}

int Load_Snapshot(Am_Object top_module, Am_String file_name)
{
  if (!file_name.Valid()) return 0;
  Set_zooming(top_module, false);

  ifstream ifile;
  ifile.open(file_name, ios::in);
  if (!ifile) {
    cerr << "can't open " << file_name << endl;
    return 0;
  }
  int id = 500;
  while(Am_Object mod = load_single_module_from_file(top_module, ifile)) {
    items[id] = mod;
    ++id;
  }
  Item_Load_from_file(top_module, ifile);

  Set_zooming(top_module, true);
}


// -------- Processes --------
// Beg_Procs
// ... process ...
// End_Procs
int Save_proc_part(Am_Value_List procs, ofstream& ofile)
{
  ofile << Beg_Procs << endl;
  int id = 1;
  for (procs.Start(); !procs.Last(); procs.Next(), ++id) {
    Am_Object proc = procs.Get();
    Save_process(proc.Set(Id, id), ofile);
  }
  ofile << End_Procs << endl;
}

int Load_proc_part(Am_Object module, ifstream& ifile)
{
  Am_String mname = module.MGet(Name);
  char tag;
  ifile >> tag; if (tag != Beg_Procs) {Err("Beg_Procs", mname); return 0;}
  while (Load_process(module, ifile));
  ifile >> tag; if (tag != End_Procs) {Err("End_Procs", mname); return 0;}
}

// -------- Process --------
// Beg_Proc Id Name position_info
// ... ports ... 
// End_Proc
int Save_process(Am_Object process, ofstream& ofile)
{
  int id         = process.Get(Id);
  Am_String name = process.MGet(Name);

  ofile << Beg_Proc << " ";
  ofile << id << " " << name;
  Save_position(process, ofile);
  ofile << endl;
  Save_port_part(process.Get(PortList), ofile);
  ofile << End_Proc << endl;
}

int Load_process(Am_Object module, ifstream& ifile)
{
  char tag;
  int id;
  char name[1000];

  ifile >> tag; if (tag != Beg_Proc) {ifile.putback(tag); return 0;}
  ifile >> id >> name;
  { cerr << "Load process: " << name << " id: " << id << endl; }

  Am_Object new_proc = New_obj(ProcessProto);
  new_proc.MSet(Name, name);

  Load_position(new_proc, ifile);

  Set_zooming(new_proc, false);
  Load_port_part(new_proc, ifile);
  Set_zooming(new_proc, true);

  Process_table[id] = new_proc; // global table $B$KEPO?!%(B
  Add_obj(module, new_proc);

  ifile >> tag; if (tag != End_Proc) {Err("End_Proc", name); return 0;}
  return 1;
}

// -------- Patterns Part --------
// Beg_Patts
// ... patterns ...
// End_Patts
int Save_patterns_part(Am_Value_List patterns, ofstream& ofile)
{
  ofile << Beg_Patts << endl;
  for (patterns.Start(); !patterns.Last(); patterns.Next()) {
    Am_Object patt = patterns.Get();
    Save_pattern(patt, ofile);
  }
  ofile << End_Patts << endl;

}

int Load_patterns_part(Am_Object module, ifstream& ifile)
{
  Am_String mname = module.MGet(Name);
  char tag;
  ifile >> tag; if (tag != Beg_Patts) {Err("Beg_Patts", mname); return 0;}
  while (Load_pattern(module, ifile));
  ifile >> tag; if (tag != End_Patts) {Err("End_Patts", mname); return 0;}
}

// -------- Rules Part --------
// Beg_Rules
// ... rules ...
// End_Rules
int Save_rules_part(Am_Value_List rules, ofstream& ofile)
{
  ofile << Beg_Rules << endl;
  for (rules.Start(); !rules.Last(); rules.Next()) {
    Am_Object rule = rules.Get();
    Save_rule(rule, ofile);
  }
  ofile << End_Rules << endl;
}

int Load_rules_part(Am_Object module, ifstream& ifile)
{
  Am_String mname = module.MGet(Name);
  char tag;
  ifile >> tag; if (tag != Beg_Rules) {Err("Beg_Rules", mname); return 0;}
  while (Load_rule(module, ifile));
  ifile >> tag; if (tag != End_Rules) {Err("End_Rules", mname); return 0;}
}


// -------- Rule --------
// Beg_Rule Proc Type position_info
// ... ports ...
// ... goals ...
// ... binders...
// End_Rule

int Save_rule(Am_Object rule, ofstream& ofile)
{
  ofile << Beg_Rule << " ";
  int proc = 0;
  if (!rule.Is_Instance_Of(MainRuleProto)) proc = rule.Get_Object(Process).Get(Id);
  int type = rule.Get(Type);
  ofile << proc << " " << type;
  Save_position(rule, ofile);
  ofile << endl;
  Save_port_part(rule.Get(PortList), ofile);
  Save_goals_part(rule, ofile);
  Save_binder_part(rule, ofile);
  Save_guard_part(rule, ofile);
  ofile << End_Rule << endl;
}

int Load_rule(Am_Object module, ifstream& ifile)
{
  char tag;
  ifile >> tag; if (tag != Beg_Rule) {ifile.putback(tag); return 0;}
  int proc, type;
  ifile >> proc >> type;
  cerr << "Load rule, Proc id: " << proc << " type: " << type << endl;
  Am_Object rule;
  if   (type == TransitionRule) rule = TransitionRuleProto;
  else if (type == NetworkRule) rule = NetworkRuleProto;
  else return 0;

  Am_Object p = Process_table[proc];
  if (proc != 0) {
    rule = Rule_Create(p, rule);
  } else {
    rule = module.Get_Part(ContentsPart).Get_Part(MainRule);
    rule.Remove_From_Owner();
  }

  Load_position(rule, ifile);

  Set_zooming(rule, false);  
  Load_exist_port_part(rule, ifile);
  Load_goals_part(rule, ifile);
  Load_binder_part(rule, ifile);
  Load_guard_part(rule, ifile);
  Set_zooming(rule, true);  

  if (proc != 0) Add_obj(module, rule);
  else module.Get_Part(ContentsPart).Add_Part(MainRule, rule);

  ifile >> tag; if (tag != End_Rule) {Err("End_Rule", ""); return 0;}
  return 1;
}

// -------- Goals Part --------
// Beg_Goals
// ... goals ...
// End_Goals

int Save_goals_part(Am_Object rule, ofstream& ofile)
{
  ofile << Beg_Goals << endl;
  Am_Value_List goals = rule.Get(ProcessList);
  for (goals.Start(); !goals.Last(); goals.Next()) {
    Am_Object goal = goals.Get();
    if (goal.Is_Instance_Of(ProcessProto)) Save_goal(goal, ofile);
    if (goal.Is_Instance_Of(PatternProto)) Save_pattern(goal, ofile);
  }
  ofile << End_Goals << endl;
}

int Load_goals_part(Am_Object rule, ifstream& ifile)
{
  Am_String rname = rule.MGet(Name);
  char tag;
  ifile >> tag; if (tag != Beg_Goals) {Err("Beg_Goals", rname); return 0;}
  while (Load_goal(rule, ifile));
  ifile >> tag; if (tag != End_Goals) {Err("End_Goals", rname); return 0;}
}

// -------- Goal --------
// Beg_Goal Id Hole Name ModuleName position_info
// ... ports ...
// End_Goal
int Save_goal(Am_Object goal, ofstream& ofile)
{
  ofile << Beg_Goal << " ";
  int type = goal.Get(Type);
  int id = 0; // InterGoal$B$N$H$-(B0
  Am_String module_name;
  if (type == ProcessGoal) {
    Am_Object proc = goal.Get_Prototype();
    id = proc.Get(Id);
    module_name = "module";
  } else {
    id = 0;
    module_name = goal.MGet(ModuleName);
  }
  int hole = 0; // Pattern$B$N$H$-$NBP1~!%(B
  Am_Object hl = goal.Get(Hole);
  if (hl.Valid()) {
    //    if(goal.Get(Parent).Valid())
    //      hole = Object_List_Index(goal.Get_Object(Parent).Get(HoleList), hl);
    //    else
      hole = Object_List_Index(hl.Get_Object(Parent).Get(HoleList), hl);
  }
  ofile << id << " " << hole << " " << goal.MGet(Name) << " " << module_name;
  Save_position(goal, ofile);
  ofile << endl;
  Save_port_part(goal.Get(PortList), ofile);
  ofile << End_Goal << endl;
}

int Load_goal(Am_Object rule, ifstream& ifile)
{
  char tag;
  ifile >> tag;
  if (tag == Beg_Goal) return Load_really_goal(rule, ifile);
  if (tag == Beg_Patt) {ifile.putback(tag); return Load_pattern(rule, ifile);}
  ifile.putback(tag);
  return 0;
}

int Load_really_goal(Am_Object rule, ifstream& ifile)
{
  char tag;
  // tag check omitted by Load_goal()
  int id, hole;
  char name[1000];
  char mname[1000];
  ifile >> id >> hole >> name >> mname;

  Am_Object goal = 0;
  if (id == 0) goal = Process_Inter_Module_Goal_New(name, mname);
  else         goal = Process_Goal_Create(Process_table[id]);
  Load_position(goal, ifile);

  if (id == 0) {
    rule.Get_Part(ContentsPart).Add_Part(goal); // fake
    Set_zooming(goal, false);
    Load_port_part(goal, ifile);
    Set_zooming(goal, true);
    goal.Remove_From_Owner();
  }

  if (hole != 0) {
    Am_Object h = Object_List_Get(rule.Get(HoleList), hole);
    Single_Hole_Replace(h, goal);
  } else {
    Add_obj(rule, goal);
  }

  if (id != 0) {
    Set_zooming(goal, false);
    Load_exist_port_part(goal, ifile);
    Set_zooming(goal, true);
  }

  ifile >> tag; if (tag != End_Goal) {Err("End_Goal", name); return 0;}
  return 1;
}

// -------- Binder Part --------
// Beg_Bndrs
// ... binders ...
// End_Bndrs
int Save_binder_part(Am_Object rule, ofstream& ofile)
{
  ofile << Beg_Bndrs << endl;
  Am_Value_List binders = rule.Get_Object(BindersPart).Get(Am_GRAPHICAL_PARTS);
  for (binders.Start(); !binders.Last(); binders.Next()) {
    Am_Object binder = binders.Get();
    Save_binder(rule, binder, ofile);
  }
  ofile << End_Bndrs << endl;
}

int Load_binder_part(Am_Object rule, ifstream& ifile)
{
  char tag;
  ifile >> tag; if (tag != Beg_Bndrs) return 0;
  while (Load_binder(rule, ifile));
  ifile >> tag; if (tag != End_Bndrs) return 0;
}


// -------- Binder --------
// Beg_Bndr
// Pos1.1 Pos1.2 ... -1
// Pos2.1 Pos2.2 ... -1
// End_Bndr

int Save_binder(Am_Object rule, Am_Object binder, ofstream& ofile)
{
  Am_Object pair[2];
  pair[0] = binder.Get(SrcPort);
  pair[1] = binder.Get(DestPort);
  if (!Obj_To_Path(pair[0], rule).Valid()) return 1;
  if (!Obj_To_Path(pair[1], rule).Valid()) return 1; 
  
  ofile << Beg_Bndr << endl;
  int i;
  for (i = 0; i < 2; ++i) {
    Am_Value_List path = Obj_To_Path(pair[i], rule);
    for (path.Start(); !path.Last(); path.Next()) {
      int pos = path.Get();
      ofile << pos << " ";
    }
    ofile << -1 << endl;
  }
  ofile << End_Bndr << endl;
}

int Load_binder(Am_Object rule, ifstream& ifile)
{
  char tag;
  ifile >> tag; if (tag != Beg_Bndr) {ifile.putback(tag); return 0;}
  int pos;
  Am_Value_List paths[2] = {Am_Value_List(), Am_Value_List()};
  int i;
  for (i = 0; i < 2; ++i) {
    while (ifile >> pos, pos >= 0) {
      paths[i].Add(pos);
    }
  }
  Am_Object src_port = Path_To_Obj(rule, paths[0]);
  Am_Object des_port = Path_To_Obj(rule, paths[1]);
  Am_Object new_line = Single_Binding_Line_Create(src_port, des_port);
  //  cout << "create binder: " << src_port << des_port << new_line << endl;
  ifile >> tag; if (tag != End_Bndr) return 0;
}

  
// -------- Guard Part --------
// Beg_Gurds
// ... gurads ...
// End_Gurds

int Save_guard_part(Am_Object rule, ofstream& ofile)
{
  ofile << Beg_Gurds << endl;
  Am_Value_List guards = rule.Get(GuardList);
  for (guards.Start(); !guards.Last(); guards.Next()) {
    Am_Object guard = guards.Get();
    Save_guard(guard, ofile);
  }
  ofile << End_Gurds << endl;
}

int Load_guard_part(Am_Object rule, ifstream& ifile)
{
  char tag;
  ifile >> tag; if (tag != Beg_Gurds) return 0;
  while (Load_guard(rule, ifile));
  ifile >> tag; if (tag != End_Gurds) return 0;
}

// -------- Guard --------
// Beg_Gurd position_info Contents
// End_Gurd

int Save_guard(Am_Object guard, ofstream& ofile)
{
  ofile << Beg_Gurd << " ";
  Save_position(guard, ofile);
  ofile << guard.MGet(Name) << endl;
  ofile << End_Gurd << endl;
}

int Load_guard(Am_Object rule, ifstream& ifile)
{
  char tag;
  ifile >> tag; if (tag != Beg_Gurd) {ifile.putback(tag); return 0;}
  Am_Object new_guard = New_obj(GuardProto);
  Load_position(new_guard, ifile);
  char Contents[1000];
  ifile >> ws;
  ifile.getline(Contents, 1000);
  new_guard.MSet(Name, Contents);
  Add_obj(rule, new_guard);
  ifile >> tag; if (tag != End_Gurd) return 0;
  return 1;
}

// -------- Layout Names --------
// Beg_LName
// name1
// name2
// ...
// End_LName

int Save_layout_names(Am_Object pattern, ofstream& ofile)
{
  ofile << Beg_LName << endl;
  Am_Value_List names = pattern.Get(LayoutNameList);
  for (names.Start(); !names.Last(); names.Next()) {
    Am_String n = names.Get();
    ofile << n << endl;
  }
  ofile << End_LName << endl;
  return 1;
}

int Load_layout_names(Am_Object pattern, ifstream& ifile)
{
  char tag;
  ifile >> tag; if (tag != Beg_LName) {ifile.putback(tag); return 1;}
  Am_Value_List names = Am_Value_List();
  ifile >> tag; 
  while (tag != End_LName) {
    ifile.putback(tag);
    char name[1000];
    ifile.getline(name, 1000);
    names.Add(name);
    ifile >> tag;
  }
  pattern.Set(LayoutNameList, names);
  return 1;
}

// -------- Size List --------
// Beg_SList
// w1 h1
// w2 h2
// ...
// End_SList

int Save_size_list(Am_Object pattern, ofstream& ofile)
{
  ofile << Beg_SList << endl;
  Am_Value_List sizes = pattern.Get(LayoutSizeList);
  for (sizes.Start(); !sizes.Last(); sizes.Next()) {
    Am_Inter_Location loc = sizes.Get();
    bool as_line;
    Am_Object ref;
    int x, y, w, h;
    loc.Get_Location(as_line, ref, x, y, w, h);
    ofile << w << " " << h << endl;
  }
  ofile << End_SList << endl;
  return 1;
}

int Load_size_list(Am_Object pattern, ifstream& ifile)
{
  char tag;
  ifile >> tag; if (tag != Beg_SList) {ifile.putback(tag); return 1;}
  Am_Value_List sizes = Am_Value_List();
  ifile >> tag; 
  while (tag != End_SList) {
    ifile.putback(tag);
    int w, h;
    ifile >> w >> h;
    Am_Inter_Location loc = Am_Inter_Location(false, pattern, 0, 0, w, h);
    sizes.Add(loc);
    ifile >> tag;
  }
  pattern.Set(LayoutSizeList, sizes);
  return 1;
}

// -------- Pattern --------
// Beg_Patt Type Name Dir Hole position_with_icon_info
// ... ports ...
// ... holes ...
// ... goals ...
// ... binders ...
// ... layout names ...(optional)
// ... size list ...(optional)
// End_Patt

int Save_pattern(Am_Object pattern, ofstream& ofile)
{
  ofile << Beg_Patt << " ";
  Am_String name = pattern.MGet(Name);
  int type = pattern.MGet(Type);
  int dir  = pattern.Get(Dir);

  int hole = 0; // Pattern$B$N$H$-$NBP1~!%(B
  Am_Object hl = pattern.Get(Hole);
  if (hl.Valid()) {
    //    if (pattern.Get(Parent).Valid())
    //      hole = Object_List_Index(pattern.Get_Object(Parent).Get(HoleList), hl);
    //    else
      hole = Object_List_Index(hl.Get_Object(Parent).Get(HoleList), hl);
  }
  ofile << type << " " << name << " " << dir << " " << " " << hole;
  Save_position(pattern, ofile); ofile << endl;

  // Modified at Version 3.3
  //  if (strcmp(Version, Ver_3_3_String) >= 0) {
    if (type == ReplPattern) {
      Am_Object repl_holes = pattern.Get_Part(ContentsPart).Get_Part(ContentsPart);
      Save_position(repl_holes, ofile);
    }
    //  }

  Save_port_part(pattern.Get(PortList), ofile);
  Save_holes_part(pattern.Get(HoleList), ofile);
  Save_goals_part(pattern, ofile);
  Save_binder_part(pattern, ofile);
  Save_layout_names(pattern, ofile);
  Save_size_list(pattern, ofile);
  ofile << End_Patt << endl;
}

int Load_pattern(Am_Object rule, ifstream& ifile)
{
  char tag;
  ifile >> tag; if (tag != Beg_Patt) {ifile.putback(tag); return 0;}
  int type, dir, hole;
  char name[1000];
  ifile >> type >> name >> dir >> hole;
  cout << "Load pattern: " << name << " type: " << type << " hole: " << hole << endl;
  
  Am_Object new_patt;
  if (type == NormalPattern) new_patt = New_obj(PatternProto);
  if (type == ReplPattern)  new_patt = New_obj(ReplPatternProto).Set(Dir, dir);
  new_patt.Get_Object(Model).Set(Name, name);

  Load_position(new_patt, ifile);

  Set_zooming(new_patt, false);

  // Modified at Version 3.3
  if (strcmp(Version, Ver_3_3_String) >= 0) {
    if (type == ReplPattern) {
      Am_Object repl_holes = new_patt.Get_Part(ContentsPart).Get_Part(ContentsPart);
      Load_position(repl_holes, ifile);
      SetOrigXYWH(repl_holes,
		  (int)repl_holes.Get(Am_LEFT),
		  (int)repl_holes.Get(Am_TOP),
		  (int)repl_holes.Get(Am_WIDTH),
		  (int)repl_holes.Get(Am_HEIGHT));
    }
  }
  Load_port_part(new_patt, ifile);
  Load_holes_part(new_patt, ifile);
  Load_goals_part(new_patt, ifile);
  Load_binder_part(new_patt, ifile);
  Load_layout_names(new_patt, ifile);
  Load_size_list(new_patt, ifile);
  Set_zooming(new_patt, true);

  if (hole != 0) {
    Am_Object dest = Object_List_Get(rule.Get(HoleList), hole);
    Single_Hole_Replace(dest, new_patt);
  } else {
    Add_obj(rule, new_patt);
  }
  ifile >> tag; if (tag != End_Patt) return 0;
  cout << "end Load pattern " << endl;
  return 1;
}

// -------- Hole Part --------
// Beg_Holes
// ... holes ...
// End_Holes

int Save_holes_part(Am_Value_List holes, ofstream& ofile)
{
  ofile << Beg_Holes << endl;
  for (holes.Start(); !holes.Last(); holes.Next()) {
    Am_Object hole = holes.Get();
    Save_hole(hole, ofile);
  }
  ofile << End_Holes << endl;
}

int Load_holes_part(Am_Object patt, ifstream& ifile)
{
  char tag;
  ifile >> tag; if (tag != Beg_Holes) return 0;
  int i = 1;
  while (Load_hole(patt, i, ifile)) ++i;
  ifile >> tag; if (tag != End_Holes) return 0;
}

// -------- Hole --------
// Beg_Hole Name position_info
// ... ports ...
// End_Hole
// $B?F$OC/$+!)(B

int Save_impls_part(Am_Object hole, ofstream& ofile)
{
  ofile << Beg_Goals << endl;
  Am_Object proc = hole.Get(Process);

  Am_Value_List goals = hole.Get(ProcessList);
  for (goals.Start(); !goals.Last(); goals.Next()) {
    Am_Object goal = goals.Get();
    if (!(goal == proc)) {
      if (goal.Is_Instance_Of(ProcessProto)) Save_goal(goal, ofile);
      if (goal.Is_Instance_Of(PatternProto)) Save_pattern(goal, ofile);
    }
  }
  ofile << End_Goals << endl;
}

int Load_impls_part(Am_Object pattern, ifstream& ifile)
{
  char tag;
  ifile >> tag; if (tag != Beg_Goals) {ifile.putback(tag); return 0;}
  while (Load_goal(pattern, ifile));
  ifile >> tag; if (tag != End_Goals) return 0;
}


int Save_hole(Am_Object hole, ofstream& ofile)
{
  ofile << Beg_Hole << " ";
  Am_String name = hole.MGet(Name);
  ofile << name;
  Am_Object proc = hole.Get(Process);
  if (proc.Valid()) {
    Save_position(proc, ofile); ofile << endl;
  } else {
    Save_position(hole, ofile); ofile << endl;
  }
  Save_port_part(hole.Get(PortList), ofile);
  Save_layout_names(hole, ofile);
  Save_size_list(hole, ofile);
  Save_impls_part(hole, ofile);
  ofile << End_Hole << endl;
}

int Load_hole(Am_Object patt, int i, ifstream& ifile)
{
  char tag;
  ifile >> tag; if (tag != Beg_Hole) {ifile.putback(tag); return 0;}
  char name[1000];
  ifile >> name;
  Am_Object hole;
  if (patt.Is_Instance_Of(ReplPatternProto)) {
    Am_Value_List hole_list = patt.Get(HoleList);  hole_list.Start();
    hole = Object_List_Get(hole_list, i);
    //    hole = hole_list.Get();
  } else {
    hole = New_obj(HoleProto);
  } 
  hole.Get_Object(Model).Set(Name, name);
  Load_position(hole, ifile);
  if (!patt.Is_Instance_Of(ReplPatternProto)) Add_obj(patt, hole);
  Am_Value_List ports = hole.Get(PortList);

  Set_zooming(hole, false);
  if ((patt.Is_Instance_Of(ReplPatternProto) && i == 1) || ports.Empty())
    Load_port_part(hole, ifile); // $B$-$?$M!<!#(B
  else Load_exist_port_part(hole, ifile);
  Set_zooming(hole, true);
  Load_layout_names(hole, ifile);
  Load_size_list(hole, ifile);

  Load_impls_part(patt, ifile);
  ifile >> tag; if (tag != End_Hole) {Err("End_Hole", name); return 0;}
  return 1;
}


// -------- Port Part --------
// Beg_Ports
// ... ports ...
// End_Ports

int Save_port_part(Am_Value_List ports, ofstream& ofile)
{
  ofile << Beg_Ports << endl;
  for (ports.Start(); !ports.Last(); ports.Next()) {
    Am_Object port = ports.Get();
    Save_port(port, ofile);
  }
  ofile << End_Ports << endl;
}

int Load_port_part(Am_Object process, ifstream& ifile)
{
  char tag;
  ifile >> tag; if (tag != Beg_Ports) return 0;
  while (Load_port(process, ifile));
  ifile >> tag; if (tag != End_Ports) return 0;
}

int Load_exist_port_part(Am_Object process, ifstream& ifile)
{
  char tag;
  ifile >> tag; if (tag != Beg_Ports) return 0;

  Am_Value_List ports = process.Get(PortList);
  for (ports.Start(); !ports.Last(); ports.Next()) {
    Am_Object port = (Am_Object)ports.Get();
    if (!Load_exist_port(port, ifile)) return 0;
  }

  ifile >> tag; if (tag != End_Ports) return 0;
}

// -------- Port --------
// Beg_Port Name Type Age Mode Open Dir Spec position_info
// ... Values ...
// End_Port

int Save_port(Am_Object port, ofstream& ofile)
{
  ofile << Beg_Port  << " ";
  Am_String name = port.MGet(Name);
  int type = port.MGet(Type);
  int age  = port.Get(Age);
  int mode = port.Get(Mode);
  int state = port.Get(State);
  int dir  = port.Get(Dir);
  int Spec = port.MGet(Special);
  ofile << name <<" "<< type <<" "<< age <<" "<< mode <<" "<< state <<" "<< dir <<" "<< Spec;
  Save_position(port, ofile);
  ofile << endl;
  Save_value_part(port.Get(ValueList), ofile);
  ofile << End_Port << endl;
}

int Load_port(Am_Object process, ifstream& ifile)
{
  char tag;
  ifile >> tag; if (tag != Beg_Port) {ifile.putback(tag); return 0;}
  int type, age, mode, state, dir, Spec;
  char name[1000];
  ifile >> name >> type >> age >> mode >> state >> dir >> Spec;
  Am_Object new_port;
  if (Spec == PortNormal) {
    if (mode == PortInput  && type == PortSingleton) new_port = New_obj(InSingletonProto);
    if (mode == PortOutput && type == PortSingleton) new_port = New_obj(OutSingletonProto);
    if (mode == PortInput  && type == PortStream) new_port = New_obj(InStreamProto);
    if (mode == PortOutput && type == PortStream) new_port = New_obj(OutStreamProto);
  } else {
    if (Spec == PortPatSingleton) new_port = New_obj(SpecialSingletonProto);
    if (Spec == PortPatStream) new_port = New_obj(SpecialStreamProto);

    if (Spec == PortMap && mode == PortOutput) new_port = New_obj(MapInProto);
    if (Spec == PortMap && mode == PortInput) new_port = New_obj(MapOutProto);
    if (Spec == PortBroadcast && type == PortSingleton) new_port = New_obj(BroadSingletonProto);
    if (Spec == PortBroadcast && type == PortStream) new_port = New_obj(BroadStreamProto);
    if (Spec == PortMerge) new_port = New_obj(MergeProto);
  }
  new_port
    //    .Set(Age,  age)
    //.Set(Mode, mode)
    .Set(State, state)
    .Set(Dir,  dir)
    .Get_Object(Model)
      .Set(Name, name)
    ;
  //  if (process.Is_Instance_Of(RuleProto) && age == PortOld) new_port.Set(Mode, 1 - mode);
  Load_position(new_port, ifile);
  Add_obj(process, new_port);
  Load_value_part(new_port, ifile);
  ifile >> tag; if (tag != End_Port) return 0;
  return 1;
}

int Load_exist_port(Am_Object port, ifstream& ifile)
{
  char tag;
  ifile >> tag; if (tag != Beg_Port) {ifile.putback(tag); return 0;}
  int type, age, mode, state, dir, Spec;
  char name[1000];
  ifile >> name >> type >> age >> mode >> state >> dir >> Spec;
  port.Set(State, state)
      .Set(Dir,  dir)
    ;
  Am_Object owner = port.Get_Owner();
  port.Remove_From_Owner(); // $B1x$$!#(B
  Load_position(port, ifile);
  owner.Add_Part(port); // $B1x$$!#(B
  Load_value_part(port, ifile);
  ifile >> tag; if (tag != End_Port) return 0;
  return 1;
}

int Load_val_port(Am_Object port, ifstream& ifile)
{
  char tag;
//  ifile >> tag; if (tag != Beg_Port) {ifile.putback(tag); return 0;}
  int type, age, mode, state, dir, Spec;
  char name[1000];
  ifile >> name >> type >> age >> mode >> state >> dir >> Spec;

  //  Am_Object rule = Parent_Rule(port);
  //  if (!rule.Valid()) return 0;
  //  Am_Object old_port = 0;
  //  Am_Value_List inner_ports = (Am_Value_List)rule.Get(InnerPortList);
  //  old_port = Object_List_Get_By_Model_Key(inner_ports, Name, name);

  Am_Object new_port;
  //  if (old_port.Valid()) {
  //    new_port = old_port.Create(); // ***
  //    Load_position(new_port, ifile);
  //    port.Get_Part(ContentsPart).Add_Part(new_port);
  //  } else
  if (Spec == PortNormal) {
    if (mode == PortInput  && type == PortSingleton) new_port = New_obj(InSingletonProto);
    if (mode == PortOutput && type == PortSingleton) new_port = New_obj(OutSingletonProto);
    if (mode == PortInput  && type == PortStream) new_port = New_obj(InStreamProto);
    if (mode == PortOutput && type == PortStream) new_port = New_obj(OutStreamProto);
    new_port
      .Set(Dir,  dir)
      //.Set(Mode, mode)
      .Set(Age,  age)
      .Set(State, state)
      .Get_Object(Model)
        .Set(Name, name)
      ;
    Load_position(new_port, ifile);
    //    Add_obj(port, new_port);
    port.Get_Part(ContentsPart).Add_Part(new_port);
  } else return 0;
  Load_value_part(new_port, ifile);
  ifile >> tag; if (tag != End_Port) return 0;
  return 1;
}

// -------- Values Part --------
// Beg_Vals
// ... values ...
// End_Vals

int Save_value_part(Am_Value_List vals, ofstream& ofile)
{
  ofile << Beg_Vals << endl;
  for (vals.Start(); !vals.Last(); vals.Next()) {
    Am_Object val = vals.Get();
    Save_value(val, ofile);
  }
  ofile << End_Vals << endl;
}

int Load_value_part(Am_Object port, ifstream& ifile)
{
  char tag;
  ifile >> tag; if (tag != Beg_Vals) return 0;
  while (Load_value(port, ifile));
  ifile >> tag; if (tag != End_Vals) return 0;
}

// -------- Value --------

int Save_value(Am_Object value, ofstream& ofile)
{
  if (value.Is_Instance_Of(AtomProto)) return Save_atom(value, ofile);
  if (value.Is_Instance_Of(MessageProto)) return Save_structure(value, ofile);
  if (value.Is_Instance_Of(PortProto)) return Save_port(value, ofile);
  return 0;
}

int Load_value(Am_Object port, ifstream& ifile)
{
  char tag;
  ifile >> tag;
  if (tag == Beg_Atom) return Load_atom(port, ifile);
  if (tag == Beg_Stru) return Load_structure(port, ifile);
  if (tag == Beg_Port) return Load_val_port(port, ifile);
  ifile.putback(tag);
  return 0;
}

// -------- Atom --------
// Beg_Atom Contents
// End_Atom
int Save_atom(Am_Object value, ofstream& ofile)
{
  ofile << Beg_Atom << " ";
  ofile << (Am_String)value.Get_Object(Model).Get(Name) << endl;
  ofile << End_Atom << endl;
}

int Load_atom(Am_Object port, ifstream& ifile)
{
  char tag;
  char Contents[1000];
  ifile >> ws;
  ifile.getline(Contents, 1000);
  Am_Object new_atom = New_obj(AtomProto);
  new_atom.Get_Object(Model)
    .Set(Name, Contents);
  //  Add_obj(port, new_atom);
  port.Get_Part(ContentsPart).Add_Part(new_atom);
  ifile >> tag; if (tag != End_Atom) return 0;
  return 1;
}

// -------- Structure --------
// Beg_Stru Name
// ... values ...
// End_Stru

int Save_structure(Am_Object value, ofstream& ofile)
{
  ofile << Beg_Stru << " ";
  ofile << (Am_String)value.Get_Object(Model).Get(Name) << endl;
  Save_value_part(value.Get(ValueList), ofile);
  ofile << End_Stru << endl;
}

int Load_structure(Am_Object port, ifstream& ifile)
{
  char tag;
  char name[1000];
  ifile >> name;
  Am_Object new_str = New_obj(MessageProto);
  new_str.Get_Object(Model)
    .Set(Name, name);
  //  Add_obj(port, new_str);
  port.Get_Part(ContentsPart).Add_Part(new_str);
  Load_value_part(new_str, ifile);
  ifile >> tag; if (tag != End_Stru) return 0;
  return 1;
}

