// Copyright (C) 1999  Toyoda Masashi (toyoda@is.titech.ac.jp)

#include <amulet.h>

#include <fstream.h>
#include <strstream.h>

#include "ToyWidgets.h"

DEFINE_SLOT(MId);

DEFINE_SLOT(ImageLarge);
DEFINE_SLOT(ImageSmall);
DEFINE_SLOT(MochiItems);

DEFINE_SLOT(ItemFrame);
DEFINE_SLOT(BitmapLarge);
DEFINE_SLOT(BitmapSmall);
DEFINE_SLOT(FileLarge);
DEFINE_SLOT(FileSmall);

DEFINE_SLOT(HyperLink);
DEFINE_SLOT(LinkArrow);

DEFINE_SLOT(SrcItem);
DEFINE_SLOT(DesItem);

Am_Object TopItemProto = 0;
Am_Object ItemProto = 0;
Am_Object TextItemProto = 0;

Am_Object LinkProto = 0;
Am_Object LinkPart = 0;
Am_Object LinkMaker = 0;
Am_Object LinkInteractor = 0;


// Operations
// ----------------------------------------------------------------------
Am_Object Item_Get_Parent(Am_Object item)
{
  Am_Object group = item.Get_Owner();
  if (!group.Valid()) return 0;
  
  Am_Object paren = group.Get_Owner();
  if (!paren.Valid()) return 0;
  if (!paren.Is_Instance_Of(ItemProto)) return 0;

  return paren;
}

void Item_Add(Am_Object target, Am_Object obj)
{
  target.Get_Object(propagateChildren).Add_Part(obj);
  target.Get(Am_LEFT); // invoke demon
}

void Item_Remove(Am_Object target, Am_Object obj)
{
  target.Get_Object(propagateChildren).Remove_Part(obj);
  target.Get(Am_LEFT); // invoke demon
}

void Item_Set_LImage(Am_Object item, char* filename)
{
  item.Set(ImageLarge, Am_Image_Array(filename));
  item.Set(FileLarge, filename);
}

void Item_Set_SImage(Am_Object item, char* filename)
{
  item.Set(ImageSmall, Am_Image_Array(filename));
  item.Set(FileSmall, filename);
}

void Item_Set_HLink(Am_Object item, Am_Object des)
{
  Am_Object new_link = LinkProto.Create()
    .Set(SrcItem, item)
    .Set(DesItem, des)
    ;
  LinkPart.Add_Part(new_link);

  item.Set(HyperLink, des);

  Am_Object link = item.Get(LinkArrow);
  if (link.Valid()) {
    link.Remove_From_Owner();
    link.Destroy();
  }
  item.Set(LinkArrow, new_link);
}

void Item_Remove_Link(Am_Object item)
{
  item.Set(HyperLink, 0);

  Am_Object link = item.Get(LinkArrow);
  item.Set(LinkArrow, 0);
  if (link.Valid()) {
    link.Remove_From_Owner();
    link.Destroy();
  }
}

void Item_Set_Link_Visibility(Am_Object item, bool visible)
{
  Am_Object link = item.Get(LinkArrow);
  if (link.Valid()) 
    link.Set(Am_ACTIVE, visible);
}

void Item_Toggle_Link_Visibility(Am_Object item)
{
  Am_Object link = item.Get(LinkArrow);
  if (link.Valid()) {
    link.Set(Am_ACTIVE, !link.Get(Am_ACTIVE));
  }
}

void Item_Retrieve_Link(Am_Object item, Am_Object selection)
{
  Am_Object des = item.Get(HyperLink);
  if (!des.Valid()) return;

  Am_Object src = Item_Get_Parent(item);
  if (!src.Valid()) src = item;
  Retrieve_Hyper_Link(src, des, selection);
  return;
}

// drag & drop
Am_Define_Method(Bin_Check_Method, bool, Item_Check_Drop, (Am_Object source, Am_Object target))
{
  // $B<+J,<+?H$K$O%I%m%C%W$G$-$J$$!#(B
  if (source == target) return false;
  // $BF1$83,AX$G$N0\F0$O%I%m%C%W$K$J$i$J$$!#(B
  if (Item_Get_Parent(source) == target) return false;
  // $B0\F0$9$k%*%V%8%'%/%H$N;R$I$b$K$O%I%m%C%W$G$-$J$$!#(B
  if (target.Is_Part_Of(source)) return false;
  return true;
}

Am_Define_Method(Drop_Method, bool, Item_Drop, (Am_Object source, Am_Object target, Am_Object cmd))
{
  Am_Object group = target.Get(propagateChildren);

  Am_Inter_Location new_loc = cmd.Get(Am_VALUE);
  bool as_line;
  Am_Object nref, oref;
  int nx, ny, ox, oy, w, h;
  new_loc.Get_Location(as_line, nref, nx, ny, w, h);
  Am_Translate_Coordinates(nref, nx, ny, group, nx, ny);

  Am_Object copy = Align_Zoom_Copy_Part(source);
  copy.Set(Am_LEFT, nx)
    .Set(Am_TOP, ny);

  Item_Add(target, copy);
  return true;
}



// Save and Load
// ----------------------------------------------------------------------
// Normal Item
// "item" id paren_id x y w h sw sh lw lh "small_image" "large_image"
// Text Item
// "text" id paren_id x y w h sw sh lw lh fg bg ff fb fi fu "text"

Am_Object *items = new Am_Object[1000];

void save_color(Am_Style sty, ofstream& ofile)
{
  if (sty == Am_No_Style) ofile << 0 << " ";
  else if (sty == Am_Red)      ofile << 1 << " ";
  else if (sty == Am_Green)    ofile << 2 << " ";
  else if (sty == Am_Blue)	  ofile << 3 << " ";
  else if (sty == Am_Yellow)   ofile << 4 << " ";
  else if (sty == Am_Purple)   ofile << 5 << " ";
  else if (sty == Am_Cyan)     ofile << 6 << " ";
  else if (sty == Am_Orange)   ofile << 7 << " ";
  else if (sty == Am_Black)    ofile << 8 << " ";
  else if (sty == Am_White)    ofile << 9 << " ";
  else if (sty == Am_Amulet_Purple) ofile << 10 << " ";
  else if (sty == Am_Motif_Gray)        ofile << 11 << " ";
  else if (sty == Am_Motif_Light_Gray)  ofile << 12 << " ";
  else if (sty == Am_Motif_Blue)        ofile << 13 << " ";
  else if (sty == Am_Motif_Light_Blue)  ofile << 14 << " ";
  else if (sty == Am_Motif_Green)       ofile << 15 << " ";
  else if (sty == Am_Motif_Light_Green) ofile << 16 << " ";
  else if (sty == Am_Motif_Orange)      ofile << 17 << " ";
  else if (sty == Am_Motif_Light_Orange) ofile << 18 << " ";
  else ofile << 19 << " ";
}

Am_Style load_color(ifstream& ifile)
{
  int sty;
  ifile >> sty;
  switch (sty) {
  case 0: return Am_No_Style;
  case 1: return Am_Red;
  case 2: return Am_Green;
  case 3: return Am_Blue;
  case 4: return Am_Yellow;
  case 5: return Am_Purple;
  case 6: return Am_Cyan;
  case 7: return Am_Orange;
  case 8: return Am_Black;
  case 9: return Am_White;
  case 10: return Am_Amulet_Purple;
  case 11: return Am_Motif_Gray;
  case 12: return Am_Motif_Light_Gray;
  case 13: return Am_Motif_Blue;
  case 14: return Am_Motif_Light_Blue;
  case 15: return Am_Motif_Green;
  case 16: return Am_Motif_Light_Green;
  case 17: return Am_Motif_Orange;
  case 18: return Am_Motif_Light_Orange;
  default: return Am_Black;
  }
}

int item_save_iter(Am_Object item, int id, ofstream& ofile)
{
  item.Set(MId, id);
  items[id] = item;

  if (id != 0) {
    if (item.Is_Instance_Of(TextItemProto)) { ofile << "text "; }
    else                                    { ofile << "item "; }

    { ofile << id << " "; }

    Am_Object paren = Item_Get_Parent(item);
    if (paren.Valid()) {
      ofile << (int)paren.Get(MId) << " ";
    } else {
      ofile << 0 << " ";
    }
//     else {
//       cerr << "save error: no parent " << item << endl;
//       return 0;
//     }

    { OutputCoords(item, ofile); }

    if (item.Is_Instance_Of(TextItemProto)) {
      Am_Object text_obj = item.Get(BitmapLarge);
      Am_String text = text_obj.Get_Part(TextPart).Get(Am_TEXT);
      Am_Style fg = text_obj.Get_Part(TextPart).Get(Am_LINE_STYLE);
      Am_Style bg = item.Get_Part(ItemFrame).Get(Am_FILL_STYLE);
      { ofile << (int)text_obj.Get(FontFamily) << " "; }
      { ofile << (int)text_obj.Get(FontIsBold) << " "; }
      { ofile << (int)text_obj.Get(FontIsItalic) << " "; }
      { ofile << (int)text_obj.Get(FontIsUnderline) << " "; }
      save_color(fg, ofile);
      save_color(bg, ofile);
      { ofile << text; }
    } else {
      Am_String fileL = item.Get(FileLarge);
      Am_String fileS = item.Get(FileSmall);
      if (fileS.Valid())                    { ofile << fileS << " "; }
      else                                  { ofile << ". " ; }
      if (fileL.Valid())                    { ofile << fileL; }
      else                                  { ofile << "."; }
    }

    { ofile << endl; }
  }
  ++id;
  Am_Value_List children = item.Get_Object(propagateChildren).Get(Am_GRAPHICAL_PARTS);
  for (children.Start(); !children.Last(); children.Next()) {
    Am_Object ch = children.Get();
    if (ch.Is_Instance_Of(ItemProto))
      id = item_save_iter(ch, id, ofile);
  }
  return id;
}

void item_save_links(Am_Object item, ofstream& ofile)
{
  if (item.Is_Instance_Of(ItemProto)) {
    Am_Object des = item.Get(HyperLink);
    if (des.Valid()) {
      ofile << "link " << (int)item.Get(MId) << " " << (int)des.Get(MId) << endl;
    }
  }
  Am_Value_List children = item.Get_Object(propagateChildren).Get(Am_GRAPHICAL_PARTS);
  for (children.Start(); !children.Last(); children.Next()) {
    Am_Object ch = children.Get();
    if (ch.Is_Instance_Of(ItemProto))
      item_save_links(ch, ofile);
  }
}

void Item_Save_to_file(Am_Object top_module, ofstream& ofile)
{
  ofile << "1 MochiPointVersion0.1" << endl;
  item_save_iter(top_module, 0, ofile);
  item_save_links(top_module, ofile);
}

void Item_Save(char* name)
{
  ofstream ofile;
  ofile.open(name, ios::out);
  if (!ofile) {
    cerr << "can't save to file: " << name << endl;
    return;
  }
  Item_Save_to_file(TopItemProto, ofile);
}

void load_item(ifstream& ifile)
{
  int id, pid, x, y, cw, ch, sw, sh, lw, lh;
  char sfile[100], lfile[100];

  ifile >> id >> pid >> x >> y >> cw >> ch >> sw >> sh >> lw >> lh >> sfile >> lfile;
  Am_Object new_item = ItemProto.Create()
    .Set(Am_LEFT, x)
    .Set(Am_TOP, y)
    .Set(Am_WIDTH, cw)
    .Set(Am_HEIGHT, ch)
    .Get_Part(propagateChildren)
    .Set(Am_ACTIVE, false)
    .Get_Owner()
    ;
  items[id] = new_item;
  Am_Object parent = items[pid];

  Item_Add(parent, new_item);
  SetPointers(new_item, cw, ch, sw, sh, lw, lh);
  if (strcmp(sfile, ".") != 0) Item_Set_SImage(new_item, sfile);
  if (strcmp(lfile, ".") != 0) Item_Set_LImage(new_item, lfile);
}

void load_text(ifstream& ifile)
{
  int id, pid, x, y, cw, ch, sw, sh, lw, lh;
  int ff, fb, fi, fu;
  char text[1000];

  ifile >> id >> pid >> x >> y >> cw >> ch >> sw >> sh >> lw >> lh;
  ifile >> ff >> fb >> fi >> fu;
  Am_Style fg = load_color(ifile);
  Am_Style bg = load_color(ifile);
  ifile.get();
  ifile.getline(text, 1000);

  Am_Object new_text = TextItemProto.Create()
    .Set(Am_LEFT, x)
    .Set(Am_TOP, y)
    .Set(Am_WIDTH, cw)
    .Set(Am_HEIGHT, ch)
    .Get_Part(ItemFrame)
    .Set(Am_FILL_STYLE, bg)
    .Get_Owner()
    .Get_Part(propagateChildren)
    .Set(Am_ACTIVE, false)
    .Get_Owner()
    ;
  new_text.Get_Object(BitmapLarge)
    .Set(FontFamily, ff)
    .Set(FontIsBold, fb)
    .Set(FontIsItalic, fi)
    .Set(FontIsUnderline, fu)
    .Get_Part(TextPart)
      .Set(Am_TEXT, text)
      .Set(Am_LINE_STYLE, fg)
    ;
  items[id] = new_text;
  Am_Object parent = items[pid];

  Item_Add(parent, new_text);
  SetPointers(new_text, cw, ch, sw, sh, lw, lh);
}

void load_link(ifstream& ifile)
{
  int src_id, des_id;
  ifile >> src_id >> des_id;
  Item_Set_HLink(items[src_id], items[des_id]);
}

void activate_items(Am_Object item)
{
  item.Get_Object(propagateChildren).Set(Am_ACTIVE, true);
  Am_Value_List children = item.Get_Object(propagateChildren).Get(Am_GRAPHICAL_PARTS);
  for (children.Start(); !children.Last(); children.Next()) {
    Am_Object ch = children.Get();
    if (ch.Is_Instance_Of(ItemProto))
      activate_items(ch);
  }
}


void Item_Load_from_file(Am_Object item, ifstream& ifile)
{
  char version[100];
  ifile.getline(version, 100);

  items[0] = item;
  item.Get_Object(propagateChildren).Set(Am_ACTIVE, false);

  char tag[10];
  while (ifile >> tag) {
    if (strcmp(tag, "item") == 0) load_item(ifile);
    if (strcmp(tag, "text") == 0) load_text(ifile);
    if (strcmp(tag, "link") == 0) load_link(ifile);
  }
  activate_items(item);
}

void Item_Load(Am_Object item, char* name)
{
  ifstream ifile;
  ifile.open(name, ios::in);
  if (!ifile) {
    cerr << "can't load from: " << name << endl;
    return;
  }
  Item_Load_from_file(item, ifile);
}


// Formulas
// ----------------------------------------------------------------------
Am_Define_Formula(bool, BitmapLargeVisibleForm)
{
  Am_Object conts = self.GV_Sibling(propagateChildren); // items part
  if (!conts.Valid()) return true;

  int thresh = conts.GV(detailThreshold);
  if ((int)conts.GV(Am_WIDTH) < thresh ||
      (int)conts.GV(Am_HEIGHT) < thresh) return false;
  return true;
}

Am_Define_Formula(bool, BitmapSmallVisibleForm)
{
  Am_Object conts = self.GV_Sibling(propagateChildren); // items part
  if (!conts.Valid()) return false;

  int thresh = conts.GV(detailThreshold);
  if ((int)conts.GV(Am_WIDTH) >= thresh &&
      (int)conts.GV(Am_HEIGHT) >= thresh) return false;
  return true;
}

Am_Define_Formula(bool, TextVisibleForm)
{
  Am_Object conts = self.GV_Sibling(propagateChildren); // items part
  if (!conts.Valid()) return true;

  int thresh = conts.GV(detailThreshold);
  Am_Value_List objs = conts.GV(Am_GRAPHICAL_PARTS);

  if ((int)conts.GV(Am_WIDTH) < thresh || (int)conts.GV(Am_HEIGHT) < thresh)
    return true;
  else if (objs.Empty())
    return true;
  else
    return false;
}

Am_Define_Value_List_Formula(MochiHyperLinkListForm)
{
  Am_Value_List links = Am_Value_List();

  Am_Value_List items = self.GV_Object(propagateChildren).GV(Am_GRAPHICAL_PARTS);
  for (items.Start(); !items.Last(); items.Next()) {
    Am_Object i = items.Get();
    if (i.Is_Instance_Of(ItemProto)) {
      Am_Object des = i.GV(HyperLink);
      if (des.Valid()) links.Add(des);
    }
  }
  return links;
}



// Initialize Prototype
// ----------------------------------------------------------------------
void initItem()
{
  Am_Object imageL = Am_Bitmap.Create("ImageLarge")
    .Set(Am_LEFT, Am_Center_X_Is_Center_Of_Owner)
    .Set(Am_TOP, Am_Center_Y_Is_Center_Of_Owner)
    .Set(Am_IMAGE, Am_From_Owner(ImageLarge))
    .Set(Am_VISIBLE, BitmapLargeVisibleForm)
    ;

  Am_Object imageS = imageL.Create("ImageSmall")
    .Set(Am_IMAGE, Am_From_Owner(ImageSmall))
    .Set(Am_VISIBLE, BitmapSmallVisibleForm)
    ;

  Am_Object conts = AlignZoomGroup.Create("MochiItems")
    .Set(propagateParent, PropagateForm)
    .Set(detailThreshold, 70)
    .Set_Single_Constraint_Mode(Am_WIDTH, false)
    .Set_Single_Constraint_Mode(Am_HEIGHT, false)
    
    .Set(Am_WIDTH, Am_From_Owner(Am_WIDTH))
    .Set(Am_HEIGHT, Am_From_Owner(Am_HEIGHT))
    ;

  Am_Object frame = Am_Rectangle.Create("ItemFrame")
    .Set(Am_WIDTH, Am_From_Owner(Am_WIDTH))
    .Set(Am_HEIGHT, Am_From_Owner(Am_HEIGHT))
    .Set(Am_FILL_STYLE, Am_No_Style)
    ;

  ItemProto = Am_Group.Create("ItemProto")
    .Set(MId, 0)

    .Set(CheckDropMethod, Item_Check_Drop)
    .Set(DropMethod, Item_Drop)


    .Set(ImageLarge, 0)
    .Set(ImageSmall, 0)
    .Set(FileLarge, 0)
    .Set(FileSmall, 0)

    .Add_Part(ItemFrame, frame)
    .Add_Part(BitmapLarge, imageL)
    .Add_Part(BitmapSmall, imageS)
    .Add_Part(propagateChildren, conts)

    .Set(HyperLink, 0)
    .Set(hyperLinks, MochiHyperLinkListForm)
    .Set(LinkArrow, 0)

    .Set(Am_WIDTH, 150)
    .Set(Am_HEIGHT, 100)
    ;

  ItemProto.Set(DropPoints, Am_Value_List().Add(ItemProto));

  TopItemProto = ItemProto.Create("TopItemProto")
    .Set(Am_WIDTH, Am_From_Owner(Am_WIDTH))
    .Set(Am_HEIGHT, Am_From_Owner(Am_HEIGHT))
    .Get_Part(propagateChildren)
    .Set(propagateParent, 0)
    .Get_Owner()
    ;

  ItemProto
    .Set_Single_Constraint_Mode(Am_WIDTH, false)
    .Set_Single_Constraint_Mode(Am_HEIGHT, false)
    .Set(Am_WIDTH, Am_From_Part(propagateChildren, Am_WIDTH))
    .Set(Am_HEIGHT, Am_From_Part(propagateChildren, Am_HEIGHT))
    ;

  // text item

  Am_Object text = ZoomingText.Create("TextLarge")
    .Set(FontFamily, 2)
    .Set(FontIsBold, true)
    .Set(detailThreshold, 70)
    .Set_Single_Constraint_Mode(Am_WIDTH, false)
    .Set_Single_Constraint_Mode(Am_HEIGHT, false)
    
    .Set(Am_WIDTH, Am_From_Owner(Am_WIDTH))
    .Set(Am_HEIGHT, Am_From_Owner(Am_HEIGHT))
    .Set(Am_VISIBLE, TextVisibleForm)

    .Get_Part(TextPart)
    .Set(Am_TEXT, "Text Item")
    .Set(Am_LINE_STYLE, Am_White)
    .Add_Part(Am_INTERACTOR, Am_Text_Edit_Interactor.Create()
	      .Set(Am_START_WHEN, "middle_down")
	      .Set(Am_PRIORITY, 5))
    .Get_Owner()
    ;

  TextItemProto = ItemProto.Create("TextItemProto")
    .Get_Part(ItemFrame)
    .Set(Am_LINE_STYLE, Am_No_Style)
    .Set(Am_FILL_STYLE, Am_Blue)
    .Get_Owner()
    ;
  TextItemProto.Remove_Part(BitmapLarge);
  TextItemProto.Remove_Part(BitmapSmall);

  TextItemProto
    .Add_Part(BitmapLarge, text)
    ;
}


// Initialize Hyper Link
// ----------------------------------------------------------------------
Am_Define_Formula(bool, LinkLayout) {
  if (!self.GV(Am_ACTIVE)) return false;

  // src port $B$H(Bdestination port
  Am_Object src = self.GV(SrcItem);
  Am_Object des = self.GV(DesItem);

  // BindersPart$B$NI}!"9b$5$,JQ$o$C$?$i:F7W;;(B
  Am_Object binders_part = self.GV_Owner();
  if (!binders_part.Valid()) return true;
  binders_part.GV(Am_WIDTH);
  binders_part.GV(Am_HEIGHT); 

  bool visible = true;

  Am_Object src_owner = src.GV_Owner();
  Am_Object src_grand = src_owner;
  while(src_grand.Valid()) {
    src_grand.GV(Am_WIDTH); src_grand.GV(Am_HEIGHT);
    src_grand.GV(Am_LEFT); src_grand.GV(Am_TOP);

    if (src_grand.GV_Owner().Valid()) src_grand = src_grand.GV_Owner();
    else break;
    if (!src_grand.GV(Am_VISIBLE)) visible = false;
  }

  Am_Object des_owner = des.GV_Owner();
  Am_Object des_grand = des_owner;
  while(des_grand.Valid()) {
    des_grand.GV(Am_WIDTH); des_grand.GV(Am_HEIGHT);
    des_grand.GV(Am_LEFT); des_grand.GV(Am_TOP);

    if (des_grand.GV_Owner().Valid()) des_grand = des_grand.GV_Owner();
    else break;
    if (!des_grand.GV(Am_VISIBLE)) visible = false;
  }

  if (!visible) return false;

  if (!src_owner.GV(Am_VISIBLE)) src = src_owner;
  if (!des_owner.GV(Am_VISIBLE)) des = des_owner;

  int src_x, src_y, des_x, des_y;
  Am_Translate_Coordinates(src, 0, 0, binders_part, src_x, src_y);
  Am_Translate_Coordinates(des, 0, 0, binders_part, des_x, des_y);

  int src_w = src.GV(Am_WIDTH);
  int src_h = src.GV(Am_HEIGHT);
  int des_w = des.GV(Am_WIDTH);
  int des_h = des.GV(Am_HEIGHT);

  src.GV(Am_TOP);  src.GV(Am_LEFT);
  des.GV(Am_TOP);  des.GV(Am_LEFT);

  int x1 = src_x + (src_w / 2);
  int y1 = src_y + (src_h / 2);
  if (des_x + des_w < src_x) x1 = src_x; 
  if (src_x + src_w < des_x) x1 = src_x + src_w;
  if (des_y + des_h < src_y) y1 = src_y;
  if (src_y + src_h < des_y) y1 = src_y + src_h;
  if (x1 == src_x + (src_w / 2) && y1 == src_y + (src_h / 2)) return false;

  int x2 = des_x + (des_w / 2);
  int y2 = des_y + (des_h / 2);
  if (src_x + src_w < des_x) x2 = des_x;
  if (des_x + des_w < src_x) x2 = des_x + des_w;
  if (src_y + src_h < des_y) y2 = des_y;
  if (des_y + des_h < src_y) y2 = des_y + des_h;
  if (x2 == des_x + (des_w / 2) && y2 == des_y + (des_h / 2)) return false;

  self.Set(Am_X1, x1);
  self.Set(Am_X2, x2);
  self.Set(Am_Y1, y1);
  self.Set(Am_Y2, y2);
  return true;
}

void initLinkArrow(void)
{
  LinkProto = Am_Arrow.Create("LinkProto")
    .Set(Am_ARROW_H_WIDTH, 15)
    .Set(Am_ARROW_H_HEIGHT, 30)
    .Set(Am_LINE_STYLE, Am_Line_2)
    .Set(Am_ACTIVE, true)
    .Set(Am_VISIBLE, LinkLayout)
    ;

  LinkPart = Am_Group.Create("LinkPart")
    .Set(Am_WIDTH, Am_From_Owner(Am_WIDTH))
    .Set(Am_HEIGHT, Am_From_Owner(Am_HEIGHT))
    ;
}


// Initialize Menu
// ----------------------------------------------------------------------
Am_Define_Method(Am_Object_Method, void, New_Item_Do, (Am_Object command_obj))
{
  Am_Object menu = command_obj.Get_Owner().Get_Owner(); // $B%a%K%e!<(B
  Am_Object target = menu.Get(TargetObj); // $BDI2C@h(B

  int x = menu.Get(XinTarget);  // $B:BI8(B
  int y = menu.Get(YinTarget);

  Am_Object new_obj = ItemProto.Create("NewItem");
  new_obj
    .Set(Am_LEFT, x - (int)new_obj.Get(Am_WIDTH) / 2) // $B%+!<%=%k$N0LCV$rCf?4$K(B
    .Set(Am_TOP, y - (int)new_obj.Get(Am_HEIGHT) / 2)
    ;
  Item_Add(target, new_obj);
}

Am_Define_Method(Am_Object_Method, void, New_Text_Do, (Am_Object command_obj))
{
  Am_Object menu = command_obj.Get_Owner().Get_Owner(); // $B%a%K%e!<(B
  Am_Object target = menu.Get(TargetObj); // $BDI2C@h(B

  int x = menu.Get(XinTarget);  // $B:BI8(B
  int y = menu.Get(YinTarget);

  Am_Object new_obj = TextItemProto.Create("NewTextItem");
  new_obj
    .Set(Am_LEFT, x - (int)new_obj.Get(Am_WIDTH) / 2) // $B%+!<%=%k$N0LCV$rCf?4$K(B
    .Set(Am_TOP, y - (int)new_obj.Get(Am_HEIGHT) / 2)
    ;
  Item_Add(target, new_obj);
}

Am_Define_Method(Am_Object_Method, void, Delete_Item_Do, (Am_Object command_obj))
{
  Am_Object menu = command_obj.Get_Owner().Get_Owner(); // $B%a%K%e!<(B
  Am_Object target = menu.Get(TargetObj); // $BDI2C@h(B

  Am_Object paren = Item_Get_Parent(target);

  if (paren.Valid()) Item_Remove(paren, target);
}

Am_Define_Method(Am_Object_Method, void, Link_Visible_Do, (Am_Object command_obj))
{
  Am_Object menu = command_obj.Get_Owner().Get_Owner(); // $B%a%K%e!<(B
  Am_Object target = menu.Get(TargetObj); // $BDI2C@h(B

  Item_Toggle_Link_Visibility(target);
}

Am_Define_Method(Am_Object_Method, void, Link_Remove_Do, (Am_Object command_obj))
{
  Am_Object menu = command_obj.Get_Owner().Get_Owner(); // $B%a%K%e!<(B
  Am_Object target = menu.Get(TargetObj); // $BDI2C@h(B

  Item_Remove_Link(target);
}

Am_Define_Method(Am_Object_Method, void, Set_LImage_Do, (Am_Object command_obj))
{
  Am_Object menu = command_obj.Get_Owner().Get_Owner(); // $B%a%K%e!<(B
  Am_Object target = menu.Get(TargetObj); // $BJQ99$9$k%*%V%8%'%/%H(B
  Am_Object window = menu.Get_Owner();

  Am_Finish_Pop_Up_Waiting(window, Am_No_Value);

  Am_String file_name = Am_Get_Input_From_Dialog(Am_Value_List().Add("Import Large Image"), "", 100, 100, false);
  if (file_name.Valid()) {
    Item_Set_LImage(target, (char*)file_name);
  }
}

Am_Define_Method(Am_Object_Method, void, Set_SImage_Do, (Am_Object command_obj))
{
  Am_Object menu = command_obj.Get_Owner().Get_Owner(); // $B%a%K%e!<(B
  Am_Object target = menu.Get(TargetObj); // $BJQ99$9$k%*%V%8%'%/%H(B
  Am_Object window = menu.Get_Owner();

  Am_Finish_Pop_Up_Waiting(window, Am_No_Value);

  Am_String file_name = Am_Get_Input_From_Dialog(Am_Value_List().Add("Import Small Image"), "", 100, 100, false);
  if (file_name.Valid()) {
    Item_Set_SImage(target, (char*)file_name);
  }
}

Am_Define_Method(Am_Object_Method, void, Item_Save_Do, (Am_Object command_obj))
{
  Am_Object menu = command_obj.Get_Owner().Get_Owner(); // $B%a%K%e!<(B
  Am_Object target = menu.Get(TargetObj); // $BJQ99$9$k%*%V%8%'%/%H(B
  Am_Object window = menu.Get_Owner();

  Am_Finish_Pop_Up_Waiting(window, Am_No_Value);

  Am_String file_name = Am_Get_Input_From_Dialog(Am_Value_List().Add("Save to File"), "", 100, 100, false);
  if (file_name.Valid()) {
    Item_Save((char*)file_name);
  }
}

Am_Define_Method(Am_Object_Method, void, Load_Do, (Am_Object command_obj))
{
  Am_Object menu = command_obj.Get_Owner().Get_Owner(); // $B%a%K%e!<(B
  Am_Object target = menu.Get(TargetObj); // $BJQ99$9$k%*%V%8%'%/%H(B
  Am_Object window = menu.Get_Owner();

  Am_Finish_Pop_Up_Waiting(window, Am_No_Value);

  Am_String file_name = Am_Get_Input_From_Dialog(Am_Value_List().Add("Load File"), "", 100, 100, false);
  if (file_name.Valid()) {
    Item_Load(target, (char*)file_name);
  }
}

void initMenu(void)
{
  Am_Object orig_size_cmd = OrigSizeCmd.Create();
  Am_Object edit_size_cmd = EditSizeCmd.Create();

  Am_Object ItemMenu = PopUpMenu.Create("ItemMenu")
    .Get_Part(MenuBody)
    //    .Set(Am_FONT, Am_Font("-*-helvetica-*-r-*--24-*"))
    .Set(Am_ITEMS, Am_Value_List()
	 .Add(MenuCommand.Create().Set(Am_LABEL, "Item Menu"))
	 .Add(Am_Menu_Line_Command.Create())
	 .Add(orig_size_cmd)
	 .Add(edit_size_cmd)
	 .Add(Am_Menu_Line_Command.Create())
	 // .Add(SetOrigCmd.Create())
	 // .Add(SetEditCmd.Create())
	 .Add(Am_Menu_Line_Command.Create())
	 .Add(MenuCommand.Create().Set(Am_LABEL, "New Item")
	      .Set(Am_DO_METHOD, New_Item_Do))
	 .Add(MenuCommand.Create().Set(Am_LABEL, "New Text")
	      .Set(Am_DO_METHOD, New_Text_Do))
	 .Add(Am_Menu_Line_Command.Create())
	 .Add(MenuCommand.Create().Set(Am_LABEL, "Link Visible")
	      .Set(Am_DO_METHOD, Link_Visible_Do))
	 .Add(MenuCommand.Create().Set(Am_LABEL, "Remove Link")
	      .Set(Am_DO_METHOD, Link_Remove_Do))
	 .Add(Am_Menu_Line_Command.Create())
	 .Add(MenuCommand.Create().Set(Am_LABEL, "Import Large Image")
	      .Set(Am_DO_METHOD, Set_LImage_Do))
	 .Add(MenuCommand.Create().Set(Am_LABEL, "Import Small Image")
	      .Set(Am_DO_METHOD, Set_SImage_Do))
	 .Add(Am_Menu_Line_Command.Create())
	 .Add(MenuCommand.Create().Set(Am_LABEL, "Load")
	      .Set(Am_DO_METHOD, Load_Do))
	 .Add(MenuCommand.Create().Set(Am_LABEL, "Save")
	      .Set(Am_DO_METHOD, Item_Save_Do))
	 .Add(Am_Menu_Line_Command.Create())
	 .Add(MenuCommand.Create().Set(Am_LABEL, "Delete")
	      .Set(Am_DO_METHOD, Delete_Item_Do))
	 )
    .Get_Owner();
  Am_Screen.Add_Part(ItemMenu);

  ItemMenu.Set(Am_VISIBLE, true);
  Am_Do_Events();
  ItemMenu.Set(Am_VISIBLE, false);
  
  orig_size_cmd.Get_Owner().Set(Am_FONT, Am_Font("-*-helvetica-*-r-*--24-*"));
  edit_size_cmd.Get_Owner().Set(Am_FONT, Am_Font("-*-helvetica-*-r-*--24-*"));
//  set_orig_cmd.Get_Owner().Set(Am_FONT, Am_Font("-*-helvetica-*-r-*--24-*"));
//  set_edit_cmd.Get_Owner().Set(Am_FONT, Am_Font("-*-helvetica-*-r-*--24-*"));

  Am_Object ItemMenuPopper = PopUpMenuInteractor.Create("ItemMenuPopper")
    .Set(MenuWindow, ItemMenu)
    .Set(Am_START_WHEN, "RIGHT_DOWN")
    ;
  ItemProto.Add_Part(ItemMenuPopper);
  TextItemProto.Add_Part(ItemMenuPopper.Create());

  Am_Object TopItemMenu = PopUpMenu.Create("ItemMenu")
    .Get_Part(MenuBody)
    //    .Set(Am_FONT, Am_Font("-*-helvetica-*-r-*--24-*"))
    .Set(Am_ITEMS, Am_Value_List()
	 .Add(MenuCommand.Create().Set(Am_LABEL, "Top Item Menu"))
	 .Add(Am_Menu_Line_Command.Create())
	 .Add(MenuCommand.Create().Set(Am_LABEL, "New Item")
	      .Set(Am_DO_METHOD, New_Item_Do))
	 .Add(MenuCommand.Create().Set(Am_LABEL, "New Text")
	      .Set(Am_DO_METHOD, New_Text_Do))
	 .Add(Am_Menu_Line_Command.Create())
	 .Add(MenuCommand.Create().Set(Am_LABEL, "Import Large Image")
	      .Set(Am_DO_METHOD, Set_LImage_Do))
	 .Add(MenuCommand.Create().Set(Am_LABEL, "Import Small Image")
	      .Set(Am_DO_METHOD, Set_SImage_Do))
	 .Add(Am_Menu_Line_Command.Create())
	 .Add(MenuCommand.Create().Set(Am_LABEL, "Load")
	      .Set(Am_DO_METHOD, Load_Do))
	 .Add(MenuCommand.Create().Set(Am_LABEL, "Save")
	      .Set(Am_DO_METHOD, Item_Save_Do))
	 .Add(Am_Menu_Line_Command.Create())
	 .Add(MenuCommand.Create().Set(Am_LABEL, "Delete")
	      .Set(Am_DO_METHOD, Delete_Item_Do))
	 )
    .Get_Owner();
  Am_Screen.Add_Part(TopItemMenu);

  Am_Object TopItemMenuPopper = PopUpMenuInteractor.Create("TopItemMenuPopper")
    .Set(MenuWindow, TopItemMenu)
    .Set(Am_START_WHEN, "RIGHT_DOWN")
    ;
  TopItemProto.Add_Part(TopItemMenuPopper);
}


// Initialize Hyper Link
// ----------------------------------------------------------------------
Am_Define_Method(Am_Where_Method, Am_Object, LinkMaker_Where_Test,
		 (Am_Object inter, Am_Object /* object */, Am_Object event_window,
		  Am_Input_Char /* ic */, int x, int y))
{
  Am_Value_List items = inter.Get(selectableObjects);

  Am_Object top_item = inter.Get_Owner();
  Am_Object obj = Content_Instance_Of(top_item, x, y, event_window, items);
  if (obj.Valid()) return obj;
  //  return top_item; // click on background
  return 0;
}

Am_Define_Method(Am_Create_New_Object_Method, Am_Object, New_HLink_Do,
		 (Am_Object inter, Am_Inter_Location location, Am_Object /* old_object */ ))
{
  Am_Value_List items = inter.Get(selectableObjects);

  Am_Object top_item = inter.Get_Owner();
  Am_Object window = inter.Get(Am_WINDOW);
  bool as_line;
  Am_Object ref;
  int x1, y1, x2, y2;
  location.Get_Location(as_line, ref, x1, y1, x2, y2);
  Am_Object src = Content_Instance_Of(top_item, x1, y1, ref, items);
  Am_Object des = Content_Instance_Of(top_item, x2, y2, ref, items);
  //  cout << src << des << endl;
  if (!src.Valid() || !des.Valid()) return Am_No_Object;
  if (src == des) return Am_No_Object;
  Item_Set_HLink(src, des);

  Am_Do_Events();

  return 0;
}

Am_Define_Method(Am_Object_Method, void, ItemHyperLinkDo, (Am_Object command_obj))
{
  Am_Object src = command_obj.Get_Owner().Get_Owner();
  Am_Object selection = src.Get_Object(Am_WINDOW).Get(Am_SELECTION_WIDGET);
  Item_Retrieve_Link(src, selection);
}

void Initialize_Hyperlink(Am_Object top_window)
{
  Am_Object LinkFeedback = Am_Arrow.Create("LinkFeedback")
    .Set(Am_ARROW_H_WIDTH, 15)
    .Set(Am_ARROW_H_HEIGHT, 30)
    .Set(Am_LINE_STYLE, Am_Line_2)
    .Set(Am_VISIBLE, false)
    ;

  LinkMaker = Am_New_Points_Interactor.Create("LinkMaker")
    .Set(Am_AS_LINE, true)
    .Set(Am_PRIORITY, 4)
    .Set(selectableObjects, Am_Value_List().Add(ItemProto))
    .Set(Am_START_WHERE_TEST, LinkMaker_Where_Test)
    .Set(Am_START_WHEN, "SHIFT_LEFT_DOWN")
    .Set(Am_FEEDBACK_OBJECT, LinkFeedback)
    .Set(Am_CREATE_NEW_OBJECT_METHOD, New_HLink_Do)
    ;

  top_window
    .Add_Part(LinkPart)
    .Add_Part(LinkMaker)
    .Add_Part(LinkFeedback)
    ;

  LinkInteractor = Am_One_Shot_Interactor.Create("LinkInteractor")
    .Set(Am_START_WHEN, "double_left_down")
    .Set(Am_PRIORITY, 2)
    .Get_Part(Am_COMMAND)
    .Set(Am_DO_METHOD, ItemHyperLinkDo)
    .Get_Owner()
    ;

  ItemProto.Add_Part(Am_INTERACTOR, LinkInteractor);
  TextItemProto.Add_Part(Am_INTERACTOR, LinkInteractor.Create());

  ItemProto.Add_Part(ToggleSizeCmd.Create().Set(Am_START_WHEN, " "));
  TextItemProto.Add_Part(ToggleSizeCmd.Create().Set(Am_START_WHEN, " "));
}





// Initialize All
// ----------------------------------------------------------------------
void Initialize_Item(void)
{
  initItem();
  initLinkArrow();
  initMenu();
}
