%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%%      trace.pl : tracer of PRISM program
%%%
%%%        Mainly developed by H. Ishimaru, T.I.T, 1998-1999.
%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%%  Copyright (C) 1999
%%%    Taisuke Sato, Yoshitaka Kameya, Yasushi Hagiwara,
%%%      Nobuhisa Ueda, Hiroshi Ishimaru
%%%    Dept. of Computer Science, Tokyo Institute of Technology.
%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%%  (1) Debugging of probf function
%%%  (2) Debugging of sample function
%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% control routines for original input and consequent output in pronpt :
 
%p_trace(Option,Goal) :-
%    ( ( Option = 'expl' ; Option = 'e' ),!,
%	  e_trace(Goal)
%    ; ( Option = 'sample' ; Option = 's' ),!,
%      s_trace(Goal) 
%    ; format("Usage: p_trace(Options,Goal) ~n",[]),
%	  format("where Options are: ~n",[]),
%	  format("    expl[e]          explanation trace mode ~n",[]),
%	  format("    sample[s]        sampling trace mode ~n",[]) ).

% trace mode of probf function
e_trace :-
    format("{PRISM e_trace on.}~n",[]),
    format("{E_TRACE} ?- ",[]),
    read(X),
    !,e_trace(X).

e_trace(Goal) :- 
    init,
    ( make_prob_expl_body(Goal,Expl_Goal),!,
      ( trace_process(1,0,1,Expl_Goal),
        output_current_expl,          % output current switches
        push_all_expl,                % store current switches
        request_call                  % get user request for other switches
      ; output_all_expl(Goal) )       % output all switches
    ; trace_process(1,0,1,Goal) ).
                                          % if Goal is not prob pred
% trace mode of sample function
s_trace :-
    format("{PRISM s_trace on.}~n",[]),
    format("{S_TRACE} ?- ",[]),
    read(X),
    !,s_trace(X).
    
s_trace(Goal) :- 
    init,
    make_prob_sample_body(Goal,Sample_Goal),!,
    ( trace_process(1,0,1,Sample_Goal),
      output_sampled
    ; output_sampled,!,fail ).

% initiation setting 
init :- init_atom_id,
    init_choice_point,
    init_skip_point,
    init_spy_point,
    init_current_expl,
    init_all_expl,
    init_sample,
    init_sampled,!.

% change original name to system prob name for trace process :

% make_prob_expl_body changes Goal to expl pred
% [NOTE] make_prob_expl_body/7 is borrowed from make_routines
make_prob_expl_body(Goal,Expl_Goal) :-
    make_prob_expl_body((s0,_),tail,_,_,Goal,Expl_Goal),
	( Expl_Goal = (Goal,_),!,fail
    ; true ).

% make_prob_sample_body changes Goal to sample pred
make_prob_sample_body(msw(G_id,N,R),Sample_Goal) :-
    Sample_Goal = sample_msw(G_id,N,R),!.
make_prob_sample_body(msw(G_id,R),Sample_Goal) :-
    Sample_Goal = sample_msw(G_id,nil,R),!.
make_prob_sample_body(bsw(G_id,N,R),Sample_Goal) :-
    Sample_Goal = sample_msw(G_id,N,R),!.
make_prob_sample_body(bsw(G_id,R),Sample_Goal) :-
    Sample_Goal = sample_msw(G_id,nil,R),!.
make_prob_sample_body(Goal,Sample_Goal) :-
    functor(Goal,GF,GArgN),
    ( '*Prob_Pred*'(GF,GArgN,_,Sample_GF),!,
      Goal =.. [GF|GArgs],
      Sample_Goal =.. [Sample_GF|GArgs]
    ; format("{PRISM ERROR: s_trace(~w) -- predicate ~w/~w must be probabilistic.}",[Goal,GF,GArgN]),!,abort ).       
					  % Goal is not prob pred

% main trace process routines :
% trace_process(ID,ParentID,Depth,Goal)

trace_process(I,_,N,true) :-
    !,
    call_msg(I,N,true),
    exit_msg(I,N,true).

trace_process(_,_,_,system_true):- !.
trace_process(_,_,_,'*body_true*'):- !.

trace_process(I,_,N,fail) :-
    !,
    call_msg(I,N,fail),
    fail_msg(I,N,fail),!,fail.

trace_process(I,P,N,bagof(X,A,Xs)) :-
    !,
    call_msg(I,N,bagof(X,A,Xs)),
    ( get_atom_id(I1),
      bagof(X,trace_process(I1,P,N,A),Xs),
      exit_msg(I,N,bagof(X,A,Xs))
    ; fail_msg(I,N,bagof(X,A,Xs)),!,fail ).

trace_process(I,P,N,!) :- 
    !,
    call_msg(I,N,!),
    exit_msg(I,N,!),
    reduce_choice_point(P),!.

trace_process(I,P,N,setof(X,A^B,Xs)) :-
    !,
    call_msg(I,N,setof(X,A^B,Xs)),
    ( get_atom_id(I1),
      setof(X,A^(trace_process(I1,P,N,B)),Xs),
      exit_msg(I,N,setof(X,A^B,Xs))
    ; fail_msg(I,N,setof(X,A^B,Xs)),!,fail ).

trace_process(I,P,N,setof(X,A,Xs)) :-
    !,
    call_msg(I,N,setof(X,A,Xs)),
    ( get_atom_id(I1),
      setof(X,trace_process(I1,P,N,A),Xs),
      exit_msg(I,N,setof(X,A,Xs))
    ; fail_msg(I,N,setof(X,A,Xs)),!,fail ).

trace_process(I,P,N,findall(X,A,Xs)) :-
    !,
    call_msg(I,N,findall(X,A,Xs)),
    ( get_atom_id(I1),
      findall(X,trace_process(I1,P,N,A),Xs),
      exit_msg(I,N,findall(X,A,Xs))
    ; fail_msg(I,N,findall(X,A,Xs)),!,fail ).

trace_process(_,P,N,(A,B)) :- 
    !,
    get_atom_id(I1),
    trace_process(I1,P,N,A),
    get_atom_id(I2),
    trace_process(I2,P,N,B).

trace_process(_,P,N,(A;B)) :- 
    !,
    % [NOTE] We must push/pop symbols which are not IDs. The ID of (A;B)
    %        must be ignored while reducing the choice-point stack, so
    %        meaningless atom '*or*' is used.
    ( ( push_choice_point('*or*')          % push choice point to stack
      ; \+on_top('*or*'),                  % if '*or*' is already removed
        !,fail                             %    from the stack, kill (A;B).
      ; pop_choice_point('*or*'),fail ),   % backtrack in general  
      get_atom_id(I1),
      trace_process(I1,P,N,A) 
    ; get_atom_id(I2),trace_process(I2,P,N,B) ).

% \+(A) is defined as (A,!,fail ; true), so we expand other definitions
% of trace_process/4.

trace_process(I,_,N,\+(A)):-!,
    ( call_msg(I,N,\+(A)),
      % [NOTE] Push/pop the ID of itself.
      ( push_choice_point(I)
      ; on_top(I),
        pop_choice_point(I),fail ),
      N1 is N+1,
      get_atom_id(I1),
      trace_process(I1,I,N1,A),
      % kill itself since it has a cut in the body..
      %   [NOTE] We remove both of I and (I,*) from the choice-point stack.
      reduce_choice_point_all(I), 
      fail_msg(I,N,\+(A)),
	  !,
      fail
    ; exit_msg(I,N,\+(A)) ).      

%%
%% Specialized for PRISM programs:
%%

% case of switch at "e_trace mode"
trace_process(I,_,N,Expl_SW) :-
    Expl_SW =.. ['*Expl_SW*',G,T,V,X,Y],!,
    '*Values*'(G,Values),!,
    '*member*'(V,Values),
    trace_process(I,_,N,'*expl_sw*'(G,T,V,X,Y)).

trace_process(I,_,N,Expl_SW) :-
    functor(Expl_SW,'*expl_sw*',_),!,
    make_msw(Expl_SW,Msw_SW),             % change name to 'msw' for output
    add_switch_to_expl(Msw_SW,Prob),
    expl_msg(I,N,Expl_SW,Msw_SW,Prob).    % message for expl switch

add_switch_to_expl(Msw_SW,Prob):-
    clause('*Current_Expl_List*'(PreSWs,_),true),!,
    ( '*member*'(Msw_SW,PreSWs),!,
      get_prob(Msw_SW,Prob)
    ; get_prob(Msw_SW,Prob),              % get switch probability
      push_current_expl(Msw_SW,Prob) ).   % storing current expl switch

% case of switch at "s_trace mode"
trace_process(I,_,N,Sample_SW) :-
    functor(Sample_SW,sample_msw,3),!,
    make_msw(Sample_SW,Msw_SW),
    sample_call_msg(I,N,Sample_SW,Msw_SW),
    ( sampling(I,Sample_SW),!,        % sampling is true
      sample_exit_msg(I,N,Sample_SW,Msw_SW)
    ; sample_fail_msg(I,N,Sample_SW,Msw_SW),!,fail ).
                                          % sampling is fail
trace_process(I,_,N,A) :- 
    call_msg(I,N,A),
    body_call(I,N,A,B),
    N1 is N+1,
    get_atom_id(I1),
    trace_process(I1,I,N1,B),
    exit_msg(I,N,A).

% call body clauses routines :

% call body clause and operate choice point
body_call(I,N,A,B) :-
    ( body_call(A,B)
    ; !,fail_msg(I,N,A),fail ),
    ( push_choice_point(I)            % First, push choice point to stack.
    ; \+on_top(I),                    % Backtrack after cutting
      ( get_zombi(I),        
        fail_msg(I,N,A)
      ; fail_for_cut_msg(I,N,A) ),
      !,fail
    ; pop_choice_point(I),            % Backtrack normally..
      redo_msg(I,N,A),                % -- pop choice point from stack
      fail ).  

% call a body clause by using function - 'clause' 
body_call(Head,Body):-
    functor(Head,HF,HArgN),
    ( is_user_pred(HF,HArgN),!,
      clause(Head,Body)
    ; call(Head),Body=system_true ).  % atom is system pred

% check whether atom is applicable
% is_user_pred returns 'yes' if applicable
% is_user_pred returns 'no' if not applicable
is_user_pred(Name,ArgN) :-
    ( '*Prob_Pred*'(_,ArgN1,Name,_),ArgN is ArgN1+2    % atom is expl pred
    ; '*Prob_Pred*'(_,ArgN,_,Name)                     % atom is sample pred
    ; '*Sentence*'(_,normal,[Name/ArgN|_]) ).          % atom is non-prob pred
                                          
% operating choice point for implementation of cut action :

init_choice_point :- 
    retractall('*Choice_Point_Stack*'(_)),
    asserta('*Choice_Point_Stack*'([])),!.

% memorize choice point for backtracking
% push choice point - 'Id' to choice point stack
push_choice_point(Id) :-
    retract('*Choice_Point_Stack*'(List)),
    asserta('*Choice_Point_Stack*'([Id|List])),
    % format("Push stack: ~w <- ~w+~w~n",[[Id|List],Id,List]),
    !.

% relieve choice point recapitulated by backtracking
% pop choice point - 'Id' from choice point stack
pop_choice_point(Id) :-
    retract('*Choice_Point_Stack*'([Id|List])),
    asserta('*Choice_Point_Stack*'(List)),
    % format("Pop stack: ~w <- ~w-~w~n",[List,Id,[Id|List]]),
    !.

% on_top(Id) succeeds if Id is on the top of the choice point stack.
% [NOTE] if it fails, we can say that Id is removed by `!'.

on_top(Id) :- clause('*Choice_Point_Stack*'([Id|_]),true).

% If zombi (Id to print) still remains, return the Id and pop;
get_zombi(Id):- 
    clause('*Choice_Point_Stack*'([(Id,*)|_]),true),
    pop_choice_point(_).

% Eliminate choice point in the effective extent of cut action:

reduce_choice_point(Id):-
    % clause('*Choice_Point_Stack*'(List),true),
  	reduce_choice_point0(Id).
    % clause('*Choice_Point_Stack*'(List0),true),
    % format("Reduced stack: ~w <- ~w~n",[List0,List]).
    
reduce_choice_point0(Id) :-
    ( clause('*Choice_Point_Stack*'([(Id,*)|_]),true),!  % do nothing
    ; clause('*Choice_Point_Stack*'([Id|_]),true),!,
      pop_choice_point(_),!,
      push_choice_point((Id,*))
    ; pop_choice_point(_),!,reduce_choice_point0(Id) ),!.

reduce_choice_point_all(Id):-
    % clause('*Choice_Point_Stack*'(List),true),
  	reduce_choice_point_all0(Id).
    % clause('*Choice_Point_Stack*'(List0),true),
    % format("Reduced stack: ~w <- ~w~n",[List0,List]).

reduce_choice_point_all0(Id) :-
    ( clause('*Choice_Point_Stack*'([(Id,*)|_]),true),!,
      pop_choice_point(_)
    ; clause('*Choice_Point_Stack*'([Id|_]),true),!,
      pop_choice_point(_)
    ; pop_choice_point(_),!,reduce_choice_point_all0(Id) ),!.

% message routines :

call_msg(I,N,G) :- trace_msg(I,'CALL',N,G),!.
exit_msg(I,N,G) :- trace_msg(I,'EXIT',N,G),!.
fail_msg(I,N,G) :- trace_msg(I,'FAIL',N,G),!.
redo_msg(I,N,G) :- trace_msg(I,'REDO',N,G),!.

% If you want to message for cut, use following clause:
fail_for_cut_msg(I,N,G):- trace_msg(I,'CUT!',N,G),!.

trace_msg(I,Msg,N,G):-
    ( spy_point_check(G),               % check position of spy action
      skip_point_check(I,Msg),          % check position of skip action
      leash_point_check(Msg),           % check position of leash action
      write_n_tabs(N),
      make_original(G,OG),              % change prob pred to original
      format("~w[~w] ~w",[Msg,N,OG]),
      stand_by(I,Msg,N,G),!             % stand by for user input
    ; true ).                           % no message if spy is still on
                                        % no message if skip is still on
:- asserta('*Indent*'(inact)).

indent_on :- 
	retractall('*Indent*'(_)),
	asserta('*Indent*'(act)),
	format("{Indent on}~n",[]).

indent_off :- 
	retractall('*Indent*'(_)),
	asserta('*Indent*'(inact)),
	format("{Indent off}~n",[]).

write_n_tabs(N) :-
       ( clause('*Indent*'(act),true),!,write_n_tabs0(N)
       ; clause('*Indent*'(inact),true),write('  ') ),!.

write_n_tabs0(0).
write_n_tabs0(N) :- 
      N>0,
      write('  '),
      N1 is N-1,
      write_n_tabs0(N1).

% change system prob name to orignal for output :

make_original(Pred,Original) :-
    ( functor(Pred,PF,_),
      ( '*Prob_Pred*'(_,_,PF,_),!,expl2origin(Pred,Original)
                                     % case of expl pred
      ; '*Prob_Pred*'(_,_,_,PF),!,sample2origin(Pred,Original) )
                                     % case of sample pred
    ; Pred = Original ).           % case of not prob pred

% expl2origin changes from Expl pred to Original for output message. 
expl2origin(Expl,Original) :-
    functor(Expl,EF,EArgN),
    OArgN is EArgN-2,
    '*Prob_Pred*'(OF,OArgN,EF,_),
    Expl =.. [EF|EArgs],
    first_n_elements(OArgN,EArgs,OArgs),
    Original =.. [OF|OArgs],!.

% sample2origin changes from Sample pred to Original for output message.
sample2origin(Sample,Original) :-
    functor(Sample,SF,SArgN),
    '*Prob_Pred*'(OF,SArgN,_,SF),!,
    Sample =.. [SF|SArgs],
    Original =.. [OF|SArgs].

% standing-by action for the input of trace option :
stand_by(I,Msg,N,G) :-
    format(" ? ",[]),    
    get_com(ComCode),                 % wating for user input
    ( ComCode < 0,!,   
      not_apply_msg,nl,
      trace_msg(I,Msg,N,G)            % for unapplicable input
    ; ComCode = 0,!                   % for return action input
    ; code2com(ComCode,Com),          % for applicable input
      ( Com = a,!,abort
      ; Com = h,!,help_msg,trace_msg(I,Msg,N,G)
      ; Com = l,!,leap(I,Msg,N,G)
      ; Com = p,!,output_spy_points_for_option,trace_msg(I,Msg,N,G)
      ; Com = s,!,mark_skip_point(I,Msg,N,G)
      ; Com = x,!,output_current_expl_for_option,trace_msg(I,Msg,N,G)
      ; Com = y,!,output_all_expl_for_option,trace_msg(I,Msg,N,G)
      ; Com = z,!,output_sampled_for_option,trace_msg(I,Msg,N,G)
      ; not_apply_msg,nl,trace_msg(I,Msg,N,G) ) ).

% code2com parses ComCode to Com
code2com(ComCode,Com) :-
    name(a,[Na]),
    ComCode1 is ComCode+Na-1,
    name(Com,[ComCode1]).

% warning message for unapplicable input by user
not_apply_msg :- format("{PRISM: Option not applicable at this port}",[]),nl,!.

% skip action routines :

init_skip_point :- retractall('*Skip_Point*'(_)).

% memorize skip goal for no messaging during skip action
% mark skip point - 'Id' in starting skip action by user
mark_skip_point(I,Msg,N,G) :-
    ( ( Msg = 'CALL' 
      ; Msg = 'REDO' ),     % skip action is on only CALL or REDO
      asserta('*Skip_Point*'(I)),!
    ; not_apply_msg,nl,trace_msg(I,Msg,N,G) ).

% check whether skip is goal point or still on or off 
% skip_point_check returns 'yes' at goal point and during skip off
% skip_point_check returns 'no' during skip still on
skip_point_check(I,Msg) :-
    ( clause('*Skip_Point*'(Id),true),
      ( Id = I,                       % match skip goal
	    ( Msg = 'EXIT' 
        ; Msg = 'FAIL'
	    ; Msg = 'CUT!' ),
        retract('*Skip_Point*'(I)),!
      ; !,fail )                      % skip is still on
    ; true ).                         % skip is off

% spy action routines :

init_spy_point :- spy_point_off.

% in prompt
p_spy([]) :- !.
p_spy([X|Xs]) :- 
    !,
    p_spy(X),p_spy(Xs).

% case of switch
p_spy(msw) :-
    spy_process(msg,msw,sample_msw,3),
    spy_process(nomsg,msw,'*expl_sw*',5).

p_spy(msw,2) :-
    spy_process(msg,msw,sample_msw,3),
    spy_process(nomsg,msw,'*expl_sw*',5).

p_spy(msw,3) :-
    spy_process(msg,msw,sample_msw,3),
    spy_process(nomsg,msw,'*expl_sw*',5).

% in prompt
% change original name to system prob name

p_spy(Name/ArgN) :- p_spy(Name,ArgN).

p_spy(Name) :-
    ( '*Prob_Pred*'(Name,_,Expl,Sample),
      spy_process(msg,Name,Sample),      % case of sample pred
      spy_process(nomsg,Name,Expl)       % case of expl pred
    ; spy_process(msg,Name,Name) ).          % others
	
p_spy(Name,ArgN) :-
    ( '*Prob_Pred*'(Name,ArgN,Expl,Sample),
      spy_process(msg,Name,Sample,ArgN),
      ArgN1 is ArgN+2,
      spy_process(nomsg,Name,Expl,ArgN1)
    ; spy_process(msg,Name,Name,ArgN) ).
	

% memorize function name and arg of spy atm
% spy_process asserts name and arg of spy atm and start spy action
% spy_point shows Name,ArgN and state
spy_process(Msg,Name,NewName) :-
    ( is_user_or_sw_pred(NewName,ArgN),
      spy_process(Msg,Name,NewName,ArgN),
      fail
    ; clause('*Spy_Point*'(Name,_,_,_),true),!
    ; ( Msg = nomsg,!
      ; format("{p_spy: ~w/_ does not exist}~n",[Name]) ) ).

spy_process(Msg,Name,NewName,ArgN) :-
    ( is_user_or_sw_pred(NewName,ArgN),!,
      ( clause('*Spy_Point*'(Name,NewName,ArgN,_),true),
	    ( Msg = nomsg,!
	    ; format("{p_spy: There is already a spypoint on ~w/~w}~n",
                 [Name,ArgN]),! )
        ; asserta('*Spy_Point*'(Name,NewName,ArgN,off)),
	      ( Msg = nomsg,!
	      ; format("{p_spy: Spypoint placed on ~w/~w}~n",[Name,ArgN]),! ) )
    ;   ( Msg = nomsg,!
        ; format("{p_spy: ~w/~w does not exist}~n",
                 [Name,ArgN]) ) ).

% Check whether atm is applicable
% is_user_sw_pred returns 'yes' if applicable
% is_user_sw_pred returns 'no' if not applicable
is_sw_pred('*expl_sw*',5) :- !.
is_sw_pred('sample_msw',3) :- !.

is_user_or_sw_pred(Name,ArgN) :-
    ( is_sw_pred(Name,ArgN); is_user_pred(Name,ArgN) ).

% In prompt,
% remove function name and arg of spy atm
% nop_spy retract name and arg of spy atm 

p_nospy([]) :- !.
p_nospy([X|Xs]) :- 
	!,
	p_nospy(X),p_nospy(Xs).


% case of switch
p_nospy(msw) :-
    nospy_process(msg,msw,sample_msw,3),
    nospy_process(nomsg,msw,'*expl_sw*',5).

p_nospy(msw,2) :-
    nospy_process(msg,msw,sample_msw,3),
    nospy_process(nomsg,msw,'*expl_sw*',5).

p_nospy(msw,3) :-
    nospy_process(msg,msw,sample_msw,3),
    nospy_process(nomsg,msw,'*expl_sw*',5).

p_nospy(Name/ArgN) :- p_nospy(Name,ArgN).

p_nospy(Name) :-
    ( '*Prob_Pred*'(Name,_,Expl,Sample),!,
      nospy_process(msg,Name,Sample),
      nospy_process(nomsg,Name,Expl)
    ; nospy_process(msg,Name,Name) ).

p_nospy(Name,ArgN) :-
    ( '*Prob_Pred*'(Name,ArgN,Expl,Sample),!,
      nospy_process(msg,Name,Sample,ArgN),
      ArgN1 is ArgN+2,
      nospy_process(nomsg,Name,Expl,ArgN1)
    ; nospy_process(msg,Name,Name,ArgN) ).

p_nospy :- 
    retractall('*Spy_Point*'(_,_,_,_)),
    format("{p_nospy: All spypoint removed}~n",[]),!.

% remove function name and arg of spy atm
% nop_spy retract name and arg of spy atm 
nospy_process(Msg,Name,NewName) :-
    ( is_user_or_sw_pred(NewName,_),!,
      ( is_user_or_sw_pred(NewName,ArgN),
        nospy_process(Msg,Name,NewName,ArgN),
        fail
      ; true )
    ; ( Msg = nomsg,!
      ; format("{p_nospy: ~w/_ does not exist}~n",[Name]) )).

nospy_process(Msg,Name,NewName,ArgN) :-
    ( is_user_or_sw_pred(NewName,ArgN),!,
      ( retract('*Spy_Point*'(Name,NewName,ArgN,_)),
        ( Msg = nomsg,!
        ; format("{p_nospy: Spypoint removed from ~w/~w}~n",[Name,ArgN]),! )
      ; ( Msg = nomsg,!
        ; format("{p_nospy: There is no spypoint on ~w/~w}~n",[Name,ArgN]),! ) )
    ; ( Msg = nomsg,!
      ; format("{nospy: ~w/~w does not exist}~n",
               [Name,ArgN]) ) ).

% check whether spy is goal point
% spy_point_check returns 'yes' at goal point and during spy off
% spy_point_check returns 'no' during spy still on
spy_point_check(Goal) :-
    functor(Goal,GF,GArgN),
    ( clause('*Spy_Point*'(_,Name,ArgN,on),true),
      Name = GF,                      % match spy goal
      ArgN = GArgN,
      spy_point_off,
      retractall('*Skip_Point*'(_)),!              
			      % if skip action is on, action is terminated
    ; clause('*Spy_Point*'(_,_,_,on),true),
      !,
      fail                    % spy is still on
    ; true ).                 % spy is off

% at spy goal point 
% spy action is off till leap action by user again
% spy_point_off make state of spy_point off
spy_point_off :- 
    ( clause('*Spy_Point*'(_,Name,ArgN,on),true),
      spy_point_off(Name,ArgN),
      fail
    ; true ).

spy_point_off(Name,ArgN) :-
    retract('*Spy_Point*'(_,Name,ArgN,on)),
    asserta('*Spy_Point*'(_,Name,ArgN,off)),!.

spy_point_on :-
    ( clause('*Spy_Point*'(_,Name,ArgN,off),true),
      spy_point_on(Name,ArgN),
      fail
    ; true ).

spy_point_on(Name,ArgN) :-
    retract('*Spy_Point*'(_,Name,ArgN,off)),
    asserta('*Spy_Point*'(_,Name,ArgN,on)),!.

% leap action at standing by action
%spy_point_on make state of all spy_point on
leap(I,Msg,N,G):- 
    ( clause('*Spy_Point*'(_,_,_,_),true),
      spy_point_on,!
    ; not_apply_msg,nl,trace_msg(I,Msg,N,G) ).

% Alias as system predicate:
show_p_spy:- output_spy_points_for_option.

% output all spy points for option
output_spy_points_for_option :-
    nl,
    format(" [SPY POINTS]~n",[]),
    ( clause('*Spy_Point*'(Original,System,ArgN,_),true),
      out_put_spy_points_for_option(Original,System,ArgN),
      fail
    ; nl,! ).

out_put_spy_points_for_option(Original,System,ArgN) :-
    ( '*Prob_Pred*'(_,_,System,_)                  % atom is expl pred
    ; format("   ~w/~w ~n",[Original,ArgN]) ),!.


% leash and indent routine
% example, 
% by p_leash([call,fail]) tracer will be stoppoing only at [call,fail] ports

% init 
:- asserta('*Leash_Point*'('CALL')),
   asserta('*Leash_Point*'('REDO')),
   asserta('*Leash_Point*'('EXIT')),
   asserta('*Leash_Point*'('FAIL')).

p_leash_all :- p_leash([call,exit,fail,redo]).

p_leash(Xs) :- 
	retractall('*Leash_Point*'(_)),
	leash_process(Xs),!.

% add leash point
p_leash_plus(Xs) :-
	leash_process(Xs),!.

leash_process([]) :- !.
leash_process([X|Xs]) :-
	( is_msg(X,X1),
	  asserta('*Leash_Point*'(X1)),
          format("{PRISM: The debugger will leash stopping at [~w] port}~n",
                 [X1])
        ; format("{PRISM: The debugger can't leash at [~w] port}~n",[X]) ),
	leash_process(Xs),!.

is_msg(X,X1) :-
    ( ( X = c ; X = 'C' ; X = call ; X = 'Call' ; X = 'CALL' ),!,
	X1 = 'CALL'
    ; ( X = r ; X = 'R' ; X = redo ; X = 'Redo' ; X = 'REDO' ),!,
	X1 = 'REDO'
    ; ( X = e ; X = 'E' ; X = exit ; X = 'Exit' ; X = 'EXIT' ),!,
	X1 = 'EXIT'
    ; ( X = f ; X = 'F' ; X = fail ; X = 'Fail' ; X = 'FAIL' ),!,
	X1 = 'FAIL' 
    ; ( X = '!' ; X = cut  ; X = 'Cut'  ; X = 'CUT'  ),!,
	X1 = 'CUT!' ),!.

% check whether Msg is leash point
% if Msg is leash point, returns yes
% otherwise returns no
leash_point_check(Msg) :-
	clause('*Leash_Point*'(Msg),true).


% operating switches action routines :

% change system switch name to 'msw' for output and store
%make_msw(System,Msw_SW) :-
%    ( functor(System,'*expl_sw*',_)
%    ; functor(System,sample_msw,3) ),!,
%    System =.. [_|Args],
%    Msw_SW =.. [msw|Args].	

make_msw(System,Msw_SW) :-
    System =.. [F|Args],
    ( F = '*expl_sw*',!,first_n_elements(3,Args,SwArgs)
    ; F = sample_msw, SwArgs = Args ),!,
    Msw_SW =.. [msw|SwArgs].

%%% getting probability of switch
% [NOTE] If SW does not exist in the switch database, assertz it!
get_prob(SW,Prob) :-
    arg(1,SW,SW_name),
    arg(3,SW,SW_value),
    ( clause('*Switch*'(SW_name,_,_,_,_,undef),true),!,Prob = undef
    ; clause('*Switch*'(SW_name,_,_,_,Values,Probs),true),!,
      get_prob(SW_name,SW_value,Values,Prob,Probs)
    ; msw_values(SW_name,Size,Values),       % if SW does not exist..
      unique_code(Code),!,
      assertz('*Switch*'(SW_name,Code,unfixed,Size,Values,undef)),!,
	  Prob = undef ).

get_prob(_,SW_value,[SW_value|_],Prob,[Prob|_]).
get_prob(SW_name,SW_value,[_|Values],Prob,[_|Probs]) :-
    get_prob(SW_name,SW_value,Values,Prob,Probs).

% storing switch in current flow
init_current_expl :-
    retractall('*Current_Expl_List*'(_,_)),
    asserta('*Current_Expl_List*'([],undef)).

push_current_expl(New_SW,Prob) :-
    null_trial_check(New_SW,New_SW1),
    clause('*Current_Expl_List*'(Pre_SWs,Pre_Probs),true),
    New_SWs = [New_SW1|Pre_SWs],
    ( Pre_Probs = 'undef',!,
      New_Probs = Prob
    ; New_Probs is Prob * Pre_Probs ),
    ( retract('*Current_Expl_List*'(Pre_SWs,Pre_Probs)),
      asserta('*Current_Expl_List*'(New_SWs,New_Probs))  % push switches
    ; retract('*Current_Expl_List*'(New_SWs,New_Probs)),
      asserta('*Current_Expl_List*'(Pre_SWs,Pre_Probs)),!,fail ).
                                          % pop switches for backtrack
% case of e_trace
% number of switch arg is 2
null_trial_check(SW1,SW2) :-
	( arg(2,SW1,null),!,
      SW1 =.. [_,S1,null,S3],
      SW2 =.. [msw,S1,S3]
    ; SW1 = SW2 ).

%null_trial_check(SW1,SW2) :- SW1 = SW2.

% output current switch list
% for result
output_current_expl :-
    clause('*Current_Expl_List*'(SWs,Prob),true),
    nl,
    format("Conjunction of switches in current path:~n",[]),
	'*reverse*'(SWs,RevSWs),
    print_probf('  ',RevSWs),
    ( Prob = undef,!,format("~nThe probability: undef~n~n",[])
    ; format("~nThe probability: ~6f~n~n",[Prob]) ).

% for option
output_current_expl_for_option :-
    clause('*Current_Expl_List*'(SWs,Prob),true),
    format("~n [CURRENT SWITCHES]~n",[]),
	'*reverse*'(SWs,RevSWs),
    print_probf('    ',RevSWs),
    format(" [PROBABILITY]~n",[]),
    ( Prob = undef,!,format("    undef~n~n",[])
    ; format("    ~6f~n~n",[Prob]) ).

% storing all switches in flow till now
init_all_expl :-
    retractall('*All_Expl_List*'(_,_)),
    asserta('*All_Expl_List*'([],undef)).

push_all_expl :-
    clause('*Current_Expl_List*'(New_SW,Prob),true),
    clause('*All_Expl_List*'(Pre_SWs,Pre_Probs),true),
    New_SWs = [New_SW|Pre_SWs],
    ( Pre_Probs = 'undef',!,
      New_Probs = Prob
    ; New_Probs is Prob + Pre_Probs ),
    retract('*All_Expl_List*'(Pre_SWs,Pre_Probs)),
    asserta('*All_Expl_List*'(New_SWs,New_Probs)),!.
                                          % push switches
% output all switch list
% for result
output_all_expl(Goal) :-
    clause('*All_Expl_List*'(AllExpls,Prob),true),
    format("~n----------------~n",[]),
    print_term(0,Goal),nl,!,
    '*reverse*'(AllExpls,RevAllExpls),!,
    sub_output_all_expl(RevAllExpls),
    ( Prob = undef,!,format("~nThe probability: undef~n~n",[])
    ; format("~nThe probability: ~6f~n~n",[Prob]) ).

sub_output_all_expl([]):- !,format("is false with probablity one.~n",[]).
sub_output_all_expl(Expls):-
    format("is explained by~n",[]),!,
    sub_output_all_expl(0,Expls).

% print_probf/1 is borrowed from exec.pl.
sub_output_all_expl(N,[Expl|Expls]) :-
    '*reverse*'(Expl,RevExpl),
    ( N>0,!,print_probf('v ',RevExpl) ; print_probf('  ',RevExpl) ),
    N1 is N+1,!,
    sub_output_all_expl(N1,Expls).
sub_output_all_expl(_,[]).

% output for option
output_all_expl_for_option :- 
    clause('*All_Expl_List*'(Expls,Prob),true),
    nl,	
    format(" [ALL SWITCHES]~n",[]),
    sub_output_all_expl_for_option(Expls),
    format(" [PROBABILITY]~n",[]),
    ( Prob = undef,!,format("    undef~n~n",[])
    ; format("    ~6f~n~n",[Prob]) ).

sub_output_all_expl_for_option(Expls):- sub_output_all_expl(0,Expls).

% getting request for other switches
% request_call returns 'no' in case of default and input 'y'
% request_call returns 'yes' in case of input 'n'
request_call :-
    format("Do you want another explanation? (y/n default:y) ",[]),
    get_com(ComCode),                 % wating for user input
    ( ComCode < 0,!,   
      request_call                    % for unapplicable input
    ; ComCode = 0,!,nl,fail           % for return action input
    ; code2com(ComCode,Com),          % for applicable input
      ( Com = y,!,nl,fail 
      ; Com = n,! 
      ; request_call ) ).

% sampling routine :

init_sample :- 
    retractall('*Sampled_MSW*'(_,_,_)).

sampling(I,Sample_SW) :-
   Sample_SW =..[_,S,N,V],       % if it was already sampled..
   ( clause('*Sampled_MSW*'(S,N,V1),true),!,
     format("~n{msw(~w,~w,_) already sampled.}~n",[S,N]),
     Msw_SW =..[msw,S,N,V1],
     get_prob(Msw_SW,Prob),     
     sampled_msg(I,Msw_SW,Prob),nl,
     ( V = V1,! 
      ; !,fail )                       % if sampling is done first time..

   ; format("~n{Sampling msw(~w,~w,_)...}",[S,N]),
     ( spy_point_check(temp),
       skip_point_check(I,undef),!,get_com(_)
     ; true ),
     ( call(Sample_SW),!,             % succeed to sample..
       Msw_SW =.. [msw,S,N,V],
       get_prob(Msw_SW,Prob),
	   push_sampled(Msw_SW,Prob),
	   sampled_msg(I,Msw_SW,Prob),nl  % fail to sample..
     ; clause('*Sampled_MSW*'(S,N,V1),true),!,
       Msw_SW =..[msw,S,N,V1],
       get_prob(Msw_SW,Prob),
       push_sampled(Msw_SW,Prob),
       sampled_msg(I,Msw_SW,Prob),nl,!,
       fail ) ).

% storing result of sampled switch
init_sampled :- 
	retractall('*Sampled_List*'(_,_)),
    asserta('*Sampled_SW_List*'([],undef)).

push_sampled(New_SW,Prob) :-
    arg_check2(New_SW,New_SW1),
    clause('*Sampled_SW_List*'(Pre_SWs,Pre_Probs),true),
    New_SWs = [New_SW1|Pre_SWs],
    ( Pre_Probs = 'undef',!,
      New_Probs = Prob
    ; New_Probs is Prob * Pre_Probs ),
    retract('*Sampled_SW_List*'(Pre_SWs,Pre_Probs)),
    asserta('*Sampled_SW_List*'(New_SWs,New_Probs)),!.  % push switches
                                          
% case of s_trace
% number of switch arg is 2
arg_check2(SW1,SW2) :-
    ( arg(2,SW1,'nil'),!,
      functor(SW2,'msw',2),
      arg(1,SW1,S1),arg(1,SW2,S1),
      arg(3,SW1,S3),arg(2,SW2,S3)
    ; SW1 = SW2 ).

output_sampled :-
    clause('*Sampled_SW_List*'(SWs,Prob),true),
    nl,
    format("----------------~n",[]),
    format("Conjunction of sampled switches:~n",[]),
	'*reverse*'(SWs,RevSWs),
    print_probf('  ',RevSWs),
    ( Prob = undef,!,format("~nThe probability: undef~n~n",[])
    ; format("~nThe probability: ~6f~n~n",[Prob]) ).

output_sampled_for_option :-
    clause('*Sampled_SW_List*'(SWs,Prob),true),
    nl,
    format(" [SAMPLING]~n",[]),
	'*reverse*'(SWs,RevSWs),
    print_probf('    ',RevSWs),
    format(" [PROBABILITY]~n",[]),
    ( Prob = undef,!,format("    undef~n",[])
    ; format("    ~6f~n",[Prob]) ),nl.

% message routine for switches
expl_msg(I,N,Expl,Msw,Prob) :-
    null_trial_check(Msw,Msw1),
    write_n_tabs(N),
    ( Prob = undef,!,format("SWITCH[~w] ~w: undef ? ",[N,Msw1])
    ; format("SWITCH[~w] ~w: ~6f ? ",[N,Msw1,Prob]) ),
    ( spy_point_check(Expl),
      skip_point_check(I,undef),!,
      get_com(ComCode),
      ( ComCode = 0,!
      ; code2com(ComCode,Com),          % for applicable input
        ( Com = a,!,abort
        ; Com = h,!,help_msg,expl_msg(I,N,Expl,Msw1,Prob)
        ; Com = l,!,
          ( clause('*Spy_Point*'(_,_,_,_),true),
	        spy_point_on,!
          ; not_apply_msg,nl,expl_msg(I,N,Expl,Msw1,Prob) )
        ; Com = p,!,
	      output_current_expl_for_option,expl_msg(I,N,Expl,Msw1,Prob)
        ; Com = q,!,
          output_all_expl_for_option,expl_msg(I,N,Expl,Msw1,Prob)
        ; Com = r,!,
          output_sampled_for_option,expl_msg(I,N,Expl,Msw1,Prob)
        ; not_apply_msg,nl,
          expl_msg(I,N,Expl,Msw1,Prob) ) )
        ; nl ).

sample_call_msg(I,N,Sample,Msw) :- switch_msg(I,'CALL',N,Sample,Msw).
sample_exit_msg(I,N,Sample,Msw) :- switch_msg(I,'EXIT',N,Sample,Msw).
sample_fail_msg(I,N,Sample,Msw) :- switch_msg(I,'FAIL',N,Sample,Msw).

switch_msg(I,Msg,N,Sample,Msw) :-
    arg_check2(Msw,Msw1),
    write_n_tabs(N),
    format("SAMPLING ~w[~w] ~w ? ",[Msg,N,Msw1]), 
    ( spy_point_check(Sample),
      skip_point_check(I,undef),!,
      get_com(ComCode),
      ( ComCode = 0,!
      ; code2com(ComCode,Com),          % for applicable input
        ( Com = a,!,abort
        ; Com = h,!,help_msg,switch_msg(I,Msg,N,Sample,Msw)
        ; Com = l,!,
          ( clause('*Spy_Point*'(_,_,_,_),true),
            spy_point_on,!
          ; not_apply_msg,nl,switch_msg(I,Msg,N,Sample,Msw) )
        ; Com = p,!,
          output_current_expl_for_option,switch_msg(I,Msg,N,Sample,Msw)
        ; Com = q,!,
          output_all_expl_for_option,switch_msg(I,Msg,N,Sample,Msw)
        ; Com = r,!,
          output_sampled_for_option,switch_msg(I,Msg,N,Sample,Msw)
        ; not_apply_msg,nl,
          switch_msg(I,Msg,N,Sample,Msw) ) )
        ; nl ).

sampled_msg(I,Msw,Prob) :-
    arg_check2(Msw,Msw1),
    ( Prob = undef,!,format("Sampled switch: ~w   undef",[Msw1])
    ; format("Sampled switch: ~w   ~6f",[Msw1,Prob]) ),
    ( spy_point_check(Msw),
      skip_point_check(I,undef),!,get_com(_) 
    ; nl ).
