%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%%  sw_handling.pl: miscellaneous commands with respect to switches.
%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%  Copyright (C) 1998
%%%    Taisuke Sato, Yoshitaka Kameya, Yasushi Hagiwara, Nobuhisa Ueda,
%%%      Dept. of Computer Science, Tokyo Institute of Technology.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%% bsw_check/{1,2}, msw_check/3, sw_pbplus_check/6, msw_values/3:
%%% 
%%% [NOTE]
%%% (1) G_id must be instanciated in advance.
%%% (2) For example, Suppose that following 2 msw declaration is written
%%%     in PRISM program,
%%%         values(class1(_),[v1,v2,v3,v4]). --- (A)
%%%         values(class1(sw1),[1,0]).       --- (B)
%%%     Then a switch class(sw1) matches (A) before (B), so PRISM system
%%%     judges  switch class(sw1) is multiple,  and  its possible values
%%%     are {v1,v2,v3,v4}.  For that reason,  we must have matching test
%%%     of G_id  in advance of matching test of Values.

bsw_check(G_id) :-
	'*Sentence*'(_,msw,[G_id,_,Values]),!,Values=[1,0],!.

bsw_check(G_id,V) :-
	'*Sentence*'(_,msw,[G_id,_,Values]),!, 
	Values=[1,0],(V=1;V=0),!.

%%% msw_values(G_id,Size,Values):
%%%     answers switch G_id's possible values,
%%%     and fails if switch G_id doesn't exist.

msw_values(G_id,Size,Values) :-
	( '*Switch*'(G_id,_,_,Size,Values,_),!
	; '*Sentence*'(_,msw,[G_id,Size,Values]),!
	; message("{PRISM ERROR: No such switch `~w'.}",[G_id]),!,fail ),!.

%%% msw_check(G_id,V,N):
%%%   Code is code No. of G_id, and V is switch G_id's Nth value.
%%% [NOTE] fails if V is not switch G_id's value.

msw_check(G_id,V,N) :-
	msw_values(G_id,_,Values),!,nth_member(V,Values,N),!.

%%% sw_pbplus_check(G_id,PbPlus,Size,Values,Pbs,Pred): answers 
%%% if switch G_id can have PbPlus as parameters.
%%% - Is No. of parameters in PbPlus equal to No. of possible values
%%%   of switch G_id?
%%% - Does PbPlus sum to 1.0?
%%% In addition, msw_pbplus_check returns Pbs (list version of PbPlus).
%%% and Size and Values of switch G_id.

% modified by hagi on Oct/1/1997,Nov/20/1997,Feb/24/1998.
sw_pbplus_check(G_id,PbPlus,Size,Values,Pbs,Pred) :-
	msw_values(G_id,Size,Values),!,
	( ( plus_to_list_w_sum(PbPlus,Pbs,Sum),!
	  ; sum_list(PbPlus,Sum),Pbs=PbPlus),
	  ( abs(1-Sum)<0.000001,!,
	    ( length(Pbs,Size),!
	    ; message("{PRISM ERROR: ~w(~w,~w) -- Switch `~w' must have ~w possible values.}",
		  [Pred,G_id,PbPlus,G_id,Size]),!,fail)
	  ; message("{PRISM ERROR: ~w(~w,~w) -- ~w must be the form of Pb1+Pb2+..+PbN or [Pb1,Pb2,..,PbN] which equals 1.0.}",
	          [Pred,G_id,PbPlus,PbPlus]),!,fail)
	; ratio_to_list(PbPlus,Pbs),!,
	  ( length(Pbs,Size),!
	  ; message("{PRISM ERROR: ~w(~w,~w) -- Switch `~w' must have ~w possible values.}",
		  [Pred,G_id,PbPlus,G_id,Size]),!,fail)
	; message("{PRISM ERROR: ~w(~w,~w) -- ~w must be the form of Pb1+Pb2+..+PbN which equals 1.0 or R1:R2:..:RN, and each Pbi or Ri must be a non-negative number.}",
	          [Pred,G_id,PbPlus,PbPlus]),!,fail),!.

%%% make_fixed/1: realizes fixed-parameter decl.
% Suppose that msw declaration values(i,[v1,v2,v3]) is written in PRISM
% program.    Then   fixed-parameter   declaration   is   the  form  of 
% fixed([i=0.5+0.2+0.3,..]), etc.

make_fixed :-
	( setof((N,GPs),'*Sentence*'(N,fixed,GPs),NGPs),!,
	  make_fixed(NGPs)
	; true ),!.

make_fixed([(_,GPs)|NGPs]) :-
	( fix_sw(GPs),!
	; ( GPs=[G_id=PbOrPlus],!,
	    message("{PRISM ERROR: fixed(~w,~w) or fixed(~w) -- Illegal fixed-parameter declaration.}",[G_id,PbOrPlus,GPs]),!,
	    fail
	  ; message("{PRISM ERROR: fixed(~w) -- Illegal fixed-parameter declaration.}",[GPs]),!,
	    fail ) ),!,
	make_fixed(NGPs).
make_fixed([]) :- !.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%                                                             %%%
%%%               Miscellaneous User's commands                 %%% 
%%%                                                             %%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%% fix_{bsw,msw,sw} modified by hagi on Feb/24/1998.
%%% fix_bsw/{1,2}: fixes parameters of BSW by query.
%%%            --> also available in utility program.
% <ex.>
% | ?- fix_bsw([i=0.5,j=0.7]).
% {Fix switch `i' <-- 0.5+0.5}
% {Fix switch `j' <-- 0.7+0.3}
% modified by hagi on Nov/22/1997

fix_bsw([G_id=Pb|GPs]) :-!,fix_sw(G_id,Pb,fix_bsw),!,fix_bsw(GPs).
fix_bsw([G_id|GPs]) :- !,fix_bsw(G_id),!,fix_bsw(GPs).

fix_bsw([]) :- !.

fix_bsw(G_id) :- fix_sw(G_id,0.5+0.5,fix_bsw),!.
fix_bsw(G_id,Pb) :- fix_sw(G_id,Pb,fix_bsw),!.

%%% fix_msw/{1,2}: fixes paramters of MSW by query.
%%%            --> also available in utility program.
% <ex.>
% | ?- fix_msw([i=0.5+0.3+0.2,j=6:1:2:1,k]).
% {Fix switch `i' <-- 0.5+0.3+0.2}
% {Fix switch `j' <-- 0.6+0.1+0.2+0.1}
% {Fix switch `k' <-- 0.2+0.2+0.2+0.2+0.2}
%
% where No. of possible values of switch i,j,k are 3,4,5 respectively.
% modified by hagi on Nov/22/1997

fix_msw([G_id=PbPlus|GPs]) :- !,fix_sw(G_id,PbPlus,fix_msw),!,fix_msw(GPs).
fix_msw([G_id|GPs]) :- !,fix_msw(G_id),!,fix_msw(GPs).

fix_msw([]) :- !.

fix_msw(G_id) :-
	make_uniform_ratio(G_id,PbRatio),!,
	fix_sw(G_id,PbRatio,fix_msw),!.

fix_msw(G_id,PbPlus) :- fix_sw(G_id,PbPlus,fix_msw).

%%% fix_sw/{1,2}: mixture of fix_msw/{1,2} and fix_bsw/{1,2}.
%%%
% modified by hagi on Nov/22/1997
fix_sw([G_id=PbOrPlus|GPs]) :- !,fix_sw(G_id,PbOrPlus,fix_sw),!,fix_sw(GPs).
fix_sw([G_id|GPs]) :- !,fix_sw(G_id),!,fix_sw(GPs).

fix_sw([]) :- !.

fix_sw(G_id) :-
	make_uniform_ratio(G_id,PbRatio),!,
	fix_sw(G_id,PbRatio,fix_sw),!.

fix_sw(G_id,PbOrPlus) :- fix_sw(G_id,PbOrPlus,fix_sw).

fix_sw(G_id,PbOrPlus,Pred) :-
	( var_check(G_id,yes),!,
	  message("{PRISM ERROR: ~w(~w,_) -- Switch `~w' must be instanciated.}",
	          [Pred,G_id,G_id]),!,fail
	; var_check(PbOrPlus,yes),!,
	  message("{PRISM ERROR: ~w(_,~w) -- ~w must be instanciated.}",
	          [Pred,PbOrPlus,PbOrPlus]),!,fail
	; sw_pbplus_check(G_id,PbOrPlus,Size,Values,Pbs,Pred),!,
	  fix_sw_w_no_check(G_id,PbOrPlus,Size,Values,Pbs,Pred)).

%%% fix_sw_w_no_check(G_id,PbPlus,Size,Values,Pbs,Pred):
% [NOTE] following condition must be satisfied in advance.
%   - G_id is instanciated.
%   - PbPlus is the form of Pb1+Pb2+..+PbN or [Pb1,Pb2,..,PbN] (equals 1.0) 
%     or R1:R2:..:RN.
%   - Pbs is the form of [Pb1,Pb2,..,PbN] (summed to 1.0).
% modified by hagi on Feb/24/1998.

fix_sw_w_no_check(G_id,PbPlus,Size,Values,Pbs,Pred) :-
	( clause('*Switch*'(G_id,_,fixed,_,_,_),true),!,
	  message("{PRISM WARNING: ~w(~w,~w) ignored -- switch `~w' is already fixed.}",[Pred,G_id,PbPlus,G_id])
	; ( retract('*Switch*'(G_id,Code,unfixed,_,_,_)),!
	  ; unique_code(Code) ),!,
	  assertz('*Switch*'(G_id,Code,fixed,Size,Values,Pbs)),!,
	  format("{Fix switch `~w' <-- ", [G_id]),
	  list_to_plus_w_write(Pbs),!,
	  message('}') ),!.

% unfix_{bsw,msw,sw} modified by hagi on Feb/24/1998.
%%% unfix_bsw/1, unfix_msw/1: unfixes BSWs or MSWs
% [NOTE]
% - Two forms allowed for unfix_bsw(G_ids) --
%   (1) G_ids is the list of G_id -- unfix_bsw([1,2,3]).
%   (2) G_ids is G_id (which must be instanciated) -- unfix_bsw(1).
% - unfix_bsw1/1 is just an alias to unfix_msw1/1, except that
%   unfix_bsw1(G_id) checks if switch G_id is BSW.

unfix_bsw([G_id|G_ids]) :- !,unfix_sw(G_id,unfix_bsw),!,unfix_bsw(G_ids).
unfix_bsw([]) :- !.
unfix_bsw(G_id) :- unfix_sw(G_id,unfix_bsw),!.

unfix_msw([G_id|G_ids]) :- !,unfix_sw(G_id,unfix_msw),!,unfix_msw(G_ids).
unfix_msw([]) :- !.
unfix_msw(G_id) :- unfix_sw(G_id,unfix_msw),!.

%%% unfix_sw/1: mixture of unfix_msw/1 and unfix_bsw/1.
%%%
% unfix_sw/1 created by hagi on Nov/22/1997

unfix_sw([G_id|G_ids]) :- !,unfix_sw(G_id,unfix_sw),unfix_sw(G_ids).
unfix_sw([]) :- !.

unfix_sw(G_id) :- unfix_sw(G_id,unfix_sw).

unfix_sw(G_id,Pred) :-
	( var_check(G_id,yes),!,
	  message("{PRISM ERROR: ~w(~w) -- Switch `~w' must be instanciated.}", [Pred,G_id,G_id]), fail
	; ( clause('*Switch*'(G_id,_,unfixed,_,_,_),true),!,
	    message("{PRISM WARNING: ~w(~w) ignored -- Switch `~w' already unfixed.}",[Pred,G_id,G_id])
	  ; retract('*Switch*'(G_id,Code,fixed,Size,Values,Pbs)),!,
	    assertz('*Switch*'(G_id,Code,unfixed,Size,Values,Pbs)),!,
	    format("{Unfix MSW ~w (",[G_id]),
	    list_to_plus_w_write(Pbs),!,
	    message(')}')
	  ; message("{PRISM ERROR: ~w(~w) -- No such Switch `~w'.}",
		  [Pred,G_id,G_id]),!,fail ) ),!.

%%% set_{bsw,msw,sw} modified by hagi on Feb/24/1998.

%%% set_bsw/1,set_bsw/2: initialize the prob. of MSW
%%% (a) | ?- set_bsw(i,0.6),set_bsw(j,0.3).
%%%     {Set switch `i' <-- 0.6+0.4}
%%%     {Set switch `j' <-- 0.3+0.7}
%%%     yes
%%% (b) | ?- set_bsw([i=0.6,j=0.3]).
%%%     {Set switch `i' <-- 0.6+0.4}
%%%     {Set switch `j' <-- 0.3+0.7}
%%%     yes
% modified by hagi on Nov/22/1997

set_bsw([G_id=Pb|GPs]) :-!,set_sw(G_id,Pb,set_bsw),!,set_bsw(GPs).
set_bsw([G_id|GPs]) :- !,set_bsw(G_id),!,set_bsw(GPs).
set_bsw([]) :- !.  % cut-symbol added by kame on Dec/8/1997.

set_bsw(G_id) :- set_sw(G_id,0.5+0.5,set_bsw),!.
set_bsw(G_id,Pb) :- set_sw(G_id,Pb,set_bsw).

%%% set_msw/1,set_msw/2: initialize the prob. of MSW
%%% (a) | ?- set_msw(i,0.6+0.4),set_msw(j,1:2:1),set_msw(k).
%%%     {Set switch `i' <-- 0.6+0.4}
%%%     {Set switch `j' <-- 0.25+0.5+0.25}
%%%     {Set switch `k' <-- 0.25+0.25+0.25+0.25}
%%%     yes
%%% (b) | ?- set_msw([i=0.6+0.4,j=3:5:2,k]).
%%%     {Set switch `i' <-- 0.6+0.4}
%%%     {Set switch `j' <-- 0.3+0.5+0.2}
%%%     {Set switch `k' <-- 0.25+0.25+0.25+0.25}
%%%     yes
%%% where, No. of possible values of switch i,j,k are 2,3,4, respectively.

% modified by hagi on Nov/22/1997

set_msw([G_id=PbPlus|GPs]) :- !,set_sw(G_id,PbPlus,set_msw),!,set_msw(GPs).
set_msw([G_id|GPs]) :- !,set_msw(G_id),!,set_msw(GPs).

set_msw([]) :- !.

set_msw(G_id) :-
	make_uniform_ratio(G_id,PbRatio),!,
	set_sw(G_id,PbRatio,set_msw),!.

set_msw(G_id,PbPlus) :- set_sw(G_id,PbPlus,set_msw).

%%% set_sw/{1,2}: mixture of set_msw/{1,2} and set_bsw/{1,2}.
%%%
%%% [NOTE] set_sw(G_id,_) -- in error if switch G_id fixed.

% modified by hagi on Nov/22/1997

set_sw([G_id=PbOrPlus|GPs]) :- !,set_sw(G_id,PbOrPlus,set_sw),!,set_sw(GPs).
set_sw([G_id|GPs]) :- !,set_sw(G_id),!,set_sw(GPs).
set_sw([]) :- !.

set_sw(G_id) :-
	make_uniform_ratio(G_id,PbRatio),!,set_sw(G_id,PbRatio,set_sw),!.

set_sw(G_id,PbOrPlus) :- set_sw(G_id,PbOrPlus,set_sw).

set_sw(G_id,PbOrPlus,Pred) :-
	( var_check(G_id,yes),!,
	  message("{PRISM ERROR: ~w(~w,_) -- Switch `~w' must be instanciated.}",
	          [Pred,G_id,G_id]),!,fail
	; var_check(PbOrPlus,yes),!,
	  message("{PRISM ERROR: ~w(_,~w) -- ~w must be instanciated.}",
	          [Pred,PbOrPlus,PbOrPlus]),!,fail
	; sw_pbplus_check(G_id,PbOrPlus,Size,Values,Pbs,Pred),!,
	  set_sw_w_no_check(G_id,PbOrPlus,Size,Values,Pbs,Pred),! ).

%%% set_sw_w_no_check(G_id,PbPlus,Size,Values,Pbs,Pred):
% [NOTE]
%   This must satistify the the same condition as that of
%   fix_sw_w_no_check/6.
% modified by hagi Feb/24/1998.

set_sw_w_no_check(G_id,PbPlus,Size,Values,Pbs,Pred) :-
	( clause('*Switch*'(G_id,_,fixed,_,_,_),true),!,
	  message("{PRISM ERROR: ~w(~w,~w) -- Switch ~w fixed. Unfix this switch in advance by using unfix_sw(~w).}",[Pred,G_id,PbPlus,G_id,G_id]),!,fail
	; ( retract('*Switch*'(G_id,Code,unfixed,_,_,_)),!
	  ; unique_code(Code) ),!,
	  assertz('*Switch*'(G_id,Code,unfixed,Size,Values,Pbs)),!,
	  format("{Set switch `~w' <-- ", [G_id]),
	  list_to_plus_w_write(Pbs),!,
	  message('}') ),!.

%%% show_sw/0: print the probability and status (fixed or unfixed) of
%%%             all the swhitchs.
%%% modified by hagi on Feb/24/1998.

show_sw :- 
	( setof((G_id,Status,Values,Pbs),
	        (Code,Size)^clause('*Switch*'(G_id,Code,Status,
			                              Size,Values,Pbs),true),
			GSVPs),!,
	  show_sw1(5,GSVPs)
	; message('{PRISM ERROR: No switch found. -- Maybe EM learning is not done yet.}'),!,fail ),!.

show_sw([G_id|G_ids]) :- !,
	show_sw0(G_id),!,show_sw(G_ids).
show_sw([]) :- !.

show_sw(G_id) :- show_sw0(G_id).

show_sw0(G_id) :-
	( var(G_id),!,
	  message("{PRISM ERROR: show_sw(~w) -- ~w must be instanciated.}",[G_id,G_id]),!,fail
	; setof((G_id,Status,Values,Pbs),
	        (Code,Size)^clause('*Switch*'(G_id,Code,Status,
		                                      Size,Values,Pbs),true),
			GSVPs),!,
	  show_sw1(5,GSVPs)
	; message("{PRISM ERROR: show_sw(~w) -- No such switch `~w'.}",[G_id,G_id]),!,fail ).


show_sw1(Column,[(G_id,Status,Values,Pbs)|GSVPs]) :-
	show_sw2(Status,Status_w_paren),!,
	format("Switch ~w~w: ",[G_id,Status_w_paren]),!,
	show_sw3(Column,Column,Values,Pbs),!,
	show_sw1(Column,GSVPs).
show_sw1(_,[]) :- !.

show_sw2(Status,Status_w_paren) :-
	( Status = fixed,!,Status_w_paren = ' (fixed)'
	; Status_w_paren = '' ),!.

show_sw3(Column,C,[V|Values],[P|Pbs]) :-
	( Column=C,!,write('   ') 
	; C=0,!,nl,write('   ')
	; write('  ') ),!,
	format("~w (~6f)",[V,P]),!,
	( C=0,!,C1 is Column-1
	; C1 is C-1),!,
	show_sw3(Column,C1,Values,Pbs).
% bug-fixed by kame on Jul/25/1998
show_sw3(Column,C,[V|Values],undef) :-
	( Column=C,!,write('   ')
	; C=0,!,nl,write('   ')
	; write('  ') ),!,
  	format("~w (undef) ",[V]),!,
	( C=0,!,C1 is Column-1
  	; C1 is C-1),!,
	show_sw3(Column,C1,Values,undef).
show_sw3(_,_,[],_) :- !,nl.

%%% get_sw(Sw,Status,Values,Pbs): get information Info of switch Sw.
%%% 
%%%   Switch infomations involves:
%%%    - Status -- current status of switch Sw (`fixed' of `unfixed').
%%%    - Values -- the list of possible values of switch Sw.
%%%    - Pbs -- the list of current parameters of switch Sw.
%%%              (the order is corresponding to Values).
%%%  [NOTE]
%%%  (1) If Sw contains one or more logical variables, the information
%%%      to be returned is that of the first-matched switch. Note that
%%%      god only knows which switch is first-matched.  So if you want
%%%      all switch information, use SICStus built-in predicate setof/3
%%%      or findall/3 in such a way that
%%%
%%%        | ?- setof((foo(A),S,V,P),get_sw(foo(A),S,V,P),All_Infos).
%%%
%%%      By this query, we can get the infomations of all switches each
%%%      of which takes the form of foo(_).
%%%  (2) If Sw doesn't match with any switches, get_sw/4 fails without
%%%      error message.
%%%
%%%  created by kame on Jan/9/1998

get_sw(G_id,Status,Values,Pbs) :-
        ( var(G_id),!,
          message("{PRISM ERROR: get_sw(~w,_) -- ~w must be instanciated.}",
                  [G_id,G_id]),!,
          fail
    ; true ),!,
        clause('*Switch*'(G_id,_,Status,_,Values,Pbs),true).

%% get_sw(Sw,Info): list version of get_sw/2.
%%   - Info is the list in the form of [Status,Values,Pbs].
%%
%%  also added by kame on Jan/9/1998

get_sw(G_id,[Status,Values,Pbs]) :- get_sw(G_id,Status,Values,Pbs).

%%% save_sw/0, save_sw/1: save the prob. of MSW to file
%%% (a) | ?- save_sw(any_file_as_you_like).
%%% (b) | ?- save_sw.
%%% in the case (a), saved to the file 'any_file_as_you_like'
%%% in the case (b), saved to the default file 'saved_MSW'
%%% 
%%% default filename changed by kame on Feb/23/1998
% modified by hagi on Feb/24/1998.
save_sw(File) :-
	( tell(File),!,listing('*Switch*'),!,told,!
	; message("{PRISM ERROR: Cannot write ~w.}",[File]),!,fail),
	message("{Switch information saved in ~w}",[File]),!.
save_sw :- save_sw('Saved_SW'),!. 

%%% restore_sw/0, restore_sw/1: restore the prob. of MSW from file
%%% 
%%%  bug fixed by kame on Jan/16/1998
%%%  default filename changed by kame on Feb/23/1998
% modified by hagi on Feb/24/1998.
restore_sw :- restore_sw('Saved_SW'),!.
restore_sw(File) :-
	retractall('*Switch*'(_,_,_,_,_,_)),!,
	( see(File),!,restore_sw1(File,1,GCSSVPs0),!,seen,!
	; message("{PRISM ERROR: Cannot open ~w.}",[File]),!,fail),
	sort(GCSSVPs0,GCSSVPs),!,
	restore_sw2(GCSSVPs),!,
	message("{Switch information restored from ~w.}",[File]),!,
	show_sw,!.

restore_sw1(File,N,GCSSVPs) :-
	read(Sentence),!,
	( Sentence=end_of_file,!,
	  GCSSVPs=[] 
	; Sentence='*Switch*'(G_id,Code,Status,Size,Values,Pbs),!,
	  N1 is N+1,!,
	  GCSSVPs=['*Switch*'(G_id,Code,Status,Size,Values,Pbs)|GCSSVPs1],!,
	  restore_sw1(File,N1,GCSSVPs1)
	; nth(N,Nth),!,
	  message("{PRISM WARNING: ~w sentence ~w ignored -- File ~w seems to be broken.}",[Nth,Sentence,File]),!,
	  N1 is N+1,!,
	  GCSSVPs=['*Switch*'(G_id,Code,Status,Size,Values,Pbs)|GCSSVPs1],!,
	  restore_sw1(File,N1,GCSSVPs1) ).

restore_sw2([GCSSVP|GCSSVPs]) :- assertz(GCSSVP),!,restore_sw2(GCSSVPs).
restore_sw2([]) :- !.

% show_prob_pred/0: show probabilistic predicates
show_prob_pred :-
	setof(PF/ArgN,(EF,SF)^('*Prob_Pred*'(PF,ArgN,EF,SF),ArgN>=0),FAs),!,
	message('Probabilistic predicates are:'),nl,!,
	show_prob_pred1(FAs),!.

show_prob_pred1([PA|PAs]) :-
	format('    ~w',[PA]),nl,!,show_prob_pred1(PAs).
show_prob_pred1([]) :- !.
