%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%                                                           
%%%  misc.pl: miscellaneous predicates.     
%%%                                                           
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%  Copyright (C) 1998
%%%    Taisuke Sato, Yoshitaka Kameya, Yasushi Hagiwara, Nobuhisa Ueda,
%%%      Dept. of Computer Science, Tokyo Institute of Technology.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%   Random seed:
% set_seed_time modified by hagi on Feb/24/1998.

set_seed_time(Seed) :- 
	( var(Seed),!,c_set_seed_time(Seed)
	; message("{PRISM ERROR: set_seed_time(~w) -- ~w must not be instanciated.}",[Seed,Seed]),!,fail ).

set_seed(Seed) :-
	( integer(Seed),!,set_seed_by_manual(Seed)
	; message("{PRISM ERROR: set_seed(~w) -- ~w must be 'time' or an integer.}",[Seed,Seed]),!,fail ).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%   Epsilon handling:
% modified by hagi on Feb/24/1998.

set_epsilon(Epsilon) :- set_epsilon(Epsilon,verbose).
set_epsilon(Epsilon,OP) :-
	( ( number(Epsilon),Epsilon > 0),!,
	  ( retract('*Epsilon*'(_)) ; true ),!,
	  assertz('*Epsilon*'(Epsilon)),!,
	  set_c_epsilon(Epsilon),!,
	  ( OP=verbose,!,
	    message("{Epsilon set to ~w.}",[Epsilon])
	  ; true)
	; message("{PRISM ERROR: set_epsilon(~w) -- ~w must be a positive number.}",
	          [Epsilon,Epsilon]),!,fail ).

% added by kame on Feb/27/1998.
get_epsilon(Epsilon) :-
        ( clause('*Epsilon*'(Epsilon1),true),!,
          Epsilon is Epsilon1
    ; message("{PRISM ERROR: get_epsilon(~w) -- epsilon is undefined.}",
                  [Epsilon]),!,fail ).
        
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%   Unique code generator:

% unique_code(Code): get G_id's code Code which is unique.
% [NOTE] Code expected to be uninstanciated.

unique_code(Code) :-
	( retract('*Unique_G_id*'(Code)),!,
	  NewCode is Code+1,!,
	  assertz('*Unique_G_id*'(NewCode))
	; setof(C,(G_id,C,Stat,Sz,Vs,Ps)^'*Switch*'(G_id,C,Stat,Sz,Vs,Ps),Cs),!,
	  last(C1,Cs),!, % Since Cs is sorted, last element Code is maximum.
	  Code is C1+1, NewCode is Code+1,!,
	  assertz('*Unique_G_id*'(NewCode))
	; Code=0,!, assertz('*Unique_G_id*'(1)) ),!.
	  
% unique_args_code(ArgsCode):
%   get goal's argument Args' unique code ArgsCode.
% [NOTE] ArgsCode expected to be uninstanciated.

unique_args_code(ArgsCode) :-
	( retract('*Unique_Args*'(ArgsCode)),!,
	  NewArgsCode is ArgsCode+1,!,
	  assertz('*Unique_Args*'(NewArgsCode))
	; setof(AC,(Args,C)^'*Ans*'(Args,AC,C),ACs),!, % by kame on Feb/27/1998
	  last(AC1,ACs),!, % Since ACs is sorted, last element AC1 is maximum.
	  ArgsCode is AC1+1,!,
	  NewArgsCode is ArgsCode+1,!,
	  assertz('*Unique_Args*'(NewArgsCode))
	; ArgsCode=0,!, assertz('*Unique_Args*'(1)) ),!.

% unique_sentence_number(USN):
%   get input sentence's global unique number.
% [NOTE] USN expected to be uninstanciated.

unique_sentence_number(USN) :-
	( retract('*Unique_Sentence*'(USN)),!,
	  NewUSN is USN+1,!,
	  assertz('*Unique_Sentence*'(NewUSN))
	; setof(N,(Type,Contents)^'*Sentence*'(N,Type,Contents),Ns),!,
	  last(N1,Ns),!,
	  USN is N1+1,!,
	  NewUSN is USN+1,!,
	  assertz('*Unique_Sentence*'(NewUSN))
	; USN=0,!,assertz('*Unique_Sentence*'(1)) ),!.

% unique_local_SN(Name/Arity,ULSN):
%   get input sentence's global unique number.
% [NOTE] Name and Arity are expected to be ground, while USN expected to be
%        uninstanciated.

unique_local_SN(Name/Arity,ULSN) :-
	( retract('*Unique_Local_Sentence*'(Name/Arity,ULSN)),!,
	  NewULSN is ULSN+1,!,
	  assertz('*Unique_Local_Sentence*'(Name/Arity,NewULSN))
	; setof(M,(N,Ps,H,B)^'*Sentence*'(N,normal,[Name/Arity,M,Ps,H,B]),Ms),!,
	  last(M1,Ms),!,
	  ULSN is M1+1,!,
	  NewULSN is ULSN+1,!,
	  assertz('*Unique_Local_Sentence*'(Name/Arity,NewULSN))
	; ULSN=0,!,assertz('*Unique_Local_Sentence*'(Name/Arity,1)) ),!.

% inc_unique_local_SN(Name,Arity,K):
%   increase unique local sentence number (ULSN) by K.
% [NOTE] Name and Arity are expected to be ground, and K expected to be
%        a positive number.

inc_unique_local_SN(Name,Arity,K) :-
	( retract('*Unique_Local_Sentence*'(Name/Arity,ULSN)),!
	; setof(M,(N,Ps,H,B)^'*Sentence*'(N,normal,[Name/Arity,M,Ps,H,B]),Ms),!,
	  last(M1,Ms),!,ULSN is M1+1
	; ULSN=0 ),!,
	NewULSN is ULSN+K,!,
	assertz('*Unique_Local_Sentence*'(Name/Arity,NewULSN)),!.
	
% created by hagi on Oct/1/1997
contains(del, Model) :-
	( retract('*Contains*'(Model,_)),!,
	  message("{Deleted switch information for model ~w.}",[Model])
	; message("{PRISM ERROR: No switch information for model ~w.}",[Model]),!,fail).

contains(Model, Switch) :-
	contains(new,Model,Switch).

contains(Type,Model,Switch) :-
	( is_list(Switch),!,
	  contains0(Type,Model,Switch)
	; contains0(Type,Model,[Switch])).

contains0(new,Model,Switch) :-
	( retract('*Contains*'(Model,_)),!
	; true),
	assertz('*Contains*'(Model,Switch)),
	format("{Model ~w has switches ",[Model]),
	print_list(Switch),
	message('}').
contains0(add,Model,Switch) :-
	( retract('*Contains*'(Model,Switch0)),!
	; Switch0=[]),
	'*append*'(Switch0,Switch,Switch1),
	assertz('*Contains*'(Model,Switch1)),
	format("{Model ~w has switches ",[Model]),
	print_list(Switch1),
	message('}').

contains_check(Model,G_id) :-
	( clause('*Contains*'(Model,Switch),true),!,
	  ( '*member*'(G_id,Switch),!
	    ; message("{PRISM WARNING: Model ~w does not contain Switch ~w}",[Model,G_id]))
	; true).

show_contains :-
	( setof((Model,Switch),clause('*Contains*'(Model,Switch),true),MS),
	  show_contains(MS)
	; message('PRISM ERROR: No model information found.'),!,fail).

show_contains([(Model,Switch)|MS]) :-
	format("Model ~w:  ",[Model]),
	print_list(Switch),nl,
	show_contains(MS).
show_contains([]).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%   Export logs -- Generate C main routine
%%%   

%%%  export_logs_{on,off}/0 -- user command
export_logs_on :-
	( retract('*Export_Logs*'(_)) ; true ),!,
	assert('*Export_Logs*'(on)),
	message('{Export logs will be recorded.}').
export_logs_off :-
	( retract('*Export_Logs*'(_)) ; true ),!,
	assert('*Export_Logs*'(off)),
	message('{Export logs will not be recorded.}').

%%% if_logs_on(G) -- if *Export_Logs* is on, call G.
if_logs_on(G) :-
	( clause('*Export_Logs*'(on),true),!,call(G) ; true ).

%%% if_logs_off(G) -- if *Export_Logs* _isn't_ on, call G.
if_logs_off(G) :-
	( clause('*Export_Logs*'(on),true),! ; call(G) ).

%%% logs_opening/0, logs_ending/0 -- invoked by em_msw/1.

logs_opening :-
	if_logs_on((tell('main.c'),
	            write('#include "prism.h"'),nl,nl,
				write('extern int idNum,valueNum1;'),nl,
				write('extern struct switchInfo *switch_info;'),nl,
				write('extern double *theta;'),nl,nl,
	            write('void main(){'),nl,nl,
				wtab,write('int i,v;'),nl,nl,
				wtab,write('initRoot(); initVars(); initProb();'),nl,nl)).

logs_ending :-
	if_logs_on((wtab,write('set_seed(time(NULL));'),nl,
	            wtab,write('initTheta();'),nl,
				wtab,
				write('printf("EM converged with %d steps.\n",c_EM_loop());'),
			    nl,nl,
				wtab,write('for(i=0; i<=idNum; i++){'),nl,
				wtab,wtab,
				write('for( v=0; v<=switch_info[i].maxValue; v++)'),nl,
				wtab,wtab,wtab,
				write('printf("(%d,%d):%.6f   ", i, v, Theta(i,v,theta));'),nl,
				wtab,wtab,write('printf("\n");'),nl,wtab,write('}'),nl,
				write('}'),
			    told)).
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%%   Data handling routines:
%%%

%%% list_to_paren: convert list to conjunction
%%% <ex.> list_to_paren([a,b,c],(a,b,c)).
list_to_paren([X|Xs],(X,Ys)) :-	!,list_to_paren(Xs,Ys).
list_to_paren([X],(X)).

%%% append (list version)
list_append([],[]) :- !.
list_append([L|Ls],X) :- '*append*'(L,Y,X),!,list_append(Ls,Y).

%%% is this list?
% modified by hagi on Oct/1/1997
is_list(X) :- var(X),!,fail.
is_list([]).
is_list([_|X]) :- !,is_list(X).

% modified by hagi on Feb/24/1998.
print_codes(Codes) :-
	print_codes1(Codes,GSVPs),!,
	show_sw1(5,GSVPs),!. % subroutine of show_sw/0

print_codes1([Code|Codes],[(G_id,Status,Values,Pbs)|GSVPs]) :-
	clause('*Switch*'(G_id,Code,Status,_,Values,Pbs),true),!,
	print_codes1(Codes,GSVPs).
print_codes1([],[]).

print_G_ids(G_ids) :-
	print_G_ids1(G_ids,GSVPs),!,
	show_sw1(GSVPs),!. % subroutine of show_sw/0

print_G_ids1([G_id|G_ids],[(G_id,Status,Values,Pbs)|GSVPs]) :-
	clause('*Switch*'(G_id,_,Status,_,Values,Pbs),true),!,
	print_G_ids1(G_ids,GSVPs).
print_G_ids1([],[]) :- !.

print_loop(X):- Columns = 3, print_loop1(X,Columns,Columns).
print_loop1([],_,_):-nl,!.
print_loop1(X,N,Columns):- N<1,nl,!,print_loop1(X,Columns,Columns).
print_loop1([pb_bsw(G_id,New_Pb)|Y],N,Columns):-
	0<N,N1 is N-1,
	format("BSW ~w: ~6f   ",[G_id,New_Pb]),!,
	print_loop1(Y,N1,Columns).

display_dot :- put(46),ttyflush,!.
display_o :- put(111),ttyflush,!.
wtab :- write('    '),!.

%% list handling

'*rev*'(X,Y):- '*rev1*'(X,[],Y).
'*rev1*'([H|X],Z,Y):- !,'*rev1*'(X,[H|Z],Y).
'*rev1*'([],Z,Z).

'*flatten*'([A|As],Bs) :-!,
	'*append*'(A,Bs1,Bs),'*flatten*'(As,Bs1).
'*flatten*'([],[]) :- !.

% multi_set_diff(A,B,C,D): C is A-A&B, D is B-A&B
% [NOTE] each of A,B,C,D allowed to be multi-set.
% <ex1.> multi_set_diff([a,b,c],[b,d],[a,c],[d]).
% <ex2.> multi_set_diff([a,b,b,c],[b,d],[a,b,c],[d]).

multi_set_diff(As,[B|Bs],Xs,Ys) :-
	delete(B,As,As1),!,
	( As=As1,!, % B isn't in As
	  Ys=[B|Ys1]
	; Ys=Ys1 ),!,
	multi_set_diff(As1,Bs,Xs,Ys1).
multi_set_diff(As,[],As,[]) :- !.

% diff(A,B,C): C is A-A&B
% [NOTE] A,B must be sorted and have no duplicating elements.

diff(Xs,[Y|Ys],Zs) :- delete(Y,Xs,Xs1),!,diff(Xs1,Ys,Zs).
diff(Zs,[],Zs) :- !.

% delete(X,Ys,Zs): delete *one* X from Ys (the rest is Zs)
delete(X,[X|Ys],Ys) :- !.
delete(X,[Y|Ys],[Y|Zs]) :- !,delete(X,Ys,Zs).
delete(_,[],[]) :- !.

% max(Ls,Max): get maximum element (wrt the standard order) Max from Ls
% [NOTE] fails if Ls=[].
% <ex> max([6,a,9,"3",f(x,y),z,t,3],f(x,y)).

max([L|Ls],Max) :- max(L,Ls,Max),!.
max(M,[L|Ls],Max) :- ( M @< L,!,max(L,Ls,Max) ; !,max(M,Ls,Max) ).
max(Max,[],Max) :- !.

% min(Ls,Min): get minimum element Min from Ls.
% [NOTE] fails if Ls=[].
% <ex> min([6,a,9,"3",f(x,y),z,t,3],3).

min([L|Ls],Min) :- min(L,Ls,Min),!.
min(M,[L|Ls],Min) :- ( M @> L,!,min(L,Ls,Min) ; !,min(M,Ls,Min) ).
min(Min,[],Min) :- !.

% list version of sort/2
list_sort([A|B],[A1|C]) :- sort(A,A1),!,list_sort(B,C).
list_sort([],[]) :-!.

% get last element.
% <ex.> last(2,[t,6,a(b),2]).

last(L,[L]).
last(L,[_|Ls]) :- !,last(L,Ls).

% var_check(Term,R): check if there exists variable in Term.
% R is 'yes' or 'no'.
var_check(Term,R) :-
	( var(Term),!,R=yes ; Term =.. [_|Args],!,var_check1(Args,R) ).

var_check1([Term|Terms],R) :-
	var_check(Term,R1),!,( R1=yes,!,R=yes ; var_check1(Terms,R) ).
var_check1([],no).

message(M) :- write(M),nl,!.
message(Strings,Pat) :- format(Strings,Pat),nl,!.

% modified by kame on Feb/27/1998
add_no_use_sentences :-
	assertz('*Sentence*'('*no_use*','*no_use*','*no_use*')),!,
	assertz('*Prob_Pred*'('*no_use*',-1,'*no_use*','*no_use*')).

%% concat_name/2:
%%   generates a new atom which has a concatenated name of given atoms.
%% <ex.>
%%   | ?- concat_name([a,p,p,l,e],NewAtom).
%%
%%   NewAtom = apple ?
%%
%%  created by kame on Nov/20/1997.
%%
concat_name(Atoms,NewAtom) :-
	concat_name(Atoms,D,[]),!,name(NewAtom,D).

% concat_name/3: sub-routine of concat_name/2.
concat_name([Atom|Atoms],D0,D1) :-
	dname(Atom,D0,D2),!,
	concat_name(Atoms,D2,D1).
concat_name([],D,D) :- !.

% dname/3: differential list version of name/2 (SICStus built-in).
% <ex.>
%   dname(apple,[97,112,112,108,101|X],X).
dname(Atom,D0,D1) :- name(Atom,As),!,dname1(As,D0,D1).

% dname1/3: sub-routine of dname/3.
dname1([A|As],[A|Y],Z) :- !,dname1(As,Y,Z).
dname1([],Z,Z).

% nth -- ordinal number
% <ex> nth(3,3rd)
%   modified by kame on Nov/20/1997

nth(N,Nth) :-
        ( (N > 3, N < 21),!,Th=th
        ; 1 is N mod 10,!,Th=st
        ; 2 is N mod 10,!,Th=nd
        ; 3 is N mod 10,!,Th=rd
        ; Th=th ),!,
        concat_name([N,Th],Nth).

% generate list of N zeros
% <ex> n_zeros(3,[0,0,0]).
n_zeros(0,[]).
n_zeros(N,[0|Zeros]) :- N1 is N-1,!,n_zeros(N1,Zeros).

% X is Nth member of Ys.
% <ex> nth_member(george,[john,paul,george,ringo],2).

nth_member(X,Ys,N) :- nth_member(0,X,Ys,N),!.
nth_member(N,X,[X|_],N) :- !.
nth_member(M,X,[_|Ys],N) :- M1 is M+1,!,nth_member(M1,X,Ys,N).

% insert_in_order/3
% <ex.>
%   insert_in_order(y,[x,y,z],[x,y,z]).
%   insert_in_order(x,[w,y,z],[w,x,y,z]).

insert_in_order(T,[T1|T_ids],T_ids1) :-
	( T=T1,!,T_ids1=[T1|T_ids]
	; T @< T1,!,T_ids1=[T,T1|T_ids]
	; T_ids1=[T1|T_ids2],!,insert_in_order(T,T_ids,T_ids2) ).
insert_in_order(T,[],[T]) :- !.

% plus_to_list:
% <ex> plus_to_list(1+2+3+4+5,[1,2,3,4,5]).
% plus_to_list, plus_to_list_w_sum modified by hagi on Oct/1/1997,Nov/20/1997
plus_to_list(Plus,Ls) :- plus_to_list(Plus,Ls,[]).
plus_to_list(Plus0+Plus1,Ls0,Ls1) :- !,
	plus_to_list(Plus0,Ls0,Ls2),Ls2=[Plus1|Ls1],number(Plus1),Plus1>=0.
plus_to_list(Atom,[Atom|Ls],Ls) :- number(Atom),Atom>=0.
	
% plus_to_list_w_sum:
% <ex> plus_to_list_w_sum(1+2+3+4+5,[1,2,3,4,5],15).
%      (1+2+3+4+5 equals 15.)
plus_to_list_w_sum(Plus,Ls,Sum) :-
	plus_to_list(Plus,Ls,[]),!,Sum is Plus.

% list_to_plus: reverse of plus_to_list/2.
list_to_plus([A|As],Plus) :- list_to_plus(As,A,Plus),!.
list_to_plus([A|As],B,Ps) :-
	A1=B+A,!,list_to_plus(As,A1,Ps).
list_to_plus([],A,A) :- !.

% list_to_plus_w_write created by hagi on Nov/20/1997
list_to_plus_w_write([Pb]) :-
	format("~6g",[Pb]).
list_to_plus_w_write([Pb|Pbs]) :-
	!,format("~6g+",[Pb]),!,
	list_to_plus_w_write(Pbs).

% sum_list created by hagi on Jan/5/1998
% sum_list:
% <ex.> sum_list([1,2,3,4,5],15).
sum_list(Ls,Sum) :-
	sum_list(Ls,Sum,0).
sum_list([],Sum,Sum).
sum_list([H|Ls],Sum,Tmp) :-
	Tmp1 is Tmp+H,!,
	sum_list(Ls,Sum,Tmp1).

% ratio_to_list
% <ex.> ratio_to_list(3:1,[0.75,0.25]).
% ratio_to_list,ratio_sum,make_pbs,make_uniform_ratio,make_ratio created by hagi on Nov/20/1997
% moved sw_handling.pl -> misc.pl by hagi on Feb/24/1998

ratio_to_list(PbRatio,Pbs) :-
	ratio_sum(PbRatio,Sum,0),!,
	Sum \== 0,
	P is 1.0/Sum,!,
	make_pbs(PbRatio,P,Pbs),!.
	
ratio_sum(Ratio:PbRatio,Sum,Sum0) :- !,
	number(Ratio),Ratio>=0,
	Sum1 is Sum0+Ratio,!,
	ratio_sum(PbRatio,Sum,Sum1).
ratio_sum(Ratio,Sum,Sum0) :-
	number(Ratio),Ratio>=0,Sum is Sum0+Ratio,!.

make_pbs(Ratio:PbRatio,P,[Pb|Pbs]) :- !,
	Pb is Ratio*P,!,
	make_pbs(PbRatio,P,Pbs).
make_pbs(Ratio,P,[Pb]) :- Pb is Ratio*P,!.

make_uniform_ratio(G_id,PbRatio) :-
	msw_values(G_id,Size,_),!,
	make_ratio(Size,PbRatio),!.

make_ratio(1,1) :- !.
make_ratio(Size,1:PbRatio) :-
	Next is Size-1,!,make_ratio(Next,PbRatio).

% insert_tail
% <ex.>
% | ?- insert_tail(s,[a,p,p,l,e],X).
% 
% X = [a,p,p,l,e,s] ?

insert_tail(A,[X|Xs],[X|Ys]) :- !,insert_tail(A,Xs,Ys).
insert_tail(A,[],[A]).

% deldup(+Xs,-Ys): deletes duplicating elements in Xs
%                  and returns the rest Ys.
%   <ex.>
%     deldup([a,b,a,a,d,a,c],[a,b,d,c]).
%
% [NOTE] deldup/2 keeps the order of original elements.
%
%   deldup/2, delall/3 created by kame on Dec/13/1997.

deldup([X|Xs],[X|Ys]) :- delall(X,Xs,Zs),!,deldup(Zs,Ys),!.
deldup([],[]) :-!.

% delall(X,Ys,Zs): deletes all X in Ys and returns the rest Zs.
%   <ex.>
%     delall(a,[a,b,a,c,e,d,f(a)],[b,c,e,d,f(a)]).

delall(_,[],[]) :- !.
delall(X,[X|Ys],Zs) :- !,delall(X,Ys,Zs).
delall(X,[Y|Ys],[Y|Zs]) :- !,delall(X,Ys,Zs).

% Get first N elements:
% [NOTE] fails if N is larger than the list length.
first_n_elements(N,[A|As],[A|Bs]):- N>0,!,
    N1 is N-1,!,
    first_n_elements(N1,As,Bs).
first_n_elements(0,_,[]).

% In use of statistics provided by SICStus Prolog 2.1
time(G):-
	statistics(runtime,_),
	call(G),
	statistics(runtime,[_,Time]),write('[ '),
	write(Time),write(' msec ]'),ttynl.

time(G,Time):-
	statistics(runtime,_),call(G),
	statistics(runtime,[_,Time]).

% added by hagi Oct/1/1997
print_list([H|B]) :-
	( var(H),!,H4='_'
	; var_check(H,yes),!,H=..H2,vartochar(H2,H3),H4=..H3
	; H4 = H),
	write(H4),write(' '),print_list(B).
print_list([]).

vartochar([X|Y],[Z|W]) :-
	( var(X),!,Z='_'
	; Z=X),
	vartochar(Y,W).
vartochar([],[]).

%%% is_pred_spec: checking predicate of predicate spec.

is_pred_spec(Pred) :-
	Pred=Name/Arity,!,atom(Name),!,integer(Arity),!,Arity >= 0,!.

is_pred_spec(Pred,Name,Arity) :-
	Pred=Name/Arity,!,atom(Name),!,integer(Arity),!,Arity >= 0,!.

%%% psmfile/2
%%% | ?- psmfile('foo.psm',X).
%%% X = foo ?
%%%
%%% | ?- psmfile('foo',X).
%%% no
% created by hagi on Feb/24/1998
psmfile(File,File1) :-
	name(File,List),
	'*append*'(List1,".psm",List),
	name(File1,List1).

%%% From text of Prolog:
%%%   moved from parse.pl by kame on Nov/20/1997.

'*append*'([],A,A).
'*append*'([A|X],Y,[A|Z]):-!,'*append*'(X,Y,Z).

'*member*'(X,[X|_]).
'*member*'(X,[_|Y]) :- '*member*'(X,Y).

'*reverse*'(Xs,Zs):- '*reverse*'(Xs,[],Zs).
'*reverse*'([X|Xs],Ys,Zs):- !,'*reverse*'(Xs,[X|Ys],Zs).
'*reverse*'([],Zs,Zs).