%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%%  make_routines.pl: translation of PRISM program.
%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%  Copyright (C) 1998
%%%    Taisuke Sato, Yoshitaka Kameya, Yasushi Hagiwara, Nobuhisa Ueda,
%%%      Dept. of Computer Science, Tokyo Institute of Technology.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%   make_routines: make 3 routines
%%%%   - learn_routine (learn)
%%%%   - expl_routine  (*expl_???*)
%%%%   - sample_routine(*sample_???*)

%%%% make_learn/1: make learn_routine(top-level command on learning phase)
% make_learn, make_learn0, learn modified by hagi on Oct/1/1997,Feb/24/1998
% make_learn0 modified by hagi on Jan/5/1998

% [NOTE] OP is option `quiet' or `verbose'
make_learn(OP) :-
	( setof([N,Model,Target,ArgN],
		'*Sentence*'(N,target,[Model,Target,ArgN]),Tlist),!,
	  make_learn(Tlist,OP)
	; message('{No target declarations found -- learning commands are not available.}')),
	  assertz((mlearn(M) :- message("{PRISM ERROR: learn or mlearn(~w) is not available for model `~w'.}",[M,M]),!,fail)),
	  assertz((mlearn(M,_) :- message("{PRISM ERROR: learn(_) or mlearn(~w,_) is not available for model `~w'.}",[M,M]),!,fail)),!.

make_learn([[_,Model,Target,ArgN]|Tlist],OP) :-
	( ( clause(mlearn(Model),_) ; clause(mlearn(Model,_),_) ),!,
	  ( Model=default,!,
	    message("{PRISM WARNING: duplicating target predicate ~w/~w -- ignored.}",[Target,ArgN])
	  ; message("{PRISM WARNING: duplicating target predicate ~w/~w for model `~w' -- ignored.}",[Target,ArgN,Model]))
	; make_learn0(Model,Target,ArgN,OP) ),!,
	make_learn(Tlist,OP).
make_learn([],_) :- !.

make_learn0(Model,Target,ArgN,OP) :-
	new_arg(ArgN,Arg),!,
	'*append*'(Arg,[Ans,[]],Arg0),!,
	concat_name(['*expl_',Target,'*'],ETarget),!,
	Expl =.. [ETarget|Arg0],!,
	( Arg = [Arg1],! ; Arg = Arg1 ),  % if arity = 1
	D = (define_goal(('*goal*'(Arg1,Ans) :- Expl))),
	E = em_msw(Model,Goals),!,
	( '*Sentence*'(_,data,[Model,user]),!,
	  ( OP=verbose,!,
	    ( Model = 'default',!,
	      message("{Model ~w: learn/1 (mlearn/2) is available for EM learning.}",[Model])
	    ; message("{Model ~w: mlearn/2 is available for EM learning.}",[Model]))
	  ; true),
	  G = mk_goals(Model,user,Tdata,Goals),
	  assertz((mlearn(Model,Tdata) :- !,(D,G,!,E,!)))
	; '*Sentence*'(_,data,[Model,File]),!,
	  ( OP=verbose,!,
	    ( Model = 'default',!,
	      message("{Model ~w: learn/0 (mlearn/1) is available for EM learning.}",[Model])
	    ; message("{Model ~w: mlearn/1 is available for EM learning.}",[Model]))
	  ; true),
	  G = mk_goals(Model,File,File,Goals),
	  M = mk_goals_message(File),
	  assertz((mlearn(Model) :- !,(D,G,!,M,E,!)))
	; message('{PRISM ERROR: missing data declaration.}'),!,
	  fail ),!.
	                   
mk_goals_message(File) :- format("{Goals extracted from ~w}",[File]),nl,!.

learn :- mlearn(default),!.
learn(X) :- mlearn(default,X),!.

% generate a list of new variables with length ArgN
new_arg(ArgN,Arg) :- new_arg0(0,ArgN,Arg).

new_arg0(N,N,[]) :- !.
new_arg0(M,N,Arg) :- Arg=[_|Arg1],M1 is M+1,!,new_arg0(M1,N,Arg1).

%%% mk_goals/3: get teacher data and make goals
%%%  if data file File="user", mk_goals/3 converts teacher data to goals.
%%%  otherwise, it extracts teacher data from File and convert them to
%%%  goals.
% modified by hagi on Oct/1/1997

% modified by kame on Dec/13/1997
% modified by hagi on Feb/24/1998
mk_goals(Model,user,Tdata,Goals) :- !,mk_goals1(Model,1,Tdata,Goals),!.
mk_goals(Model,_,File,Goals) :-
	( see(File),!,mk_goals0(Model,1,File,Goals),!,seen,!
	; message("{PRISM ERROR: Cannot read ~w.}",[File]),!,fail ).

mk_goals0(Model,N,File,Goals) :-
	read(G),!,
	( G = end_of_file,!,Goals=[]
	; '*Sentence*'(_,target,[Model,F,ArgN]),!, % F/ArgN is the target pred.
	  ( G = count(G1,Count),
	    functor(G1,F,ArgN),!,
		G1 =.. [F|Args],
		( Args = [Args1] ; Args = Args1 ),
		Goals = ['*goal*'(Count,Args1,_)|Goals0]
	  ; functor(G,F,ArgN),!,
	    G =.. [F|Args],
		( Args = [Args1] ; Args = Args1 ),
		Goals = ['*goal*'(1,Args1,_)|Goals0]
	  ; nth(N,Nth),!,
	    % message/2 modified by kame on Dec/13/1997.
	    message("{PRISM WARNING: ~w goal ~w in file `~w' doesn't match with target declaration -- ignored.}",[Nth,File,G]),
	    Goals = Goals0 ),!,
	  N1 is N+1,!,
	  mk_goals0(Model,N1,File,Goals0) ),!.

mk_goals1(Model,N,[T|Tdata],Goals) :-
	'*Sentence*'(_,target,[Model,F,ArgN]),!,
	( T = count(T1,Count),
	  functor(T1,F,ArgN),!,
	  T1 =.. [F|Args],
	  ( Args = [Args1] ; Args = Args1 ),!,
	  Goals = ['*goal*'(Count,Args1,_)|Goals0]
	; functor(T,F,ArgN),!,
	  T =.. [F|Args],!,
	  ( Args = [Args1] ; Args = Args1 ),!,
	  Goals = ['*goal*'(1,Args1,_)|Goals0]
	; nth(N,Nth),!,
	  message("{PRISM WARNING: ~w goal `~w' doesn't match with target declaration -- ignored.}",[Nth,T]),!,
	  Goals = Goals0 ),!,
	N1 is N+1,!,
	mk_goals1(Model,N1,Tdata,Goals0).
mk_goals1(_,_,[],[]) :- !.

%%%% make_expl/0:
%%%%   make expl_routine (the routine to get explanations for a goal)
%%%%  [NOTE!!] N is used to keep order of the clauses of user program.
make_expl :-
	%% collect normal clauses (which is not control declaration)
	setof((Spec,M,Head,Body),
	      (N,Ps)^'*Sentence*'(N,normal,[Spec,M,Ps,Head,Body]),Clauses),!,
	make_expl(Clauses),!,
	make_expl_sw,!. % added by kame on Nov/26/1997

make_expl([(HF/HArgN,_,Head,Body)|NClauses]):-
	( '*Prob_Pred*'(HF,HArgN,EF,_),!,
	  Head=..[HF|HArgs],!,
	  make_prob_expl(EF,HArgs,Body)
	; make_nonprob_expl(Head,Body)),!,
	make_expl(NClauses).
make_expl([]):-!.

%%% make_expl for probabilistic predicates
%%% --- adds diff-list for each prob. predicate
%%%     (generates *expl_??* routines.)

make_prob_expl(EF,HArgs,Body) :-
	'*append*'(HArgs,[X,Y],HArgs1),!,
	EHead =.. [EF|HArgs1],!,
	make_prob_expl_body((s0,_),tail,X,Y,Body,EBody),!,
                               % modified by kame on Dec/15/1997.
	assertz((EHead :- EBody)),!.

%%  make_expl_body/7 is modified by kame on Dec/11/1997.
%%
%%  [DESCRIPTION]
%%  make_expl_body/7  takes one of two states,  s0 and s1,  which is held
%%  in the 1st/2nd args. "s0" is the initial state. When make_expl_body/6
%%  find one switch,  it transits to the state "s1".  If make_expl_body/6
%%  is in the state "s0" and there is no following conjunct, then it must
%%  add a formula X=Y to the translated *expl_??* conjuncts (X and Y is a
%%  pair of differential list). For example, consider the following PRISM
%%  clause and multi-valued switch declarations:
%%
%%    choose(Y) :-
%%        msw(sw1,X),
%%        ( X=others, msw(sw2,Y) ; X\==others, Y=X ).
%%    values(sw1,[c1,c2,others]).
%%    values(sw2,[c3,c4,c5]).
%%
%%  which  represents  the distribution  of  a choice  from  5 candidates
%%  c1,c2,..,c5.  The first PRISM clause means "At first, flip the switch
%%  sw1. If its value is 'others', then  filp the another switch sw2  and
%%  choose the value, e.g., c3. Otherwise, pick up the value of sw1(i.e.,
%%  c1 or c2)."  make_expl_body/7  translates  above PRISM clause  to the
%%  following explanation routine:
%%
%%    expl_choose(Y,D0,D1) :-
%%        expl_sw(sw1,null,X,D0,D2),
%%        ( X=others, expl_sw(sw2,null,Y,D2,D1)
%%        ; X\==others, Y=X, D2=D1 ).
%%
%%  [NOTE]
%%  (1) In the body of  PRISM clause, if there is a disjunction of both of
%%      probabilistic and non-probabilstic disjuncts, the non-probabilistic
%%      one must have the formula such as D2=D1.
%%  (2) On the other hand, if there exists a disjunction which contains no
%%      probabilstic atoms,   PRISM translater  adds  unnessesary formulas
%%      (such as D2=D1) to all disjuncts. This may cause a slight waste of
%%      time in the exhaustive search of the explanation routines.
%%  (3) In the example above,  the formula X\==others must not be omitted.
%%      If omitted,  PRISM system  will  return  unexpected answer  in the
%%      exhaustive search.   For  example,  answer with  the probabilistic
%%      formula of "choose(others)" is:
%%
%%        | ?- probf(choose(others)).
%%
%%        choose(others)
%%        is explained by
%%          msw(sw1,null,others).
%%
%%      This is not the answer we expect.

% modified by kame on Dec/15/1997.
make_prob_expl_body((InS,OutS),Pos,X,Y,Body,EBody) :-
	( Body=(C,Conj),!,
	  make_prob_expl_body((InS,MidS),head,X,Z,C,EC),!,
	  make_prob_expl_body((MidS,OutS),tail,Z,Y,Conj,EConj),!,
	  EBody=(EC,EConj)
	; Body=(D;Disj),!,
	  make_prob_expl_body((s0,MidS1),tail,X,Y,D,ED),!,
	  make_prob_expl_body((s0,MidS2),tail,X,Y,Disj,EDisj),!,
	  ( (MidS1=s1 ; MidS2=s1),!,OutS=s1 ; InS=OutS ),!,
	  EBody=(ED;EDisj)
	; make_prob_expl_term((InS,OutS),Pos,X,Y,Body,EBody)),!.

%%  make_prob_expl_term/5 is modified by kame on Nov/26/1997. 
%%  additionaly modified by kame on Dec/11/1997.
%%
%%  [DESCRIPTION]
%%  Consider the following PRISM program (target and data declaration is
%%  omitted): 
%%
%%    a(X,W) :- b(X,Y),c(Y,Z),d(Z,W).
%%    b(X,Y) :- msw(b(X),Y).
%%    c(Y,Z) :- msw(c(Y),Z).
%%    d(Z,W) :- bsw(d(Z),W).
%%
%%    % multi-valued switch declarations:
%%    values(b(_),[x,y,z]).  
%%    values(c(_),[p,q,r]).
%%
%%  From this program, PRISM translator generates the following explanation
%%  routines:
%%
%%    expl_a(X,W,D1,D2) :-
%%        expl_b(X,Y,D1,D3),expl_c(Y,Z,D3,D4),expl_d(Z,W,D4,D2).
%%
%%    expl_b(X,Y,D1,D2) :- expl_sw(b(X),null,Y,D1,D2).
%%    expl_c(Y,Z,D1,D2) :- expl_sw(c(Y),null,Z,D1,D2).
%%    expl_b(Z,W,D1,D2) :- expl_sw(d(Z),null,W,D1,D2).
%%
%%    expl_sw(G_id,T,V,D1,D2) :-
%%        ( G_id = b(_), Vs=[x,y,z]
%%        ; G_id = c(_), Vs=[p,q,r]
%%        ; G_id = _,    Vs=[1,0] ),
%%        !, % <--- don't miss it!
%%        expl_sw(Vs,G_id,T,V,D1,D2).
%%
%%    expl_sw(Vs,G_id,T,V,D1,D2) :-
%%        member(V,Vs), D1=[msw(G_id,T,V)|D2].
%%
%%  The translator generates  expl_sw/5  by referring to only multi-valued
%%  declarations.  The cut-symbol "!" in expl_sw/5 is essential to get all
%%  correct explanations, so  we must not miss it.   As expl_sw/6 is user-
%%  independent,  we can treat  expl_sw/6  as built-in predicate of  PRISM
%%  system. member/2 is borrowed from standard Prolog text.

% [NOTE] bsw(G_id,V)/msw(G_id,V) is the abbrev. of
%        bsw(G_id,null,V)/msw(G_id,null,V).
%  modified by kame on Dec/15/1997.

make_prob_expl_term((InS,OutS),Pos,X,Y,Term,ETerm) :-
	functor(Term,F,ArgN),!,
	( ((F,ArgN)=(bsw,3);(F,ArgN)=(msw,3)),!,
	  % The following 3 lines modified by kame on Nov/26/1997
	  Term =.. [F,G_id,T,V],!,
	  ETerm = '*Expl_SW*'(G_id,T,V,X,Y),!,
	  OutS=s1,!,
	  assert_found_sw(G_id)
	; ((F,ArgN)=(bsw,2);(F,ArgN)=(msw,2)),!,
	  % The following 3 lines modified by kame on Nov/26/1997
	  Term =.. [F,G_id,V],!,
	  ETerm = '*Expl_SW*'(G_id,null,V,X,Y),!,
	  OutS=s1,!,
	  assert_found_sw(G_id)
	; '*Prob_Pred*'(F,ArgN,EF,_),!,
	  OutS=s1,!,
	  Term =.. [F|Args0],!,
	  '*append*'(Args0,[X,Y],Args),!,
	  ETerm =.. [EF|Args]
	; ( InS=s0,!,                     %% if Term is not probabilistic atom.
	    ( Pos=tail,!,ETerm = (Term,X=Y)  % if no prob-atom in precedent conjs
                                         %   and Term is the last conjunct,
        ; ETerm = Term,!,X=Y )           %    then add the formula X=Y.
	  ; ETerm = Term, X=Y ),!,           % Otherwise, do nothing.
	  InS=OutS ),!.                      % (modified by kame on Dec/11/1997)

% assert_found_sw(G_id):
%   asserts a switch G_id to switch database if G_id is instanciated.
%
%   renamed by kame on Nov/26/1997. (original name is make_prob_expl_msw/1)

assert_found_sw(G_id) :-
	( var_check(G_id,yes),!
	; ( clause('*Switch*'(G_id,_,_,_,_,_),true),!
	  ; '*Sentence*'(_,msw,[G_id,Size,Values]),!,
        unique_code(Code),!,
	    assertz('*Switch*'(G_id,Code,unfixed,Size,Values,undef)) )),!.

%%% make_expl_sw/0: generates the *Expl_SW* clause.
%%%
%%% *Expl_SW* clauses are translated by referring to only multi-valued
%%% switch (msw) declarations.
%%% <ex.>
%%%   Suppose that the following msw declaration is given:
%%%
%%%     values(f(g(_),_,h),[p,q,r]).
%%%     values(f(g(_),_,i),[s,t,u]).
%%%     values(f(h(_),_,_),[v,w]).
%%%
%%%   Translator generates the *Expl_SW* clause such as
%%%
%%%     *Expl_SW*(G_id,T,V,X,Y) :-
%%%         ( G_id = f(g(_),_,h), Vs=[p,q,r]
%%%         ; G_id = f(g(_),_,i), Vs=[s,t,u]
%%%         ; G_id = f(h(_),_,_), Vs=[v,w]
%%%         ; G_id = _, Vs=[1,0] ),
%%%         !,
%%%         expl_sw(Vs,G_id,T,V,X,Y).
%%%
%%% [NOTE]
%%%   (1) For last disjunct, default msw declaration "values(_,[1,0])" is
%%%       also needed to be translated.
%%%   (2) When *Expl_SW*/5 is called, G_id and T must be instanciated.
%%%       (This condition must be presented clearly in specification.)
%%%
%%%  Created by kame on Nov/26/1997.

% [NOTE] N is only used to keep order of the msw declarations.
make_expl_sw :-
	setof((N,G_id,Values),
	      Size^'*Sentence*'(N,msw,[G_id,Size,Values]),
		  Decls),!,
	make_expl_sw_values(G,Vs,Decls,VDisj),!,
	% [NOTE] Both of G and Vs occur in VDisj.
	assertz(('*Values*'(G,Vs):-VDisj)),!.

%make_expl_sw :-
%	setof((N,G_id,Values),
%	      Size^'*Sentence*'(N,msw,[G_id,Size,Values]),
%		  Decls),!,
%	make_expl_sw_values(G,Vs,Decls,VDisj),!,
%	% [NOTE] Both of G and Vs occur in VDisj.
%	assertz(('*Expl_SW*'(G,T,V,X,Y):-VDisj,!,expl_sw(Vs,G,T,V,X,Y))),!.

make_expl_sw_values(G,Vs,[(_,G_id,Values)|Decls],VDisj) :-
	( Decls=[],!,VDisj=(G=G_id,Vs=Values)
    ; VDisj = ((G=G_id,Vs=Values);VDisj1),!,
	  make_expl_sw_values(G,Vs,Decls,VDisj1) ),!.

'*Expl_SW*'(G,T,V,X,Y):- '*Values*'(G,Vs),!,expl_sw(Vs,G,T,V,X,Y).

%%% expl_sw(Vs,G_id,T,V,D1,D2):
%%%   For T'th trial of switch G_id, pick up its value from Vs.
%%%
%%%  Created by kame on Nov/26/1997

% [NOTE by kame on Dec/13/1997]
%   Don't insert cut-symbol between *member* and X=[..] !!

expl_sw(Vs,G_id,T,V,X,Y) :- '*member*'(V,Vs),X=[msw(G_id,T,V)|Y].

%%% for Non-Probabilistic clause 
%%% only assertz Clause.

make_nonprob_expl(Head,Body) :- assertz((Head :- Body)),!.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% make sample_routines
%%%

%% [NOTE] Spec is the form of Name/Arity.

make_sample :-
	%% collect normal Clauses
	setof((Spec,M,Head,Body),
	      (N,Ps)^'*Sentence*'(N,normal,[Spec,M,Ps,Head,Body]),
		  Clauses),!,
	make_sample(Clauses),!.

make_sample([(F/ArgN,_,Head,Body)|Clauses]):-
	( '*Prob_Pred*'(F,ArgN,_,SF),!,
	  Head =.. [F|Arg],!,
	  SHead =.. [SF|Arg],!,
	  make_sample_body(Body,SBody),!,
	  assertz((SHead :- SBody))
	; true ),!, % Do nothing(non-prob clause is already asserted by make_expl).
	make_sample(Clauses).
make_sample([]):-!.

make_sample_body((C,Conj),(SC,SConj)) :-
	make_sample_body(C,SC),!,
	make_sample_body(Conj,SConj).
make_sample_body((D;Disj),(SD;SDisj)) :-
	make_sample_body(D,SD),!,
	make_sample_body(Disj,SDisj).
make_sample_body(Term,STerm) :- make_sample_term(Term,STerm).

make_sample_term(Term,STerm) :-
	functor(Term,F,ArgN),!,
	( ((F,ArgN)=(bsw,3);(F,ArgN)=(msw,3)),!,
	  Term =.. [F|Args],!,
	  STerm =.. [sample_msw|Args]
	; ((F,ArgN)=(bsw,2);(F,ArgN)=(msw,2)),!,
	  Term =.. [F,G_id,R],!,
	  STerm =.. [sample_msw,G_id,null,R]
	; '*Prob_Pred*'(F,ArgN,_,SF),!,
	  Term =.. [F|Args],!,
	  STerm =.. [SF|Args]
	; STerm = Term ),!.
