/****************************************************************************
 **  graph.c: display a graph of learning curve.                           **
 **                                                                        **
 **   Copyright (C) 1998                                                   **
 **     Taisuke Sato, Yoshitaka Kameya, Yasushi Hagiwara, Nobuhisa Ueda,   **
 **       Dept. of Computer Science, Tokyo Institute of Technology.        **
 **                                                                        **
 ****************************************************************************/

/*
 *  written by ueda on Feb/24/1998
 *
 */

#include "prism.h"
#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>

/* xscale for both graphs */
#define Gtx	70	/* x coordinate for leftmost of graphs */
#define Gw	500	/* width of graphs */
/* yscale for graph showing likelihood (L-graph) */
#define GtyL	50	/* y coordinate for top of L-graph */
#define GhL	350	/* height of L-graph */
/* yscale for graph showing the difference between successive
   likelihood (D-graph) */
#define Gty	450	/* y coordinate for top of D-graph */
#define Gh	320	/* height of D-graph */
#define Infty   10000   /* pseudo value of infinity */
#define Delta	0.0001	/* pseudo value of zero for avoiding division by zero */
#define Maxunit 1000	/* maximum of # iteration on the graph */
/* variables for X window */
Display		*d;
Window		rw, w;
XEvent		ev;
GC		gc, gc2;
char		fontname[] = "*helvetica-bold-r*10*";
Font		currentfont;
unsigned long	col[4];

char* string;
int c, rate[Maxunit+1], rate2[Maxunit+1], unit=500, num_eps;
float eps, eps2;
/* commands for drawing graphs */

rescale(x, y, width, h, xsize, ysize, iteration, min, max, logeps)
int x, y, width, h, xsize, iteration;
float min, max, logeps;
{
    char string2[64];
    int l1;

    for ( l1 = 0; l1 <= ysize; l1++ ){
	XDrawLine(d, w, gc, x, y+h*l1/ysize, x+width, y+h*l1/ysize);
	sprintf(string2, "%3.1f", max-(max-min)*l1/ysize);
	XDrawString(d, w, gc, 10, y+h*l1/ysize+5, string2, strlen(string2));
    }
    for ( l1 = 0; l1 <= xsize; l1++ ){
	XDrawLine(d, w, gc, x+width*l1/10, y, x+width*l1/10, y+h);
	sprintf(string2, "%3d", l1 * (unit/xsize) + (iteration-unit/2));
	XDrawString(d, w, gc, x+width*l1/10, y+h+15,
		    string2, strlen(string2));
    }
    if (logeps==0.0) {
	string = "Log-likelihood";
	XDrawString(d, w, gc, 10, GtyL-15, string, strlen(string));	}
    else {
	XSetForeground(d, gc2, col[2]);
	XDrawLine(d, w, gc2, x, y+(h*logeps/ysize), x+width, y+(h*logeps/ysize));
	XSetForeground(d, gc2, col[3]);
	string = "Gradient of log-likelihood";
	XDrawString(d, w, gc, 10, Gty-15, string, strlen(string));      }
    /* write comments */
}

void graph_redraw_em_loop(iteration,min,max,check)
int iteration;
float min,max;
unsigned int check; /*  0: redraw if there is an event 1: forcing to redraw */
{
    int l1, mask=0, count=0;
    int x1, y1, x2, y2, z1, z2;

    while ( 1 )	{
	count++;
	ev.type =0;
	XCheckMaskEvent(d, ExposureMask, &ev);
	if (ev.type == Expose || check == 1) {
	    if (check ==1 ) XClearArea(d, w, 0, 0, Gw, Gty+Gh, False);
	    if (iteration > unit/2) l1 = iteration-(iteration%(unit/2)); else l1=unit/2;
	    rescale(Gtx,Gty,Gw,Gh,10,num_eps,l1,eps,3.0,eps2);
	    rescale(Gtx,GtyL,Gw,GhL,10,10,l1,min,max,0);
	    for (l1=1; l1<unit; l1++){
		if (rate[l1] != Infty && rate[(l1-1)] != Infty){
		    x1=Gtx+l1*Gw/unit; x2=Gtx+(l1+1)*Gw/unit,
		    y1=Gty+Gh-(Gh*rate[l1-1])/(num_eps*1000);
		    y2=Gty+Gh-Gh*rate[l1]/(num_eps*1000);
    		    XDrawLine(d, w, gc2, x1, y1, x2, y2);
		    z1=GtyL+GhL-GhL*rate2[l1-1]/(max-min+Delta)/1000;
		    z2=GtyL+GhL-GhL*rate2[l1]/(max-min+Delta)/1000;
		    XDrawLine(d, w, gc2, x1, z1, x2, z2);
		}
	    }
	}
	    if (count >= 1) break;
    }
    XFlush(d);
}

void graph_setwindow_em_loop(min,max,epsilon,width)
float min, max, epsilon;
int width;
{
    Colormap	colmap;
    XColor	c0, c1;
    unsigned int c[2];
    int	wx = 620, wy = 800, ww = wx, wh = wy, fw = 2, l1;

    unit=width;
    for (l1=0; l1<=unit; l1++) rate[l1]=Infty;

    d = XOpenDisplay ( NULL );
    colmap = XDefaultColormap ( d, 0 );
    XAllocNamedColor (d, colmap, "red",   &c1, &c0);	col[2] = c1.pixel;
    XAllocNamedColor (d, colmap, "blue",  &c1, &c0);	col[3] = c1.pixel;
    XAllocNamedColor (d, colmap, "black", &c1, &c0);	col[0] = c1.pixel;
    XAllocNamedColor (d, colmap, "white", &c1, &c0);	col[1] = c1.pixel;
    rw = DefaultRootWindow (d);
    w = XCreateSimpleWindow (d, rw, wx, wy, ww, wh, fw, col[0], col[1]);
    gc = XCreateGC(d, w, 0, 0);
    XSetLineAttributes(d, gc , 0, LineSolid, CapNotLast, JoinRound);
    XSetForeground(d, gc , col[0]);
    if ((currentfont = XLoadFont(d, fontname)) != None)
	{
	    XSetFont(d, gc, currentfont);
	}
    gc2= XCreateGC(d, w, 0, 0);
    XSetLineAttributes(d, gc2, 3, LineSolid, CapNotLast, JoinRound);
    XSetForeground(d, gc2, col[3]);

    XStoreName (d, w, "PRISM: convergence");
    XMapWindow (d, w);

    eps=(int)log10(epsilon)-1; /* -1 is for rounding down */
    eps2=3.0-log10(epsilon);	num_eps=3.0-eps;

    XSelectInput (d, w, ExposureMask);
    while ( 1 )	{
	XNextEvent (d, &ev);
	if (ev.type == Expose ) {
	    rescale(Gtx,Gty,Gw,Gh,10,num_eps,unit/2,eps,3.0,eps2);
	    rescale(Gtx,GtyL,Gw,GhL,10,10,unit/2,min,max,0);
	    break;
	}
    }
    rescale(Gtx,Gty,Gw,Gh,10,num_eps,unit/2,eps,3.0,eps2);
    rescale(Gtx,GtyL,Gw,GhL,10,10,unit/2,min,max,0);
}

void newregion(ww, wh, iteration, min, max)
int iteration;
unsigned int ww, wh;
float min, max;
{
    unsigned int width, height;
    int l1;

    for (l1=1; l1<unit/2; l1++){
	rate[l1] =rate[(int)unit/2+l1];		rate[(int)unit/2+l1]=Infty;
	rate2[l1]=rate2[(int)unit/2+l1];	rate2[(int)unit/2+l1]=Infty;
    }
    XClearArea(d, w, Gtx, GtyL, Gtx+ww*2, GtyL+wh, True);
    rescale(Gtx,Gty,Gw,Gh,10,num_eps,iteration,eps,3.0,eps2);
    rescale(Gtx,GtyL,Gw,GhL,10,10,iteration,min,max,0);
}

void graph_line_em_loop(iteration,diff,loglike,min,max)
int iteration;
double diff, loglike, min, max;
{
    int x, y, z, x1,y1,z1, x2,y2,z2;

    y=(-eps+log10(diff))*1000;
    z=(loglike-min)*1000;
    if (iteration >= unit/2) x=iteration % (unit/2)+(unit/2); else x=iteration;
    if (iteration % (unit/2) == 0 && iteration >= unit) {
	newregion(600, GtyL+Gh+GhL, iteration, min, max);
	rate[x-1]=y; rate2[x-1]=z;
    }
    rate[x]=y;    rate[0]=rate[1];
    x1=Gtx+x*Gw/unit; x2=Gtx+(x+1)*Gw/unit,
    y1=Gty+Gh-(Gh*rate[x-1])/(num_eps*1000); y2=Gty+Gh-Gh*y/(num_eps*1000);
    XDrawLine(d, w, gc2, x1, y1, x2, y2);

    rate2[x]=z;    rate2[0]=rate2[1];
    z1=GtyL+GhL-(GhL*rate2[x-1])/(max-min+Delta)/1000;
    z2=GtyL+GhL-GhL*z/(max-min+Delta)/1000;
    XDrawLine(d, w, gc2, x1, z1, x2, z2);
    XFlush(d);
}

void graph_shutwindow_em_loop()
{
    XDestroyWindow(d,w);
    XFlush(d);
    XFreeGC(d,gc);
    XFreeGC(d,gc2);
    XCloseDisplay(d);
}
