%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%%  c_interface.pl: export/import the EM information.
%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%  Copyright (C) 1998
%%%    Taisuke Sato, Yoshitaka Kameya, Yasushi Hagiwara, Nobuhisa Ueda,
%%%      Dept. of Computer Science, Tokyo Institute of Technology.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%
%%% Initialization of C routines' global variables
%%%
init_c_routine :- 
	really_init_trie,!,really_init_expl_table,!,
	really_init_prob,!.

%%%  export_EM: exports the information required in EM learning
%%%             to EM routine (written in C)
%%%  <list of exports:>
%%%  (1) the number of different switches
%%%  (2) the integer code of each switch
%%%  (3) the integer code of each fixed (unfixed?) switch
%%%  (4) the number of different goal patterns.
%%%      <ex.> bloodtype program has four goal patterns: 
%%%      { bloodtype(a),bloodtype(b),bloodtype(o),bloodtype(ab) }
%%%  (5) the frequency of each goal pattern in teacher data set. 
%%%      <ex.> teacher data set (100 goals) has 40 bloodtype(a),
%%%            20 bloodtype(b), 30 bloodtype(o), 10 bloodtype(ab).
%%%  (6) value size of each switch
%%%  (7) contents of explanation table

export_EM(Codes,FixedCodes) :-
	retractall('*Exported_Codes*'(_)),!,
	assertz('*Exported_Codes*'(Codes)),!,
	if_logs_off((write('{Exporting EM information to EM routine...'),!,
	                   ttyflush)),!,
	( ( export_codes(Codes),!
  	  ; message('{PRISM INTERNAL ERROR: export_codes/1 fails.}'),!,fail ),!,
	  ( export_fixed_codes(FixedCodes),!
  	  ; message('{PRISM INTERNAL ERROR: export_fixed_codes/1 fails.}'),!,
        fail ),!,
      ( export_goals,!
  	  ; message('{PRISM INTERNAL ERROR: export_goals/0 fails.}'),!,fail ),!,
	    if_logs_off(message('done.}'))
	; message('{Exportation of EM information failed.}'),!,fail ),!.

% export_codes: exports the information about switches.
% [NOTE]
% export_code/1 (not export_code*s*) is C interface predicate.
% Also exports the number of possible values.
export_codes(Codes) :-
	get_sw_maxval(Codes,MaxVals,Tnums),!,
	max(MaxVals,MaxMaxVal),!,
	length(Codes,NC0),NC is NC0-1,!,
	if_logs_on((wtab,write(set_values(NC,MaxMaxVal)),write(';'),nl)),!,
	export_sw(NC,MaxMaxVal,R),!,
	export_error(R),!,
	export_codes_loop(Codes,MaxVals,Tnums),!.

get_sw_maxval([C|Codes],[MV|MaxVals],[Tnum|Tnums]) :-
	clause('*Switch*'(_,C,_,Size,_,_),true),!,
	clause('*Occ_SW*'(_,C,T_ids),true),!,
	length(T_ids,Tnum),!,
	MV is Size-1,!,
	get_sw_maxval(Codes,MaxVals,Tnums).
get_sw_maxval([],[],[]) :-!.

export_codes_loop([C|Codes],[MV|MaxVals],[Tnum|Tnums]) :-
	if_logs_on((wtab,write(set_idInfo(C,MV,Tnum)),write(';'),nl)),!,
	export_code(C,MV,Tnum,R),!,
	export_error(R),!,export_codes_loop(Codes,MaxVals,Tnums).
export_codes_loop([],[],[]) :- !.

% export_fixed_codes: exports the information about fixed switches.
% [NOTE]
% export_fixed_code/1 (not export_fixed_code*s*/1) is
% C interface predicate.
export_fixed_codes([FC|FixedCodes]) :-
	export_fixed_code_loop(FC),!,export_fixed_codes(FixedCodes).
export_fixed_codes([]) :- !.

%  bugs of export_fixed_code_loop/1-3 fixed by kame on Dec/18/1997.
export_fixed_code_loop(FC) :-
	clause('*Switch*'(_,FC,fixed,_,_,Pbs),true),!,
	export_fixed_code_loop(FC,0,Pbs),!.

export_fixed_code_loop(FC,N,[Pb|Pbs]) :-
    ( export_fixed_code(FC,N,Pb)
    ; message("{PRISM INTERNAL ERROR: export_fixed_code(~w,~w,~w) failed.}",[FC,N,Pb]),!,
	  fail ),!,
	N1 is N+1,!,
	export_fixed_code_loop(FC,N1,Pbs).
export_fixed_code_loop(_,_,[]) :- !.

% export_goals: exports the information about goals.
% [NOTE] export_goal_patterns/1 is C interface predicate.
%    modified for *Ans* by kame on Feb/27/1998
export_goals :-
	setof((ACode,Args),C^clause('*Ans*'(Args,ACode,C),true),As),!, 
    % Args is sorted, then each argument patterns
    % is numbered 0,1,2,... implicitly.
	length(As,NP),!,
	get_goal_weights(As,Ws),!,
	if_logs_on((wtab,write(set_goals(NP)),write(';'),nl)),!,
	export_goal_patterns(NP,R),!, % finally calls set_goals()
	export_error(R),!,
	export_goal_weights(0,Ws),!.  % finally calls set_data()

% export_goal_weight/2 is C interface predicate.
get_goal_weights([(_,Args)|As],[W|Ws]) :-
	( clause('*Ans*'(Args,_,W),true) ; W=0 ),!,
	get_goal_weights(As,Ws).
get_goal_weights([],[]) :- !. 

export_goal_weights(N,[W|Ws]) :-
	if_logs_on((wtab,write(set_data(N,W)),write(';'),nl)),!,
	export_goal_weight(N,W),!,  % C interface predicate
	N1 is N+1,!,
	export_goal_weights(N1,Ws).
export_goal_weights(_,[]) :- !.
	
%%% export_error(R) -- fails if exportation fails (R=<0).
%
%  modified by kame on Jul/23/1998.

export_error(R) :-
    ( R>0,!,true
	; message('{PRISM ERROR: Memory allocation failed.}'),!,fail ),!.

%%% export_ans/1: exports an answer (a conjunction of switch).
%%% [NOTE] export_ans_sw/4 is C interface predicate.

export_ans([(GC,TC,VC)|Coded_Ans]) :-
	( Coded_Ans=[],!,
	  if_logs_on((wtab,write(get_trie(0,GC,TC,VC)),write(';'),nl)),!,
	  export_ans_sw(0,GC,TC,VC,R),!,
	  export_error(R)
	; if_logs_on((wtab,write(get_trie(1,GC,TC,VC)),write(';'),nl)),!,
	  export_ans_sw(1,GC,TC,VC,R),!,
	  export_error(R),!,
	  export_ans(Coded_Ans) ),!.

%%% aliases for exportation logs

prepare_trie(R) :-
	if_logs_on((wtab,write('prepare_trie();'),nl)),!,
	prepare_c_trie(R),!.

prepare_expl(AC,R) :-
	if_logs_on((wtab,write(prepare_expl(AC)),write(';'),nl)),!,
	prepare_c_expl(AC,R),!.

count_ans(R) :-
	if_logs_on((wtab,write('count_expls();'),nl)),!,
	count_c_ans(R),!.

init_trie :-
	if_logs_on((wtab,write('freeTrie();'),nl)),!,
	init_c_trie,!.
	
%%%
%%% Exportation predicates for prob_calculate/2
%%% (used in prob/{1,2} and cprob/{2,3})
%%%

% Ans is an explanation, and is in the form the list of encoded
% switch, i.e., Ans=[(GC,TC,VC),...]
export_prob_ans([(GC,TC,VC)|Coded_Ans]) :-
	( Coded_Ans=[],!,
	  export_prob_ans_sw(0,GC,TC,VC,R),!,
	  export_error(R),!
	; export_prob_ans_sw(1,GC,TC,VC,R),!,
	  export_error(R),!,
	  export_prob_ans(Coded_Ans)).

export_prob_calculate(Codes) :-
	get_prob_sw_maxval(Codes,MaxVals),!,
	max(MaxVals,MMVal),!,
	length(Codes,NC0),NC is NC0-1,!,
	export_prob_sw(NC,MMVal,R),!,
	export_error(R),!,
	export_prob_codes(Codes,MaxVals),!,
	export_pbs(Codes),!.

get_prob_sw_maxval([C|Codes],[MV|MaxVals]) :-
	clause('*Switch*'(_,C,_,Size,_,_),true),!,
	MV is Size-1,!,
	get_prob_sw_maxval(Codes,MaxVals).
get_prob_sw_maxval([],[]) :- !.

export_prob_codes([C|Codes],[MV|MaxVals]) :-
	export_prob_code(C,MV,R),!,
	export_error(R),!,
	export_prob_codes(Codes,MaxVals).
export_prob_codes([],[]) :- !.

export_pbs([Code|Codes]) :-
	clause('*Switch*'(G_id,Code,_,_,_,Pbs),true),!,
	( Pbs=undef,!,
      message("{PRISM ERROR: Parameter of switch ~w is not assigned yet. Run show_sw/0 for confirmation, or run EM learning or set_sw/{1,2} for setup.}",
	          [G_id]),!,
	  fail
	; export_pbs_loop(Code,0,Pbs) ),!,
	export_pbs(Codes).
export_pbs([]) :- !.

export_pbs_loop(Code,N,[Pb|Pbs]) :-
	export_pb(Code,N,Pb,R),!,
	export_error(R),!,
	N1 is N+1,!,
	export_pbs_loop(Code,N1,Pbs).
export_pbs_loop(_,_,[]) :- !.

%%%
%%% Importation predicates
%%%

% [NOTE] import_pb/3 is C-interface predicate.
import_pbs :-
	clause('*Exported_Codes*'(ExportedCodes),true),!,
	import_pbs(ExportedCodes),!.

import_pbs([Code|Codes]) :-
	( retract('*Switch*'(G_id,Code,unfixed,Size,Values,_)),!,
	  import_pbs(0,Size,Code,NewPbs),!,
	  assertz('*Switch*'(G_id,Code,unfixed,Size,Values,NewPbs)),!
	; true ),!, % do nothing if C is fixed or switch C doesn't exist.
	import_pbs(Codes).
import_pbs([]) :- !.

import_pbs(Size,Size,_,[]) :- !.
import_pbs(N,Size,Code,NewPbs) :-
	import_pb(Code,N,NewPb),!,
	( NewPb < 0,!,
	  message("{PRISM INTERNAL ERROR: import_pb(~w,~w,_) fails.}",[Code,N]),!,
	  fail
	; NewPbs=[NewPb|NewPbs0] ),!,
	N1 is N+1,!,
	import_pbs(N1,Size,Code,NewPbs0).

