(***********************************************************************

                    Mathematica-Compatible Notebook

This notebook can be used on any computer system with Mathematica 3.0,
MathReader 3.0, or any compatible application. The data for the notebook 
starts with the line of stars above.

To get the notebook into a Mathematica-compatible application, do one of 
the following:

* Save the data starting with the line of stars above into a file
  with a name ending in .nb, then open the file inside the application;

* Copy the data starting with the line of stars above to the
  clipboard, then use the Paste menu command inside the application.

Data for notebooks contains only printable 7-bit ASCII and can be
sent directly in email or through ftp in text mode.  Newlines can be
CR, LF or CRLF (Unix, Macintosh or MS-DOS style).

NOTE: If you modify the data for this notebook not in a Mathematica-
compatible application, you must delete the line below containing the 
word CacheID, otherwise Mathematica-compatible applications may try to 
use invalid cache data.

For more information on notebooks and Mathematica-compatible 
applications, contact Wolfram Research:
  web: http://www.wolfram.com
  email: info@wolfram.com
  phone: +1-217-398-0700 (U.S.)

Notebook reader applications are available free of charge from 
Wolfram Research.
***********************************************************************)

(*CacheID: 232*)


(*NotebookFileLineBreakTest
NotebookFileLineBreakTest*)
(*NotebookOptionsPosition[     57055,       1810]*)
(*NotebookOutlinePosition[     57907,       1840]*)
(*  CellTagsIndexPosition[     57863,       1836]*)
(*WindowFrame->Normal*)



Notebook[{

Cell[CellGroupData[{
Cell["Modelling Electric Circuits", "Title",
  FontColor->RGBColor[0, 0, 0.500008]],

Cell["\<\
\[Copyright]1998-1999 Research Institute for Symbolic Computation \
(RISC-Linz) and Institute of Information Sciences and Electronics, Tsukuba.\
\
\>", "Text",
  InitializationCell->True],

Cell[CellGroupData[{

Cell["NO WARRANTY", "Subsubsection",
  InitializationCell->True],

Cell["\<\
The program was produced on an experimental basis in the
course of the research and development conducted during the project
and is provided to users as so produced on an experimental basis.
Accordingly, the program is provided without any warranty whatsoever,
whether express, implied, statutory or otherwise. The term \"warranty\"
used herein includes, but is not limited to, any warranty of the
quality, performance, merchantability and fitness for a particular
purpose of the program and the nonexistence of any infringement or
violation of any right of any third party.

Each user of the program will agree and understand, and be
deemed to have agreed and understood, that there is no warranty
whatsoever for the program and, accordingly, the entire risk arising
from or otherwise connected with the program is assumed by the user.

Therefore, neither ICOT, the copyright holder, or any other
organization that participated in or was otherwise related to the
development of the program and their respective officials, directors,
officers and other employees shall be held liable for any and all
damages, including, without limitation, general, special, incidental
and consequential damages, arising out of or otherwise in connection
with the use or inability to use the program or any product, material
or result produced or otherwise obtained by using the program,
regardless of whether they have been advised of, or otherwise had
knowledge of, the possibility of such damages at any time during the
project or thereafter. Each user will be deemed to have agreed to the
foregoing by his or her commencement of use of the program. The term
\"use\" as used herein includes, but is not limited to, the use,
modification, copying and distribution of the program and the
production of secondary products from the program.

In the case where the program, whether in its original form or
modified, was distributed or delivered to or received by a user from
any person, organization or entity other than ICOT, unless it makes or
grants independently of ICOT any specific warranty to the user in
writing, such person, organization or entity, will also be exempted
from and not be held liable to the user for any such damages as noted
above as far as the program is concerned.\t\
\>", "Text",
  InitializationCell->True]
}, Closed]],

Cell[CellGroupData[{

Cell["Package Description", "Subtitle"],

Cell[TextData[
"This notebook is an electronic tutorial for CFLP users. It ilustrates on a \
particular case\[LongDash]the theory of electrical circuits\[LongDash]how to \
write a package for CFLP. More precisey, we illustrate the way how a \
potential CFLP user can represent an equational theory as a set of CFLP \
rewrite rules and how to solve problems in this theory with CFLP. \nThe \
package ElDemo.m which contains definitions for electrical components \
(resistors and capacitors) and for elementary operations, like serial and \
parallel connections, used in building electrical circuits."], "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Implementation", "Subtitle"],

Cell[TextData[{
  "There is a standard sequence of ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " commands that are typically used when writing a ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " package. All these commands must be written in input initialization \
cells. This means that, after you write the command in a ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " notebook, you have to perform the following steps:"
}], "Text"],

Cell[TextData[{
  StyleBox["1",
    FontWeight->"Bold"],
  ". select the input cell of the newly typed command by clicking with the \
mouse pointer on the cell bracket displayed on the right of your command,\n",
  StyleBox["2",
    FontWeight->"Bold"],
  ". switch on the ",
  StyleBox["Initialization Cell",
    FontWeight->"Bold"],
  " option from the ",
  StyleBox["Cell/Cell Properties",
    FontWeight->"Bold"],
  " submenu "
}], "Text",
  CellMargins->{{25, Inherited}, {Inherited, Inherited}},
  TextAlignment->AlignmentMarker],

Cell[CellGroupData[{

Cell["Begin", "Section"],

Cell[TextData[{
  " The first command is ",
  StyleBox["BeginPackage",
    FontWeight->"Bold"],
  ". It takes two arguments: the first argument is the name of the package \
that is going to be written, and the second one is a list of ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " packages on top of which you write the new package. In this example, this \
command is:"
}], "Text"],

Cell[BoxData[
    \(\(BeginPackage["\<ElDemo`\>", {"\<TSolve`\>"}]; \)\)], "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[TextData[{
  "and it indicates the fact that the a new package called ",
  StyleBox["ElDemo.m ",
    FontWeight->"Bold"],
  "is being written on top of the ",
  StyleBox["TSolve.m",
    FontWeight->"Bold"],
  " package. ",
  StyleBox["TSolve.m",
    FontWeight->"Bold"],
  " is the only package that has to be loaded by a ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " user who wants to use the CFLP system."
}], "Text"],

Cell[TextData[{
  "Next, we specify the symbols we want to have visible outside the package. \
For this, we define ",
  StyleBox["usage",
    FontFamily->"Fixed",
    FontWeight->"Bold"],
  " messages, which are ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " commands of the form:\n\n",
  StyleBox["symb",
    FontSlant->"Italic"],
  "::usage=\"string\"\n\nwhere ",
  StyleBox["symb",
    FontSlant->"Italic"],
  " is a symbol exported from the package, and \"string\" is a string which \
describes the meaning of ",
  StyleBox["symb",
    FontSlant->"Italic"],
  ". A message associated to a symbol ",
  StyleBox["symb",
    FontSlant->"Italic"],
  " can be visualized by the user by typing in  the command:\n\n?",
  StyleBox["symb",
    FontSlant->"Italic"]
}], "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Usage", "Section"],

Cell[BoxData[
    \(\(resistor::usage = 
      "\<resistor[R,S,V0,I0,V1,I1] describes a resistor with input \
voltage/current s[V0,I0] and output s[V1,I1].\>"\ ; \)\)], "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    \(\(capacitor::usage = 
      "\<capacitor[R,S,s[V0,I0],s[V1,I1]] describes a capacitor with input \
voltage/current s[V0,I0] and output s[V1,I].\>"\ ; \)\)], "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    \(\(junction::usage = 
      "\<junction[InSignals,OutSignals] describes a junction of inputs \
InSignals and outputs OutSignals. L and R are lists of electrical signals.\>"\
\ ; \)\)], "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    \(\(elJoin::usage = 
      "\<elJoin[S,L,V] describes the join operation of inputs L and outputs \
R. Here, L is a list of pairs s[Vi,Ii] modelling the signals, where V equals \
all Vi and I is the sum of Ii.\>"\ ; \)\)], "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    \(\(CComp::usage = 
      "\<CComp[C0] describes a capacitor with characteristic capacity C0.\>"\ \
; \)\)], "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    \(\(RComp::usage = 
      "\<RComp[R] describes a resistor with characteristic resistance R.\>"\ \
; \)\)], "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    \(s::usage; \ ElSignal::usage; \)], "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    \(\(serial::usage = 
      "\<serial[compList, s[V1,I1], s[V2,I2]] is a predicate which describes \
the relation between the input signal s[V1,I1] and the output signal s[V2,I2] \
for the serial connection of the electrical components of `compList`.\>"; 
    \)\)], "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    \(\(parallel::usage = 
      "\<serial[compList, s[V1,I1], s[V2,I2]] is a predicate which describes \
the relation between the input signal s[V1,I1] and the output signal s[V2,I2] \
for the parallel connection of the electrical components of `compList`.\>"; 
    \)\)], "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    \(\(PConnect::usage = 
      "\<PConnect[Clist, s[V0,I0], s[V1,I1]] is the electrical component \
representing the serial connection of the unary components of list Clist, \
with input s[V0,I0], output s[V1,I1] and system frequency S.\>"; \)\)], 
  "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    \(\(SConnect::usage = 
      "\<SConnect[S, Clist, s[V0,I0], s[V1,I1]] is the electrical component \
representing the serial connection of the unary components of list Clist, \
with input s[V0,I0], output s[V1,I1] and system frequency S.\>"; \)\)], 
  "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[TextData[{
  "The ",
  StyleBox["Begin",
    FontWeight->"Bold"],
  " command marks the starting point of a package context. In this example, \
this is the Private context."
}], "Text"],

Cell[BoxData[
    \(\(Begin["\<`Private`\>"]; \)\)], "Input",
  PageWidth->Infinity,
  InitializationCell->True,
  ShowSpecialCharacters->False,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[TextData[{
  "The following command makes visible the CFLP functions into the private \
context of ",
  StyleBox["ElDemo.m",
    FontWeight->"Bold"],
  ":"
}], "Text"],

Cell[BoxData[
    \(\($ContextPath = Join[$CFLPpackages, $ContextPath]; \)\)], "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[TextData[{
  "From now on the package developer can write expressions in CFLP syntax. \
The ",
  StyleBox["CFLPsyntax.nb",
    FontWeight->"Bold"],
  " palette is provided as keyboard extension with buttons to enter CFLP \
expressions."
}], "Text"],

Cell[TextData[{
  "We represent electric signals as pairs ",
  StyleBox["s[V,I]",
    FontWeight->"Bold"],
  " of voltage/current, where ",
  StyleBox["s",
    FontWeight->"Bold"],
  " is a constructor symbol. We also find useful to introduce a new type \
constructor ",
  StyleBox["ElSignal",
    FontWeight->"Bold"],
  ", to denote the type of an electrical signal."
}], "Text"],

Cell["\<\
The following command extends the set of CFLP type constructors \
with the type constructor sig.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(TypeConstructor[ElSignal]\)], "Input",
  PageWidth->Infinity,
  InitializationCell->True,
  ShowSpecialCharacters->False,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    \({ElSignal}\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "and next we impose type ",
  StyleBox["sig",
    FontWeight->"Bold"],
  " on signal terms of the form ",
  StyleBox["s[V,I]",
    FontWeight->"Bold"],
  ":"
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    RowBox[{
    "Constructor", "[", 
      \(s : Float\[Cross]Float \[ShortRightArrow] ElSignal\), 
      StyleBox["]",
        ShowSpecialCharacters->False,
        ShowStringCharacters->True]}]], "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    \({s}\)], "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Electrical components", "Section"],

Cell[TextData[{
  "We  model ",
  StyleBox["electrical components",
    FontSlant->"Italic"],
  " by predicates that express the relationship between the input and output \
signal of a system by systems of equations  in the components of input/output \
signals. The predicates we define below are:"
}], "Text"],

Cell[TextData[{
  StyleBox["resistor[R,s[V1,I1],s[V2,I2]]",
    FontWeight->"Bold"],
  ": defines the behavior of a resistor with input signal ",
  StyleBox["s[V1,I1]",
    FontWeight->"Bold"],
  ", output signal ",
  StyleBox["s[V2,I2]",
    FontWeight->"Bold"],
  ", and characteristic value ",
  StyleBox["R",
    FontWeight->"Bold"],
  ",\n",
  StyleBox["capacitor[C,s[V1,I1],s[V2,I2]]",
    FontWeight->"Bold"],
  ": defines the behavior of a resistor with input signal ",
  StyleBox["s[V1,I1]",
    FontWeight->"Bold"],
  ", output signal ",
  StyleBox["s[V2,I2]",
    FontWeight->"Bold"],
  ", and characteristic value ",
  StyleBox["C.",
    FontWeight->"Bold"]
}], "Text"],

Cell[TextData[{
  "All these predicates are defined symbols of type ",
  StyleBox["Float\[Cross]ElSignal\[Cross]ElSignal\[Rule]Bool",
    FontWeight->"Bold"]
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(DefinedSymbol[\n\t
      resistor : 
        Float\[Cross]ElSignal\[Cross]ElSignal \[ShortRightArrow] Bool, \n\t
      capacitor : 
        Float\[Cross]ElSignal\[Cross]ElSignal \[ShortRightArrow] Bool\n]\)], 
  "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    \({resistor, capacitor}\)], "Output"]
}, Open  ]],

Cell["\<\
We proceed by defining the characterizing predicates which describe \
electrical components in terms of CFLP rewrite rules.\
\>", "Text"],

Cell[TextData[{
  "A ",
  StyleBox["resistor ",
    FontSlant->"Italic"],
  " with characteristic ",
  StyleBox["R,",
    FontWeight->"Bold"],
  " input signal",
  StyleBox[" s[V1,I1]",
    FontWeight->"Bold"],
  " and output signal ",
  StyleBox["s[V2,I2]",
    FontWeight->"Bold"],
  " satisfies the equations:"
}], "Text"],

Cell["\<\
-V1+V2 = R * I1
I1 = I2 \
\>", "Text"],

Cell["We define", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(Def[{R, S, V1, I1, V2, I2}, \n
      \t{resistor[R, s[V1, I1], s[V2, I2]] \[RightArrow] True 
          \[DoubleLeftArrow] {\(-V1\) + V2 \[TildeTilde] I1\ R, 
            I1 \[TildeTilde] I2}}]\)], "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    InterpretationBox[
      \("Adding rule #"\[InvisibleSpace]1
        \[InvisibleSpace]": "\[InvisibleSpace]\(resistor[R, s[V1, I1], 
            s[V2, I2]] \[RightArrow] True 
          \[DoubleLeftArrow] {\(-V1\) + V2 \[TildeTilde] I1\ R, 
            I1 \[TildeTilde] I2}\)\),
      SequenceForm[ "Adding rule #", 1, ": ", 
        RewriteRule[ 
          resistor[ R, 
            s[ V1, I1], 
            s[ V2, I2]], True, {
          eq[ 
            Plus[ 
              Times[ -1, V1], V2], 
            Times[ I1, R]], 
          eq[ I1, I2]}]],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      RowBox[{
      "\<\"Symbol\"\>", "\[InvisibleSpace]", "\<\"s\"\>", 
        "\[InvisibleSpace]", "\<\" \"\>", "\[InvisibleSpace]", 
        InterpretationBox[\(I1, I2, R, V1, V2\),
          Infix[ {I1, I2, R, V1, V2}, ","],
          Editable->False], "\[InvisibleSpace]", 
        "\<\" declared as  variable\"\>", "\[InvisibleSpace]", "\<\"s\"\>", 
        "\[InvisibleSpace]", "\<\".\"\>"}],
      SequenceForm[ "Symbol", "s", " ", 
        Infix[ {I1, I2, R, V1, V2}, ","], " declared as  variable", "s", 
        "."],
      Editable->False]], "Print"],

Cell[BoxData[
    \("\[DownTee]"\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "A ",
  StyleBox["capacitor",
    FontSlant->"Italic"],
  "  with characteristic ",
  StyleBox["C,",
    FontWeight->"Bold"],
  " input signal",
  StyleBox[" s[V0,I0]",
    FontWeight->"Bold"],
  " and output signal ",
  StyleBox["s[V1,I1] ",
    FontWeight->"Bold"],
  "satisfies the equations\n\nV0-V1=I0/C\nI0=I1"
}], "Text"],

Cell["We define", "Text"],

Cell[BoxData[
    \(Def[{C0, V1, I1, V2, I2}, \n
      \t{capacitor[C0, s[V1, I1], s[V2, I2]] \[RightArrow] True 
          \[DoubleLeftArrow] {V1 - V2 \[TildeTilde] I1/C0, 
            I1 \[TildeTilde] I2}}]\)], "Output",
  CellEditDuplicate->False,
  Evaluatable->True,
  InitializationCell->True,
  GeneratedCell->False,
  CellAutoOverwrite->False,
  FontWeight->"Bold",
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    InterpretationBox[
      \("Adding rule #"\[InvisibleSpace]1
        \[InvisibleSpace]": "\[InvisibleSpace]\(capacitor[C0, s[V1, I1], 
            s[V2, I2]] \[RightArrow] True 
          \[DoubleLeftArrow] {V1 - V2 \[TildeTilde] I1\/C0, 
            I1 \[TildeTilde] I2}\)\),
      SequenceForm[ "Adding rule #", 1, ": ", 
        RewriteRule[ 
          capacitor[ C0, 
            s[ V1, I1], 
            s[ V2, I2]], True, {
          eq[ 
            Plus[ V1, 
              Times[ -1, V2]], 
            Times[ 
              Power[ C0, -1], I1]], 
          eq[ I1, I2]}]],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      RowBox[{
      "\<\"Symbol\"\>", "\[InvisibleSpace]", "\<\"s\"\>", 
        "\[InvisibleSpace]", "\<\" \"\>", "\[InvisibleSpace]", 
        InterpretationBox[\(C0, I1, I2, V1, V2\),
          Infix[ {C0, I1, I2, V1, V2}, ","],
          Editable->False], "\[InvisibleSpace]", 
        "\<\" declared as  variable\"\>", "\[InvisibleSpace]", "\<\"s\"\>", 
        "\[InvisibleSpace]", "\<\".\"\>"}],
      SequenceForm[ "Symbol", "s", " ", 
        Infix[ {C0, I1, I2, V1, V2}, ","], " declared as  variable", "s", 
        "."],
      Editable->False]], "Print"],

Cell[BoxData[
    \("\[DownTee]"\)], "Output"],

Cell["\<\
We find useful to introduce the following abbreviations for \
expressing electrical components with given characteristic values:\
\>", 
  "Text"],

Cell[BoxData[
    \(\(RComp[R_] = \[Lambda][{sIn, sOut}, resistor[R, sIn, sOut]]; \)\)], 
  "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    \(\(CComp[C0_] := \[Lambda][{sIn, sOut}, capacitor[C0, sIn, sOut]]; 
    \)\)], "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[TextData[{
  StyleBox["RComp[R]",
    FontWeight->"Bold"],
  " describes a resistor with characteristic resistance ",
  StyleBox["R",
    FontWeight->"Bold"],
  ". \n",
  StyleBox["CComp[C0]",
    FontWeight->"Bold"],
  " describes a resistor with characteristic capacity ",
  StyleBox["C0",
    FontWeight->"Bold"],
  "."
}], "Text"],

Cell[TextData[{
  "We call these abbreviations ",
  StyleBox["component specifications",
    FontSlant->"Italic"],
  "."
}], "Text"],

Cell["\<\
We describe now the main operations used for building up electrical \
sytems from electrical components.\
\>", "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Serial connections", "Section"],

Cell[TextData[{
  " The serial connection of  ",
  StyleBox["n",
    FontSlant->"Italic"],
  " electric components is described by a predicate of the form"
}], "Text"],

Cell[TextData[{
  StyleBox["serial[ ",
    FontWeight->"Bold"],
  StyleBox["Clist, sIn, sOut ",
    FontWeight->"Bold",
    FontSlant->"Italic"],
  StyleBox["]",
    FontWeight->"Bold"]
}], "Text",
  TextAlignment->Center],

Cell["where:", "Text"],

Cell["We define serial as follows:", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(DefinedSymbol[\n\t
      serial : TyList[ElSignal\[Cross]ElSignal \[ShortRightArrow] Bool]
              \[Cross]ElSignal\[Cross]ElSignal \[ShortRightArrow] Bool]\)], 
  "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    \({serial}\)], "Output"],

Cell[BoxData[
    \(Def[{comp, compList, sIn, sOut, sBetw}, \n
      \t{serial[{}, sIn, sIn] \[RightArrow] True, \n\t\t
        serial[Cons[comp, compList], sIn, sOut] \[RightArrow] True 
          \[DoubleLeftArrow] {comp[sIn, sBetw] \[TildeTilde] True, 
            serial[compList, sBetw, sOut] \[TildeTilde] True}}]\)], "Output",
  CellEditDuplicate->False,
  Evaluatable->True,
  InitializationCell->True,
  GeneratedCell->False,
  CellAutoOverwrite->False,
  FontWeight->"Bold",
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    InterpretationBox[
      \("Adding rule #"\[InvisibleSpace]1
        \[InvisibleSpace]": "\[InvisibleSpace]\(serial[\[Bullet], sIn, sIn] 
          \[RightArrow] True\)\),
      SequenceForm[ "Adding rule #", 1, ": ", 
        RewriteRule[ 
          serial[ \[Bullet], sIn, sIn], True, {}]],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      \("Symbol"\[InvisibleSpace]""\[InvisibleSpace]" "\[InvisibleSpace]sIn
        \[InvisibleSpace]" declared as  variable"\[InvisibleSpace]""\
\[InvisibleSpace]"."\),
      SequenceForm[ 
      "Symbol", "", " ", sIn, " declared as  variable", "", "."],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      \("Adding rule #"\[InvisibleSpace]2
        \[InvisibleSpace]": "\[InvisibleSpace]\(serial[Cons[comp, compList], 
            sIn, sOut] \[RightArrow] True 
          \[DoubleLeftArrow] {comp[sIn, sBetw] \[TildeTilde] True, 
            serial[compList, sBetw, sOut] \[TildeTilde] True}\)\),
      SequenceForm[ "Adding rule #", 2, ": ", 
        RewriteRule[ 
          serial[ 
            Cons[ comp, compList], sIn, sOut], True, {
          eq[ 
            comp[ sIn, sBetw], True], 
          eq[ 
            serial[ compList, sBetw, sOut], True]}]],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      RowBox[{
      "\<\"Symbol\"\>", "\[InvisibleSpace]", "\<\"s\"\>", 
        "\[InvisibleSpace]", "\<\" \"\>", "\[InvisibleSpace]", 
        InterpretationBox[\(comp, compList, sBetw, sIn, sOut\),
          Infix[ {comp, compList, sBetw, sIn, sOut}, ","],
          Editable->False], "\[InvisibleSpace]", 
        "\<\" declared as  variable\"\>", "\[InvisibleSpace]", "\<\"s\"\>", 
        "\[InvisibleSpace]", "\<\".\"\>"}],
      SequenceForm[ "Symbol", "s", " ", 
        Infix[ {comp, compList, sBetw, sIn, sOut}, ","], 
        " declared as  variable", "s", "."],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      \("Symbol"\[InvisibleSpace]""\[InvisibleSpace]" "\[InvisibleSpace]sBetw
        \[InvisibleSpace]" declared as extra variable"\[InvisibleSpace]""\
\[InvisibleSpace]"."\),
      SequenceForm[ 
      "Symbol", "", " ", sBetw, " declared as extra variable", "", "."],
      Editable->False]], "Print"],

Cell[BoxData[
    \("\[DownTee]"\)], "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Parallel connections", "Section"],

Cell[CellGroupData[{

Cell["elJoin", "Subsection"],

Cell[TextData[{
  "For modelling parallel connections we first introduce the concept of ",
  StyleBox["electrical joint",
    FontSlant->"Italic"],
  " of electrical signals. The term "
}], "Text"],

Cell[TextData[{
  StyleBox["elJoin[{s[V1,I1],\[Ellipsis],s[Vn,In]},V]",
    FontWeight->"Bold"],
  " "
}], "Text",
  TextAlignment->Center],

Cell[TextData[{
  "describes the sum of the intensities of the electric signals ",
  StyleBox["s[V1 ,I1],\[Ellipsis],s[Vn,In] ",
    FontWeight->"Bold"],
  "under the condition that the electric signals have all the same voltage.We \
use the ",
  StyleBox["elJoin",
    FontWeight->"Bold"],
  " function symbol to describe situations where ",
  StyleBox["s[V1 ,I1],\[Ellipsis],s[Vn,In]",
    FontWeight->"Bold"],
  " are the input signals of an electrical junction. "
}], "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell["elJoin", "Subsection"],

Cell[BoxData[
    \(DefinedSymbol[
      elJoin : TyList[ElSignal]\[Cross]Float \[ShortRightArrow] Float]\)], 
  "Output",
  CellEditDuplicate->False,
  Evaluatable->True,
  InitializationCell->True,
  GeneratedCell->False,
  CellAutoOverwrite->False,
  FontWeight->"Bold",
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    \({elJoin}\)], "Output"],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(Def[{I1, I2, V, sInList}, \n
      \t{elJoin[{}, V] \[RightArrow] 0, \n\t\t
        elJoin[Cons[s[V, I1], sInList], V] \[RightArrow] I1 + I2 
          \[DoubleLeftArrow] I2 \[TildeTilde] elJoin[sInList, V]}]; \)\)], 
  "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    InterpretationBox[
      \("Adding rule #"\[InvisibleSpace]1
        \[InvisibleSpace]": "\[InvisibleSpace]\(elJoin[\[Bullet], V] 
          \[RightArrow] 0\)\),
      SequenceForm[ "Adding rule #", 1, ": ", 
        RewriteRule[ 
          elJoin[ \[Bullet], V], 0, {}]],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      \("Symbol"\[InvisibleSpace]""\[InvisibleSpace]" "\[InvisibleSpace]V
        \[InvisibleSpace]" declared as  variable"\[InvisibleSpace]""\
\[InvisibleSpace]"."\),
      SequenceForm[ "Symbol", "", " ", V, " declared as  variable", "", "."],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      \("Adding rule #"\[InvisibleSpace]2
        \[InvisibleSpace]": "\[InvisibleSpace]\(elJoin[
            Cons[s[V, I1], sInList], V] \[RightArrow] I1 + I2 
          \[DoubleLeftArrow] I2 \[TildeTilde] elJoin[sInList, V]\)\),
      SequenceForm[ "Adding rule #", 2, ": ", 
        RewriteRule[ 
          elJoin[ 
            Cons[ 
              s[ V, I1], sInList], V], 
          Plus[ I1, I2], {
          eq[ I2, 
            elJoin[ sInList, V]]}]],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      RowBox[{
      "\<\"Symbol\"\>", "\[InvisibleSpace]", "\<\"s\"\>", 
        "\[InvisibleSpace]", "\<\" \"\>", "\[InvisibleSpace]", 
        InterpretationBox[\(I1, I2, sInList, V\),
          Infix[ {I1, I2, sInList, V}, ","],
          Editable->False], "\[InvisibleSpace]", 
        "\<\" declared as  variable\"\>", "\[InvisibleSpace]", "\<\"s\"\>", 
        "\[InvisibleSpace]", "\<\".\"\>"}],
      SequenceForm[ "Symbol", "s", " ", 
        Infix[ {I1, I2, sInList, V}, ","], " declared as  variable", "s", 
        "."],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      \("Symbol"\[InvisibleSpace]""\[InvisibleSpace]" "\[InvisibleSpace]I2
        \[InvisibleSpace]" declared as extra variable"\[InvisibleSpace]""\
\[InvisibleSpace]"."\),
      SequenceForm[ 
      "Symbol", "", " ", I2, " declared as extra variable", "", "."],
      Editable->False]], "Print"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["junction", "Subsection"],

Cell[TextData[{
  "Next, we define the ",
  StyleBox["junction",
    FontSlant->"Italic"],
  " of a list of input signals with a list of output signals."
}], "Text"],

Cell[BoxData[
    \(DefinedSymbol[
      junction : 
        TyList[ElSignal]\[Cross]TyList[ElSignal] \[ShortRightArrow] Bool]\)], 
  "Output",
  CellEditDuplicate->False,
  Evaluatable->True,
  InitializationCell->True,
  GeneratedCell->False,
  CellAutoOverwrite->False,
  FontWeight->"Bold",
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    \({junction}\)], "Output"],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(Def[{sInList, sOutList, V}, \n
      \t{junction[sInList, sOutList] \[RightArrow] True \[DoubleLeftArrow] 
          elJoin[sInList, V] \[TildeTilde] elJoin[sOutList, V]}]; \)\)], 
  "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    InterpretationBox[
      \("Adding rule #"\[InvisibleSpace]1
        \[InvisibleSpace]": "\[InvisibleSpace]\(junction[sInList, sOutList] 
          \[RightArrow] True \[DoubleLeftArrow] 
          elJoin[sInList, V] \[TildeTilde] elJoin[sOutList, V]\)\),
      SequenceForm[ "Adding rule #", 1, ": ", 
        RewriteRule[ 
          junction[ sInList, sOutList], True, {
          eq[ 
            elJoin[ sInList, V], 
            elJoin[ sOutList, V]]}]],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      RowBox[{
      "\<\"Symbol\"\>", "\[InvisibleSpace]", "\<\"s\"\>", 
        "\[InvisibleSpace]", "\<\" \"\>", "\[InvisibleSpace]", 
        InterpretationBox[\(sInList, sOutList, V\),
          Infix[ {sInList, sOutList, V}, ","],
          Editable->False], "\[InvisibleSpace]", 
        "\<\" declared as  variable\"\>", "\[InvisibleSpace]", "\<\"s\"\>", 
        "\[InvisibleSpace]", "\<\".\"\>"}],
      SequenceForm[ "Symbol", "s", " ", 
        Infix[ {sInList, sOutList, V}, ","], " declared as  variable", "s", 
        "."],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      \("Symbol"\[InvisibleSpace]""\[InvisibleSpace]" "\[InvisibleSpace]V
        \[InvisibleSpace]" declared as extra variable"\[InvisibleSpace]""\
\[InvisibleSpace]"."\),
      SequenceForm[ 
      "Symbol", "", " ", V, " declared as extra variable", "", "."],
      Editable->False]], "Print"]
}, Open  ]],

Cell[CellGroupData[{

Cell["parallel", "Subsubsection"],

Cell["\<\
 The parallel connection of  electric components is described by \
the predicate\
\>", "Text"],

Cell[TextData[{
  StyleBox["parallel[ ",
    FontWeight->"Bold"],
  StyleBox["Clist, sIn, sOut ",
    FontWeight->"Bold",
    FontSlant->"Italic"],
  StyleBox["]",
    FontWeight->"Bold"]
}], "Text",
  TextAlignment->Center],

Cell["where:", "Text"],

Cell[TextData[{
  "\[Diamond] ",
  StyleBox["CList",
    FontWeight->"Bold",
    FontSlant->"Italic"],
  " is the list with the specifications of the components which are connected \
in parallel,\n\[Diamond] ",
  StyleBox["sIn",
    FontWeight->"Bold",
    FontSlant->"Italic"],
  " is the input signal to the parallel connection\n\[Diamond] ",
  StyleBox["sOut",
    FontWeight->"Bold",
    FontSlant->"Italic"],
  " is the output signal from the parallel connection"
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(DefinedSymbol[\n\t
      parallel : 
        TyList[ElSignal\[Cross]ElSignal \[ShortRightArrow] Bool]
              \[Cross]ElSignal\[Cross]ElSignal \[ShortRightArrow] Bool]\)], 
  "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    \({parallel}\)], "Output"],

Cell[BoxData[
    \(Def[{comp, compList, sIn, sIn1, sIn2, sOut, sOut1, sOut2}, \n
      \t{parallel[{comp}, sIn, sOut] \[RightArrow] comp[sIn, sOut], \n\t\t
        parallel[Cons[comp, compList], sIn, sOut] \[RightArrow] True 
          \[DoubleLeftArrow] {comp[sIn1, sOut1] \[TildeTilde] True, \n\t\t\t\t
            parallel[compList, sIn2, sOut2] \[TildeTilde] True, 
            junction[{sIn}, {sIn1, sIn2}] \[TildeTilde] True, 
            junction[{sOut1, sOut2}, {sOut}] \[TildeTilde] True}}]\)], 
  "Output",
  CellEditDuplicate->False,
  Evaluatable->True,
  InitializationCell->True,
  GeneratedCell->False,
  CellAutoOverwrite->False,
  FontWeight->"Bold",
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    InterpretationBox[
      \("Adding rule #"\[InvisibleSpace]1
        \[InvisibleSpace]": "\[InvisibleSpace]\(parallel[
            Cons[comp, \[Bullet]], sIn, sOut] \[RightArrow] 
          comp[sIn, sOut]\)\),
      SequenceForm[ "Adding rule #", 1, ": ", 
        RewriteRule[ 
          parallel[ 
            Cons[ comp, \[Bullet]], sIn, sOut], 
          comp[ sIn, sOut], {}]],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      RowBox[{
      "\<\"Symbol\"\>", "\[InvisibleSpace]", "\<\"s\"\>", 
        "\[InvisibleSpace]", "\<\" \"\>", "\[InvisibleSpace]", 
        InterpretationBox[\(comp, sIn, sOut\),
          Infix[ {comp, sIn, sOut}, ","],
          Editable->False], "\[InvisibleSpace]", 
        "\<\" declared as  variable\"\>", "\[InvisibleSpace]", "\<\"s\"\>", 
        "\[InvisibleSpace]", "\<\".\"\>"}],
      SequenceForm[ "Symbol", "s", " ", 
        Infix[ {comp, sIn, sOut}, ","], " declared as  variable", "s", "."],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      \("Adding rule #"\[InvisibleSpace]2
        \[InvisibleSpace]": "\[InvisibleSpace]\(parallel[
            Cons[comp, compList], sIn, sOut] \[RightArrow] True 
          \[DoubleLeftArrow] {comp[sIn1, sOut1] \[TildeTilde] True, 
            parallel[compList, sIn2, sOut2] \[TildeTilde] True, 
            junction[Cons[sIn, \[Bullet]], Cons[sIn1, Cons[sIn2, \[Bullet]]]] 
              \[TildeTilde] True, 
            junction[Cons[sOut1, Cons[sOut2, \[Bullet]]], 
                Cons[sOut, \[Bullet]]] \[TildeTilde] True}\)\),
      SequenceForm[ "Adding rule #", 2, ": ", 
        RewriteRule[ 
          parallel[ 
            Cons[ comp, compList], sIn, sOut], True, {
          eq[ 
            comp[ sIn1, sOut1], True], 
          eq[ 
            parallel[ compList, sIn2, sOut2], True], 
          eq[ 
            junction[ 
              Cons[ sIn, \[Bullet]], 
              Cons[ sIn1, 
                Cons[ sIn2, \[Bullet]]]], True], 
          eq[ 
            junction[ 
              Cons[ sOut1, 
                Cons[ sOut2, \[Bullet]]], 
              Cons[ sOut, \[Bullet]]], True]}]],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      RowBox[{
      "\<\"Symbol\"\>", "\[InvisibleSpace]", "\<\"s\"\>", 
        "\[InvisibleSpace]", "\<\" \"\>", "\[InvisibleSpace]", 
        InterpretationBox[\(comp, compList, sIn, sIn1, sIn2, sOut, sOut1, 
          sOut2\),
          Infix[ {comp, compList, sIn, sIn1, sIn2, sOut, sOut1, sOut2}, ","],
          Editable->False], "\[InvisibleSpace]", 
        "\<\" declared as  variable\"\>", "\[InvisibleSpace]", "\<\"s\"\>", 
        "\[InvisibleSpace]", "\<\".\"\>"}],
      SequenceForm[ "Symbol", "s", " ", 
        Infix[ {comp, compList, sIn, sIn1, sIn2, sOut, sOut1, sOut2}, ","], 
        " declared as  variable", "s", "."],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      RowBox[{
      "\<\"Symbol\"\>", "\[InvisibleSpace]", "\<\"s\"\>", 
        "\[InvisibleSpace]", "\<\" \"\>", "\[InvisibleSpace]", 
        InterpretationBox[\(sIn1, sIn2, sOut1, sOut2\),
          Infix[ {sIn1, sIn2, sOut1, sOut2}, ","],
          Editable->False], "\[InvisibleSpace]", 
        "\<\" declared as extra variable\"\>", "\[InvisibleSpace]", 
        "\<\"s\"\>", "\[InvisibleSpace]", "\<\".\"\>"}],
      SequenceForm[ "Symbol", "s", " ", 
        Infix[ {sIn1, sIn2, sOut1, sOut2}, ","], 
        " declared as extra variable", "s", "."],
      Editable->False]], "Print"],

Cell[BoxData[
    \("\[DownTee]"\)], "Output"]
}, Open  ]],

Cell["\<\
We can now introduce abbreviations for specifying serial and \
parallel connections. The specification of the serial connection of the \
electrical components of a list compList can be specified as follows:\
\>", 
  "Text"],

Cell[BoxData[
    \(\(SConnect[compList_] = 
      \[Lambda][{sIn, sOut}, serial[compList, sIn, sOut]]; \)\)], "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell["\<\
In a similar way can be specified the parallel connection of a list \
of electrical components.\
\>", "Text"],

Cell[BoxData[
    \(\(PConnect[compList_] = 
      \[Lambda][{sIn, sOut}, parallel[compList, sIn, sOut]]; \)\)], "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["End", "Section"],

Cell[TextData[{
  "The ",
  StyleBox["End",
    FontWeight->"Bold"],
  " command closes the last context which was started with a ",
  StyleBox["Begin",
    FontWeight->"Bold"],
  " command. In our case, the next command closes the Private context of the \
package ",
  StyleBox["ElDemo.m",
    FontWeight->"Bold"],
  "."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(End[]\)], "Input",
  PageWidth->Infinity,
  InitializationCell->True,
  ShowSpecialCharacters->False,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    \("ElDemo`Private`"\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "The last command of a package is ",
  StyleBox["EndPackage",
    FontWeight->"Bold"],
  "."
}], "Text"],

Cell[BoxData[
    \(EndPackage[]\)], "Input",
  PageWidth->Infinity,
  InitializationCell->True,
  ShowSpecialCharacters->False,
  FontColor->RGBColor[0, 0, 0.500008]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Package demo", "Section"],

Cell["\<\
The following commands illustrate the use of the package ElDemo.m.
First, load the package:\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(<< ElDemo`\)], "Input"],

Cell[BoxData[
    \("Copyright 1998-1999 Institute RISC-Linz and Institute of Information \
Sciences and Electronics, Tsukuba."\)], "Print"],

Cell[BoxData[
    InterpretationBox[
      \("Adding rule #"\[InvisibleSpace]1
        \[InvisibleSpace]": "\[InvisibleSpace]\(resistor[R, s[V1, I1], 
            s[V2, I2]] \[RightArrow] True 
          \[DoubleLeftArrow] {\(-V1\) + V2 \[TildeTilde] I1\ R, 
            I1 \[TildeTilde] I2}\)\),
      SequenceForm[ "Adding rule #", 1, ": ", 
        RewriteRule[ 
          resistor[ R, 
            s[ V1, I1], 
            s[ V2, I2]], True, {
          eq[ 
            Plus[ 
              Times[ -1, V1], V2], 
            Times[ I1, R]], 
          eq[ I1, I2]}]],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      RowBox[{
      "\<\"Symbol\"\>", "\[InvisibleSpace]", "\<\"s\"\>", 
        "\[InvisibleSpace]", "\<\" \"\>", "\[InvisibleSpace]", 
        InterpretationBox[\(I1, I2, R, V1, V2\),
          Infix[ {I1, I2, R, V1, V2}, ","],
          Editable->False], "\[InvisibleSpace]", 
        "\<\" declared as  variable\"\>", "\[InvisibleSpace]", "\<\"s\"\>", 
        "\[InvisibleSpace]", "\<\".\"\>"}],
      SequenceForm[ "Symbol", "s", " ", 
        Infix[ {I1, I2, R, V1, V2}, ","], " declared as  variable", "s", 
        "."],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      \("Adding rule #"\[InvisibleSpace]1
        \[InvisibleSpace]": "\[InvisibleSpace]\(capacitor[C0, s[V1, I1], 
            s[V2, I2]] \[RightArrow] True 
          \[DoubleLeftArrow] {V1 - V2 \[TildeTilde] I1\/C0, 
            I1 \[TildeTilde] I2}\)\),
      SequenceForm[ "Adding rule #", 1, ": ", 
        RewriteRule[ 
          capacitor[ C0, 
            s[ V1, I1], 
            s[ V2, I2]], True, {
          eq[ 
            Plus[ V1, 
              Times[ -1, V2]], 
            Times[ 
              Power[ C0, -1], I1]], 
          eq[ I1, I2]}]],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      RowBox[{
      "\<\"Symbol\"\>", "\[InvisibleSpace]", "\<\"s\"\>", 
        "\[InvisibleSpace]", "\<\" \"\>", "\[InvisibleSpace]", 
        InterpretationBox[\(C0, I1, I2, V1, V2\),
          Infix[ {C0, I1, I2, V1, V2}, ","],
          Editable->False], "\[InvisibleSpace]", 
        "\<\" declared as  variable\"\>", "\[InvisibleSpace]", "\<\"s\"\>", 
        "\[InvisibleSpace]", "\<\".\"\>"}],
      SequenceForm[ "Symbol", "s", " ", 
        Infix[ {C0, I1, I2, V1, V2}, ","], " declared as  variable", "s", 
        "."],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      \("Adding rule #"\[InvisibleSpace]1
        \[InvisibleSpace]": "\[InvisibleSpace]\(serial[\[Bullet], sIn, sIn] 
          \[RightArrow] True\)\),
      SequenceForm[ "Adding rule #", 1, ": ", 
        RewriteRule[ 
          serial[ \[Bullet], sIn, sIn], True, {}]],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      \("Symbol"\[InvisibleSpace]""\[InvisibleSpace]" "\[InvisibleSpace]sIn
        \[InvisibleSpace]" declared as  variable"\[InvisibleSpace]""\
\[InvisibleSpace]"."\),
      SequenceForm[ 
      "Symbol", "", " ", sIn, " declared as  variable", "", "."],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      \("Adding rule #"\[InvisibleSpace]2
        \[InvisibleSpace]": "\[InvisibleSpace]\(serial[Cons[comp, compList], 
            sIn, sOut] \[RightArrow] True 
          \[DoubleLeftArrow] {comp[sIn, sBetw] \[TildeTilde] True, 
            serial[compList, sBetw, sOut] \[TildeTilde] True}\)\),
      SequenceForm[ "Adding rule #", 2, ": ", 
        RewriteRule[ 
          serial[ 
            Cons[ comp, compList], sIn, sOut], True, {
          eq[ 
            comp[ sIn, sBetw], True], 
          eq[ 
            serial[ compList, sBetw, sOut], True]}]],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      RowBox[{
      "\<\"Symbol\"\>", "\[InvisibleSpace]", "\<\"s\"\>", 
        "\[InvisibleSpace]", "\<\" \"\>", "\[InvisibleSpace]", 
        InterpretationBox[\(comp, compList, sBetw, sIn, sOut\),
          Infix[ {comp, compList, sBetw, sIn, sOut}, ","],
          Editable->False], "\[InvisibleSpace]", 
        "\<\" declared as  variable\"\>", "\[InvisibleSpace]", "\<\"s\"\>", 
        "\[InvisibleSpace]", "\<\".\"\>"}],
      SequenceForm[ "Symbol", "s", " ", 
        Infix[ {comp, compList, sBetw, sIn, sOut}, ","], 
        " declared as  variable", "s", "."],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      \("Symbol"\[InvisibleSpace]""\[InvisibleSpace]" "\[InvisibleSpace]sBetw
        \[InvisibleSpace]" declared as extra variable"\[InvisibleSpace]""\
\[InvisibleSpace]"."\),
      SequenceForm[ 
      "Symbol", "", " ", sBetw, " declared as extra variable", "", "."],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      \("Adding rule #"\[InvisibleSpace]1
        \[InvisibleSpace]": "\[InvisibleSpace]\(elJoin[\[Bullet], V] 
          \[RightArrow] 0\)\),
      SequenceForm[ "Adding rule #", 1, ": ", 
        RewriteRule[ 
          elJoin[ \[Bullet], V], 0, {}]],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      \("Symbol"\[InvisibleSpace]""\[InvisibleSpace]" "\[InvisibleSpace]V
        \[InvisibleSpace]" declared as  variable"\[InvisibleSpace]""\
\[InvisibleSpace]"."\),
      SequenceForm[ "Symbol", "", " ", V, " declared as  variable", "", "."],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      \("Adding rule #"\[InvisibleSpace]2
        \[InvisibleSpace]": "\[InvisibleSpace]\(elJoin[
            Cons[s[V, I1], sInList], V] \[RightArrow] I1 + I2 
          \[DoubleLeftArrow] I2 \[TildeTilde] elJoin[sInList, V]\)\),
      SequenceForm[ "Adding rule #", 2, ": ", 
        RewriteRule[ 
          elJoin[ 
            Cons[ 
              s[ V, I1], sInList], V], 
          Plus[ I1, I2], {
          eq[ I2, 
            elJoin[ sInList, V]]}]],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      RowBox[{
      "\<\"Symbol\"\>", "\[InvisibleSpace]", "\<\"s\"\>", 
        "\[InvisibleSpace]", "\<\" \"\>", "\[InvisibleSpace]", 
        InterpretationBox[\(I1, I2, sInList, V\),
          Infix[ {I1, I2, sInList, V}, ","],
          Editable->False], "\[InvisibleSpace]", 
        "\<\" declared as  variable\"\>", "\[InvisibleSpace]", "\<\"s\"\>", 
        "\[InvisibleSpace]", "\<\".\"\>"}],
      SequenceForm[ "Symbol", "s", " ", 
        Infix[ {I1, I2, sInList, V}, ","], " declared as  variable", "s", 
        "."],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      \("Symbol"\[InvisibleSpace]""\[InvisibleSpace]" "\[InvisibleSpace]I2
        \[InvisibleSpace]" declared as extra variable"\[InvisibleSpace]""\
\[InvisibleSpace]"."\),
      SequenceForm[ 
      "Symbol", "", " ", I2, " declared as extra variable", "", "."],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      \("Adding rule #"\[InvisibleSpace]1
        \[InvisibleSpace]": "\[InvisibleSpace]\(junction[sInList, sOutList] 
          \[RightArrow] True \[DoubleLeftArrow] 
          elJoin[sInList, V] \[TildeTilde] elJoin[sOutList, V]\)\),
      SequenceForm[ "Adding rule #", 1, ": ", 
        RewriteRule[ 
          junction[ sInList, sOutList], True, {
          eq[ 
            elJoin[ sInList, V], 
            elJoin[ sOutList, V]]}]],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      RowBox[{
      "\<\"Symbol\"\>", "\[InvisibleSpace]", "\<\"s\"\>", 
        "\[InvisibleSpace]", "\<\" \"\>", "\[InvisibleSpace]", 
        InterpretationBox[\(sInList, sOutList, V\),
          Infix[ {sInList, sOutList, V}, ","],
          Editable->False], "\[InvisibleSpace]", 
        "\<\" declared as  variable\"\>", "\[InvisibleSpace]", "\<\"s\"\>", 
        "\[InvisibleSpace]", "\<\".\"\>"}],
      SequenceForm[ "Symbol", "s", " ", 
        Infix[ {sInList, sOutList, V}, ","], " declared as  variable", "s", 
        "."],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      \("Symbol"\[InvisibleSpace]""\[InvisibleSpace]" "\[InvisibleSpace]V
        \[InvisibleSpace]" declared as extra variable"\[InvisibleSpace]""\
\[InvisibleSpace]"."\),
      SequenceForm[ 
      "Symbol", "", " ", V, " declared as extra variable", "", "."],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      \("Adding rule #"\[InvisibleSpace]1
        \[InvisibleSpace]": "\[InvisibleSpace]\(parallel[
            Cons[comp, \[Bullet]], sIn, sOut] \[RightArrow] 
          comp[sIn, sOut]\)\),
      SequenceForm[ "Adding rule #", 1, ": ", 
        RewriteRule[ 
          parallel[ 
            Cons[ comp, \[Bullet]], sIn, sOut], 
          comp[ sIn, sOut], {}]],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      RowBox[{
      "\<\"Symbol\"\>", "\[InvisibleSpace]", "\<\"s\"\>", 
        "\[InvisibleSpace]", "\<\" \"\>", "\[InvisibleSpace]", 
        InterpretationBox[\(comp, sIn, sOut\),
          Infix[ {comp, sIn, sOut}, ","],
          Editable->False], "\[InvisibleSpace]", 
        "\<\" declared as  variable\"\>", "\[InvisibleSpace]", "\<\"s\"\>", 
        "\[InvisibleSpace]", "\<\".\"\>"}],
      SequenceForm[ "Symbol", "s", " ", 
        Infix[ {comp, sIn, sOut}, ","], " declared as  variable", "s", "."],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      \("Adding rule #"\[InvisibleSpace]2
        \[InvisibleSpace]": "\[InvisibleSpace]\(parallel[
            Cons[comp, compList], sIn, sOut] \[RightArrow] True 
          \[DoubleLeftArrow] {comp[sIn1, sOut1] \[TildeTilde] True, 
            parallel[compList, sIn2, sOut2] \[TildeTilde] True, 
            junction[Cons[sIn, \[Bullet]], Cons[sIn1, Cons[sIn2, \[Bullet]]]] 
              \[TildeTilde] True, 
            junction[Cons[sOut1, Cons[sOut2, \[Bullet]]], 
                Cons[sOut, \[Bullet]]] \[TildeTilde] True}\)\),
      SequenceForm[ "Adding rule #", 2, ": ", 
        RewriteRule[ 
          parallel[ 
            Cons[ comp, compList], sIn, sOut], True, {
          eq[ 
            comp[ sIn1, sOut1], True], 
          eq[ 
            parallel[ compList, sIn2, sOut2], True], 
          eq[ 
            junction[ 
              Cons[ sIn, \[Bullet]], 
              Cons[ sIn1, 
                Cons[ sIn2, \[Bullet]]]], True], 
          eq[ 
            junction[ 
              Cons[ sOut1, 
                Cons[ sOut2, \[Bullet]]], 
              Cons[ sOut, \[Bullet]]], True]}]],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      RowBox[{
      "\<\"Symbol\"\>", "\[InvisibleSpace]", "\<\"s\"\>", 
        "\[InvisibleSpace]", "\<\" \"\>", "\[InvisibleSpace]", 
        InterpretationBox[\(comp, compList, sIn, sIn1, sIn2, sOut, sOut1, 
          sOut2\),
          Infix[ {comp, compList, sIn, sIn1, sIn2, sOut, sOut1, sOut2}, ","],
          Editable->False], "\[InvisibleSpace]", 
        "\<\" declared as  variable\"\>", "\[InvisibleSpace]", "\<\"s\"\>", 
        "\[InvisibleSpace]", "\<\".\"\>"}],
      SequenceForm[ "Symbol", "s", " ", 
        Infix[ {comp, compList, sIn, sIn1, sIn2, sOut, sOut1, sOut2}, ","], 
        " declared as  variable", "s", "."],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      RowBox[{
      "\<\"Symbol\"\>", "\[InvisibleSpace]", "\<\"s\"\>", 
        "\[InvisibleSpace]", "\<\" \"\>", "\[InvisibleSpace]", 
        InterpretationBox[\(sIn1, sIn2, sOut1, sOut2\),
          Infix[ {sIn1, sIn2, sOut1, sOut2}, ","],
          Editable->False], "\[InvisibleSpace]", 
        "\<\" declared as extra variable\"\>", "\[InvisibleSpace]", 
        "\<\"s\"\>", "\[InvisibleSpace]", "\<\".\"\>"}],
      SequenceForm[ "Symbol", "s", " ", 
        Infix[ {sIn1, sIn2, sOut1, sOut2}, ","], 
        " declared as extra variable", "s", "."],
      Editable->False]], "Print"]
}, Open  ]],

Cell["and start the CFLP system:", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(StartCFLP[]\)], "Input"],

Cell[BoxData[
    \("connecting\[Ellipsis]"\)], "Print"],

Cell[BoxData[
    \("done."\)], "Print"]
}, Open  ]],

Cell["\<\
In this way, the equational theory of the CFLP system is extended \
with the rewrite rules defined in the CFLP package which is loaded.
The following types of problems can be solved now with CFLP:\
\>", "Text"],

Cell[CellGroupData[{

Cell["Problem 1", "Subsection"],

Cell[TextData[{
  "Find the dependency of the output signal ",
  StyleBox["s[V2,I2]",
    FontWeight->"Bold"],
  " on the input signal ",
  StyleBox["s[V1,I1]",
    FontWeight->"Bold"],
  " for a parallel connection of 3 resistors with given characteristics",
  StyleBox[" R1,R2,R3.",
    FontWeight->"Bold"]
}], "Text"],

Cell["\<\
                _____     
            ___|  R1 |____
           |   |_____|    |   
 s[V1,I1]  |    _____     | s[V2,I2]
----->-----|___|  R2 |____|----->----- 
           |   |_____|    |
           |    _____     |
           |___| R3  |____|
               |_____|\
\>", "Text",
  Editable->False,
  CellFrame->True,
  FontFamily->"Fixed",
  FontWeight->"Bold",
  Background->GrayLevel[0.900008]],

Cell[CellGroupData[{

Cell[BoxData[
    \(TSolve[
      parallel[{RComp[R1], RComp[R2], RComp[R3]}, s[V1, I1], s[V2, I2]] 
        \[TildeTilde] True, {I2, V2}, 
      Constructor -> {I1 : Float, V1 : Float, R1 : Float, R2 : Float, 
          R3 : Float}]\)], "Input"],

Cell[BoxData[
    \({{I2 \[Rule] I1, 
        V2 \[Rule] \(I1\ R1\ R2\ R3\)\/\(R2\ R3 + R1\ \((R2 + R3)\)\) + V1}}
      \)], "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Problem 2", "Subsection"],

Cell[TextData[{
  "Find the dependency between the input signal ",
  StyleBox["s[V1,I1]",
    FontWeight->"Bold"],
  " and the output signal ",
  StyleBox["s[V2,I2]",
    FontWeight->"Bold"],
  " for the circuit depicted in the figure below:"
}], "Text"],

Cell["\<\
                 ____     ____
            ____| R1 |___| C1 |___
 s[V1,I1]  |    |____|   |____|   |  s[V2,I2]
----->-----|          ____        |----->-----
           |_________| R2 |_______|
                     |____|\
\>", "Text",
  Editable->False,
  CellFrame->True,
  FontFamily->"Fixed",
  FontWeight->"Bold",
  Background->GrayLevel[0.900008]],

Cell[CellGroupData[{

Cell[BoxData[
    \(TSolve[\n\tSConnect[{RComp[R1], CComp[C1]}], 
      \(RComp[R2]\)[s[V1, I1], s[V, I]] \[TildeTilde] True, \n\t
      \(PConnect[{SConnect[{RComp[R1], CComp[C1]}], RComp[R2]}]\)[s[V1, I1], 
          s[V2, I2]] \[TildeTilde] True, {I2, V2}, 
      Constructor -> {I1 : Float, V1 : Float, R1 : Float, R2 : Float, 
          C1 : Float}]\)], "Input"],

Cell[BoxData[
    \({{I2 \[Rule] I1, 
        V2 \[Rule] 
          \(I1\ \((\(-1\) + C1\ R1)\)\ R2\)\/\(\(-1\) + C1\ \((R1 + R2)\)\) + 
            V1}}\)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(TSolve[\n\t
      \(PConnect[{SConnect[{RComp[R1], CComp[C1]}], RComp[R2]}]\)[s[V1, I1], 
          s[V2, I2]] \[TildeTilde] True, {I1, V1}, 
      Constructor -> {I2 : Float, V2 : Float, R1 : Float, R2 : Float, 
          C1 : Float}]\)], "Input"],

Cell[BoxData[
    \({{I1 \[Rule] I2, 
        V1 \[Rule] 
          \(I2\ \((1 - C1\ R1)\)\ R2\)\/\(\(-1\) + C1\ \((R1 + R2)\)\) + V2}}
      \)], "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Problem 3", "Subsection"],

Cell["\<\
Find the characteristic of a capacitor that can replace a parallel \
connection of 3 capacitors.\
\>", "Text"],

Cell[TextData[
"                _____     \n            ___|  C1 |____\n           |   \
|_____|    |   \n s[V1,I1]  |    _____     | s[V2,I2]\n----->-----|___|  C2 \
|____|----->----- \n           |   |_____|    |\n           |    _____     |\n\
           |___| C3  |____|\n               |_____|\n                  \
\[DoubleUpArrow]\n                  \[DoubleDownArrow]\ns[V1,I1]        _____ \
     s[V2,I2]\n----->---------|  C0 |---------->-----\n                ----- \
"], "Text",
  Editable->False,
  CellFrame->True,
  FontFamily->"Fixed",
  FontWeight->"Bold",
  Background->GrayLevel[0.900008]],

Cell[CellGroupData[{

Cell[BoxData[
    \(TSolve[\n\t
      parallel[{CComp[C1], CComp[C2], CComp[C3]}, s[V1, I1], s[V2, I2]] 
        \[TildeTilde] capacitor[C0, s[V1, I1], s[V2, I2]], {I1, V1, C0}, 
      Constructor -> {I2 : Float, V2 : Float, C1, C2, C3}\n]\)], "Input"],

Cell[BoxData[
    \({{C0 \[Rule] C1 + C2 + C3, I1 \[Rule] I2, 
        V1 \[Rule] \(I2 + \((C1 + C2 + C3)\)\ V2\)\/\(C1 + C2 + C3\)}}\)], 
  "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Problem 4", "Subsection"],

Cell["\<\
Find the characteristic of a capacitor that can replace a serial \
connection of 3 capacitors.\
\>", "Text"],

Cell[TextData[
"s[V1,I1]  ____    ____    ____   s[V2,I2]\n----->---| C1 |--| C2 |--| C3 \
|------>-----\n          ----    ----    ----\n                   \
\[DoubleUpArrow]\n                   \[DoubleDownArrow]\ns[V1,I1]         \
_____          s[V2,I2]\n----->----------|  C0 |-------------->-----\n        \
         ----- "], "Text",
  Editable->False,
  CellFrame->True,
  FontFamily->"Fixed",
  FontWeight->"Bold",
  Background->GrayLevel[0.900008]],

Cell[CellGroupData[{

Cell[BoxData[
    \(TSolve[\n\t
      serial[{CComp[C1], CComp[C2], CComp[C3]}, s[V1, I1], s[V2, I2]] 
        \[TildeTilde] capacitor[C0, s[V1, I1], s[V2, I2]], {I1, V1, C0}, 
      Constructor -> {I2 : Float, V2 : Float, C1, C2, C3}, \n\t
      TypeCheck -> False\n]\)], "Input"],

Cell[BoxData[
    \({{C0 \[Rule] \(C1\ C2\ C3\)\/\(C2\ C3 + C1\ \((C2 + C3)\)\), 
        I1 \[Rule] I2, V1 \[Rule] I2\/C1 + I2\/C2 + I2\/C3 + V2}}\)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(CSConnect[]\)], "Input"],

Cell[BoxData[
    \("connection to scheduler established."\)], "Print"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Problem 5", "Subsection"],

Cell[TextData[{
  "Find the the characteristic ",
  StyleBox["R",
    FontWeight->"Bold"],
  " of a resistor such that  for a capacitor with specified capacity ",
  StyleBox["C0",
    FontWeight->"Bold"],
  ", the specification depicted below holds:"
}], "Text"],

Cell["\<\
                  ____    
            _____| R  |_____
s[220,I0]  |     |____|     |  s[110,I0]
----->-----|      ____      |------>------
           |_____| C0 |_____| 
                 |____|\
\>", "Text",
  Editable->False,
  CellFrame->True,
  FontFamily->"Fixed",
  FontWeight->"Bold",
  Background->GrayLevel[0.900008]],

Cell[CellGroupData[{

Cell[BoxData[
    \(TSolve[
      parallel[{RComp[R], CComp[C0]}, s[220, I0], s[110, I0]] \[TildeTilde] 
        True, {R : Float}, Constructor -> {I0 : Float, C0 : Float}, \n\t
      TypeCheck -> False\n]\)], "Input"],

Cell[BoxData[
    \({{R \[Rule] 110\/\(\(-I\) + 110\ C0\)}}\)], "Output"]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}, Open  ]]
},
FrontEndVersion->"X 3.0",
ScreenRectangle->{{0, 800}, {0, 600}},
AutoGeneratedPackage->Automatic,
WindowToolbars->"RulerBar",
WindowSize->{711, 482},
WindowMargins->{{Automatic, 1}, {-13, Automatic}},
PrintingPageRange->{Automatic, Automatic},
PrintingOptions->{"PaperSize"->{612, 792},
"PaperOrientation"->"Portrait",
"Magnification"->1},
Magnification->1
]


(***********************************************************************
Cached data follows.  If you edit this Notebook file directly, not using
Mathematica, you must remove the line containing CacheID at the top of 
the file.  The cache data will then be recreated when you save this file 
from within Mathematica.
***********************************************************************)

(*CellTagsOutline
CellTagsIndex->{}
*)

(*CellTagsIndex
CellTagsIndex->{}
*)

(*NotebookFileOutline
Notebook[{

Cell[CellGroupData[{
Cell[1731, 51, 83, 1, 102, "Title"],
Cell[1817, 54, 197, 5, 50, "Text",
  InitializationCell->True],

Cell[CellGroupData[{
Cell[2039, 63, 64, 1, 42, "Subsubsection",
  InitializationCell->True],
Cell[2106, 66, 2326, 40, 698, "Text",
  InitializationCell->True]
}, Closed]],

Cell[CellGroupData[{
Cell[4469, 111, 39, 0, 53, "Subtitle"],
Cell[4511, 113, 609, 8, 104, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[5157, 126, 34, 0, 64, "Subtitle"],
Cell[5194, 128, 466, 12, 68, "Text"],
Cell[5663, 142, 534, 16, 68, "Text"],

Cell[CellGroupData[{
Cell[6222, 162, 24, 0, 54, "Section"],
Cell[6249, 164, 393, 10, 68, "Text"],
Cell[6645, 176, 150, 3, 27, "Input",
  InitializationCell->True],
Cell[6798, 181, 437, 14, 50, "Text"],
Cell[7238, 197, 783, 25, 176, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[8058, 227, 24, 0, 54, "Section"],
Cell[8085, 229, 244, 5, 43, "Input",
  InitializationCell->True],
Cell[8332, 236, 252, 5, 43, "Input",
  InitializationCell->True],
Cell[8587, 243, 279, 6, 43, "Input",
  InitializationCell->True],
Cell[8869, 251, 315, 6, 59, "Input",
  InitializationCell->True],
Cell[9187, 259, 202, 5, 27, "Input",
  InitializationCell->True],
Cell[9392, 266, 201, 5, 27, "Input",
  InitializationCell->True],
Cell[9596, 273, 129, 3, 27, "Input",
  InitializationCell->True],
Cell[9728, 278, 359, 7, 59, "Input",
  InitializationCell->True],
Cell[10090, 287, 363, 7, 59, "Input",
  InitializationCell->True],
Cell[10456, 296, 342, 7, 59, "Input",
  InitializationCell->True],
Cell[10801, 305, 345, 7, 59, "Input",
  InitializationCell->True],
Cell[11149, 314, 190, 6, 32, "Text"],
Cell[11342, 322, 183, 5, 27, "Input",
  InitializationCell->True],
Cell[11528, 329, 172, 6, 32, "Text"],
Cell[11703, 337, 154, 3, 27, "Input",
  InitializationCell->True],
Cell[11860, 342, 253, 7, 50, "Text"],
Cell[12116, 351, 380, 12, 50, "Text"],
Cell[12499, 365, 120, 3, 32, "Text"],

Cell[CellGroupData[{
Cell[12644, 372, 180, 5, 27, "Input",
  InitializationCell->True],
Cell[12827, 379, 44, 1, 27, "Output"]
}, Open  ]],
Cell[12886, 383, 187, 8, 32, "Text"],

Cell[CellGroupData[{
Cell[13098, 395, 288, 8, 27, "Input",
  InitializationCell->True],
Cell[13389, 405, 37, 1, 27, "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[13475, 412, 40, 0, 54, "Section"],
Cell[13518, 414, 310, 7, 50, "Text"],
Cell[13831, 423, 681, 24, 86, "Text"],
Cell[14515, 449, 169, 4, 32, "Text"],

Cell[CellGroupData[{
Cell[14709, 457, 307, 8, 75, "Input",
  InitializationCell->True],
Cell[15019, 467, 55, 1, 27, "Output"]
}, Open  ]],
Cell[15089, 471, 147, 3, 32, "Text"],
Cell[15239, 476, 325, 14, 32, "Text"],
Cell[15567, 492, 48, 3, 50, "Text"],
Cell[15618, 497, 25, 0, 32, "Text"],

Cell[CellGroupData[{
Cell[15668, 501, 293, 6, 43, "Input",
  InitializationCell->True],
Cell[15964, 509, 606, 17, 23, "Print"],
Cell[16573, 528, 601, 13, 23, "Print"],
Cell[17177, 543, 46, 1, 27, "Output"]
}, Open  ]],
Cell[17238, 547, 346, 14, 86, "Text"],
Cell[17587, 563, 25, 0, 32, "Text"],
Cell[17615, 565, 412, 11, 43, "Output",
  Evaluatable->True,
  InitializationCell->True],
Cell[18030, 578, 634, 18, 39, "Print"],
Cell[18667, 598, 604, 13, 23, "Print"],
Cell[19274, 613, 46, 1, 27, "Output"],
Cell[19323, 616, 155, 4, 32, "Text"],
Cell[19481, 622, 167, 4, 27, "Input",
  InitializationCell->True],
Cell[19651, 628, 173, 4, 27, "Input",
  InitializationCell->True],
Cell[19827, 634, 339, 13, 50, "Text"],
Cell[20169, 649, 132, 5, 32, "Text"],
Cell[20304, 656, 128, 3, 32, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[20469, 664, 37, 0, 54, "Section"],
Cell[20509, 666, 167, 5, 32, "Text"],
Cell[20679, 673, 222, 9, 32, "Text"],
Cell[20904, 684, 22, 0, 32, "Text"],
Cell[20929, 686, 44, 0, 32, "Text"],

Cell[CellGroupData[{
Cell[20998, 690, 265, 6, 43, "Input",
  InitializationCell->True],
Cell[21266, 698, 42, 1, 27, "Output"],
Cell[21311, 701, 523, 12, 75, "Output",
  Evaluatable->True,
  InitializationCell->True],
Cell[21837, 715, 344, 8, 23, "Print"],
Cell[22184, 725, 327, 7, 23, "Print"],
Cell[22514, 734, 634, 15, 39, "Print"],
Cell[23151, 751, 646, 13, 23, "Print"],
Cell[23800, 766, 341, 7, 23, "Print"],
Cell[24144, 775, 46, 1, 27, "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[24239, 782, 39, 0, 54, "Section"],

Cell[CellGroupData[{
Cell[24303, 786, 28, 0, 45, "Subsection"],
Cell[24334, 788, 197, 5, 32, "Text"],
Cell[24534, 795, 139, 5, 32, "Text"],
Cell[24676, 802, 479, 12, 68, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[25192, 819, 28, 0, 45, "Subsection"],
Cell[25223, 821, 312, 10, 27, "Output",
  Evaluatable->True,
  InitializationCell->True],
Cell[25538, 833, 42, 1, 27, "Output"],

Cell[CellGroupData[{
Cell[25605, 838, 317, 7, 59, "Input",
  InitializationCell->True],
Cell[25925, 847, 324, 8, 23, "Print"],
Cell[26252, 857, 316, 6, 23, "Print"],
Cell[26571, 865, 531, 14, 23, "Print"],
Cell[27105, 881, 604, 13, 23, "Print"],
Cell[27712, 896, 335, 7, 23, "Print"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[28096, 909, 30, 0, 45, "Subsection"],
Cell[28129, 911, 165, 5, 32, "Text"],
Cell[28297, 918, 333, 11, 27, "Output",
  Evaluatable->True,
  InitializationCell->True],
Cell[28633, 931, 44, 1, 27, "Output"],

Cell[CellGroupData[{
Cell[28702, 936, 280, 6, 43, "Input",
  InitializationCell->True],
Cell[28985, 944, 510, 12, 39, "Print"],
Cell[29498, 958, 610, 13, 23, "Print"],
Cell[30111, 973, 333, 7, 23, "Print"]
}, Open  ]],

Cell[CellGroupData[{
Cell[30481, 985, 33, 0, 42, "Subsubsection"],
Cell[30517, 987, 104, 3, 32, "Text"],
Cell[30624, 992, 224, 9, 32, "Text"],
Cell[30851, 1003, 22, 0, 32, "Text"],
Cell[30876, 1005, 480, 15, 68, "Text"],

Cell[CellGroupData[{
Cell[31381, 1024, 276, 7, 43, "Input",
  InitializationCell->True],
Cell[31660, 1033, 44, 1, 27, "Output"],
Cell[31707, 1036, 707, 15, 91, "Output",
  Evaluatable->True,
  InitializationCell->True],
Cell[32417, 1053, 435, 11, 23, "Print"],
Cell[32855, 1066, 586, 12, 23, "Print"],
Cell[33444, 1080, 1186, 29, 71, "Print"],
Cell[34633, 1111, 714, 14, 23, "Print"],
Cell[35350, 1127, 632, 13, 23, "Print"],
Cell[35985, 1142, 46, 1, 27, "Output"]
}, Open  ]],
Cell[36046, 1146, 233, 5, 50, "Text"],
Cell[36282, 1153, 186, 4, 27, "Input",
  InitializationCell->True],
Cell[36471, 1159, 119, 3, 32, "Text"],
Cell[36593, 1164, 188, 4, 27, "Input",
  InitializationCell->True]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[36842, 1175, 22, 0, 54, "Section"],
Cell[36867, 1177, 333, 12, 50, "Text"],

Cell[CellGroupData[{
Cell[37225, 1193, 160, 5, 27, "Input",
  InitializationCell->True],
Cell[37388, 1200, 51, 1, 27, "Output"]
}, Open  ]],
Cell[37454, 1204, 122, 5, 32, "Text"],
Cell[37579, 1211, 167, 5, 27, "Input",
  InitializationCell->True]
}, Open  ]],

Cell[CellGroupData[{
Cell[37783, 1221, 31, 0, 54, "Section"],
Cell[37817, 1223, 115, 3, 50, "Text"],

Cell[CellGroupData[{
Cell[37957, 1230, 43, 1, 27, "Input"],
Cell[38003, 1233, 140, 2, 39, "Print"],
Cell[38146, 1237, 606, 17, 23, "Print"],
Cell[38755, 1256, 601, 13, 23, "Print"],
Cell[39359, 1271, 634, 18, 39, "Print"],
Cell[39996, 1291, 604, 13, 23, "Print"],
Cell[40603, 1306, 344, 8, 23, "Print"],
Cell[40950, 1316, 327, 7, 23, "Print"],
Cell[41280, 1325, 634, 15, 39, "Print"],
Cell[41917, 1342, 646, 13, 23, "Print"],
Cell[42566, 1357, 341, 7, 23, "Print"],
Cell[42910, 1366, 324, 8, 23, "Print"],
Cell[43237, 1376, 316, 6, 23, "Print"],
Cell[43556, 1384, 531, 14, 23, "Print"],
Cell[44090, 1400, 604, 13, 23, "Print"],
Cell[44697, 1415, 335, 7, 23, "Print"],
Cell[45035, 1424, 510, 12, 39, "Print"],
Cell[45548, 1438, 610, 13, 23, "Print"],
Cell[46161, 1453, 333, 7, 23, "Print"],
Cell[46497, 1462, 435, 11, 23, "Print"],
Cell[46935, 1475, 586, 12, 23, "Print"],
Cell[47524, 1489, 1186, 29, 71, "Print"],
Cell[48713, 1520, 714, 14, 23, "Print"],
Cell[49430, 1536, 632, 13, 23, "Print"]
}, Open  ]],
Cell[50077, 1552, 42, 0, 32, "Text"],

Cell[CellGroupData[{
Cell[50144, 1556, 44, 1, 27, "Input"],
Cell[50191, 1559, 56, 1, 23, "Print"],
Cell[50250, 1562, 40, 1, 23, "Print"]
}, Open  ]],
Cell[50305, 1566, 220, 4, 68, "Text"],

Cell[CellGroupData[{
Cell[50550, 1574, 31, 0, 45, "Subsection"],
Cell[50584, 1576, 320, 10, 50, "Text"],
Cell[50907, 1588, 410, 15, 183, "Text"],

Cell[CellGroupData[{
Cell[51342, 1607, 246, 5, 43, "Input"],
Cell[51591, 1614, 135, 3, 45, "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[51775, 1623, 31, 0, 45, "Subsection"],
Cell[51809, 1625, 254, 8, 32, "Text"],
Cell[52066, 1635, 364, 12, 132, "Text"],

Cell[CellGroupData[{
Cell[52455, 1651, 367, 6, 75, "Input"],
Cell[52825, 1659, 167, 4, 45, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[53029, 1668, 268, 5, 59, "Input"],
Cell[53300, 1675, 156, 4, 45, "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[53505, 1685, 31, 0, 45, "Subsection"],
Cell[53539, 1687, 120, 3, 32, "Text"],
Cell[53662, 1692, 608, 12, 268, "Text"],

Cell[CellGroupData[{
Cell[54295, 1708, 252, 4, 91, "Input"],
Cell[54550, 1714, 150, 3, 43, "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[54749, 1723, 31, 0, 45, "Subsection"],
Cell[54783, 1725, 118, 3, 32, "Text"],
Cell[54904, 1730, 459, 10, 166, "Text"],

Cell[CellGroupData[{
Cell[55388, 1744, 281, 5, 107, "Input"],
Cell[55672, 1751, 161, 2, 45, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[55870, 1758, 44, 1, 27, "Input"],
Cell[55917, 1761, 71, 1, 23, "Print"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[56037, 1768, 31, 0, 45, "Subsection"],
Cell[56071, 1770, 262, 8, 50, "Text"],
Cell[56336, 1780, 336, 12, 132, "Text"],

Cell[CellGroupData[{
Cell[56697, 1796, 218, 4, 75, "Input"],
Cell[56918, 1802, 73, 1, 43, "Output"]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}
]
*)




(***********************************************************************
End of Mathematica Notebook file.
***********************************************************************)

