%% file   : parser.pl (version )
%% author : Sosuke Kaneko
%% date   : Dec.1997 - Apr.1998
%   COPYRIGHT (C) 1998 Sosuke Kaneko


parse(FileName,ParseTree):-
        seeing(InFile),
        see(FileName),
	tokenize(Tokens),
        preparse(Tokens,TokList,LTNList),
        'Goal'(ParseTree,TokList,[]),
	errorReport(LTNList,Tokens),
        seen,
        see(InFile).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%                      %%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%-------------------------------------------%
%/֤ν(ȥ/ʸʳ)
%Ȥν
%ȡֹղ
%/ȡֹбꥹȤ
%-------------------------------------------%

preparse(Tokens,TokenList,LTNList):-
        preparse(Tokens,TokenList,[_|LTNList],1,0,0).


:-mode preparse(+,-,-,+,+,+).
%%preparse(Tokens,TokenList,LineTokenNumberList,TokenNumber,LineNumber,Flag)
% Flag=(0:ȥ/ʸ,1:ȥ,2:ʸ)

preparse([],[],[],_,_,_).
preparse(['\',''''|ToksTail],TokList,LTNL,TN,LN,Flag):- !,  % escape for '
	TN1 is TN+1,
	TokList=[tok('\',TN),tok('''',TN1)|TokListTail],
	TN2 is TN1+1,
        preparse(ToksTail,TokListTail,LTNL,TN2,LN,Flag).
preparse([''''|ToksTail],TokList,LTNL,TN,LN,Flag):- !,
	TokList=[tok('''',TN)|TokListTail],
	(Flag=:=0 -> NFlag:=1 ; (Flag=:=1 -> NFlag:=0 ; NFlag:=2)),
	TN1 is TN+1,
        preparse(ToksTail,TokListTail,LTNL,TN1,LN,NFlag).
preparse(['\','"'|ToksTail],TokList,LTNL,TN,LN,Flag):- !,    % escape for "
	TN1 is TN+1,
	TokList=[tok('\',TN),tok('"',TN1)|TokListTail],
	TN2 is TN1+1,
        preparse(ToksTail,TokListTail,LTNL,TN2,LN,Flag).
preparse(['"'|ToksTail],TokList,LTNL,TN,LN,Flag):- !,
	TokList=[tok('"',TN)|TokListTail],
	(Flag=:=0 -> NFlag:=2 ; (Flag=:=2 -> NFlag:=0 ; NFlag:=1)),
	TN1 is TN+1,
        preparse(ToksTail,TokListTail,LTNL,TN1,LN,NFlag).
preparse(['/','/'|ToksTail],TokList,LTNL,TN,LN,0):- !,	% // comments
	TN1 is TN+2,
	skip_slushSlushComments(ToksTail,ToksTail1,TN1,TN2),
	preparse(ToksTail1,TokList,LTNL,TN2,LN,0).
preparse(['/','*'|ToksTail],TokList,LTNL,TN,LN,0):- !,	% /* comments */
	TN1 is TN+2,
	skip_slushStarComments(ToksTail,ToksTail1,TN1,TN2),
	preparse(ToksTail1,TokList,LTNL,TN2,LN,0).
preparse([' '|ToksTail],TokList,LTNL,TN,LN,0):- !,		% ignore space
	TN1 is TN+1,
        preparse(ToksTail,TokList,LTNL,TN1,LN,0).
preparse([' '|ToksTail],TokList,LTNL,TN,LN,Flag):-
	Flag =\= 0,!,
	TokList=[tok(' ',TN)|TokListTail],
	TN1 is TN+1,
        preparse(ToksTail,TokListTail,LTNL,TN1,LN,Flag).
preparse(['	'|ToksTail],TokList,LTNL,TN,LN,0):- !,	% ignore tab
	TN1 is TN+1,
        preparse(ToksTail,TokList,LTNL,TN1,LN,0).
preparse(['	'|ToksTail],TokList,LTNL,TN,LN,Flag):-
	Flag =\= 0, !,
	TokList=[tok('	',TN)|TokListTail],
	TN1 is TN+1,
        preparse(ToksTail,TokListTail,LTNL,TN1,LN,Flag).
preparse([line(LN1)|ToksTail],TokList,LTNL,TN,LN,0):- !,	% line
	TN0 is TN-1,
	LTNL=[line(LN,TN0)|LTNLTail],
	preparse(ToksTail,TokList,LTNLTail,TN,LN1,0).
preparse([Token|ToksTail],TokList,LTNL,TN,LN,Flag):-		% others
	TokList=[tok(Token,TN)|TokListTail],
	TN1 is TN+1,
        preparse(ToksTail,TokListTail,LTNL,TN1,LN,Flag).

:-mode skip_slushSlushComments(+,-,+,-).
skip_slushSlushComments([line(LN)|ToksTail],[line(LN)|ToksTail],TN,TN):-!.
skip_slushSlushComments([_|ToksTail],Tokens,TN0,TN):-
	TN1 is TN0+1,
	skip_slushSlushComments(ToksTail,Tokens,TN1,TN).

:-mode skip_slushStarComments(+,-,+,-).
skip_slushStarComments(['*','/'|ToksTail],ToksTail,TN0,TN):-!,
	TN is TN0+2.
skip_slushStarComments([line(_)|ToksTail],Tokens,TN0,TN):-!,
	skip_slushStarComments(ToksTail,Tokens,TN0,TN).
skip_slushStarComments([_|ToksTail],Tokens,TN0,TN):-
	TN1 is TN0+1,
	skip_slushStarComments(ToksTail,Tokens,TN1,TN).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%      ʸ      %%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

'Goal'('Goal'(CompilationUnit)) -->
        'CompilationUnit'(CompilationUnit).

%-------  19.3  Lexical Structure  --------%

%Literal:
%        FloatingPointLiteral
%        IntegerLiteral
%        BooleanLiteral
%        CharacterLiteral
%        StringLiteral
%        NullLiteral
'Literal'('Literal'(
		FloatingPointLiteral)) -->
	'FloatingPointLiteral'(FloatingPointLiteral),
	{!}.
'Literal'('Literal'(
             IntegerLiteral)) -->
        'IntegerLiteral'(IntegerLiteral),
	{!}.
'Literal'('Literal'(
	      BooleanLiteral)) -->
	'BooleanLiteral'(BooleanLiteral),
	{!}.
'Literal'('Literal'(
		CharacterLiteral)) -->
	'CharacterLiteral'(CharacterLiteral),
	{!}.
'Literal'('Literal'(
		StringLiteral)) -->
	'StringLiteral'(StringLiteral).
	{!}.
'Literal'('Literal'(
		NullLiteral)) -->
	'NullLiteral'(NullLiteral).

% poor integer and floatingpoint literal
'IntegerLiteral'('IntegerLiteral'(Token)) -->
	[tok(Token,_)],
	{integer(Token)}.
'FloatingPointLiteral'(T,[tok(T1,_),tok('.',_),tok(T2,_)|S],S):-
	integer(T1),integer(T2),
	T='FloatingPointLiteral'(Token),
	name(T1,L1),name(T2,L2),
	append(L1,[46|L2],L),
	name(Token,L).


%%BooleanLiteral: one of
%%	true false
'BooleanLiteral'(T,[tok(true,_)|S0],S):-
	S0=S,
	T='BooleanLiteral'(true).
'BooleanLiteral'(T,[tok(false,_)|S0],S):-
	S0=S,
	T='BooleanLiteral'(false).



%%CharacterLiteral:
%%	' SingleCharacter '
%%	' EscapeSequence '
'CharacterLiteral'(T,[tok('''',_)|S0],S):-
	T='CharacterLiteral'('''',SingleCharacter,''''),
	'SingleCharacter'(SingleCharacter,S0,S1),
	S1=[tok('''',_)|S].
'CharacterLiteral'(T,[tok('''',_)|S0],S):-
	T='CharacterLiteral'('''',EscapeSequence,''''),
	'EscapeSequence'(EscapeSequence,S0,S1),
	S1=[tok('''',_)|S].

%%SingleCharacter:
%%	InputCharacter but not " or \
'SingleCharacter'(InputCharacter) -->
	[tok(InputCharacter,_)],
	{InputCharacter \= '''',
	 InputCharacter \= '\',
	 ((atom(InputCharacter),
	   atom_length(InputCharacter,Length), 	% SingleCharacter must be
	   Length < 2);				% a single character.
	  (integer(InputCharacter),
	   InputCharacter<10))}.


%%StringLiteral:
%%	" StringCharactersopt "
'StringLiteral'(T,[tok('"',_)|S0],S):-
	'StringCharactersopt'(StringCharactersopt,S0,S1),
	S1=[tok(Tok,TN)|S],
	(Tok=='"' -> true ; error1('" expected',TN)),
	name(String,StringCharactersopt),
	T='StringLiteral'('"',String,'"').

%%StringCharacters:
%%	StringCharacter StringCharactersopt
'StringCharacters'(StringCharacters) -->
	'StringCharacter'(StringCharacter),
	'StringCharactersopt'(StringCharactersopt),
	{append(StringCharacter,StringCharactersopt,StringCharacters)}.

'StringCharactersopt'(StringCharacters) -->
	'StringCharacters'(StringCharacters),
	{!}.
'StringCharactersopt'([]) --> [].

%%StringCharacter:
%%	EscapeSequence
%%	InputCharacter but not " or \
'StringCharacter'(EscapeSequence) -->
	'EscapeSequence'(EscapeSequence),{!}.
'StringCharacter'(InputCharacter) -->
	[tok(Token,_)],
	{Token \== '"',
	 name(Token,InputCharacter)}.

%%EscapeSequence: one of
%%	\b \t \n \f \r \" \' \\
'EscapeSequence'(EscapeSequence,[tok('\',_)|S0],S):-
	escape(EscapeSequence,S0,S).

:-mode escape(-,+,-).
escape(T,[tok(b,_)|S0],S):- !,T=[92,98],S0=S.
escape(T,[tok(t,_)|S0],S):- !,T=[92,116],S0=S.
escape(T,[tok(n,_)|S0],S):- !,T=[92,110],S0=S.
escape(T,[tok(f,_)|S0],S):- !,T=[92,102],S0=S.
escape(T,[tok(r,_)|S0],S):- !,T=[92,114],S0=S.
escape(T,[tok('"',_)|S0],S):- !,T=[92,34],S0=S.
escape(T,[tok('''',_)|S0],S):- !,T=[92,39],S0=S.
escape(T,[tok('\',_)|S0],S):- T=[92,92],S0=S.


%%NullLiteral: 
%%	null
'NullLiteral'(T,[tok(null,_)|S0],S):-
	S0=S,
	T='NullLiteral'(null).

%-------  19.4  Types,Values,and Variables  --------%

%Type:
%       ReferenceType
%       PrimitiveType
'Type'('Type'(ReferenceType)) -->
        'ReferenceType'(ReferenceType),{!}.
'Type'('Type'(PrimitiveType)) -->
        'PrimitiveType'(PrimitiveType).

%PrimitiveType:
%       boolean
%       NumericType
'PrimitiveType'(T,[tok(boolean,_)|S0],S):-!,
	T='PrimitiveType'(boolean),
	S0=S.
'PrimitiveType'('PrimitiveType'(NumericType),S0,S):-
        'NumericType'(NumericType,S0,S).

%NumericType:
%       IntegralType
%       FloatingPointType
'NumericType'('NumericType'(
                  IntegralType)) -->
        'IntegralType'(IntegralType),{!}.
'NumericType'('NumericType'(
                  FloatingPointType)) -->
        'FloatingPointType'(FloatingPointType).

%IntegralType: one of
%       byte short int long char
:-mode 'IntegralType'(-,+,-).
'IntegralType'(T,[tok(byte,_)|S],S):- !,T='IntegralType'(byte).
'IntegralType'(T,[tok(short,_)|S],S):- !,T='IntegralType'(short).
'IntegralType'(T,[tok(int,_)|S],S):- !,T='IntegralType'(int).
'IntegralType'(T,[tok(long,_)|S],S):- !,T='IntegralType'(long).
'IntegralType'(T,[tok(char,_)|S],S):- T='IntegralType'(char).

%FloatingPointType: one of
%       float double
:-mode 'FloatingPointType'(-,+,-).
'FloatingPointType'(T,[tok(float,_)|S],S):- !,T='FloatingPointType'(float).
'FloatingPointType'(T,[tok(double,_)|S],S):- T='FloatingPointType'(double).

%ReferenceType:
%       ArrayType
%       ClassOrInterfaceType
'ReferenceType'('ReferenceType'(
                    ArrayType)) -->
        'ArrayType'(ArrayType),{!}.
'ReferenceType'('ReferenceType'(
                    ClassOrInterfaceType)) -->
        'ClassOrInterfaceType'(ClassOrInterfaceType).

%ClassOrInterfaceType:
%       Name
'ClassOrInterfaceType'('ClassOrInterfaceType'(
                           Name)) -->
        'Name'(Name).

%ClassType:
%       ClassOrInterfaceType
'ClassType'('ClassType'(
                ClassOrInterfaceType)) -->
        'ClassOrInterfaceType'(ClassOrInterfaceType).

%InterfaceType:
%       ClassOrInterfaceType
'InterfaceType'('InterfaceType'(
                    ClassOrInterfaceType)) -->
        'ClassOrInterfaceType'(ClassOrInterfaceType).

%ArrayType:
%       PrimitiveType Dims
%       Name Dims
'ArrayType'('ArrayType'(
                PrimitiveType,
                Dims)) -->
        'PrimitiveType'(PrimitiveType),
	'Dims'(Dims).
'ArrayType'('ArrayType'(
                Name,
                Dims)) -->
        'Name'(Name),
	'Dims'(Dims).



%-------  19.5  Names  --------%

%Name:
%        QualifiedName
%        SimpleName
'Name'('Name'(QualifiedName)) -->
        'QualifiedName'(QualifiedName).
'Name'('Name'(SimpleName)) -->
        'SimpleName'(SimpleName).

%SimpleName:
%        Identifier
'SimpleName'('SimpleName'(Identifier)) -->
        'Identifier'(Identifier).

%QualifiedName:
%        Identifier . Name
'QualifiedName'('QualifiedName'(
                    Identifier,
                    '.',
                    Name)) -->
        'Identifier'(Identifier),
        [tok('.',_)],
        'Name'(Name).


%--------  19.6  Packages  ---------%

%CompilationUnit:
%        PackageDeclarationopt ImportDeclarationsopt TypeOrConstraintDeclarationsopt 
'CompilationUnit'('CompilationUnit'(
                      PackageDeclarationopt,
                      ImportDeclarationsopt,
                      TypeOrConstraintDeclarationsopt)) -->
        'PackageDeclarationopt'(PackageDeclarationopt),
        'ImportDeclarationsopt'(ImportDeclarationsopt),
        'TypeOrConstraintDeclarationsopt'(TypeOrConstraintDeclarationsopt).

%ImportDeclarations:
%        ImportDeclaration ImportDeclarationsopt
'ImportDeclarations'('ImportDeclarations'(
                         ImportDeclaration,
                         ImportDeclarationsopt)) -->
        'ImportDeclaration'(ImportDeclaration),
        'ImportDeclarationsopt'(ImportDeclarationsopt).

'ImportDeclarationsopt'(ImportDeclarations) -->
	'ImportDeclarations'(ImportDeclarations),
	{!}.
'ImportDeclarationsopt'(nil) --> [].

%TypeOrConstraintDeclarations:
%        TypeOrConstraintDeclaration TypeOrConstraintDeclarationsopt
'TypeOrConstraintDeclarations'('TypeOrConstraintDeclarations'(
                       TypeOrConstraintDeclaration,
                       TypeOrConstraintDeclarationsopt)) -->
        'TypeOrConstraintDeclaration'(TypeOrConstraintDeclaration),
        'TypeOrConstraintDeclarationsopt'(TypeOrConstraintDeclarationsopt).

'TypeOrConstraintDeclarationsopt'(TypeOrConstraintDeclarations) -->
	'TypeOrConstraintDeclarations'(TypeOrConstraintDeclarations),
	{!}.
'TypeOrConstraintDeclarationsopt'(nil) --> [].

%PackageDeclaration:
%        package Name ;
'PackageDeclaration'(T,[tok(package,TN)|S0],S):-
	T='PackageDeclaration'(package,Name,';'),
        ('Name'(Name,S0,S1) -> true ; error('invalid name',TN)),
	S1=[tok(Tok1,TN1)|S],
	(Tok1==';' -> true ; error1('; expected',TN1)).

'PackageDeclarationopt'(PackageDeclaration) -->
	'PackageDeclaration'(PackageDeclaration),
	{!}.
'PackageDeclarationopt'(nil) --> [].

%ImportDeclaration:
%        SingleTypeImportDeclaration
%        TypeImportOnDemandDeclaration
'ImportDeclaration'('ImportDeclaration'(
                        SingleTypeImportDeclaration)) -->
        'SingleTypeImportDeclaration'(SingleTypeImportDeclaration),{!}.
'ImportDeclaration'('ImportDeclaration'(
                        TypeImportOnDemandDeclaration)) -->
        'TypeImportOnDemandDeclaration'(TypeImportOnDemandDeclaration).

%SingleTypeImportDeclaration:
%        import Name ;
'SingleTypeImportDeclaration'(T,[tok(import,TN)|S0],S):-
	T='SingleTypeImportDeclaration'(import,Name,';'),
        ('Name'(Name,S0,S1) -> true ; error('invalid name',TN)),
	S1=[tok(';',_)|S].

%TypeImportOnDemandDeclaration:
%        import Name . * ;
'TypeImportOnDemandDeclaration'(T,[tok(import,_)|S0],S):-
	T='TypeImportOnDemandDeclaration'(import,Name,'.','*',';'),
        'Name'(Name,S0,S1),
	S1=[tok(Tok1,TN),tok(Tok2,_),tok(Tok3,_)|S],
        (Tok1=='.',Tok2=='*',Tok3==';' -> true ;
	 error1('; or .*; expected',TN)).

%TypeOrConstraintDeclaration:
%        ClassDeclaration
%        InterfaceDeclaration
%        ConstraintDeclaration
%        ;
'TypeOrConstraintDeclaration'('TypeOrConstraintDeclaration'(
                     ClassDeclaration)) -->
        'ClassDeclaration'(ClassDeclaration),{!}.
'TypeOrConstraintDeclaration'('TypeOrConstraintDeclaration'(
                     InterfaceDeclaration)) -->
        'InterfaceDeclaration'(InterfaceDeclaration),{!}.
'TypeOrConstraintDeclaration'('TypeOrConstraintDeclaration'(
                     ConstraintDeclaration)) -->
        'ConstraintDeclaration'(ConstraintDeclaration),{!}.
'TypeOrConstraintDeclaration'(T,[tok(';',_)|S0],S):-!,
	S0=S,
	T='TypeOrConstraintDeclaration'(';').
% error handling
'TypeOrConstraintDeclaration'(T,S0,S):-
	S0=[tok(_,TN)|_],error1('invalid declaration head',TN).
'TypeOrConstraintDeclaration'(T,S0,S):-
	T=nil,
	errorRecovery(S0,S,WrongStatemts).
%	reportError(['wrong class, interface, or constraint declaration'|WrongStatemts]).

%-------  19.7  for parsing  -------%

%Modifiers:
%        Modifier Modifiersopt
'Modifiers'('Modifiers'(
                Modifier,
                Modifiersopt)) -->
        'Modifier'(Modifier),
        'Modifiersopt'(Modifiersopt).

'Modifiersopt'(Modifiers) -->
	'Modifiers'(Modifiers),
	{!}.
'Modifiersopt'(nil) --> [].

%Modifier: one of
%	 main
%        public protected private
%        static
%        abstract final native synchronized transient volatile
:-mode 'Modifier'(-,+,-).
'Modifier'(T,[tok(main,_)|S0],        S):-!,S0=S,T=main.
'Modifier'(T,[tok(public,_)|S0],      S):-!,S0=S,T=public.
'Modifier'(T,[tok(protected,_)|S0],   S):-!,S0=S,T=protected.
'Modifier'(T,[tok(private,_)|S0],     S):-!,S0=S,T=private.
'Modifier'(T,[tok(static,_)|S0],      S):-!,S0=S,T=static.
'Modifier'(T,[tok(abstract,_)|S0],    S):-!,S0=S,T=abstract.
'Modifier'(T,[tok(final,_)|S0],       S):-!,S0=S,T=final.
'Modifier'(T,[tok(native,_)|S0],      S):-!,S0=S,T=native.
'Modifier'(T,[tok(synchronized,_)|S0],S):-!,S0=S,T=synchronized.
'Modifier'(T,[tok(transient,_)|S0],   S):-!,S0=S,T=transient.
'Modifier'(T,[tok(volatile,_)|S0],    S):-!,S0=S,T=volatile.



%-------  19.8  Classes  -------%

%% 19.8.1 Class Declaration

%ClassDeclaration:
%        Modifiersopt class Identifier Superopt Interfacesopt ClassBody
'ClassDeclaration'('ClassDeclaration'(
                       Modifiersopt,
                       class,
                       Identifier,
                       Superopt,
                       Interfacesopt,
                       ClassBody)) -->
        'Modifiersopt'(Modifiersopt),
        [tok(class,TN)],
        ('Identifier'(Identifier) -> [] ;{error('invalid identifier',TN)}),
        'Superopt'(Superopt),
        'Interfacesopt'(Interfacesopt),
        'ClassBody'(ClassBody).

%Super:
%        extends ClassType
'Super'(T,[tok(extends,TN)|S0],S):-
	T='Super'(extends,ClassType),
        ('ClassType'(ClassType,S0,S) -> true ; error('invalid name',TN)).

'Superopt'(Super) --> 
	'Super'(Super),
	{!}.
'Superopt'(nil) --> [].

%Interfaces:
%        implements InterfaceTypeList
'Interfaces'(T,[tok(implements,TN)|S0],S):-
	T='Interfaces'(implements,InterfaceTypeList),
        ('InterfaceTypeList'(InterfaceTypeList,S0,S) -> true ;
	 error('invalid name',TN)).

'Interfacesopt'(Interfaces) --> 
	'Interfaces'(Interfaces),
	{!}.
'Interfacesopt'(nil) --> [].

%InterfaceTypeList:
%        InterfaceType InterfaceTypeListRestopt
'InterfaceTypeList'('InterfaceTypeList'(
			InterfaceType,
			InterfaceTypeListRestopt)) -->
	'InterfaceType'(InterfaceType),
	'InterfaceTypeListRestopt'(InterfaceTypeListRestopt).

%InterfaceTypeListRest:
%        , InterfaceTypeList
'InterfaceTypeListRest'(T,[tok(',',_)|S0],S):-
	T='InterfaceTypeListRest'(',',InterfaceTypeList),
	'InterfaceTypeList'(InterfaceTypeList,S0,S).

'InterfaceTypeListRestopt'(InterfaceTypeListRest) -->
	'InterfaceTypeListRest'(InterfaceTypeListRest).
	{!}.
'InterfaceTypeListRestopt'(nil) --> [].

%ClassBody:
%        { ClassBodyDeclarationsopt }
'ClassBody'(T,[tok('{',_)|S0],S):-
	T='ClassBody'('{',ClassBodyDeclarationsopt,'}'),
        'ClassBodyDeclarationsopt'(ClassBodyDeclarationsopt,S0,S1),
        S1=[tok(Token,TN)|S],
	(Token=='}' -> true ; error1('} expected',TN)),!.
% error handling
'ClassBody'(T,[tok(Tok,TN)|S0],S):-
	Tok \== '{',
	T=nil,S0=S,
	error1('{ expected',TN).

%ClassBodyDeclarations:
%        ClassBodyDeclaration ClassBodyDeclarationsopt
'ClassBodyDeclarations'('ClassBodyDeclarations'(
                            ClassBodyDeclaration,
                            ClassBodyDeclarationsopt)) -->
        'ClassBodyDeclaration'(ClassBodyDeclaration),
        'ClassBodyDeclarationsopt'(ClassBodyDeclarationsopt).

'ClassBodyDeclarationsopt'(ClassBodyDeclarations) -->
	'ClassBodyDeclarations'(ClassBodyDeclarations).
	{!}.
'ClassBodyDeclarationsopt'(nil) --> [].

%ClassBodyDeclaration:
%        ConstructorDeclaration
%        ClassMemberDeclaration
%        StaticInitializer
'ClassBodyDeclaration'('ClassBodyDeclaration'(
                           ConstructorDeclaration)) -->
        'ConstructorDeclaration'(ConstructorDeclaration),{!}.
'ClassBodyDeclaration'('ClassBodyDeclaration'(
                           ClassMemberDeclaration)) -->
        'ClassMemberDeclaration'(ClassMemberDeclaration),{!}.
'ClassBodyDeclaration'('ClassBodyDeclaration'(
                           StaticInitializer)) -->
        'StaticInitializer'(StaticInitializer),{!}.
%error handling
'ClassBodyDeclaration'(T,S0,S):-
	S0=[tok(Tok,TN)|_],
	Tok\=='}',
	error1('invalid class body declaration',TN).
'ClassBodyDeclaration'(T,S0,S):-			
	T=nil,
	S0\=[tok('}',_)|_],
	errorRecovery(S0,S,WrongClassMemberDeclaration).
%	reportError(['wrong class body declaration'|WrongClassMemberDeclaration]).

%ClassMemberDeclaration:
%        ComponentDeclaration  % for dj
%        AttributeDeclaration  % for dj
%        Constraint ;          % for dj
%        MethodDeclaration
%        FieldDeclaration
%%        ActionDeclaration     % for dj
'ClassMemberDeclaration'('ClassMemberDeclaration'(
                              ComponentDeclaration)) -->
        'ComponentDeclaration'(ComponentDeclaration),{!}.
'ClassMemberDeclaration'('ClassMemberDeclaration'(
                              AttributeDeclaration)) -->
        'AttributeDeclaration'(AttributeDeclaration),{!}.
'ClassMemberDeclaration'('ClassMemberDeclaration'(
                              Constraint,
			      ';')) -->
        'Constraint'(Constraint),
	[tok(Tok,TN)],
	{Tok==';' -> true ; error1('; expected',TN)},{!}.
'ClassMemberDeclaration'('ClassMemberDeclaration'(
                             MethodDeclaration)) -->
        'MethodDeclaration'(MethodDeclaration),{!}.
'ClassMemberDeclaration'('ClassMemberDeclaration'(
                             FieldDeclaration)) -->
        'FieldDeclaration'(FieldDeclaration),{!}.
/*
'ClassMemberDeclaration'('ClassMemberDeclaration'(
                              ActionDeclaration)) -->
        'ActionDeclaration'(ActionDeclaration),{!}.
*/

%% 19.8.2  Field Declarations

%FieldDeclaration:
%       Modifiersopt Type VariableDeclarators ;
'FieldDeclaration'('FieldDeclaration'(
                        Modifiersopt,
                        Type,
                        VariableDeclarators,
                        ';')) -->
        'Modifiersopt'(Modifiersopt),
        'Type'(Type),
        'VariableDeclarators'(VariableDeclarators),
        [tok(Token,TN)],
	{(Token==';' -> true ; error1('; expected',TN))}.

%VariableDeclarators:
%       VariableDeclarator VariableDeclaratorsRestopt
'VariableDeclarators'('VariableDeclarators'(
	                        VariableDeclarator,
				VariableDeclaratorsRestopt)) -->
        'VariableDeclarator'(VariableDeclarator),
	'VariableDeclaratorsRestopt'(VariableDeclaratorsRestopt).

%VariableDeclaratorsRest:
%	, VariableDeclarators
'VariableDeclaratorsRest'(T,[tok(',',_)|S0],S):-
	T='VariableDeclaratorsRest'(',',VariableDeclarators),
        'VariableDeclarators'(VariableDeclarators,S0,S).

'VariableDeclaratorsRestopt'(VariableDeclaratorsRest) -->
	'VariableDeclaratorsRest'(VariableDeclaratorsRest).
%	{!}.
'VariableDeclaratorsRestopt'(nil) --> [].

%VariableDeclarator:
%       VariableDeclaratorId VariableDeclaratorRestopt
'VariableDeclarator'('VariableDeclarator'(
        	                VariableDeclaratorId,
				VariableDeclaratorRestopt)) -->
        'VariableDeclaratorId'(VariableDeclaratorId),
	'VariableDeclaratorRestopt'(VariableDeclaratorRestopt).

%VariableDeclaratorRest:
%       = VariableInitializer
'VariableDeclaratorRest'(T,[tok('=',TN)|S0],S):-
	T='VariableDeclaratorRest'('=',VariableInitializer),
        ('VariableInitializer'(VariableInitializer,S0,S) -> true;
	 error('variable initializer expected',TN)).

'VariableDeclaratorRestopt'(VariableDeclaratorRest) -->
	'VariableDeclaratorRest'(VariableDeclaratorRest).
	{!}.
'VariableDeclaratorRestopt'(nil) --> [].

%VariableDeclaratorId:
%       Identifier Dimsopt
'VariableDeclaratorId'('VariableDeclaratorId'(
                           Identifier,
			   Dimsopt)) -->
        'Identifier'(Identifier),
	'Dimsopt'(Dimsopt).

%VariableInitializer:
%       Expression
%       ArrayInitializer
'VariableInitializer'('VariableInitializer'(
                          Expression)) -->
        'Expression'(Expression).
'VariableInitializer'('VariableInitializer'(
                          ArrayInitializer)) -->
        'ArrayInitializer'(ArrayInitializer).




%--------  19.8.3  Method Declarations  --------%

%MethodDeclaration:
%        MethodHeader MethodBody
'MethodDeclaration'('MethodDeclaration'(
			MethodHeader,
			MethodBody)) -->
	'MethodHeader'(MethodHeader),
	'MethodBody'(MethodBody).

%MethodHeader:
%        Modifiersopt MethodHeaderRest
'MethodHeader'('MethodHeader'(
			Modifiersopt,
			MethodHeaderRest)) -->
	'Modifiersopt'(Modifiersopt),
	'MethodHeaderRest'(MethodHeaderRest).

%MethodHeaderRest:
%	void MethodDeclarator Throwsopt
%       Type MethodDeclarator Throwsopt
'MethodHeaderRest'(T,[tok(void,_)|S0],S):-!,
	T='MethodHeaderRest'(void,MethodDeclarator,Throwsopt),
	'MethodDeclarator'(MethodDeclarator,S0,S1),
	'Throwsopt'(Throwsopt,S1,S).
'MethodHeaderRest'('MethodHeaderRest'(
			Type,
			MethodDeclarator,
			Throwsopt)) -->
	'Type'(Type),
	'MethodDeclarator'(MethodDeclarator),
	'Throwsopt'(Throwsopt).

%MethodDeclarator:
%        Identifier ( FormalParameterListopt ) []opt
'MethodDeclarator'('MethodDeclarator'(
		       Identifier,
		       '(',
		       FormalParameterListopt,
		       ')',
		       Dim)) -->
	'Identifier'(Identifier),
	[tok('(',_)],
	'FormalParameterListopt'(FormalParameterListopt),
	[tok(Token,TN)],
	{Token==')' -> true ; error1(') expected',TN)},
	( [tok('[',_),tok(']',_)] -> {Dim='[]'};{Dim=nil}).

%FormalParameterList:
%        FormalParameter FormalParameterListRestopt
'FormalParameterList'('FormalParameterList'(
			  FormalParameter,
			  FormalParameterListRestopt)) -->
	'FormalParameter'(FormalParameter),
	'FormalParameterListRestopt'(FormalParameterListRestopt).

'FormalParameterListopt'(FormalParameterList) --> 
	'FormalParameterList'(FormalParameterList).
	{!}.
'FormalParameterListopt'(nil) --> [].

%FormalParameterListRest:
%	, FormalParameterList
'FormalParameterListRest'(T,[tok(',',_)|S0],S):-
	T='FormalParameterListRest'(',',FormalParameterList),
	'FormalParameterList'(FormalParameterList,S0,S).

'FormalParameterListRestopt'(FormalParameterListRest) -->
	'FormalParameterListRest'(FormalParameterListRest).
	{!}.
'FormalParameterListRestopt'(nil) --> [].

%FormalParameter:
%        Type VariableDeclaratorId
'FormalParameter'('FormalParameter'(
		      Type,
		      VariableDeclaratorId)) -->
	'Type'(Type),
	'VariableDeclaratorId'(VariableDeclaratorId).

%Throws:
%        throws ClassTypeList
'Throws'(T,[tok(throws,TN)|S0],S):-
	T='Throws'(throws,ClassTypeList),
	('ClassTypeList'(ClassTypeList,S0,S) -> true ; error('invalid name',TN)).

'Throwsopt'(Throws) --> 
	'Throws'(Throws).
	{!}.
'Throwsopt'(nil) --> [].

%ClassTypeList:
%        ClassType ClassTypeListRestopt
'ClassTypeList'('ClassTypeList'(
			ClassType,
			ClassTypeListRestopt)) -->
	'ClassType'(ClassType),
	'ClassTypeListRestopt'(ClassTypeListRestopt).

%ClassTypeListRest:
%        , ClassTypeList
'ClassTypeListRest'(T,[tok(',',_)|S0],S):-
	T='ClassTypeListRest'(',',ClassTypeList),
	'ClassTypeList'(ClassTypeList,S0,S).

'ClassTypeListRestopt'(ClassTypeListRest) -->
	'ClassTypeListRest'(ClassTypeListRest).
	{!}.
'ClassTypeListRestopt'(nil) --> [].

%MethodBody:
%	;
%       Block <error_handling_MethodBody accepts ;>
'MethodBody'(T,[tok(';',_)|S0],S):-!,
	T='MethodBody'(';'),S0=S.
'MethodBody'('MethodBody'(
		  Block)) -->
	'Block'(Block),
	error_handling_MethodBody. % error handling

error_handling_MethodBody([tok(';',_)|S],S):-!.
error_handling_MethodBody(S,S).


%--------  19.8.4  Static Initializers  --------%

%StaticInitializer:
%        static Block
'StaticInitializer'(T,[tok(static,_)|S0],S):-
	T='StaticInitializer'(static,Block),
	'Block'(Block,S0,S).



%--------  19.8.5  Constructor Declarations  --------%

%ConstructorDeclaration:
%        Modifiersopt ConstructorDeclarator Throwsopt ConstructorBody
'ConstructorDeclaration'('ConstructorDeclaration'(
			     Modifiersopt,
			     ConstructorDeclarator,
			     Throwsopt,
			     ConstructorBody)) -->
	'Modifiersopt'(Modifiersopt),
	'ConstructorDeclarator'(ConstructorDeclarator),
	'Throwsopt'(Throwsopt),
	'ConstructorBody'(ConstructorBody).

%ConstructorDeclarator:
%        SimpleName ( FormalParameterListopt )
'ConstructorDeclarator'('ConstructorDeclarator'(
			    SimpleName,
			    '(',
			    FormalParameterListopt,
			    ')')) -->
	'SimpleName'(SimpleName),
	[tok('(',_)],
	'FormalParameterListopt'(FormalParameterListopt),
	[tok(')',TN)].

%ConstructorBody:
%        { ExplicitConstructorInvocationopt BlockStatementsopt } <error_handling_ConstructorBody accepts ;>
'ConstructorBody'(T,[tok('{',_)|S0],S):-
	T='ConstructorBody'(
		      '{',
		      ExplicitConstructorInvocationopt,
		      BlockStatementsopt,
		      '}'),
	'ExplicitConstructorInvocationopt'(ExplicitConstructorInvocationopt,S0,S1),
	'BlockStatementsopt'(BlockStatementsopt,S1,S2),
	S2=[tok(Token,TN)|S3],
	(Token=='}' -> true ; error1('} expected',TN)),
	error_handling_ConstructorBody(S3,S). % error handling

error_handling_ConstructorBody([tok(';',_)|S],S):-!.
error_handling_ConstructorBody(S,S).

%ExplicitConstructorInvocation:
%        this ( ArgumentListopt ) ;
%        super ( ArgumentListopt ) ;
'ExplicitConstructorInvocation'(T,[tok(this,_),tok('(',_)|S0],S):-!,
	T='ExplicitConstructorInvocation'(this,'(',ArgumentListopt,')',';'),
	'ArgumentListopt'(ArgumentListopt,S0,S1),
	S1=[tok(Tok1,TN)|S2],
	(Tok1==')' -> true ; error1(') expected',TN)),
	(S2=[tok(';',_)|S] -> true ; error('; expected',TN)).
'ExplicitConstructorInvocation'(T,[tok(super,_),tok('(',_)|S0],S):-
	T='ExplicitConstructorInvocation'(super,'(',ArgumentListopt,')',';'),
	'ArgumentListopt'(ArgumentListopt,S0,S1),
	S1=[tok(Tok1,TN)|S2],
	(Tok1==')' -> true ; error1(') expected',TN)),
	(S2=[tok(';',_)|S] -> true ; error('; expected',TN)).

'ExplicitConstructorInvocationopt'(ExplicitConstructorInvocation) -->
	'ExplicitConstructorInvocation'(ExplicitConstructorInvocation).
	{!}.
'ExplicitConstructorInvocationopt'(nil) --> [].



%--------  19.9  Interfaces  --------%

%InterfaceDeclaration:
%        Modifiersopt interface Identifier ExtendsInterfacesopt InterfaceBody
'InterfaceDeclaration'('InterfaceDeclaration'(
				Modifiersopt,
				interface,
				Identifier,
				ExtendsInterfacesopt,
				InterfaceBody)) -->
	'Modifiersopt'(Modifiersopt),
	[tok(interface,TN)],
	('Identifier'(Identifier) -> [] ; {error('invalid identifier',TN)}),
	'ExtendsInterfacesopt'(ExtendsInterfacesopt),
	'InterfaceBody'(InterfaceBody).

%ExtendsInterfaces:
%        extends InterfaceTypes
'ExtendsInterfaces'(T,[tok(extends,TN)|S0],S):-
	T='ExtendsInterfaces'(extends,InterfaceTypes),
	('InterfaceTypes'(InterfaceTypes,S0,S) -> true ; error('invalid name',TN)).

'ExtendsInterfacesopt'(ExtendsInterfaces) -->
	'ExtendsInterfaces'(ExtendsInterfaces).
	{!}.
'ExtendsInterfacesopt'(nil) --> [].

%InterfaceTypes :
%	InterfaceType InterfaceTypesRestopt
'InterfaceTypes'('InterfaceTypes'(
			InterfaceType,
			InterfaceTypesRestopt)) -->
	'InterfaceType'(InterfaceType),
	'InterfaceTypesRestopt'(InterfaceTypesRestopt).

%InterfaceTypesRest :
%	, InterfaceTypes
'InterfaceTypesRest'(T,[tok(',',_)|S0],S):-
	T='InterfaceTypesRest'(',',InterfaceTypes),
	'InterfaceTypes'(InterfaceTypes,S0,S).

'InterfaceTypesRestopt'(InterfaceTypesRest) -->
	'InterfaceTypesRest'(InterfaceTypesRest).
	{!}.
'InterfaceTypesRestopt'(nil) --> [].

%InterfaceBody:
%        { InterfaceMemberDeclarationsopt }
'InterfaceBody'(T,[tok('{',_)|S0],S):-!,
	T='InterfaceBody'('{',InterfaceMemberDeclarationsopt,'}'),
	'InterfaceMemberDeclarationsopt'(InterfaceMemberDeclarationsopt,S0,S1),
	S1=[tok(Token,TN)|S],
	(Token=='}' -> true ; error1('} expected',TN)).
%error handling
'InterfaceBody'(T,[tok(Tok,TN)|S0],S):-
	Tok \== '{',
	T=nil,S0=S,
	error1('{ expected',TN).

%InterfaceMemberDeclarations:
%        InterfaceMemberDeclaration InterfaceMemberDeclarationsopt
'InterfaceMemberDeclarations'('InterfaceMemberDeclarations'(
					InterfaceMemberDeclaration,
					InterfaceMemberDeclarationsopt)) -->
	'InterfaceMemberDeclaration'(InterfaceMemberDeclaration),
	'InterfaceMemberDeclarationsopt'(InterfaceMemberDeclarationsopt).

'InterfaceMemberDeclarationsopt'(InterfaceMemberDeclarations) -->
	'InterfaceMemberDeclarations'(InterfaceMemberDeclarations).
	{!}.
'InterfaceMemberDeclarationsopt'(nil) --> [].

%InterfaceMemberDeclaration:
%        ConstantDeclaration
%        AbstractMethodDeclaration
'InterfaceMemberDeclaration'('InterfaceMemberDeclaration'(
					ConstantDeclaration)) -->
	'ConstantDeclaration'(ConstantDeclaration).
'InterfaceMemberDeclaration'('InterfaceMemberDeclaration'(
					AbstractMethodDeclaration)) -->
	'AbstractMethodDeclaration'(AbstractMethodDeclaration).

%ConstantDeclaration:
%        FieldDeclaration
'ConstantDeclaration'('ConstantDeclaration'(
				FieldDeclaration)) -->
	'FieldDeclaration'(FieldDeclaration).

%AbstractMethodDeclaration:
%        MethodHeader ;
'AbstractMethodDeclaration'('AbstractMethodDeclaration'(
				MethodHeader,
				';')) -->
	'MethodHeader'(MethodHeader),
	[tok(Token,TN)],
	{Token==';' -> true ; error1('; expected',TN)}.




%--------  19.10  Arrays  --------%

%ArrayInitializer:
%        { VariableInitializersopt ,opt }
'ArrayInitializer'(T,[tok('{',_)|S0],S):-
	T='ArrayInitializer'('{',VariableInitializersopt,Comma,'}'),
	'VariableInitializersopt'(VariableInitializersopt,S0,S1),
	'Comma'(Comma,S1,S2),
	S2=[tok(Token,TN)|S],
	(Token=='}' -> true ; error1('} expected',TN)).

'Comma'(T,[tok(',',_)|S0],S):-!,S0=S,T=','.
'Comma'(T,S0,S):- S0=S,T=nil.

%VariableInitializers:
%        VariableInitializer VariableInitializersRestopt
'VariableInitializers'('VariableInitializers'(
				VariableInitializer,
				VariableInitializersRestopt)) -->
	'VariableInitializer'(VariableInitializer),
	'VariableInitializersRestopt'(VariableInitializersRestopt).

'VariableInitializersopt'(VariableInitializers) -->
	'VariableInitializers'(VariableInitializers).
	{!}.
'VariableInitializersopt'(nil) --> [].

%VariableInitializersRest:
%        , VariableInitializers
'VariableInitializersRest'(T,[tok(',',_)|S0],S):-
	T='VariableInitializersRest'(',',VariableInitializers),
	'VariableInitializers'(VariableInitializers,S0,S).

'VariableInitializersRestopt'(VariableInitializersRest) -->
	'VariableInitializersRest'(VariableInitializersRest).
	{!}.
'VariableInitializersRestopt'(nil) --> [].



%--------  Identifier  --------%

'Identifier'('Identifier'(Identifier)) -->
        [tok(Identifier,_)],
        {atom(Identifier),
	 not reserved_word(Identifier),
         atom_codes(Identifier,[TopIDCode|_]),
         ((97 =< TopIDCode,TopIDCode =< 122,!);   % alphabets
          (65 =< TopIDCode,TopIDCode =< 90,!);    % ALPHABETs
          (TopIDCode =:= 95,!);                   % '_'
          (TopIDCode =:= 36))}.                   % '$'

'Identifieropt'(Identifier) -->
	'Identifier'(Identifier),
	{!}.
'Identifieropt'(nil) --> [].


%--- ͽ ---%
:-mode reserved_word(+).
%%% Java (Version1.0)
reserved_word(abstract).	reserved_word(boolean).
reserved_word(break).		reserved_word(byte).
reserved_word(case).		reserved_word(catch).
reserved_word(char).		reserved_word(class).
reserved_word(cintinue).	reserved_word(default).
reserved_word(do).		reserved_word(double).
reserved_word(else).		reserved_word(extends).
reserved_word(false).		reserved_word(final).
reserved_word(finally). 	reserved_word(float).
reserved_word(for).		reserved_word(if).
reserved_word(implements).	reserved_word(import).
reserved_word(instanceof).	reserved_word(int).
reserved_word(interface).	reserved_word(ling).
reserved_word(native).		reserved_word(new).
reserved_word(null).		reserved_word(package).
reserved_word(private).		reserved_word(protected).
reserved_word(public).		reserved_word(return).
reserved_word(short).		reserved_word(static).
reserved_word(super).		reserved_word(switch).
reserved_word(synchronized).	reserved_word(this).
reserved_word(throws).		reserved_word(trasient).
reserved_word(true).		reserved_word(try).
reserved_word(void).		reserved_word(volatile).
reserved_word(while).

%%% DJ 
reserved_word(attribute).
reserved_word(component).
reserved_word(constraint).
reserved_word(in).
%reserved_word(event).
reserved_word(sum).
reserved_word(minimum).
reserved_word(maximum).
reserved_word(times).
reserved_word(when).





%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%       error handling       %%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% error recovery with panic mode
% skip until semicolon, or a block
errorRecovery(S0,S,WrongStatements):-
	S0 \== [],
	retract('$error'),
	errorRecovery(S0,S,WrongStatements,0).

errorRecovery([],S,WrongStatements,Count):-
	!,
	WrongStatements=[],
	S=[].
errorRecovery([tok(';',_)|S0],S,WrongStatements,0):-
	!,
	WrongStatements=[';'],
	S=S0.
errorRecovery([tok('{',_)|S0],S,WrongStatements,Count):-
	!,
	WrongStatements=['{'|WrongStatementsRest],
	NCount is Count+1,
	errorRecovery(S0,S,WrongStatementsRest,NCount).
errorRecovery([tok('}',_)|S0],S,WrongStatements,1):-
	!,
	WrongStatements=['}'],
	S=S0.
errorRecovery([tok('}',_)|S0],S,WrongStatements,Count):-
	Count > 1,!,
	NCount is Count-1,
	WrongStatements=['}'|WrongStatementsRest],
	errorRecovery(S0,S,WrongStatementsRest,NCount).
errorRecovery([tok(Token,_)|S0],S,WrongStatements,Count):-
	WrongStatements=[Token|WrongStatementsRest],
	errorRecovery(S0,S,WrongStatementsRest,Count).


%% check errors in 
%error(ErrorMessage,TokenNumber)
:-mode error(+,+).
:-mode error1(+,+).
error(EM,TN):-
	(clause('$error',true) ->
		true
	; 
		TN1 is TN+1,
	 	assertz('$error'(EM,TN1)),
		assertz('$error')	%flag
	), 
	fail.
error1(EM,TN):-
	(clause('$error',true) ->
		true
	; 
	 	assertz('$error'(EM,TN)),
		assertz('$error')	%flag
	), 
	fail.


%% report errors to users
:- mode errorReport(+,+).
errorReport(LTNList,Tokens):-
	retract('$error'(EM,TN)),!,
	getLineNumber(LTNList,TN,LN),
	write('** syntax error ** '),
	write(LN),write(':  '),write(EM),nl,
	writeLine(Tokens,LN,TN),nl,
	errorReport(LTNList,Tokens).
errorReport(_,_):-
	retract('$error'),!,	% clean up flags
	errorReport(_,_).
errorReport(_,_).


%% write a line where a syntax error is found.
writeLine(Tokens,LN,TN0):-
	writeLine(Tokens,LN,TN0,1).

:- mode writeLine(+,+,+,+).
writeLine([line(LN)|Tail],LN0,TN0,TN):-
	LN0 > LN,!,
	writeLine(Tail,LN0,TN0,TN).
writeLine([line(LN)|Tail],LN0,TN0,TN):-
	LN0==LN,!,
	writeLine_aux(Tail,TN0,TN).
writeLine([Token|Tail],LN,TN0,TN):-
	TN1 is TN+1,
	writeLine(Tail,LN,TN0,TN1).

:- mode writeLine(+,+,+).
writeLine_aux([line(_)|Tail],_,_):-!.
writeLine_aux([Token|Tail],TN0,TN):-
	TN0 =\= TN,!,
	write(Token),
	TN1 is TN+1,
	writeLine_aux(Tail,TN0,TN1).
writeLine_aux([Token|Tail],TN0,TN):-
	TN0 == TN,
	write('<<here>>'),write(Token),
	TN1 is TN+1,
	writeLine_aux(Tail,TN0,TN1).

:- mode getLineNumber(+,+,-).
getLineNumber([line(LN,TN)|Tail],TN0,LN0):-
	TN < TN0,!,
	getLineNumber(Tail,TN0,LN0).
getLineNumber([line(LN,TN)|Tail],TN0,LN0):-
	TN >= TN0,
	LN0=LN.
