%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%   File   : dj2pro.pl
%   Author : Neng-Fa ZHOU
%   Last update : 1998
%   Purpose: translate DJ trees to Prolog terms
%   COPYRIGHT (C) 1998 Neng-Fa ZHOU
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
dj2proIdentifier('Identifier'(Identifier),PIdentifier):-true : PIdentifier=Identifier.
dj2proIdentifier(_,PIdentifier):-true : cmpError(['An identifier expected']).

dj2proClassType('ClassType'(ClassOrInterfaceType),PClass):-
    dj2proClassOrInterface(ClassOrInterfaceType,PClass).
    
dj2proClassOrInterface('ClassOrInterfaceType'(Name),PClass):-
    dj2proName(Name,PClass).

dj2proType('Type'(Something),PType):-true : dj2proType(Something,PType).
dj2proType('PrimitiveType'(Something),PType):-true : dj2proType(Something,PType).
dj2proType(boolean,PType):-true : PType=boolean.
dj2proType('NumericType'(Something),PType):-true : dj2proType(Something,PType).
dj2proType('IntegralType'(Something),PType):-true : dj2proType(Something,PType).
dj2proType(byte,PType):-true : PType=byte.
dj2proType(short,PType):-true : PType=short.
dj2proType(int,PType):-true : PType=int.
dj2proType(long,PType):-true : PType=long.
dj2proType(char,PType):-true : PType=char.
dj2proType('FloatingType'(Something),PType):-true : dj2proType(Something,PType).
dj2proType(double,PType):-true : PType=double.
dj2proType(float,PType):-true : PType=float.
dj2proType('ReferenceType'(ReferenceType),PType):-true : dj2proType(ReferenceType,PType).
dj2proType('ClassOrInterfaceType'(Something),PType):-true : dj2proName(Something,PType).

dj2proType('ArrayType'(SomeType,Dims),PType):-true : 
    dj2proType(SomeType,ElmType),
    dj2proArrayType(Dims,ElmType,PType).
dj2proType(Type,PType):-
    'Name'(_)<=Type : dj2proName(Type,PType).

dj2proName('Name'(Something),PName):-true : dj2proName(Something,PName).
dj2proName('SimpleName'(Identifier),PName):-true : dj2proIdentifier(Identifier,PName).
dj2proName('QualifiedName'(Identifier,'.',Name),PName):-true :
    PName=(PIdentifier.PName1),
    dj2proIdentifier(Identifier,PIdentifier),
    dj2proName(Name,PName1).

dj2proModifiers('Modifiers'(Modifier,Modifiers),PModifiers):-
    true :
    PModifiers=[Modifier|PModifiers1],
    dj2proModifiers(Modifiers,PModifiers1).
dj2proModifiers(nil,PModifiers):-
    true :
    PModifiers=[].

dj2proArrayType('Dims'('[]',Dims),ElmType,PType):-
    true :
    PType=array(PType1),
    dj2proArrayType(Dims,ElmType,PType1).
dj2proArrayType(nil,ElmType,PType):-
    true :
    PType=ElmType.

dj2proConstraintsOrConstraintBlocks(Constraints,PConstraints):-
    'ConstraintBlocks'(_,_)<=Constraints :
    dj2proConstraintBlocks(Constraints,PConstraints).
dj2proConstraintsOrConstraintBlocks(Constraints,PConstraints):-
    true :
    dj2proConstraints(Constraints,PConstraints).


dj2proConstraintBlocks('ConstraintBlocks'(ConstraintBlock,ConstraintBlocksRest),
			PConstraintBlocks):-
    PConstraintBlocks=[PConstraintBlock|PConstraintBlocks1],
    dj2proConstraintBlock(ConstraintBlock,PConstraintBlock),
    dj2proConstraintBlocksRest(ConstraintBlocksRest,PConstraintBlocks1).

dj2proConstraintBlock('ConstraintBlock'('{',Constraints,'}'),PConstraintBlock):-true :
    PConstraintBlock={PConstraints},
    dj2proConstraints(Constraints,PConstraints).
    
dj2proConstraintBlocksRest('ConstraintBlocksRest'(',',ConstraintBlocks),PConstraintBlocks):-
    true :
    dj2proConstraintBlocks(ConstraintBlocks,PConstraintBlocks).
dj2proConstraintBlocksRest(nil,PConstraintBlocks):-
    true : PConstraintBlocks=[].

dj2proConstraints('Constraints'(Constraint,ConstraintsRest),PConstraints):-
    true :
    PConstraints=[PConstraint|PConstraints1],
    dj2proConstraint(Constraint,PConstraint),
    dj2proConstraintsRest(ConstraintsRest,PConstraints1).
dj2proConstraints(nil,PConstraints):-true : PConstraints=[].

dj2proConstraintsRest('ConstraintsRest'(';',Constraints),PConstraints):-
    true :
    dj2proConstraints(Constraints,PConstraints).
dj2proConstraintsRest(nil,PConstraints):-
    true :
    PConstraints=[].

dj2proConstraint('Constraint'(ConstraintFactor,CompositeConstraintRest),PConstraint):-
    true : 
    dj2proConstraintFactor(ConstraintFactor,PConstraintFactor),
    dj2proCompositeConstraintRest(CompositeConstraintRest,PConstraintFactor,PConstraint).
dj2proConstraint('Constraint'('~',ConstraintFactor),PConstraint):-
    true : 
    PConstraint='$not'(PConstraintFactor),
    dj2proConstraintFactor(ConstraintFactor,PConstraintFactor).
dj2proConstraint('Constraint'(ForConstraint),PConstraint):-
    true : 
    dj2proConstraint(ForConstraint,PConstraint).
dj2proConstraint('ForConstraint'(for,'(',EnumeratorConditionList, ')',Constraint),PConstraint):-
    dj2proConstraint(Constraint,PConstraint1),
    PConstraint='$for'(PConstraint1,PEnumeratorConditionList),
    dj2proEnumeratorConditionList(EnumeratorConditionList,PEnumeratorConditionList).

dj2proEnumeratorConditionList('EnumeratorConditionList'(EnumeratorOrCondition,EnumeratorConditionListRest),PEnumeratorConditionList):-true :
    PEnumeratorConditionList=[PEnumeratorOrCondition|PEnumeratorConditionList1],
    dj2proEnumeratorOrCondition(EnumeratorOrCondition,PEnumeratorOrCondition),
    dj2proEnumeratorConditionList(EnumeratorConditionListRest,PEnumeratorConditionList1).
dj2proEnumeratorConditionList('EnumeratorConditionListRest'(',',EnumeratorConditionList),PEnumeratorConditionList):-true :
    dj2proEnumeratorConditionList(EnumeratorConditionList,PEnumeratorConditionList).
dj2proEnumeratorConditionList(nil,PEnumeratorConditionList):-true :
    PEnumeratorConditionList=[].

dj2proEnumeratorOrCondition('EnumeratorOrCondition'(Constraint),PEnumerator):-
    true :
    dj2proConstraint(Constraint,PEnumerator).

dj2proConstraintFactor('ConstraintFactor'(Name,in,Range),PConstraint):-
    true : 
    dj2proName(Name,PName),
    dj2proRange(Range,PRange),
    PConstraint=(PName in PRange).
dj2proConstraintFactor('ConstraintFactor'(Something),PConstraint):-
    true : 
    dj2proConstraintFactor(Something,PConstraint).
dj2proConstraintFactor('ArithmeticConstraint'(Exp1,Rest),PConstraint):-
    true :
    dj2proDJExpression(Exp1,PExp1),
    dj2proArithmeticConstraintRest(Rest,PR,PExp2),
    PConstraint=..[PR,PExp1,PExp2].
dj2proConstraintFactor('ConstraintCall'(Identifier,'(',ConstraintArgumentList,')'),PConstraint):-
    true :
    dj2proIdentifier(Identifier,PIdentifier),
    dj2proConstraintArgumentList(ConstraintArgumentList,PArgumentList),
    PConstraint=..[PIdentifier|PArgumentList].
dj2proConstraintFactor(Constraint,PConstraint):-
    'ConstraintBlock'('{',Constraints,'}')<=Constraint :
    dj2proConstraintBlock(Constraint,PConstraint).
dj2proConstraintFactor(ConstraintFactor,PConstraint):-
    'ConstraintFactor'('(',Constraint,')')<=ConstraintFactor :
    dj2proConstraint(Constraint,PConstraint).


dj2proCompositeConstraintRest(nil,PConstraintFactor,PConstraint):-true :
    PConstraint=PConstraintFactor.
dj2proCompositeConstraintRest(
    'CompositeConstraintRest'(R,Constraint),
    PConstraintFactor,PConstraint):-true :
    dj2proOperator(R,PR),
    PConstraint=..[PR,PConstraintFactor,PConstraint1],
    dj2proConstraint(Constraint,PConstraint1).

dj2proArithmeticConstraintRest(
    'ArithmeticConstraintRest'(R,Expression),
    PR,PExp):-
    dj2proOperator(R,PR),
    dj2proDJExpression(Expression,PExp).

dj2proArgumentList(
    'ArgumentList'(Expression,ArgumentListRest),PArgumentList):-
    true :
    PArgumentList=[PExpression|PArgumentList1],
    dj2proExpression(Expression,PExpression),
    (ArgumentListRest==nil->PArgumentList1=[];
     ArgumentListRest='ArgumentListRest'(',',ArgumentList),
     dj2proArgumentList(ArgumentList,PArgumentList1)).

dj2proConstraintArgumentList('ConstraintArgumentList'(ConstraintArgument,ConstraintArgumentListRest),PTerm):-
    true :
    PTerm=[PArg|PTerm1],
    dj2proConstraintArgument(ConstraintArgument,PArg),
    dj2proConstraintArgumentList(ConstraintArgumentListRest,PTerm1).
dj2proConstraintArgumentList(nil,PTerm):-
    true : PTerm=[].
dj2proConstraintArgumentList('ConstraintArgumentListRest'(',',ConstraintArgumentList),PTerm):-
    true :
    dj2proConstraintArgumentList(ConstraintArgumentList,PTerm).

dj2proConstraintArgument('ConstraintArgument'(Something),PTerm):-
    true :
    dj2proConstraintArgument(Something,PTerm).
dj2proConstraintArgument('AnonymousArray'('{',AnonymousArrayElements,'}'),PTerm):-
    true :
    dj2proAnonymousArrayElements(AnonymousArrayElements,PElements,0,Size),
    PTerm=anonymousArray(Size,PElements).
dj2proConstraintArgument(Expression,PTerm):-
    true :
    dj2proExpression(Expression,PTerm).
    
dj2proAnonymousArrayElements('AnonymousArrayElements'(ConstraintArgument,AnonymousArrayElementsRest),PTerm,Size0,Size):-
    true :
    PTerm=[PArg|PTerm1],
    Size1 is Size0+1,
    dj2proConstraintArgument(ConstraintArgument,PArg),
    dj2proAnonymousArrayElements(AnonymousArrayElementsRest,PTerm1,Size1,Size).
dj2proAnonymousArrayElements('AnonymousArrayElementsRest'(',',AnonymousArrayElements),PTerm,Size0,Size):-
    true :
    dj2proAnonymousArrayElements(AnonymousArrayElements,PTerm,Size0,Size).
dj2proAnonymousArrayElements(nil,PTerm,Size0,Size):-
    true :
    Size=Size0,
    PTerm=[].

:-mode dj2proOperator(+,-).
:-mode dj2proOperator(-,+).
dj2proOperator('=','#=').
dj2proOperator('==','#=').
dj2proOperator('!=','#\=').
dj2proOperator('<=','#=<').
dj2proOperator('<','#<').
dj2proOperator('>=','#>=').
dj2proOperator('>','#>').
dj2proOperator('->','$entail').
dj2proOperator('||','$or').
dj2proOperator('&&','$and').
dj2proOperator('~','$not').
dj2proOperator('+','+').
dj2proOperator('-','-').
dj2proOperator('*','*').
dj2proOperator('/','/').

dj2proRange('Range'(DJExp1,'..',DJExp2),PRange):-
    true :
    PRange=(I1..I2),
    dj2proDJExpression(DJExp1,I1),
    dj2proDJExpression(DJExp2,I2).

dj2proRange('Range'('{',ArgumentList,'}'),PRange):-
    true :
    dj2proArgumentList(ArgumentList,PList),
    PRange=PList.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
dj2proDJExpression('DJExpression'(Expression),PExpression):-
    dj2proAdditiveExpression(Expression,PExpression).

dj2proExpression(
    'Expression'(AssignmentExpression),PExpression):-
    true :
    dj2proExpression(AssignmentExpression,PExpression).
dj2proExpression(
    'AssignmentExpression'(ConditionalExpression),PExpression):-
    true :
    dj2proExpression(ConditionalExpression,PExpression).
dj2proExpression(
    'ConditionalExpression'(ConditionalOrExpression,Rest),PExpression):-
    true :
    dj2proExpression(ConditionalOrExpression,PExpression),
    (Rest==nil->true;
     cmpError(['Conditional expression cannot appear in constraints'])).
dj2proExpression(
   'ConditionalOrExpression'(ConditionalAndExpression,Rest),PExpression):-
    true :
    dj2proExpression(ConditionalAndExpression,PExpression),
    (Rest==nil->true;cmpError(['Conditional or expression || cannot appear in constraints'])).
dj2proExpression(
    'ConditionalAndExpression'(InclusiveOrExpression,Rest),PExpression):-
    true :
    dj2proExpression(InclusiveOrExpression,PExpression),
    (Rest==nil->true;
     cmpError(['Conditional and expression && cannot appear in constraints'])).
dj2proExpression(
    'InclusiveOrExpression'(ExclusiveOrExpression,Rest),PExpression):-
    true :
    dj2proExpression(ExclusiveOrExpression,PExpression),
    (Rest==nil->true;
     cmpError(['Inclusive or expression | cannot appear in constraints'])).
dj2proExpression(
    'ExclusiveOrExpression'(AndExpression,Rest),PExpression):-
    true :
    dj2proExpression(AndExpression,PExpression),
    (Rest==nil -> true;
     cmpError(['Exclusive or expression '^' cannot appear in constraints'])).
dj2proExpression(
    'AndExpression'(EqualityExpression,Rest),PExpression):-
    true :
    dj2proExpression(EqualityExpression,PExpression),
    (Rest==nil->true;
     cmpError(['And expression & cannot appear in constraints'])).
dj2proExpression(
    'EqualityExpression'(RelationalExpression,Rest),PExpression):-
    true :
    dj2proExpression(RelationalExpression,PExpression),
    (Rest==nil->true;
     cmpError(['Equality expression cannot appear in constraints: ',Rest])).
dj2proExpression(
    'RelationalExpression'(ShiftExpression,Rest),PExpression):-
    true :
    dj2proExpression(ShiftExpression,PExpression),
    (Rest==nil->true;
     cmpError(['Relational expression cannot appear in constraints'])).
dj2proExpression(
    'ShiftExpression'(AdditiveExpression,Rest),PExpression):-
    true :
    dj2proAdditiveExpression(AdditiveExpression,PExpression),
    (Rest==nil->true;
     cmpError(['Additive expression cannot appear in constraints'])).

dj2proAdditiveExpression(
   'AdditiveExpression'(MultiplicativeExpression,Rest),
   PExpression):-
    true :
    dj2proMultiplicativeExpression(MultiplicativeExpression,PMultiplicativeExpression),
    dj2proAdditiveExpressionRest(Rest,PMultiplicativeExpression,PExpression).

dj2proAdditiveExpressionRest(nil,PMultiplicativeExpression,PExpression):-
    true :
    PExpression= PMultiplicativeExpression.
dj2proAdditiveExpressionRest(
    'AdditiveExpressionRest'(R,AdditiveExpression),
    PMultiplicativeExpression,PExpression):-
    true :
    dj2proOperator(R,PR),
    PExpression=..[PR,PMultiplicativeExpression,PAdditiveExpression],
    dj2proAdditiveExpression(AdditiveExpression,PAdditiveExpression).

dj2proMultiplicativeExpression(
    'MultiplicativeExpression'(UnaryExpression,Rest),
    PExpression):-
    true :
    dj2proUnaryExpression(UnaryExpression,PUnaryExpression),
    dj2proMultiplicativeExpressionRest(Rest,PUnaryExpression,PExpression).

dj2proMultiplicativeExpressionRest(nil,PUnaryExpression,PExpression):-
    true :
    PExpression= PUnaryExpression.
dj2proMultiplicativeExpressionRest(
    'MultiplicativeExpressionRest'(R,MultiplicativeExpression),
    PUnaryExpression,PExpression):-
    true :
    dj2proOperator(R,PR),
    PExpression=..[PR,PUnaryExpression,PMultiplicativeExpression],
    dj2proMultiplicativeExpression(MultiplicativeExpression,PMultiplicativeExpression).

dj2proUnaryExpression(
    'UnaryExpression'(Something),PExpression):-
    true :
    dj2proUnaryExpression(Something,PExpression).
dj2proUnaryExpression(
    'UnaryExpression'('+',Something),PExpression):-
    true :
    dj2proUnaryExpression(Something,PExpression).
dj2proUnaryExpression(
    'UnaryExpression'('-',Something),PExpression):-
    true :
    PExpression=(-PExpression1),
    dj2proUnaryExpression(Something,PExpression1).
dj2proUnaryExpression(
    'UnaryExpressionNotPlusMinus'(PostfixExpression),PExpression):-
    true :
    dj2proUnaryExpression(PostfixExpression,PExpression).
dj2proUnaryExpression(
    'PostfixExpression'(PrimaryOrName,PlusPlusOrMinusMinus),PExpression):-   %%% not completed
    true :
    dj2proUnaryExpression(PrimaryOrName,PExpression).
dj2proUnaryExpression(
    'PrimaryOrName'(Something),PExpression):-
    true :
    dj2proUnaryExpression(Something,PExpression). %Primary or Name
dj2proUnaryExpression(
    'Primary'(PrimaryNoNewArray),PExpression):-
    true :
    dj2proUnaryExpression(PrimaryNoNewArray,PExpression).
dj2proUnaryExpression(
    'PrimaryNoNewArray'(this),PExpression):-
    true :
    PExpression=this.
dj2proUnaryExpression(
   'Literal'(Literal),PExpression):-
    true :
    dj2proUnaryExpression(Literal,PExpression).
dj2proUnaryExpression(
   'IntegerLiteral'(Integer),PExpression):-
    atom(Integer) :
    atom_to_integer(Integer,PExpression).
dj2proUnaryExpression(
   'IntegerLiteral'(Integer),PExpression):-
    true :
    PExpression=Integer.
dj2proUnaryExpression(
    'StringLiteral'(_,String,_),PExpression):-
    true :
    PExpression='StringConst'(String).
dj2proUnaryExpression(
   'BooleanLiteral'(Boolean),PExpression):-
    true :
    PExpression=Boolean.
dj2proUnaryExpression(Expression,PExpression):-
    'Name'(_)<=Expression :
    dj2proName(Expression,PExpression).
dj2proUnaryExpression(
 'PrimaryNoNewArray'(Something),PExpression):- %FieldAccess,MethodInvocation,ArrayAccess
    true :
    dj2proUnaryExpression(Something,PExpression).
dj2proUnaryExpression(
    'PrimaryNoNewArray'('(',Expression,')'),PExpression):-
    true :
    dj2proExpression(Expression,PExpression).
dj2proUnaryExpression(
    'FieldAccess'(super,'.', Identifier),PExpression):-
    true :
    PExpression=(super.PIdentifier),
    dj2proIdentifier(Identifier,PIdentifier).	
dj2proUnaryExpression(
    'FieldAccess'(Primary,'.', Identifier),PExpression):-
    true :
    PExpression=(PPrimary.PIdentifier),
    dj2proIdentifier(Identifier,PIdentifier),
    dj2proUnaryExpression(Primary,PPrimary).
dj2proUnaryExpression(
    'ArrayAccess'(	Name,'[', Expression,']'),PExpression):-
    'Name'(_)<=Name :
    PExpression='array'(PName,PExpression1),
    dj2proName(Name,PName),
    dj2proExpression(Expression,PExpression1).
dj2proUnaryExpression(
    'ArrayAccess'(	PrimaryNoNewArray,'[', Expression,']'),PExpression):-
    true :
    PExpression='array'(PPrimaryNoNewArray,PExpression1),	
    dj2proUnaryExpression(PrimaryNoNewArray,PPrimaryNoNewArray),
    dj2proExpression(Expression,PExpression1).
dj2proUnaryExpression(
    'MethodInvocation'(Name,'(',ArgumentList,')'),PExpression):-
    'Name'('SimpleName'(Identifier))<=Name :
    dj2proIdentifier(Identifier,PIdentifier),
    dj2proArgumentList(ArgumentList,PArgumentList),
    PExpression=..[PIdentifier|PArgumentList].
dj2proUnaryExpression(
    'MethodInvocation'(Name,'(',ArgumentList,')'),PExpression):-
    'Name'(QualifiedName)<=Name,
    'QualifiedName'(Identifier,'.',Name1)<=QualifiedName :
    dj2proIdentifier(Identifier,PIdentifier),
    dj2proUnaryExpression('MethodInvocation'(Name1,'(',ArgumentList,')'),PExpression1),
    PExpression=(PIdentifier.PExpression1).
dj2proUnaryExpression(
    'MethodInvocation'(Primary,'.',Identifier,'(',ArgumentList,')'),PExpression):-
    dj2proUnaryExpression(Primary,PPrimary),
    dj2proIdentifier(Identifier,PIdentifier),
    dj2proArgumentList(ArgumentList,PArgumentList),
    PCall=..[PIdentifier|PArgumentList],
    PExpression=(PPrimary.PCall).
dj2proUnaryExpression(
    'MethodInvocation'(super,'.',Identifier,'(',ArgumentList,')'),PExpression):-
    dj2proIdentifier(Identifier,PIdentifier),
    dj2proArgumentList(ArgumentList,PArgumentList),
    PCall=..[PIdentifier|PArgumentList],
    PExpression=(super.PCall).
dj2proUnaryExpression(
   'Primary'(sum,'(','Template'(Expression),',',EnumeratorConditionList,')'),PExpression):-
    true :
    dj2proExpression(Expression,PTemplate),
    dj2proEnumeratorConditionList(EnumeratorConditionList,Args),
    PExpression=..['$sum',PTemplate,Args].
dj2proUnaryExpression(
    'Primary'(minimum,'(','Template'(Expression),',',EnumeratorConditionList,')'),PExpression):-
    true :
    dj2proExpression(Expression,PTemplate),
    dj2proEnumeratorConditionList(EnumeratorConditionList,Args),
    PExpression=..['$minimum',PTemplate,Args].
dj2proUnaryExpression(
    'Primary'(maximum,'(','Template'(Expression),',',EnumeratorConditionList,')'),PExpression):-
    true :
    dj2proExpression(Expression,PTemplate),
    dj2proEnumeratorConditionList(EnumeratorConditionList,Args),
    PExpression=..['$maximum',PTemplate,Args].
    
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

dj2proDimExprs('DimExprs'(DimExpr,DimExprs),PDims):-
    true :
    PDims=[PDim|PDims1],
    dj2proDimExpr(DimExpr,PDim),
    dj2proDimExprs(DimExprs,PDims1).
dj2proDimExprs(nil,PDims):-
    true : PDims=[].

dj2proDimExpr('DimExpr'('[',Expression,']'),PDim):-
    true :
    dj2proExpression(Expression,PDim).

atom_to_integer(Atom,Integer):-
    name(Atom,String),
    Zero=0'0,
    atoms_to_integer(String,Zero,0,Integer).

atoms_to_integer([],Zero,I0,I):-true : I=I0.
atoms_to_integer([D|Ds],Zero,I0,I):-D>=48, D=<57 :
    I1 is I0*10+D-Zero,
    atoms_to_integer(Ds,Zero,I1,I).
atoms_to_integer([D|Ds],Zero,I0,I):-true :
    cmpError(['Integer expected']).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
dj2proFormalParameterList( 'FormalParameterList'(FormalParameter,FormalParameterListRest),PParameters):-
    true :
    PParameters=[PParameter|PParameters1],
    dj2proFormalParameter(FormalParameter,PParameter),
    dj2proFormalParameterList(FormalParameterListRest,PParameters1).
dj2proFormalParameterList( 'FormalParameterListRest'(',',FormalParameterList),PParameters):-
    true :
    dj2proFormalParameterList(FormalParameterList,PParameters).
dj2proFormalParameterList(nil,PParameters):-
    true :
    PParameters=[].
    
dj2proFormalParameter('FormalParameter'(Type,VariableDeclaratorId),PParameter):-
    true :
    PParameter=(PType,Var),
    dj2proType(Type,PType1),
    dj2proVariableDeclaratorId(VariableDeclaratorId,PType1,PType,Var).

dj2proVariableDeclaratorId('VariableDeclaratorId'(Identifier,nil),Type0,Type,Var):-
    true :
    Var=PIdentifier,
    Type=Type0,
    dj2proIdentifier(Identifier,PIdentifier).
dj2proVariableDeclaratorId('VariableDeclaratorId'(Identifier,Dims),Type0,Type,Var):-
    true :
    dj2proIdentifier(Identifier,Var),
    dj2proArrayType(Dims,Type0,Type).

dj2proConstraintBody('ConstraintBody'(ConstraintBlock),PConstraintBody):-
    true :
    dj2proConstraintBlock(ConstraintBlock,PConstraintBody).
    


    
    
