//
// HiRiseDemo: a Demo Application of the HiRise Constraint Solver
//
// Copyright (C) 1998 Hiroshi HOSOBE
//
////////////////////////////////////////////////////////////////

// TreeView.cpp : CTreeView NX̓̒`s܂B
//

#include <time.h>
#include "stdafx.h"
#include "HiRiseDemo.h"

#include "MainFrm.h"
#include "TreeDoc.h"
#include "TreeView.h"

#include "TreeNode.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// CTreeView

IMPLEMENT_DYNCREATE(CTreeView, CView)

BEGIN_MESSAGE_MAP(CTreeView, CView)
	//{{AFX_MSG_MAP(CTreeView)
	ON_WM_MOUSEMOVE()
	ON_WM_LBUTTONDOWN()
	ON_WM_LBUTTONUP()
	ON_WM_RBUTTONDOWN()
	ON_COMMAND(ID_TREE_ADD, OnTreeAdd)
	ON_COMMAND(ID_TREE_REMOVE, OnTreeRemove)
	ON_COMMAND(ID_TREE_STAY, OnTreeStay)
	ON_COMMAND(ID_TREE_LINE, OnTreeLine)
	//}}AFX_MSG_MAP
	// WR}h
	ON_COMMAND(ID_FILE_PRINT, CView::OnFilePrint)
	ON_COMMAND(ID_FILE_PRINT_DIRECT, CView::OnFilePrint)
	ON_COMMAND(ID_FILE_PRINT_PREVIEW, CView::OnFilePrintPreview)
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CTreeView NX̍\z/

CTreeView::CTreeView()
#ifdef WEAKER
: m_editX(3), m_editY(3)
#else // !WEAKER
: m_editX(2), m_editY(2)
#endif // WEAKER
{
	// TODO: ̏ꏊɍ\zp̃R[hǉĂB

	m_firstDraw = TRUE;
	m_inputMode = IM_NORMAL;
	m_pDraggedTreeNode = 0;
}

CTreeView::~CTreeView()
{
	while (m_stayTreeData.GetSize() > 0) {
		StayTreeData* pData = (StayTreeData*) m_stayTreeData.GetAt(0);
		m_stayTreeData.RemoveAt(0);
		delete pData;
	}

	while (m_lineTreeData.GetSize() > 0) {
		LineTreeData* pData = (LineTreeData*) m_lineTreeData.GetAt(0);
		m_lineTreeData.RemoveAt(0);
		delete pData;
	}
}

BOOL CTreeView::PreCreateWindow(CREATESTRUCT& cs)
{
	// TODO: ̈ʒu CREATESTRUCT cs C Window NX܂̓X^C
	//  CĂB

	return CView::PreCreateWindow(cs);
}

/////////////////////////////////////////////////////////////////////////////
// CTreeView NX̕`

void CTreeView::CalcEndPoints(CRect rect, 
									CPoint point0, CPoint point1, 
									CPoint& end0, CPoint& end1)
{
	CPoint end[4];
	double a, b, c;
	double x, y;
	int i;

	a = point1.y - point0.y;
	b = -(point1.x - point0.x);
	c = point0.x * a + point0.y * b;

	int n = 0;

	if (b != 0) {
		// left
		y = (c - a * rect.left) / b;
		if (y >= rect.top && y <= rect.bottom) {
			end[n].x = rect.left;
			end[n].y = (LONG) y;
			n++;
		}
		
		// right
		y = (c - a * rect.right) / b;
		if (y >= rect.top && y <= rect.bottom) {
			end[n].x = rect.right;
			end[n].y = (LONG) y;
			n++;
		}
	}

	if (a != 0) {
		// top
		x = (c - b * rect.top) / a;
		if (x >= rect.left && x <= rect.right) {
			end[n].x = (LONG) x;
			end[n].y = rect.top;
			n++;
		}

		// bottom
		x = (c - b * rect.bottom) / a;
		if (x >= rect.left && x <= rect.right) {
			end[n].x = (LONG) x;
			end[n].y = rect.bottom;
			n++;
		}
	}

	if (n == 0)
		end0.x = end0.y = end1.x = end1.y = 0;
	else if (n == 1)
		end0 = end1 = end[0];
	else if (n == 2) {
		end0 = end[0];
		end1 = end[1];
	}
	else if (n > 2) {
		end0 = end[0];
		for (i = 1; i < n; i++) {
			if (end[i].x != end0.x || end[i].y != end0.y) {
				end1 = end[i];
				break;
			}
		}
		if (i == n)
			end1 = end0;
	}
}

void CTreeView::DrawTree(TreeNode* pTree, int nodeSize, CDC* pDC)
{
	int i;

	CPoint p;
	p.x = (LONG) *pTree->m_x;
	p.y = (LONG) *pTree->m_y;

	for (i = 0; i < pTree->m_children.GetSize(); i++) {
		TreeNode* child = (TreeNode*) pTree->m_children.GetAt(i);
		if (child != 0) {
			pDC->MoveTo(p);
			pDC->LineTo((LONG) *child->m_x, (LONG) *child->m_y);
		}
	}

	CRect rect;
	rect.left = p.x - nodeSize / 2;
	rect.right = p.x + nodeSize / 2;
	rect.top = p.y - nodeSize / 2;
	rect.bottom = p.y + nodeSize / 2;

	if (pTree == m_pDraggedTreeNode)
		pDC->FillSolidRect(&rect, RGB(0, 0xff, 0));
	else {
		BOOL done = FALSE;

		for (i = 0; i < m_stayTreeData.GetSize(); i++) {
			StayTreeData* pData = (StayTreeData*)
				m_stayTreeData.GetAt(i);
			if (pData->m_pNode == pTree) {
				pDC->FillSolidRect(&rect, RGB(0xff, 0, 0));
				done = TRUE;
				break;
			}
		}

		if (!done) {
			for (i = 0; i < m_lineTreeData.GetSize(); i++) {
				LineTreeData* pData = (LineTreeData*)
					m_lineTreeData.GetAt(i);
				if (pData->m_pNode == pTree) {
					pDC->FillSolidRect(&rect, RGB(0xff, 0, 0));
					done = TRUE;
					break;
				}
			}
		}

		if (!done)
			pDC->FillSolidRect(&rect, RGB(0, 0, 0xff));
	}

	for (i = 0; i < pTree->m_children.GetSize(); i++) {
		TreeNode* child = (TreeNode*) pTree->m_children.GetAt(i);
		if (child != 0)
			DrawTree(child, nodeSize, pDC);
	}
}

void CTreeView::OnDraw(CDC* pDC)
{
	CTreeDoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);

	// TODO: ̏ꏊɃlCeBu f[^p̕`R[hǉ܂B

	if (m_firstDraw) {
		char buf[BUFSIZ];
		sprintf(buf, "plan: %f ms, exec: %f ms (#vars: %d, #cons: %d)",
			    pDoc->planTimeForConstruction, pDoc->execTimeForConstruction,
				pDoc->m_solver.countVars(), pDoc->m_solver.countCons());
		extern CHiRiseDemoApp theApp;
		((CMainFrame*) theApp.m_pMainWnd)->SetStatusBar(buf);
		m_firstDraw = FALSE;
	}

	int i;

	CRect rect;
	GetClientRect(rect);

	CPen *pOldPen = pDC->GetCurrentPen();

	if (m_inputMode == IM_SPECIFY_LINE) {
		CPoint end0;
		CPoint end1;
		CalcEndPoints(rect, m_lineEnd0, m_lineEnd1, end0, end1);

		CPen pen;
		pen.CreatePen(PS_SOLID, pDoc->m_lineWidth, RGB(0, 0xff, 0));
		pDC->SelectObject(&pen);

		pDC->MoveTo(end0);
		pDC->LineTo(end1);
	}

	CPen pen;
	pen.CreatePen(PS_SOLID, pDoc->m_lineWidth, RGB(0xff, 0, 0));
	pDC->SelectObject(&pen);

	for (i = 0; i < m_lineTreeData.GetSize(); i++) {
		LineTreeData* pData = (LineTreeData*)
			m_lineTreeData.GetAt(i);

		CPoint end0;
		CPoint end1;
		CalcEndPoints(rect, pData->m_lineEnd0, pData->m_lineEnd1,
					  end0, end1);

		pDC->MoveTo(end0);
		pDC->LineTo(end1);
	}

	CPen pen1;
	pen1.CreatePen(PS_SOLID, pDoc->m_lineWidth, RGB(0, 0, 0));
	pDC->SelectObject(&pen1);

	DrawTree(pDoc->m_pTree, pDoc->m_nodeSize, pDC);

	pDC->SelectObject(pOldPen);
}

void CTreeView::planAndExec()
{
	CTreeDoc* pDoc = GetDocument();

	CWaitCursor wc;

	clock_t t0 = clock();
	pDoc->m_solver.plan();
	clock_t t1 = clock();
	pDoc->m_solver.execute();
	clock_t t2 = clock();

	double planTime = (t1 - t0) / (double) CLOCKS_PER_SEC * 1000;
	double execTime = (t2 - t1) / (double) CLOCKS_PER_SEC * 1000;

	char buf[BUFSIZ];
	sprintf(buf, "plan: %f ms, exec: %f ms (#vars: %d, #cons: %d)",
			planTime, execTime,
			pDoc->m_solver.countVars(), pDoc->m_solver.countCons());

	extern CHiRiseDemoApp theApp;
	((CMainFrame*) theApp.m_pMainWnd)->SetStatusBar(buf);
}

void CTreeView::error(char* fname, char* message)
{
	if (message == 0) {
		char buf[BUFSIZ];
		sprintf(buf, "CTreeView::%d", fname);
		perror(buf);
	}
	else
		fprintf(stderr, "CTreeView::%s: %s\n", fname, message);

	exit(1);
}

/////////////////////////////////////////////////////////////////////////////
// CTreeView NẌ

BOOL CTreeView::OnPreparePrinting(CPrintInfo* pInfo)
{
	// ftHg̈
	return DoPreparePrinting(pInfo);
}

void CTreeView::OnBeginPrinting(CDC* /*pDC*/, CPrintInfo* /*pInfo*/)
{
	// TODO: O̓ʂȏǉĂB
}

void CTreeView::OnEndPrinting(CDC* /*pDC*/, CPrintInfo* /*pInfo*/)
{
	// TODO: ̌㏈ǉĂB
}

/////////////////////////////////////////////////////////////////////////////
// CTreeView NX̐ff

#ifdef _DEBUG
void CTreeView::AssertValid() const
{
	CView::AssertValid();
}

void CTreeView::Dump(CDumpContext& dc) const
{
	CView::Dump(dc);
}

CTreeDoc* CTreeView::GetDocument() // fobO o[W̓CCłB
{
	ASSERT(m_pDocument->IsKindOf(RUNTIME_CLASS(CTreeDoc)));
	return (CTreeDoc*)m_pDocument;
}
#endif //_DEBUG

/////////////////////////////////////////////////////////////////////////////
// CTreeView NX̃bZ[W nh

TreeNode* CTreeView::PickTreeNode(TreeNode* pTree, int nodeSize,
										CPoint point)
{
	int i;

	if (point.x >= (LONG) *pTree->m_x - nodeSize / 2 &&
		point.x <= (LONG) *pTree->m_x + nodeSize / 2 &&
	    point.y >= (LONG) *pTree->m_y - nodeSize / 2 &&
		point.y <= (LONG) *pTree->m_y + nodeSize / 2)
		return pTree;

	for (i = 0; i < pTree->m_children.GetSize(); i++) {
		TreeNode* child = (TreeNode*) pTree->m_children.GetAt(i);
		if (child != 0) {
			TreeNode* pickedNode = PickTreeNode(child, nodeSize, point);
			if (pickedNode != 0)
				return pickedNode;
		}
	}

	return 0;
}

void CTreeView::OnMouseMove(UINT nFlags, CPoint point) 
{
	// TODO: ̈ʒuɃbZ[W nhp̃R[hǉ邩܂̓ftHg̏ĂяoĂ
	
	CTreeDoc* pDoc = GetDocument();

	switch (m_inputMode) {
	case IM_NORMAL:
		break;
	case IM_DRAG_OBJECT:
		m_editX.set((HRValue) point.x);
		m_editY.set((HRValue) point.y);
		pDoc->m_solver.execute();

		RedrawWindow();
		break;
	case IM_SPECIFY_LINE:
		m_lineEnd1 = point;

		RedrawWindow();
		break;
	}

	CView::OnMouseMove(nFlags, point);
}

void CTreeView::OnLButtonDown(UINT nFlags, CPoint point) 
{
	CTreeDoc* pDoc = GetDocument();

	switch (m_inputMode) {
	case IM_NORMAL:
		m_pDraggedTreeNode = PickTreeNode(pDoc->m_pTree, pDoc->m_nodeSize,
										  point);

		if (m_pDraggedTreeNode != 0) {
			m_inputMode = IM_DRAG_OBJECT;

			m_editX.clear();
			m_editX.add(m_pDraggedTreeNode->m_x, (HRValue) point.x);
			pDoc->m_solver.add(m_editX);
		
			m_editY.clear();
			m_editY.add(m_pDraggedTreeNode->m_y, (HRValue) point.y);
			pDoc->m_solver.add(m_editY);

			RedrawWindow();

			planAndExec();
		}
		break;
	case IM_DRAG_OBJECT:
		m_inputMode = IM_NORMAL;
		break;
	case IM_SPECIFY_LINE:
		break;
	}

	CView::OnLButtonDown(nFlags, point);
}

void CTreeView::OnLButtonUp(UINT nFlags, CPoint point) 
{
	// TODO: ̈ʒuɃbZ[W nhp̃R[hǉ邩܂̓ftHg̏ĂяoĂ
	
	CTreeDoc* pDoc = GetDocument();

	switch (m_inputMode) {
	case IM_NORMAL:
		break;
	case IM_DRAG_OBJECT:
		m_inputMode = IM_NORMAL;

		pDoc->m_solver.remove(m_editX);		
		pDoc->m_solver.remove(m_editY);		

		m_pDraggedTreeNode = 0;

		RedrawWindow();

		planAndExec();
		break;
	case IM_SPECIFY_LINE:
		m_inputMode = IM_NORMAL;

		LineTreeData* pData = new LineTreeData;
		if (pData == 0)
			error("OnLButtonUp");

		pData->m_pNode = m_pLinedTreeNode;
		pData->m_lineEnd0 = m_lineEnd0;
		pData->m_lineEnd1 = m_lineEnd1;

		pData->m_line.clear();
		pData->m_line.add(m_pLinedTreeNode->m_x,
						  (HRValue) m_lineEnd1.y - m_lineEnd0.y);
		pData->m_line.add(m_pLinedTreeNode->m_y,
						  (HRValue) -(m_lineEnd1.x - m_lineEnd0.x));
		pData->m_line.setConst((HRValue) m_lineEnd0.x * (m_lineEnd1.y - m_lineEnd0.y)
							   - m_lineEnd0.y * (m_lineEnd1.x - m_lineEnd0.x));
		pDoc->m_solver.add(pData->m_line);

		m_lineTreeData.Add(pData);

		RedrawWindow();

		planAndExec();
		break;
	}

	CView::OnLButtonUp(nFlags, point);
}

void CTreeView::RemoveStaysAndLines(TreeNode* pNode)
{
	int i;

	for (i = 0; i < pNode->m_children.GetSize(); i++) {
		TreeNode* child = (TreeNode*) pNode->m_children.GetAt(i);
		if (child != 0)
			RemoveStaysAndLines(child);
	}

	for (i = 0; i < m_stayTreeData.GetSize(); i++) {
		StayTreeData* pData = (StayTreeData*) m_stayTreeData.GetAt(i);
		if (pData->m_pNode == pNode) {
			m_stayTreeData.RemoveAt(i);
			delete pData;
			break;
		}
	}
	
	for (i = 0; i < m_lineTreeData.GetSize(); i++) {
		LineTreeData* pData = (LineTreeData*) m_lineTreeData.GetAt(i);
		if (pData->m_pNode == pNode) {
			m_lineTreeData.RemoveAt(i);
			delete pData;
			break;
		}
	}
}

void CTreeView::OnRButtonDown(UINT nFlags, CPoint point) 
{
	// TODO: ̈ʒuɃbZ[W nhp̃R[hǉ邩܂̓ftHg̏ĂяoĂ
	
	CTreeDoc* pDoc = GetDocument();
	int i;

	m_popupPoint = point;

	m_pSelectedNode = PickTreeNode(pDoc->m_pTree, pDoc->m_nodeSize,
								   point);
	if (m_pSelectedNode != 0) {
		CMenu menu;
		menu.LoadMenu(IDR_TREE);
		CMenu* pPopup = menu.GetSubMenu(0);	

		if (m_pSelectedNode->m_parent != 0)
			pPopup->EnableMenuItem(ID_TREE_REMOVE,
								   MF_BYCOMMAND | MF_ENABLED);
		else
			pPopup->EnableMenuItem(ID_TREE_REMOVE,
								   MF_BYCOMMAND | MF_GRAYED);
		
		for (i = 0; i < m_stayTreeData.GetSize(); i++) {
			StayTreeData* pData = (StayTreeData*)
				m_stayTreeData.GetAt(i);
			if (pData->m_pNode == m_pSelectedNode)
				break;
		}
		if (i < m_stayTreeData.GetSize())
			pPopup->CheckMenuItem(ID_TREE_STAY,
								  MF_BYCOMMAND | MF_CHECKED);
		else
			pPopup->CheckMenuItem(ID_TREE_STAY,
								  MF_BYCOMMAND | MF_UNCHECKED);

		for (i = 0; i < m_lineTreeData.GetSize(); i++) {
			LineTreeData* pData = (LineTreeData*)
				m_lineTreeData.GetAt(i);
			if (pData->m_pNode == m_pSelectedNode)
				break;
		}
		if (i < m_lineTreeData.GetSize())
			pPopup->CheckMenuItem(ID_TREE_LINE,
								  MF_BYCOMMAND | MF_CHECKED);
		else
			pPopup->CheckMenuItem(ID_TREE_LINE,
								  MF_BYCOMMAND | MF_UNCHECKED);

		CPoint sPoint = point;
		ClientToScreen(&sPoint);
		pPopup->TrackPopupMenu(TPM_LEFTALIGN | TPM_RIGHTBUTTON, 
							   sPoint.x, sPoint.y, this);
		menu.DestroyMenu();
	}

	CView::OnRButtonDown(nFlags, point);
}

void CTreeView::OnTreeAdd() 
{
	// TODO: ̈ʒuɃR}h nhp̃R[hǉĂ
	
	CTreeDoc* pDoc = GetDocument();
	HRSolver* solver = &pDoc->m_solver;

	if (m_pSelectedNode == 0)
		return;

	TreeNode* child = new TreeNode(&pDoc->m_solver, pDoc->m_xUnit,
								   pDoc->m_yUnit, m_pSelectedNode, 1, 0);
	m_pSelectedNode->m_children.Add(child);
		
#ifdef WEAKER
	HRLinear* yInterval = new HRLinear(1);
#else // !WEAKER
	HRLinear* yInterval = new HRLinear(0);
#endif // WEAKER
	m_pSelectedNode->m_yIntervals.Add(yInterval);

	yInterval->sum(m_pSelectedNode->m_y, pDoc->m_yUnit, child->m_y);
	solver->add(*yInterval);

	int nChildren = m_pSelectedNode->m_children.GetSize();
	if (nChildren == 1) {
		m_pSelectedNode->m_xInterval.clear();

		m_pSelectedNode->m_leftAlign.clear();
		m_pSelectedNode->m_leftAlign.equal(m_pSelectedNode->m_left,
										   child->m_left);
		solver->add(m_pSelectedNode->m_leftAlign);
	}
	else {
		HRLinear* childrenGlue = new HRLinear(0);
		m_pSelectedNode->m_childrenGlues.Add(childrenGlue);
				
		TreeNode* lastChild = (TreeNode*)
			m_pSelectedNode->m_children.GetAt(nChildren - 2);
		childrenGlue->equal(lastChild->m_right, child->m_left);
		solver->add(*childrenGlue);
	}

#if 0
	m_pSelectedNode->m_midAlign.clear();

	TreeNode* firstChild = (TreeNode*) m_pSelectedNode->m_children.GetAt(0);
	if (firstChild == child)
		m_pSelectedNode->m_midAlign.equal(child->m_x, m_pSelectedNode->m_x);
	else
		m_pSelectedNode->m_midAlign.center(firstChild->m_x, child->m_x,
										   m_pSelectedNode->m_x);
	solver->add(m_pSelectedNode->m_midAlign);
#endif

	m_pSelectedNode->m_rightAlign.clear();

	m_pSelectedNode->m_rightAlign.equal(m_pSelectedNode->m_right,
										child->m_right);
	solver->add(m_pSelectedNode->m_rightAlign);

	planAndExec();

	RedrawWindow();
}

void CTreeView::OnTreeRemove() 
{
	// TODO: ̈ʒuɃR}h nhp̃R[hǉĂ
	
	CTreeDoc* pDoc = GetDocument();
	HRSolver* solver = &pDoc->m_solver;
	int i;

	if (m_pSelectedNode == 0 || 
		m_pSelectedNode->m_parent == 0)
		return;

	TreeNode* pParent = m_pSelectedNode->m_parent; 

	int nBros = pParent->m_children.GetSize();

	int idx;
	for (i = 0; i < nBros; i++) {
		if (pParent->m_children.GetAt(i) == m_pSelectedNode) {
			pParent->m_children.RemoveAt(i);
			idx = i;
			nBros--;
			break;
		}
	}

	RemoveStaysAndLines(m_pSelectedNode);
	delete m_pSelectedNode;

	if (nBros == 0) {
#if 0
		pParent->m_midAlign.clear();
		pParent->m_midAlign.center(pParent->m_left, pParent->m_right,
								   pParent->m_x);
		pDoc->m_solver.add(pParent->m_midAlign);
#endif

		pParent->m_xInterval.clear();
		pParent->m_xInterval.sum(pParent->m_left, pDoc->m_xUnit,
								 pParent->m_right);
		pDoc->m_solver.add(pParent->m_xInterval);
	}
	else {
		if (idx == 0) {
			TreeNode* firstBro = (TreeNode*) pParent->m_children.GetAt(0);

			pParent->m_leftAlign.clear();
			pParent->m_leftAlign.equal(pParent->m_left, firstBro->m_left);
			solver->add(pParent->m_leftAlign);

#if 0
			pParent->m_midAlign.clear();
			if (nBros == 1)
				pParent->m_midAlign.equal(firstBro->m_x, pParent->m_x);
			else {
				TreeNode* lastBro = (TreeNode*)
					pParent->m_children.GetAt(nBros - 1);
				pParent->m_midAlign.center(firstBro->m_x, lastBro->m_x,
										   pParent->m_x);
			}
			solver->add(pParent->m_midAlign);
#endif
		}
		else if (idx < nBros) {
			TreeNode* prevBro = (TreeNode*) pParent->m_children.GetAt(idx - 1);
			TreeNode* nextBro = (TreeNode*) pParent->m_children.GetAt(idx);

			HRLinear* childrenGlue = new HRLinear(0);
			pParent->m_childrenGlues.Add(childrenGlue);
				
			childrenGlue->equal(prevBro->m_right, nextBro->m_left);
			solver->add(*childrenGlue);
		}
		else {
			TreeNode* lastBro = (TreeNode*)
				pParent->m_children.GetAt(nBros - 1);

			pParent->m_rightAlign.clear();
			pParent->m_rightAlign.equal(pParent->m_right, lastBro->m_right);
			solver->add(pParent->m_rightAlign);

#if 0
			pParent->m_midAlign.clear();
			if (nBros == 1)
				pParent->m_midAlign.equal(lastBro->m_x, pParent->m_x);
			else {
				TreeNode* firstBro = (TreeNode*) pParent->m_children.GetAt(0);
				pParent->m_midAlign.center(firstBro->m_x, lastBro->m_x,
										   pParent->m_x);
			}
			solver->add(pParent->m_midAlign);
#endif
		}
	}

	planAndExec();

	RedrawWindow();
}

void CTreeView::OnTreeStay() 
{
	// TODO: ̈ʒuɃR}h nhp̃R[hǉĂ
	
	CTreeDoc* pDoc = GetDocument();
	int i;

	if (m_pSelectedNode == 0)
		return;

	StayTreeData* pData;
	for (i = 0; i < m_stayTreeData.GetSize(); i++) {
		pData = (StayTreeData*) m_stayTreeData.GetAt(i);
		if (pData->m_pNode == m_pSelectedNode)
			break;
	}
	if (i < m_stayTreeData.GetSize()) {
		m_stayTreeData.RemoveAt(i);
		delete pData;
	}
	else {
		pData = new StayTreeData;
	
		pData->m_pNode = m_pSelectedNode;
		pData->m_stayX.add(m_pSelectedNode->m_x);
		pDoc->m_solver.add(pData->m_stayX);
		pData->m_stayY.add(m_pSelectedNode->m_y);
		pDoc->m_solver.add(pData->m_stayY);

		m_stayTreeData.Add(pData);
	}

	planAndExec();

	RedrawWindow();
}

void CTreeView::OnTreeLine() 
{
	// TODO: ̈ʒuɃR}h nhp̃R[hǉĂ
	
	CTreeDoc* pDoc = GetDocument();
	int i;

	if (m_pSelectedNode == 0)
		return;

	LineTreeData* pData;
	for (i = 0; i < m_lineTreeData.GetSize(); i++) {
		pData = (LineTreeData*)
			m_lineTreeData.GetAt(i);
		if (pData->m_pNode == m_pSelectedNode)
			break;
	}
	if (i < m_lineTreeData.GetSize()) {
		m_lineTreeData.RemoveAt(i);
		delete pData;

		planAndExec();
	}
	else {
		m_inputMode = IM_SPECIFY_LINE;
	
		m_pLinedTreeNode = m_pSelectedNode;
		m_lineEnd0.x = (LONG) *m_pSelectedNode->m_x;
		m_lineEnd0.y = (LONG) *m_pSelectedNode->m_y;
		m_lineEnd1 = m_popupPoint;
	}

	RedrawWindow();
}
