//
// The HiRise Constraint Solver
//
// Copyright (C) 1998 Hiroshi HOSOBE
//
////////////////////////////////////////////////////////////////

#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include "HRSolver.h"

#ifdef __GNUC__
#include "HRList.cpp"

template class HRList<int>;

template class HRList<HRIndexAndValue>;

class HRVar;
template class HRList<HRVar*>;

class HRCon;
template class HRList<HRCon*>;

class HREdit;
template class HRList<HREdit*>;

struct HRLowerVec;
template class HRList<HRLowerVec*>;

struct HRUpperVec;
template class HRList<HRUpperVec*>;
#endif // __GNUC__

void HRSolver::getFirst(int& str, int& idx)
{
	idx = 0;

	while (str < HR_N_LEVELS) {
		while (idx < m_level[str].count()) {
			if (m_level[str].get(idx) != 0)
				break;

			idx++;
		}
		if (idx < m_level[str].count())
			break;

		str++;
		idx = 0;
	}
}

void HRSolver::getNext(int& str, int& idx)
{
	idx++;

	while (str < HR_N_LEVELS) {
		while (idx < m_level[str].count()) {
			if (m_level[str].get(idx) != 0)
				break;

			idx++;
		}
		if (idx < m_level[str].count())
			break;

		str++;
		idx = 0;
	}
}

HRSolver::HRSolver()
{
    m_nRealVars = 0;
    m_nRealCons = 0;

	m_strongestRemovedRowStr.str = HR_N_LEVELS + 1;
	m_strongestRemovedRowStr.idx = 0;
	m_firstPrefConIndex = 0;
}

HRSolver::~HRSolver()
{
	int i;

    if (m_lowerVecs.count() > 0) {
		for (i = 0; i < m_lowerVecs.count(); i++)
			delete m_lowerVecs.get(i);
		for (i = 0; i < m_upperVecs.count(); i++)
			delete m_upperVecs.get(i);

		free(m_row);
		free(m_row2);
        free(m_fsValue);
        free(m_bsValue);
        free(m_waStr);
    }
}

HRBool HRSolver::add(HRCon& con)
{
    int i;

    int str = con.getStrength();
    if (str < 0 || str >= HR_N_LEVELS) {
        char buf[BUFSIZ];
        sprintf(buf, "constraint with invalid strength: %s\n",
                con.getClassAndName());
        warning("add", buf);
        return HR_FALSE;
    }

    if (m_level[str].index(&con) != -1) {
        char buf[BUFSIZ];
        sprintf(buf, "constraint added twice: %s\n", con.getClassAndName());
        warning("add", buf);
        return HR_FALSE;
    }
    
	con.setSolver(this);
    m_level[str].add(&con);
    m_nRealCons++;

	con.setRow(-1);

    for (i = 0; i < con.countVars(); i++) {
		HRVar* var = con.getVar(i);
		var->add(con);
		if (m_vars.index(var) == -1) {
			m_vars.add(var);
			m_nRealVars++;
		}
	}
	
	switch (con.getType()) {
	case HR_EDIT:
		m_edits.add((HREdit*) &con);
		break;

	case HR_LINEAR:
		{
			HRLinear& lin = (HRLinear&) con;
			HRList<HRIndexAndValue>* coefs = lin.getCoefs();
			for (i = 0; i < lin.countVars(); i++) {
				HRIndexAndValue coef;
				coef.index = m_vars.index(lin.getVar(i));
				coef.value = coefs->get(i).value;
				coefs->set(i, coef);
			}
		}
		break;

	case HR_CON:
	case HR_STAY:
	case HR_WEAKEST_STAY:
		break;
	}

	for (i = 0; i < m_addedCons.count(); i++) {
		HRCon* c = m_addedCons.get(i);
		if (c->getStrength() > str) {
			m_addedCons.insert(i, &con);
			break;
		}
	}
	if (i == m_addedCons.count())
		m_addedCons.add(&con);

    return HR_TRUE;
}

HRBool HRSolver::remove(HRCon& con)
{
	int i;

    int str = con.getStrength();
	int idx = m_level[str].index(&con);

    if (idx == -1) {
        char buf[BUFSIZ];
        sprintf(buf, "nonexistent constraint: %s\n", con.getClassAndName());
        warning("remove", buf);
        return HR_FALSE;
    }

	int row = con.getRow();
	if (row != -1) {
		m_lowerVecs.get(row)->idx = -1;

		for (i = 0; i < m_removedRows.count(); i++) {
			if (m_removedRows.get(i) > row) {
				m_removedRows.insert(i, row);
				break;
			}
		}
		if (i == m_removedRows.count())
			m_removedRows.add(row);

		if (m_strongestRemovedRowStr.str > str ||
			(m_strongestRemovedRowStr.str == str &&
			 m_strongestRemovedRowStr.idx > idx)) {
			m_strongestRemovedRowStr.str = str;
			m_strongestRemovedRowStr.idx = idx;
		}
	}
	else 
		m_inactCons.remove(&con);

	m_addedCons.remove(&con);

	if (con.getType() == HR_EDIT)
		m_edits.remove((HREdit*) &con);

	con.setSolver(0);
    m_level[str].set(idx, 0);
    m_nRealCons--;

    for (i = 0; i < con.countVars(); i++) {
		HRVar* var = con.getVar(i);
		var->remove(con);

		if (var->countCons() == 0) {
			int idx = m_vars.index(var);
#ifdef DEBUG
			if (idx == -1)
				error("remove", "fatal error\n");
#endif // DEBUG
			m_vars.set(idx, 0);
			m_removedVars.add(idx);
			m_nRealVars--;
		}
	}

    return HR_TRUE;
}

HRBool HRSolver::refresh()
{
	int nVars0 = m_vars.count();
	int i, j, k;

	int nLowerVecs = m_lowerVecs.count();
	for (i = 0; i < nLowerVecs; i++) {
		HRLowerVec* lowerVec = m_lowerVecs.get(i);
		if (lowerVec->str < HR_N_LEVELS &&
			lowerVec->idx != -1)
			m_level[lowerVec->str].get(lowerVec->idx)->setRow(-1);
		delete m_lowerVecs.get(i);
	}
	m_lowerVecs.clear();

	int* varIndex;
	if ((varIndex = (int*) malloc(sizeof(int) * nVars0)) == 0)
		error("refresh");

	int idx = 0;
	for (i = 0; i < nVars0; i++) {
		HRVar* var = m_vars.get(i);
		if (var == 0)
			varIndex[i] = -1;
		else {
			m_vars.set(idx, var);
			varIndex[i] = idx;
			idx++;
		}
	}
	for (i = nVars0 - 1; i >= idx; i--)
		m_vars.removeAt(i);
	
	for (i = 0; i < HR_N_LEVELS; i++) {
		HRCons* lev = &m_level[i];
		int nCons0 = lev->count();

		idx = 0;
		for (j = 0; j < nCons0; j++) {
			HRCon* con = lev->get(j);
			if (con == 0)
				continue;

			lev->set(idx, con);
			idx++;

			if (con->getType() == HR_LINEAR) {
				HRLinear* lin = (HRLinear*) con;
				HRList<HRIndexAndValue>* coefs = lin->getCoefs();
				int nCoefs = coefs->count();
				for (k = 0; k < nCoefs; k++) {
					HRIndexAndValue coef = coefs->get(k);
					coef.index = varIndex[coef.index];
					coefs->set(k, coef);
				}
			}
		}

		for (j = nCons0 - 1; j >= idx; j--)
			lev->removeAt(j);
	}

	free(varIndex);

	int nVars = m_vars.count();

    if (nLowerVecs == 0) {
		if ((m_row = (HRValue*) malloc(sizeof(HRValue) * nVars)) == 0 ||
			(m_row2 = (HRValue*) malloc(sizeof(HRValue) * nVars)) == 0 ||
			(m_fsValue = (HRValue*) malloc(sizeof(HRValue) * nVars)) == 0 ||
			(m_bsValue = (HRValue*) malloc(sizeof(HRValue) * nVars)) == 0 ||
			(m_waStr = (HRStr*) malloc(sizeof(HRStr) * nVars)) == 0)
			error("refresh");
	}
	else {
		if ((m_row = (HRValue*)
			 realloc(m_row, sizeof(HRValue) * nVars)) == 0 ||
			(m_row2 = (HRValue*)
			 realloc(m_row2, sizeof(HRValue) * nVars)) == 0 ||
			(m_fsValue = (HRValue*)
			 realloc(m_fsValue, sizeof(HRValue) * nVars)) == 0 ||
			(m_bsValue = (HRValue*)
			 realloc(m_bsValue, sizeof(HRValue) * nVars)) == 0 ||
			(m_waStr = (HRStr*)
			 realloc(m_waStr, sizeof(HRStr) * nVars)) == 0)
			error("refresh");
    }

    refactorize();

	return HR_TRUE;
}

void HRSolver::refactorize()
{
    int nVars = m_vars.count();
	int i, j, k;

	for (i = 0; i < m_lowerVecs.count(); i++) {
		HRLowerVec* lowerVec = m_lowerVecs.get(i);
		if (lowerVec->str < HR_N_LEVELS &&
			lowerVec->idx != -1)
			m_level[lowerVec->str].get(lowerVec->idx)->setRow(-1);
		delete m_lowerVecs.get(i);
	}
	m_lowerVecs.clear();

	for (i = 0; i < m_upperVecs.count(); i++)
		delete m_upperVecs.get(i);
	m_upperVecs.clear();

	m_inactCons.clear();
	m_addedCons.clear();
	m_removedRows.clear();
	m_strongestRemovedRowStr.str = HR_N_LEVELS + 1;
	m_strongestRemovedRowStr.idx = 0;
	m_removedVars.clear();
	m_firstPrefConIndex = 0;

	//
	// add constraints
	//

	int str = 0;
	int idx;
	getFirst(str, idx);

	i = 0;
	while (i < nVars) {
		// set elements for the current constraint
		HRValue cst;
		setRow(m_row, cst, str, idx);
			
		// determine the i-th row of Ak; the first to (i-1)-th entries of
		// the i-th row of Ak is those of the i-th row of L
		for (j = 0; j < m_upperVecs.count(); j++) {
			HRUpperVec* upperVec = m_upperVecs.get(j);
			int src = upperVec->srcCol;
			int dst = upperVec->dstCol;
			if (upperVec->shift) {
				HRValue val = m_row[src];
				for (k = src; k < dst; k++)
					m_row[k] = m_row[k + 1];
				m_row[dst] = val;
			}
			else {
				if (dst != src) {
					HRValue val = m_row[src];
					m_row[src] = m_row[dst];
					m_row[dst] = val;
				}

				HRValue piv = m_row[src];
				m_row[src] = piv * upperVec->elems.get(0).value;
				for (k = 1; k < upperVec->elems.count(); k++) {
					HRIndexAndValue elem = upperVec->elems.get(k);
					m_row[elem.index] += piv * elem.value;
				}
			}
		}

		// find column to swap
		for (j = i; j < nVars; j++)
			if (!nearEq(m_row[j], (HRValue) 0))
				break;

		if (j == nVars) {
			if (str < HR_N_LEVELS) {
				HRCon* con = m_level[str].get(idx);
				m_inactCons.add(con);
			}
		}
		else {
			// the current constraint is hierarchically independent

			if (str == 0)
				m_firstPrefConIndex = i + 1;

			HRLowerVec* lowerVec;
			HRUpperVec* upperVec;
			if ((lowerVec = new HRLowerVec) == 0 ||
				(upperVec = new HRUpperVec) == 0)
				error("refactorize");
			if (i < m_lowerVecs.count()) {
				// dispose previous lower vector
				HRLowerVec* lVec = m_lowerVecs.get(i);
				delete lVec;
				m_lowerVecs.set(i, lowerVec);
			}
			else 
				m_lowerVecs.add(lowerVec);
			m_upperVecs.add(upperVec);

			// swap columns
			if (j != i) {
				HRValue val = m_row[j];
				m_row[j] = m_row[i];
				m_row[i] = val;
			}

			// record column swap
			upperVec->shift = HR_FALSE;
			upperVec->srcCol = i;
			upperVec->dstCol = j;

			// determine the i-th row of Uk
			HRValue rowiInv = 1 / m_row[i];
			HRIndexAndValue elem;
			elem.index = i;
			elem.value = rowiInv;
			upperVec->elems.add(elem);
			for (j = i + 1; j < nVars; j++) {
				if (!nearEq(m_row[j], (HRValue) 0)) {
					elem.index = j;
					elem.value = -m_row[j] * rowiInv;
					upperVec->elems.add(elem);
				}
			}

			// set lower elements
			for (j = 0; j < i; j++) {
				if (!nearEq(m_row[j], (HRValue) 0)) {
					elem.index = j;
					elem.value = m_row[j];
					lowerVec->elems.add(elem);
				}
			}

			// set constant
			lowerVec->cst = cst;

			// set other info
			lowerVec->str = str;
			lowerVec->idx = idx;
			if (str < HR_N_LEVELS)
				m_level[str].get(idx)->setRow(i);

			i++;
		}

		getNext(str, idx);
	}

	while (str < HR_N_LEVELS) {
		HRCon* con = m_level[str].get(idx);
		m_inactCons.add(con);
		getNext(str, idx);
	}
}

void HRSolver::refactorizePrefCons()
{
    int nVars = m_vars.count();
	int i, j, k;

	for (i = m_lowerVecs.count() - 1; i >= m_firstPrefConIndex; i--) {
		HRLowerVec* lowerVec = m_lowerVecs.removeAt(i);
		if (lowerVec->str < HR_N_LEVELS &&
			lowerVec->idx != -1)
			m_level[lowerVec->str].get(lowerVec->idx)->setRow(-1);
		delete lowerVec;
	}

	for (i = m_upperVecs.count() - 1; i >= m_firstPrefConIndex; i--)
		delete m_upperVecs.removeAt(i);

	m_inactCons.clear();
	m_addedCons.clear();
	m_removedRows.clear();
	m_strongestRemovedRowStr.str = HR_N_LEVELS + 1;
	m_strongestRemovedRowStr.idx = 0;
	m_removedVars.clear();

	//
	// add constraints
	//

	int str = 0;
	int idx;
	getFirst(str, idx);

	i = m_firstPrefConIndex;
	while (i < nVars) {
		if (str == 0) {
			HRCon* con = m_level[0].get(idx);
			if (con->getRow() != -1) {
				getNext(str, idx);
				continue;
			}
		}

		// set elements for the current constraint
		HRValue cst;
		setRow(m_row, cst, str, idx);
			
		// determine the i-th row of Ak; the first to (i-1)-th entries of
		// the i-th row of Ak is those of the i-th row of L
		for (j = 0; j < m_upperVecs.count(); j++) {
			HRUpperVec* upperVec = m_upperVecs.get(j);
			int src = upperVec->srcCol;
			int dst = upperVec->dstCol;
			if (upperVec->shift) {
				HRValue val = m_row[src];
				for (k = src; k < dst; k++)
					m_row[k] = m_row[k + 1];
				m_row[dst] = val;
			}
			else {
				if (dst != src) {
					HRValue val = m_row[src];
					m_row[src] = m_row[dst];
					m_row[dst] = val;
				}

				HRValue piv = m_row[src];
				m_row[src] = piv * upperVec->elems.get(0).value;
				for (k = 1; k < upperVec->elems.count(); k++) {
					HRIndexAndValue elem = upperVec->elems.get(k);
					m_row[elem.index] += piv * elem.value;
				}
			}
		}

		// find column to swap
		for (j = i; j < nVars; j++)
			if (!nearEq(m_row[j], (HRValue) 0))
				break;

		if (j == nVars) {
			if (str < HR_N_LEVELS) {
				HRCon* con = m_level[str].get(idx);
				m_inactCons.add(con);
			}
		}
		else {
			// the current constraint is hierarchically independent

			if (str == 0)
				m_firstPrefConIndex = i + 1;

			HRLowerVec* lowerVec;
			HRUpperVec* upperVec;
			if ((lowerVec = new HRLowerVec) == 0 ||
				(upperVec = new HRUpperVec) == 0)
				error("refactorize");
			if (i < m_lowerVecs.count()) {
				// dispose previous lower vector
				HRLowerVec* lVec = m_lowerVecs.get(i);
				delete lVec;
				m_lowerVecs.set(i, lowerVec);
			}
			else 
				m_lowerVecs.add(lowerVec);
			m_upperVecs.add(upperVec);

			// swap columns
			if (j != i) {
				HRValue val = m_row[j];
				m_row[j] = m_row[i];
				m_row[i] = val;
			}

			// record column swap
			upperVec->shift = HR_FALSE;
			upperVec->srcCol = i;
			upperVec->dstCol = j;

			// determine the i-th row of Uk
			HRValue rowiInv = 1 / m_row[i];
			HRIndexAndValue elem;
			elem.index = i;
			elem.value = rowiInv;
			upperVec->elems.add(elem);
			for (j = i + 1; j < nVars; j++) {
				if (!nearEq(m_row[j], (HRValue) 0)) {
					elem.index = j;
					elem.value = -m_row[j] * rowiInv;
					upperVec->elems.add(elem);
				}
			}

			// set lower elements
			for (j = 0; j < i; j++) {
				if (!nearEq(m_row[j], (HRValue) 0)) {
					elem.index = j;
					elem.value = m_row[j];
					lowerVec->elems.add(elem);
				}
			}

			// set constant
			lowerVec->cst = cst;

			// set other info
			lowerVec->str = str;
			lowerVec->idx = idx;
			if (str < HR_N_LEVELS)
				m_level[str].get(idx)->setRow(i);

			i++;
		}

		getNext(str, idx);
	}
}

HRBool HRSolver::plan()
{
    int nVars = m_vars.count();

    if (m_lowerVecs.count() == 0) {
		if ((m_row = (HRValue*) malloc(sizeof(HRValue) * nVars)) == 0 ||
			(m_row2 = (HRValue*) malloc(sizeof(HRValue) * nVars)) == 0 ||
			(m_fsValue = (HRValue*) malloc(sizeof(HRValue) * nVars)) == 0 ||
			(m_bsValue = (HRValue*) malloc(sizeof(HRValue) * nVars)) == 0 ||
			(m_waStr = (HRStr*) malloc(sizeof(HRStr) * nVars)) == 0)
			error("plan");

		refactorize();
	}
	else {
		if ((m_row = (HRValue*)
			 realloc(m_row, sizeof(HRValue) * nVars)) == 0 ||
			(m_row2 = (HRValue*)
			 realloc(m_row2, sizeof(HRValue) * nVars)) == 0 ||
			(m_fsValue = (HRValue*)
			 realloc(m_fsValue, sizeof(HRValue) * nVars)) == 0 ||
			(m_bsValue = (HRValue*)
			 realloc(m_bsValue, sizeof(HRValue) * nVars)) == 0 ||
			(m_waStr = (HRStr*)
			 realloc(m_waStr, sizeof(HRStr) * nVars)) == 0)
			error("plan");

		factorize();
    }

    return HR_TRUE;
}

void HRSolver::factorize()
{
    int nVars = m_vars.count();
    int i, j, k, l, i0;
	
#ifdef DEBUG
	clock_t t0 = clock();
#endif // DEBUG

	//
	// remove constraints
	//

	HRBool removed = (m_removedRows.count() > 0);
	if (removed) {
		int nActCons = m_lowerVecs.count() - m_removedRows.count();

		for (i = m_removedRows.count() - 1; i >= 0; i--) {
			int row = m_removedRows.get(i);
			if (row < m_firstPrefConIndex)
				m_firstPrefConIndex = row - 1;
			delete m_lowerVecs.removeAt(row);
		}

		int nUpperVecs = m_upperVecs.count();
		int nRemovedRows = 0;
		for (i = m_removedRows.get(0); i < nActCons; i++) {
			while (nRemovedRows < m_removedRows.count()) { 
				if (i + nRemovedRows < m_removedRows.get(nRemovedRows))
					break;

				HRUpperVec* upperVec;
				if ((upperVec = new HRUpperVec) == 0)
					error("factorize");
				m_upperVecs.add(upperVec);
				
				upperVec->shift = HR_TRUE;
				upperVec->srcCol = i;
				upperVec->dstCol = nVars - 1;

				nRemovedRows++;
			}

			HRLowerVec* lowerVec = m_lowerVecs.get(i);

			if (lowerVec->elems.count() > 0) {
				k = 0;
				for (j = 0; j < nRemovedRows; j++) {
					m_row[nVars - nRemovedRows + j] = (HRValue) 0;

					int removedRow = m_removedRows.get(j);
					for ( ; k < lowerVec->elems.count(); k++) {
						HRIndexAndValue elem = lowerVec->elems.get(k);
						if (elem.index > removedRow)
							break;
						else if (elem.index == removedRow) {
							m_row[nVars - nRemovedRows + j] = elem.value;
							lowerVec->elems.removeAt(k);
							break;
						}
						else {
							elem.index -= j;
							lowerVec->elems.set(k, elem);
						}
					}
				}
				for ( ; k < lowerVec->elems.count(); k++) {
					HRIndexAndValue elem = lowerVec->elems.get(k);
					elem.index -= nRemovedRows;
					lowerVec->elems.set(k, elem);
				}

				int n = 0;
				k = 0;
				for (j = nUpperVecs; j < m_upperVecs.count(); j++) {
					HRUpperVec* upperVec = m_upperVecs.get(j);

					if (upperVec->shift) {
						n++;
						continue;
					}
					
					int src = upperVec->srcCol;
					for ( ; k < lowerVec->elems.count(); k++) {
						HRIndexAndValue elem = lowerVec->elems.get(k);
						if (elem.index > src)
							break;
						else if (elem.index == src) {
							for (l = 1; l < upperVec->elems.count(); l++) {
								HRIndexAndValue e = upperVec->elems.get(l);
								m_row[e.index - nRemovedRows + n]
									+= elem.value * e.value;
							}
							k++;
							break;
						}
					}
					if (k == lowerVec->elems.count())
						break;
				}

				HRUpperVec* upperVec;
				if ((upperVec = new HRUpperVec) == 0)
					error("factorize");

				upperVec->shift = HR_FALSE;
				upperVec->srcCol = i;
				upperVec->dstCol = i;

				HRIndexAndValue elem;
				elem.index = i;
				elem.value = (HRValue) 1;
				upperVec->elems.add(elem);

				for (j = nVars - nRemovedRows; j < nVars; j++) {
					if (!nearEq(m_row[j], (HRValue) 0)) {
						elem.index = j;
						elem.value = -m_row[j];
						upperVec->elems.add(elem);
					}
				}

				if (upperVec->elems.count() == 1)
					delete upperVec;
				else
					m_upperVecs.add(upperVec);

			}				

			if (lowerVec->str < HR_N_LEVELS)
				m_level[lowerVec->str].get(lowerVec->idx)->setRow(i);
		}
	}

#ifdef DEBUG
	clock_t t1 = clock();
	printf("t1: %f ms\n", (t1 - t0) / (double) CLOCKS_PER_SEC * 1000);
#endif // DEBUG

	//
	// process removed variables
	//

	i = m_lowerVecs.count();
	for (i0 = 0; i0 < m_removedVars.count(); i0++) {
		int idx = m_removedVars.get(i0);

		// set elements for the current constraint
		HRValue cst;
		setRowForWeakestStay(m_row, cst, idx);
			
		// determine the i-th row of Ak; the first to (i-1)-th entries of
		// the i-th row of Ak is those of the i-th row of L
		for (j = 0; j < m_upperVecs.count(); j++) {
			HRUpperVec* upperVec = m_upperVecs.get(j);
			int src = upperVec->srcCol;
			int dst = upperVec->dstCol;
			if (upperVec->shift) {
				HRValue val = m_row[src];
				for (k = src; k < dst; k++)
					m_row[k] = m_row[k + 1];
				m_row[dst] = val;
			}
			else {
				if (dst != src) {
					HRValue val = m_row[src];
					m_row[src] = m_row[dst];
					m_row[dst] = val;
				}

				HRValue piv = m_row[src];
				m_row[src] = piv * upperVec->elems.get(0).value;
				for (k = 1; k < upperVec->elems.count(); k++) {
					HRIndexAndValue elem = upperVec->elems.get(k);
					m_row[elem.index] += piv * elem.value;
				}
			}
		}
			
		// find column to swap
		for (j = i; j < nVars; j++)
			if (!nearEq(m_row[j], (HRValue) 0))
				break;

#ifdef DEBUG
		if (j == nVars)
			error("factorize", "fatal error\n");
#endif // DEBUG
				
		HRLowerVec* lowerVec;
		HRUpperVec* upperVec;
		if ((lowerVec = new HRLowerVec) == 0 ||
			(upperVec = new HRUpperVec) == 0)
			error("factorize");
		if (i < m_lowerVecs.count()) {
			// dispose previous lower vector
			HRLowerVec* lVec = m_lowerVecs.get(i);
			delete lVec;
			m_lowerVecs.set(i, lowerVec);
		}
		else 
			m_lowerVecs.add(lowerVec);
		m_upperVecs.add(upperVec);

		// swap columns
		if (j != i) {
			HRValue val = m_row[j];
			m_row[j] = m_row[i];
			m_row[i] = val;
		}

		// record column swap
		upperVec->shift = HR_FALSE;
		upperVec->srcCol = i;
		upperVec->dstCol = j;

		// determine the i-th row of Uk
		HRValue rowiInv = 1 / m_row[i];
		HRIndexAndValue elem;
		elem.index = i;
		elem.value = rowiInv;
		upperVec->elems.add(elem);
		for (j = i + 1; j < nVars; j++) {
			if (!nearEq(m_row[j], 0)) {
				elem.index = j;
				elem.value = -m_row[j] * rowiInv;
				upperVec->elems.add(elem);
			}
		}

		// set lower elements
		for (j = 0; j < i; j++) {
			if (!nearEq(m_row[j], 0)) {
				elem.index = j;
				elem.value = m_row[j];
				lowerVec->elems.add(elem);
			}
		}

		// set constant
		lowerVec->cst = cst;

		// set other info
		lowerVec->str = HR_N_LEVELS;
		lowerVec->idx = idx;

		i++;
	}

	m_removedVars.clear();

#ifdef DEBUG
	clock_t t2 = clock();
	printf("t2: %f ms\n", (t2 - t1) / (double) CLOCKS_PER_SEC * 1000);
#endif // DEBUG

	//
	// add inactive constraints
	//

	if (removed) {
		HRCons inactCons;
		inactCons.append(m_inactCons);
		m_inactCons.clear();

		for (i0 = 0; i0 < inactCons.count(); i0++) {
			HRCon* con = inactCons.get(i0);
			int str = con->getStrength();
			if (str < m_strongestRemovedRowStr.str) {
				m_inactCons.add(con);
				continue;
			}
			if (str == m_strongestRemovedRowStr.str) {
				int idx = m_level[str].index(con);
				if (idx < m_strongestRemovedRowStr.idx) {
					m_inactCons.add(con);
					continue;
				}
			}
			break;
		}
		
		m_removedRows.clear();
		m_strongestRemovedRowStr.str = HR_N_LEVELS + 1;
		m_strongestRemovedRowStr.idx = 0;

		i = m_lowerVecs.count();
		for ( ; i0 < inactCons.count(); i0++) {
			if (i == nVars)
				break;

			HRCon* con = inactCons.get(i0);
			int str = con->getStrength();

			// set elements for the current constraint
			HRValue cst;
			setRow(m_row, cst, *con);
			
			// determine the i-th row of Ak; the first to (i-1)-th entries of
			// the i-th row of Ak is those of the i-th row of L
			for (j = 0; j < m_upperVecs.count(); j++) {
				HRUpperVec* upperVec = m_upperVecs.get(j);
				int src = upperVec->srcCol;
				int dst = upperVec->dstCol;
				if (upperVec->shift) {
					HRValue val = m_row[src];
					for (k = src; k < dst; k++)
						m_row[k] = m_row[k + 1];
					m_row[dst] = val;
				}
				else {
					if (dst != src) {
						HRValue val = m_row[src];
						m_row[src] = m_row[dst];
						m_row[dst] = val;
					}

					HRValue piv = m_row[src];
					m_row[src] = piv * upperVec->elems.get(0).value;
					for (k = 1; k < upperVec->elems.count(); k++) {
						HRIndexAndValue elem = upperVec->elems.get(k);
						m_row[elem.index] += piv * elem.value;
					}
				}
			}

			// find column to swap
			for (j = i; j < nVars; j++)
				if (!nearEq(m_row[j], (HRValue) 0))
					break;

			if (j < nVars) {
				// the current constraint is hierarchically independent

				HRLowerVec* lowerVec;
				HRUpperVec* upperVec;
				if ((lowerVec = new HRLowerVec) == 0 ||
					(upperVec = new HRUpperVec) == 0)
					error("factorize");
				if (i < m_lowerVecs.count()) {
					// dispose previous lower vector
					HRLowerVec* lVec = m_lowerVecs.get(i);
					delete lVec;
					m_lowerVecs.set(i, lowerVec);
				}
				else 
					m_lowerVecs.add(lowerVec);
				m_upperVecs.add(upperVec);

				// swap columns
				if (j != i) {
					HRValue val = m_row[j];
					m_row[j] = m_row[i];
					m_row[i] = val;
				}

				// record column swap
				upperVec->shift = HR_FALSE;
				upperVec->srcCol = i;
				upperVec->dstCol = j;

				// determine the i-th row of Uk
				HRValue rowiInv = 1 / m_row[i];
				HRIndexAndValue elem;
				elem.index = i;
				elem.value = rowiInv;
				upperVec->elems.add(elem);
				for (j = i + 1; j < nVars; j++) {
					if (!nearEq(m_row[j], (HRValue) 0)) {
						elem.index = j;
						elem.value = -m_row[j] * rowiInv;
						upperVec->elems.add(elem);
					}
				}

				// set lower elements
				for (j = 0; j < i; j++) {
					if (!nearEq(m_row[j], (HRValue) 0)) {
						elem.index = j;
						elem.value = m_row[j];
						lowerVec->elems.add(elem);
					}
				}

				// set constant
				lowerVec->cst = cst;

				// set other info
				lowerVec->str = str;
				lowerVec->idx = m_level[str].index(con);
				con->setRow(i);

				i++;
			}
		}

		for ( ; i0 < inactCons.count(); i0++)
			m_inactCons.add(inactCons.get(i0));
	}

#ifdef DEBUG
	clock_t t3 = clock();
	printf("t3: %f ms\n", (t3 - t2) / (double) CLOCKS_PER_SEC * 1000);
#endif // DEBUG

	//
	// add new constraints by revoking weaker ones
	//

	updateWalkaboutStrengths();

	for (i0 = 0; i0 < m_addedCons.count(); i0++) {
		HRCon* con = m_addedCons.get(i0);
		int str = con->getStrength();
		int idx = m_level[str].index(con);

		// set elements for the current constraint
		HRValue cst;
		setRow(m_row, cst, *con);

		HRStr waStr;
		waStr.str = -1;
		waStr.idx = 0;
		for (i = 0; i < nVars; i++) {
			if (nearEq(m_row[i], (HRValue) 0))
				continue;

			HRStr waStr1 = m_waStr[i];
			if (waStr1.str > waStr.str ||
				(waStr1.str == waStr.str && waStr1.idx > waStr.idx))
				waStr = waStr1;
		}

		if (waStr.str < str ||
			(waStr.str == str && waStr.idx < idx)) {
			for (j = 0; j < m_inactCons.count(); j++) {
				HRCon* c = m_inactCons.get(j);
				if (c->getStrength() > str) {
					m_inactCons.insert(j, con);
					break;
				}
			}
			if (j == m_inactCons.count())
				m_inactCons.add(con);
		}
		else {
			int nActCons = m_lowerVecs.count();
			int removedRow = -1;
			HRList<HRUpperVec*> newUpperVecs;
			HRList<HRLowerVec*> newLowerVecs;

			if (waStr.str <= HR_N_LEVELS) {
				nActCons--;

				if (waStr.str < HR_N_LEVELS) {
					HRCon* removedCon = m_level[waStr.str].get(waStr.idx);
					removedRow = removedCon->getRow();
				}
				else {
					for (i = 0; i < m_lowerVecs.count(); i++) {
						HRLowerVec* lowerVec = m_lowerVecs.get(i);
						if (lowerVec->str == HR_N_LEVELS &&
							lowerVec->idx == waStr.idx) {
							removedRow = i;
							break;
						}
					}
				}
#ifdef DEBUG
				if (removedRow == -1)
					error("factorize", "fatal error");
#endif // DEBUG

				HRUpperVec* upperVec;
				if ((upperVec = new HRUpperVec) == 0)
					error("factorize");
				newUpperVecs.add(upperVec);
				
				upperVec->shift = HR_TRUE;
				upperVec->srcCol = removedRow;
				upperVec->dstCol = nVars - 1;

				for (i = removedRow; i < nActCons; i++) {
					HRLowerVec* lowerVec0 = m_lowerVecs.get(i + 1);

					HRLowerVec* lowerVec;
					if ((lowerVec = new HRLowerVec) == 0)
						error("factorize");
					newLowerVecs.add(lowerVec);

					if (lowerVec0->elems.count() > 0) {
						m_row2[nVars - 1] = (HRValue) 0;

						for (k = 0; k < lowerVec0->elems.count(); k++) {
							HRIndexAndValue elem = lowerVec0->elems.get(k);
							if (elem.index > removedRow)
								break;
							else if (elem.index == removedRow) {
								m_row2[nVars - 1] = elem.value;
								k++;
								break;
							}
							else
								lowerVec->elems.add(elem);
						}
						for ( ; k < lowerVec0->elems.count(); k++) {
							HRIndexAndValue elem = lowerVec0->elems.get(k);
							elem.index -= 1;
							lowerVec->elems.add(elem);
						}

						int n = 0;
						k = 0;
						for (j = 0; j < newUpperVecs.count(); j++) {
							HRUpperVec* upperVec = newUpperVecs.get(j);
							
							if (upperVec->shift) {
								n++;
								continue;
							}
					
							int src = upperVec->srcCol;
							for ( ; k < lowerVec->elems.count(); k++) {
								HRIndexAndValue elem = lowerVec->elems.get(k);
								if (elem.index > src)
									break;
								else if (elem.index == src) {
									for (l = 1; l < upperVec->elems.count(); l++) {
										HRIndexAndValue e = upperVec->elems.get(l);
										m_row2[e.index - 1 + n]
											+= elem.value * e.value;
									}
									k++;
									break;
								}
							}
							if (k == lowerVec->elems.count())
								break;
						}

						HRUpperVec* upperVec;
						if ((upperVec = new HRUpperVec) == 0)
							error("factorize");

						upperVec->shift = HR_FALSE;
						upperVec->srcCol = i;
						upperVec->dstCol = i;

						HRIndexAndValue elem;
						elem.index = i;
						elem.value = (HRValue) 1;
						upperVec->elems.add(elem);

						if (!nearEq(m_row2[nVars - 1], (HRValue) 0)) {
							elem.index = nVars - 1;
							elem.value = -m_row2[nVars - 1];
							upperVec->elems.add(elem);
						}

						if (upperVec->elems.count() == 1)
							delete upperVec;
						else
							newUpperVecs.add(upperVec);
					}				

					lowerVec->cst = lowerVec0->cst;
					lowerVec->str = lowerVec0->str;
					lowerVec->idx = lowerVec0->idx;
				}
			}

			// determine the i-th row of Ak; the first to (i-1)-th entries of
			// the i-th row of Ak is those of the i-th row of L
			for (j = 0; j < m_upperVecs.count(); j++) {
				HRUpperVec* upperVec = m_upperVecs.get(j);
				int src = upperVec->srcCol;
				int dst = upperVec->dstCol;
				if (upperVec->shift) {
					HRValue val = m_row[src];
					for (k = src; k < dst; k++)
						m_row[k] = m_row[k + 1];
					m_row[dst] = val;
				}
				else {
					if (dst != src) {
						HRValue val = m_row[src];
						m_row[src] = m_row[dst];
						m_row[dst] = val;
					}

					HRValue piv = m_row[src];
					m_row[src] = piv * upperVec->elems.get(0).value;
					for (k = 1; k < upperVec->elems.count(); k++) {
						HRIndexAndValue elem = upperVec->elems.get(k);
						m_row[elem.index] += piv * elem.value;
					}
				}
			}
			for (j = 0; j < newUpperVecs.count(); j++) {
				HRUpperVec* upperVec = newUpperVecs.get(j);
				int src = upperVec->srcCol;
				int dst = upperVec->dstCol;
				if (upperVec->shift) {
					HRValue val = m_row[src];
					for (k = src; k < dst; k++)
						m_row[k] = m_row[k + 1];
					m_row[dst] = val;
				}
				else {
					if (dst != src) {
						HRValue val = m_row[src];
						m_row[src] = m_row[upperVec->dstCol];
						m_row[dst] = val;
					}

					HRValue piv = m_row[src];
					m_row[src] = piv * upperVec->elems.get(0).value;
					for (k = 1; k < upperVec->elems.count(); k++) {
						HRIndexAndValue elem = upperVec->elems.get(k);
						m_row[elem.index] += piv * elem.value;
					}
				}
			}

			// find column to swap
			for (j = nActCons; j < nVars; j++)
				if (!nearEq(m_row[j], (HRValue) 0))
					break;

			if (j == nVars) {
				refactorizePrefCons();
				return;
			}
			else {
				// the current constraint is hierarchically independent

				if (waStr.str < HR_N_LEVELS) {
					HRCon* revokedCon = m_level[waStr.str].get(waStr.idx);
					
					revokedCon->setRow(-1);

					for (i = 0; i < m_inactCons.count(); i++) {
						HRCon* c = m_inactCons.get(i);
						if (c->getStrength() > str) {
							m_inactCons.insert(i, revokedCon);
							break;
						}
					}
					if (i == m_inactCons.count())
						m_inactCons.add(revokedCon);
				}

				for (i = 0; i < newLowerVecs.count(); i++) {
					HRLowerVec* lowerVec = newLowerVecs.get(i);
					delete m_lowerVecs.get(removedRow + i);
					m_lowerVecs.set(removedRow + i, lowerVec);
					if (lowerVec->str < HR_N_LEVELS)
						m_level[lowerVec->str].get(lowerVec->idx)
							->setRow(removedRow + i);
				}
					
				for (i = 0; i < newUpperVecs.count(); i++)
					m_upperVecs.add(newUpperVecs.get(i));

				HRLowerVec* lowerVec;
				HRUpperVec* upperVec;
				if ((lowerVec = new HRLowerVec) == 0 ||
					(upperVec = new HRUpperVec) == 0)
					error("factorize");
				if (waStr.str <= HR_N_LEVELS) {
					delete m_lowerVecs.get(nActCons);
					m_lowerVecs.set(nActCons, lowerVec);
				}
				else
					m_lowerVecs.add(lowerVec);
				m_upperVecs.add(upperVec);

				// swap columns
				if (j != nActCons) {
					HRValue val = m_row[j];
					m_row[j] = m_row[nActCons];
					m_row[nActCons] = val;
				}

				// record column swap
				upperVec->shift = HR_FALSE;
				upperVec->srcCol = nActCons;
				upperVec->dstCol = j;

				// determine the lowest row of Uk
				HRValue rowiInv = 1 / m_row[nActCons];
				HRIndexAndValue elem;
				elem.index = nActCons;
				elem.value = rowiInv;
				upperVec->elems.add(elem);
				for (j = nActCons + 1; j < nVars; j++) {
					if (!nearEq(m_row[j], (HRValue) 0)) {
						elem.index = j;
						elem.value = -m_row[j] * rowiInv;
						upperVec->elems.add(elem);
					}
				}

				// set lower elements
				for (j = 0; j < nActCons; j++) {
					if (!nearEq(m_row[j], (HRValue) 0)) {
						elem.index = j;
						elem.value = m_row[j];
						lowerVec->elems.add(elem);
					}
				}

				// set constant
				lowerVec->cst = cst;

				// set other info
				lowerVec->str = str;
				lowerVec->idx = m_level[str].index(con);
				con->setRow(nActCons);

				updateWalkaboutStrengths();
			}
		}
	}

	m_addedCons.clear();

#ifdef DEBUG
	clock_t t4 = clock();
	printf("t4: %f ms\n", (t4 - t3) / (double) CLOCKS_PER_SEC * 1000);
#endif // DEBUG

	//
	// add the weakest stay constraints
	//

	i = m_lowerVecs.count();
	int idx = 0;
	while (i < nVars) {
		if (m_vars.get(idx) == 0) {
			idx++;
			continue;
		}

#if 0
		// preprocess for filtering out unnecessary constraints

		char* dirty = (char*) malloc(sizeof(char) * nVars);
		for (j = 0; j < i; j++)
			dirty[j] = (char) HR_FALSE;
		for (j = i; j < nVars; j++)
			dirty[j] = (char) HR_TRUE;

		for (j = m_upperVecs.count() - 1; j >= 0; j--) {
			HRUpperVec* upperVec = m_upperVecs.get(j);
			int src = upperVec->srcCol;
			int dst = upperVec->dstCol;
			if (upperVec->shift) {
				char dir = dirty[dst];
				for (k = dst; k > src; k--)
					dirty[k] = dirty[k - 1];
				dirty[src] = dir;
			}
			else {
				if (!(HRBool) dirty[src]) {
					for (k = 1; k < upperVec->elems.count(); k++) {
						HRIndexAndValue elem = upperVec->elems.get(k);
						if ((HRBool) dirty[elem.index]) {
							dirty[src] = (char) HR_TRUE;
							break;
						}
					}
				}
				
				if (dst != src) {
					char dir = dirty[src];
					dirty[src] = dirty[dst];
					dirty[dst] = dir;
				}
			}
		}

		for ( ; ; idx++) {
#ifdef DEBUG
			if (idx == nVars)
				error("factorize", "fatal error\n");
#endif // DEBUG

			if (!(HRBool) dirty[idx])
				continue;

			for (j = 0; j < m_lowerVecs.count(); j++) {
				HRLowerVec* lowerVec = m_lowerVecs.get(j);
				if (lowerVec->str == HR_N_LEVELS &&
					lowerVec->idx == idx)
					break;
			}
			if (j == m_lowerVecs.count())
				break;
		}

		free(dirty);
#else
		for (j = 0; j < m_lowerVecs.count(); j++) {
			HRLowerVec* lowerVec = m_lowerVecs.get(j);
			if (lowerVec->str == HR_N_LEVELS &&
				lowerVec->idx == idx)
				break;
		}
		if (j < m_lowerVecs.count()) {
			idx++;
			continue;
		}
#endif
		
		// set elements for the current constraint
		HRValue cst;
		setRowForWeakestStay(m_row, cst, idx);
			
		// determine the i-th row of Ak; the first to (i-1)-th entries of
		// the i-th row of Ak is those of the i-th row of L
		for (j = 0; j < m_upperVecs.count(); j++) {
			HRUpperVec* upperVec = m_upperVecs.get(j);
			int src = upperVec->srcCol;
			int dst = upperVec->dstCol;
			if (upperVec->shift) {
				HRValue val = m_row[src];
				for (k = src; k < dst; k++)
					m_row[k] = m_row[k + 1];
				m_row[dst] = val;
			}
			else {
				if (dst != src) {
					HRValue val = m_row[src];
					m_row[src] = m_row[dst];
					m_row[dst] = val;
				}

				HRValue piv = m_row[src];
				m_row[src] = piv * upperVec->elems.get(0).value;
				for (k = 1; k < upperVec->elems.count(); k++) {
					HRIndexAndValue elem = upperVec->elems.get(k);
					m_row[elem.index] += piv * elem.value;
				}
			}
		}
			
		// find column to swap
		for (j = i; j < nVars; j++)
			if (!nearEq(m_row[j], (HRValue) 0))
				break;

		if (j < nVars) {
			// the current constraint is hierarchically independent
				
			HRLowerVec* lowerVec;
			HRUpperVec* upperVec;
			if ((lowerVec = new HRLowerVec) == 0 ||
				(upperVec = new HRUpperVec) == 0)
				error("factorize");
			if (i < m_lowerVecs.count()) {
				// dispose previous lower vector
				HRLowerVec* lVec = m_lowerVecs.get(i);
				delete lVec;
				m_lowerVecs.set(i, lowerVec);
			}
			else 
				m_lowerVecs.add(lowerVec);
			m_upperVecs.add(upperVec);

			// swap columns
			if (j != i) {
				HRValue val = m_row[j];
				m_row[j] = m_row[i];
				m_row[i] = val;
			}

			// record column swap
			upperVec->shift = HR_FALSE;
			upperVec->srcCol = i;
			upperVec->dstCol = j;

			// determine the i-th row of Uk
			HRValue rowiInv = 1 / m_row[i];
			HRIndexAndValue elem;
			elem.index = i;
			elem.value = rowiInv;
			upperVec->elems.add(elem);
			for (j = i + 1; j < nVars; j++) {
				if (!nearEq(m_row[j], 0)) {
					elem.index = j;
					elem.value = -m_row[j] * rowiInv;
					upperVec->elems.add(elem);
				}
			}

			// set lower elements
			for (j = 0; j < i; j++) {
				if (!nearEq(m_row[j], 0)) {
					elem.index = j;
					elem.value = m_row[j];
					lowerVec->elems.add(elem);
				}
			}

			// set constant
			lowerVec->cst = cst;

			// set other info
			lowerVec->str = HR_N_LEVELS;
			lowerVec->idx = idx;

			i++;
		}
			
		idx++;
	}

#ifdef DEBUG
	clock_t t5 = clock();
	printf("t5: %f ms\n", (t5 - t4) / (double) CLOCKS_PER_SEC * 1000);
#endif // DEBUG
}

void HRSolver::updateWalkaboutStrengths()
{
	int nVars = m_vars.count();
    int i, k;

    for (i = 0; i < m_lowerVecs.count(); i++) {
		HRLowerVec* lowerVec = m_lowerVecs.get(i);

		HRStr waStr;
		waStr.str = lowerVec->str;
		waStr.idx = lowerVec->idx;
		if (lowerVec->str == HR_N_LEVELS &&
			m_vars.get(lowerVec->idx) == 0) {
			waStr.str = -1;
			waStr.idx = 0;
		}
		for (k = 0; k < lowerVec->elems.count(); k++) {
			HRIndexAndValue elem = lowerVec->elems.get(k);
			HRStr waStr1 = m_waStr[elem.index];
			if (waStr1.str > waStr.str ||
				(waStr1.str == waStr.str && waStr1.idx > waStr.idx))
				waStr = waStr1;
		}
        m_waStr[i] = waStr;
    }
	for ( ; i < nVars; i++) {
		m_waStr[i].str = HR_N_LEVELS + 1;
		m_waStr[i].idx = 0;
	}
	
	for (i = m_upperVecs.count() - 1; i >= 0; i--) {
		HRUpperVec* upperVec = m_upperVecs.get(i);
		int src = upperVec->srcCol;
		int dst = upperVec->dstCol;
		if (upperVec->shift) {
			HRStr waStr = m_waStr[dst];
			for (k = dst; k > src; k--)
				m_waStr[k] = m_waStr[k - 1];
			m_waStr[src] = waStr;
		}
		else {
			HRStr waStr = m_waStr[upperVec->elems.get(0).index];
			for (k = 1; k < upperVec->elems.count(); k++) {
				HRIndexAndValue elem = upperVec->elems.get(k);
				HRStr waStr1 = m_waStr[elem.index];
				if (waStr1.str > waStr.str ||
					(waStr1.str == waStr.str && waStr1.idx > waStr.idx))
					waStr = waStr1;
			}
			m_waStr[upperVec->srcCol] = waStr;
		
			if (dst != src) {
				waStr = m_waStr[src];
				m_waStr[src] = m_waStr[dst];
				m_waStr[dst] = waStr;
			}
		}
	}

#if 0
	{ 
		char buf[BUFSIZ], buf1[BUFSIZ];
		sprintf(buf, "");
		for (i = 0; i < nVars; i++) {
			sprintf(buf1, "(%d,%d),", m_waStr[i].str, m_waStr[i].idx);
			strcat(buf, buf1);
		}
		strcat(buf, "\n");
		OutputDebugString(buf);
	}
#endif
}

HRBool HRSolver::execute()
{
    changeConstants();

    forwardSubstitution();
    backwardSubstitution();

	// TODO: inactReqCons$B$N(Bsatisfiability test

    return HR_TRUE;
}

void HRSolver::changeConstants()
{
    int i;

    for (i = 0; i < m_edits.count(); i++) {
        HREdit* edit = m_edits.get(i);
		
		int row = edit->getRow();
		if (row == -1)
			continue;

        m_lowerVecs.get(row)->cst = edit->get();
    }
}

void HRSolver::forwardSubstitution()
{
	int nVars = m_vars.count();
    int i, k;

    for (i = 0; i < nVars; i++) {
		HRLowerVec* lowerVec = m_lowerVecs.get(i);

        HRValue val = lowerVec->cst;
		for (k = 0; k < lowerVec->elems.count(); k++) {
			HRIndexAndValue elem = lowerVec->elems.get(k);
			val -= elem.value * m_fsValue[elem.index];
		}
        m_fsValue[i] = val;
    }
}

void HRSolver::backwardSubstitution()
{
    int nVars = m_vars.count();
    int i, k;

	for (i = 0; i < nVars; i++)
		m_bsValue[i] = m_fsValue[i];

	for (i = m_upperVecs.count() - 1; i >= 0; i--) {
		HRUpperVec* upperVec = m_upperVecs.get(i);
		int src = upperVec->srcCol;
		int dst = upperVec->dstCol;
		if (upperVec->shift) {
			HRValue val = m_bsValue[dst];
			for (k = dst; k > src; k--)
				m_bsValue[k] = m_bsValue[k - 1];
			m_bsValue[src] = val;
		}
		else {
			HRValue val = 0;
			for (k = 0; k < upperVec->elems.count(); k++) {
				HRIndexAndValue elem = upperVec->elems.get(k);
				val += elem.value * m_bsValue[elem.index];
			}
			m_bsValue[upperVec->srcCol] = val;
		
			if (dst != src) {
				val = m_bsValue[src];
				m_bsValue[src] = m_bsValue[dst];
				m_bsValue[dst] = val;
			}
		}
	}

	for (i = 0; i < nVars; i++) {
		HRVar* var = m_vars.get(i);
		if (var != 0)
			var->set(m_bsValue[i]);
	}
}

void HRSolver::setRow(HRValue* row, HRValue& cst, HRCon& con)
{
	switch (con.getType()) {
	case HR_CON:
		warning("setRow", "constraint of type HR_CON given\n");
		break;
		
	case HR_STAY:
		setRowForStay(row, cst, (HRStay&) con);
		break;

	case HR_EDIT:
		setRowForEdit(row, cst, (HREdit&) con);
		break;

	case HR_LINEAR:
		setRowForLinear(row, cst, (HRLinear&) con);
		break;

	case HR_WEAKEST_STAY:
		break;
	}
}

void HRSolver::setRow(HRValue* row, HRValue& cst, int str, int idx)
{
	if (str < HR_N_LEVELS) {
		HRCon* con = m_level[str].get(idx);
		switch (con->getType()) {
		case HR_CON:
			warning("setRow", "constraint of type HR_CON given\n");
			break;
		
		case HR_STAY:
			setRowForStay(row, cst, (HRStay&) *con);
			break;

		case HR_EDIT:
			setRowForEdit(row, cst, (HREdit&) *con);
			break;

		case HR_LINEAR:
			setRowForLinear(row, cst, (HRLinear&) *con);
			break;

		case HR_WEAKEST_STAY:
			break;
		}
	}
	else
        setRowForWeakestStay(row, cst, idx);
}

void HRSolver::setRowForStay(HRValue* row, HRValue& cst, HRStay& stay)
{
    int nVars = m_vars.count();
    int j;

    if (stay.countVars() == 0) {
        char buf[BUFSIZ];
        sprintf(buf, "stay constraint with no variable: %s\n",
                stay.getClassAndName());
        error("setRowForStay", buf);
    }

    HRVar* var = stay.getVar(0);
    int idx = m_vars.index(var);
    if (idx == -1) {
        char buf[BUFSIZ];
        sprintf(buf, "variable not found: %s\n", stay.getClassAndName());
        error("setRowForStay", buf);
    }
                
    for (j = 0; j < nVars; j++)
		row[j] = (HRValue) 0;
    row[idx] = (HRValue) 1;
    
    cst = var->get();
}

void HRSolver::setRowForEdit(HRValue* row, HRValue& cst, HREdit& edit)
{
    int nVars = m_vars.count();
    int j;

    if (edit.countVars() == 0) {
        char buf[BUFSIZ];
        sprintf(buf, "edit constraint with no variable: %s\n",
                edit.getClassAndName());
        error("setRowForEdit", buf);
    }

    HRVar* var = edit.getVar(0);
    int idx = m_vars.index(var);
    if (idx == -1) {
        char buf[BUFSIZ];
        sprintf(buf, "variable not found: %s\n", edit.getClassAndName());
        error("setRowForEdit", buf);
    }
                
    for (j = 0; j < nVars; j++)
		row[j] = (HRValue) 0;
    row[idx] = (HRValue) 1;
    
    cst = edit.get();
}

void HRSolver::setRowForLinear(HRValue* row, HRValue& cst, HRLinear& lin)
{
    int nVars = m_vars.count();
    int j;

    for (j = 0; j < nVars; j++)
		row[j] = (HRValue) 0;

	HRList<HRIndexAndValue>* coefs = lin.getCoefs();
	for (j = 0; j < lin.countVars(); j++) {
        HRVar* var = lin.getVar(j);
		HRIndexAndValue coef = coefs->get(j);
		row[coef.index] = coef.value;
	}

    cst = lin.getConst();
}

void HRSolver::setRowForWeakestStay(HRValue* row, HRValue& cst, int vIdx)
{
    int nVars = m_vars.count();
    int j;

    for (j = 0; j < nVars; j++)
		row[j] = (HRValue) 0;
	row[vIdx] = (HRValue) 1;

	HRVar* var = m_vars.get(vIdx);
	if (var == 0)
		cst = (HRValue) 0;
	else
		cst = var->get();
}
