//
// The HiRise Constraint Solver
//
// Copyright (C) 1998 Hiroshi HOSOBE
//
////////////////////////////////////////////////////////////////

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#ifdef DEBUG
#include <time.h>
#endif // DEBUG
#include "HRList.h"
#include "HRVar.h"
#include "HRCon.h"
#include "HRStay.h"
#include "HREdit.h"
#include "HRLinear.h"
#include "HRParser.h"

class HRParserHashEntry {
public:
	HRString name;
	HRVar* var;
	HRParserHashEntry* next;

	HRParserHashEntry(HRString n, HRVar* v);
	~HRParserHashEntry();
};

HRParserHashEntry::HRParserHashEntry(HRString n, HRVar* v)
{
	name = strdup(n);
	var = v;
}

HRParserHashEntry::~HRParserHashEntry()
{
	free(name);
	delete var;
}

class HRParserLinearTerm {
public:
	HRVar* var;
	HRValue coef;
	HRParserLinearTerm* next;
	
	HRParserLinearTerm(HRVar* v, HRValue c);

	HRBool isConst() { return var == 0 && next == 0; }
};

HRParserLinearTerm::HRParserLinearTerm(HRVar* v, HRValue c)
{
	var = v;
	coef = c;
	next = 0;
}

HRParser::HRParser()
{
	int i;

	for (i = 0; i < HR_PASER_HASH_SIZE; i++)
		m_hash[i] = 0;
}

HRParser::~HRParser()
{
	int i;

	for (i = 0; i < HR_PASER_HASH_SIZE; i++) {
		HRParserHashEntry* entry;
		HRParserHashEntry* next;
		for (entry = m_hash[i]; entry != 0; entry = next) {
			next = entry->next;
			delete entry;
		}
	}
}

int HRParser::calcHashVal(const HRString name)
{
	int i;

	int val = 0;
	for (i = 0; name[i] != '\0'; i++)
		val = name[i] + 31 * val;

	return val % HR_PASER_HASH_SIZE;
}

HRBool HRParser::insertHashEntry(const HRString name, HRVar* var)
{
	int hVal = calcHashVal(name);

	HRParserHashEntry* entry;
	for (entry = m_hash[hVal]; entry != 0; entry = entry->next)
		if (strcmp(entry->name, name) == 0)
			break;
	if (entry != 0)
		return HR_FALSE;

	entry = new HRParserHashEntry(name, var);

	entry->next = m_hash[hVal];
	m_hash[hVal] = entry;

	return HR_TRUE;
}

HRBool HRParser::deleteHashEntry(const HRString name)
{
	int hVal = calcHashVal(name);

	HRParserHashEntry* prev = 0;
	HRParserHashEntry* entry;
	for (entry = m_hash[hVal]; entry != 0; entry = entry->next) {
		if (strcmp(entry->name, name) == 0)
			break;
		prev = entry;
	}
	if (entry == 0)
		return HR_FALSE;

	if (prev == 0)
		m_hash[hVal] = entry->next;
	else
		prev->next = entry->next;

	delete entry;

	return HR_TRUE;
}

HRParserHashEntry* HRParser::getHashEntry(const HRString name)
{
	int hVal = calcHashVal(name);

	HRParserHashEntry* entry;
	for (entry = m_hash[hVal]; entry != 0; entry = entry->next)
		if (strcmp(entry->name, name) == 0)
			break;

	return entry;
}

HRVar* HRParser::createVar(const HRString name, HRValue val)
{
	if (!checkNameValidity(name)) {
		char buf[BUFSIZ];
		sprintf(buf, "invalid name: %s\n", name);
		error("createVar", buf);
	}

#ifdef DEBUG
	HRVar* var = new HRVar(name, val);
#else // !DEBUG
	HRVar* var = new HRVar(val);
#endif // DEBUG

	if (!insertHashEntry(name, var)) {
		char buf[BUFSIZ];
		sprintf(buf, "name already registered: %s\n", name);
		error("createVar", buf);
	}

	return var;
}

HRBool HRParser::checkNameValidity(const HRString name)
{
	int i;

	if (strcmp(name, "stay") == 0 ||
		strcmp(name, "edit") == 0)
		return HR_FALSE;

	if (!isAlphabet(name[0]))
		return HR_FALSE;

	for (i = 1; name[i] != '\0'; i++)
		if (!isAlphabet(name[i]) && !isDigit(name[i]))
			return HR_FALSE;

	return HR_TRUE;
}

void HRParser::deleteVar(const HRString name)
{
	if (!deleteHashEntry(name)) {
		char buf[BUFSIZ];
		sprintf(buf, "nonexistent variable: %s\n", name);
		error("deleteVar", buf);
	}
}

HRVar* HRParser::getVar(const HRString name)
{
	HRParserHashEntry* entry = getHashEntry(name);
	if (entry == 0)
		return 0;

	return entry->var;
}

HRCon* HRParser::createCon(int str, const HRString expr, ...)
{
	HRList<HRVar*> vars;
	va_list varList;
	int i;

	va_start(varList, expr);

	int nLocalVars = countLocalVars(expr);
	for (i = 0; i < nLocalVars; i++) {
		HRVar* var = va_arg(varList, HRVar*);
		vars.add(var);
	}

	va_end(varList);
	
	HRConType type = getConType(expr);
	if (type == -1)
		return 0;

	HRCon* con;
	switch (type) {
	case HR_STAY:
	case HR_EDIT:
		con = createStayOrEditCon(str, expr, vars);
		break;
	case HR_LINEAR:
		con = createLinearCon(str, expr, vars);
		break;
	}

	return con;
}

int HRParser::countLocalVars(const HRString expr)
{
	int i;

	int n = 0;
	for (i = 0; expr[i] != '\0'; i++) {
		if (expr[i] != '$')
			continue;
		
		i++;

		char buf[BUFSIZ];
		int j = 0;
		while (isDigit(expr[i]))
			buf[j++] = expr[i++];
		buf[j] = '\0';
		if (j == 0)
			return 0;

		int index = atoi(buf);
		if (index >= n)
			n = index + 1;
	}

	return n;
}

HRConType HRParser::getConType(const HRString expr)
{
	int i, j;

	i = 0;
	skipSpaces(expr, i);

	char firstToken[BUFSIZ];
	j = 0;
	while (isAlphabet(expr[i]))
		firstToken[j++] = expr[i++];
	firstToken[j] = '\0';

	if (strcmp(firstToken, "stay") == 0)
		return HR_STAY;
	if (strcmp(firstToken, "edit") == 0)
		return HR_EDIT;

	return HR_LINEAR;
}

HRCon* HRParser::createStayOrEditCon(int str, const HRString expr,
									 HRList<HRVar*>& vars)
{
	int i, j;

	i = 0;
	skipSpaces(expr, i);

	char firstToken[BUFSIZ];
	j = 0;
	while (isAlphabet(expr[i]))
		firstToken[j++] = expr[i++];
	firstToken[j] = '\0';

	HRBool isStay = HR_TRUE;
	if (strcmp(firstToken, "stay") != 0) {
		if (strcmp(firstToken, "edit") == 0)
			isStay = HR_FALSE;
		else {
			error("createStayOrEditCon", "fatal error");
			return 0;
		}
	}
	
	skipSpaces(expr, i);

	if (expr[i] != '(') {
		char buf[BUFSIZ];
		sprintf(buf, "left parenthesis required: %s\n", expr);
		error("createStayOrEditCon", buf);
		return 0;
	}
	i++;

	skipSpaces(expr, i);

	HRParserLinearTerm* term = parseVariable(expr, i, vars);
	if (term == 0) {
		char buf[BUFSIZ];
		sprintf(buf, "variable required: %s", expr);
		error("createStayOrEditCon", buf);
		return 0;
	}

	skipSpaces(expr, i);

	if (expr[i] != ')') {
		char buf[BUFSIZ];
		sprintf(buf, "right parenthesis required: %s\n", expr);
		error("createStayOrEditCon", buf);
		return 0;
	}
	i++;

	skipSpaces(expr, i);

	if (expr[i] != '\0') {
		char buf[BUFSIZ];
		sprintf(buf, "invalid description: %s", expr + i);
		error("createStayOrEditCon", buf);
		return 0;
	}		

	HRCon* con;
	if (isStay)
		con = new HRStay(str);
	else
		con = new HREdit(str);
	con->add(*term->var);

	delete term;

	return con;
}

HRCon* HRParser::createLinearCon(int str, const HRString expr,
								 HRList<HRVar*>& vars)
{
	int i;

	i = 0;
	HRParserLinearTerm* term = parseExpression(expr, i, vars);
	if (term == 0)
		return 0;
	
	skipSpaces(expr, i);

	if (expr[i] != '=') {
		error("createLinearCon", "equal missing\n");
		freeTerms(term);
		return 0;
	}
	i++;

	HRParserLinearTerm* rhs = parseExpression(expr, i, vars);
	if (rhs == 0)
		return 0;
	
	skipSpaces(expr, i);

	if (expr[i] != '\0') {
		char buf[BUFSIZ];
		sprintf(buf, "invalid description: %s", expr + i);
		error("createLinearCon", buf);
		return 0;
	}

	addTerms(term, rhs, -1);

	HRLinear* con = new HRLinear(str);

	HRParserLinearTerm* t;
	for (t = term; t != 0; t = t->next) {
		if (t->var != 0) {
			if (!nearEq(t->coef, 0))
				con->add(*t->var, t->coef);
		}
		else
			con->setConst(-t->coef);
	}
	
	freeTerms(term);
		
	return con;
}

void HRParser::freeTerms(HRParserLinearTerm* term)
{
	HRParserLinearTerm* next;

	for ( ; term != 0; term = next) {
		next = term->next;
		delete term;
	}
}
	
void HRParser::addTerms(HRParserLinearTerm* term0, HRParserLinearTerm* term1,
						HRValue scale)
{
	HRParserLinearTerm* next;

	for ( ; term1 != 0; term1 = next) {
		next = term1->next;

		HRParserLinearTerm* t0prev = 0;
		HRParserLinearTerm* t0;
		for (t0 = term0; t0 != 0; t0prev = t0, t0 = t0->next) {
			if (t0->var == term1->var) {
				t0->coef += scale * term1->coef;
				delete term1;
				break;
			}
		}
		if (t0 == 0) {
			term1->coef *= scale;
			term1->next = 0;
			t0prev->next = term1;
		}
	}
}

void HRParser::multTerms(HRParserLinearTerm* term, HRValue scale)
{
	for ( ; term != 0; term = term->next)
		term->coef *= scale; 
}

void HRParser::skipSpaces(const HRString expr, int& i)
{
	while (isSpace(expr[i]))
		i++;
}

HRParserLinearTerm* HRParser::parseVariable(const HRString expr, int& i,
											HRList<HRVar*>& vars)
{
    HRParserLinearTerm* term = 0;

    if (expr[i] == '$') {
		i++;

		char digits[BUFSIZ];
		int j = 0;
		while (isDigit(expr[i]))
			digits[j++] = expr[i++];
		digits[j] = '\0';
		if (j == 0)
			return 0;

		int varIndex = atoi(digits);
		if (varIndex < 0 || varIndex >= vars.count()) {
			error("parseValue", "variable index out of range\n");
			return 0;
		}			
		term = new HRParserLinearTerm(vars.get(varIndex), 1);
    }
	else if (isAlphabet(expr[i])) {
		char name[BUFSIZ];
		int j = 0;
		while (isAlphabet(expr[i]) || isDigit(expr[i]))
			name[j++] = expr[i++];
		name[j] = '\0';
		
		HRParserHashEntry* entry = getHashEntry(name);
		if (entry == 0) {
			char buf[BUFSIZ];
			sprintf(buf, "variable not found: %d\n", name);
			error("createStayOrEditCon", buf);
			return 0;
		}

		term = new HRParserLinearTerm(entry->var, 1);
	}

	return term;
}

HRParserLinearTerm* HRParser::parseValue(const HRString expr, int& i,
										 HRList<HRVar*>& vars)
{
    HRParserLinearTerm* term = 0;

	skipSpaces(expr, i);

    if (expr[i] == '+') {
		i++;

		term = parseValue(expr, i, vars);
    }
    else if (expr[i] == '-') {
		i++;

		if ((term = parseValue(expr, i, vars)) == 0)
			return 0;
		
		term->coef = -term->coef;
    }
    else if (isDigit(expr[i]) || expr[i] == '.') {
		char buf[BUFSIZ];
		int j = 0;
		while (isDigit(expr[i]))
			buf[j++] = expr[i++];
		if (expr[i] == '.')
			buf[j++] = expr[i++];
		while (isDigit(expr[i]))
			buf[j++] = expr[i++];
		buf[j] = '\0';
		if (j == 0)
			return 0;

		HRValue value = (HRValue) atof(buf);
		term = new HRParserLinearTerm(0, value);
	}
    else if (expr[i] == '$' || isAlphabet(expr[i]))
		term = parseVariable(expr, i, vars);
    else if (expr[i] == '(') {
		i++;

		if ((term = parseExpression(expr, i, vars)) == 0)
			return 0;

		if (expr[i] == ')')
			i++;
		else {
			error("parseValue", "right parenthesis missing\n");
			freeTerms(term);
			return 0;
		}
    }
    
    return term;
}

HRParserLinearTerm* HRParser::parseTerm(const HRString expr, int& i,
										HRList<HRVar*>& vars)
{
    HRParserLinearTerm* term = parseValue(expr, i, vars);
    if (term == 0)
		return 0;

    for ( ; ; ) {
		skipSpaces(expr, i);

		if (expr[i] == '*') {
			i++;

			HRParserLinearTerm* tmp = parseValue(expr, i, vars);
			if (tmp == 0) {
				freeTerms(term);
				return 0;
			}
			if (!term->isConst() && !tmp->isConst()) {
				error("parserTerm", "expression not linear\n");
				freeTerms(term);
				freeTerms(tmp);
				return 0;
			}

			if (term->isConst()) {
				multTerms(tmp, term->coef);
				delete term;
				term = tmp;
			}
			else {
				multTerms(term, tmp->coef);
				delete tmp;
			}
		}
		else if (expr[i] == '/') {
			i++;
			
			HRParserLinearTerm* tmp = parseValue(expr, i, vars);
			if (tmp == 0) {
				freeTerms(term);
				return 0;
			}
			if (!tmp->isConst()) {
				error("parserTerm", "expression not linear\n");
				freeTerms(term);
				freeTerms(tmp);
				return 0;
			}
				
			multTerms(term, 1 / tmp->coef);
			delete tmp;
		}
		else
			break;
    }

    return term;
}

HRParserLinearTerm* HRParser::parseExpression(const HRString expr, int& i,
											  HRList<HRVar*>& vars)
{
    HRParserLinearTerm* term = parseTerm(expr, i, vars);
	if (term == 0)
		return 0;

    for ( ; ; ) {
		skipSpaces(expr, i);

		if (expr[i] == '+') {
			i++;
			
			HRParserLinearTerm* tmp = parseTerm(expr, i, vars);
			if (tmp == 0) {
				freeTerms(term);
				return 0;
			}
			
			addTerms(term, tmp, 1);
		}
		else if (expr[i] == '-') {
			i++;
			
			HRParserLinearTerm* tmp = parseTerm(expr, i, vars);
			if (tmp == 0) {
				freeTerms(term);
				return 0;
			}
			
			addTerms(term, tmp, -1);
		}
		else
			break;
    }

    return term;
}
