/***************************************************************************
* Functions
*
* FunTree.h (Management of binary trees representing functions)
*
* Date:    1 feb 98
* Author:  L. Granvilliers - LIFO Orleans
****************************************************************************/
/*************************************************************************/
/*                                                                       */
/*       Copyright (C) 1998 Universite de Nantes                         */
/*                                                                       */
/*************************************************************************/

#ifndef FUNTREE_H
#define FUNTREE_H

#include "FiaInterval.h"


/* OPERATOR = INTERVAL FUNCTION **********************************************
*****************************************************************************/
#      define FunOpAddII    FiaAddII
#      define FunOpAddRI    FiaAddRI
#      define FunOpSubII    FiaSubII
#      define FunOpSubRI    FiaSubRI
#      define FunOpSubIR    FiaSubIR
#      define FunOpNegI     FiaNegI
#      define FunOpMulII    FiaMulII
#      define FunOpMulRI    FiaMulRI
#      define FunOpMulRnegI FiaMulRnegI
#      define FunOpMulRposI FiaMulRposI
#      define FunOpDivII    FiaDivII
#      define FunOpDivIR    FiaDivIR
#      define FunOpDivRI    FiaDivRI
#      define FunOpDivIRneg FiaDivIRneg
#      define FunOpDivIRpos FiaDivIRpos
#      define FunOpDivRnegI FiaDivRnegI
#      define FunOpDivRposI FiaDivRposI
#      define FunOpSqrI     FiaSqrI
#      define FunOpSqrtI    FiaSqrtI
#      define FunOpPowI     FiaPowI


/* TYPES OF TREE NODES ******************************************************
*****************************************************************************/
#      define FunNodeVar 0      /* variable */
#      define FunNodeOp  1      /* operator */
#      define FunNodeItv 2      /* interval = constant */
#      define FunNodeUseless 3  /* empty (right) node for operators of arity 1 */



/* TYPE OF THE BACKWARDS EVALUATION FUNCTIONS *******************************
*****************************************************************************/
typedef void (* FunEvalBwdPath)(void *);


/* EVALUATION FUNCTIONS AT ONE OPERATOR NODE ********************************
*****************************************************************************/
struct FunTvalFunc
{
  FiaEvalOpI evalII;             /* forward evaluation */
  FunEvalBwdPath evalDerivLeft;  /* backward evaluation for the left sub-tree */
  FunEvalBwdPath evalDerivRight; /* backward evaluation for the right sub-tree */
  FunEvalBwdPath evalDerivAll;   /* backward evaluation for the sub-trees */
};


/* VALUE AT ONE TREE'S NODE *************************************************
*****************************************************************************/
union FunTval
{
    long *var;                /* Node variable : var[0] for the index in
                                 the global array of variables;
                                 var[1] for the index in the local one */
    struct FunTvalFunc *eval; /* Node operator = evaluation functions */
    FiaItv i;                 /* Node constant or exponent */
};


/* TREE'S NODE **************************************************************
*****************************************************************************/
struct FunTnode
{
    int type;              /* type of information: var, operator, constant */
    union FunTval val;     /* value at this node */
    FiaItv forward;        /* forward interval evaluation at this node for f */
    FiaItv backward;       /* backward interval evaluation at this node for dF */
    struct FunTnode *left; /* Left subtree */
    struct FunTnode *right;/* Right subtree */
};
typedef struct FunTnode FunTree;


#      define FunLeftTree(f)    ((f)->left)
#      define FunRightTree(f)   ((f)->right)
#      define FunTypeTree(f)    ((f)->type)
#      define FunItvFwd(f)      ((f)->forward)
#      define FunItvBwd(f)      ((f)->backward)

#      define FunStructOpTree(f) ((f)->val.eval)
#      define FunOpTree(f)      ((f)->val.eval->evalII)
#      define FunDleftTree(f)   ((f)->val.eval->evalDerivLeft)
#      define FunDrightTree(f)  ((f)->val.eval->evalDerivRight)
#      define FunDallTree(f)    ((f)->val.eval->evalDerivAll)

#      define FunItvTree(f)     ((f)->val.i)
#      define FunVarTree(f)     ((f)->val.var)
#      define FunGlobVarTree(f) ((f)->val.var[0])
#      define FunLocVarTree(f)  ((f)->val.var[1])


FunTree *FunFreeTree       (FunTree *f);
FunTree *FunCopyTree       (FunTree *f);
FunTree *FunNewOpTree      (FiaEvalOpI evalII,
                            FunEvalBwdPath evalDleft,
                            FunEvalBwdPath evalDright,
                            FunEvalBwdPath evalDall,
                            FunTree *l,FunTree *r);
FunTree *FunNewExpTree     (int exp);
FunTree *FunNewItvTree     (FiaItv i);
FunTree *FunNewEmptyItvTree();
FunTree *FunNewVarTree     (long globvar, long locvar);
FunTree *FunNewUselessTree ();
int     FunTreeZero        (FunTree *f);


/* LIST OF TREE NODES *******************************************************
   needed for the sum of the backward intervals for all the occurrences of
   one variable for partial derivative computations
   The nodes are ordered with respect to the backward evaluation
*****************************************************************************/
struct FunLtreeIn
{   
    FunTree *node;
    struct FunLtreeIn *next;
};
typedef struct
{
  struct FunLtreeIn *first;
  struct FunLtreeIn *end;
} FunLtree;


/* LIST OF (TREE NODESBACKWARD EVALUATION FUNCTION **************************
   needed for the computation of the partial derivative of one variable
*****************************************************************************/
struct FunLevalTreeIn
{   
    FunTree *node;
    FunEvalBwdPath eval;
    struct FunLevalTreeIn *next;
};
typedef struct
{
  struct FunLevalTreeIn *first;
  struct FunLevalTreeIn *end;
} FunLevalTree;


/* LOCAL INFORMATIONS FOR THE VARIABLES IN A TREE ***************************
*****************************************************************************/
typedef struct
{
    long globindex;
    int Nocc;
    FiaItv deriv;
    FunLtree *VarNodes;      /* Tree nodes containing this variable */
    FunLevalTree *EvalPath;  /* Nodes which must be backward evaluated to compute
                                the partial derivative for this variable */
    FunLtree *ReEvalFwd;     /* Nodes to be forward evaluated in Newton
			        to compute f'(X) after the forward
				evaluation of f(center(X)) */
} FunLocVar;


/* FUNCTION *****************************************************************
*****************************************************************************/
typedef struct
{
    FunTree *f;        /* expression */
    int number;        /* place in Acstr */
    int Nvar;          /* Number of variables in f */
    int Nfree;         /* needed for the parsing of f */
    FunLocVar *vars;   /* Informations associated to the variables in f */
    FunLtree *EvalFwd; /* Nodes to be forward evaluated */
    FunLevalTree *EvalBwd; /* Nodes to be backward evaluated for the computation
                              of all the partial derivatives */

    int pred;         /* type of constraint predicate */
}
FunFunction;

#      define FunLocVarSizeUnit 10
#      define FunFunTree(FunF) FunF->f
#      define FunFunNbVar(FunF) FunF->Nvar
#      define FunFunPred(FunF) FunF->pred
#      define FunLocToGlobVar(FunF,locvar)  (FunF->vars[locvar].globindex)
#      define FunLocVarDeriv(FunF,locvar)   (FunF->vars[locvar].deriv)
#      define FunLocVarNocc(FunF,locvar)    (FunF->vars[locvar].Nocc)
#      define FunLocVarPaths(FunF,locvar)   (FunF->vars[locvar].paths)


FunFunction *FunCreateF (FunTree *f, int pred);
void         FunFreeF   (FunFunction *FunF);


void     FunEvalFwdConstantTree (FunTree *f);
void     FunEvalFwdTree    (FunFunction *func, FiaItv ivar, long globvar);
void     FunEvalFwdTreeBis (FunFunction *func);
void     FunEvalCenterFwdTree (FunFunction *FunF);

void     FunEvalBwdTreeOne (FunFunction *func, long locvar);
void     FunEvalBwdTreeAll (FunFunction *func);

void FunReEvalFwdTree(FunFunction *FunF, FiaItv ivar, long globvar, long locvar);




/* PROTOTYPES OF BACKWARD EVALUATION FUNCTIONS ******************************
*****************************************************************************/

void FunEvalBwdAddII      (FunTree *f);
void FunEvalBwdSubII      (FunTree *f);
void FunEvalBwdNegI       (FunTree *f);
void FunEvalBwdMulII      (FunTree *f);
void FunEvalBwdDivII      (FunTree *f);
void FunEvalBwdSqrI       (FunTree *f);
void FunEvalBwdPowI       (FunTree *f);
void FunEvalBwdSqrtI      (FunTree *f);

void FunEvalBwdLeftAddII  (FunTree *f);
void FunEvalBwdRightAddII (FunTree *f);
void FunEvalBwdLeftSubII  (FunTree *f);
void FunEvalBwdRightSubII (FunTree *f);
void FunEvalBwdLeftMulII  (FunTree *f);
void FunEvalBwdRightMulII (FunTree *f);
void FunEvalBwdLeftDivII  (FunTree *f);
void FunEvalBwdRightDivII (FunTree *f);




/* INTERFACE WITH BSSOLVE ***************************************************
   (parsing, access to extern variable domains)
*****************************************************************************/


/* Name of the solver -> needed in BSSolve structures */
#define NEWTON_REP "newton"


/* Creation of the internal representation of the constraint represented by
   the wrapped string in s */
FunFunction * ParseFunTree(char *s);


/* To get the external domain of a variable  -> addition of the internal
   representation in BSSolve structures if it does not exist yet */
FiaBounds *FunGetExternDom(long globvar);



#        define FunCstrEq  1
#        define FunCstrSup 2
#        define FunCstrInf 3
#        define MAXCARFLOAT 40

/* Lexical units */
#        define ADD      1
#        define SUB      2
#        define MUL      3
#        define DIV      4
#        define POW      5
#        define MINUS    7
#        define SQR     11
#        define SQRT    13
#        define VAR     15
#        define LBR     16
#        define RBR     17
#        define TOKINT  18
#        define TOKRE   19
#        define DEC     20
#        define SEP     22
#        define TOKITV  23
#        define END     30
#        define TOKRAT  40


void GetTok();
#        define GetToken GetTok();

FunTree   * Expr();
FiaBounds * ExprFloat();               
int         ExprInt();
long        ExprLong();
FiaBounds * ExprItv();
FiaBounds * ExprRat();
int         Pred();
int         InvPred(int pred);
FiaBounds   *wstring2FiaItv(char *s);

#endif
