/***************************************************************************
* Functions
*
* FunCstr.c (Management of constraints and narrowing operators)
*
* Date:    1 feb 98
* Author:  L. Granvilliers - LIFO Orleans
****************************************************************************/
/*************************************************************************/
/*                                                                       */
/*       Copyright (C) 1998 Universite de Nantes                         */
/*                                                                       */
/*************************************************************************/


#include "FunCstr.h"



FiaR PragmaBoxPrecision = 0.0;


/***************************************************************************
* FILTERING and NARROWING ALGORITHMS
****************************************************************************/


int FunNarNewtonIter(FunTree *f, FunFunction *fun, long locvar, long globvar,
                     FiaBounds *in, FiaBounds *out)
/***************************************************************************
*  Iterative step of Newton narrowing
*  fun->f == f
*/
{
  FiaItv imid, idiv, isub;

#    ifdef DEBUGGING
  printf("tree{");
  FunWriteTree(stdout,f,Avars,9);
  printf("}\n");
#    endif

  /* 0 in f(X) ? */
  FunEvalFwdTree(fun,in,globvar);

#    ifdef DEBUGGING
           printf("f(X) : "); FiaWriteI(stdout,FunItvFwd(f),9); printf("\n");
#    endif

  if( !FiaRinI(FunItvFwd(f),0.0) )
  {
#    ifdef DEBUGGING
           printf("No zero of f\n");
#    endif

    FiaSetEmptyI(out);
    return( FunNarNewtonFail );
  }

  /* 0 in df(X) ? */
  FunEvalBwdTreeOne(fun,locvar);

#    ifdef DEBUGGING
           printf("f'(X) : "); FiaWriteI(stdout,FunLocVarDeriv(fun,locvar),9);
           printf("\n");
#    endif

  if( FiaRinI(FunLocVarDeriv(fun,locvar),0.0) )
  {
#    ifdef DEBUGGING
           printf("Zero of f'\n");
#    endif

    FiaCopyI(out,in);
    return( FunNarNewtonFlounder );
  }

  /* 0 in f(X) and 0 not in df(X) */
  FiaCopyI(out,in);

  while( !FiaCanonicalNarI(out) ) /*!FiaCaconicalNarI(out) )*/
  {
    /* Midpoint of the current domain of the variable */
    FiaRoundDown;
    FiaMinI(imid) = FiaMidRR(FiaMinI(out),FiaMaxI(out));
    FiaRoundUp;
    FiaMaxI(imid) = FiaMidRR(FiaMinI(out),FiaMaxI(out));

#    ifdef DEBUGGING
           printf("X       : "); FiaWriteI(stdout,out,9); printf("\n");
           printf("m(X)    : "); FiaWriteI(stdout,imid,9); printf("\n");
#    endif

    /* evaluation of f(X) and df(X) */
    FunEvalFwdTree(fun,imid,globvar);
    FunReEvalFwdTree(fun,out,globvar,locvar);
    FunEvalBwdTreeOne(fun,locvar);

#    ifdef DEBUGGING
           printf("f(m(X)) : "); FiaWriteI(stdout,FunItvFwd(f),9);
           printf("\nf'(X)   : "); FiaWriteI(stdout,FunLocVarDeriv(fun,locvar),9);
           printf("\n");
#    endif

    /* idiv <- f(X)/df(X) */
    FiaDivII(idiv,FunItvFwd(f),FunLocVarDeriv(fun,locvar));

    /* isub <- center(X) - f(X)/df(X) */
    FiaSubII(isub,imid,idiv);

    /* X <- X inter (center(X) - f(X)/df(X)) */
    if( FiaIsEqualInterII(out,out,isub) ) return( FunNarNewtonFlounder );
    if( FiaEmptyI(out) )    return( FunNarNewtonFail );
  }

/*printf("I:");FiaWriteI(stdout,out,9);printf("\n");*/

  return( FunNarNewtonSuccess );
}


int FunNarNewtonLeftBounded(FunFunction *fun, long globvar, FiaR x)
/***************************************************************************
*  in==[a,b]
*  To test if [a,a+] is the leftmost zero of f
*/
{
  FiaItv bound;
  FiaMinI(bound) = x;
  FiaMaxI(bound) = FiaNextR(x+PragmaBoxPrecision);

  /* 0 in f([a,a+]) ? */
  FunEvalFwdTree(fun,bound,globvar);

  if( FiaRinI(FunItvFwd(fun->f),0.0) ) return( 1 );
  else return( 0 );
}




int FunNarNewtonLeft(FunTree *f, FunFunction *fun, long locvar, long globvar,
                     FiaBounds *in, FiaBounds *out)
/***************************************************************************
*  To search the leftmost zero of f
*/
{
  FiaItv *stack = (FiaItv *)malloc(FunStackAllocUnit*sizeof(FiaItv));
  int current = 0, size = FunStackAllocUnit, found = 0, n;
  FiaR center;

  /* Copy of the input domain: the first one to be considered */
  FiaCopyI(stack[0],in);

  while( current>=0 && !found )
  {
    /* Try to reduce the current (leftmost) domain */
    n = FunNarNewtonIter(f,fun,locvar,globvar,stack[current],out);

    switch(n)
    {
      case FunNarNewtonFail:
              current --;
              break;

      case FunNarNewtonFlounder:
           if( FunNarNewtonLeftBounded(fun,globvar,FiaMinI(out)) )
                found = 1;
           else
           {
             if( current==size-1 )  /* needs of allocating */
	     {
               size += FunStackAllocUnit;
               stack = (FiaItv *)realloc(stack,size*sizeof(FiaItv));
	     }
             /* Split out in [a,b] cup [b,c] */
             center = FiaMidRR(FiaMinI(out),FiaMaxI(out));
             FiaCopyI(stack[current],out);
             FiaMinI(stack[current]) = center;
             current ++;
             FiaCopyI(stack[current],out);
             FiaMaxI(stack[current]) = center;
           }
           break;

      default: /* then success */
           found = 1;
    }
  }
  free(stack);
  if( !found )
  {
    FiaSetEmptyI(out);
    return( FunNarNewtonFail );
  }
  else return( FunNarNewtonSuccess );
}


int FunNarNewtonRightBounded(FunFunction *fun, long globvar, FiaR x)
/***************************************************************************
*  in==[a,b]
*  To test if [x-,x] is the leftmost zero of f
*/
{
  FiaItv bound;
  FiaMinI(bound) = FiaPrevR(x-PragmaBoxPrecision);
  FiaMaxI(bound) = x;

  /* 0 in f([x-,x]) ? */
  FunEvalFwdTree(fun,bound,globvar);

  if( FiaRinI(FunItvFwd(fun->f),0.0) ) return( 1 );
  else return( 0 );
}


int FunNarNewtonRight(FunTree *f, FunFunction *fun, long locvar, long globvar,
                      FiaBounds *in, FiaBounds *out)
/***************************************************************************
*  To search the rightmost zero of f
*/
{
  FiaItv *stack = (FiaItv *)malloc(FunStackAllocUnit*sizeof(FiaItv));
  int current = 0, size = FunStackAllocUnit, found = 0, n;
  FiaR center;

  /* Copy of the input domain: the first one to be considered */
  FiaCopyI(stack[0],in);

  while( current>=0 && !found )
  {
    /* Try to reduce the current (rightmost) domain */
    n = FunNarNewtonIter(f,fun,locvar,globvar,stack[current],out);

    switch(n)
    {
      case FunNarNewtonFail:
              current --;
              break;

      case FunNarNewtonFlounder:
           if( FunNarNewtonRightBounded(fun,globvar,FiaMaxI(out)) )
                found = 1;
           else
           {
             if( current==size-1 )  /* needs of allocating */
	     {
               size += FunStackAllocUnit;
               stack = (FiaItv *)realloc(stack,size*sizeof(FiaItv));
	     }
             /* Split out in [a,b] cup [b,c] */
             center = FiaMidRR(FiaMinI(out),FiaMaxI(out));
             FiaCopyI(stack[current],out);
             FiaMaxI(stack[current]) = center;
             current ++;
             FiaCopyI(stack[current],out);
             FiaMinI(stack[current]) = center;
           }
           break;

      default: /* then success */
           found = 1;
    }
  }
  free(stack);
  if( !found )
  {
    FiaSetEmptyI(out);
    return( FunNarNewtonFail );
  }
  else return( FunNarNewtonSuccess );
}



int FunNarNewtonEq(FunFunction *fun, long locvar, long globvar, FiaBounds *out)
/***************************************************************************
*  Implementation of the Newton narrowing operator for equations
*/
{
  FiaItv out2, in3;
  FiaBounds *in = FunGetExternDom(globvar);
  int n;

#    ifdef DEBUGGING
  printf("Dom(%s) : ",FunNameVar(Avars,globvar));
  FiaWriteI(stdout,FunDomV(d,globvar),9);
#    endif



  n = FunNarNewtonIter(FunFunTree(fun),fun,locvar,globvar,in,out);

#    ifdef DEBUGGING
  printf("    ->   Box = ");
  FiaWriteI(stdout,out,9);
  printf("\n");
#    endif



  if( n==FunNarNewtonFlounder )
  {
    n = FunNarNewtonLeft(FunFunTree(fun),fun,locvar,globvar,out,out2);


#    ifdef DEBUGGING
    printf("    Left = ");
    FiaWriteI(stdout,out2,9);
    printf("\n");
#    endif





    if( n==FunNarNewtonSuccess ) /* FiaMinI(out2) is the leftmost zeros */
    {
      FiaMinI(in3) = FiaMinI(out2); FiaMaxI(in3) = FiaMaxI(out);
      n = FunNarNewtonRight(FunFunTree(fun),fun,locvar,globvar,in3,out);


#    ifdef DEBUGGING
      printf("    Right = ");
      FiaWriteI(stdout,out,9);
      printf("\n");
#    endif


      if( n==FunNarNewtonFail )
      {
        FiaSetEmptyI(out);
        return( 0 );
      }
      else
      {
        FiaMinI(out) = FiaMinI(out2);
        return( 1 );
      }
    }
    else
    {
      FiaSetEmptyI(out); /* no zero of fun->f in out */
      return( 0 );
    }
  }
  return( n );
}


int FunNarNewtonSup(FunFunction *fun, long locvar, long globvar, FiaBounds *out)
/***************************************************************************
*  Implementation of the Newton narrowing operator for f >= 0
*/
{
  FiaItv bound, out2, in3;
  FiaBounds *in = FunGetExternDom(globvar);
  int n;

  FiaMinI(bound) = FiaMinI(in);
  FiaMaxI(bound) = FiaNextR(FiaMinI(in));

  /* right(f([left,left+])) >= 0 ? */
  FunEvalFwdTree(fun,bound,globvar);
  if( FiaMaxI(FunItvFwd(fun->f)) >= 0 )
  {
    FiaCopyI(out,in);
  }
  else
  {
    n = FunNarNewtonLeft(FunFunTree(fun),fun,locvar,globvar,in,out);
    if( n!=FunNarNewtonSuccess )
    {
      FiaSetEmptyI(out);
      return( 0 );
    }
    else
    {
      FiaMaxI(out) = FiaMaxI(in);
    }
  }

  FiaMinI(bound) = FiaPrevR(FiaMaxI(out));
  FiaMaxI(bound) = FiaMaxI(out);

  /* right(f([right,right+])) >= 0 ? */
  FunEvalFwdTree(fun,bound,globvar);
  if( FiaMaxI(FunItvFwd(fun->f)) >= 0 )
  {
    return( 1 );
  }
  else
  {
    n = FunNarNewtonRight(FunFunTree(fun),fun,locvar,globvar,out,out2);
    if( n!=FunNarNewtonSuccess )
    {
      FiaSetEmptyI(out);
      return( 0 );
    }
    else
    {
      FiaMaxI(out) = FiaMaxI(out2);
      return( 1 );
    }
  }
}


int FunNarNewtonInf(FunFunction *fun, long locvar, long globvar, FiaBounds *out)
/***************************************************************************
*  Implementation of the Newton narrowing operator for f <= 0
*/
{
  FiaItv bound, out2, in3;
  FiaBounds *in = FunGetExternDom(globvar);
  int n;

  FiaMinI(bound) = FiaMinI(in);
  FiaMaxI(bound) = FiaNextR(FiaMinI(in));

  /* left(f([left,left+])) <= 0 ? */
  FunEvalFwdTree(fun,bound,globvar);
  if( FiaMinI(FunItvFwd(fun->f)) <= 0 )
  {
    FiaCopyI(out,in);
  }
  else
  {
    n = FunNarNewtonLeft(FunFunTree(fun),fun,locvar,globvar,in,out);
    if( n!=FunNarNewtonSuccess )
    {
      FiaSetEmptyI(out);
      return( 0 );
    }
    else
    {
      FiaMaxI(out) = FiaMaxI(in);
    }
  }

  FiaMinI(bound) = FiaPrevR(FiaMaxI(out));
  FiaMaxI(bound) = FiaMaxI(out);

  /* left(f([right,right+])) <= 0 ? */
  FunEvalFwdTree(fun,bound,globvar);
  if( FiaMinI(FunItvFwd(fun->f)) <= 0 )
  {
    return( 1 );
  }
  else
  {
    n = FunNarNewtonRight(FunFunTree(fun),fun,locvar,globvar,out,out2);
    if( n!=FunNarNewtonSuccess )
    {
      FiaSetEmptyI(out);
      return( 0 );
    }
    else
    {
      FiaMaxI(out) = FiaMaxI(out2);
      return( 1 );
    }
  }
}


int FunNarTaylorEq(FunFunction *fun, long locvar, long globvar, FiaBounds *out)
/***************************************************************************
*  Implementation of the Taylor narrowing operator for equations
*  in the case of no preconditionning
*/
{
  FiaBounds *in = FunGetExternDom(globvar);
  FiaItv isum, imid, isub, imul, idiv;
  FiaBounds *itv;

  int i;

  /* isum <- f(center(X)) */
  FunEvalCenterFwdTree(fun);
  FiaCopyI(isum,FunItvFwd(fun->f));

#    ifdef DEBUGGING
           printf("f(center)   : "); FiaWriteI(stdout,isum,9); printf("\n");
#    endif

  /* evaluation of f(X) and all the partial derivatives df(X)/dx */
  FunEvalFwdTree(fun,in,globvar);
  FunEvalBwdTreeAll(fun);

  for( i=0; i<locvar; i++ )
  {
#    ifdef DEBUGGING
           printf("f'(X_%d)   : ",i); FiaWriteI(stdout,FunLocVarDeriv(fun,i),9);
           printf("\n");
#    endif

    /* midpoint of dom(var of local index i) */
    itv = FunGetExternDom(FunLocToGlobVar(fun,i));
    FiaRoundDown;
    FiaMinI(imid) = FiaMidI(itv);
    FiaRoundUp;
    FiaMaxI(imid) = FiaMidI(itv);

    /* isub <- dom(var i) - center(dom(var i)) */
    FiaSubII(isub,itv,imid);

    /* imul <- df/d(var i) * [dom(var i) - center(dom(var i))] */
    FiaMulII(imul,FunLocVarDeriv(fun,i),isub);

    FiaAddII(isum,isum,imul);
  }
  for( i=locvar+1; i<fun->Nvar; i++ )
  {
#    ifdef DEBUGGING
           printf("f'(X_%d)   : ",i); FiaWriteI(stdout,FunLocVarDeriv(fun,i),9);
           printf("\n");
#    endif

    /* midpoint of dom(var of local index i) */
    itv = FunGetExternDom(FunLocToGlobVar(fun,i));
    FiaRoundDown;
    FiaMinI(imid) = FiaMidI(itv);
    FiaRoundUp;
    FiaMaxI(imid) = FiaMidI(itv);

    /* isub <- dom(var i) - center(dom(var i)) */
    FiaSubII(isub,itv,imid);

    /* imul <- df/d(var i) * [dom(var i) - center(dom(var i))] */
    FiaMulII(imul,FunLocVarDeriv(fun,i),isub);

    FiaAddII(isum,isum,imul);
  }

#    ifdef DEBUGGING
           printf("f'(X_%d)   : ",locvar);
           FiaWriteI(stdout,FunLocVarDeriv(fun,locvar),9);
           printf("\nSum   : "); FiaWriteI(stdout,isum,9); printf("\n");
#    endif

    /* midpoint of dom(var of local index locvar) */
    FiaRoundDown;
    FiaMinI(imid) = FiaMidI(in);
    FiaRoundUp;
    FiaMaxI(imid) = FiaMidI(in);

    /* idiv <- isum / df/d(locvar) */
    FiaDivII(idiv,isum,FunLocVarDeriv(fun,locvar));

#    ifdef DEBUGGING
           printf("Div   : "); FiaWriteI(stdout,idiv,9); printf("\n");
#    endif

    /* imul contains the interval evaluation of the Taylor expression */
    FiaSubII(imul,imid,idiv);

#    ifdef DEBUGGING
           printf("Sub   : "); FiaWriteI(stdout,imul,9); printf("\n");
           printf("Input   : "); FiaWriteI(stdout,in,9); printf("\n");
#    endif

    FiaInterII(out,in,imul);

#    ifdef DEBUGGING
           printf("Output   : "); FiaWriteI(stdout,out,9); printf("\n");
#    endif

    if( FiaEmptyI(out) ) return( 0 );
    else return( 1 );


}



int FunNarNewtonTaylorEq(FunFunction *fun, long locvar, long globvar, FiaBounds *out)
/***************************************************************************
*  Narrowing operator computing in sequence (Taylor o Newton)(d)
*/
{
  FiaItv out2, save;
  FiaBounds *itv;

  if( FunNarNewtonEq(fun,locvar,globvar,out2) )
  {
    itv = FunGetExternDom(globvar);
    FiaCopyI(save,itv);
    FiaCopyI(itv,out2);
    if( FunNarTaylorEq(fun,locvar,globvar,out) )
    {
      FiaCopyI(itv,save);
      return( 1 );
    }
    else
    {
      return( 0 );
    }
  }
  else
  {
    FiaSetEmptyI(out);
    return( 0 );
  }
}





/***************************************************************************
* CREATION OF NARROWING OPERATORS from CONSTRAINTS
****************************************************************************/



long FunSearchLocvar(FunFunction *f, long globvar)
/***************************************************************************
*  To search the local index of the variable globvar in f
*/
{
  int i;
  for( i=0; i<FunFunNbVar(f); i++ )
  {
    if( FunLocToGlobVar(f,i)==globvar ) return( i );
  }
}


FunOpNar *FunOpNarCreateOne(FunFunction *f, long globvar)
/***************************************************************************
*  Creation of one narrowing operator for the projection of c wrt globvar
*/
{
  FunOpNar *op = (FunOpNar *)malloc(sizeof(FunOpNar));
/*   switch( FunTypeCstr(FunFunPred(f)) ) */
  switch( FunFunPred(f) )
  {
    case FunCstrEq:
         op->fun = f;
         op->locvar = FunSearchLocvar(f,globvar);
         op->globvar = globvar;
         op->prune = (FunPruneProj)FunNarNewtonTaylorEq;
         return( op );

    case FunCstrInf:
         op->fun = f;
         op->locvar = FunSearchLocvar(f,globvar);
         op->globvar = globvar;
         op->prune = (FunPruneProj)FunNarNewtonInf;
         return( op );

    case FunCstrSup:
         op->fun = f;
         op->locvar = FunSearchLocvar(f,globvar);
         op->globvar = globvar;
         op->prune = (FunPruneProj)FunNarNewtonSup;
         return( op );
  }
}


int FunOpNarApply(FunOpNar *op)
/***************************************************************************
*  To apply the narrowing operator op
*  Returns  0 if the domain is empty
*           1 if the domain has not changed
*           2 otherwise
*/
{
  FiaItv out;
  FiaBounds *in;
  int n;
  void *bsrep;
  
  /* Input domain for the variable globvar */
  in = FunGetExternDom(op->globvar);
  /*
      printf( "VARIABLE %d AVANT PRUNE -> ", op->globvar);
      FiaWriteI( stdout, in, 25);
      putchar('\n');
  */
  /* application */
  n = (* op->prune)(op->fun,op->locvar,op->globvar,out);

  if( n )  /* success */
  {
    if( FiaIdiffI(in,out) )  /* domain has changed */
    {
      /* get the bssolve rep. of the variable */
      bsrep = LocVarGetVar(op->globvar);

      
      /* the pointer in is a structure allocated in processor -> it will contain
         the new domain for the variable */
      FiaCopyI(in,out);

      /* Update the domain of the variable, locally on processor,
         and for my representation */
      VarUpdateRep(bsrep,NEWTON_REP,in);
      /*
      printf( "VARIABLE %d UPDATE -> ", op->globvar);
      FiaWriteI( stdout, in, 25);
      putchar('\n');
      */
      return( 2 );
    }
    else   /* domain has not changed */
    {
      return( 1 );
    }
  }

  /* else the domain is empty */


  return( 0 );
}
