/***************************************************************************
* Groebner
*
* Groebner.c (Management of groebner bases)
*
* Date:    1 feb 98
* Author:  L. Granvilliers - LIFO Orleans
****************************************************************************/

/*************************************************************************/
/*                                                                       */
/*       Copyright (C) 1998 Universite de Nantes                         */
/*                                                                       */
/*************************************************************************/

#include "Groebner.h"
#include <floatingpoint.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <bssolve.h>


#define GbStrToInt(s)  atoi(s)   /* string to integer */
#define GbStrToR(s)    atof(s)   /* string to double */
#define GbStrToLong(s) atol(s)   /* string to long */



/**********************************************************************************
*                      POLYNOMIAL BASIC OPERATIONS
***********************************************************************************/

GbTerm *GbTermAlloc()
/***************************************************************************
* To allocate a Gb TERM
*/
{
  return( (GbTerm *)malloc(sizeof(GbTerm)) );
}



GbMonomial *GbMonomialAlloc(GbTerm *t, long num, unsigned long den)
/***************************************************************************
* To allocate a Gb MONOMIAL
*/
{
  mpq_t q;
  GbMonomial *m = (GbMonomial *)malloc(sizeof(GbMonomial));
  Mlterm(m) = (GbListTerm *)malloc(sizeof(GbListTerm));

  mpq_init(Mcoeff(m));
  mpq_set_si(Mcoeff(m),num,den);
  mpq_canonicalize(Mcoeff(m));

  LTfirst(Mlterm(m)) = LTend(Mlterm(m)) = t;
  return( m );
}


GbPoly *GbGetPolyZero()
/***************************************************************************
* To allocate the 0-polynomial
*/
{
  GbPoly *p;
  p = (GbPoly *)malloc(sizeof(GbPoly));
  Pmono(p) = GbMonomialAlloc(NULL,0,1);
  Pnext(p) = NULL;
  return( p );
}

void GbFreeT(GbTerm *t)
/***************************************************************************
* To desallocate a term
*/
{
  GbTerm *t1;
  while( t!=NULL ) { t1 = t; t = Tnext(t); free(t1); }
} 


void GbFreeLT(GbListTerm *lt)
/***************************************************************************
* To desallocate a list of terms
*/
{
  GbTerm *t1, *t2;
  if( (lt!=NULL) && (LTfirst(lt)!=NULL) ) GbFreeT(LTfirst(lt));
  free(lt);
}
      

void GbFreeM(GbMonomial *m)
/***************************************************************************
* To desallocate a monomial
*/
{
  GbFreeLT(Mlterm(m));
  mpq_clear(Mcoeff(m));
  free(m);
}


void GbFreeP(GbPoly *p)
/***************************************************************************
* To desallocate a polynomial
*/
{
  GbPoly *p1;
  while( p!=NULL ) { p1 = p; p = Pnext(p); GbFreeM(Pmono(p1)); free(p1); }
}



GbTerm *GbCopyT(GbTerm *t)
/***************************************************************************
* To create a copy of the term t
*/
{
  GbTerm *tcopy, *taux;
 
  if( t==NULL ) return( NULL );

  tcopy = GbTermAlloc();
  Tvar(tcopy) = Tvar(t);
  Texp(tcopy) = Texp(t);
  Tprev(tcopy) = NULL;
  t = Tnext(t);
  taux = tcopy;  
   
  while( t!=NULL )
  {
    Tnext(taux) = GbTermAlloc();
    Tvar(Tnext(taux)) = Tvar(t);
    Texp(Tnext(taux)) = Texp(t);
    Tprev(Tnext(taux)) = taux;
    taux = Tnext(taux);
    t = Tnext(t);
  }
  Tnext(taux) = NULL;
  return( tcopy );
}


GbListTerm *GbCopyLT(GbListTerm *lt)
/***************************************************************************
* To create a copy of the list of terms `lt'
*/
{
  GbListTerm *lcopy;
  GbTerm *t;
   
  lcopy = (GbListTerm *)malloc(sizeof(GbListTerm));
  LTfirst(lcopy) = GbCopyT(LTfirst(lt));

  t = LTfirst(lcopy);
  if( t==NULL ) { LTend(lcopy) = NULL; }

  else { while( Tnext(t)!=NULL ) t = Tnext(t); LTend(lcopy) = t; }

  return( lcopy );
}


int GbMcmplexM(GbMonomial *m1, GbMonomial *m2)
/***************************************************************************
* To compare two monomials with the lexicographic ordering
*/
{
  GbTerm *t1, *t2;
  int CMP = GbVarEq, found = 0;

  t1 = LTfirst(Mlterm(m1)); t2 = LTfirst(Mlterm(m2));

  while( (!found) && (t1!=NULL) && (t2!=NULL) )
  {
    CMP = GbCmpVar(Tvar(t1),Tvar(t2));
    if( CMP==GbVarSup ) found = 1;
    else if( CMP==GbVarInf ) found = 1;
    else
    {
      if( Texp(t1) > Texp(t2) ) { found = 1; CMP = GbVarSup; }
      else if( Texp(t1) < Texp(t2) ) { found = 1; CMP = GbVarInf; }
      else { t1 = Tnext(t1); t2 = Tnext(t2); }
    }
  }
  if( (t1==NULL) && (t2!=NULL) ) return( GbVarInf );
  else if( (t1!=NULL) && (t2==NULL) ) return( GbVarSup );
  else return( CMP );
}


int GbMcmpM(GbMonomial *m1, GbMonomial *m2)
/***************************************************************************
* To compare two monomials
*/
{
  if( (Mlterm(m1)==NULL) || (LTfirst(Mlterm(m1))==NULL) )
  {
    if( (Mlterm(m2)==NULL) || (LTfirst(Mlterm(m2))==NULL) )
      return( GbVarEq );
    else return( GbVarInf );
  }
  else if( (Mlterm(m2)==NULL) || (LTfirst(Mlterm(m2))==NULL) )
    return( GbVarSup );

  return( GbMcmplexM(m1,m2) );
}


int BsGbPolyEqual(GbPoly *p1, GbPoly *p2)
/***************************************************************************
* To test if p1 = p2
*/
{
  GbPoly *p3, *p4;
  int n;

  p3 = p1;
  p4 = p2;
  while( (p3!=NULL) && (p4!=NULL) )
  {
    n = GbMcmpM(Pmono(p3),Pmono(p4));
    if( n==GbVarInf ) return( 0 );
    else if( n==GbVarSup ) return( 0 );
    else
    {
      p3 = Pnext(p3); p4 = Pnext(p4);
    }
  }
  if( (p3==NULL) && (p4==NULL) ) return( 1 );
  else return( 0 );
}



int GbPolyMin(GbPoly *p1, GbPoly *p2)
/***************************************************************************
* To test if p1 < p2
*  Returns 1 if p1 < p2, 2 otherwise
*/
{
  GbPoly *p3, *p4;
  int n;

  if( p1==NULL ) return( 1 );
  if( p2==NULL ) return( 2 );

  p3 = p1;
  p4 = p2;
  while( (p3!=NULL) && (p4!=NULL) )
  {
    n = GbMcmpM(Pmono(p3),Pmono(p4));
    if( n==GbVarInf ) return( 1 );
    else if( n==GbVarSup ) return( 2 );
    else
    {
      p3 = Pnext(p3); p4 = Pnext(p4);
    }
  }
  if( p3==NULL ) return( 1 );
  else return( 2 );
}


void BsGbPolyMin(int uselesstag, char **res, int *size,
                 char *sp1, int size1, char *sp2, int size2)
/***************************************************************************
*  To compare two polynomials
*/  
{
  GbPoly *p1, *p2;

  p1 = GbParsePoly(sp1);
  p2 = GbParsePoly(sp2);

  if( GbPolyMin(p1,p2)==1 )
  {
    *size = size1;
    *res = malloc( size1);
    memcpy(*res, sp1, size1);
  }
  else
  {
    *size = size2;
    *res = malloc( size2);
    memcpy(*res, sp2, size2);
  }
  GbFreeP(p1);
  GbFreeP(p2); 
}





int GbPolyZero(GbPoly *p)
/***************************************************************************
* To test if p = 0
*/
{
  if( p==NULL ) return( 1 );
  if( Pnext(p)!=NULL ) return( 0 );
  if( mpq_sgn(Mcoeff(Pmono(p)))==0 ) return( 1 );
  else return( 0 );
}


GbPoly *GbPdelMzero(GbPoly *p)
/***************************************************************************
* To delete in `p' the monomials with a coefficient equal to 0
*/
{
  GbPoly *cur, *prev, *aux;

  aux = cur = p;
  while( (cur!=NULL) && (mpq_sgn(Mcoeff(Pmono(cur)))==0)) cur = Pnext(cur);
  p = cur;

  /* To delete the leftmost zeros in `p' */
  while( aux!=p )
  {
    prev = aux;
    aux = Pnext(aux);  
    GbFreeM(Pmono(prev));
    free(prev);
  }
  
  if( p!=NULL )
  {
    prev = p;
    cur = Pnext(p);
    while( cur!=NULL )
    {
      if( mpq_sgn(Mcoeff(Pmono(cur))) ==0 )
      {
        Pnext(prev) = Pnext(cur);
        GbFreeM(Pmono(cur));
        free(cur);
        cur = Pnext(prev);
      }
      else
      {
        prev = cur;
        cur = Pnext(cur);
      }
    }
  }
  return( p );   
}


void GbLTaddT(GbListTerm *lt, GbTerm *t)
{  
  GbTerm *current;
  int CMPVAR;

  if( LTfirst(lt)==NULL )
  {
    LTfirst(lt) = LTend(lt) = t;
    return;
  }

  CMPVAR = GbCmpVar(Tvar(t),Tvar(LTfirst(lt)));
  if( CMPVAR==GbVarEq )
  {
    Texp(LTfirst(lt)) += Texp(t);
    GbFreeT(t);
    return;
  }
  if( CMPVAR==GbVarSup )
  {
    Tnext(t) = LTfirst(lt);
    Tprev(t) = NULL;
    Tprev(LTfirst(lt)) = t;
    LTfirst(lt) = t;
    return;
  }

  current = LTfirst(lt);
  while( (Tnext(current)!=NULL) && (GbCmpVar(Tvar(t),Tvar(Tnext(current)))==GbVarInf) )
     current = Tnext(current);

  if( (Tnext(current)!=NULL) && (GbCmpVar(Tvar(t),Tvar(Tnext(current)))==GbVarEq) )
  {
    Texp(Tnext(current)) += Texp(t);
    GbFreeT(t);
    return;
  }
  else
  {
    Tnext(t) = Tnext(current);
    Tprev(t) = current;
    Tnext(current) = t;
    if( Tnext(t)!=NULL ) Tprev(Tnext(t)) = t;
    else LTend(lt) = t;
  }
}
   


GbMonomial *GbMfusionM(GbMonomial *m1, GbMonomial *m2)
/***************************************************************************
* Return the polynomial `m1*m2'
* `m1' and `m2' are transformed by this operation
*/
{
  mpq_t q;
  GbTerm *t, *copy;

  /* Coefficient */
  mpq_init(q);
  mpq_mul(q,Mcoeff(m1),Mcoeff(m2));
  mpq_set(Mcoeff(m1),q);
  mpq_set(Mcoeff(m2),q);
  mpq_clear(q);

  if( Mlterm(m2)==NULL ) return( m1 );
  else if( Mlterm(m1)==NULL ) return( m2 );
  else
  {
    t = LTfirst(Mlterm(m2));
    while(t!=NULL )
    {
      copy = GbTermAlloc();
      Tvar(copy) = Tvar(t);
      Texp(copy) = Texp(t);
      Tprev(copy) = Tnext(copy) = NULL;
      GbLTaddT(Mlterm(m1),copy);
      t = Tnext(t);
    }
    GbFreeM(m2);
    return( m1 );
  }
}


GbPoly *GbPfusionP(GbPoly *p1, GbPoly *p2)
/***************************************************************************
* Return the polynomial `p1+p2'
* `p1' and `p2' are transformed by this operation
*/
{
  GbPoly *pfusion, *paux, *cur1, *cur2, *pdel;
  int CMP;
        
  if( p1==NULL ) return( p2 );
  if( p2==NULL ) return( p1 );
 
  CMP = GbMcmpM(Pmono(p1),Pmono(p2));
  if( CMP==GbVarSup )
  {
    pfusion = paux = p1;
    cur1 = Pnext(p1);
    cur2 = p2;
  }
  else if( CMP==GbVarInf )
  {
    pfusion = paux = p2;
    cur1 = p1;
    cur2 = Pnext(p2);
  }
  else
  { 
    mpq_add(Mcoeff(Pmono(p1)),Mcoeff(Pmono(p1)),Mcoeff(Pmono(p2)));
    pfusion = paux = p1;
    cur1 = Pnext(p1);
    cur2 = Pnext(p2);
    GbFreeM(Pmono(p2));
    free(p2);
  }

  while( (cur1!=NULL) && (cur2!=NULL) )
  {
    CMP = GbMcmpM(Pmono(cur1),Pmono(cur2));
    if( CMP==GbVarSup )
    {  
      Pnext(paux) = cur1;
      paux = cur1;
      cur1 = Pnext(cur1);
    }
    else if( CMP==GbVarInf )
    {
      Pnext(paux) = cur2;
      paux = cur2;
      cur2 = Pnext(cur2);
    }
    else
    {
      mpq_add(Mcoeff(Pmono(cur1)),Mcoeff(Pmono(cur1)),Mcoeff(Pmono(cur2)));
      Pnext(paux) = cur1;   
      paux = cur1;
      cur1 = Pnext(cur1);
      pdel = cur2;  
      cur2 = Pnext(cur2);
      GbFreeM(Pmono(pdel));
      free(pdel);
    }
  }

  if( cur1!=NULL ) Pnext(paux) = cur1;
  else Pnext(paux) = cur2;

  pfusion = GbPdelMzero(pfusion);

  return( pfusion );
} 





GbMonomial *GbMmulM(GbMonomial *m1, GbMonomial *m2)
/***************************************************************************
* Return the monomial `m1*m2'
*/
{
  GbMonomial *mnew;
  GbTerm *tnew, *t1, *t2, *t3;
  int CMPVAR;

  mnew = GbMonomialAlloc(NULL,1,1);
  mpq_mul(Mcoeff(mnew),Mcoeff(m1),Mcoeff(m2));

  t1 = LTfirst(Mlterm(m1));
  t2 = LTfirst(Mlterm(m2));

  if( (t1==NULL) || (t2==NULL) )
  {
    if( t2==NULL )
    {
      if( t1==NULL ) return( mnew );
      else { free(Mlterm(mnew)); Mlterm(mnew) = GbCopyLT(Mlterm(m1)); return( mnew ); }
    }
    else { free(mnew->lterm); Mlterm(mnew) = GbCopyLT(Mlterm(m2)); return( mnew ); }
  }

  LTfirst(Mlterm(mnew)) = GbTermAlloc();
  LTend(Mlterm(mnew)) = LTfirst(Mlterm(mnew));

  tnew = LTfirst(Mlterm(mnew));
  Tprev(tnew) = NULL;

  CMPVAR = GbCmpVar(Tvar(t1),Tvar(t2));

  if( CMPVAR==GbVarSup )
    { Tvar(tnew) = Tvar(t1); Texp(tnew) = Texp(t1); t1 = Tnext(t1); }

  else if( CMPVAR==GbVarInf )
    { Tvar(tnew) = Tvar(t2); Texp(tnew) = Texp(t2); t2 = Tnext(t2); }

  else
  {
    Tvar(tnew) = Tvar(t2); Texp(tnew) = Texp(t2) + Texp(t1);
    t1 = Tnext(t1); t2 = Tnext(t2);
  }

  while( (t1!=NULL) && (t2!=NULL) )
  {
    Tnext(tnew) = GbTermAlloc();
    Tprev(Tnext(tnew)) = tnew;
    tnew = Tnext(tnew);
    LTend(Mlterm(mnew)) = tnew;

    CMPVAR = GbCmpVar(Tvar(t1),Tvar(t2));

    if( CMPVAR==GbVarSup )
      { Tvar(tnew) = Tvar(t1); Texp(tnew) = Texp(t1); t1 = Tnext(t1); }

    else if( CMPVAR==GbVarInf )
      { Tvar(tnew) = Tvar(t2); Texp(tnew) = Texp(t2); t2 = Tnext(t2); }

    else
    {
      Tvar(tnew) = Tvar(t2); Texp(tnew) = Texp(t2) + Texp(t1);
      t1 = Tnext(t1); t2 = Tnext(t2);
    }
  }

  if( t1==NULL ) t3 = t2;
  else t3 = t1;

  if( t3!=NULL )
  {
    while( t3!=NULL )
    {
      Tnext(tnew) = GbTermAlloc();
      Tprev(Tnext(tnew)) = tnew;
      tnew = Tnext(tnew);
      LTend(Mlterm(mnew)) = tnew;
      Tvar(tnew) = Tvar(t3);
      Texp(tnew) = Texp(t3);
      t3 = Tnext(t3);
    }
  }
  Tnext(tnew) = NULL;
  LTend(Mlterm(mnew)) = tnew;
  return( mnew );
}


GbPoly *GbPmulM(GbPoly *p, GbMonomial *m)
/***************************************************************************
* Return the polynomial `m*p'
*/
{
  GbPoly *pmul, *pmul2;

  pmul = (GbPoly *)malloc(sizeof(GbPoly));
  Pmono(pmul) = GbMmulM(Pmono(p),m);
  pmul2 = pmul;
  p = Pnext(p);

  while( p!=NULL )
  {
    Pnext(pmul2) = (GbPoly *)malloc(sizeof(GbPoly));
    pmul2 = Pnext(pmul2);
    Pmono(pmul2) = GbMmulM(Pmono(p),m);
    p = Pnext(p);
  }
  Pnext(pmul2) = NULL;
  return( pmul );
}


void GbWriteRat(FILE *out, mpq_t q)
{
  int inv = 0;
  mpz_t num, den;
  mpz_init(num); mpz_init(den);

  if( mpq_sgn(q) < 0 )
  {
    mpq_neg(q,q);
    inv = 1;
  }

  mpq_get_num(num,q); mpq_get_den(den,q);

  if( mpz_cmp_ui(den,1)==0 )
  {
    if( mpz_cmp_ui(num,1)==0 )
      fprintf(stdout,"i(1)");
    else
    {
      fprintf(stdout,"i(");
      mpz_out_str(out,10,num);
      fprintf(stdout,")");
    }
  }
  else
  {
    fprintf(out,"q(");
    mpz_out_str(out,10,num); fprintf(out,":"); mpz_out_str(out,10,den);
    fprintf(out,")");
  }
  mpz_clear(num); mpz_clear(den);

  if( inv ) mpq_neg(q,q);
}


void GbWriteP(FILE *out, GbPoly *p)
{
  GbPoly *p1;
  GbTerm *t;

  if( p==NULL ) return;

  if( mpq_sgn(Mcoeff(Pmono(p))) < 0 ) fprintf(out,"-");
  GbWriteRat(out,Mcoeff(Pmono(p)));
  
  t = LTfirst(Mlterm(Pmono(p)));
  while( t!=NULL )
  {
    fprintf(out,"*x(%ld)",Tvar(t));
    if( Texp(t)>1 ) fprintf(out,"^%d",Texp(t));
    t = Tnext(t);
  }

  p1 = Pnext(p);

  while( p1 != NULL )
  {
    if( mpq_sgn(Mcoeff(Pmono(p1))) < 0 ) fprintf(out," - ");
    else fprintf(out," + ");

    GbWriteRat(out,Mcoeff(Pmono(p1)));
  
    t = LTfirst(Mlterm(Pmono(p1)));
    while( t!=NULL )
    {
      fprintf(out,"*x(%ld)",Tvar(t));
      if( Texp(t)>1 ) fprintf(out,"^%d",Texp(t));
      t = Tnext(t);
    }

    p1 = Pnext(p1);
  }
}


/**********************************************************************************
*                         POLYNOMIAL OPERATIONS for GB
***********************************************************************************/


GbTerm  *GbLcmMonomial(GbMonomial *m1, GbMonomial *m2)
/***************************************************************************
* To compute the Least Common Multiple of `m1' and `m2'
*/
{
  GbTerm *tfirst, *tnew, *t1, *t2, *t3;
  int CMPVAR;

  t1 = LTfirst(Mlterm(m1));
  t2 = LTfirst(Mlterm(m2));

  tnew = GbTermAlloc();
  Tprev(tnew) = NULL;
  tfirst = tnew;

  CMPVAR = GbCmpVar(Tvar(t1),Tvar(t2));

  if( CMPVAR==GbVarSup )
    { Tvar(tnew) = Tvar(t1); Texp(tnew) = Texp(t1); t1 = Tnext(t1); }

  else if( CMPVAR==GbVarInf )
    { Tvar(tnew) = Tvar(t2); Texp(tnew) = Texp(t2); t2 = Tnext(t2); }

  else
  {
    Tvar(tnew) = Tvar(t2); Texp(tnew) = GbMaxRR(Texp(t2),Texp(t1));
    t1 = Tnext(t1); t2 = Tnext(t2);
  }

  while( (t1!=NULL) && (t2!=NULL) )
  {
    Tnext(tnew) = GbTermAlloc();
    Tprev(Tnext(tnew)) = tnew; tnew = Tnext(tnew);

    CMPVAR = GbCmpVar(Tvar(t1),Tvar(t2));
    if( CMPVAR==GbVarSup )
      { Tvar(tnew) = Tvar(t1); Texp(tnew) = Texp(t1); t1 = Tnext(t1); }

    else if( CMPVAR==GbVarInf )
      { Tvar(tnew) = Tvar(t2); Texp(tnew) = Texp(t2); t2 = Tnext(t2); }

    else
    {
      Tvar(tnew) = Tvar(t2); Texp(tnew) = GbMaxRR(Texp(t2),Texp(t1));
      t1 = Tnext(t1); t2 = Tnext(t2);
    }
  }

  if( t1==NULL ) t3 = t2;
  else t3 = t1;

  if( t3!=NULL )
  {
    while( t3!=NULL )
    {
      Tnext(tnew) = GbTermAlloc();
      Tprev(Tnext(tnew)) = tnew; tnew = Tnext(tnew);
      Tvar(tnew) = Tvar(t3); Texp(tnew) = Texp(t3);
      t3 = Tnext(t3);
    }
  }
  Tnext(tnew) = NULL;
  return( tfirst );
}



GbMonomial *GbMdivM(GbMonomial *m, GbMonomial *div)
/***************************************************************************
* To compute the monomial `m1/m2'
*/
{
  GbMonomial *mnew;
  GbTerm *tnew, *t, *tdiv, *t3;
  int CMPVAR;

  mnew = GbMonomialAlloc(NULL,1,1);
  mpq_div(Mcoeff(mnew),Mcoeff(m),Mcoeff(div));

  t = LTfirst(Mlterm(m));
  tdiv = LTfirst(Mlterm(div));

  while( (t!=NULL) && 
         (tdiv!=NULL) && 
         (GbCmpVar(Tvar(t),Tvar(tdiv))==GbVarEq) &&
         (Texp(t)==Texp(tdiv)) )
     { t = Tnext(t); tdiv = Tnext(tdiv); }

  if( t==NULL ) return( mnew );

  while( tdiv!=NULL )
  {
    if( GbCmpVar(Tvar(t),Tvar(tdiv))==GbVarEq )
    {
      if( Texp(t)==Texp(tdiv) )
        { t = Tnext(t); tdiv = Tnext(tdiv); }

      else
      {
        if( LTfirst(Mlterm(mnew))==NULL )
        {
          LTfirst(Mlterm(mnew)) = GbTermAlloc();
          LTend(Mlterm(mnew)) = LTfirst(Mlterm(mnew));
          Tprev(LTfirst(Mlterm(mnew))) = NULL;
          Tvar(LTfirst(Mlterm(mnew))) = Tvar(t);
          Texp(LTfirst(Mlterm(mnew))) = Texp(t) - Texp(tdiv);
          t = Tnext(t);
          tdiv = Tnext(tdiv);
          tnew = LTfirst(Mlterm(mnew));
	}
        else
        {
          Tnext(tnew) = GbTermAlloc();
          Tprev(Tnext(tnew)) = tnew;
          tnew = Tnext(tnew);
          LTend(Mlterm(mnew)) = tnew;
          Tvar(tnew) = Tvar(t);
          Texp(tnew)= Texp(t) - Texp(tdiv);
          t = Tnext(t);
          tdiv = Tnext(tdiv);
        }
      }
    }
    else
    {
      if( LTfirst(Mlterm(mnew))==NULL )
      {
        LTfirst(Mlterm(mnew)) = GbTermAlloc();
        LTend(Mlterm(mnew)) = LTfirst(Mlterm(mnew));
        Tprev(LTfirst(Mlterm(mnew))) = NULL;
        Tvar(LTfirst(Mlterm(mnew))) = Tvar(t);
        Texp(LTfirst(Mlterm(mnew))) = Texp(t);
        t = Tnext(t);
        tnew = LTfirst(Mlterm(mnew));
	tdiv = Tnext(tdiv);
      }
      else
      {
        Tnext(tnew) = GbTermAlloc();
        Tprev(Tnext(tnew)) = tnew;
        tnew = Tnext(tnew);
        LTend(Mlterm(mnew)) = tnew;
        Tvar(tnew) = Tvar(t);
        Texp(tnew) = Texp(t);
        t = Tnext(t);
	tdiv = Tnext(tdiv);
      }
    }
  }

  while( t!=NULL )
  {
    if( LTfirst(Mlterm(mnew))==NULL )
    {

      LTfirst(Mlterm(mnew)) = GbTermAlloc();
      LTend(Mlterm(mnew)) = LTfirst(Mlterm(mnew));
      Tprev(LTfirst(Mlterm(mnew))) = NULL;
      Tvar(LTfirst(Mlterm(mnew))) = Tvar(t);
      Texp(LTfirst(Mlterm(mnew))) = Texp(t);
      t = Tnext(t);
      tnew = LTfirst(Mlterm(mnew));
    }
    else
    {
      Tnext(tnew) = GbTermAlloc();
      Tprev(Tnext(tnew)) = tnew;
      tnew = Tnext(tnew);
      LTend(Mlterm(mnew)) = tnew;
      Tvar(tnew) = Tvar(t);
      Texp(tnew) = Texp(t);
      t = Tnext(t);
    }
  }
  Tnext(tnew) = NULL;
  return( mnew );
}


GbMonomial *GbTdivM(GbTerm *x_gamma, GbMonomial *m)
/***************************************************************************
* To compute the division of `x_gamma' by `m' 
* `m' must divise `x_gamma'
*/
{
  GbMonomial *mnew;
  GbTerm *tnew, *tm;

  mnew = GbMonomialAlloc(NULL,1,1);
  mpq_div(Mcoeff(mnew),Mcoeff(mnew),Mcoeff(m));

  tm = LTfirst(Mlterm(m));

  while( tm!=NULL )
  {
    if( GbCmpVar(Tvar(tm),Tvar(x_gamma))==GbVarEq )
    {
      if( Texp(x_gamma) == Texp(tm) )
      {
        x_gamma = Tnext(x_gamma);
        tm = Tnext(tm);
      }
      else
      {
        if( LTfirst(Mlterm(mnew))==NULL )
        {
           LTfirst(Mlterm(mnew)) = GbTermAlloc();
           LTend(Mlterm(mnew)) = LTfirst(Mlterm(mnew));
           Tprev(LTfirst(Mlterm(mnew))) = NULL;
           Tvar(LTfirst(Mlterm(mnew))) = Tvar(x_gamma);
           Texp(LTfirst(Mlterm(mnew))) = Texp(x_gamma) - Texp(tm);
           x_gamma = Tnext(x_gamma);
           tm = Tnext(tm);
           tnew = LTfirst(Mlterm(mnew));
        }
        else
        {
           Tnext(tnew) = GbTermAlloc();
           Tprev(Tnext(tnew)) = tnew;
           tnew = Tnext(tnew);
           LTend(Mlterm(mnew)) = tnew;
           Tvar(tnew) = Tvar(x_gamma);
           Texp(tnew) = Texp(x_gamma) - Texp(tm);
           x_gamma = Tnext(x_gamma);
           tm = Tnext(tm);
        }
      }
    }
    else
    {
      if( LTfirst(Mlterm(mnew))==NULL )
      {
         LTfirst(Mlterm(mnew)) = GbTermAlloc();
         LTend(Mlterm(mnew)) = LTfirst(Mlterm(mnew));
         Tprev(LTfirst(Mlterm(mnew))) = NULL;
         Tvar(LTfirst(Mlterm(mnew))) = Tvar(x_gamma);
         Texp(LTfirst(Mlterm(mnew))) = Texp(x_gamma);
         x_gamma = Tnext(x_gamma);
         tnew = LTfirst(Mlterm(mnew));
      }
      else
      {
         Tnext(tnew) = GbTermAlloc();
         Tprev(Tnext(tnew)) = tnew;
         tnew = Tnext(tnew);
         LTend(Mlterm(mnew)) = tnew;
         Tvar(tnew) = Tvar(x_gamma);
         Texp(tnew) = Texp(x_gamma);
         x_gamma = Tnext(x_gamma);
      }
    }
  }

  while( x_gamma != NULL )
  {
    if( LTfirst(Mlterm(mnew))==NULL )
    {
       LTfirst(Mlterm(mnew)) = GbTermAlloc();
       LTend(Mlterm(mnew)) = LTfirst(Mlterm(mnew));
       Tprev(LTfirst(Mlterm(mnew))) = NULL;
       Tvar(LTfirst(Mlterm(mnew))) = Tvar(x_gamma);
       Texp(LTfirst(Mlterm(mnew))) = Texp(x_gamma);
       x_gamma = Tnext(x_gamma);
       tnew = LTfirst(Mlterm(mnew));
    }
    else
    {
       Tnext(tnew) = GbTermAlloc();
       Tprev(Tnext(tnew)) = tnew;
       tnew = Tnext(tnew);
       LTend(Mlterm(mnew)) = tnew;
       Tvar(tnew) = Tvar(x_gamma);
       Texp(tnew) = Texp(x_gamma);
       x_gamma = Tnext(x_gamma);
    }
  }

  Tnext(tnew) = NULL;
  return( mnew );
}



int GbTisdivT(GbTerm *t, GbTerm *div)
/***************************************************************************
* To test if `div' divise `t'
*/
{
  int is_div = 1;

  while( is_div && (div!=NULL) && (t!=NULL) )
  {
    while( (t!=NULL) && (GbCmpVar(Tvar(t),Tvar(div))!=GbVarEq) ) t = Tnext(t);

    if( t==NULL ) is_div = 0;
    else if( Texp(t) < Texp(div) ) is_div = 0;
    else
      { t = Tnext(t); div = Tnext(div); }
  }
  if( div!=NULL ) is_div = 0;
  return( is_div );
}


GbPoly *BsGbSpoly(long idp1, long idp2)
/***************************************************************************
* CALL in BSsolve
* To compute the S-polynomial of `p1' and `p2'
*/
{
  GbPoly *p1, *p2, *p3;
  BssCtr *bsrep;
  char *bsstring;
  void *locrep;

  locrep = LocCtrGetRep(idp1,GROEBNER_REP);
  if( locrep==NULL )
  {
    bsrep = LocCtrGetCtr(idp1);
    bsstring = Fml2wstring(CtrFormula(bsrep));
    p1 = GbParsePoly(bsstring);
    free( bsstring);
    CtrAddRep(bsrep,GROEBNER_REP,(void *)p1);
 
  }
  else
    p1 = ((GbPoly**)locrep)[0];

  locrep = LocCtrGetRep(idp2,GROEBNER_REP);
  if( locrep==NULL )
  {
    bsrep = LocCtrGetCtr(idp2);
    bsstring = Fml2wstring(CtrFormula(bsrep));
    p2 = GbParsePoly(bsstring);
    free( bsstring);
    CtrAddRep(bsrep,GROEBNER_REP,(void *)p2);
  }
  else
    p2 = ((GbPoly**)locrep)[0];

/*
  printf("  Spoly(");
  GbWriteP(stdout,p1);
  printf(" , ");
  GbWriteP(stdout,p2);
  printf(") = ");
*/
  p3 = GbSpoly(p1,p2);
/*
  GbWriteP(stdout,p3);
  printf("\n");
*/
  return( p3 );
}


GbPoly *GbSpoly(GbPoly *p1, GbPoly *p2)
/***************************************************************************
* To compute the S-polynomial of `p1' and `p2'
*/
{
  GbTerm *x_gamma;
  GbMonomial *m_p1, *m_p2;
  GbPoly *pr1, *pr2, *paux;

  x_gamma = GbLcmMonomial(Pmono(p1),Pmono(p2));
  m_p1 = GbTdivM(x_gamma,Pmono(p1));
  m_p2 = GbTdivM(x_gamma,Pmono(p2));
  mpq_neg(Mcoeff(m_p2),Mcoeff(m_p2));

  pr1 = GbPmulM(p1,m_p1);
  pr2 = GbPmulM(p2,m_p2);
  pr1 = GbPfusionP(pr1,pr2);

  GbFreeM(m_p1); GbFreeM(m_p2); GbFreeT(x_gamma);

  return( pr1 );
}


void GbReduce(GbPoly **p1, GbPoly *p2)
/***************************************************************************
* To reduce `p1' with respect to `p2'
*/
{
  GbPoly *pr, *p3, *p4;
  GbMonomial *m;
/*
  printf("Reduce\n\t");
  GbWriteP(stdout,*p1);
  printf("\n   wrt \n\t");
  GbWriteP(stdout,p2);
  printf("\n");
*/
  if( GbTisdivT(LTfirst(Mlterm(Pmono(*p1))),LTfirst(Mlterm(Pmono(p2)))) )
  {
    m = GbMdivM(Pmono(*p1),Pmono(p2));

    mpq_neg(Mcoeff(m),Mcoeff(m));
    pr = GbPmulM(p2,m);

    /* The leading term of p1 is deleted using the leading term of pr */
    p3 = *p1;
    *p1 = Pnext(*p1);
    p4 = pr;
    pr = Pnext(pr);
/*
  printf("AvantFusion : ");
  GbWriteP(stdout,*p1);
  printf("\n");
*/
    *p1 = GbPfusionP(*p1,pr);


    if( *p1==NULL )
    {
      *p1 = GbGetPolyZero();
    }
/* 
  printf("ApresFusion : ");
  GbWriteP(stdout,*p1);
  printf("\n");
  */

    GbFreeM(m);
    GbFreeM(Pmono(p3));
    free(p3);
    GbFreeM(Pmono(p4));
    free(p4);
  }
/*
  printf("ApresReduce = ");
  GbWriteP(stdout,*p1);
  printf("\n"); 
*/
}


void BsGbReduce(long idp1, long idp2)
/***************************************************************************
* To reduce `p1' with respect to `p2'
*/
{
  GbPoly *p1, *p2;
  BssCtr *bsrep;
  char *bsstring;
  void *locrep;

  locrep = LocCtrGetRep(idp1,GROEBNER_REP);
  if( locrep==NULL )
  {
    bsrep = LocCtrGetCtr(idp1);
    bsstring = Fml2wstring(CtrFormula(bsrep));
    p1 = GbParsePoly(bsstring);
    CtrAddRep(bsrep,GROEBNER_REP,(void *)p1);
 
  }
  else
    p1 = ((GbPoly**)locrep)[0];

  locrep = LocCtrGetRep(idp2,GROEBNER_REP);
  if( locrep==NULL )
  {
    bsrep = LocCtrGetCtr(idp2);
    bsstring = Fml2wstring(CtrFormula(bsrep));
    p2 = GbParsePoly(bsstring);
    CtrAddRep(bsrep,GROEBNER_REP,(void *)p2);
  }
  else
    p2 = ((GbPoly**)locrep)[0];

  GbReduce(&p1,p2);
}



/**********************************************************************************
*                            INTERFACE with BSSOLVE
***********************************************************************************/


long GbExprLong(char *s, int *pos)
{
  int i, j;
  long n;
  char strnum[60];

  j = 0;
  i = *pos;
  while( s[i]!=')' )
  {
    strnum[j] = s[i];
    j++; i++;
  }
  strnum[j] = '\0';
  *pos = i;
  return( GbStrToLong(strnum) );
}


int GbExprInt(char *s, int *pos)
{
  int n, i, j;
  char strnum[40];

  j = 0;
  i = *pos;
  while( s[i]!=')' )
  {
    strnum[j] = s[i];
    j++; i++;
  }
  strnum[j] = '\0';
  *pos = i;
  return( GbStrToInt(strnum) );
}


void GbFloatToRat(mpq_t q, char *s, int *pos)
{
  int k, j, i, ed, ef, exp_sgn, exponent = 1;
  char sresult[100], sexp[5], snum[60];
  mpz_t num;
  mpz_t den;
       
  j = 0;
  i = *pos;
  while( s[i]!=')' )
  {
    snum[j] = s[i];
    j++; i++;
  }
  snum[j] = '\0';
  *pos = i;


  i=0;
  mpz_init(den);
  while( (snum[i]!='\0') && (snum[i]!='.') ) i++;
       
  if( snum[i]!='.' )   /* snum contains an integer */
  {
    if( snum[0]=='+' ) strcpy(snum,&snum[1]);
    mpz_init_set_str(num,snum,10);
    mpz_set_ui(den,1);
    mpq_set_num(q,num);
    mpq_set_den(q,den);
  }
  else                /* snum contains a float */
  {
    j = i+1;
    while( (snum[j]!='\0') && (snum[j]!='e') && (snum[j]!='E') ) j++;
     
    if( snum[j]!='\0' )  /* there exists an exponent */
    {
      if( snum[j+1]=='-' )
      {
        exp_sgn = -1;
        ed = j+2;
      }
      else if( snum[j+1]=='+' )
      {
        exp_sgn = 1;
        ed = j+2;
      }
      else
      {
        exp_sgn = 1;
        ed = j+1;
      }
      ef = ed+1;
      while( snum[ef]!='\0' ) ef++;
      
      strncpy(sexp,&snum[ed],ef-ed+1);

      exponent = atoi(sexp);
      exponent *= exp_sgn;
    }  
    else
    {
      j += 1;
    }
    strncpy(sresult,snum,i);
    strncpy(&sresult[i],&snum[i+1],j-i-1);
    sresult[j-1] = '\0';
       
    mpz_init_set_str(num,sresult,10);
   
    mpz_set_ui(den,1);
    for(k=0; k<j-i-1; k++) mpz_mul_ui(den,den,10);

    if( exponent<0 )   
    {
      exponent *= -1;
      for( k=0; k<exponent; k++ ) mpz_mul_ui(den,den,10);
   
      mpq_set_num(q,num);
      mpq_set_den(q,den);
    }
    else
    {
      for( k=0; k<exponent; k++ ) mpz_mul_ui(num,num,10);
       
      mpq_set_num(q,num);
      mpq_set_den(q,den);
    }  
  }
  mpq_canonicalize(q); mpz_clear(num); mpz_clear(den);
}


void GbGetRat(mpq_t q, char *s, int *pos)
{
  char snum[100], sden[100];
  int i,j;
  mpz_t num;
  mpz_t den;

  j = 0;
  i = *pos;
  while( s[i]!=':' )
  {
    snum[j] = s[i];
    j++; i++;
  }
  snum[j] = '\0';
  *pos = i+1;

  j = 0;
  i = *pos;
  while( s[i]!=')' )
  {
    sden[j] = s[i];
    j++; i++;
  }
  sden[j] = '\0';
  *pos = i;

  if( snum[0]=='+' ) strcpy(snum,&snum[1]);

  mpz_init_set_str(num,snum,10);
  mpz_init_set_str(den,sden,10);

  mpq_set_num(q,num);
  mpq_set_den(q,den);

  mpq_canonicalize(q); mpz_clear(num); mpz_clear(den);
}


void GbGetIdent(char *id, char *s, int *pos)
{
  int i=0;

  while( (isalpha(s[*pos])) && (s[*pos]!='\0') )
  {
    id[i] = s[*pos];
    (*pos)++;
    i++;
  }
  id[i] = '\0';
}


int GbGetToken(char *s, int *pos)
{
  char c;
  char id[40];

  c = s[*pos];  (*pos)++;

  if ( c=='o') 
    {
      (*pos)++; /* o( */
      c = s[*pos]; (*pos)++;
      if( c=='+' ) return( GbADD );
      else if( c=='-' ) return( GbSUB );
      else if( c=='*' ) return( GbMUL );
      else
	{
	  (*pos)--;
	  GbGetIdent(id,s,pos);
	  if( strcmp(id,"pow")==0 ) return( GbPOW );
	  else if( strcmp(id,"neg")==0 ) return( GbNEG );
	  else if( strcmp(id,"sqr")==0 ) return( GbSQR );
	}
    }
  else if( c == 'i' ) return( GbInt );
  else if( c == 'f' ) return( GbFloat );
  else if( c == 'q' ) return( GbRat );
  else if( c == 'v' ) return( GbVAR );
}



GbMonomial *GbExprMonomial(char *s, int *pos)
{
  long var;
  GbMonomial *m, *m1, *m2;
  int j, n = GbGetToken(s,pos);

  if( n==GbSQR )
  {
    (*pos)+=3;  /* (v( */
    var = GbExprLong(s,pos);
    (*pos)+=3;  /* ))) */
    m = GbMonomialAlloc(NULL,1,1);
    Mlterm(m) = (GbListTerm *)malloc(sizeof(GbListTerm));
    LTfirst(Mlterm(m)) = GbTermAlloc();
    Tvar(LTfirst(Mlterm(m))) = var;
    Texp(LTfirst(Mlterm(m))) = 2;
    Tprev(LTfirst(Mlterm(m))) = Tnext(LTfirst(Mlterm(m))) = NULL;
    LTend(Mlterm(m)) = LTfirst(Mlterm(m));

    return( m );
  }
  else if( n==GbPOW )
  {
    (*pos)+=3;  /* (v( */
    var = GbExprLong(s,pos);
    (*pos)+=4;  /* ),i( */
    j = GbExprInt(s,pos);
    (*pos)+=3;  /* ))) */

    m = GbMonomialAlloc(NULL,1,1);
    Mlterm(m) = (GbListTerm *)malloc(sizeof(GbListTerm));
    LTfirst(Mlterm(m)) = GbTermAlloc();
    Tvar(LTfirst(Mlterm(m))) = var;
    Texp(LTfirst(Mlterm(m))) = j;
    Tprev(LTfirst(Mlterm(m))) = Tnext(LTfirst(Mlterm(m))) = NULL;
    LTend(Mlterm(m)) = LTfirst(Mlterm(m));

    return( m );
  }
  else if( n==GbMUL )
  {
    (*pos)++;   /* ( */
    m1 = GbExprMonomial(s,pos);
    (*pos)++;   /* , */
    m2 = GbExprMonomial(s,pos);
    (*pos)+=2;   /* )) */
    return( GbMfusionM(m1,m2) );
  }
  else if( n==GbRat )
  {
    (*pos)++;  /* ( */
    m = GbMonomialAlloc(NULL,1,1);
    GbGetRat(Mcoeff(m),s,pos);
    (*pos)++;  /* ) */

    return( m );    
  }
  else if( n==GbInt )
  {
    (*pos)++;  /* ( */
    j = GbExprInt(s,pos);
    (*pos)++;  /* ) */

    return( GbMonomialAlloc(NULL,j,1) );
  }
  else if( n==GbFloat )
  {
    (*pos)++;  /* ( */
    m = GbMonomialAlloc(NULL,1,1);
    GbFloatToRat(Mcoeff(m),s,pos);
    (*pos)++;  /* ) */

    return( m );
  }
  else  /* variable */
  {
    (*pos)++;  /* ( */
    var = GbExprLong(s,pos);
    (*pos)++;  /* ) */

    m = GbMonomialAlloc(NULL,1,1);
    Mlterm(m) = (GbListTerm *)malloc(sizeof(GbListTerm));
    LTfirst(Mlterm(m)) = GbTermAlloc();
    Tvar(LTfirst(Mlterm(m))) = var;
    Texp(LTfirst(Mlterm(m))) = 1;
    Tprev(LTfirst(Mlterm(m))) = Tnext(LTfirst(Mlterm(m))) = NULL;
    LTend(Mlterm(m)) = LTfirst(Mlterm(m));

    return( m );
  }
}



GbPoly *GbExprPoly(char *s, int *pos)
/***************************************************************************
*  Parsing of the polynomial represented by the wrapped string in s
*/
{
  GbPoly *p1, *p2, *aux;
  GbMonomial *m;
  int nsave = *pos;

  int n = GbGetToken(s,pos);

  if( n==GbADD )
  {
    (*pos)++;   /* ( */
    p1 = GbExprPoly(s,pos);
    (*pos)++;   /* , */
    p2 = GbExprPoly(s,pos);
    (*pos)+=2;   /* )) */
    return( GbPfusionP(p1,p2) );
  }
  else if( n==GbSUB )
  {
    (*pos)++;   /* ( */
    p1 = GbExprPoly(s,pos);
    (*pos)++;   /* , */
    p2 = GbExprPoly(s,pos);
    (*pos)+=2;   /* )) */

    aux = p2;
    while( aux!=NULL )
    {
      mpq_neg(Mcoeff(Pmono(aux)),Mcoeff(Pmono(aux)));
      aux = Pnext(aux);
    }
    return( GbPfusionP(p1,p2) );
  }
  else  if( n==GbNEG )
  {
    (*pos)++;   /* ( */
    p1 = GbExprPoly(s,pos);
    (*pos)+=2;   /* )) */

    aux = p1;
    while( aux!=NULL )
    {
      mpq_neg(Mcoeff(Pmono(aux)),Mcoeff(Pmono(aux)));
      aux = Pnext(aux);
    }
    return( p1 );
  }
  else
  {
    *pos = nsave;
    m = GbExprMonomial(s,pos);
    p1 = (GbPoly *)malloc(sizeof(GbPoly));
    Pnext(p1) = NULL;
    Pmono(p1) = m;
    return( p1 );
  }
}


GbPoly *GbParsePoly(char * s)
/***************************************************************************
*  Parsing of the polynomial represented by the wrapped string in s
*/
{
  GbPoly *p1, *p2, *aux;
  int pos = 4;    /* o(=( */

  p1 = GbExprPoly(s,&pos);
  pos++;          /* , */

  p2 = GbExprPoly(s,&pos);
  pos+=2;          /* )) */

  if( GbPolyZero(p1) )
  {
    if( GbPolyZero(p2) )  /*  0 = 0 */
    {
      GbFreeP(p1);
      GbFreeP(p2);
      return( NULL );
    }
    else                  /* 0 = p2 */
    {
      GbFreeP(p1);
      return( p2 );
    }
  }
  else
  {
    if( GbPolyZero(p2) ) /* p1 = 0 */
    {
      GbFreeP(p2);
      return( p1 );
    }
    else                 /* p1 = p2 */
    {
      aux = p2;
      while( aux!=NULL )
      {
        mpq_neg(Mcoeff(Pmono(aux)),Mcoeff(Pmono(aux)));
        aux = Pnext(aux);
      }

      return( GbPfusionP(p1,p2) );
    }
  }
}


char *GbOneTerm2wstring(GbTerm *t)
{
  char s1[60], *svar, *s2, *s;
  int n;
 
  sprintf(s1,"%ld",Tvar(t));

  n = strlen(s1) + 4;
  svar = (char *)malloc(n*sizeof(char));
  svar[0] = 'v';
  svar[1] = '(';
  strcpy(&(svar[2]),s1);
  svar[n-2] = ')';
  svar[n-1] = '\0';

  if( Texp(t)==1 ) return( svar );
  else if( Texp(t)==2 )
  {
    n = strlen(svar) + 9;
    s2 = (char *)malloc(n*sizeof(char));
    s2[0] = 'o';
    s2[1] = '(';
    s2[2] = 's';
    s2[3] = 'q';
    s2[4] = 'r';
    s2[5] = '(';
    strcpy(&(s2[6]),svar);
    s2[n-3] = ')';
    s2[n-2] = ')';
    s2[n-1] = '\0';
    free(svar);
    return( s2 );
  }
  else
  {
    sprintf(s1,"%d",Texp(t));

    n = strlen(s1) + 4;
    s2 = (char *)malloc(n*sizeof(char));
    s2[0] = 'i';
    s2[1] = '(';
    strcpy(&(s2[2]),s1);
    s2[n-2] = ')';
    s2[n-1] = '\0';

    n = strlen(svar) + strlen(s2) + 10;
    s = (char *)malloc(n*sizeof(char));
    s[0] = 'o';
    s[1] = '(';
    s[2] = 'p';
    s[3] = 'o';
    s[4] = 'w';
    s[5] = '(';
    strcpy(&(s[6]),svar);
    s[strlen(svar)+6] = ',';
    strcpy(&(s[strlen(svar)+7]),s2);
    s[n-3] = ')';
    s[n-2] = ')';
    s[n-1] = '\0';

    free(s2); free(svar);
    return( s );
  }
}

char *GbNonZeroTerm2wstring(GbTerm *t)
{
  char *s1, *s2, *s;
  int n;

  if( t==NULL ) return( NULL );

  s1 = GbOneTerm2wstring(t);
  s2 = GbNonZeroTerm2wstring(Tnext(t));

  if( s2==NULL ) return( s1 );
  else
  {
    n = strlen(s1)+strlen(s2)+8;
    s = (char *)malloc(n*sizeof(char));
    s[0] = 'o';
    s[1] = '(';
    s[2] = '*';
    s[3] = '(';
    strcpy(&(s[4]),s1);
    s[strlen(s1)+4] = ',';
    strcpy(&(s[strlen(s1)+5]),s2);
    s[n-3] = ')';
    s[n-2] = ')';
    s[n-1] = '\0';

    free(s1); free(s2);
    return( s );
  }
}


char *GbTerm2wstring(GbListTerm *l)
{
  GbTerm *t;
  if( l==NULL ) return( NULL );
  if( LTfirst(l)==NULL ) return( NULL );

  return( GbNonZeroTerm2wstring(LTfirst(l)) );
}


char *GbRat2wstring(mpq_t q)
{
  char *s, *snum, *sden;
  int n,m,size;
  mpz_t num, den;
  mpz_init(num);
  mpz_init(den);

  mpq_get_num(num,q); mpq_get_den(den,q);

  n = mpz_sizeinbase(num,10)+2;
  m = mpz_sizeinbase(den,10)+2;

  snum = (char *)malloc(n*sizeof(char));
  sden = (char *)malloc(m*sizeof(char));

  mpz_get_str(snum,10,num);
  mpz_get_str(sden,10,den);

  if( mpz_cmp_ui(den,1)==0 )  /* integer */
  {
    size = strlen(snum)  + 4;
    s = (char *)malloc(size*sizeof(char));

    s[0] = 'i';
    s[1] = '(';
    strcpy(&(s[2]),snum);
    s[size-2] = ')';
    s[size-1] = '\0';

    free(snum); free(sden);
    mpz_clear(num); mpz_clear(den);
    return( s );
  }

  size = strlen(snum) + strlen(sden) + 5;
  s = (char *)malloc(size*sizeof(char));

  s[0] = 'q';
  s[1] = '(';
  strcpy(&(s[2]),snum);
  s[strlen(snum)+2] = ':';
  strcpy(&(s[strlen(snum)+3]),sden);
  s[size-2] = ')';
  s[size-1] = '\0';

  free(snum); free(sden);
  mpz_clear(num); mpz_clear(den);
  return( s );
}


char *GbMonomial2wstring(GbMonomial *m)
{
  char *s1, *s2, *s;
  int n;

  s1 = GbRat2wstring(Mcoeff(m));
  s2 = GbTerm2wstring(Mlterm(m));

  if( strcmp(s1,"i(1)")==0 )
  {
    if( s2==NULL ) return( s1 );
    else return( s2 );
  }
  if( strcmp(s1,"i(-1)")==0 )
  {
    if( s2==NULL ) return( s1 );
  }

  if( s2==NULL ) return( s1 );
  else
  {
    n = strlen(s1)+strlen(s2)+8;
    s = (char *)malloc(n*sizeof(char));
    s[0] = 'o';
    s[1] = '(';
    s[2] = '*';
    s[3] = '(';
    strcpy(&(s[4]),s1);
    s[strlen(s1)+4] = ',';
    strcpy(&(s[strlen(s1)+5]),s2);
    s[n-3] = ')';
    s[n-2] = ')';
    s[n-1] = '\0';

    free(s1); free(s2);

/*    
printf("  Monome:%s\n",s);
*/

    return( s );
  }
}


char *GbPolyIter2wstring(GbPoly *p)
{
  GbPoly *p1;
  GbTerm *t;
  char *s1, *s2, *s;
  int n;


  if( p==NULL ) return( NULL );

  s1 = GbMonomial2wstring(Pmono(p));
  s2 = GbPolyIter2wstring(Pnext(p));

  if( s2==NULL ) return( s1 );
  else /* +(s1,s2) */
  {
    n = strlen(s1)+strlen(s2)+8;
    s = (char *)malloc(n*sizeof(char));
    s[0] = 'o';
    s[1] = '(';
    s[2] = '+';
    s[3] = '(';
    strcpy(&(s[4]),s1);
    s[strlen(s1)+4] = ',';
    strcpy(&(s[strlen(s1)+5]),s2);
    s[n-3] = ')';
    s[n-2] = ')';
    s[n-1] = '\0';

    free(s1); free(s2);
    return( s );
  }
}


char *GbPoly2wstring(GbPoly *p)
{
  char *s1, *s;
  int n;

  s1 = GbPolyIter2wstring(p);
  if( s1==NULL ) return( NULL );

  n = strlen(s1) + 12;
  s = (char *)malloc(n*sizeof(char));

  s[0] = 'o';
  s[1] = '(';
  s[2] = '=';
  s[3] = '(';
  strcpy(&(s[4]),s1);
  s[strlen(s1)+4] = ',';
  s[n-7] = 'i';
  s[n-6] = '(';
  s[n-5] = '0';
  s[n-4] = ')';
  s[n-3] = ')';
  s[n-2] = ')';
  s[n-1] = '\0';

  free(s1);
  return( s );
}
