/***************************************************************************/
/* rational.h                                                               */
/*                                                                         */
/* Description file for operations on rationals                             */
/*                                                                         */
/* Rateger  Arithmetic Library                                             */
/*									   */
/***************************************************************************/

/*************************************************************************/
/*                                                                       */
/*       Copyright (C) 1998 Universite de Nantes                         */
/*                                                                       */
/*************************************************************************/
 
/************* 
 Edit history: 
       27/2/98 - Nicolas Romero : Creation
*************/

#ifndef RATIONAL_H
#define RATIONAL_H

#include <stdio.h>
#include <stdlib.h>
/* #include <floatingpoint.h> */
#include <gmp.h>

#define RatVal( i) ((mpq_ptr)i)
#define RatValNum( i) (mpz_get_si(mpq_numref(i)))
#define RatValDen( i) (mpz_get_ui(mpq_denref(i)))
#define Rat2double( i)  (mpq_get_d( i))

#define RAT_BSTRING_SIZE (sizeof( signed long) + sizeof( unsigned long))
/*-------------------------------------------------------------------------*/
/* BSS RATIONAL                                                             */
/*-------------------------------------------------------------------------*/

typedef  mpq_t BssRatType;
typedef  BssRatType BssRat;
typedef  BssRatType *BssRatP;

BssRatP RatAlloc();

#define RatInit(r) mpq_init( r)

BssRatP RatAllocCopy( const BssRat i);
BssRatP RatAllocInit(signed long int n, unsigned long int d);

#define  RatSet(i, n, d)\
     mpq_set_si( RatVal(i), n, d), mpq_canonicalize( RatVal(i))
#define  RatCopy(i1, i2)\
     mpq_set(RatVal(i1),RatVal(i2))
#define RatFree(i)\
     mpq_clear(i)

#define RatAdd( r, i1, i2) \
     (mpq_add( RatVal(r), RatVal(i1), RatVal(i2)), \
      mpq_canonicalize( RatVal(r)))
#define RatSub( r, i1, i2) \
     (mpq_sub( RatVal(r), RatVal(i1), RatVal(i2)), \
      mpq_canonicalize( RatVal(r)))
#define RatMinus( r, i1) \
     (mpq_neg( RatVal(r), RatVal(i1)))
#define RatMul( r, i1, i2) \
     (mpq_mul( RatVal(r), RatVal(i1), RatVal(i2)), \
      mpq_canonicalize( RatVal(r)))
#define RatDiv( r, i1, i2) \
     (mpq_div( RatVal(r), RatVal(i1), RatVal(i2)), \
      mpq_canonicalize( RatVal(r)))
#define RatSqrt( r, i1) \
     (((mpz_sgn( mpq_numref( RatVal(i1))) >= 0) \
      ? (mpz_sqrt( mpq_numref( RatVal( r)), mpq_numref( RatVal( i1))), \
	 mpz_sqrt( mpq_denref( RatVal( r)), mpq_denref( RatVal( i1)))) \
      :0), \
      mpq_canonicalize( RatVal(r)))
#define RatAbs( r, i1) \
     ((mpq_sgn( RatVal(i1)) < 0) \
      ? mpq_neg( RatVal( r), RatVal(i1)) \
      : mpq_set( RatVal(r), RatVal(i1)))
#define RatSqr( r, i1) \
     (mpq_mul( RatVal(r), RatVal(i1), RatVal(i1)), \
      mpq_canonicalize( RatVal(r)))
void RatPow(BssRat r, const BssRat x, int n);

#define  RatInter( i, i1, i2)\
     ((!mpq_cmp(RatVal( i1), RatVal(i2)))\
     ? (RatCopy( i, i1),1)\
     : 0)
     
void RatWrite(FILE *out, const BssRat i, int digits);

char* Rat2bstring( const BssRat v, unsigned int *size);
BssRatP bstring2Rat( const char *s);

extern BssRat *klic2Rat( void *);
extern void *Rat2klic( const BssRat );

char *Rat2wstring( const BssRat v);
BssRatP wstring2Rat( const char *s);
     
#define RatCmp( i1, i2, pcmp) \
       (*pcmp=1, \
	mpq_cmp(RatVal( i1), RatVal(i2)))

#endif   
