/*************************************************************************/
/*                                                                       */
/*       Copyright (C) 1998 Universite de Nantes                         */
/*                                                                       */
/*************************************************************************/
 
/************* 
 Edit history: 
       27/2/98 - Nicolas Romero : Creation
*************/

#include "interval.h"
#include <stdlib.h>
#include <string.h>
#include "value.h"
#include "float.h"

#define LIBERE

int _itv_cmp;

static char *ExStr;  /* For Res_Exact and ClearExceptions */

#ifdef _LINUX_
#  define ClearExceptions
#define Res_Exact (1) 
#else
#  define ClearExceptions  ieee_f("clear","exception","all",&ExStr)
#  define Res_Exact        \
     (!(ieee_f("get","exception","all",&ExStr) & (1<<fp_inexact)))
#endif

#define BracketL2LL(r1,r2) (((!Res_Exact) ||                  \
                            (ItvIsLeftOpen(r1)) ||               \
                            (ItvIsLeftOpen(r2)))                 \
                              ? LEFT_OPEN                \
			      : LEFT_CLOSED)

#define BracketR2LL(r1,r2) (((!Res_Exact) ||                  \
                            (ItvIsLeftOpen(r1)) ||               \
                            (ItvIsLeftOpen(r2)))                 \
                              ? RIGHT_OPEN                  \
			      : RIGHT_CLOSED)

#define BracketL2LR(r1,r2)   (((!Res_Exact) ||                \
                              (ItvIsLeftOpen(r1)) ||             \
                              (ItvIsRightOpen(r2)))              \
                              ? LEFT_OPEN                   \
			      : LEFT_CLOSED)

#define BracketR2LR(r1,r2)   (((!Res_Exact) ||                \
                              (ItvIsLeftOpen(r1)) ||             \
                              (ItvIsRightOpen(r2)))              \
                              ? RIGHT_OPEN                  \
			      : RIGHT_CLOSED)

#define BracketL2RL(r1,r2)   BracketL2LR(r2,r1)
#define BracketR2RL(r1,r2)   BracketR2LR(r2,r1)

#define BracketL2RR(r1,r2)   (((!Res_Exact) ||                \
                              (ItvIsRightOpen(r1)) ||            \
                              (ItvIsRightOpen(r2)))              \
                              ? LEFT_OPEN                   \
			      : LEFT_CLOSED)

#define BracketR2RR(r1,r2)   (((!Res_Exact) ||                \
                              (ItvIsRightOpen(r1)) ||            \
                              (ItvIsRightOpen(r2)))              \
                              ? RIGHT_OPEN                  \
			      : RIGHT_CLOSED)

#define BracketL2LLDM(R,r1,r2)                                         \
((BssCmp( ItvMinRP(R), zero, &cmp) == 0)                                       \
 ? ((((BssCmp( ItvMinRP(r1), zero, &cmp) == 0) && ItvIsLeftClosed(r1)) ||  \
     ((BssCmp(ItvMinRP(r2), zero, &cmp) ==0) && ItvIsLeftClosed(r2)))    \
    ? LEFT_CLOSED                        \
    : LEFT_OPEN)                       \
 : BracketL2LL(r1,r2))      
     
#define BracketL2LRDM(R,r1,r2)                                         \
     ((BssCmp( ItvMinRP(R), zero, &cmp)==0)                                       \
      ? ((((BssCmp( ItvMinRP(r1), zero, &cmp)==0) && ItvIsLeftClosed(r1)) ||  \
	  ((BssCmp( ItvMaxRP(r2), zero, &cmp)==0) && ItvIsRightClosed(r2)))   \
	 ? LEFT_CLOSED                        \
	 : LEFT_OPEN)                       \
      : BracketL2LR(r1,r2))      

#define BracketL2RLDM(R,r1,r2)  BracketL2LRDM(R,r2,r1)

#define BracketL2RRDM(R,r1,r2)                                         \
     ((BssCmp( ItvMinRP(R), zero, &cmp)==0)                                       \
      ? (((BssCmp( ItvMaxRP(r1), zero, &cmp)==0 && ItvIsRightClosed(r1)) || \
	  (BssCmp( ItvMaxRP(r2), zero, &cmp)==0 && ItvIsRightClosed(r2)))   \
	 ? LEFT_CLOSED                        \
	 : LEFT_OPEN)                       \
      : BracketL2RR(r1,r2))      
     
#define BracketR2LLDM(R,r1,r2)                                         \
     ((BssCmp( ItvMaxRP(R), zero, &cmp)==0)                                       \
      ? (((BssCmp( ItvMinRP(r1), zero, &cmp)==0 && ItvIsLeftClosed(r1)) ||  \
	  (BssCmp( ItvMinRP(r2), zero, &cmp)==0 && ItvIsLeftClosed(r2)))    \
	 ? RIGHT_CLOSED                       \
	 : RIGHT_OPEN)                      \
      : BracketR2LL(r1,r2))      

#define BracketR2LRDM(R,r1,r2)                                         \
     ((BssCmp( ItvMaxRP(R), zero, &cmp)==0)                                       \
      ? (((BssCmp( ItvMinRP(r1), zero, &cmp)==0 && ItvIsLeftClosed(r1)) ||  \
	  (BssCmp( ItvMaxRP(r2), zero, &cmp)==0 && ItvIsRightClosed(r2)))   \
	 ? RIGHT_CLOSED                       \
	 : RIGHT_OPEN)                      \
      : BracketR2LR(r1,r2))    
  
#define BracketR2RLDM(R,r1,r2)  BracketR2LRDM(R,r2,r1)

#define BracketR2RRDM(R,r1,r2)                                         \
     ((BssCmp( ItvMaxRP(R), zero, &cmp)==0)                                       \
      ? (((BssCmp( ItvMaxRP(r1), zero, &cmp)==0 && ItvIsRightClosed(r1)) || \
	  (BssCmp( ItvMaxRP(r2), zero, &cmp)==0 && ItvIsRightClosed(r2)))   \
	 ? RIGHT_CLOSED                       \
	 : RIGHT_OPEN)                      \
      : BracketR2RR(r1,r2))      



#define BracketL2LRDM2(D,r1,r2)                                        \
     ((BssCmp(D, zero, &cmp)==0)                                       \
      ? (((BssCmp( ItvMinRP(r1), zero, &cmp)==0 && ItvIsLeftClosed(r1)) ||  \
	  (BssCmp( ItvMaxRP(r2), zero, &cmp)==0 && ItvIsRightClosed(r2)))   \
	 ? LEFT_CLOSED                        \
	 : LEFT_OPEN)                       \
      : BracketL2LR(r1,r2))      

#define BracketR2LLDM2(D,r1,r2)                                        \
     ((BssCmp( D, zero, &cmp)==0)                                       \
      ? (((BssCmp( ItvMinRP(r1), zero, &cmp)==0 && ItvIsLeftClosed(r1)) ||  \
	  (BssCmp( ItvMinRP(r2), zero, &cmp)==0 && ItvIsLeftClosed(r2)))    \
	 ? RIGHT_CLOSED                       \
	 : RIGHT_OPEN)                      \
      : BracketR2LL(r1,r2))      

#define BracketR2RRDM2(D,r1,r2)                                        \
     ((BssCmp( D, zero, &cmp)==0)                                       \
      ? (((BssCmp( ItvMaxRP(r1), zero, &cmp)==0 && ItvIsRightClosed(r1)) || \
	  (BssCmp( ItvMaxRP(r2), zero, &cmp)==0 && ItvIsRightClosed(r2)))   \
	 ? RIGHT_CLOSED                       \
	 : RIGHT_OPEN)                      \
      : BracketR2RR(r1,r2))      

#define BracketL2RLDM2(D,r1,r2)  BracketL2LRDM2(D,r2,r1)

void ItvSetLeftBracket(BssItv r, char b)
{
  (ItvBracketsRP( r) = (ItvBracketsRP( r) & MASK_LEFT) | (b));
}

void ItvSetRightBracket(BssItv r, char b)
{
  ((ItvBracketsRP( r)) = (((ItvBracketsRP( r)) & MASK_RIGHT) | (b)));
}

void ItvInit(BssItv r)
{
  
  ItvMinRP(r) = BssAlloc();
  ItvMaxRP( r) = BssAlloc();
}

BssItvP ItvAlloc()
{
  BssItvP r;
  r = ((BssItvP)malloc(sizeof(BssItvType)));
  ItvMinRP(r)= BssAlloc();
  ItvMaxRP(r)= BssAlloc();
  ItvInit(r);
  return r;
}


/*-------------------------------------------------------------------------*/
/* ItvAllocCopy                                                       */
/*-------------------------------------------------------------------------*/
BssItvP ItvAllocCopy(const BssItv r)
{
  BssItvP rn;
  rn = ItvAlloc();/*(BssItvP)malloc(sizeof(BssItvType));*/
  BssCopy( ItvMinRP(rn), ItvMinRP(r)); 
  BssCopy(ItvMaxRP(rn),ItvMaxRP(r));
  ItvCopyBrackets(rn, r);
  return( rn );
}


/*-------------------------------------------------------------------------*/
/* ItvAllocInit                                                       */
/*-------------------------------------------------------------------------*/
BssItvP ItvAllocInit(const BssValue min, const BssValue max,
			    const char leftclosed, const char rightclosed)
{
  BssItvP rn;
  rn = ItvAlloc();/*(BssItvP )malloc(sizeof(BssItvType));*/
  BssCopy( ItvMinRP(rn),min);
  BssCopy(ItvMaxRP(rn),max);
  if (leftclosed)
    ItvSetLeftBracketClosed(rn);
  else
    ItvSetLeftBracketOpen(rn);
  if (rightclosed)
    ItvSetRightBracketClosed(rn);
  else
    ItvSetRightBracketOpen(rn);
  return( rn );
}


/*-------------------------------------------------------------------------*/
/* ItvCopy                                                            */
/*-------------------------------------------------------------------------*/
void ItvCopy(BssItv r, const BssItv r1)
{
  BssCopy(ItvMinRP(r), ItvMinRP(r1));
  BssCopy(ItvMaxRP(r), ItvMaxRP(r1));
  ItvCopyBrackets(r, r1);  
}

/*-------------------------------------------------------------------------*/
/* ItvSet                                                                  */
/*-------------------------------------------------------------------------*/
void ItvSet(BssItv r, const BssValue min, const BssValue max,
	    const char leftclosed, const char rightclosed)
{
  BssCopy( ItvMinRP(r), min);
  BssCopy( ItvMaxRP(r) ,max);
  if (leftclosed)
    ItvSetLeftBracketClosed(r);
  else
    ItvSetLeftBracketOpen(r);
  if (rightclosed)
    ItvSetRightBracketClosed(r);
  else
    ItvSetRightBracketOpen(r);
}


/*-------------------------------------------------------------------------*/
/* ItvFree                                                            */
/*-------------------------------------------------------------------------*/
void ItvFree(BssItvP r)
{
  BssFreeValue( ItvMinRP(r));
  BssFreeValue( ItvMaxRP(r));
  free(r);
}


/*-------------------------------------------------------------------------*/
/* ItvSetMaxRange                                                          */
/*-------------------------------------------------------------------------*/
void ItvSetMaxRange(BssItv r)
{
  ItvSetBracketsOpen( r);
  BssSet( ItvMinRP(r), Minus_infinity);
  BssSet( ItvMaxRP(r), Plus_infinity);
}

/*-------------------------------------------------------------------------*/
/* ItvSetZeroRange                                                          */
/*-------------------------------------------------------------------------*/
void ItvSetZeroRange(BssItv r)
{
  ItvSetBracketsClosed( r);
  BssSet( ItvMinRP(r), Integer, 0);
  BssSet( ItvMaxRP(r), Integer, 0);
}

/*-------------------------------------------------------------------------*/
/* ItvIsZeroRange                                                          */
/*-------------------------------------------------------------------------*/
int ItvIsZeroRange(BssItv r)
{
  int cmp;
  BssValue zero;

  BssInit(zero);

  BssSet( zero, Integer, 0); 
  
  return ItvIsClosed(r) && (BssCmp(ItvMinRP(r),zero, &cmp)==0)
                        && (BssCmp(ItvMaxRP(r),zero, &cmp)==0);
}

/*-------------------------------------------------------------------------*/
/* ItvAdd                                                                  */
/*-------------------------------------------------------------------------*/
void ItvAdd( BssItv r, const BssItv r1, const BssItv r2)
{
  if( ItvIsEmptyRange(r1) || ItvIsEmptyRange(r2) )
  {
    ItvSetEmptyRange(r);
    return;
  }

  if (BssIsFlt( ItvMinRP( r1)) || BssIsFlt( ItvMinRP( r2)))
    FltRoundDownWard;
  BssAdd( ItvMinRP(r), ItvMinRP(r1) ,ItvMinRP(r2));

  if (ItvIsLeftOpen( r1) || ItvIsLeftOpen( r2))
    ItvSetLeftBracketOpen( r);
  else
    ItvSetLeftBracketClosed( r);
  
  if (BssIsFlt( ItvMaxRP( r)) || BssIsFlt( ItvMaxRP( r2)))
    FltRoundUpWard;
  BssAdd( ItvMaxRP(r), ItvMaxRP(r1), ItvMaxRP(r2));

  if (ItvIsRightOpen( r1) || ItvIsRightOpen( r2))
    ItvSetRightBracketOpen( r);
  else
    ItvSetRightBracketClosed( r);
}


/*-------------------------------------------------------------------------*/
/* ItvSub                                                                  */
/*-------------------------------------------------------------------------*/
void ItvSub( BssItv r, const BssItv r1, const BssItv r2)
{
  if( ItvIsEmptyRange(r1) || ItvIsEmptyRange(r2) )
  {
    ItvSetEmptyRange(r);
    return;
  }

  if (BssIsFlt( ItvMinRP( r1)) || BssIsFlt( ItvMaxRP( r2)))
    FltRoundDownWard;
  BssSub( ItvMinRP(r), ItvMinRP(r1) ,ItvMaxRP(r2));

  if (ItvIsLeftOpen( r1) || ItvIsRightOpen( r2))
    ItvSetLeftBracketOpen( r);
  else
    ItvSetLeftBracketClosed( r);
  
  if (BssIsFlt( ItvMaxRP( r)) || BssIsFlt( ItvMinRP( r2)))
    FltRoundUpWard;
  BssSub( ItvMaxRP(r), ItvMaxRP(r1), ItvMinRP(r2));

  if (ItvIsRightOpen( r1) || ItvIsLeftOpen( r2))
    ItvSetRightBracketOpen( r);
  else
    ItvSetRightBracketClosed( r);
}


/*-------------------------------------------------------------------------*/
/* ItvMinus                                                                */
/*-------------------------------------------------------------------------*/
void ItvMinus( BssItv r, const BssItv r1)
{
  if( ItvIsEmptyRange(r1))
  {
    ItvSetEmptyRange(r);
    return;
  }

  BssMinus(ItvMinRP(r),ItvMaxRP(r1));
  BssMinus(ItvMaxRP(r),ItvMinRP(r1));

  ItvExchangeBrackets( r, r1);
}


/*-------------------------------------------------------------------------*/
/* ItvMul                                                                  */
/*-------------------------------------------------------------------------*/
void ItvMul( BssItv Res, const BssItv r1, const BssItv r2)
{
  BssValue a1b2,a2b1,a1b1,a2b2;
  char a1b2B,a2b1B,a1b1B,a2b2B;
  BssValue zero;

  int cmp; /* In fact, values appearing in BssItv can always be
	      compared */

  if( ItvIsEmptyRange(r1) || ItvIsEmptyRange(r2) )
  {
    ItvSetEmptyRange(Res);
    return;
  }

  if ((ItvIsZeroRange(r1)) || (ItvIsZeroRange(r2)))
    {
      ItvCopy(Res, r1);
      return;
    }

  BssInit( a1b2);
  BssInit( a2b1);
  BssInit( a1b1);
  BssInit( a2b2);

  BssInit( zero);
  BssSet( zero, Float, 0.0);
  
  if (BssCmp( ItvMinRP(r1), zero, &cmp) >= 0)    /* r1 >= 0 */
   {
    if (BssCmp( ItvMinRP(r2), zero, &cmp) >= 0)     /* r2 >= 0 */
      {
	FltRoundDownWard;
       ClearExceptions;
       BssMul( ItvMinRP(Res), ItvMinRP(r1), ItvMinRP(r2));
       ItvSetLeftBracket(Res,BracketL2LLDM(Res,r1,r2));
       FltRoundUpWard;
	ClearExceptions;
       BssMul( ItvMaxRP(Res), ItvMaxRP(r1),ItvMaxRP(r2));
       ItvSetRightBracket(Res,BracketR2RRDM(Res,r1,r2));
       return;
      }
    else
      if (BssCmp(ItvMaxRP(r2), zero, &cmp) <= 0)   /* r2 <= 0 */
        {
         FltRoundDownWard;
	ClearExceptions;
	 BssMul( ItvMinRP(Res), ItvMaxRP(r1),ItvMinRP(r2));
         ItvSetLeftBracket(Res,BracketL2RLDM(Res,r1,r2));
         FltRoundUpWard;
	ClearExceptions;
	 BssMul(ItvMaxRP(Res), ItvMinRP(r1),ItvMaxRP(r2));
         ItvSetRightBracket(Res,BracketR2LRDM(Res,r1,r2));
         return;
        }
      else                /* 0 \in r2 */
        {
         FltRoundDownWard;
	ClearExceptions;
         BssMul( ItvMinRP(Res), ItvMaxRP(r1),ItvMinRP(r2));
         ItvSetLeftBracket(Res,BracketL2RLDM(Res,r1,r2));
         FltRoundUpWard;
	ClearExceptions;
	 BssMul( ItvMaxRP(Res),ItvMaxRP(r1),ItvMaxRP(r2));
	 ItvSetRightBracket(Res,BracketR2RRDM(Res,r1,r2));
         return;
        }
   }
 else
   if (BssCmp( ItvMaxRP(r1), zero, &cmp) <= 0)        /* Z <= 0 */
     {
      if (BssCmp(ItvMinRP(r2), zero, &cmp) >= 0)     /* r2 >= 0 */
        {
         FltRoundDownWard;
	ClearExceptions;
	 BssMul(ItvMinRP(Res), ItvMinRP(r1),ItvMaxRP(r2));
         ItvSetLeftBracket(Res,BracketL2LRDM(Res,r1,r2));
         FltRoundUpWard;
	ClearExceptions;
	 BssMul( ItvMaxRP(Res), ItvMaxRP(r1),ItvMinRP(r2));
         ItvSetRightBracket(Res,BracketR2RLDM(Res,r1,r2));
         return;
        }
     else
        if (BssCmp( ItvMaxRP(r2), zero, &cmp) <= 0)   /* r2 <= 0 */
          {
           FltRoundDownWard;
	ClearExceptions;
	   BssMul( ItvMinRP(Res), ItvMaxRP(r1),ItvMaxRP(r2));
           ItvSetLeftBracket(Res,BracketL2RRDM(Res,r1,r2));
           FltRoundUpWard;
	ClearExceptions;
	   BssMul( ItvMaxRP(Res), ItvMinRP(r1),ItvMinRP(r2));
           ItvSetRightBracket(Res,BracketR2LLDM(Res,r1,r2));
           return;
          }
        else                /* 0 \in r2 */
          {
           FltRoundDownWard;
	ClearExceptions;
	   BssMul( ItvMinRP(Res), ItvMinRP(r1),ItvMaxRP(r2));
           ItvSetLeftBracket(Res,BracketL2LRDM(Res,r1,r2));
           FltRoundUpWard;
	ClearExceptions;
	   BssMul( ItvMaxRP(Res), ItvMinRP(r1),ItvMinRP(r2));
           ItvSetRightBracket(Res,BracketR2LLDM(Res,r1,r2));
           return;
          }
     }
   else                  /* 0 \in r1 */
     {
      if (BssCmp( ItvMinRP(r2), zero, &cmp) >= 0)     /* r2 >= 0 */
        {
         FltRoundDownWard;
	ClearExceptions;
	 BssMul( ItvMinRP(Res), ItvMinRP(r1),ItvMaxRP(r2));
         ItvSetLeftBracket(Res,BracketL2LRDM(Res,r1,r2));
         FltRoundUpWard;
	ClearExceptions;
	 BssMul( ItvMaxRP(Res), ItvMaxRP(r1),ItvMaxRP(r2));
         ItvSetRightBracket(Res,BracketR2RRDM(Res,r1,r2));
         return;
        }
     else
        if (BssCmp( ItvMaxRP(r2), zero, &cmp) <= 0)   /* r2 <= 0 */
          {
           FltRoundDownWard;
	ClearExceptions;
	   BssMul( ItvMinRP(Res), ItvMaxRP(r1),ItvMinRP(r2));
           ItvSetLeftBracket(Res,BracketL2RLDM(Res,r1,r2));
           FltRoundUpWard;
	ClearExceptions;
	   BssMul( ItvMaxRP(Res), ItvMinRP(r1),ItvMinRP(r2));
           ItvSetRightBracket(Res,BracketR2LLDM(Res,r1,r2));
           return;
          }
        else                /* 0 \in r2 */
          {
           FltRoundDownWard;
	ClearExceptions;
	   BssMul(a1b2, ItvMinRP(r1),ItvMaxRP(r2));
           a1b2B=BracketL2LRDM2(a1b2,r1,r2);
	   BssMul(a2b1, ItvMaxRP(r1),ItvMinRP(r2));
           a2b1B=BracketL2RLDM2(a2b1,r1,r2);
           if (BssCmp( a1b2, a2b1, &cmp) < 0)
             {
              BssCopy( ItvMinRP(Res), a1b2);
              ItvSetLeftBracket(Res,a1b2B);
             }
           else
             if (BssCmp(a2b1, a1b2, &cmp) < 0)
               {
                BssCopy( ItvMinRP(Res),a2b1);
                ItvSetLeftBracket(Res,a2b1B);
               }
             else   /* a1b2 == a2b1 but brackets may be differents */
               {
                BssCopy(ItvMinRP(Res),a2b1);
                ItvSetLeftBracket(Res,((a1b2B<=a2b1B) ? a1b2B : a2b1B));
               }

           FltRoundUpWard;
	ClearExceptions;
	   BssMul(a1b1, ItvMinRP(r1),ItvMinRP(r2));
           a1b1B=BracketR2LLDM2(a1b1,r1,r2);
	   BssMul(a2b2, ItvMaxRP(r1),ItvMaxRP(r2));
           a2b2B=BracketR2RRDM2(a2b2,r1,r2);
           if (BssCmp(a1b1,a2b2,&cmp) > 0)
             {
              BssCopy( ItvMaxRP(Res), a1b1);
              ItvSetRightBracket(Res,a1b1B);
             }
           else
             if (BssCmp(a2b2,a1b1, &cmp) > 0)
               {
                BssCopy( ItvMaxRP(Res), a2b2);
                ItvSetRightBracket(Res,a2b2B);
               }
             else   /* a1b1 = a2b2 but brackets may be differents */
               {
                BssCopy( ItvMaxRP(Res),a2b2);
                ItvSetRightBracket(Res,((a1b1B>=a2b2B) ? a1b1B : a2b2B));
               }
           return;
          }      
     }
}


/*
{
  BssFlt acDn, adDn, bcDn, bdDn, acUp, adUp, bcUp, bdUp;

  FltRoundDownWard;
  acDn = BssMul(ItvMinRP(r1),ItvMinRP(r2));
  adDn = BssMul(ItvMinRP(r1),ItvMaxRP(r2));
  bcDn = BssMul(ItvMaxRP(r1),ItvMinRP(r2));
  bdDn = BssMul(ItvMaxRP(r1),ItvMaxRP(r2));

  if( BssCmp( acDn,adDn) == -1 )
  {
    if( BssCmp( bcDn, bdDn) == -1)
    {
      ItvMinRP(r) = BssMin(acDn,bcDn);

      if (ItvIsLeftOpen( r1) || ItvIsLeftOpen( r2))
	ItvSetRightBracketOpen( r);
      else
	ItvSetRightBracketClosed( r);
      FltRoundUpWard;
      adUp = BssMul(ItvMinRP(r1),ItvMaxRP(r2));
      bdUp = BssMul(ItvMaxRP(r1),ItvMaxRP(r2));
      ItvMaxRP(r) = BssMax(adUp,bdUp);
    }
    else
    {
      ItvMinRP(r) = BssMin(acDn,bdDn);
      FltRoundUpWard;
      adUp = BssMul(ItvMinRP(r1),ItvMaxRP(r2));
      bcUp = BssMul(ItvMaxRP(r1),ItvMinRP(r2));
      ItvMaxRP(r) = BssMax(adUp,bcUp);
    }
  }
  else
  {
    if( BssCmp(bcDn,bdDn)==-1 )
    {
      ItvMinRP(r) = BssMin(adDn,bcDn);
      FltRoundUpWard;
      acUp = BssMul(ItvMinRP(r1),ItvMinRP(r2));
      bdUp = BssMul(ItvMaxRP(r1),ItvMaxRP(r2));
      ItvMaxRP(r) = BssMax(acUp,bdUp);
    }
    else
    {
      ItvMinRP(r) = BssMin(adDn,bdDn);
      FltRoundUpWard;
      acUp = BssMul(ItvMinRP(r1),ItvMinRP(r2));
      bcUp = BssMul(ItvMaxRP(r1),ItvMinRP(r2));
      ItvMaxRP(r) = BssMax(acUp,bcUp);
    }
  }
} */


/*-------------------------------------------------------------------------*/
/* ItvDiv                                                        */
/*-------------------------------------------------------------------------*/
void ItvDiv( BssItv Res, const BssItv r1, const BssItv r2)
{
  BssValue zero;
  int cmp;

  if( ItvIsEmptyRange(r1) || ItvIsEmptyRange(r2) )
    {
      ItvSetEmptyRange(Res);
      return;
    }

  BssInit( zero);
  BssSet( zero, Float, 0.0);

  if (ItvIsZeroRange(r2))  /* r2 = [0,0] */
   {
    ItvSetMaxRange(Res);
    return;
   }
 
 if (ItvIsZeroRange(r1)) /* r1 = [0,0] */
   {
    if ((BssCmp( ItvMinRP(r2), zero, &cmp)<=0) && (BssCmp( ItvMaxRP(r2), zero, &cmp)>=0))
       ItvSetMaxRange(Res);
    else
       ItvSetZeroRange(Res);
    return;
   }

 if (BssCmp( ItvMinRP(r1), zero, &cmp)>=0)                   /* r1 >= 0 */
   {
    if (BssCmp( ItvMinRP(r2), zero, &cmp)>0)                 /* r2 > 0  */
      {
       FltRoundDownWard;
	ClearExceptions;
       BssDiv( ItvMinRP(Res),ItvMinRP(r1),ItvMaxRP(r2));
       ItvSetLeftBracket(Res,BracketL2LRDM(Res,r1,r2));
       FltRoundUpWard;
	ClearExceptions;
       BssDiv( ItvMaxRP(Res), ItvMaxRP(r1),ItvMinRP(r2));
       ItvSetRightBracket(Res,BracketR2RLDM(Res,r1,r2)); 
       return;
      }
    else 
      if (BssCmp( ItvMaxRP(r2), zero, &cmp)<0)              /* r2 < 0  */
        {
         FltRoundDownWard;
	ClearExceptions;
	 BssDiv( ItvMinRP(Res), ItvMaxRP(r1),ItvMaxRP(r2));
         ItvSetLeftBracket(Res,BracketL2RRDM(Res,r1,r2));
         FltRoundUpWard;
	ClearExceptions;
	 BssDiv(ItvMaxRP(Res),ItvMinRP(r1),ItvMinRP(r2));
         ItvSetRightBracket(Res,BracketR2LLDM(Res,r1,r2)); 
         return;
        }
      else                           /* 0 \in r2 */
        {
         ItvSetMaxRange(Res);
         return;
        }
   }
 else
   if (BssCmp( ItvMaxRP(r1), zero, &cmp)<=0)               /* r1 <= 0 */
     {
      if (BssCmp( ItvMinRP(r2), zero, &cmp)>0)             /* r2 > 0 */
        {  
         FltRoundDownWard;
	ClearExceptions;
	 BssDiv( ItvMinRP(Res), ItvMinRP(r1),ItvMinRP(r2));
         ItvSetLeftBracket(Res,BracketL2LLDM(Res,r1,r2));
         FltRoundUpWard;
	ClearExceptions;
	 BssDiv( ItvMaxRP(Res), ItvMaxRP(r1),ItvMaxRP(r2));
         ItvSetRightBracket(Res,BracketR2RRDM(Res,r1,r2)); 
         return;
        }
      else
        if (BssCmp( ItvMaxRP(r2), zero, &cmp)<0)          /* r2 < 0 */
          {
           FltRoundDownWard;
	ClearExceptions;
	   BssDiv( ItvMinRP(Res), ItvMaxRP(r1),ItvMinRP(r2));
           ItvSetLeftBracket(Res,BracketL2RLDM(Res,r1,r2));
           FltRoundUpWard;
	ClearExceptions;
	   BssDiv(ItvMaxRP(Res),ItvMinRP(r1),ItvMaxRP(r2));
           ItvSetRightBracket(Res,BracketR2LRDM(Res,r1,r2)); 
           return;
          }
        else                     /* 0 \in r2 */
          {
           ItvSetMaxRange(Res);
           return;           
          }   
     }
   else                        /* 0 in r1 */
     {
      if (BssCmp( ItvMinRP(r2), zero, &cmp)>0)        /* r2 > 0 */
        {
         FltRoundDownWard;
	ClearExceptions;
	 BssDiv( ItvMinRP(Res), ItvMinRP(r1),ItvMinRP(r2));
         ItvSetLeftBracket(Res,BracketL2LLDM(Res,r1,r2));
         FltRoundUpWard;
	ClearExceptions;
	 BssDiv( ItvMaxRP(Res),ItvMaxRP(r1),ItvMinRP(r2));
         ItvSetRightBracket(Res,BracketR2RLDM(Res,r1,r2));
         return;
        }
      else
       if (BssCmp( ItvMaxRP(r2), zero, &cmp)<0)       /* r2 < 0 */
         {
          FltRoundDownWard;
	ClearExceptions;
	  BssDiv( ItvMinRP(Res),ItvMaxRP(r1),ItvMaxRP(r2));
          ItvSetLeftBracket(Res,BracketL2RRDM(Res,r1,r2));
          FltRoundUpWard;
	  ClearExceptions;
	  BssDiv( ItvMaxRP(Res),ItvMinRP(r1),ItvMaxRP(r2));
          ItvSetRightBracket(Res,BracketR2LRDM(Res,r1,r2)); 
          return;
         }
       else
         {
          ItvSetMaxRange(Res);
          return;           
         }
     }
}


/* {
  BssFlt acDn, adDn, bcDn, bdDn, acUp, adUp, bcUp, bdUp;

  if( ItvIsNumInRange(r2,0.0) )
  {
    ItvSetMaxRange(&r);
    return;
  }

  FltRoundDownWard;
  acDn = FltMathDiv(ItvMinRP(r1),ItvMinRP(r2));
  adDn = FltMathDiv(ItvMinRP(r1),ItvMaxRP(r2));
  bcDn = FltMathDiv(ItvMaxRP(r1),ItvMinRP(r2));
  bdDn = FltMathDiv(ItvMaxRP(r1),ItvMaxRP(r2));

  if( acDn<adDn )
  {
    if( bcDn<bdDn )
    {
      ItvMinRP(r) = FltMathMin(acDn,bcDn);
      FltRoundUpWard;
      adUp = FltMathDiv(ItvMinRP(r1),ItvMaxRP(r2));
      bdUp = FltMathDiv(ItvMaxRP(r1),ItvMaxRP(r2));
      ItvMaxRP(r) = FltMathMax(adUp,bdUp);
    }
    else
    {
      ItvMinRP(r) = FltMathMin(acDn,bdDn);
      FltRoundUpWard;
      adUp = FltMathDiv(ItvMinRP(r1),ItvMaxRP(r2));
      bcUp = FltMathDiv(ItvMaxRP(r1),ItvMinRP(r2));
      ItvMaxRP(r) = FltMathMax(adUp,bcUp);
    }
  }
  else
  {
    if( bcDn<bdDn )
    {
      ItvMinRP(r) = FltMathMin(adDn,bcDn);
      FltRoundUpWard;
      acUp = FltMathDiv(ItvMinRP(r1),ItvMinRP(r2));
      bdUp = FltMathDiv(ItvMaxRP(r1),ItvMaxRP(r2));
      ItvMaxRP(r) = FltMathMax(acUp,bdUp);
    }
    else
    {
      ItvMinRP(r) = FltMathMin(adDn,bdDn);
      FltRoundUpWard;
      acUp = FltMathDiv(ItvMinRP(r1),ItvMinRP(r2));
      bcUp = FltMathDiv(ItvMaxRP(r1),ItvMinRP(r2));
      ItvMaxRP(r) = FltMathMax(acUp,bcUp);
    }
  }
} */


/*-------------------------------------------------------------------------*/
/* ItvAbs                                                             */
/*-------------------------------------------------------------------------*/
void ItvAbs( BssItv r, const BssItv r1)
{
  double x;
  BssValue a, b;
  int cmp, rescmp;

  if( ItvIsEmptyRange(r1) )
  {
    ItvSetEmptyRange(r);
    return;
  }

  BssInit( a);
  BssInit( b);
  BssAbs(a, ItvMinRP(r1));
  BssAbs(b, ItvMaxRP(r1));
  
  rescmp = BssCmp( a, b, &cmp);

  if (cmp)
    if (rescmp <= 0) {
      BssCopy( ItvMinRP(r), a);
      BssCopy( ItvMaxRP(r), b);
      ItvCopyBrackets( r, r1);
    }
    else {
      BssCopy(ItvMinRP( r), b);
      BssCopy( ItvMaxRP( r), a);
      ItvExchangeBrackets( r, r1);
    }
}


/*-------------------------------------------------------------------------*/
/* ItvSqr                                                             */
/*-------------------------------------------------------------------------*/
void ItvSqr( BssItv r, const BssItv r1)
{
  BssItv tmp;
  if( ItvIsEmptyRange(r1) )
  {
    ItvCopy(r, r1);
    return;
  }

  ItvInit( tmp);
  ItvAbs(tmp,r1);
  if (BssIsFlt( ItvMinRP( tmp)))
    FltRoundDownWard;
  BssSqr(ItvMinRP(r), ItvMinRP(tmp));
    if (BssIsFlt( ItvMaxRP( tmp)))
      FltRoundUpWard;
  BssSqr(ItvMaxRP(r),ItvMaxRP(tmp));
}


/*-------------------------------------------------------------------------*/
/* ItvSqrt                                                            */
/*-------------------------------------------------------------------------*/
void ItvSqrt( BssItv r, const BssItv r1)
{
  int cmp;
  BssValue zero;
  BssItv tmp;

  ItvInit( tmp);
  ItvSetZeroRange( tmp);
  BssInit( zero);
  BssSet( zero,Float, 0.0);
  if( (ItvIsEmptyRange(r1)) || (!(ItvCmp( r1, tmp, &cmp)) || !cmp))
  {
    ItvSetEmptyRange(r);
    return;
  }
  ItvAbs(tmp,r1);
  FltRoundDownWard;
  BssSqrt(ItvMinRP(r), ItvMinRP(tmp));
  FltRoundUpWard;
  BssSqrt( ItvMaxRP(r), ItvMaxRP(tmp));
  ItvCopyBrackets( r, r1);
}


/*-------------------------------------------------------------------------*/
/* ItvPow                                                             */
/*-------------------------------------------------------------------------*/
void ItvPow( BssItv r, const BssItv r1, const int n)
{
  BssValue a, b;
  int cmp, rescmp;
  
  if( ItvIsEmptyRange(r1) )
  {
    ItvSetEmptyRange(r);
    return;
  }

  BssInit(a);
  BssInit(b);
  if (BssIsFlt( ItvMinRP( r1)))
    FltRoundDownWard;
  BssPow( a, ItvMinRP( r1),n);
  if (BssIsFlt( ItvMaxRP( r1)))
    FltRoundUpWard;
  BssPow( b, ItvMaxRP( r1),n);

  rescmp = BssCmp( a, b, &cmp);

  if (cmp)
    if (rescmp <= 0) {
      BssCopy(ItvMinRP(r), a);
      BssCopy(ItvMaxRP(r),b);
      ItvCopyBrackets( r, r1);
    }
    else {
      BssCopy(ItvMinRP(r), b);
      BssCopy(ItvMaxRP(r), a);
      ItvExchangeBrackets( r, r1);
    }
  else
    ItvSetEmptyRange(r);
}
    
/*-------------------------------------------------------------------------*/
/* ItvInter                                                                */
/*-------------------------------------------------------------------------*/
int ItvInter( BssItv r, const BssItv r1, const BssItv r2)
{
  int cmpval, cmp;
  
  if( ItvIsEmptyRange(r1) || ItvIsEmptyRange(r2) )
    ItvSetEmptyRange(r);
  else {
    /* left side */
    cmpval = BssCmp( ItvMinRP(r1),ItvMinRP(r2), &cmp);
    if (!cmp) {
      ItvSetEmptyRange( r);
      return 1;
    }
    
    if ( cmpval > 0) {
      BssCopy( ItvMinRP(r), ItvMinRP(r1));
      if (ItvIsLeftOpen( r1))
	ItvSetLeftBracketOpen( r);
      else
	ItvSetLeftBracketClosed( r);
    }
    else if ( cmpval < 0) {
      BssCopy( ItvMinRP(r), ItvMinRP(r2));
      if (ItvIsLeftOpen( r2))
	ItvSetLeftBracketOpen( r);
      else
	ItvSetLeftBracketClosed( r);
    }
    else {
      BssCopy( ItvMinRP(r), ItvMinRP(r1));
      if (ItvIsLeftOpen(r1) || ItvIsLeftOpen( r2))
	ItvSetLeftBracketOpen( r);
      else
	ItvSetLeftBracketClosed( r);
    }
    /* right side */
    cmpval = BssCmp( ItvMaxRP(r1),ItvMaxRP(r2), &cmp);
    if (!cmp) {
      ItvSetEmptyRange( r);
      return 1;
    }
    if ( cmpval < 0) {
      BssCopy( ItvMaxRP(r), ItvMaxRP(r1));
      if (ItvIsRightOpen( r1))
	ItvSetRightBracketOpen( r);
      else
	ItvSetRightBracketClosed( r);
    }
    else if ( cmpval > 0) {
      BssCopy( ItvMaxRP(r), ItvMaxRP(r2));
      if (ItvIsRightOpen( r2))
	ItvSetRightBracketOpen( r);
      else
	ItvSetRightBracketClosed( r);
    }
    else {
      BssCopy( ItvMaxRP(r), ItvMaxRP(r1));
      if (ItvIsRightOpen(r1) || ItvIsRightOpen( r2))
	ItvSetRightBracketOpen( r);
      else
	ItvSetRightBracketClosed( r);
    }
  }

  if ((BssCmp(ItvMinRP( r), ItvMaxRP( r), &cmp) == 0)
      && (ItvIsLeftOpen( r) || ItvIsRightOpen( r)))
    ItvSetEmptyRange( r);
  
  return 1;
}


/*-------------------------------------------------------------------------*/
/* ItvWrite                                                           */
/*-------------------------------------------------------------------------*/
void ItvWrite(FILE *out, const BssItv r, int digits)
{
  if( ItvIsEmptyRange(r) )
  {
    fprintf(out,"(+Inf,-Inf)");
    return;
  }
  else
  {
    if (ItvIsLeftOpen(r))
      fputc('(', out);
    else
      fputc('[', out);
    if (BssIsFlt( ItvMinRP( r)))
      FltRoundDownWard;
    BssWrite( out, ItvMinRP( r), digits);
    fputc(',', out);
/*     fputc('.', out); */
    if (BssIsFlt( ItvMaxRP( r)))
      FltRoundUpWard;
    BssWrite( out, ItvMaxRP( r), digits);   
    if (ItvIsRightOpen(r))
      fputc(')', out);
    else
      fputc(']', out);
  }
}

char* Itv2bstring(  BssItv r, unsigned int *size)
{
  char *s;
/*   char *s, *ls, *rs;
 *   int lsize, rsize;
 * 
 *   ls = Bss2bstring( ItvMinRP( r), &lsize);
 *   rs = Bss2bstring( ItvMaxRP( r), &rsize);
 * 
 *   s = (char*)malloc( lsize + rsize +1);
 * 
 *   s[0] = ItvGetBrackets( r);
 * 
 *   for (*size=1; *size <= lsize; (*size)++)
 *     s[*size] = ls[*size-1];
 *   for (; *size <= lsize + rsize; (*size)++)
 *     s[*size] = rs[*size-lsize];
 *   
 *   return s;
 */
  s = Itv2wstring( r);
  size[0] = strlen(s) + 1;
  return s;
}
  
  
BssItvP bstring2Itv( char *s)
{
/*   char b;
 *   BssValue *l, *r;
 *   BssItvP i;
 * 
 *   i = ItvAlloc();
 *   ItvSetBrackets( i[0],  s[0]);
 *   s++;
 *   l = bstring2Bss( s);
 * 
 *   switch (BssGetType( l[0])) {
 *   case Integer:
 *     s += INT_BSTRING_SIZE;
 *     break;
 *   case Rational:
 *     s += RAT_BSTRING_SIZE;
 *     break;
 *   case Float:
 *     s += FLT_BSTRING_SIZE;
 *     break;
 *   case Minus_infinity:
 *   case Plus_infinity:
 *     s += INF_BSTRING_SIZE;
 *     break;    
 *   }    
 * 
 *   r = bstring2Bss( s);
 * 
 *   BssCopy( ItvMinRP( i[0]), l[0]);
 *   BssCopy(ItvMaxRP( i[0]), r[0]);  
 * 
 *   return i;
 */
  return wstring2Itv( s);
}
		  
char *Itv2wstring( const BssItv r)
{
  char *s;
  char *ls, *rs;
  
  if( ItvIsEmptyRange(r) )
  {
    
    s = (char*)malloc(22);
    sprintf(s, "r((l(+Inf)..l(-Inf)))");
  }
  else
  {
    if (BssIsFlt( ItvMinRP( r)))
      FltRoundDownWard;

    ls = Bss2wstring(ItvMinRP( r));
    if (BssIsFlt( ItvMaxRP( r)))    
      FltRoundUpWard;

    rs = Bss2wstring( ItvMaxRP( r));   
    s = (char*)malloc(10+strlen(ls)+strlen(rs));
    sprintf( s, "r(%c%s..%s%c)",
	     ItvIsLeftOpen(r)  ? '(':'[',
	     ls,
	     rs,
	     ItvIsRightOpen(r) ? ')':']'
	     );
#ifdef LIBERE
    free(ls);
    free(rs);
#endif    
  }
    return s;
}
     
BssItvP wstring2Itv( char *source)
{
  char l,r;
  char *p, *s, *s1, *s2;
  BssValueP lval, rval;
  int cmp, cmpval, pile, index;
  BssItvP i;

  p = s = strdup( source);
  
  l = (s[2]=='[' ? 1 : 0);
  /* search the right bracket */
      pile = 1;
      index = 3;
      do {
	switch (s[index]) {
	case '(':
	  pile++;
	  break;
	case ')':
	case ']':
	  pile--;
	}
	index++;
      } while (pile!=0);
  r = (s[index - 1]==']' ? 1 : 0);
  s1 = &s[3];
  s1[strlen(s)-5] = '\0';
  for (s2 = s1; !((s2[0]=='.') && (s2[1]=='.')); s2++)
    l+= 0;
  *s2='\0';
  s2+=2;

  FltRoundDownWard;
  lval = wstring2Bss( s1);
  FltRoundUpWard;
  rval = wstring2Bss( s2);

#ifdef LIBERE  
  free( p);
#endif
  
  cmpval = BssCmp(lval, rval, &cmp);
  if ((cmpval > 0) || ((cmpval == 0) && (!l || !r))) {
    i = ItvAlloc();
    ItvSetZeroRange(i); 
    ItvSetEmptyRange(i);
    ItvSetLeftBracketOpen(i);
    ItvSetRightBracketOpen(i);
    return i;
  }
  else
    return ItvAllocInit( lval, rval, l, r);
}    

int ItvCmp( const BssItv i1, const BssItv i2, int *cmp)
{
  int cmpminmax, cmpval;
  int r1empty, r2empty;

  r1empty = ItvIsEmptyRange( i1);
  r2empty = ItvIsEmptyRange( i2);
  if ( r1empty && r2empty) {
    *cmp = 1;
    return 0;
  }
  else if (r1empty || r2empty) {
    *cmp = 0;
    return -1;
  }
  else if (!BssCmp(  ItvMinRP( i1), ItvMinRP( i2), &cmpval)
	   &&  !BssCmp(ItvMaxRP( i1), ItvMaxRP( i2), &cmpval)) {
    *cmp = 1;
    return 0;
  }
  else {
    *cmp = 1;

    /* i1 = m1..M1 et i2 = m2..M2 */
    cmpminmax = BssCmp( ItvMinRP( i1), ItvMaxRP( i2), &cmpval);
    
    if (cmpval) {
      if (cmpminmax > 0) /* m1>M2 */
	return 1;
      else { /* m1<=M2 */
	cmpminmax = BssCmp( ItvMinRP( i2), ItvMaxRP( i1), &cmpval);
	if (cmpval) {
	  if (cmpminmax > 0) /* m1<=M2, m2>M1 */
	    return -1;
	  else {/* m1<=M2, m2<=M1 */
	    cmpminmax = BssCmp( ItvMinRP( i1), ItvMinRP( i2), &cmpval);
	    if (cmpval) {
	      if (cmpminmax < 0) { /* m1<=M2, m2<=M1, m1<m2 */
		cmpminmax = BssCmp( ItvMaxRP( i1), ItvMaxRP( i2), &cmpval);
		if (cmpval) {
		  if (cmpminmax < 0) /* m1<=M2, m2<=M1, m1<m2, M1<M2 */
		    return -1;
		  else { /* m1<=M2, m2<=M1, m1<m2, M1>=M2 */
		    *cmp = 0;
		    return -1;
		  }
		}
		else { /* !cmpval */ 
		  *cmp = 0;
		  return -1;
		}
	      }
	      else {/* m1<=M2, m2<=M1, m1>=m2 */
		cmpminmax = BssCmp( ItvMaxRP( i1), ItvMaxRP( i2), &cmpval);
		if (cmpval) {
		  if (cmpminmax > 0) /* m1<=M2, m2<=M1, m1>=m2, M1>M2 */
		    return 1;
		  else { /* m1<=M2, m2<=M1, m1>=m2, M1<=M2 */
		    *cmp = 0;
		    return -1;
		  }
		}
		else { /* !cmpval */ 
		  *cmp = 0;
		  return -1;
		}
	      }
	    }
	    else { /* !cmpval */ 
	      *cmp = 0;
	      return -1;
	    }
	  }
	}
	else { /* !cmpval */ 
	  *cmp = 0;
	  return -1;
	}
      }
    }
    else { /* !cmpval */ 
      *cmp = 0;
      return -1;
    }
  }
}
