/***************************************************************************/
/* float.h                                                                 */
/*                                                                         */
/* Description file for operations on floats                               */
/*                                                                         */
/* Float    Arithmetic Library                                             */
/*									   */
/* Derived from Laurent Granvilliers' solver				   */
/***************************************************************************/

/*************************************************************************/
/*                                                                       */
/*       Copyright (C) 1998 Universite de Nantes                         */
/*                                                                       */
/*************************************************************************/
 
/************* 
 Edit history: 
       27/2/98 - Nicolas Romero : Creation
*************/

#ifndef FLOAT_H
#define FLOAT_H

#include <stdio.h>

#include <floatingpoint.h>

/*-------------------------------------------------------------------------*/
/* IEEE rounding : Source : Sun Release 4.1 - Mathematical Library         */
/*-------------------------------------------------------------------------*/

extern double  infinity();
extern int     isinf();
extern double  nextafter();
extern double  sqrt();
extern double  ceil();
extern double  floor();

#undef NaN
#ifdef NaN
#define NaNTest( f) ( isnan(FltMathArg=f) ? 0.0 : FltMathArg )
#else
#define NaNTest( f) f
#endif

#include <whichsun.h>

char *FltRound;
#define FltRoundDownWard  ieee_f("set","direction","negative",&FltRound)
#define FltRoundUpWard    ieee_f("set","direction","positive",&FltRound)
#define FltRoundNearest   ieee_f("set","direction","nearest",&FltRound)
#define FltSetPrecision   ieee_f("set","precision","double",&FltRound)

/* static inline volatile void RoundDown()
 * {
 *   int cw;
 *   asm volatile ("st %%fsr,%0" : "=m" (*&cw));
 *   cw = (cw & 0x3fffffff) | 0xC0000000;
 *   asm volatile ("ld %0,%%fsr" : : "m" (*&cw));
 * }
 * 
 * static inline volatile void RoundUp()
 * {
 *   int cw;
 *   asm volatile ("st %%fsr,%0" : "=m" (*&cw));
 *   cw = (cw & 0x3fffffff) | 0x80000000;
 *   asm volatile ("ld %0,%%fsr" : : "m" (*&cw));
 * }
 * #ifndef UpOrDownSeen
 * char UpOrDown;
 * #define UpOrDownSeen
 * #else
 * extern char UpOrDown;
 * #endif
 * #define FltRoundDownWard (UpOrDown = -1)
 * #define FltRoundUpWard (UpOrDown = 1)
 * #define FltRound ((UpOrDown == -1 ? RoundDown()\
 * 		                  : UpOrDown ==1 ? RoundUp()\
 * 		                                 : 0)\
 * 		  , UpOrDown=0)
 * #endif
 */

#define FltPiLow  3.1415926535897931
#define FltPiHigh 3.1415926535897932

/* static double FltInfinity =  1/0.0; */

#define FltMaxDouble 1.797693134862315708e+308
#define FltMinDouble -FltMaxDouble
#define FltNextZero  4.94065645841246544e-324

#define FltIsInfinity(a) (isinf(a))

static double FltMathArg;
#define FltMathMul(x,y)  NaNTest(x*y)
#define FltMathDiv(x,y)  NaNTest(x/y)

#define FltMathAbs(x) ((x<=0.0) ? -(x) : x)

#define FltMathIsOdd(n) (((n)%2)==1)
#define FltMathIsEven(n) (((n)%2)==0)

#define FltMathMin(x,y) (((x)<(y))? x : y)
#define FltMathMax(x,y) (((x)<(y))? y : x)
#define FltMathMin4(x,y,z,t) FltMathMin(FltMathMin(x,y),FltMathMin(z,t))
#define FltMathMax4(x,y,z,t) FltMathMax(FltMathMax(x,y),FltMathMax(z,t))

#define FltMathCenter(a,b) ((FltMathMax(a,FltMinDouble)/2) +   \
                         (FltMathMin(FltMaxDouble,b)/2))

/*
#define FltMathCenter(a,b) \
 ( FltIsInfinity(a) ? \
      (FltIsInfinity(b) ? 0.0 : ( (FltMinDouble/2 + (b)/2) ) ) : \
      (FltIsInfinity(b) ? (FltMaxDouble/2) : ((a)/2 + (b)/2) ) )
*/

#define FltStrToDouble(s) atof(s)

#define FltNextDouble(x) nextafter(x,FltMaxRange)
#define FltPrevDouble(x) nextafter(x,FltMinRange)

#define FltVal( f) (((double*)(f))[0])

#define FLT_PRECISION 25
#define FLT_BASE_STRING_SIZE 7

#define FLT_BSTRING_SIZE (sizeof (double))
     
/*-------------------------------------------------------------------------*/
/* BSS FLOAT                                                               */
/*-------------------------------------------------------------------------*/

typedef double BssFltType;
typedef BssFltType BssFlt[1];
typedef BssFltType *BssFltP;

#define  FltAlloc() \
     (BssFltP)malloc( sizeof(BssFltType))
BssFltP FltAllocCopy(const BssFlt f);
BssFltP FltAllocInit(double d);

#define FltInit(f)

#define  FltSet(r1, d)\
         (FltRound,FltVal(r1)=d)
#define  FltCopy(r1, r2)\
     (FltRound,FltVal(r1)=FltVal(r2))
#define FltFree(f)\
     free(f)

#define FltAdd(f, f1, f2) \
     (FltRound,FltSet( f, FltVal(f1)+FltVal(f2)))
#define FltSub(f, f1, f2) \
     (FltRound,FltSet( f,  FltVal(f1)-FltVal(f2)))
#define FltMinus(f, f1) \
     (FltRound,FltSet( f,  -FltVal(f1)))
#define FltMul(f, f1, f2) \
     (FltRound,FltSet( f,  FltMathMul(FltVal(f1),FltVal(f2))))
#define FltDiv(f, f1, f2) \
     (FltRound,FltSet( f,  FltMathDiv(FltVal(f1),FltVal(f2))))
#define FltSqrt(f, f1) \
     (FltRound,FltSet( f,  sqrt(FltVal(f1))))
#define FltAbs(f, f1) \
     (FltRound,FltSet( f,  FltMathAbs(FltVal(f1))))
#define FltSqr(f, f1) \
     (FltRound,FltSet( f,  FltMathMul(FltVal(f1),FltVal(f1))))
void FltPow(BssFlt r, const BssFlt x, int n);

#define  FltInter(f, f1, f2)\
     (FltRound,\
     ((FltVal(f1)==FltVal(f2))\
     ? (FltCopy( f,  f1), 1)\
     : 0))

#define FltWrite(out, f,digits)\
     (FltRound,fprintf(out,"%+.*g",digits,FltVal(f)))

char* Flt2bstring( const BssFlt v, unsigned int *size);
BssFltP bstring2Flt( const char *s);
     
extern BssFlt *klic2Flt( void *);
extern void *Flt2klic( const BssFlt );

char *Flt2wstring( const BssFlt v);
BssFltP wstring2Flt( const char *s);
     
int FltCmp( BssFlt f1, BssFlt f2, int *cmp);

#endif   
