(***********************************************************************

                    Mathematica-Compatible Notebook

This notebook can be used on any computer system with Mathematica 3.0,
MathReader 3.0, or any compatible application. The data for the notebook 
starts with the line of stars above.

To get the notebook into a Mathematica-compatible application, do one of 
the following:

* Save the data starting with the line of stars above into a file
  with a name ending in .nb, then open the file inside the application;

* Copy the data starting with the line of stars above to the
  clipboard, then use the Paste menu command inside the application.

Data for notebooks contains only printable 7-bit ASCII and can be
sent directly in email or through ftp in text mode.  Newlines can be
CR, LF or CRLF (Unix, Macintosh or MS-DOS style).

NOTE: If you modify the data for this notebook not in a Mathematica-
compatible application, you must delete the line below containing the 
word CacheID, otherwise Mathematica-compatible applications may try to 
use invalid cache data.

For more information on notebooks and Mathematica-compatible 
applications, contact Wolfram Research:
  web: http://www.wolfram.com
  email: info@wolfram.com
  phone: +1-217-398-0700 (U.S.)

Notebook reader applications are available free of charge from 
Wolfram Research.
***********************************************************************)

(*CacheID: 232*)


(*NotebookFileLineBreakTest
NotebookFileLineBreakTest*)
(*NotebookOptionsPosition[     50487,       1636]*)
(*NotebookOutlinePosition[     51314,       1665]*)
(*  CellTagsIndexPosition[     51270,       1661]*)
(*WindowFrame->Normal*)



Notebook[{

Cell[CellGroupData[{
Cell["Modelling Electric Circuits", "Title",
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    \( (*\ 
      \[Copyright]1998\ Research\ Institute\ for\ Symbolic\ Computation\ 
        \((RISC - Linz)\)\ *) \)], "Input",
  InitializationCell->True],

Cell[BoxData[
    \( (*\nNO\ WARRANTY\n\n
      The\ program\ was\ produced\ on\ an\ experimental\ basis\ in\ the\n
      course\ of\ the\ research\ and\ development\ conducted\ during\ the\ 
        project\n
      and\ is\ provided\ to\ users\ as\ so\ produced\ on\ an\ experimental\ 
        basis.\nAccordingly, \ 
      the\ program\ is\ provided\ without\ any\ warranty\ whatsoever, \n
      whether\ express, \ implied, \ 
      statutory\ or\ otherwise.\ The\ term\ "\<warranty\>"\n
      used\ herein\ includes, \ but\ is\ not\ limited\ to, \ 
      any\ warranty\ of\ the\n
      quality, \ performance, \ 
      merchantability\ and\ fitness\ for\ a\ particular\n
      purpose\ of\ the\ program\ and\ the\ nonexistence\ of\ any\ infringement
        \ or\n
      violation\ of\ any\ right\ of\ any\ third\ party.\n\nEach\ user\ of\ the
        \ program\ will\ agree\ and\ understand, \ and\ be\n
      deemed\ to\ have\ agreed\ and\ understood, \ 
      that\ there\ is\ no\ warranty\n
      whatsoever\ for\ the\ program\ and, \ accordingly, \ 
      the\ entire\ risk\ arising\n
      from\ or\ otherwise\ connected\ with\ the\ program\ is\ assumed\ by\ the
        \ user.\n\nTherefore, \ neither\ ICOT, \ the\ copyright\ holder, \ 
      or\ any\ other\n
      organization\ that\ participated\ in\ or\ was\ otherwise\ related\ to\ 
        the\n
      development\ of\ the\ program\ and\ their\ respective\ officials, \ 
      directors, \n
      officers\ and\ other\ employees\ shall\ be\ held\ liable\ for\ any\ and
        \ all\n
      damages, \ including, \ without\ limitation, \ general, \ special, \ 
      incidental\n
      and\ consequential\ damages, \ 
      arising\ out\ of\ or\ otherwise\ in\ connection\n
      with\ the\ use\ or\ inability\ to\ use\ the\ program\ or\ any\ product, 
      \ material\n
      or\ result\ produced\ or\ otherwise\ obtained\ by\ using\ the\ program, 
      \nregardless\ of\ whether\ they\ have\ been\ advised\ of, \ 
      or\ otherwise\ had\n
      knowledge\ of, \ 
      the\ possibility\ of\ such\ damages\ at\ any\ time\ during\ the\n
      project\ or\ thereafter.\ Each\ user\ will\ be\ deemed\ to\ have\ agreed
        \ to\ the\n
      foregoing\ by\ his\ or\ her\ commencement\ of\ use\ of\ the\ 
        program.\ The\ term\n
      "\<use\>"\ as\ used\ herein\ includes, \ but\ is\ not\ limited\ to, \ 
      the\ use, \nmodification, \ 
      copying\ and\ distribution\ of\ the\ program\ and\ the\n
      production\ of\ secondary\ products\ from\ the\ program.\n\nIn\ the\ 
        case\ where\ the\ program, \ whether\ in\ its\ original\ form\ or\n
      modified, \ 
      was\ distributed\ or\ delivered\ to\ or\ received\ by\ a\ user\ from\n
      any\ person, \ organization\ or\ entity\ other\ than\ ICOT, \ 
      unless\ it\ makes\ or\n
      grants\ independently\ of\ ICOT\ any\ specific\ warranty\ to\ the\ user
        \ in\n
      writing, \ such\ person, \ organization\ or\ entity, \ 
      will\ also\ be\ exempted\n
      from\ and\ not\ be\ held\ liable\ to\ the\ user\ for\ any\ such\ damages
        \ as\ noted\n
      above\ as\ far\ as\ the\ program\ is\ \(concerned.\)\t\n*) \)], "Input",\

  InitializationCell->True],

Cell[CellGroupData[{

Cell["Package Description", "Subtitle",
  FontColor->RGBColor[0, 0, 0.500008]],

Cell["\<\
This package contains definitions for electrical components and for \
elementary operations, like serial and parallel connections, used in building \
electrical circuits.
We start by modelling the electrical signals as pairs of current/voltage \
ranging over the domain of reals. Next, we model electrical components such \
as resistors, capacitors and inductors by predicates expressing equational \
constraints between the input and output signals. We found useful to \
introduce the notion of component instance which describes the behavior of an \
electric component with a given characteristic value. Component instances are \
modelled as lambda-terms and are used later in specifying the operations \
involved in describing electrical circuits. 
We end by defining the operations of serial connection, parallel connection, \
and junction of signals, which suffice for describing any electrical circuit. \

A sample session of how to use this package is provided in the notebook \
ElDemo.nb.\
\>", "Text"],

Cell[CellGroupData[{

Cell["Initialization", "Subsubtitle"],

Cell[TextData[{
  "There is a standard sequence of ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " commands that is typically used to set up the contexts in a package. \
These commands set the values of $Context and $ContextPath so that the new \
symbols which are introduced are created in the appropriate contexts."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(BeginPackage["\<Electrical`\>", \n{"\<TSolve`\>"}]; \)\)], "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    \("connection to scheduler established."\)], "Print"],

Cell[BoxData[
    \("connection to constraint solvers established."\)], "Print"]
}, Open  ]],

Cell[TextData[{
  StyleBox["BeginPackage",
    FontWeight->"Bold"],
  " sets ",
  StyleBox["Electrical`",
    FontWeight->"Bold"],
  "  to be the current context and specifies that the context of package \
TSolve` (which contains the actual implementation of the CFLP system) is \
needed in addition to the package ",
  StyleBox["System`.",
    FontWeight->"Bold"],
  " As a consequence, all the definitions provided in the package TSolve` are \
made visible in the package Electrical`. The two messages"
}], "Text"],

Cell[BoxData[
    \("connection to scheduler established."\)], "Print"],

Cell[BoxData[
    \("connection to constraint solvers established."\)], "Print"],

Cell[TextData[{
  "are displayed by the command ",
  StyleBox["CSConnect",
    FontWeight->"Bold"],
  " which is called when reading package ",
  StyleBox["TSolve`",
    FontWeight->"Bold"],
  "  and establishes the ",
  StyleBox["MathLink",
    FontSlant->"Italic"],
  " connections interpreter/scheduler and scheduler/constraint solvers."
}], "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Usage", "Subsubtitle"],

Cell[TextData[{
  "The command ",
  StyleBox["Off",
    FontWeight->"Bold"],
  " is useful for switching off the warning messages printed by ",
  StyleBox["Mathematica. ",
    FontSlant->"Italic"],
  "We use it here to switch off the warnings for defining symbols with \
similar names (e.g. ",
  StyleBox["CInst",
    FontWeight->"Bold"],
  " and ",
  StyleBox["RInst",
    FontWeight->"Bold"],
  ")."
}], "Text"],

Cell[BoxData[
    \(Off[General::spell]\)], "Input",
  InitializationCell->True,
  FontColor->RGBColor[0.6, 0, 0]],

Cell["\<\
This sequence of commands introduces the objects intended for \
export (and no others). It also attaches usage messages to symbols; these \
messages can be visualized by the user by typing in  \
\>", "Text"],

Cell[TextData[{
  "?",
  StyleBox["symb",
    FontSlant->"Italic"]
}], "Text",
  TextAlignment->Center],

Cell[BoxData[
    \(\(resistor::usage = 
      "\<resistor[R,S,V0,I0,V1,I1] describes a resistor with input \
voltage/current s[V0,I0] and output s[V1,I1].\>"\ ; \)\)], "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    \(\(inductor::usage = 
      "\<inductor[R,S,V0,I0,V1,I1] describes a inductor with input \
voltage/current s[V0,I0] and output s[V1,I1].\>"\ ; \)\)], "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    \(\(capacitor::usage = 
      "\<capacitor[R,S,s[V0,I0],s[V1,I1]] describes a capacitor with input \
voltage/current s[V0,I0] and output s[V1,I].\>"\ ; \)\)], "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    \(\(source::usage = 
      "\<source[R,S,s[V0,I0],s[V1,I1]] describes a voltage source with input \
voltage/current s[V0,I0] and output s[V1,I1].\>"\ ; \)\)], "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    \(\(earth::usage = 
      "\<earth[S,s[V1,I1]] describes earth with output voltage/current \
s[V1,I1].\>"\ ; \)\)], "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    \(\(junction::usage = 
      "\<junction[L,R] describes a junction of inputs L and outputs R. L and \
R are lists of pairs s[Vi,Ii].\>"\ ; \)\)], "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    \(\(elJoin::usage = 
      "\<elJoin[S,L,V] describes the join operation of inputs L and outputs \
R. Here, L is a list of pairs s[Vi,Ii] modelling the signals, where V equals \
all Vi and I is the sum of Ii.\>"\ ; \)\)], "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    \(\(CInst::usage = 
      "\<CInst[C] is the capacitor with characteristic capacity C.\>"\ ; 
    \)\)], "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    \(\(LInst::usage = 
      "\<LInst[L] is the inductor with characteristic value L.\>"\ ; \)\)], 
  "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    \(\(EInst::usage = 
      "\<EInst[E] is the voltage source with characteristic value E.\>"\ ; 
    \)\)], "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    \(\(RInst::usage = 
      "\<RInst[R] is the the resistor with characteristic value R.\>"\ ; 
    \)\)], "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    \(s::usage; \ sig::usage; \)], "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    \(\(serial::usage = 
      "\<serial[S, Clist, s[V0,I0], s[V1,I1]] is the electrical component \
representing the serial connection of the unary components of list Clist, \
with input s[V0,I0], output s[V1,I1] and system frequency S.\>"; \)\)], 
  "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    \(\(parallel::usage = 
      "\<parallel[S, Clist, s[V0,I0], s[V1,I1]] is the electrical component \
representing the serial connection of the unary components of list Clist, \
with input s[V0,I0], output s[V1,I1] and system frequency S.\>"; \)\)], 
  "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Begin", "Subsubtitle"],

Cell[BoxData[
    \(\(Begin["\<`Private`\>"]; \)\)], "Input",
  PageWidth->Infinity,
  InitializationCell->True,
  ShowSpecialCharacters->False,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[TextData[{
  "This command sets the current context to ",
  StyleBox["Electrical`Private`",
    FontWeight->"Bold"],
  "."
}], "Text"],

Cell[BoxData[
    \(\($ContextPath = 
      Join[\n\t{"\<TypeSyntax`\>", "\<TermSyntax`\>", "\<Variables`\>", \n\t\t
          "\<RewriteRules`\>", "\<RewriteRuleDefs`\>", "\<Types`\>", \n\t\t
          "\<Terms`\>", "\<Calculus`\>", "\<TypeChecker`\>"}, \n
        \t\t$ContextPath]; \)\)], "Input",
  PageWidth->Infinity,
  InitializationCell->True,
  ShowSpecialCharacters->False,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[TextData[{
  StyleBox["$ContextPath",
    FontWeight->"Bold"],
  " is a global variable that gives a list of contexts, after ",
  StyleBox["$Context",
    FontWeight->"Bold"],
  ", to search in trying to find a symbol that has been entered. From now on, \
we can make use of the commands defined in the packages TypeSyntax`, \
TermSyntax`, Variables`, RewriteRules`, RewriteRuleDefs`, Types`, Terms`, \
Calculus`, TypeChecker` and TSolve`."
}], "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Electric signals", "Subsubtitle"],

Cell[TextData[{
  "By convention, electric signals are defined as pairs ",
  StyleBox["V/I",
    FontWeight->"Bold"],
  " of voltage/current. We model them in our system as terms ",
  StyleBox["s[V,I]",
    FontWeight->"Bold"],
  " where ",
  StyleBox["s",
    FontWeight->"Bold"],
  " is a binary constructor over reals. We also find useful for the type \
checker to define a particular type, say ",
  StyleBox["sig",
    FontWeight->"Bold"],
  ", for the terms denoting electric signals. \nFirst, we define the type \
constructor ",
  StyleBox["sig",
    FontWeight->"Bold"],
  ":"
}], "Text"],

Cell[BoxData[
    \(\(TypeConstructor[sig]; \)\)], "Input",
  PageWidth->Infinity,
  InitializationCell->True,
  ShowSpecialCharacters->False,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[TextData[{
  "and next we impose type ",
  StyleBox["sig",
    FontWeight->"Bold"],
  " on signal terms"
}], "Text"],

Cell[BoxData[
    RowBox[{
      RowBox[{"Constructor", "[", 
        StyleBox[\(TypeOf[s, FuncType[Cross[Real, Real], sig]]\),
          ShowSpecialCharacters->False,
          ShowStringCharacters->True], "]"}], ";"}]], "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Electrical components", "Subsubtitle"],

Cell[TextData[{
  "We  model ",
  StyleBox["electrical components",
    FontSlant->"Italic"],
  " by predicates that express the relationship between the input and output \
signal of a system as a conjunction of equations  in the components of \
input/output signals,  and two extra variables: \n\n\[Bullet] a variable ",
  StyleBox["S",
    FontWeight->"Bold"],
  " called system frequency, which characterizes the electrical sytem as a \
whole,\n\[Bullet] a variable ",
  StyleBox["X",
    FontWeight->"Bold"],
  " that is a characteristic parameter to a particular system. \n \n Both ",
  StyleBox["S",
    FontWeight->"Bold"],
  " and ",
  StyleBox["X",
    FontWeight->"Bold"],
  " range over ",
  StyleBox["Real.",
    FontWeight->"Bold"],
  "\n We distinguish two kinds of electrical components:\n \[Diamond] ",
  StyleBox["unary",
    FontSlant->"Italic"],
  " components: these components get an input signal ",
  StyleBox["s[V0, I0]",
    FontWeight->"Bold"],
  " and produce an output signal ",
  StyleBox["s[V1, I1]",
    FontWeight->"Bold"],
  ";\n To this type belong the resistors, capacitors, inductors, and sources.\
\n \[Diamond] ",
  StyleBox["constant",
    FontSlant->"Italic"],
  " components: these components emit an output signal ",
  StyleBox["s[V1, I1]",
    FontWeight->"Bold"],
  " without receiving any input."
}], "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[{
  StyleBox["Unary",
    FontSlant->"Italic"],
  "  components"
}], "Subsubtitle"],

Cell["Are modelled by predicates of the form ", "Text"],

Cell[TextData[{
  StyleBox["C[ Z,S, s[V0, I0], s[V1, I1]]",
    FontWeight->"Bold"],
  " "
}], "Text",
  TextAlignment->Center],

Cell["where", "Text"],

Cell[TextData[{
  " ",
  StyleBox["s[V0, I0]",
    FontWeight->"Bold"],
  " is the input signal \n ",
  StyleBox["s[V1, I1]",
    FontWeight->"Bold"],
  " is the output signal of the system\n ",
  StyleBox["S",
    FontWeight->"Bold"],
  " is a ",
  StyleBox["Real",
    FontWeight->"Bold"],
  " variable (the frequency variable) that plays role in expressing the \
relationship between ",
  StyleBox["V0, I0, V1, I1",
    FontWeight->"Bold"],
  "\n",
  StyleBox["Z",
    FontWeight->"Bold"],
  " is a ",
  StyleBox["Real",
    FontWeight->"Bold"],
  " constant that characterizes the system ",
  StyleBox["C",
    FontWeight->"Bold"],
  ". "
}], "Text"],

Cell["\<\
In this setting we can model resistors, inductors, capacitors and \
sources . All of them have the same type signature .\
\>", "Text"],

Cell[CellGroupData[{

Cell["Type Signatures", "Subsubsection"],

Cell[TextData[{
  "We impose the type  Real\[Cross]Real\[Cross]sig\[Cross]sig\
\[ShortRightArrow]Bool to the predicates ",
  StyleBox["resistor",
    FontSlant->"Italic"],
  ", ",
  StyleBox["inductor",
    FontSlant->"Italic"],
  ", ",
  StyleBox["capacitor",
    FontSlant->"Italic"],
  " and ",
  StyleBox["source",
    FontSlant->"Italic"],
  "."
}], "Text"],

Cell[BoxData[
    \(\(DefinedFunctionSymbol[\n\t
      TypeOf[resistor, FuncType[Cross[Real, Real, sig, sig], Bool]], \n\t
      TypeOf[inductor, FuncType[Cross[Real, Real, sig, sig], Bool]], \n\t
      TypeOf[capacitor, FuncType[Cross[Real, Real, sig, sig], Bool]], \n\t
      TypeOf[source, FuncType[Cross[Real, Real, sig, sig], Bool]]\n]; \)\)], 
  "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Definitions", "Subsubsection"],

Cell[TextData[{
  "We proceed by defining the characterizing predicates for resistors, \
capacitors, inductors, and sources. The CFLP language constructs for this \
purpose are the conditional rewrite rules.\nA ",
  StyleBox["resistor ",
    FontSlant->"Italic"],
  " with characteristic ",
  StyleBox["R,",
    FontWeight->"Bold"],
  " input signal",
  StyleBox[" s[V0,I0]",
    FontWeight->"Bold"],
  " and output signal ",
  StyleBox["s[V1,I1]",
    FontWeight->"Bold"],
  " is an electric system  that satisfies the following equations"
}], "Text"],

Cell["\<\
-V0+V1 = R * I0
I0 = I1 \
\>", "Text"],

Cell[TextData[{
  "We model such a system by CFLP conditional rewrite rules. The conditional \
rewrite rule for this system is\n\n",
  StyleBox[
  "resistor[R,S,s[V0,I0],s[V1,I1]]\[RightArrow]True\[DoubleLeftArrow]-V0+V1\
\[TildeTilde]I0 R&&I0\[TildeTilde]I1",
    FontWeight->"Bold"],
  "\n\nAnd is declared with the command ",
  StyleBox["Def",
    FontWeight->"Bold"],
  ". "
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(Def[{R, S, V0, I0, V1, I1}, \n
      \t{RewriteRule[resistor[R, S, s[V0, I0], s[V1, I1]], True, 
          And[eq[\(-\ V0\) + V1, R\ *\ I0], eq[I0, I1]]]}]; \)\)], "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    InterpretationBox[
      \("Adding rule #"\[InvisibleSpace]1
        \[InvisibleSpace]": "\[InvisibleSpace]\(resistor[R, S, s[V0, I0], 
            s[V1, I1]] \[RightArrow] True \[DoubleLeftArrow] 
          \(\(-V0\) + V1 \[TildeTilde] I0\ R && I0 \[TildeTilde] I1\)\)\),
      SequenceForm[ "Adding rule #", 1, ": ", 
        RewriteRule[ 
          resistor[ R, S, 
            s[ V0, I0], 
            s[ V1, I1]], True, 
          And[ 
            eq[ 
              Plus[ 
                Times[ -1, V0], V1], 
              Times[ I0, R]], 
            eq[ I0, I1]]]],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      RowBox[{
      \("Symbol"\), "\[InvisibleSpace]", \("s"\), "\[InvisibleSpace]", 
        \(" "\), "\[InvisibleSpace]", 
        InterpretationBox[\(I0, I1, R, S, V0, V1\),
          Infix[ {I0, I1, R, S, V0, V1}, ","],
          Editable->False], "\[InvisibleSpace]", \(" declared as  variable"\),
         "\[InvisibleSpace]", \("s"\), "\[InvisibleSpace]", \("."\)}],
      SequenceForm[ "Symbol", "s", " ", 
        Infix[ {I0, I1, R, S, V0, V1}, ","], " declared as  variable", "s", 
        "."],
      Editable->False]], "Print"]
}, Open  ]],

Cell[TextData[{
  "An ",
  StyleBox["inductor ",
    FontSlant->"Italic"],
  " with characteristic ",
  StyleBox["L,",
    FontWeight->"Bold"],
  " input signal",
  StyleBox[" s[V0,I0]",
    FontWeight->"Bold"],
  " and output signal ",
  StyleBox["s[V1,I1]",
    FontWeight->"Bold"],
  " is an electric system  that satisfies the following equations\n\n\
V0-V1=L*S*I0\nI0=I1\n\nThe corresponding",
  StyleBox[" Def ",
    FontWeight->"Bold"],
  "command is:"
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(Def[{L, S, V0, I0, V1, I1}, \n
      \t{RewriteRule[inductor[L, S, s[V0, I0], s[V1, I1]], True, 
          And[eq[V0 - V1, L*S*I0], eq[I0, I1]]]}]; \)\)], "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    InterpretationBox[
      \("Adding rule #"\[InvisibleSpace]1
        \[InvisibleSpace]": "\[InvisibleSpace]\(inductor[L, S, s[V0, I0], 
            s[V1, I1]] \[RightArrow] True \[DoubleLeftArrow] 
          \(V0 - V1 \[TildeTilde] I0\ L\ S && I0 \[TildeTilde] I1\)\)\),
      SequenceForm[ "Adding rule #", 1, ": ", 
        RewriteRule[ 
          inductor[ L, S, 
            s[ V0, I0], 
            s[ V1, I1]], True, 
          And[ 
            eq[ 
              Plus[ V0, 
                Times[ -1, V1]], 
              Times[ I0, L, S]], 
            eq[ I0, I1]]]],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      RowBox[{
      \("Symbol"\), "\[InvisibleSpace]", \("s"\), "\[InvisibleSpace]", 
        \(" "\), "\[InvisibleSpace]", 
        InterpretationBox[\(I0, I1, L, S, V0, V1\),
          Infix[ {I0, I1, L, S, V0, V1}, ","],
          Editable->False], "\[InvisibleSpace]", \(" declared as  variable"\),
         "\[InvisibleSpace]", \("s"\), "\[InvisibleSpace]", \("."\)}],
      SequenceForm[ "Symbol", "s", " ", 
        Infix[ {I0, I1, L, S, V0, V1}, ","], " declared as  variable", "s", 
        "."],
      Editable->False]], "Print"]
}, Open  ]],

Cell[TextData[{
  "A ",
  StyleBox["capacitor",
    FontSlant->"Italic"],
  "  with characteristic ",
  StyleBox["C,",
    FontWeight->"Bold"],
  " input signal",
  StyleBox[" s[V0,I0]",
    FontWeight->"Bold"],
  " and output signal ",
  StyleBox["s[V1,I1] ",
    FontWeight->"Bold"],
  "satisfies\n\nV0-V1=I0/C\nI0=I1\n\nand is modeled with the conditional \
rewrite rule \n\n",
  Cell[BoxData[
      StyleBox[
        \(capacitor[C, S, s[V0, I0], s[V1, I1]] \[RightArrow] True 
            \[DoubleLeftArrow] V0 - V1 \[TildeTilde] I0\/C && 
          I0 \[TildeTilde] I1\),
        FontWeight->"Bold"]]]
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(Def[{C, S, V0, I0, V1, I1}, \n
      \t{RewriteRule[capacitor[C, S, s[V0, I0], s[V1, I1]], True, 
          And[eq[V0 - V1, I0/C], eq[I0, I1]]]}]; \)\)], "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    InterpretationBox[
      \("Adding rule #"\[InvisibleSpace]1
        \[InvisibleSpace]": "\[InvisibleSpace]\(capacitor[C, S, s[V0, I0], 
            s[V1, I1]] \[RightArrow] True \[DoubleLeftArrow] 
          \(V0 - V1 \[TildeTilde] I0\/C && I0 \[TildeTilde] I1\)\)\),
      SequenceForm[ "Adding rule #", 1, ": ", 
        RewriteRule[ 
          capacitor[ C, S, 
            s[ V0, I0], 
            s[ V1, I1]], True, 
          And[ 
            eq[ 
              Plus[ V0, 
                Times[ -1, V1]], 
              Times[ 
                Power[ C, -1], I0]], 
            eq[ I0, I1]]]],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      RowBox[{
      \("Symbol"\), "\[InvisibleSpace]", \("s"\), "\[InvisibleSpace]", 
        \(" "\), "\[InvisibleSpace]", 
        InterpretationBox[\(C, I0, I1, S, V0, V1\),
          Infix[ {C, I0, I1, S, V0, V1}, ","],
          Editable->False], "\[InvisibleSpace]", \(" declared as  variable"\),
         "\[InvisibleSpace]", \("s"\), "\[InvisibleSpace]", \("."\)}],
      SequenceForm[ "Symbol", "s", " ", 
        Infix[ {C, I0, I1, S, V0, V1}, ","], " declared as  variable", "s", 
        "."],
      Editable->False]], "Print"]
}, Open  ]],

Cell[TextData[{
  "A ",
  StyleBox["source",
    FontSlant->"Italic"],
  " with characteristic ",
  StyleBox["E",
    FontWeight->"Bold"],
  ", input signal",
  StyleBox[" s[V0,I0]",
    FontWeight->"Bold"],
  " and output signal ",
  StyleBox["s[V1,I1] ",
    FontWeight->"Bold"],
  "satisfies\n\nV0-V1=E\nI0=I1\n\nand is modelled with the conditional \
rewrite rule\n\n",
  StyleBox[
  "source[E,S,s[V0,I0],s[V1,I1]]\[RightArrow]True\[DoubleLeftArrow]V0-V1\
\[TildeTilde]E&&I0\[TildeTilde]I1.",
    FontWeight->"Bold"]
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(Def[{E, S, V0, I0, V1, I1}, \n
      \t{RewriteRule[source[E, S, s[V0, I0], s[V1, I1]], True, 
          And[eq[V0 - V1, E], eq[I0, I1]]]}]; \)\)], "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    InterpretationBox[
      \("Adding rule #"\[InvisibleSpace]1
        \[InvisibleSpace]": "\[InvisibleSpace]\(source[E, S, s[V0, I0], 
            s[V1, I1]] \[RightArrow] True \[DoubleLeftArrow] 
          \(V0 - V1 \[TildeTilde] E && I0 \[TildeTilde] I1\)\)\),
      SequenceForm[ "Adding rule #", 1, ": ", 
        RewriteRule[ 
          source[ E, S, 
            s[ V0, I0], 
            s[ V1, I1]], True, 
          And[ 
            eq[ 
              Plus[ V0, 
                Times[ -1, V1]], E], 
            eq[ I0, I1]]]],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      RowBox[{
      \("Symbol"\), "\[InvisibleSpace]", \("s"\), "\[InvisibleSpace]", 
        \(" "\), "\[InvisibleSpace]", 
        InterpretationBox[\(E, I0, I1, S, V0, V1\),
          Infix[ {E, I0, I1, S, V0, V1}, ","],
          Editable->False], "\[InvisibleSpace]", \(" declared as  variable"\),
         "\[InvisibleSpace]", \("s"\), "\[InvisibleSpace]", \("."\)}],
      SequenceForm[ "Symbol", "s", " ", 
        Infix[ {E, I0, I1, S, V0, V1}, ","], " declared as  variable", "s", 
        "."],
      Editable->False]], "Print"]
}, Open  ]],

Cell[TextData[{
  "All electric systems considered so far are ",
  StyleBox["unary",
    FontSlant->"Italic"],
  " in the sense that they receive just ",
  StyleBox["one",
    FontSlant->"Italic"],
  " input signal and send ",
  StyleBox["one",
    FontSlant->"Italic"],
  " output signal. We may also assume the degenerate case when no input \
signal exists. These types of electric systems we call ",
  StyleBox["unidirectional",
    FontSlant->"Italic"],
  "."
}], "Text"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[{
  StyleBox["Constant",
    FontSlant->"Italic"],
  " components"
}], "Subsubtitle"],

Cell[TextData[{
  "are modelled by predicates of the form ",
  StyleBox["C[ Z, S, s[V1, I1]]",
    FontWeight->"Bold"],
  " where  ",
  StyleBox["s[V1, I1]",
    FontWeight->"Bold"],
  " is the output signal of the system. ",
  StyleBox["S",
    FontWeight->"Bold"],
  " and ",
  StyleBox["Z",
    FontWeight->"Bold"],
  " have the same meaning as before.\nWe consider ",
  StyleBox["earth",
    FontSlant->"Italic"],
  " as a unidirectional system."
}], "Text"],

Cell[CellGroupData[{

Cell["Type Signature", "Subsubsection"],

Cell[TextData[{
  "We impose the type Real\[Cross]Real\[Cross]sig\[ShortRightArrow]Bool to ",
  StyleBox["earth",
    FontWeight->"Bold"],
  "."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(DefinedFunctionSymbol[\n\t
      TypeOf[earth, FuncType[Cross[Real, Real, sig], Bool]]]\)], "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    \({earth}\)], "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Definition", "Subsubsection"],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(Def[{E, S, V1, I1}, \n
      \t{RewriteRule[earth[E, S, s[V1, I1]], True, eq[V1, 0]]}]; \)\)], 
  "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    InterpretationBox[
      \("Adding rule #"\[InvisibleSpace]1
        \[InvisibleSpace]": "\[InvisibleSpace]\(earth[E, S, s[V1, I1]] 
          \[RightArrow] True \[DoubleLeftArrow] V1 \[TildeTilde] 0\)\),
      SequenceForm[ "Adding rule #", 1, ": ", 
        RewriteRule[ 
          earth[ E, S, 
            s[ V1, I1]], True, 
          eq[ V1, 0]]],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      RowBox[{
      \("Symbol"\), "\[InvisibleSpace]", \("s"\), "\[InvisibleSpace]", 
        \(" "\), "\[InvisibleSpace]", 
        InterpretationBox[\(E, I1, S, V1\),
          Infix[ {E, I1, S, V1}, ","],
          Editable->False], "\[InvisibleSpace]", \(" declared as  variable"\),
         "\[InvisibleSpace]", \("s"\), "\[InvisibleSpace]", \("."\)}],
      SequenceForm[ "Symbol", "s", " ", 
        Infix[ {E, I1, S, V1}, ","], " declared as  variable", "s", "."],
      Editable->False]], "Print"]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[StyleBox["Component Instances",
  FontSlant->"Italic"]], "Subsubtitle"],

Cell[TextData[{
  "A ",
  StyleBox["component instance",
    FontSlant->"Italic"],
  " of an electric system is the electric component for which the \
characterizing value is specified. We model this concept with lambda terms. "
}], "Text"],

Cell[CellGroupData[{

Cell["Definitions", "Subsubsection"],

Cell["A resistor instance:", "Text"],

Cell[BoxData[
    \(RInst[x_] := \n\t
      Module[{S, L, R}, \n\t\t\[Lambda][{S, L, R}, resistor[x, S, L, R]]\n\t]
        \)], "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell["A capacitor instance:", "Text"],

Cell[BoxData[
    \(CInst[x_] := \n\t
      Module[{S, L, R}, \n\t\t\[Lambda][{S, L, R}, capacitor[x, S, L, R]]\n\t]
        \)], "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell["An inductor instance:", "Text"],

Cell[BoxData[
    \(LInst[x_] := \n\t
      Module[{S, L, R}, \n\t\t\[Lambda][{S, L, R}, inductor[x, S, L, R]]\n\t]
        \)], "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell["A source instance:", "Text"],

Cell[BoxData[
    \(EInst[x_] := \n\t
      Module[{S, L, R}, \n\t\t\[Lambda][{S, L, R}, source[x, S, L, R]]\n\t]
        \)], "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[TextData[{
  "Note that we wrapped all CFLP lambda-terms in some ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " Module constructs. The reason for this is to quantify the quantities S, \
L, and R.  For example, EInst[E] invoked twice generates two different \
instances of the lambda-term intended for a source instance:\n\n\
\[Lambda][{S$496,L$496,R$496},source[E,S$496,L$496,R$496]] \n\
\[Lambda][{S$497,L$497,R$497},source[E,S$497,L$497,R$497]]"
}], "Text"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[StyleBox["Electrical connections",
  FontSize->15,
  FontWeight->"Bold"]], "Subsubtitle"],

Cell["\<\
We describe now the main operations used for building up electrical \
sytems starting from electrical components. \
\>", "Text"],

Cell[CellGroupData[{

Cell["Serial connections", "Section"],

Cell[TextData[{
  " The serial connection of  ",
  StyleBox["n",
    FontSlant->"Italic"],
  " electric components is of the form "
}], "Text"],

Cell["serial[S,Clist,InputSignal,OutputSignal]", "Text",
  TextAlignment->Center],

Cell[TextData[
"where\n\[Diamond] S is the frequency variable of the system, \n\[Diamond] \
CList is a CFLP list containing the instances of electric components that are \
connected in serial,\n\[Diamond] InputSignal is the input signal\n\[Diamond] \
OutputSignal is the output signal"], "Text"],

Cell["\<\
We give below the type signature and the definition of this \
predicate.\
\>", "Text"],

Cell[CellGroupData[{

Cell["Signature(s)", "Subsubsection"],

Cell[CellGroupData[{

Cell[BoxData[
    \(DefinedFunctionSymbol[\n\t
      TypeOf[serial, 
        FuncType[
          Cross[Real, TyList[FuncType[Cross[Real, sig, sig], Bool]], sig, 
            sig], Bool]]\n\t]\)], "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    \({serial}\)], "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Definition(s)", "Subsubsection"],

Cell[TextData[{
  "We define the behavior of a serial connection by means of two conditional \
rewrite rules\n\n",
  StyleBox[
  "serial[S,\[Bullet],L,R]\[RightArrow]True\[DoubleLeftArrow]L\[TildeTilde]R\n\
serial[S,\[LeftAngleBracket]C1,CS\[RightAngleBracket],L,R]\[RightArrow]True\
\[DoubleLeftArrow]C1[S,L,L1]\[TildeTilde]True&&serial[S,CS,L1,R]\[TildeTilde]\
True\n",
    FontWeight->"Bold"],
  "\nThe first rule specifies that the input signal ",
  StyleBox["L",
    FontWeight->"Bold"],
  " coincides with the output signal ",
  StyleBox["R",
    FontWeight->"Bold"],
  " when no components are connected. The second rule stands for the \
recursive definition of a serial connection."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    RowBox[{
      RowBox[{"Def", "[", 
        RowBox[{\({S, L, L1, R, C1, CS}\), ",", "\n", "\t", 
          RowBox[{"{", 
            RowBox[{
              
              StyleBox[
                \(RewriteRule[serial[S, \[Bullet], L, R], True, eq[L, R]]\),
                ShowSpecialCharacters->False,
                ShowStringCharacters->True], ",", "\n", "\t\t", 
              
              StyleBox[
                \(RewriteRule[serial[S, Cons[C1, CS], L, R], True, 
                  And[eq[C1[S, L, L1], True], eq[serial[S, CS, L1, R], True]]]
                  \),
                ShowSpecialCharacters->False,
                ShowStringCharacters->True]}], "}"}]}], "]"}], ";"}]], "Input",\

  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    InterpretationBox[
      \("Adding rule #"\[InvisibleSpace]1
        \[InvisibleSpace]": "\[InvisibleSpace]\(serial[S, \[Bullet], L, R] 
          \[RightArrow] True \[DoubleLeftArrow] L \[TildeTilde] R\)\),
      SequenceForm[ "Adding rule #", 1, ": ", 
        RewriteRule[ 
          serial[ S, \[Bullet], L, R], True, 
          eq[ L, R]]],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      RowBox[{
      \("Symbol"\), "\[InvisibleSpace]", \("s"\), "\[InvisibleSpace]", 
        \(" "\), "\[InvisibleSpace]", 
        InterpretationBox[\(L, R, S\),
          Infix[ {L, R, S}, ","],
          Editable->False], "\[InvisibleSpace]", \(" declared as  variable"\),
         "\[InvisibleSpace]", \("s"\), "\[InvisibleSpace]", \("."\)}],
      SequenceForm[ "Symbol", "s", " ", 
        Infix[ {L, R, S}, ","], " declared as  variable", "s", "."],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      \("Adding rule #"\[InvisibleSpace]2
        \[InvisibleSpace]": "\[InvisibleSpace]\(serial[S, 
            \[LeftAngleBracket]C1, CS\[RightAngleBracket], L, R] 
          \[RightArrow] True \[DoubleLeftArrow] 
          \(C1[S, L, L1] \[TildeTilde] True && 
            serial[S, CS, L1, R] \[TildeTilde] True\)\)\),
      SequenceForm[ "Adding rule #", 2, ": ", 
        RewriteRule[ 
          serial[ S, 
            Cons[ C1, CS], L, R], True, 
          And[ 
            eq[ 
              C1[ S, L, L1], True], 
            eq[ 
              serial[ S, CS, L1, R], True]]]],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      RowBox[{
      \("Symbol"\), "\[InvisibleSpace]", \("s"\), "\[InvisibleSpace]", 
        \(" "\), "\[InvisibleSpace]", 
        InterpretationBox[\(C1, CS, L, L1, R, S\),
          Infix[ {C1, CS, L, L1, R, S}, ","],
          Editable->False], "\[InvisibleSpace]", \(" declared as  variable"\),
         "\[InvisibleSpace]", \("s"\), "\[InvisibleSpace]", \("."\)}],
      SequenceForm[ "Symbol", "s", " ", 
        Infix[ {C1, CS, L, L1, R, S}, ","], " declared as  variable", "s", 
        "."],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      \("Symbol"\[InvisibleSpace]""\[InvisibleSpace]" "\[InvisibleSpace]L1
        \[InvisibleSpace]" declared as extra variable"\[InvisibleSpace]""\
\[InvisibleSpace]"."\),
      SequenceForm[ 
      "Symbol", "", " ", L1, " declared as extra variable", "", "."],
      Editable->False]], "Print"]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Parallel connections", "Section"],

Cell[TextData[{
  "For modelling parallel connections we first introduce the concept of ",
  StyleBox["electrical joint",
    FontSlant->"Italic"],
  " of electrical signals. The predicate "
}], "Text"],

Cell[TextData[{
  StyleBox["elJoin[Signals,V]",
    FontWeight->"Bold"],
  " "
}], "Text",
  TextAlignment->Center],

Cell[TextData[{
  "joins a list ",
  StyleBox["Signals",
    FontWeight->"Bold"],
  " of signals",
  StyleBox[" s[",
    FontWeight->"Bold"],
  Cell[BoxData[
      FormBox[
        SubscriptBox[
          StyleBox["I",
            FontSlant->"Plain"], "i"], TraditionalForm]],
    FontWeight->"Bold"],
  StyleBox[",",
    FontWeight->"Bold"],
  Cell[BoxData[
      FormBox[
        SubscriptBox[
          StyleBox["V",
            FontSlant->"Plain"], "i"], TraditionalForm]],
    FontWeight->"Bold"],
  StyleBox["]",
    FontWeight->"Bold"],
  " where ",
  StyleBox["V",
    FontWeight->"Bold"],
  " equals all ",
  Cell[BoxData[
      FormBox[
        SubscriptBox[
          StyleBox["V",
            FontSlant->"Plain"], "i"], TraditionalForm]],
    FontWeight->"Bold"],
  ", and returns the sum of currents ",
  StyleBox["\[Sum] ",
    FontWeight->"Bold"],
  Cell[BoxData[
      FormBox[
        SubscriptBox[
          StyleBox["I",
            FontSlant->"Plain"], "i"], TraditionalForm]],
    FontWeight->"Bold"],
  "."
}], "Text"],

Cell[CellGroupData[{

Cell["elJoin", "Subsection"],

Cell[CellGroupData[{

Cell["Signature", "Subsubsection"],

Cell[CellGroupData[{

Cell[BoxData[
    RowBox[{"DefinedFunctionSymbol", "[", "\n", "\t", 
      StyleBox[\(TypeOf[elJoin, FuncType[Cross[TyList[sig], Real], Real]]\),
        ShowSpecialCharacters->False,
        ShowStringCharacters->True], "\n", "\t", "]"}]], "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    \({elJoin}\)], "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Definitions", "Subsubsection"],

Cell[CellGroupData[{

Cell[BoxData[
    RowBox[{
      RowBox[{"Def", "[", 
        RowBox[{\({S, V, I0, V0, RS}\), ",", "\n", "\t", 
          RowBox[{"{", 
            RowBox[{
              StyleBox[\(RewriteRule[elJoin[\[Bullet], V], 0, True]\),
                ShowSpecialCharacters->False,
                ShowStringCharacters->True], 
              StyleBox[",",
                ShowSpecialCharacters->False,
                ShowStringCharacters->True], 
              StyleBox["\n",
                ShowSpecialCharacters->False,
                ShowStringCharacters->True], 
              StyleBox["\t\t",
                ShowSpecialCharacters->False,
                ShowStringCharacters->True], 
              TagBox[
                
                StyleBox[
                  \(RewriteRule[elJoin[Cons[s[V0, I0], RS], V], 
                    Plus[I0, elJoin[RS, V]], eq[V0, V]]\),
                  ShowSpecialCharacters->False,
                  ShowStringCharacters->True],
                FullForm]}], "}"}]}], "]"}], ";"}]], "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    InterpretationBox[
      \("Adding rule #"\[InvisibleSpace]1
        \[InvisibleSpace]": "\[InvisibleSpace]\(elJoin[\[Bullet], V] 
          \[RightArrow] 0\)\),
      SequenceForm[ "Adding rule #", 1, ": ", 
        RewriteRule[ 
          elJoin[ \[Bullet], V], 0, True]],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      \("Symbol"\[InvisibleSpace]""\[InvisibleSpace]" "\[InvisibleSpace]V
        \[InvisibleSpace]" declared as  variable"\[InvisibleSpace]""\
\[InvisibleSpace]"."\),
      SequenceForm[ "Symbol", "", " ", V, " declared as  variable", "", "."],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      \("Adding rule #"\[InvisibleSpace]2
        \[InvisibleSpace]": "\[InvisibleSpace]\(elJoin[
            \[LeftAngleBracket]s[V0, I0], RS\[RightAngleBracket], V] 
          \[RightArrow] I0 + elJoin[RS, V] \[DoubleLeftArrow] 
          V0 \[TildeTilde] V\)\),
      SequenceForm[ "Adding rule #", 2, ": ", 
        RewriteRule[ 
          elJoin[ 
            Cons[ 
              s[ V0, I0], RS], V], 
          Plus[ I0, 
            elJoin[ RS, V]], 
          eq[ V0, V]]],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      RowBox[{
      \("Symbol"\), "\[InvisibleSpace]", \("s"\), "\[InvisibleSpace]", 
        \(" "\), "\[InvisibleSpace]", 
        InterpretationBox[\(I0, RS, V, V0\),
          Infix[ {I0, RS, V, V0}, ","],
          Editable->False], "\[InvisibleSpace]", \(" declared as  variable"\),
         "\[InvisibleSpace]", \("s"\), "\[InvisibleSpace]", \("."\)}],
      SequenceForm[ "Symbol", "s", " ", 
        Infix[ {I0, RS, V, V0}, ","], " declared as  variable", "s", "."],
      Editable->False]], "Print"]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["junction", "Subsection"],

Cell[TextData[{
  "Next we define the ",
  StyleBox["junction",
    FontSlant->"Italic"],
  " of a list ",
  StyleBox["L",
    FontWeight->"Bold"],
  " of input signals with a list  ",
  StyleBox["R",
    FontWeight->"Bold"],
  " of output signals."
}], "Text"],

Cell[CellGroupData[{

Cell["Signature", "Subsubsection"],

Cell[CellGroupData[{

Cell[BoxData[
    RowBox[{"DefinedFunctionSymbol", "[", "\n", "\t", 
      StyleBox[
        \(TypeOf[junction, FuncType[Cross[TyList[sig], TyList[sig]], Bool]]\),
        
        ShowSpecialCharacters->False,
        ShowStringCharacters->True], "\n", "\t", "]"}]], "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    \({junction}\)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    RowBox[{
      RowBox[{"Def", "[", 
        RowBox[{\({L, R, V}\), ",", "\n", "\t", 
          RowBox[{"{", 
            
            StyleBox[
              \(RewriteRule[junction[L, R], True, 
                eq[elJoin[L, V], elJoin[R, V]]]\),
              ShowSpecialCharacters->False,
              ShowStringCharacters->True], "}"}]}], "]"}], ";"}]], "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    InterpretationBox[
      \("Adding rule #"\[InvisibleSpace]1
        \[InvisibleSpace]": "\[InvisibleSpace]\(junction[L, R] \[RightArrow] 
          True \[DoubleLeftArrow] elJoin[L, V] \[TildeTilde] elJoin[R, V]\)\),
      
      SequenceForm[ "Adding rule #", 1, ": ", 
        RewriteRule[ 
          junction[ L, R], True, 
          eq[ 
            elJoin[ L, V], 
            elJoin[ R, V]]]],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      RowBox[{
      \("Symbol"\), "\[InvisibleSpace]", \("s"\), "\[InvisibleSpace]", 
        \(" "\), "\[InvisibleSpace]", 
        InterpretationBox[\(L, R, V\),
          Infix[ {L, R, V}, ","],
          Editable->False], "\[InvisibleSpace]", \(" declared as  variable"\),
         "\[InvisibleSpace]", \("s"\), "\[InvisibleSpace]", \("."\)}],
      SequenceForm[ "Symbol", "s", " ", 
        Infix[ {L, R, V}, ","], " declared as  variable", "s", "."],
      Editable->False]], "Print"],

Cell[BoxData[
    InterpretationBox[
      \("Symbol"\[InvisibleSpace]""\[InvisibleSpace]" "\[InvisibleSpace]V
        \[InvisibleSpace]" declared as extra variable"\[InvisibleSpace]""\
\[InvisibleSpace]"."\),
      SequenceForm[ 
      "Symbol", "", " ", V, " declared as extra variable", "", "."],
      Editable->False]], "Print"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Definition", "Subsubsection"],

Cell[TextData[{
  "Like serial connections, parallel connections of ",
  StyleBox["n",
    FontSlant->"Italic"],
  " components C1, \[Ellipsis], Cn can also be  expressed recursively. The \
predicate "
}], "Text"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["parallel", "Subsection"],

Cell[TextData[{
  " The parallel connection of ",
  StyleBox["n",
    FontSlant->"Italic"],
  " electric components is of the form "
}], "Text"],

Cell["parallel[S,Clist,InputSignal,OutputSignal]", "Text",
  TextAlignment->Center],

Cell[TextData[
"where\n\[Diamond] S is the frequency variable of the system, \n\[Diamond] \
CList is a CFLP list containing the instances of electric components that are \
connected in parallel,\n\[Diamond] InputSignal is the input signal\n\
\[Diamond] OutputSignal is the output signal"], "Text"],

Cell[CellGroupData[{

Cell["Signature", "Subsubsection"],

Cell[CellGroupData[{

Cell[BoxData[
    RowBox[{"DefinedFunctionSymbol", "[", "\n", "\t", 
      StyleBox[
        \(TypeOf[parallel, 
          FuncType[
            Cross[Real, TyList[FuncType[Cross[Real, sig, sig], Bool]], sig, 
              sig], Bool]]\),
        ShowSpecialCharacters->False,
        ShowStringCharacters->True], "\n", "\t", "]"}]], "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    \({parallel}\)], "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Definitions", "Subsubsection"],

Cell[TextData[{
  "We define the behavior of a parallel connection by means of two \
conditional rewrite rules\n\n",
  StyleBox[
  "parallel[S,\[LeftAngleBracket]C,\[Bullet]\[RightAngleBracket],InS,OutS]\
\[RightArrow]True\[DoubleLeftArrow]C[S,InS,OutS]\[TildeTilde]True",
    FontWeight->"Bold"],
  "\n",
  StyleBox[
  "parallel[S,\[LeftAngleBracket]C,Cs\[RightAngleBracket],InS,OutS]\
\[RightArrow]True\[DoubleLeftArrow]\n\tC[S,InS1,OutS1]\[TildeTilde]True&&\n   \
 \tparallel[S,Cs,InS2,OutS2]\[TildeTilde]True&&\n    \tjunction[\
\[LeftAngleBracket]InS,\[Bullet]\[RightAngleBracket],\[LeftAngleBracket]InS1,\
\[LeftAngleBracket]InS2,\[Bullet]\[RightAngleBracket]\[RightAngleBracket]]\
\[TildeTilde]True&&\n    \tjunction[\[LeftAngleBracket]OutS1,\
\[LeftAngleBracket]OutS2,\[Bullet]\[RightAngleBracket]\[RightAngleBracket],\
\[LeftAngleBracket]OutS,\[Bullet]\[RightAngleBracket]]\[TildeTilde]True\n",
    FontWeight->"Bold"],
  "\nA difference to the definition of serial connections is the fact that \
for parallel connection we have the base case when only one component is \
connected."
}], "Text"],

Cell[BoxData[
    RowBox[{
      RowBox[{"Def", "[", 
        RowBox[{
        \({S, C, Cs, InS, OutS, InS1, OutS1, InS2, OutS2}\), ",", "\n", "\t", 
          
          RowBox[{"{", 
            
            StyleBox[
              \(RewriteRule[parallel[S, Cons[C, \[Bullet]], InS, OutS], True, 
                eq[C[S, InS, OutS], True]], 
              RewriteRule[parallel[S, Cons[C, Cs], InS, OutS], True, 
                And[eq[C[S, InS1, OutS1], True], 
                  eq[parallel[S, Cs, InS2, OutS2], True], \n\t\t\t\t
                  eq[junction[Cons[InS, \[Bullet]], 
                      Cons[InS1, Cons[InS2, \[Bullet]]]], True], \n\t\t\t\t
                  eq[junction[Cons[OutS1, Cons[OutS2, \[Bullet]]], 
                      Cons[OutS, \[Bullet]]], True]]]\),
              ShowSpecialCharacters->False,
              ShowStringCharacters->True], 
            StyleBox["}",
              ShowSpecialCharacters->False,
              ShowStringCharacters->True]}]}], "]"}], ";"}]], "Input",
  InitializationCell->True,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[BoxData[
    RowBox[{"Def", "[", 
      RowBox[{
      \({S, C, Cs, InS, OutS, InS1, OutS1, InS2, OutS2}\), ",", "\n", "\t", 
        RowBox[{"{", 
          
          StyleBox[
            \(RewriteRule[parallel[S, {C}, InS, OutS], C[S, InS, OutS], 
              True], RewriteRule[parallel[S, Cons[C, Cs], InS, OutS], True, 
              And[eq[C[S, InS1, OutS1], True], 
                eq[parallel[S, Cs, InS2, OutS2], True], \n\t\t\t\t
                eq[junction[Cons[InS, \[Bullet]], 
                    Cons[InS1, Cons[InS2, \[Bullet]]]], True], \n\t\t\t\t
                eq[junction[Cons[OutS1, Cons[OutS2, \[Bullet]]], 
                    Cons[OutS, \[Bullet]]], True]]]\),
            ShowSpecialCharacters->False,
            ShowStringCharacters->True], 
          StyleBox["}",
            ShowSpecialCharacters->False,
            ShowStringCharacters->True]}]}], "]"}]], "Input"]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["End", "Subsubtitle"],

Cell[BoxData[
    \(On[General::spell]\)], "Input",
  InitializationCell->True,
  FontColor->RGBColor[0.6, 0, 0]],

Cell[BoxData[
    \(\(End[]; \)\)], "Input",
  PageWidth->Infinity,
  InitializationCell->True,
  ShowSpecialCharacters->False,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[TextData[{
  "This command reverts to the previous context, in this case",
  StyleBox[" Electrical`.",
    FontWeight->"Bold"]
}], "Text"],

Cell[BoxData[
    \(\(EndPackage[]; \)\)], "Input",
  PageWidth->Infinity,
  InitializationCell->True,
  ShowSpecialCharacters->False,
  FontColor->RGBColor[0, 0, 0.500008]],

Cell[TextData[{
  "This ends the package",
  StyleBox[" Electric`",
    FontWeight->"Bold"],
  ", prepending it to the context search path."
}], "Text"]
}, Open  ]]
}, Open  ]]
}, Open  ]]
},
FrontEndVersion->"X 3.0",
ScreenRectangle->{{0, 1280}, {0, 1024}},
AutoGeneratedPackage->Automatic,
WindowSize->{778, 734},
WindowMargins->{{201, Automatic}, {72, Automatic}},
PrintingPageRange->{Automatic, Automatic},
PrintingOptions->{"PaperSize"->{612, 792},
"PaperOrientation"->"Portrait",
"Magnification"->1},
Magnification->1
]


(***********************************************************************
Cached data follows.  If you edit this Notebook file directly, not using
Mathematica, you must remove the line containing CacheID at the top of 
the file.  The cache data will then be recreated when you save this file 
from within Mathematica.
***********************************************************************)

(*CellTagsOutline
CellTagsIndex->{}
*)

(*CellTagsIndex
CellTagsIndex->{}
*)

(*NotebookFileOutline
Notebook[{

Cell[CellGroupData[{
Cell[1731, 51, 83, 1, 104, "Title"],
Cell[1817, 54, 171, 4, 27, "Input",
  InitializationCell->True],
Cell[1991, 60, 3225, 63, 683, "Input",
  InitializationCell->True],

Cell[CellGroupData[{
Cell[5241, 127, 78, 1, 54, "Subtitle"],
Cell[5322, 130, 1020, 17, 158, "Text"],

Cell[CellGroupData[{
Cell[6367, 151, 37, 0, 48, "Subsubtitle"],
Cell[6407, 153, 340, 7, 50, "Text"],

Cell[CellGroupData[{
Cell[6772, 164, 156, 3, 43, "Input",
  InitializationCell->True],
Cell[6931, 169, 71, 1, 23, "Print"],
Cell[7005, 172, 80, 1, 23, "Print"]
}, Open  ]],
Cell[7100, 176, 516, 13, 68, "Text"],
Cell[7619, 191, 71, 1, 23, "Print"],
Cell[7693, 194, 80, 1, 23, "Print"],
Cell[7776, 197, 352, 11, 50, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[8165, 213, 28, 0, 48, "Subsubtitle"],
Cell[8196, 215, 413, 15, 50, "Text"],
Cell[8612, 232, 114, 3, 27, "Input",
  InitializationCell->True],
Cell[8729, 237, 217, 4, 50, "Text"],
Cell[8949, 243, 103, 5, 32, "Text"],
Cell[9055, 250, 244, 5, 43, "Input",
  InitializationCell->True],
Cell[9302, 257, 244, 5, 43, "Input",
  InitializationCell->True],
Cell[9549, 264, 252, 5, 43, "Input",
  InitializationCell->True],
Cell[9804, 271, 252, 5, 43, "Input",
  InitializationCell->True],
Cell[10059, 278, 209, 5, 27, "Input",
  InitializationCell->True],
Cell[10271, 285, 239, 5, 43, "Input",
  InitializationCell->True],
Cell[10513, 292, 315, 6, 59, "Input",
  InitializationCell->True],
Cell[10831, 300, 198, 5, 27, "Input",
  InitializationCell->True],
Cell[11032, 307, 192, 5, 27, "Input",
  InitializationCell->True],
Cell[11227, 314, 200, 5, 27, "Input",
  InitializationCell->True],
Cell[11430, 321, 198, 5, 27, "Input",
  InitializationCell->True],
Cell[11631, 328, 124, 3, 27, "Input",
  InitializationCell->True],
Cell[11758, 333, 341, 7, 59, "Input",
  InitializationCell->True],
Cell[12102, 342, 345, 7, 59, "Input",
  InitializationCell->True]
}, Open  ]],

Cell[CellGroupData[{
Cell[12484, 354, 28, 0, 48, "Subsubtitle"],
Cell[12515, 356, 183, 5, 27, "Input",
  InitializationCell->True],
Cell[12701, 363, 139, 5, 32, "Text"],
Cell[12843, 370, 422, 9, 91, "Input",
  InitializationCell->True],
Cell[13268, 381, 457, 10, 68, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[13762, 396, 39, 0, 48, "Subsubtitle"],
Cell[13804, 398, 595, 19, 68, "Text"],
Cell[14402, 419, 181, 5, 27, "Input",
  InitializationCell->True],
Cell[14586, 426, 121, 5, 32, "Text"],
Cell[14710, 433, 297, 7, 27, "Input",
  InitializationCell->True]
}, Open  ]],

Cell[CellGroupData[{
Cell[15044, 445, 44, 0, 48, "Subsubtitle"],
Cell[15091, 447, 1352, 39, 212, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[16480, 491, 97, 4, 48, "Subsubtitle"],
Cell[16580, 497, 55, 0, 32, "Text"],
Cell[16638, 499, 127, 5, 32, "Text"],
Cell[16768, 506, 21, 0, 32, "Text"],
Cell[16792, 508, 654, 27, 86, "Text"],
Cell[17449, 537, 144, 3, 32, "Text"],

Cell[CellGroupData[{
Cell[17618, 544, 40, 0, 42, "Subsubsection"],
Cell[17661, 546, 362, 15, 32, "Text"],
Cell[18026, 563, 427, 8, 107, "Input",
  InitializationCell->True]
}, Open  ]],

Cell[CellGroupData[{
Cell[18490, 576, 36, 0, 42, "Subsubsection"],
Cell[18529, 578, 552, 16, 68, "Text"],
Cell[19084, 596, 48, 3, 50, "Text"],
Cell[19135, 601, 390, 11, 104, "Text"],

Cell[CellGroupData[{
Cell[19550, 616, 261, 5, 43, "Input",
  InitializationCell->True],
Cell[19814, 623, 628, 17, 23, "Print"],
Cell[20445, 642, 577, 12, 23, "Print"]
}, Open  ]],
Cell[21037, 657, 471, 18, 140, "Text"],

Cell[CellGroupData[{
Cell[21533, 679, 252, 5, 43, "Input",
  InitializationCell->True],
Cell[21788, 686, 629, 17, 23, "Print"],
Cell[22420, 705, 577, 12, 23, "Print"]
}, Open  ]],
Cell[23012, 720, 618, 21, 159, "Text"],

Cell[CellGroupData[{
Cell[23655, 745, 251, 5, 43, "Input",
  InitializationCell->True],
Cell[23909, 752, 654, 18, 40, "Print"],
Cell[24566, 772, 577, 12, 23, "Print"]
}, Open  ]],
Cell[25158, 787, 532, 19, 158, "Text"],

Cell[CellGroupData[{
Cell[25715, 810, 245, 5, 43, "Input",
  InitializationCell->True],
Cell[25963, 817, 588, 16, 23, "Print"],
Cell[26554, 835, 577, 12, 23, "Print"]
}, Open  ]],
Cell[27146, 850, 475, 15, 50, "Text"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[27670, 871, 99, 4, 48, "Subsubtitle"],
Cell[27772, 877, 462, 17, 68, "Text"],

Cell[CellGroupData[{
Cell[28259, 898, 39, 0, 42, "Subsubsection"],
Cell[28301, 900, 156, 5, 32, "Text"],

Cell[CellGroupData[{
Cell[28482, 909, 187, 4, 43, "Input",
  InitializationCell->True],
Cell[28672, 915, 41, 1, 27, "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[28762, 922, 35, 0, 42, "Subsubsection"],

Cell[CellGroupData[{
Cell[28822, 926, 195, 5, 43, "Input",
  InitializationCell->True],
Cell[29020, 933, 405, 10, 21, "Print"],
Cell[29428, 945, 544, 11, 21, "Print"]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[30033, 963, 85, 1, 46, "Subsubtitle"],
Cell[30121, 966, 240, 6, 44, "Text"],

Cell[CellGroupData[{
Cell[30386, 976, 36, 0, 40, "Subsubsection"],
Cell[30425, 978, 36, 0, 29, "Text"],
Cell[30464, 980, 204, 5, 72, "Input",
  InitializationCell->True],
Cell[30671, 987, 37, 0, 29, "Text"],
Cell[30711, 989, 205, 5, 72, "Input",
  InitializationCell->True],
Cell[30919, 996, 37, 0, 29, "Text"],
Cell[30959, 998, 204, 5, 72, "Input",
  InitializationCell->True],
Cell[31166, 1005, 34, 0, 29, "Text"],
Cell[31203, 1007, 202, 5, 72, "Input",
  InitializationCell->True],
Cell[31408, 1014, 476, 9, 106, "Text"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[31933, 1029, 103, 2, 49, "Subsubtitle"],
Cell[32039, 1033, 138, 3, 44, "Text"],

Cell[CellGroupData[{
Cell[32202, 1040, 37, 0, 51, "Section"],
Cell[32242, 1042, 143, 5, 29, "Text"],
Cell[32388, 1049, 81, 1, 29, "Text"],
Cell[32472, 1052, 295, 4, 106, "Text"],
Cell[32770, 1058, 96, 3, 29, "Text"],

Cell[CellGroupData[{
Cell[32891, 1065, 37, 0, 40, "Subsubsection"],

Cell[CellGroupData[{
Cell[32953, 1069, 271, 7, 72, "Input",
  InitializationCell->True],
Cell[33227, 1078, 42, 1, 25, "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[33318, 1085, 38, 0, 40, "Subsubsection"],
Cell[33359, 1087, 702, 17, 136, "Text"],

Cell[CellGroupData[{
Cell[34086, 1108, 789, 20, 87, "Input",
  InitializationCell->True],
Cell[34878, 1130, 397, 9, 21, "Print"],
Cell[35278, 1141, 529, 11, 21, "Print"],
Cell[35810, 1154, 659, 17, 35, "Print"],
Cell[36472, 1173, 574, 12, 21, "Print"],
Cell[37049, 1187, 335, 7, 21, "Print"]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[37445, 1201, 39, 0, 51, "Section"],
Cell[37487, 1203, 202, 5, 44, "Text"],
Cell[37692, 1210, 115, 5, 29, "Text"],
Cell[37810, 1217, 1040, 43, 44, "Text"],

Cell[CellGroupData[{
Cell[38875, 1264, 28, 0, 43, "Subsection"],

Cell[CellGroupData[{
Cell[38928, 1268, 34, 0, 40, "Subsubsection"],

Cell[CellGroupData[{
Cell[38987, 1272, 316, 6, 56, "Input",
  InitializationCell->True],
Cell[39306, 1280, 42, 1, 25, "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[39397, 1287, 36, 0, 40, "Subsubsection"],

Cell[CellGroupData[{
Cell[39458, 1291, 1096, 27, 72, "Input",
  InitializationCell->True],
Cell[40557, 1320, 326, 8, 21, "Print"],
Cell[40886, 1330, 316, 6, 21, "Print"],
Cell[41205, 1338, 553, 15, 35, "Print"],
Cell[41761, 1355, 547, 11, 21, "Print"]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[42369, 1373, 30, 0, 43, "Subsection"],
Cell[42402, 1375, 261, 11, 29, "Text"],

Cell[CellGroupData[{
Cell[42688, 1390, 34, 0, 40, "Subsubsection"],

Cell[CellGroupData[{
Cell[42747, 1394, 343, 8, 72, "Input",
  InitializationCell->True],
Cell[43093, 1404, 44, 1, 25, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[43174, 1410, 450, 12, 56, "Input",
  InitializationCell->True],
Cell[43627, 1424, 452, 12, 35, "Print"],
Cell[44082, 1438, 529, 11, 21, "Print"],
Cell[44614, 1451, 333, 7, 21, "Print"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[44996, 1464, 35, 0, 40, "Subsubsection"],
Cell[45034, 1466, 213, 6, 44, "Text"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[45296, 1478, 30, 0, 43, "Subsection"],
Cell[45329, 1480, 144, 5, 29, "Text"],
Cell[45476, 1487, 83, 1, 29, "Text"],
Cell[45562, 1490, 297, 4, 106, "Text"],

Cell[CellGroupData[{
Cell[45884, 1498, 34, 0, 40, "Subsubsection"],

Cell[CellGroupData[{
Cell[45943, 1502, 410, 10, 72, "Input",
  InitializationCell->True],
Cell[46356, 1514, 44, 1, 25, "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[46449, 1521, 36, 0, 40, "Subsubsection"],
Cell[46488, 1523, 1104, 21, 182, "Text"],
Cell[47595, 1546, 1082, 24, 179, "Input",
  InitializationCell->True],
Cell[48680, 1572, 906, 19, 179, "Input"]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[49659, 1599, 26, 0, 46, "Subsubtitle"],
Cell[49688, 1601, 113, 3, 26, "Input",
  InitializationCell->True],
Cell[49804, 1606, 166, 5, 26, "Input",
  InitializationCell->True],
Cell[49973, 1613, 143, 4, 29, "Text"],
Cell[50119, 1619, 173, 5, 26, "Input",
  InitializationCell->True],
Cell[50295, 1626, 152, 5, 29, "Text"]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}
]
*)




(***********************************************************************
End of Mathematica Notebook file.
***********************************************************************)

