// Copyright (C) 1996,1997 Buntarou Shizuki(shizuki@is.titech.ac.jp)

#include "CZoom.h"

Am_Slot_Key CZNODE		= Am_Register_Slot_Name("CZNODE");
Am_Slot_Key CZSTATE		= Am_Register_Slot_Name("CZSTATE");
Am_Slot_Key CLICK_GROW		= Am_Register_Slot_Name("CLICK_GROW");
Am_Slot_Key CLICK_SHRINK	= Am_Register_Slot_Name("CLICK_SHRINK");
Am_Slot_Key CHANGE_FOCUS	= Am_Register_Slot_Name("CHANGE_FOCUS");
Am_Slot_Key CZNODE_BORDER	= Am_Register_Slot_Name("CZNODE_BORDER");
Am_Slot_Key ELLIPSIS_MODE	= Am_Register_Slot_Name("ELLIPSIS_MODE");
Am_Slot_Key ELLIPSIS_DOT_1	= Am_Register_Slot_Name("ELLIPSIS_DOT_1");
Am_Slot_Key ELLIPSIS_DOT_2	= Am_Register_Slot_Name("ELLIPSIS_DOT_2");
Am_Slot_Key ELLIPSIS_DOT_3	= Am_Register_Slot_Name("ELLIPSIS_DOT_3");

static Am_Object ellipsis_proto;

Am_Object CZNODE_PROTO;

Am_Object CZNode::CZwindow;

CZNode *CZNode::nodes_opened = NULL;
CZNode *CZNode::nodes_others = NULL;
CZNode *CZNode::root = NULL;

int CZNode::CZinterpolation_step = 1;

int CZNode::CZnumber_created = 0;
int CZNode::CZdebug_member = 0;
int CZNode::CZdebug_resection = 0;
int CZNode::CZdebug_section = 0;
int CZNode::CZdebug_list = 0;
int CZNode::CZdebug_calcpoint = 0;
int CZNode::CZdebug_scale = 0;
int CZNode::CZdebug_calc = 0;
int CZNode::CZdebug_focus = 0;

static void cznode_init();
static void ellipsis_init();

CZNode::CZNode()
{
  // for debugging
  CZid = CZnumber_created++;

  CZdoi = 1;
  state = CZ_be_auto;
  Xreq = tmpXreq = 0;
  Yreq = tmpYreq = 0;
  Xscale = tmpXscale = 0;
  Yscale = tmpYscale = 0;
  x_sections = NULL;
  y_sections = NULL;

  CZparent = NULL;
  next_opened = NULL;
  next_others = NULL;

  current_visible = false;

  current_expansion = CZ_closed;
  future_expansion = CZ_closed;
  forced_expansion = CZ_be_auto;

  display_method = new DisplayNormally(this);
}

CZNode::~CZNode()
{
  delete x_sections;
  delete y_sections;
  delete display_method;
}

//
// InitCZoom();
//   :
//   :
//
// for (;;) {
//	 :
//	CZCalc(screen size);
//	CZSetPoint(size);
//	 :
//	CZShow();
//	 :
// }
//
void
CZNode::CZInitCZoom(Am_Object target_window)
{
  CZwindow = target_window;

  cznode_init();
  ellipsis_init();
  return;
}

void
CZNode::CZCalcInit(CZNode *creator)
{
  DoublePointSize creator_ps = creator->OldPoint;

  double x = creator_ps.GetLeft() + creator_ps.GetWidth()/2;
  double y = creator_ps.GetTop() + creator_ps.GetHeight()/2;

  OldPoint.SetPoint(x, y);
  OldPoint.SetSize(0, 0);
}

void
CZNode::CZSetState(CZNodeState s)
{
  state = s;
  Pix p;
  CZNode *n;
  for (p = CZmember.first(); p != 0; CZmember.next(p)) {
    n = CZmember(p);
    n->CZSetState(s);
  }
}

void
CZNode::CZSetParent(CZNode *n)
{
  // establish parent-child relationship
  CZparent = n;
  if (CZparent) {
    CZparent->CZmember.append(this);
    CZparent->display_method->update();	// XXX
  }
}

void
CZNode::CZResection()
{
  CZMakeSection();
  CZTmpSection();

  if (CZdebug_resection) {
    int limit = GetLength()*2+2;

    printf("Resection of node(%d)@%x: x_sections:\n", CZid, this);
    if (GetLength() > 0)
      dump_sections(x_sections, limit);
    
    printf("Resection of node(%d)@%x: y_sections:\n", CZid, this);
    if (GetLength() > 0)
      dump_sections(y_sections, limit);
  }
}

void
CZNode::CZFocus()
{
  // focus me...
  CZparent->display_method->Focus(this);
  CZparent->CZResection();
}

void
CZNode::CZDeleteMember()
{
  Pix ni;
  DLList<CZNode *> *group = CZparent->CZGetMember();
  for (ni = group->first(); ni != 0; group->next(ni)) {
    if ((*group)(ni) == this)
      break;
  }
  if (ni == 0) {
    printf("DeleteMember: internal error, can't find node(%d)@%x in group.\n",
	   CZid, this);
  } else {
    group->del(ni);
    current_visible = false;	// XXX
  }
  CZparent->display_method->update(); // XXX
}

void
CZNode::CZSetDOI(double D)
{
  CZdoi = D;

  //
  // propagete upward(and downward...?)
  //
//   CZNode *p = CZparent;
//   while (p && p->CZdoi < D) {
//     p->CZdoi = D;
//     p = p->CZparent;
//   }
}

void
CZNode::CZSetMethod(DisplayMethod *method)
{
  if (display_method)
    free(display_method);
  display_method = method;
}

void
CZNode::CZShow()
{
  int i;

  root->CZShowPrologue();
  //  Am_Do_Events();		// XXX
  for (i = 1; i <= CZinterpolation_step; i++) {
    root->CZShowIter(i, CZinterpolation_step);
    Am_Do_Events();		// XXX
  }
  root->CZShowEpilogue();
  if (CZdebug_member)
    root->CZDumpMember(0);
}

void
CZNode::CZShowPrologue()
{
  Am_Object image = Image();

  if (CZparent) {
    if (CZparent->future_expansion == CZ_closed && !current_visible) {
      //current_visible = false;
      //image.Set(Am_VISIBLE, false);
      return;
    }
    if (CZparent->future_expansion == CZ_closed && current_visible) {
      current_visible = false;
      image.Set(Am_VISIBLE, false);
    }
    if (CZparent->future_expansion == CZ_opened && !current_visible) {
      current_visible = true;
      image.Set(Am_VISIBLE, true);
      CZparent->CZOldPoint(this);
    }
    if (CZparent->future_expansion == CZ_opened && current_visible) {
      //current_visible = true;
      //image.Set(Am_VISIBLE, true);
    }
  } else {
    current_visible = true;
    image.Set(Am_VISIBLE, true);
  }

  CZNode *n;
  for (n = GetFirst(); !IsDone(); n = GetNext())
    n->CZShowPrologue();

  for (n = GetOtherFirst(); !IsOtherDone(); n = GetOtherNext())
    n->CZOthersDelete();
  
//   Pix ni;
//   for (ni = CZmember.first(); ni != 0; CZmember.next(ni)) {
//     CZNode *n = CZmember(ni);
//     n->CZShowPrologue();
//   }
}

void
CZNode::CZOthersDelete()
{
  CZNode *n;
  
  if (! this)
    return;

  if (current_visible) {
    Image().Set(Am_VISIBLE, false);
    current_visible = false;
    
    for (n = GetFirst(); !IsDone(); n = GetNext())
      n->CZOthersDelete();
  }
}

void
CZNode::CZShowIter(int count, int limit)
{
  if (!current_visible)
    return;
  
  Am_Object image = Image();

  double p = limit-count;
  double q = count;
  
  int L = (int)((p*OldPoint.GetLeft()   + q*NewPoint.GetLeft())   / limit);
  int T = (int)((p*OldPoint.GetTop()    + q*NewPoint.GetTop())    / limit);
  int W = (int)((p*OldPoint.GetWidth()  + q*NewPoint.GetWidth())  / limit);
  int H = (int)((p*OldPoint.GetHeight() + q*NewPoint.GetHeight()) / limit);

  int state;
  if (future_expansion == CZ_opened)
    state = CZSTATE_CLUSTER;
  else
    state = CZSTATE_LEAF;

  if ((int)image.Get(Am_LEFT) != L)
    image.Set(Am_LEFT, L);
  if ((int)image.Get(Am_TOP) != T)
    image.Set(Am_TOP, T);
  if ((int)image.Get(Am_WIDTH) != W)
    image.Set(Am_WIDTH, W);
  if ((int)image.Get(Am_HEIGHT) != H)
    image.Set(Am_HEIGHT, H);
  if ((int)image.Get(CZSTATE) != state)
    image.Set(CZSTATE, state);
  if ((bool)image.Get(Am_VISIBLE) != true)
    image.Set(Am_VISIBLE, true);
  
  if (count == limit) {
    current_visible = true;
    OldPoint = NewPoint;
  }

  if (future_expansion == CZ_opened) {
    CZNode *n;
    for (n = GetFirst(); ! IsDone(); n = GetNext())
      n->CZShowIter(count, limit);
  }
}

void
CZNode::CZShowEpilogue()
{
  CZNode *n;
  
  current_expansion = future_expansion;
  for (n = GetFirst(); ! IsDone(); n = GetNext())
    n->CZShowEpilogue();
}

void
CZNode::CZMakeSection()
{
  int node_number = GetLength();
  int limit = node_number*2+2;
  
  delete x_sections;
  delete y_sections;
  x_sections = new section[limit];
  y_sections = new section[limit];
  bzero(x_sections, sizeof(section)*limit);
  bzero(y_sections, sizeof(section)*limit);
  
  x_sections[0].node = NULL;
  x_sections[0].orig = 0;
  x_sections[0].start = GetLeft();
  x_sections[limit-1].node = NULL;
  x_sections[limit-1].orig = limit;
  x_sections[limit-1].scale = -1;
  x_sections[limit-1].start = GetLeft()+GetWidth();
  x_sections[limit-1].width = -1;
  
  y_sections[0].node = NULL;
  y_sections[0].orig = 0;
  y_sections[0].start = GetTop();
  y_sections[limit-1].node = NULL;
  y_sections[limit-1].orig = limit;
  y_sections[limit-1].scale = -1;
  y_sections[limit-1].start = GetTop()+GetHeight();
  y_sections[limit-1].width = -1;
  
  //
  // Initialize each scale factor of sub-nodes
  //
  CZNode *n;
  for (n = GetFirst(); ! IsDone(); n = GetNext())
    n->CZCalcScale();

  //
  // breaks up into intervals
  //
  int i = 1;
  for (n = GetFirst(); ! IsDone(); n = GetNext()) {
    x_sections[i].node = n;
    x_sections[i].orig = i;
    x_sections[i].start = n->GetLeft();
    x_sections[i+1].node = n;
    x_sections[i+1].orig = i+1;
    x_sections[i+1].start = n->GetRight();
    y_sections[i].node = n;
    y_sections[i].orig = i;
    y_sections[i].start = n->GetTop();
    y_sections[i+1].node = n;
    y_sections[i+1].orig = i+1;
    y_sections[i+1].start = n->GetBottom();
    
    i = i + 2;
  }
  qsort(x_sections, limit, sizeof(section),
	(int (*)(const void *, const void *))compare_section);
  qsort(y_sections, limit, sizeof(section),
	(int (*)(const void *, const void *))compare_section);
  
  for (i = 0; i < limit-1; i++) {
    x_sections[i].width = x_sections[i+1].start - x_sections[i].start;
    y_sections[i].width = y_sections[i+1].start - y_sections[i].start;
  }  
  
  //
  // set scale factor of each section
  //
  for (i = 0; i < limit-1; i++) {
    x_sections[i].scale = -1;
    y_sections[i].scale = -1;
    x_sections[i].doi = 0;
    y_sections[i].doi = 0;
    for (n = GetFirst(); ! IsDone(); n = GetNext()) {
      double n_doi = n->CZdoi;
      if (x_sections[i].start >= n->GetLeft()
	  && n->GetRight() > x_sections[i].start) {
	if (x_sections[i].width > 0) {
	  double x_scale = n->Xscale;
	  if (x_sections[i].scale < x_scale) {
	    if (CZdebug_scale)
	      printf("node(%d)%x: MakeSection: setting x_scale[%d]=%g\n",
		     CZid, this, i, x_scale);
	    x_sections[i].scale = x_scale;
	  }
	  if (x_sections[i].doi < n_doi) {
	    x_sections[i].doi = n_doi;
	  }
	}
      }
      if (y_sections[i].start >= n->GetTop()
	  && n->GetBottom() > y_sections[i].start) {
	if (y_sections[i].width > 0) {
	  double y_scale = n->Yscale;
	  if (y_sections[i].scale < y_scale) {
	    if (CZdebug_scale)
	      printf("node(%d)%x: MakeSection: setting y_scale[%d]=%g\n",
		     CZid, this, i, y_scale);
	    y_sections[i].scale = y_scale;
	  }
	  if (y_sections[i].doi < n_doi) {
	    y_sections[i].doi = n_doi;
	  }
	}
      }
    }
  }
  
  //
  // set minimum requirement for blank sections
  //
#define BLANK_REQ 16
  for (i = 0; i < limit-1; i++) {
    if (x_sections[i].scale < 0 && x_sections[i].width > 0) {
      x_sections[i].scale = BLANK_REQ / x_sections[i].width;
      if (CZdebug_scale)
	printf("node(%d)%x: MakeSection: setting y_scale[%d]=%g\n",
	       CZid, this, i, x_sections[i].scale);
    }
    if (y_sections[i].scale < 0 && y_sections[i].width > 0) {
      y_sections[i].scale = BLANK_REQ / y_sections[i].width;
      if (CZdebug_scale)
	printf("node(%d)%x: MakeSection: setting y_scale[%d]=%g\n",
	       CZid, this, i, y_sections[i].scale);
    }
  }
  if (CZdebug_section) {
    int limit = GetLength()*2+2;

    printf("Making section of node(%d)@%x: x_sections:\n", CZid, this);
    if (GetLength() > 0)
      dump_sections(x_sections, limit);
    
    printf("Making section of node(%d)@%x: y_sections:\n", CZid, this);
    if (GetLength() > 0)
      dump_sections(y_sections, limit);
  }
}

void
CZNode::CZInitSection()
{
  if (GetLength() > 0) {
    CZNode  *n;
    for (n = GetFirst(); !IsDone(); n = GetNext()) {
      n->Xreq = n->tmpXreq = n->XMin();
      n->Yreq = n->tmpYreq = n->YMin();
      n->CZCalcScale();
      n->CZTmpScale();
    }
    CZMakeSection();
    CZTmpSection();
  }
}

void
CZNode::CZTmpSection()
{
  if (GetLength() > 0) {
    int limit = GetLength()*2+2;
    int i;
    for (i = 0; i < limit; i++) {
      x_sections[i].tmp = x_sections[i].scale;
      y_sections[i].tmp = y_sections[i].scale;
    }
  }
}

//
// CZCalc()
//
void
CZNode::CZCalc()
{
  if (CZdebug_calc)
    printf("CZCalc:\n");

  double W = (int)CZwindow.Get(Am_WIDTH) -20;
  double H = (int)CZwindow.Get(Am_HEIGHT)-20;

  CZMakeList();

  //
  // phase 1: show nodes that must be shown
  //
  {
    CZNode *node;
    for (node = nodes_opened; node != NULL; node = node->next_opened) {
      CZNode *n = node;
      n->CZTmpSection();
      while (n != NULL) {
	n->CZCalcMin();		// XXX
	n->CZCommitExpand();	// XXX
	n = n->CZparent;
      }
    }
  }

  double allowed_w = W;
  double allowed_h = H;

  //
  // phase 2: show nodes based on DOI within screen
  //
  CZNode *node = NULL;
  CZNode *n = NULL;
  {
    for (node = nodes_others; node != NULL; node = node->next_others) {
//       node->future_expansion = CZ_closed;
//       node->Xreq = node->tmpXreq = node->XMin();
//       node->Yreq = node->tmpYreq = node->YMin();
//       node->CZCalcScale();
//       node->CZTmpScale();
      if (CZdebug_calc)
	printf("\tCZCalc(begin): node(%d)@%x (%g,%g)(%g,%g)\n",
	       node->CZid, node,
	       root->Xreq, root->Yreq,
	       root->tmpXreq, root->tmpYreq);
      if (node->CZparent != NULL
	  && node->CZparent->future_expansion == CZ_closed
	  && node->CZparent->forced_expansion != CZ_be_closed) {
	for (n = node; n != root; n = n->CZparent)
	  n->CZparent->CZCalcMin(n);

	if (allowed_w >= root->tmpXreq && allowed_h >= root->tmpYreq) {
	  for (n = node->CZparent; n != NULL; n = n->CZparent)
	    n->CZCommitExpand();
	} else {
	  for (n = node->CZparent; n != NULL; n = n->CZparent)
	    n->CZTmpSection();
	  break;
	}
      }
      if (CZdebug_calc)
	printf("\tCZCalc( end ): node(%d)@%x (%g,%g(%g,%g))\n",
	       node->CZid, node,
	       root->Xreq, root->Yreq,
	       root->tmpXreq, root->tmpYreq);
    }
  }
}  
/*
void
CZNode::CZCalc()
{
  if (CZdebug_calc)
    printf("CZCalc:\n");

  double W = (int)CZwindow.Get(Am_WIDTH) -20;
  double H = (int)CZwindow.Get(Am_HEIGHT)-20;

  CZMakeList();

  //
  // phase 1: show nodes that must be shown
  //
  {
    CZNode *node;
    for (node = nodes_opened; node != NULL; node = node->next_opened) {
      CZNode *n = node;
      n->CZTmpSection();
      while (n != NULL) {
	n->CZCalcMin();		// XXX
	n->CZCommitExpand();	// XXX
	n = n->CZparent;
      }
    }
  }

  double allowed_w = W;
  double allowed_h = H;

  //
  // phase 2: show nodes based on DOI within screen
  //
  CZNode *node = NULL;
  CZNode *n = NULL;
  {
    for (node = nodes_others; node != NULL; node = node->next_others) {
      node->future_expansion = CZ_closed;
      node->Xreq = node->tmpXreq = node->XMin();
      node->Yreq = node->tmpYreq = node->YMin();
      node->CZCalcScale();
      node->CZTmpScale();
      if (CZdebug_calc)
	printf("\tCZCalc(begin): node(%d)@%x (%g,%g)(%g,%g)\n",
	       node->CZid, node,
	       root->Xreq, root->Yreq,
	       root->tmpXreq, root->tmpYreq);
      if (node->CZparent != NULL
	  && node->CZparent->future_expansion == CZ_closed
	  && node->CZparent->forced_expansion != CZ_be_closed) {
	for (n = node; n != root; n = n->CZparent)
	  n->CZparent->CZCalcMin(n);

	if (allowed_w >= root->tmpXreq && allowed_h >= root->tmpYreq) {
	  for (n = node->CZparent; n != NULL; n = n->CZparent)
	    n->CZCommitExpand();
	} else {
	  for (n = node->CZparent; n != NULL; n = n->CZparent)
	    n->CZTmpSection();
	  break;
	}
      }
      if (CZdebug_calc)
	printf("\tCZCalc( end ): node(%d)@%x (%g,%g(%g,%g))\n",
	       node->CZid, node,
	       root->Xreq, root->Yreq,
	       root->tmpXreq, root->tmpYreq);
    }
  }
}  
*/
void
CZNode::CZSetPoint()
{
  double W = (int)CZwindow.Get(Am_WIDTH) -20;
  double H = (int)CZwindow.Get(Am_HEIGHT)-20;
  root->CZCalcPoint(0, 0, W, H);
}

//
// CalcPoint(screen offset, screen size)
//
// This alogorithm is the one that is described in the
// Continuos Zoom paper and I think this is improper being used
// with semantic zooming....
//
void
CZNode::CZCalcPoint(double L, double T, double W, double H)
{
  if (CZdebug_calcpoint)
    printf("Setting node(%d)@%x with (%g,%g,%g,%g)\n",
	   CZid, this, L, T, W, H);

  NewPoint.SetPoint(L, T);
  NewPoint.SetSize(W, H);

  if (future_expansion == CZ_opened)
    AssignRemainder(L, T, W, H);
}  

void
CZNode::AssignRemainder(double L, double T, double W, double H)
{
  double XLfs = (double)(int)(W - Xreq); // XXX
  double YLfs = (double)(int)(H - Yreq); // XXX
  
  if (XLfs < 0 || YLfs < 0) {
    printf("Warning(internal error): space requirement not satisfied(X=%g,Y=%g)\n",
	   XLfs, YLfs);
    if (XLfs < 0)
      XLfs = 0;
    if (YLfs < 0)
      YLfs = 0;
  }

  int limit = GetLength()*2+2;

  section x_sec[limit];
  section y_sec[limit];
  memcpy(x_sec, x_sections, sizeof(section)*limit);
  memcpy(y_sec, y_sections, sizeof(section)*limit);

  double XDOIsum = 0;
  double YDOIsum = 0;
  int i;
  for (i = 0; i < limit-1; i++) {
    XDOIsum += x_sec[i].doi * x_sec[i].width;
    YDOIsum += y_sec[i].doi * y_sec[i].width;
  }

  //
  // set width destructively
  //
  double XDOIf;
  double YDOIf;
  if (XDOIsum > 0) {
    for (i = 0; i < limit-1; i++) {
      XDOIf = x_sec[i].doi*x_sec[i].width / XDOIsum;
      x_sec[i].width = x_sec[i].width*x_sec[i].scale + XDOIf*XLfs;
    }
  } else {
    for (i = 0; i < limit-1; i++)
      x_sec[i].width = x_sec[i].width*x_sec[i].scale;
  }
  if (YDOIsum > 0) {
    for (i = 0; i < limit-1; i++) {
      YDOIf = y_sec[i].doi*y_sec[i].width / YDOIsum;
      y_sec[i].width = y_sec[i].width*y_sec[i].scale + YDOIf*YLfs;
    }
  } else {
    for (i = 0; i < limit-1; i++)
      y_sec[i].width = y_sec[i].width*y_sec[i].scale;
  }

  if (CZdebug_calcpoint) {
    printf("AssignRemainder node(%d)@%x: x_sec\n", CZid, this);
    dump_sections(x_sec, limit);
    printf("AssignRemainder node(%d)@%x: y_sec\n", CZid, this);
    dump_sections(y_sec, limit);
  }
  
  CZNode *n;
  for (n = GetFirst(); !IsDone(); n = GetNext()) {
    i = 0;

    double child_L = L;
    double child_W = 0;
    while (x_sec[i].node != n) {
      child_L += x_sec[i].width;
      i++;
    }
    do {
      child_W += x_sec[i].width;
      i++;
    } while (x_sec[i].node != n);
    
    i = 0;
    double child_T = T;
    double child_H = 0;
    while (y_sec[i].node != n) {
      child_T += y_sec[i].width;
      i++;
    }
    do {
      child_H += y_sec[i].width;
      i++;
    } while (y_sec[i].node != n);
    n->CZCalcPoint(child_L, child_T, child_W, child_H);
  }
}

void
CZNode::CZMakeList()
{
  // first, sort nodes that must be opened using 'next_opened' field
  nodes_opened = root->CZMakeListOpened();

  // second, make list of nodes using 'next_others' field
  nodes_others = root->CZMakeListOthers();

  // close and minimize all nodes
  CZNode *node;
  for (node = nodes_opened; node != NULL; node = node->next_opened) {
    node->future_expansion = CZ_closed;
    node->Xreq = node->tmpXreq = node->XMin();
    node->Yreq = node->tmpYreq = node->YMin();
    node->CZCalcScale();
    node->CZTmpScale();
  }
  for (node = nodes_others; node != NULL; node = node->next_others) {
    node->future_expansion = CZ_closed;
    node->Xreq = node->tmpXreq = node->XMin();
    node->Yreq = node->tmpYreq = node->YMin();
    node->CZCalcScale();
    node->CZTmpScale();
  }
}

CZNode *
CZNode::CZMakeListOpened()
{
  if (this == NULL)
    return NULL;

  next_opened = NULL;

  if (CZGetState() != CZ_be_opened)
    return NULL;

  // when i am a leaf node
  if (GetLength() == 0)
    return this;

  // when i am a cluster node
  CZNode *head1 = this;
  CZNode *head2 = NULL;
  CZNode *node;
  for (node = GetFirst(); ! IsDone(); node = GetNext()) {
    head2 = node->CZMakeListOpened();
    head1 = merge_list_opened(head1, head2);
  }
  return head1;
}

CZNode *
CZNode::CZMakeListOthers()
{
  if (this == NULL)
    return NULL;
  
  next_others = NULL;

  if (CZGetState() == CZ_be_closed) {
    unvisible();		// XXX
    return this;		// XXX
  }

  CZNode *head1 = NULL;
  CZNode *head2 = NULL;
  CZNode *node;
  for (node = GetFirst(); ! IsDone(); node = GetNext()) {
    head2 = node->CZMakeListOthers();
    head1 = merge_list_others(head1, head2);
  }

  if (CZGetState() == CZ_be_opened)
    return head1;
  else {
    next_others = head1;
    return this;
  }
}

void
CZNode::CZOldPoint(CZNode *child)
{
  display_method->SetOld(child);
}

void
CZNode::unvisible()
{
  CZNode *n;
  for (n = GetFirst(); !IsDone(); n = GetNext()) {
    n->Image().Set(Am_VISIBLE, false);
    n->unvisible();
  }
}

void
CZNode::CZCalcScale()
{
  Xscale = Xreq / GetWidth();
  Yscale = Yreq / GetHeight();
}

void
CZNode::CZTmpScale()
{
  tmpXscale = tmpXreq / GetWidth();
  tmpYscale = tmpYreq / GetHeight();
}

void
CZNode::CZCommitScale()
{
  Xscale = tmpXscale;
  Yscale = tmpYscale;
}

void
CZNode::CZCalcMin(CZNode *updated)
{
  if (GetLength() > 0) {
    int limit = GetLength()*2+2;

    //
    // set scale factor of each section
    //
    int i;
    for (i = 0; i < limit-1; i++) {
      if (x_sections[i].start >= updated->GetLeft()
	  && updated->GetRight() > x_sections[i].start) {
	if (x_sections[i].width > 0) {
	  double x_scale = updated->tmpXscale;
	  if (x_sections[i].tmp < x_scale) {
	    x_sections[i].tmp = x_scale;
	  }
	}
      }
      if (y_sections[i].start >= updated->GetTop()
	  && updated->GetBottom() > y_sections[i].start) {
	if (y_sections[i].width > 0) {
	  double y_scale = updated->tmpYscale;
	  if (y_sections[i].tmp < y_scale) {
	    y_sections[i].tmp = y_scale;
	  }
	}
      }
    }
    //
    // calculate this cluster's minimum requested size
    //
    tmpXreq = 0;
    tmpYreq = 0;
    for (i = 0; i < limit; i++) {
      tmpXreq += x_sections[i].tmp * x_sections[i].width;
      tmpYreq += y_sections[i].tmp * y_sections[i].width;
    }
  } else {
    tmpXreq = XMin();
    tmpYreq = YMin();
  }
  CZTmpScale();
}

void
CZNode::CZSetMin(double X, double Y)
{
  Xreq = X;
  Yreq = Y;
  CZCalcScale();
}  

void
CZNode::CZCalcMin()
{
  if (GetLength() > 0) {
    int limit = GetLength()*2+2;

    //
    // set scale factor of each section
    //
    int i;
    CZNode *n;
    for (n = GetFirst(); !IsDone(); n = GetNext()) {
      for (i = 0; i < limit-1; i++) {
	if (x_sections[i].start >= n->GetLeft()
	    && n->GetRight() > x_sections[i].start) {
	  if (x_sections[i].width > 0) {
	    double x_scale = n->tmpXreq / x_sections[i].width; // XXX
	    if (x_sections[i].tmp < x_scale) {
	      x_sections[i].tmp = x_scale;
	    }
	  }
	}
	if (y_sections[i].start >= n->GetTop()
	    && n->GetBottom() > y_sections[i].start) {
	  if (y_sections[i].width > 0) {
	    double y_scale = n->tmpYreq / y_sections[i].width; // XXX
	    if (y_sections[i].tmp < y_scale) {
	      y_sections[i].tmp = y_scale;
	    }
	  }
	}
      }
    }
    //
    // calculate this cluster's minimum requested size
    //
    tmpXreq = 0;
    tmpYreq = 0;
    for (i = 0; i < limit; i++) {
      tmpXreq += x_sections[i].tmp * x_sections[i].width;
      tmpYreq += y_sections[i].tmp * y_sections[i].width;
    }
  } else {
    tmpXreq = XMin();
    tmpYreq = YMin();
  }
  CZTmpScale();
}

void
CZNode::CZCommitExpand()
{
  Xreq = tmpXreq;
  Yreq = tmpYreq;
  CZCommitScale();

  if (GetLength() > 0) {
    int limit = GetLength()*2+2;
    int i;
    for (i = 0; i < limit; i++) {
      x_sections[i].scale = x_sections[i].tmp;
      y_sections[i].scale = y_sections[i].tmp;
      if (CZdebug_scale) {
	printf("node(%d)%x: CommitExpand: setting x_scale[%d]=%g\n",
	       CZid, this, i, x_sections[i].scale);
	printf("node(%d)%x: CommitExpand: setting y_scale[%d]=%g\n",
	       CZid, this, i, y_sections[i].scale);
      }
    }
    future_expansion = CZ_opened;
  } else {
    future_expansion = CZ_closed;
  }
}

void
CZNode::CZDumpMember(int level)
{
  int i;
  for (i = 0; i < level; i++)
    printf(" ");
  printf("node(%d)@%x: %s-%s{%g,(%g,%g)(%g,%g)(%g,%g),(%g,%g,%g,%g),(%g,%g,%g,%g)}\n",
	 CZid,
	 this,
	 current_visible == true ? "visible" : "hidden",
	 current_expansion == CZ_be_opened ? "open" : "close",
	 CZdoi,
	 Xreq, Yreq,
	 tmpXreq, tmpYreq,
	 XMin(), YMin(),
	 OldPoint.GetLeft(), OldPoint.GetTop(),
	 OldPoint.GetWidth(), OldPoint.GetHeight(),
	 NewPoint.GetLeft(), NewPoint.GetTop(),
	 NewPoint.GetWidth(), NewPoint.GetHeight());
  CZNode *n;
  for (n = GetFirst(); ! IsDone(); n = GetNext())
    n->CZDumpMember(level+1);
}

void
CZNode::CZDumpSection()
{
  int limit = GetLength()*2+2;

  printf("x_sections of node(%d)@%x: \n", CZid, this);
  if (GetLength() > 0)
    dump_sections(x_sections, limit);

  printf("y_sections of node(%d)@%x: \n", CZid, this);
  if (GetLength() > 0)
    dump_sections(y_sections, limit);

  CZNode *n;
  for (n = GetFirst(); ! IsDone(); n = GetNext())
    n->CZDumpSection();
}


CZNode *
merge_list_opened(CZNode *list1, CZNode *list2)
{
  if (list2 == NULL)
    return list1;
  if (list1 == NULL)
    return list2;

  CZNode *head, *cur, *p, *q;
  if (list1->CZdoi < list2->CZdoi) {
    head = cur = list2;
    p = list1;
    q = list2->next_opened;
  } else {
    head = cur = list1;
    p = list1->next_opened;
    q = list2;
  }
  
  while (p != NULL && q != NULL) {
    if (p->CZdoi >= q->CZdoi) {
      cur->next_opened = p;
      cur = p;
      p = p->next_opened;
    } else {
      cur->next_opened = q;
      cur = q;
      q = q->next_opened;
    }
  }
  if (p != NULL)
    cur->next_opened = p;
  if (q != NULL)
    cur->next_opened = q;
  
  return head;
}
  
CZNode *
merge_list_others(CZNode *list1, CZNode *list2)
{
  if (CZNode::CZdebug_list) {
    printf("merge_list_others: list1="); print_list_others(list1); printf("\n");
    printf("                   list2="); print_list_others(list2); printf("\n");
  }
  
  if (list2 == NULL)
    return list1;
  if (list1 == NULL)
    return list2;

  CZNode *head, *cur, *p, *q;
  if (list1->CZdoi < list2->CZdoi) {
    head = cur = list2;
    p = list1;
    q = list2->next_others;
  } else {
    head = cur = list1;
    p = list1->next_others;
    q = list2;
  }
  
  while (p != NULL && q != NULL) {
    if (p->CZdoi >= q->CZdoi) {
      cur->next_others = p;
      cur = p;
      p = p->next_others;
    } else {
      cur->next_others = q;
      cur = q;
      q = q->next_others;
    }
  }
  if (p != NULL)
    cur->next_others = p;
  if (q != NULL)
    cur->next_others = q;

  if (CZNode::CZdebug_list) {
    printf("                  return="); print_list_others(head); printf("\n");
  }  

  return head;
}

void
print_list_others(CZNode *node)
{
  CZNode *n = node;
  while (n) {
    if (CZNode::CZdebug_list)
      printf(" (node %d,doi=%g)@%x", n->CZid, n->CZdoi, n);
    n = n->next_others;
  }
}

int
compare_section(CZNode::section *s1, CZNode::section *s2)
{
  if (s1->start > s2->start)
    return 1;
  if (s1->start < s2->start)
    return -1;
  return 0;
}

void
dump_sections(CZNode::section *sec, int n)
{
  int i;
  for (i = 0; i < n; i++) {
    printf("\t[%d]: start=%g, width=%g(%d), doi=%g, scale=%g, tmp=%g, node@%x\n",
	   i,
	   sec[i].start,
	   sec[i].width,
	   (int)(sec[i].width * sec[i].scale),
	   sec[i].doi,
	   sec[i].scale,
	   sec[i].tmp,
	   sec[i].node);
  }
}

static void do_grow(Am_Object cmd_obj);
static void do_shrink(Am_Object cmd_obj);

extern void remove_links();	// XXX
extern void draw_links();	// XXX

void
CZNode::CZRedraw()
{
  CZNode::CZCalc();
  CZNode::CZSetPoint();
  remove_links();		// XXX
  CZNode::CZShow();
  draw_links();			// XXX
}

Am_Define_Method(Am_Object_Method, void, click_grow,
		 (Am_Object cmd_obj))
{
  CZNode::click_grow(cmd_obj);
}

void
CZNode::click_grow(Am_Object cmd_obj)
{
  if (cmd_obj.Valid()) {
    Am_Object o = cmd_obj.Get_Owner().Get_Owner();
    if (o.Valid()) {
      CZNode *n = (CZNode*)(void*)o.Get(CZNODE);
      printf("Clicked node(%d)@%x, growing...\n", n->CZid, n);
      if (n->do_grow() == 0)
	CZNode::CZRedraw();
    }
  }
}

Am_Define_Method(Am_Object_Method, void, click_shrink, (Am_Object cmd_obj))
{
  CZNode::click_shrink(cmd_obj);
}

void
CZNode::click_shrink(Am_Object cmd_obj)
{
  if (cmd_obj.Valid()) {
    Am_Object o = cmd_obj.Get_Owner().Get_Owner();
    if (o.Valid()) {
      CZNode *n = (CZNode*)(void*)o.Get(CZNODE);
      printf("Clicked node(%d)@%x, shrinking...\n", n->CZid, n);
      if (n->do_shrink() == 0)
	CZNode::CZRedraw();
    }
  }
}

Am_Define_Method(Am_Object_Method, void, leave_focus_mode, (Am_Object inter))
{
  if (CZNode::CZdebug_focus)
    printf("Leave focus mode:\n");

  Am_Object_Method method = Am_Choice_Interactor.Get(Am_DO_METHOD);
  method.Call(inter);

  inter.Set(Am_ACTIVE, false);
}

void
CZNode::leave_focus_mode(Am_Object cmd_obj)
{
  return;
}

Am_Define_Method(Am_Object_Method,
		 void, focus_start, (Am_Object inter))
{
  if (CZNode::CZdebug_focus)
    printf("Start focusing...\n");

  Am_Object_Method method = Am_Choice_Interactor.Get(Am_START_DO_METHOD);
  method.Call(inter);
}

Am_Define_Method(Am_Object_Method,
		 void, focus_interim, (Am_Object inter))
{
  if (CZNode::CZdebug_focus)
    printf("Changing focus...\n");

  Am_Object_Method method = Am_Choice_Interactor.Get(Am_INTERIM_DO_METHOD);
  method.Call(inter);

  Am_Object focus_obj = (Am_Object)inter.Get(Am_INTERIM_VALUE);
  if (CZNode::CZdebug_focus) {
    cout << " --Focus_interim: " << inter << "\n";
    cout << " --Focus_interim: " << focus_obj << "\n";
  }
  if (focus_obj.Valid()) {
    CZNode *focus_node = (CZNode*)(void*)focus_obj.Get(CZNODE);
    focus_node->SetTmpMin();	// XXX
    focus_node->CZparent->CZInitSection(); // XXX
    focus_node->CZFocus();
    CZNode::CZRedraw();
    focus_node->SetNormalMin();	// XXX
    focus_node->CZparent->CZInitSection(); // XXX
  }
}

Am_Define_Method(Am_Object_Method,
		 void, focus_abort, (Am_Object inter))
{
  Am_Object_Method method = Am_Choice_Interactor.Get(Am_ABORT_DO_METHOD);
  method.Call(inter);

  CZNode *node = (CZNode*)(void*)inter.Get_Owner().Get(CZNODE); // XXX
  CZNode *n;
  for (n = node; n != NULL; n = n->CZparent) // XXX
    n->CZInitSection();		// XXX
  CZNode::CZRedraw();		// XXX
  if (CZNode::CZdebug_focus)
    printf("Aborting focus mode...\n");
}

Am_Define_Style_Formula(cznode_border_style)
{
  if (CZNode::CZdebug_focus)
    printf("Cznode_border_style: changing...\n");

  if (self.GV(Am_INTERIM_SELECTED))
    return Am_Line_4;
  else
    return Am_Thin_Line;
}

static int debug_interactor = 0;

Am_Define_Method(Am_Where_Method, Am_Object, CZNODE_IN_CLUSTER,
		 (Am_Object inter,
		  Am_Object object, Am_Object event_window,
		  Am_Input_Char ic, int x, int y)) {
  if (debug_interactor)
    cout << " --CZNODE_IN_CLUSTER: " << inter << object << "\n";
  
  CZNode *cluster = (CZNode*)(void*)inter.Get_Owner().Get(CZNODE);
  if (debug_interactor)
    printf("--CZNODE_IN_CLUSTER: cluster node(%d)@%x\n",
	   cluster->CZGetID(), cluster);
  CZNode *n;
  Am_Object win = CZNode::CZGetWindow();
  for (n = cluster->GetFirst(); !cluster->IsDone(); n = cluster->GetNext()) {
    Am_Object ret = Am_Point_In_Obj(n->Image(), x, y, win);
    if (ret && !ret.Is_Instance_Of(ellipsis_proto)) {
      if (debug_interactor) {
	cout << " --CZNODE_IN_CLUSTER: returns " << ret << "\n";
	CZNode *selected = (CZNode*)(void*)ret.Get(CZNODE);
	printf("--CZNODE_IN_CLUSTER: selected node(%d)@%x\n",
	       selected->CZGetID(), selected);
      }
      return ret;
    }
  }
  if (debug_interactor)
    cout << " --CZNODE_IN_CLUSTER: returns NULL\n";
  return Am_No_Object;
}

static void
cznode_init()
{
  Am_Object left_click = Am_One_Shot_Interactor.Create("growing")
    .Set(Am_PRIORITY, 6)
    .Set(Am_HOW_SET, false)
    .Get_Part(Am_COMMAND)
      .Set(Am_DO_METHOD, click_grow)
      .Get_Owner()
    ;
  Am_Object right_click = Am_One_Shot_Interactor.Create("shrinking")
    .Set(Am_PRIORITY, 6)
    .Set(Am_START_WHEN, "RIGHT_DOWN")
    .Set(Am_HOW_SET, false)
    .Get_Part(Am_COMMAND)
      .Set(Am_DO_METHOD, click_shrink)
      .Get_Owner()
    ;
  Am_Object change_focus = Am_Choice_Interactor.Create("focusing")
    .Set(Am_PRIORITY, 100)
    .Set(Am_ACTIVE, false)
    .Set(Am_START_WHEN, "MIDDLE_DOWN")
    .Set(Am_START_WHERE_TEST, CZNODE_IN_CLUSTER)
    .Set(Am_STOP_WHEN, "ANY_MOUSE_UP")

    .Set(Am_START_DO_METHOD, focus_start)
    .Set(Am_INTERIM_DO_METHOD, focus_interim)
    .Set(Am_ABORT_DO_METHOD, focus_abort)
    .Set(Am_DO_METHOD, leave_focus_mode)
    ;

  Am_Object cznode_border = Am_Rectangle.Create("border")
    .Set(Am_LEFT,  0)
    .Set(Am_TOP, 0)
    .Set(Am_WIDTH,  Am_From_Owner(Am_WIDTH))
    .Set(Am_HEIGHT, Am_From_Owner(Am_HEIGHT))
    .Set(Am_SELECTED, Am_From_Owner(Am_SELECTED))
    .Set(Am_INTERIM_SELECTED, Am_From_Owner(Am_INTERIM_SELECTED))
    .Set(Am_VISIBLE, Am_From_Owner(Am_INTERIM_SELECTED))
    .Set(Am_FILL_STYLE, Am_No_Style)
    .Set(Am_LINE_STYLE, cznode_border_style)
    ;

  CZNODE_PROTO = Am_Group.Create("CZNODE_PROTO")
    .Set(Am_PRETEND_TO_BE_LEAF, true)
    .Set(Am_VISIBLE, false)
    .Set(Am_LEFT, 0)
    .Set(Am_TOP, 0)
    .Set(Am_WIDTH, 0)
    .Set(Am_HEIGHT, 0)
    .Set(Am_SELECTED, false)
    .Set(Am_INTERIM_SELECTED, false)
    .Set(CZNODE, NULL)
    .Set(CZSTATE, CZSTATE_LEAF)
    .Add_Part(CLICK_GROW, left_click)
    .Add_Part(CLICK_SHRINK, right_click)
    .Add_Part(CHANGE_FOCUS, change_focus)
    .Add_Part(CZNODE_BORDER, cznode_border)
    ;
}

CZNode *
CZNode::GetAllFirst()
{
  all_index = CZmember.first();
  if (all_index == 0)
    return NULL;
  else
    return CZmember(all_index);
}

CZNode *
CZNode::GetAllNext()
{
  CZmember.next(all_index);
  if (all_index == 0)
    return NULL;
  else
    return CZmember(all_index);
}

bool
CZNode::IsAllDone()
{
  return all_index == 0;
}


Am_Define_Formula(int, dot_diameter)
{
  Am_Object e1, e2, e3;
  Am_Object owner = self.GV_Owner();

  int l, t, w, h;
  int d, r;
  double base;

  if (owner.Valid()) {
    l = owner.GV(Am_LEFT);
    t = owner.GV(Am_TOP);
    w = owner.GV(Am_WIDTH);
    h = owner.GV(Am_HEIGHT);
    base = log((double)min(w, h)); // XXX
    d = (int)(base*base); // XXX
    r = d/2;
    
    e1 = owner.GV_Part(ELLIPSIS_DOT_1);
    e2 = owner.GV_Part(ELLIPSIS_DOT_2);
    e3 = owner.GV_Part(ELLIPSIS_DOT_3);

    if ((int)owner.GV(ELLIPSIS_MODE) == ELLIPSIS_HORIZONTAL) {
      // horizontal mode
      int e1_l, e2_l, e3_l;
      int e_t;

      e1_l = w*2/10 - r;
      e2_l = w*5/10 - r;
      e3_l = w*8/10 - r;
      e_t = (h-d)/2;

      e1.Set(Am_LEFT, e1_l).Set(Am_TOP, e_t).Set(Am_WIDTH, d); //.Set(Am_HEIGHT, d);
      e2.Set(Am_LEFT, e2_l).Set(Am_TOP, e_t).Set(Am_WIDTH, d).Set(Am_HEIGHT, d);
      e3.Set(Am_LEFT, e3_l).Set(Am_TOP, e_t).Set(Am_WIDTH, d).Set(Am_HEIGHT, d);
    } else {
      // vertical mode
      int e_l;
      int e1_t, e2_t, e3_t;

      e_l = (w-d)/2;
      e1_t = h*2/10 - r;
      e2_t = h*5/10 - r;
      e3_t = h*8/10 - r;

      e1.Set(Am_LEFT, e_l).Set(Am_TOP, e1_t).Set(Am_WIDTH, d); //.Set(Am_HEIGHT, d);
      e2.Set(Am_LEFT, e_l).Set(Am_TOP, e2_t).Set(Am_WIDTH, d).Set(Am_HEIGHT, d);
      e3.Set(Am_LEFT, e_l).Set(Am_TOP, e3_t).Set(Am_WIDTH, d).Set(Am_HEIGHT, d);
    }
    return d;
  } else
    return 2;			// assumes the region is 10x10.
}


Am_Define_Method(Am_Object_Method,
		 void, enter_focus_mode, (Am_Object cmd_obj))
{
  CZNode::enter_focus_mode(cmd_obj);
}

void
CZNode::enter_focus_mode(Am_Object cmd_obj)
{
  if (CZNode::CZdebug_focus)
    printf("Enter focus change mode ");
  if (cmd_obj.Valid()) {
    Am_Object o = cmd_obj.Get_Owner().Get_Owner();
    if (o.Valid()) {
      CZNode *enode   = (CZNode*)(void*)o.Get(CZNODE);
      CZNode *cluster = enode->CZparent;
      if (CZNode::CZdebug_focus)
	printf("node(%d)@%x...\n", cluster->CZid, cluster);
      cluster->Image()
	.Get_Part(CHANGE_FOCUS)
	  .Set(Am_ACTIVE, true)
	  .Get_Owner()
	;
    }
  } else
    if (CZNode::CZdebug_focus)
      printf("\n");
}

static void
ellipsis_init()
{
  Am_Object left_click = Am_One_Shot_Interactor.Create("focus change")
    .Set(Am_START_WHEN, "MIDDLE_DOWN")
    .Set(Am_HOW_SET, false)
    .Set(Am_PRIORITY, 100)	// XXX
    .Get_Part(Am_COMMAND)
      .Set(Am_DO_METHOD, enter_focus_mode)
      .Get_Owner()
    ;

  Am_Object dot = Am_Arc.Create("dot");
    ;

  ellipsis_proto = Am_Group.Create("ellipsis")
    .Set(Am_VISIBLE, false)
    .Set(Am_WIDTH, 10)
    .Set(Am_HEIGHT, 10)
    .Set(CZSTATE, CZSTATE_LEAF)
    .Set(ELLIPSIS_MODE, ELLIPSIS_HORIZONTAL)
    .Add_Part(ELLIPSIS_DOT_1, dot.Create("ellipsis_dot1")
	      .Set(Am_HEIGHT, dot_diameter))
    .Add_Part(ELLIPSIS_DOT_2, dot.Create("ellipsis_dot2"))
    .Add_Part(ELLIPSIS_DOT_3, dot.Create("ellipsis_dot3"))
    .Add_Part(CHANGE_FOCUS, left_click)
    ;
}

Ellipsis::Ellipsis(int mode)
{
  image = ellipsis_proto.Create()
    .Set(CZNODE, (void *)this)
    .Set(ELLIPSIS_MODE, mode)
    ;
  CZwindow.Add_Part(image);
}

Ellipsis::~Ellipsis()
{
  image.Destroy();
}

void
Ellipsis::SetPoint()
{
  SetLeft(ref_node->GetLeft());
  SetTop(ref_node->GetTop());
  SetWidth(ref_node->GetWidth());
  SetHeight(ref_node->GetHeight());

}

//
// XXX
//

DisplayMethod::DisplayMethod() {}
DisplayMethod::~DisplayMethod() {}

DisplayNormally::DisplayNormally(CZNode *node)
{
  group = node;
  group_member = group->CZGetMember();
}

DisplayNormally::~DisplayNormally()
{
}

CZNode *
DisplayNormally::GetFirst()
{
  node_index = group_member->first();
  if (node_index == 0)
    return NULL;
  else
    return (*group_member)(node_index);
}

CZNode *
DisplayNormally::GetNext()
{
  group_member->next(node_index);
  if (node_index == 0)
    return NULL;
  else
    return (*group_member)(node_index);
}

bool
DisplayNormally::IsDone()
{
  return node_index == 0;
}

int
DisplayNormally::GetLength()
{
  return group_member->length();
}

void
//DisplayNormally::SetOld(CZNode *child)
DisplayMethod::SetOld(CZNode *child)
{
//   double x = group->NewPoint.GetLeft() + group->NewPoint.GetWidth() /2;
//   double y = group->NewPoint.GetTop()  + group->NewPoint.GetHeight()/2;
  double x = group->OldPoint.GetLeft() + group->OldPoint.GetWidth() /2;
  double y = group->OldPoint.GetTop()  + group->OldPoint.GetHeight()/2;
  child->OldPoint.SetPoint(x, y);
  child->OldPoint.SetSize(0, 0);
}


DisplayEllipsis::DisplayEllipsis(CZNode *node, int mode)
{
  int i;

  group = node;
  group_member = group->CZGetMember();
  ellipsis1 = new Ellipsis(mode);
  ellipsis2 = new Ellipsis(mode);
  ellipsis1->CZparent = node;	// XXX
  ellipsis2->CZparent = node;	// XXX
  ellipsis1_index = -1;		// ellipsis1 is not displayed
  ellipsis2_index = -1;		// ellipsis2 is not displayed
  for (i = 0; i < CZ_ELLIPSIS_NODES; i++)
    nodes[i] = NULL;
  nodes_length = 0;
  nodes_index = 0;
  focus_index = -1;		// default: focus is at the last one
  other_index = 0;
}

void
DisplayEllipsis::Focus(CZNode *node)
{
  int i;
  CZNode *n;
  Pix p;
  for (p=group_member->first(),i=0; p!=0; group_member->next(p),i++) {
    n = (*group_member)(p);
    if (n == node)
      break;
  }

//   for (n = group->GetFirst(),i = 0; !group->IsDone(); n=group->GetNext(),i++) {
//     if (n == node)
//       break;
//   }

  int len  = group->GetLength();
  if (i == len) {
    // internal error
  } else if (i == len-1)
    focus_index = -1;
  else
    focus_index = i;

  if (CZNode::CZdebug_focus)
    printf("Changing focus of node(%d)@%x to %d(node(%d)@%x)\n",
	   group->CZGetID(), group, focus_index, node->CZGetID(), node);

  update();
}

// initialization:
//	nodes
//	nodes_length
//	nodes_index;
CZNode *
DisplayEllipsis::GetFirst()
{
  nodes_index = 0;
  return nodes[nodes_index];
}

CZNode *
DisplayEllipsis::GetNext()
{
  nodes_index++;
  if (nodes_index == ellipsis1_index) {
    ellipsis1->SetPoint();
    return ellipsis1;
  } else if (nodes_index == ellipsis2_index) {
    ellipsis2->SetPoint();
    return ellipsis2;
  } else if (nodes_index < nodes_length) {
    return nodes[nodes_index];
  } else 
    return NULL;
}

bool
DisplayEllipsis::IsDone()
{
  return nodes_index >= nodes_length;
}

int
DisplayEllipsis::GetLength()
{
  int len = group_member->length();
  int focus = focus_index < 0 ? len-1 : focus_index;

  if (len < 5)
    return len;
  else
    return min(focus+4, 3-focus+len);
}

CZNode *
DisplayEllipsis::GetOtherFirst()
{
  other_index_ellipsis = 0;
  other_index = group_member->first();
  return GetOtherNext();
}

CZNode *
DisplayEllipsis::GetOtherNext()
{
  int i;
  CZNode *n;

  for (;;) {
    //
    // fetch a CZNode
    //
    if (other_index != 0)
      group_member->next(other_index);

    if (other_index == 0) {
      if (other_index_ellipsis == 0)
	n = ellipsis1;
      else if (other_index_ellipsis == 1)
	n = ellipsis2;
      else {
	other_index_ellipsis = -1;
	n = NULL;
	break;
      }
      other_index_ellipsis++;
    }
    else 
      n = (*group_member)(other_index);

    //
    // test it
    //
    for (i = 0; i < CZ_ELLIPSIS_NODES; i++) {
      if (nodes[i] == n)
	break;
    }
    if (i == CZ_ELLIPSIS_NODES)
      return n;
  }
  return NULL;
}

bool
DisplayEllipsis::IsOtherDone()
{
  return other_index == 0 && other_index_ellipsis < 0;
}

static int debug_ellipsis_update = 0;

void
DisplayEllipsis::update()
{
  int i, j;
  Pix p;
  CZNode *n;
  int pi;
  int len = group_member->length();
  int focus = focus_index < 0 ? len-1 : focus_index;

  if (debug_ellipsis_update)
    printf("Ellipsis::Update: focus = %d\n", focus);

  ellipsis1_index = -1;
  ellipsis2_index = -1;

  Pix node_pix[5];
  int node_ind[6];

  if (len < 5) {
    for (p = group_member->first(),pi = 0; p!=0; group_member->next(p),pi++)
      nodes[pi] = (*group_member)(p);
    nodes_length = len;
  } else {
    node_ind[0] = 0;
    node_ind[1] = focus - 1;
    node_ind[2] = focus;
    node_ind[3] = focus + 1;
    node_ind[4] = len - 1;
    node_ind[5] = -1;

    for (p = group_member->first(),pi = 0; p != 0; group_member->next(p),pi++) {
      if (pi == focus)
	break;
    }
    Pix p1 = p; group_member->prev(p1);
    Pix p2 = p;
    Pix p3 = p; group_member->next(p3);

    node_pix[0] = group_member->first();
    node_pix[1] = p1;
    node_pix[2] = p2;
    node_pix[3] = p3;
    node_pix[4] = group_member->last();

    j = 0;
    int prev_i = -1;
    for (i = 0; i < 5 && prev_i < len-1; i++) {
      int diff  = node_ind[i] - prev_i;
      if (diff < 1) {
	;
      } else if (diff == 1 && node_ind[i] < len) {
	n = (*group_member)(node_pix[i]);
	if (debug_ellipsis_update)
	  printf("Ellipsis::Update: nodes[%d] = %d, node(%d)@%x\n",
		 j, node_ind[i], n->CZGetID(), n);
	nodes[j++] = n;
	prev_i = node_ind[i];
	//    else if (diff == 2) {
	//      nodes[j++] = (*group_member)(node_pix[i]);
	//      prev_i = node_ind[i];
      } else if (diff == 1 /*&& node_ind[i] >= len*/) {
	;
      } else {
	if (i == 1) {
 	  Pix tmp = node_pix[i];
 	  group_member->prev(tmp);
 	  CZNode *n = (*group_member)(tmp);
	  ellipsis1->SetRef(n);
	  ellipsis1_index = j;
	  if (debug_ellipsis_update)
	    printf("Ellipsis::Update: nodes[%d] = ellipsis1\n", j);
	  nodes[j++] = ellipsis1;

// 	  Pix tmp = node_pix[i];
// 	  group_member->prev(tmp);
// 	  CZNode *n = (*group_member)(tmp);
// 	  ellipsis1->SetLeft(n->GetLeft());
// 	  ellipsis1->SetTop(n->GetTop());
// 	  ellipsis1->SetWidth(n->GetWidth());
// 	  ellipsis1->SetHeight(n->GetHeight());
	} else {
 	  Pix tmp = node_pix[i];
 	  group_member->prev(tmp);
 	  CZNode *n = (*group_member)(tmp);
	  ellipsis2->SetRef(n);
	  ellipsis2_index = j;
	  if (debug_ellipsis_update)
	    printf("Ellipsis::Update: nodes[%d] = ellipsis2\n", j);
	  nodes[j++] = ellipsis2;

// 	  Pix tmp = node_pix[i];
// 	  group_member->prev(tmp);
// 	  CZNode *n = (*group_member)(tmp);
// 	  ellipsis2->SetLeft(n->GetLeft());
// 	  ellipsis2->SetTop(n->GetTop());
// 	  ellipsis2->SetWidth(n->GetWidth());
// 	  ellipsis2->SetHeight(n->GetHeight());
	}
	n = (*group_member)(node_pix[i]);
	if (debug_ellipsis_update)
	  printf("Ellipsis::Update: nodes[%d] = %d, node(%d)@%x\n",
		 j, node_ind[i], n->CZGetID(), n);
	nodes[j++] = n;
	prev_i = node_ind[i];
      }
    }
    nodes_length = j;
    for (; j < CZ_ELLIPSIS_NODES; j++)
      nodes[j] = NULL;
  }
  nodes_index = 0;
}

// eof
