/************************************************************************
 **  trie.c: build TRIE structure of the explanations.
 **
 **  Copyright (C) 1998
 **    Taisuke Sato, Yoshitaka Kameya, Yasushi Hagiwara, Nobuhisa Ueda,
 **      Dept. of Computer Science, Tokyo Institute of Technology.
 ************************************************************************/

#include "prism.h"

/* Global variables */
struct glink *Root;
struct glink *ERoot;
struct swlink *Crr;

/* Global variables used in em.c */
extern int idNum,idNum1,valueNum,valueNum1,valueNum2,ivNum12,ivNum11;
extern int goals;
extern struct goal *G;
extern struct switchInfo *switch_info;

/*
 *  trie.c: PRISM keeps explanations in the form of trie structure.
 *
 *  <ex.>
 *  hmm([a,b]) is explained by
 *
 *    bsw(init,nil,0) & bsw(obs(s0),1,0) & bsw(trans(s0),0) & ...
 *  v bsw(init,nil,0) & bsw(obs(s0),1,0) & bsw(trans(s0),1) & ...
 *  v ...
 *  v bsw(init,nil,1) & bsw(obs(s1),1,1) & bsw(trans(s1),1) & ...
 *
 *  An explanation is a conjunction of switches, e.g, bsw(init,nil,0) &
 *  bsw(obs(s0),1,0) & bsw(trans(s0),0) &...
 *  PRISM keeps all explanations in the following form, which is called
 *  trie structure.
 *
 *  bsw(init,nil,0) -+-> bsw(obs(s0),1,0) -+-> bsw(trans(s0),0) -->..
 *                   |                     |
 *                   |                     +-> bsw(trans(s0),1) -->..
 *                   |
 *                   +-> bsw(obs(s0),1,1) -+-> ...
 */  

void initRoot(void) {
  Root=GNULL; ERoot=GNULL; Crr=LNULL;
}

/* Functions for preparation of trie structure */
int prepare_trie(void) {

  get_trie(-1,(int)NULL,(int)NULL,(int)NULL);
  /* get_trie $B$N(B static $BJQ?t(B ignore $B$r(B 0 $B$K=i4|2=(B */
  if ((Root = galloc())==GNULL) return(0);
  else {
	Root->Expl=LNULL;
	Root->NextG=GNULL;
	ERoot=Root;
	return(1);
  }
}

void cancel_trie(void) {

  free(Root);

}

int prepare_expl(int goal) {

  if (ERoot->Expl==LNULL) {

	ERoot->Goal=goal;
	ERoot->Count=0;
	ERoot->NextG=GNULL;

	if ((ERoot->Expl=lalloc())==LNULL) return(0);
	(ERoot->Expl)->Sw=SNULL;
	(ERoot->Expl)->NextSw=LNULL;
	Crr=ERoot->Expl;
	return(1);
  }
  else if (ERoot->NextG==GNULL) {

	if ((ERoot->NextG = galloc())==GNULL) return(0);

	ERoot=ERoot->NextG;

	ERoot->Goal=goal;
	ERoot->Count=0;
	ERoot->NextG=GNULL;

	if ((ERoot->Expl=lalloc())==LNULL) return(0);
	(ERoot->Expl)->Sw=SNULL;
	(ERoot->Expl)->NextSw=LNULL;
	Crr=ERoot->Expl;
	return(1);
  }
  else {
	ERoot=ERoot->NextG;
	return prepare_expl(goal);
  }
}

/* 'Return to root' functions */
void return_to_root(void) {
  ERoot=Root;
}

void return_to_expl_root(void) {
  Crr=ERoot->Expl;
}

/* Increment or decrement the count of explanations */
void add_expl_count(int n) {
  ERoot->Count += n;
}

/* Memory allocation functions */
struct sw *salloc(void) {
  return (struct sw *)malloc(sizeof(struct sw));
}

struct swlink *lalloc(void) {
  return (struct swlink *)malloc(sizeof(struct swlink));
}

struct glink *galloc(void) {
  return (struct glink *)malloc(sizeof(struct glink));
}

/*
 * PRISM $B$O$"$k%4!<%k(B(goal)$B$N@bL@<0(B(switch $B$N(B $BA*8@I8=`7A(B)$B$+$i(B switch $B$N(B
 * $B%H%i%$(B(trie)$B9=B$$r9=C[$9$k!%$?$@$7@bL@<0$NA*8@;R(B (switch $B$N(B $BO"8@(B) $B$O(B
 * switch $B$NBh(B1$B0z?t$HBh(B2$B0z?t$NAH$r%-!<$K$7$F(B  sort  $B$7$F$"$j!$(B $B=EJ#$9$k(B
 * switch $B$,=P8=$7$J$$$b$N$H$9$k!%(B
 *
 * get_trie() $B$O8D!9$N(B switch $B$r<u$1<h$j=g$K%H%i%$9=B$$r9=C[$9$k!%Nc$($P(B
 * sw(1,1,0) & sw(1,2,0) & sw(2,2,0) v sw(1,1,0) & sw(2,1,0) & sw(3,2,0)
 * $B$,M?$($i$l$?@bL@<0$G$"$C$?$H$-!$(B
 *
 *   get_trie(1,1,1,0);
 *   get_trie(1,1,2,1);
 *   get_trie(0,2,2,0);
 * 
 *   get_trie(1,1,1,0);
 *   get_trie(1,1,2,1);
 *   get_trie(0,3,2,0);
 *
 * $B$H8F$S=P$;$P$h$$!%(Bget_trie() $B$NBh(B1$B0z?t$r(B 0 $B$H$9$k$3$H$G0l$D$N(B switch
 * $B$NO"8@$,0l$D=*$o$C$?$3$H$rCN$i$;$k!%(B
 */
int get_trie(int flag, int g_id, int t_id, int val) {

  static int ignore=0;
  /*
   * $B@bL@<0$,(B A&B v A&B&C&D v B&C $B$N$H$-!$(BA&B&C&D $B$r:o=|$9$kI,MW$,$"$k!%(B
   * $B$b$C$H6qBNE*$K$$$&$H!$(Bget_trie() $B$,(B 2$BHVL\$N(B disjunct $BCf$N(B B $B$r<u$1(B
   * $B<h$C$?;~E@$G(B C $B0J2<$rL5;k$9$l$P$h$$(B(A $B$H(B B $B$O(B 1 $BHVL\$N(B disjunct $B$K(B
   * $B%^!<%8$5$l$F$$$k(B)$B!%$H$3$m$,(B Prolog $BB&$G$O2?$b9M$($:99$K(B C $B0J2<$rAw(B
   * $B$j$D$1$F$/$k$N$G(B static $BJQ?t(B ignore $B$rMQ0U$7!$(BB $B$r<u$1<h$C$?;~E@$G(B
   * ignore $B$r(B 1 $B$K%;%C%H$7!$(Bignore $B$,(B 1 $B$G$"$k8B$j2?$b9T$J$o$:$KL5;k$9(B
   * $B$k$h$&$K$9$k!%$?$@$7F~NO(B flag $B$,(B 0 $B$G$"$C$?>l9g!$$=$3$G0l$D$N(B
   * disjunct $B$,=*$o$C$?$3$H$K$J$k$N$G(B ignore $B$r(B 0 $B$K%j%;%C%H$7!$?7$?$J(B
   * disjunct B&C $B$rBT$D!%(B
   */

  /* printf("get_trie(%d,%d,%d,%d)\n",flag,g_id,t_id,val); */

  if (flag==-1) { /* -1 $B$N(B flag $B$r<u$1<h$C$?$i(B ignore $B$r(B 0 $B$K%;%C%H(B */
	ignore=0;
	return(1);
  }

  if (ignore)
	switch(flag){
	case 1:
	  return(1);           /* $B2?$b$7$J$$(B */
	case 0:
	  /* disjunct $B$,=*$o$C$?$N$G(B ignore $B$r%j%;%C%H$7!$(Broot $B$KLa$k(B */
	  ignore=0; return_to_expl_root(); return(1);
	default:
	  return(0);
	}

  if (Crr->Sw==SNULL) {
	/* printf("Crr->Sw==SNULL\n"); */

	if ((Crr->Sw = salloc())==SNULL) return(0);

	(Crr->Sw)->G_id = (short)g_id;	
	(Crr->Sw)->T_id = (short)t_id;
	(Crr->Sw)->Val  = (short)val;

	switch(flag){
	case 1:
	  if (((Crr->Sw)->Child = lalloc())==LNULL) return(0);
	  ((Crr->Sw)->Child)->Sw=SNULL;
	  ((Crr->Sw)->Child)->NextSw=LNULL;
	  Crr=(Crr->Sw)->Child;
	  break;
	case 0:
	  (Crr->Sw)->Child = LNULL;
	  add_expl_count(1);
	  return_to_expl_root();
	  break;
	default:
	  printf("{PRISM INTERNAL ERROR: get_trie(%d,_,_,_) --");
	  printf("%d must be 0 or 1}",flag);
	  return(0);
	}
	return(1);
  }
  else {
	
	while(1){
	  if (compare_sw(Crr->Sw,g_id,t_id,val)) { /* $B0JA0$N$b$N$H%N!<%I$r6&M-(B */
		if ((Crr->Sw)->Child==LNULL)
		  switch(flag){
		  case 1:     /* $BF~NO$HESCf$^$GF1$8(B explanation $B$,0JA0$K$bB8:_$7$?(B */
			ignore=1;          /* flag $B$,(B 0 $B$N8F$S=P$7$,$"$k$^$G2?$b$7$J$$(B */
			break;
		  case 0:         /* $BF~NO$HA4$/F1$8(B explanation $B$,0JA0$K$bB8:_$7$?(B */
			return_to_expl_root();       /* add_expl_count(1) $B$O9T$J$o$J$$(B */
			break;
		  default:
			printf("{PRISM INTERNAL ERROR: get_Ptrie(%d,_,_,_) -- ");
			printf("%d must be 0 or 1.}",flag);
			return(0);
		  }
		else {
		  switch(flag){
		  case 1:
			Crr=(Crr->Sw)->Child;
			break;
		  case 0:
			/* $B>C$7$?;R6!$N?t(B-1 $B$r0z$/(B */
			add_expl_count(1-freeChild((Crr->Sw)->Child));
			(Crr->Sw)->Child=LNULL;
			return_to_expl_root();
			break;
		  default:
			printf("{PRISM INTERNAL ERROR: get_trie(%d,_,_,_) --");
			printf("%d must be 0 or 1}",flag);
			return(0);
		  }
		}
		break;
	  }
	  else if (Crr->NextSw==LNULL) {

		if ((Crr->NextSw = lalloc())==LNULL) return(0);
		if (((Crr->NextSw)->Sw = salloc())==SNULL) return(0);
		(Crr->NextSw)->NextSw=LNULL;

		((Crr->NextSw)->Sw)->G_id = (short)g_id;
		((Crr->NextSw)->Sw)->T_id = (short)t_id;
		((Crr->NextSw)->Sw)->Val  = (short)val;

		switch(flag){
		case 1:
		  if ((((Crr->NextSw)->Sw)->Child = lalloc())==LNULL) return(0);
		  (((Crr->NextSw)->Sw)->Child)->Sw=SNULL;
		  (((Crr->NextSw)->Sw)->Child)->NextSw=LNULL;
		  Crr=((Crr->NextSw)->Sw)->Child;
		  break;
		case 0:
		  ((Crr->NextSw)->Sw)->Child = LNULL;
		  add_expl_count(1);
		  return_to_expl_root();
		  break;
		default:
		  printf("{PRISM INTERNAL ERROR: get_trie(%d,_,_,_) --");
		  printf("%d must be 0 or 1}",flag);
		  return(0);
		}
		break;
	  }
	  else {
		Crr=Crr->NextSw;
	  }
	}
	return(1);
  }
}

/* $B:G=*E*$K>C$7$?MU$N?t(B(explanation$B$N?t(B)$B$rJV$9(B
 * Modified by kame on Nov/26/1997 
 */
int freeChild(struct swlink *childp){

  struct swlink *tmp_ptr;
  struct swlink *tmp_ptr2;
  int deleted_child=0;

  for(tmp_ptr=childp; tmp_ptr != LNULL; tmp_ptr = tmp_ptr2){

	if ((tmp_ptr->Sw)->Child != LNULL) {
	  deleted_child += freeChild((tmp_ptr->Sw)->Child);
	  (tmp_ptr->Sw)->Child=LNULL;
	}
	else deleted_child += 1;

	free(tmp_ptr->Sw);
	tmp_ptr->Sw=SNULL;

	tmp_ptr2 = tmp_ptr->NextSw;
	free(tmp_ptr);
  }
  return deleted_child;
}

/* swp $B$N;X$9(B switch $B$N0z?t$H(B g_id, t_id, val $B$,0lCW$9$k$+(B(1 or 0)? */
int compare_sw(struct sw *swp, int g_id, int t_id, int val) {

  return (swp->G_id==(short)g_id
          && swp->T_id==(short)t_id
          && swp->Val==(short)val);

}

void freeTrie(void) {

  struct glink *glp;
  struct glink *glq;
  int deleted_expls=0,de;
  
  if (Root != GNULL) {
	for(glp=Root; glp != GNULL; glp=glq) {
	  deleted_expls += (de = freeChild(glp->Expl));
	  /* printf("Goal[%d](%d expls) deleted %d expls\n",
		 glp->Goal,glp->Count,de); */
	  glp->Expl=LNULL;
	  glq=glp->NextG;
	  free(glp);
	}
	Root=GNULL;
	printf("{Previous explanations cleaned up.}\n");
  }

  ERoot=GNULL;
  Crr=LNULL;

}


/* $B%H%i%$9=B$2=$5$l$?@bL@<0$N7A>u$r0u;z(B */
void show_trie(void){

  struct glink *glp;

  for(glp=Root; glp != GNULL; glp = glp->NextG) {
	printf("[Goal %d] -- %d expls\n",glp->Goal,glp->Count);
	if (glp->Expl != LNULL) show_expl(0,glp->Expl);
  }
}

/* $B%a%b%j%"%I%l%9I=<($"$j(B
void show_expl(int indent, struct swlink *linkp){

  int i;
  if(linkp->Sw != SNULL){

	printf("%8x->%8x[%2d|%2d|%1d]",
		   linkp,linkp->Sw,
		   (linkp->Sw)->G_id,(linkp->Sw)->T_id,(linkp->Sw)->Val);

	if ((linkp->Sw)->Child != LNULL){
	  printf("->");
	  show_expl(indent+1,(linkp->Sw)->Child);
	}
	else printf("$\n");

	if (linkp->NextSw != LNULL){
	  for(i=0;i<indent;i++) printf("                             ");
	  show_expl(indent,linkp->NextSw);
	} 

	if (linkp->NextSw != LNULL){
	  for(i=0;i<indent;i++) printf("                             ");
	  show_expl(indent,linkp->NextSw);
	} else {
	  for(i=0;i<indent;i++) printf("                             ");
	  printf("~~~~~~~~             \n");
	}
  }
}
*/

/* $B%a%b%j%"%I%l%9I=<($J$7(B */
void show_expl(int indent, struct swlink *linkp){

  int i;
  if(linkp->Sw != SNULL){
	printf("[%2d|%2d|%1d]",
		   (linkp->Sw)->G_id,(linkp->Sw)->T_id,(linkp->Sw)->Val);
	if ((linkp->Sw)->Child != LNULL)
	  show_expl(indent+1,(linkp->Sw)->Child);
	else printf("\n");
	if (linkp->NextSw != LNULL){
	  for(i=0;i<indent;i++) printf("         ");
	  show_expl(indent,linkp->NextSw);
	}
  }
}

/* Explanation table $B$N9=C[(B 
 *   [NOTE]
 *   Prolog $BB&$+$i(B em.c $B$N4X?t72(B(set_goals$B$J$I(B)$B$,8F$S=P$5$l!$(B
 *   explanation table $B$KI,MW$J(B memory $B$O4{$K3NJ]$5$l$F$$$k(B
 *   $B$H$9$k!%$3$3$G$O%H%i%$9=B$2=$5$l$?@bL@<0$r;2>H$7$J$,$i(B
 *   explanation table $B$NCf?H$rKd$a$k:n6H$r9T$J$&!%(B
 */

/* [NOTE] em.c $B$N(B set_goals $B$+$i8F$S=P$5$l$k(B */
int set_Gstatus(void){

  struct glink *glp;
  int t;

  for(glp=Root; glp != GNULL; glp=glp->NextG){
	t = glp->Goal;
	G[t].Snum = glp->Count;
	if ((G[t].table=(int *)malloc(sizeof(int)*ivNum12*glp->Count))
		==(int *)NULL)
	  return(0);
	G[t].observedNum = 0;
  }
  return(1);
}

/* $B%H%i%$9=B$$N@bL@<0$N:,$+$iMU$K8~$+$C$F$?$I$j!$$=$N7PO)Cf$G3F(B switch
 * $B$N(B ON $B$N8D?t$r?t$($k!%MU$K$D$$$?$i$"$i$+$8$a3NJ]$7$F$"$k(B explanation
 * table $B$N(B1$B9T$K%3%T!<$9$k(B
 */
int count_expls(void){
  
  struct glink *glp;
  int *ebuf;
  int i,r=1;

  for(glp=Root; glp != GNULL; glp=glp->NextG){
	copy_to_table(-1,(int *)NULL);      /* copy_to_table $B$N(B index $B$r=i4|2=(B */
	if ((ebuf=(int *)malloc(sizeof(int)*ivNum12))==(int *)NULL)
	  return(0);
	for(i=0;i<ivNum12;i++) ebuf[i]=0;
	if (glp->Expl != LNULL)
	  r &= count_an_expl(glp->Goal,glp->Expl,ebuf);
  }
  return r;
}

int count_an_expl(int t, struct swlink *swlp, int *ebuf){

  struct sw *swp;
  int *ebufcopy;
  int i,j,g,flag,r1,r2; /* j is temporary */

  swp=swlp->Sw;

  /* $BJ,4t$,$"$k$H$-$O@h$:%P%C%U%!(B ebuf $B$r(B ebufcopy $B$K%3%T!<$7$F$*$/(B */
  if (swlp->NextSw != LNULL){
	if ((ebufcopy=(int *)malloc(sizeof(int)*ivNum12))==(int *)NULL) return(0);
	for(i=0;i<ivNum12;i++)
	  ebufcopy[i]=ebuf[i];
  }
  /* swp->G_id $B$O(B Prolog $BCf$G3d$jEv$F$i$l$?@0?t%3!<%I$G!$(B
	 0 $B$+$i(B idNum $B$^$G$NO"B3CM$r$H$k$H$O8B$i$J$$$?$a!$(B
	 switch_info[i].idInProlog $B$H%^%C%A$9$k(B i $B$rC5$9!%(B */
  flag=0;
  for(i=0;i<=idNum;i++)
	if(swp->G_id==switch_info[i].idInProlog) { g=i; flag=1; break; }
  
  if (flag==0){
	printf("{PRISM INTERNAL ERROR: count_an_expl() -- unknown G_id %d occurs in trie.}",swp->G_id);
	return(0);
  }
  
  ebuf[g*valueNum1+swp->Val] += 1;          /* $B=P8=$7$?(B switch $B$r%+%&%s%H(B */
  ebuf[ivNum11+g] +=1;                              /* Gamma $B$N7W;;$K;H$&(B */
  
  /* $BJ,4t$9$kA0$K<+J,$N;R$I$b$K$D$$$F%+%&%s%H(B */
  if (swp->Child == LNULL) {
	copy_to_table(t, ebuf);        /* $BMU$^$GMh$?$i(B ebuf $B$r(B table $B$K%3%T!<(B */
	r1=1;
  }
  else r1=count_an_expl(t, swp->Child, ebuf);

  /* $BJ,4t(B */
  if (swlp->NextSw != LNULL)
	r2=count_an_expl(t, swlp->NextSw, ebufcopy);
  else r2=1;

  return r1&&r2;
}

void copy_to_table(int t, int *ebuf){

  static int index=0;
  int i,j;

  if (t==-1) index=0;                           /* t==-1 $B$J$iFCJL$K=i4|2=(B */
  else {
	for(i=0;i<ivNum11;i++) *(G[t].table+index+i)=ebuf[i]; /* $BC1=c$K%3%T!<(B */
	for(i=0;i<=idNum;i++)                                 /* Gamma $B$r7W;;(B */
	  *(G[t].table+index+ivNum11+i)=switch_info[i].Tnum-ebuf[ivNum11+i]; 

	index += ivNum12;
	free(ebuf); /* $B%3%T!<$7=*$o$C$?$i>C$9(B */
  }
}

