%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%                                                             
%%%      PRISM: A Language for Statistical Modeling [ver1.0]    
%%%                                                             
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%  Copyright (C) 1998
%%%    Taisuke Sato, Yoshitaka Kameya, Yasushi Hagiwara, Nobuhisa Ueda,
%%%      Dept. of Computer Science, Tokyo Institute of Technology.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% PRISM Translator:
%%%    translates User's PRISM program to specialized program
%%%    for learning and execution phase.
%%% the task:
%%%    1. to parse PRISM program
%%%    2. to find the probabilistic predicates
%%%    3. to build expl_routines (used for learning and execution)
%%%    4. to build sample_routines (for execution)
%%%    5. to build learn_routines (for learning)

%%% pconsult/1 and pcompile/1:
%%%   PRISM version of SICStus's consult/1 and compile/1.
%%%
%%%   modified by kame on Nov/20/1997
% prism/{1-2} is mixture of bsconsult/1 and bscompile/1
% created by hagi on Feb/24/1998

bsconsult(Files) :- prism([consult,verbose],Files).
bscompile(Files) :- prism([compile,verbose],Files).

prism(Files) :- prism([compile,verbose],Files).

prism(Options,Files) :-
	( '*member*'(consult,Options),!,
	  ( '*member*'(compile,Options),!,
	    message("{PRISM ERROR: prism(~w,_) -- conflicting options `compile' and `consult'.}",[Options]),!,fail
	  ; OP1 = consult)
	; OP1 = compile),
	( '*member*'(quiet,Options),!,
	  ( '*member*'(verbose,Options),!,
	    message("{PRISM ERROR: prism(~w,_) -- conflicting options `verbose' and `quiet'.}",[Options]),!,fail
	  ; OP2 = quiet)
	; OP2 = verbose),
	prism(OP1,OP2,Files).

prism(_,_,[]) :- !.
prism(OP1,OP2,Files) :-
	( internal_consult(Files,OP2),!
	; message('{PRISM ERROR: Translation failed.}'),!,fail),!,
	( OP1 = compile,!,
	  ( Files = [File|_] ; File = Files ),!,
	  ( psmfile(File,File1) ; File1 = File),!,
	  pid(ProcID),!,
	  concat_name(['/tmp/psm',ProcID,File1,'.qsm'],PRISMfile),!,
	  print_routines(PRISMfile),!,
	  ( OP2 = verbose,!,
	    message("{Wrote translated routines into ~w}",[PRISMfile]),!
	  ; true),
	  clean_except_switch,!,
	  compile(PRISMfile),!,
	  concat_name(['rm -f ',PRISMfile],Remove),!,
	  system(Remove),!,
	  ( OP2 = verbose,!,message("{Removed ~w}",[PRISMfile])
      ; true )
	; true),
	query(OP2).

%% internal_consult/1: internal translator
%%   created by kame on Nov/20/1997
% modified by hagi on Feb/24/1998.

internal_consult(Files,OP) :-
        ( OP=verbose,!,
	  trans_start_msg(Files),! % modified by kame on Nov/20/1997.
	; true),
	add_no_use_sentences,!,   % assertz dummy *Sentence* and *Prob_Pred*.
        clean,!,              %  (modified by kame on Feb/27/1998)
        parse(Files,OP),!,         % assertz *Sentence*, *Prob_Pred* and so on.
        make_fixed,!,
        make_routines(OP),!,       % assertz *expl_??* & *sample_??* routines.
        set_seed_time(Seed),!,     % set random seed to the value of time().
        set_epsilon(0.000001,OP),!,% set epsilon to 0.000001.
        init_c_routine,!,          % initialize C routines' global variables.
	( OP=verbose,!,
	  message("{set_seed_time -- random seed set to ~w.}",[Seed]),
	  trans_end_msg(Files),!,
	  ( clause('*Switch*'(_,_,_,_,_,_),true),!,
	    message('{Switches found by translator:}'),!,
	    show_sw
	  ; true ),!
	; true ),!.

% trans_start_msg/1, trans_end_msg/1: Print which files are translated.
%  created by kame on Dec/16/1997.

trans_start_msg(Files) :-
	write('{Loading '),!,print_filenames(Files),!,write('...}'),nl,!.

trans_end_msg(Files) :-
	write('{'),!,print_filenames(Files),!,
	write(' translated to executable programs.}'),nl,!.

print_filenames([File|Files]) :- !,
        write(File),!,
		( Files = [],! ; write(','),!,print_filenames(Files) ).
print_filenames([]) :- !.
print_filenames(File) :- write(File),!.

% make_routines/1: generates learn, expl, sample routines.
% modified by hagi on Feb/24/1998.
make_routines(OP) :- make_learn(OP),!,make_expl,!,make_sample,!.

%%%%%%%%%%%%%%%%%%%%%%%%%% temporal aliases %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
p_compile(Files) :- prism([compile],Files).
p_consult(Files) :- prism([consult],Files).

%%%%%%%%%%%%%%%%% clean up %%%%%%%%%%%%%%%%%
% clean/0, clean_except_switch/0, clean_switch/0, clean_sentences/0,
% clean_routines/0, clean_normal_clauses/0
%   are all modified by kame on Dec/22/1997.
% [NOTE] Since clean_normal_clauses/0, clean_routines/0 refers to the
%        *Sentence* data base, clean_sentence/0 must be done lastly.

clean :- clean_except_switch,!,clean_switch,!.

clean_except_switch :-
	clean_normal_clauses,!,clean_routines,!,clean_sentences,!.

clean_sentences :-
	retractall('*Sentence*'(_,_,_)),!,
	retractall('*Contains*'(_,_)),!,
	retractall('*Prob_Pred*'(_,_,_,_)),!.

clean_switch :- retractall('*Switch*'(_,_,_,_,_,_)),!.

clean_routines :-
	retractall(mlearn(_)),!,
	retractall(mlearn(_,_)),!,
	( setof(EF/ArgN2,
	        (F,SF,ArgN)^('*Prob_Pred*'(F,ArgN,EF,SF),
			             \+(((F=bsw;F=msw),(ArgN=2;ArgN=3))), % added by kame
					     ArgN>=0,                             %  on Jan/6/1998
					     ArgN2 is ArgN+2),                    % bug-fixed
            EFAs),!,                                          %  on Mar/2/1998
      abolish(EFAs)
    ; true ),!,
	( setof(SF/ArgN,
            (F,EF)^('*Prob_Pred*'(F,ArgN,EF,SF),
                    \+(((F=bsw;F=msw),(ArgN=2;ArgN=3))), % added by kame
                    ArgN>=0),                            % on Jan/6/1998
            SFAs),!,
      abolish(SFAs)
    ; true ),!.

clean_normal_clauses :-
	( setof(F/ArgN,
	        (N,M,Ps,H,B,EF,SF)^('*Sentence*'(N,normal,[F/ArgN,M,Ps,H,B]),
			                    \+('*Prob_Pred*'(F,ArgN,EF,SF))),
			FAs),!,
	  abolish(FAs)
	; true ),!.

%%%%%%%%%%%% show the result of translation %%%%%%%%%
% show_db, show_sentence_db, show_routines, show_normal_clauses
%  are all modified by kame on Dec/14/1997.

show_db :- show_sentence_db,!,show_switch_db,!.

% modified by kame on Feb/27/1998
show_sentence_db :-
	listing('*Sentence*'),!,listing('*Prob_Pred*'),!.

show_switch_db :- listing('*Switch*'),!.

%  show_decl/0:  show declarations in user program.
%  <ex.>
%  Suppose that user program has two models named `m1' and `m2'.
%  | ?- show_decl.
%  
%  Target declarations:
%    m1: p/3
%    m2: q/4
%
%  Data declarations:
%    m1: user
%    m2: datafile.dat
%
%  MSW declarations:
%    switch dir: north south east west
%    switch color: red yellow blue
%
%  added by kame on Feb/27/1998

show_decl :- nl,
        show_target_decl,nl,!,show_data_decl,nl,!,show_msw_decl.

show_target_decl :-
        ( setof((SN,Model,F,ArgN),
            '*Sentence*'(SN,target,[Model,F,ArgN]),
            Ts),!,
          show_target_decl(Ts)
    ; message('{PRISM WARNING: No target declaration -- maybe no PRISM program l
oaded.}') ).

show_target_decl(Ts) :-
        message('Target declarations:'),!,show_target_decl1(Ts).

show_target_decl1([(_,Model,F,ArgN)|Ts]) :-
        format("  ~w: ~w/~w~n",[Model,F,ArgN]),!,show_target_decl1(Ts).
show_target_decl1([]).

show_data_decl :-
	( setof((SN,Model,File),'*Sentence*'(SN,data,[Model,File]),Ds),!,
	  show_data_decl(Ds)
    ; message('{PRISM WARNING: No data declaration -- maybe no PRISM program loaded.}') ).

show_data_decl(Ds) :-
        message('Data declarations:'),!,show_data_decl1(Ds).

show_data_decl1([(_,Model,File)|Ds]) :-
        format("  ~w: ~w~n",[Model,File]),!,show_data_decl1(Ds).
show_data_decl1([]).

show_msw_decl :-
        findall((SN,G_id,Values),
                '*Sentence*'(SN,msw,[G_id,_,Values]),
            Ms0),!,
        ( Ms0 = [],!,
          message('{PRISM WARNING: No multi-valued switch declaration -- maybe n
o PRISM program loaded.}') 
        ; sort(Ms0,Ms),!,
          show_msw_decl(Ms) ).

show_msw_decl(Ms) :-
        message('MSW declarations:'),!,show_msw_decl1(Ms).

show_msw_decl1([(_,G_id,Values)|Ms]) :-
        anonymous_vars(G_id,CG_id),!,
        format("  ~w: ",[CG_id]),!,show_msw_decl2(Values),nl,!,
        show_msw_decl1(Ms).
show_msw_decl1([]).

show_msw_decl2([Value|Values]) :-
        format("~w ",[Value]),!,show_msw_decl2(Values).
show_msw_decl2([]).

anonymous_vars(Term,CTerm) :-
        ( var(Term),!,CTerm='_'
    ; Term =.. [F|Args],!,
          anonymous_vars_list(Args,CArgs),!,
          CTerm =.. [F|CArgs] ).
          
anonymous_vars_list([Term|Terms],[CTerm|CTerms]) :-
        anonymous_vars(Term,CTerm),!,
        anonymous_vars_list(Terms,CTerms).
anonymous_vars_list([],[]).

% modified by kame on Jan/6/1998
show_routines :-
        listing('mlearn'),!,
        ( setof(EF/ArgN2,
                (F,SF,ArgN)^('*Prob_Pred*'(F,ArgN,EF,SF),
                         \+(((F=bsw;F=msw),(ArgN=2;ArgN=3))), 
                                     ArgN2 is ArgN+2,
                                                 ArgN>=0),
             EFAs),!,
          listing(EFAs)
    ; true ),!,
        listing('*Expl_SW*'),!, % added by kame on Dec/18/1997.
        ( setof(SF/ArgN,
                (F,EF)^('*Prob_Pred*'(F,ArgN,EF,SF),
                                \+(((F=bsw;F=msw),(ArgN=2;ArgN=3))),
                                        ArgN>=0),
            SFAs),!,
          listing(SFAs)
    ; true ),!.

show_normal_clauses :-
	( setof(F/ArgN,
	        (N,M,Ps,H,B,EF,SF)^('*Sentence*'(N,normal,[F/ArgN,M,Ps,H,B]),
                                \+('*Prob_Pred*'(F,ArgN,EF,SF))),
		    FAs),!,
	  listing(FAs)
	; true ),!.

%%% print_routines(File) -- writes routines to File.
% modified by hagi on Oct/1/1997,Feb/24/1998.
print_routines(File) :-
	nofileerrors,
	( tell(File),!,
	  show_routines,!,
	  show_normal_clauses,!,
	  listing('*Prob_Pred*'),!,listing('*Sentence*'),!,
	  told,!	
	; message("{PRISM ERROR: Cannot write ~w.}",[File]),!,fail ).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%  The status of PRISM system is one of the following,
%%%   - initial,
%%%   - compiled,
%%%   - consulted.
%%%  We assert a predicate *Status* to indicate which is the current
%%%  state. *Status*(compiled) and *Status*(consulted) indicates the
%%%  state 'compiled' and 'consulted' respectively.   If no *Status*
%%%  predicate is asserted,  it indicates that  the current state is
%%%  'initial'.
%%%
%%%  [State Transition Diagram]
%%%         _____________________________________________
%%%        |                   compile                   |
%%%        |                                             v
%%%   +---------+  consult  +-----------+  compile  +----------+
%%%   | initial | --------> | consulted | --------> | compiled |
%%%   +---------+           +-----------+           +----------+
%%%                             |   ^
%%%                             +---+ consult
%%%
%%%  The following predicates are all created by kame on Nov/20/1997.

% is_status(Status) :- The status of PRISM system is Status.
% [NOTE] if user program is compiled, modification must be forbidden.

is_status(compiled) :- clause('*Status*'(compiled),true).
is_status(consulted) :- clause('*Status*'(consulted),true).
is_status(initial).

% show_status/0 (User command): prints the current status.

show_status :- is_status(Status),!,message("Status: ~w",[Status]).

% can_read_database/0:
%   checks if we can refer to *.psm database such as *Sentence* database,
%   *expl_??* or *sample_??* routines (if not, fails with message).
% [NOTE]
%   In initial status, no *Sentence* clause is asserted.
can_read_database :-
	( is_status(initial),!,
	  message('{PRISM ERROR: No PRISM program loaded.}'),!,
	  fail
    ; true ),!.

% can_write_database/0:
%   checks if we can modify *.psm database (if not, fails with message).
% [NOTE]
%   In status 'compiled', *.psm database cannot be accessed by using
%   retract/1 or clause/2. (We already do not use clause/2.) Therefore,
%   we must forbid further modification (by p_assertz/1, for example).

can_write_database :-
	( is_status(initial),!,
	  message('{PRISM ERROR: No PRISM program loaded.}'),!,
	  fail
    ; is_status(compiled),!,
	  message('{PRISM ERROR: A previously compiled program exists -- Further modification is forbbiden.}'),!,
	  fail
    ; true ),!.
