%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%%  modifier.pl: Modification of PRISM program
%%%  
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%  Copyright (C) 1998
%%%    Taisuke Sato, Yoshitaka Kameya, Yasushi Hagiwara, Nobuhisa Ueda,
%%%      Dept. of Computer Science, Tokyo Institute of Technology.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%    available commands are:
%%%      p_asserta/1, p_assertz/1, p_retract/1, p_retractall/1,
%%%      p_abolish/1, p_abolish/2
%%%
%%%  [NOTE] 
%%%     After PRISM program is modified, it is possible that probabilstical
%%%     causation among probabilstic atom is also changed. So we should
%%%     clean-up current *expl_??* and *sample_??* routine, and should
%%%     re-make these routines.

% clean_modifier/0: clean up previous parse infomation and routines.

clean_modifier :-
	clean_routines,!,
	clean_normal_clauses,!.

% remake_routines/0: clean-up & make_routines again.

% avoid_abolish_warning added by kame on Mar/2/1998
remake_routines :-
	add_no_use_sentences,!,  % renamed by kame on Feb/27/1998
	find_prob_pred,!,        % \
	avoid_abolish_warning,!, %  [NOTE] do not change the order of 
	clean_modifier,!,        % /             these 3 clauses.
	make_fixed,!,
	make_routines(quiet),!,
	add_no_use_sentences.    % added by kame on Feb/27/1998

avoid_abolish_warning :-
	( setof(F/ArgN,(EF,SF)^('*Prob_Pred*'(F,ArgN,EF,SF),ArgN >= 0),Preds)
    ; Preds=[] ),!,
	avoid_abolish_warning(Preds).

avoid_abolish_warning([F/ArgN|Preds]) :- 
	ArgN2 is ArgN+2,!,
	gen_no_use_args(ArgN2,EArgs),!,
	gen_no_use_args(ArgN,SArgs),!,
	concat_name(['*expl_',F,'*'],EF),!,
	concat_name(['*sample_',F,'*'],SF),!,
	ETerm =.. [EF|EArgs],!,
	STerm =.. [SF|SArgs],!,
	assertz(ETerm),!,
	assertz(STerm),!,
	avoid_abolish_warning(Preds).
avoid_abolish_warning([]).

gen_no_use_args(0,[]) :- !.
gen_no_use_args(N,['*no_use*'|NoUseArgs]) :-
	N1 is N-1,
	gen_no_use_args(N1,NoUseArgs).

%%%
%%%  p_asserta/1:
%%%
p_asserta(Clauses) :-
	( is_list(Clauses),!,p_asserta_list(Clauses)
	; Clauses=(Head :- Body),!,p_asserta(Head,Body)
	; p_asserta(Clauses,true) ),!,
	remake_routines,!.

p_asserta(Head,Body) :-
	functor(Head,F,ArgN),!,
	cd_error(p_asserta,F,ArgN,Head),!,
	( setof((N,M,Ps,H,B),
	        retract('*Sentence*'(N,normal,[F/ArgN,M,Ps,H,B])),
		    NMPHBs),!,
	  inc_relevant_local_SN(1,F,ArgN,NMPHBs)
	; true ),!,
	inc_unique_local_SN(F,ArgN,1),!,
	unique_sentence_number(USN),!,
	collect_pred(Body,Preds),!,
	assertz('*Sentence*'(USN,normal,[F/ArgN,0,Preds,Head,Body])),!.
	
p_asserta_list(Clauses) :-
	divide_to_groups(Clauses,DivCls),!,
	p_asserta_list_sub(DivCls),!.

p_asserta_list_sub([[F/ArgN|HBs]|DivCls]) :-
	length(HBs,K),!,
	p_asserta_list(K,F,ArgN,HBs),!,
	p_asserta_list_sub(DivCls).
p_asserta_list_sub([]) :-!.

p_asserta_list(K,F,ArgN,HBs) :-
	( setof((N,M,Ps,H,B),
	        retract('*Sentence*'(N,normal,[F/ArgN,M,Ps,H,B])),
			NMPHBs),!,
	  inc_relevant_local_SN(K,F,ArgN,NMPHBs)
	; true ),!,
	inc_unique_local_SN(F,ArgN,K),!,
	length(HBs,NHB),!,
	p_asserta_list(0,NHB,F,ArgN,HBs),!.

p_asserta_list(M,NHB,F,ArgN,[(Head,Body)|HBs]) :-
	( cd_error(p_asserta,F,ArgN,Head),!,
	  unique_sentence_number(USN),!,
	  collect_pred(Body,Preds),!,
	  assertz('*Sentence*'(USN,normal,[F/ArgN,M,Preds,Head,Body])),!,
	  M1 is M+1,!,
	  NHB1 = NHB
	; M1 = M, NHB1 is NHB-1 ),!,
	p_asserta_list(M1,NHB1,F,ArgN,HBs).
p_asserta_list(N,N,_,_,[]) :- !.
	
% divide_to_groups/2
% <ex.>
% | ?- divide_to_groups([(a:-b),(f(X):-c(X)),(e:-g),(f(X):-d(X))],DivCls).
% 
% DivCls = [[a/0,(a,b)],[f/1,(f(X),c(X)),(f(X),d(X))],[e/0,(e,g)]] ?

divide_to_groups(Cls,DivCls) :-
	divide_to_groups(Cls,[],DivCls),!.

divide_to_groups([Clause|Cls],DivCls0,DivCls) :-
	( Clause = (Head :- Body); Head = Clause, Body = true ),!,
	HB = (Head,Body),!,
	functor(Head,F,ArgN),!,
	divide_to_groups(F,ArgN,HB,DivCls0,DivCls1),!,
	divide_to_groups(Cls,DivCls1,DivCls).
divide_to_groups([],DivCls,DivCls) :- !.

divide_to_groups(F,ArgN,HB,[[F/ArgN|HBs0]|DivCls],[[F/ArgN|HBs]|DivCls]) :- !,
	insert_tail(HB,HBs0,HBs),!.
divide_to_groups(F,ArgN,HB,[[Spec|HBs]|DivCls],[[Spec|HBs]|DivCls1]) :- !,
	divide_to_groups(F,ArgN,HB,DivCls,DivCls1).
divide_to_groups(F,ArgN,HB,[],[[F/ArgN,HB]]) :- !.	

% inc_relevant_local_SN(K,F,ArgN,NMHB):
%   increase local sentence number of sentences described as NMPHB by K.

inc_relevant_local_SN(K,F,ArgN,[(N,M,Ps,H,B)|NMPHBs]) :-
	M1 is M+K,!,
	assertz('*Sentence*'(N,normal,[F/ArgN,M1,Ps,H,B])),!,
	inc_relevant_local_SN(K,F,ArgN,NMPHBs).
inc_relevant_local_SN(_,_,_,[]) :- !.

%%%
%%%  p_assertz/1:
%%%

p_assertz(Clauses) :-
	( is_list(Clauses),!,p_assertz_list(Clauses)
	; Clauses=(Head :- Body),!,p_assertz(Head,Body)
	; p_assertz(Clauses,true) ),!,
	remake_routines,!.

p_assertz(Head,Body) :-
	functor(Head,F,ArgN),!,
	cd_error(p_assertz,F,ArgN,Head),!,
	unique_sentence_number(USN),!,
	unique_local_SN(F/ArgN,ULSN),!,
	collect_pred(Body,Preds),!,
	assertz('*Sentence*'(USN,normal,[F/ArgN,ULSN,Preds,Head,Body])),!.

p_assertz_list([(Head:-Body)|Clauses]) :-
	( p_assertz(Head,Body),!
	; true ),!,
	p_assertz_list(Clauses).
p_assertz_list([]) :-!.	

%%%
%%%  p_assert/1 is the same as p_assertz/1 except name itself.
%%%
p_assert(Clauses) :-
	( is_list(Clauses),!,p_assert_list(Clauses)
	; Clauses=(Head :- Body),!,p_assert(Head,Body)
	; p_assert(Clauses,true) ),!,
	remake_routines,!.

p_assert(Head,Body) :-
	functor(Head,F,ArgN),!,
	cd_error(p_assert,F,ArgN,Head),!,
	unique_sentence_number(USN),!,
	unique_local_SN(F/ArgN,ULSN),!,
	collect_pred(Body,Preds),!,
	assertz('*Sentence*'(USN,normal,[F/ArgN,ULSN,Preds,Head,Body])),!.

p_assert_list([(Head:-Body)|Clauses]) :-
	( p_assert(Head,Body),!
	; true ),!,
	p_assert_list(Clauses).
p_assert_list([]) :-!.	

%%%
%%%  p_retract/1:
%%%

p_retract(Clauses) :-
	( is_list(Clauses),!,p_retract_list(Clauses)
	; Clauses=(Head :- Body),!,p_retract(Head,Body)
	; p_retract(Clauses,true) ),!,
	remake_routines,!.

p_retract(Head,Body) :-
	functor(Head,F,ArgN),!,
	cd_error(p_retract,F,ArgN,Head),!,
	( retract('*Sentence*'(_,normal,[F/ArgN,_,_,Head,Body])),!
	; message("{PRISM WARNING: p_retract(~w,~w) -- No matching sentence.}",
	          [Head,Body]) ),!.

p_retract_list([(Head:-Body)|Clauses]) :-
	( p_retract(Head,Body),!
	; true ),!,
	p_retract_list(Clauses).
p_retract_list([]) :-!.

%%%  p_retractall/1
%%%
%%%  p_retractall(Head)
%%%  p_retractall(Heads) -- Heads is the list of Head

p_retractall(Heads) :-
	( is_list(Heads),!,p_retractall_list(Heads)
	; p_retractall_sub(Heads) ),!,
	remake_routines,!.

p_retractall_sub(Head) :-
	functor(Head,F,ArgN),!,
	cd_error(p_retractall,F,ArgN,Head),!,
	( retract('*Sentence*'(_,normal,[F/ArgN,_,_,Head,_])), % fail-driven loop
      fail                                                 % *do not* insert 
	; true ),!.                                            % any cut-symbols.

p_retractall_list([Head|Heads]) :-
	( p_retractall_sub(Head),!
	; true ),!,
	p_retractall_list(Heads).
p_retractall_list([]) :-!.

% control declaration check
% [NOTE] Comm is one of {p_assertz,p_asserta,p_assert,p_retract,p_retractall}
% modified by hagi on Oct/1/1997
cd_error(Comm,F,ArgN,Head) :-
	( '*member*'(F/ArgN,[target/2,target/3,data/1,data/2,fixed/1,fixed/2,values/2]),!,
	  ( ( Comm=p_assertz ; Comm=p_asserta ; Comm=p_assert ),!,
	    Act=assert
	  ; ( Comm=p_retract ; Comm=p_retractall ),!,
	    Act=retract ),!,
	  ( Body=true,!,
	    message("{PRISM ERROR: ~w(~w) -- can't ~w control delcaration ~w.}",[Comm,Head,Act,Head]),!,fail
	  ; message("{PRISM ERROR: ~w((~w :- ~w)) -- can't ~w control declaration ~w.}",[Comm,Head,Body,Act,Head]),!,fail ),
	    message("{PRISM ERROR: (~w :- ~w) -- Control declaration ~w must be unit clause.}",[Head,Body,Head]),!,fail
	; true ),!.

%%%
%%%  p_abolish/1
%%%

p_abolish(Preds) :-
	( is_list(Preds),!,p_abolish_list(Preds)
	; p_abolish_sub(Preds) ),!,
	remake_routines,!.

% modified by hagi on Oct/1/1997
% modified by kame on Mar/2/1998
p_abolish_sub(Pred) :-
	( is_pred_spec(Pred,Name,Arity),!,
	  ( '*member*'(Pred,[target/2,target/3,data/1,data/2,fixed/1,
	                     fixed/2,values/2]),!,
	    message("{PRISM ERROR: p_abolish(~w) -- can't abolish control declaration ~w.}",[Pred,Pred]),!,fail
	  ; p_abolish_pred(Name,Arity) )
    ; atom(Pred),!,
	  p_abolish_pred(Pred)
    ; message("{PRISM ERROR: p_abolish(~w) -- ~w must be the form of Name/Arity, where Name is a predicate name, Arity is a non-negative integer.}",
	          [Pred,Pred]),!,fail ),!.

% added by kame on Mar/2/1998
p_abolish_pred(Name,Arity) :-                          % fail-driven loop:
	( retract('*Sentence*'(_,normal,[Name/Arity,_,_,_,_])), 
      fail                                             %   *do not* insert
	; true ),!.                                        %   cut-symbol (!).
	
p_abolish_pred(Name) :-
	( retract('*Sentence*'(_,normal,[Name/_,_,_,_,_])), % fail-driven loop:
      fail                                              %   *do not* insert
	; true ),!.                                         %   cut-symbol (!).
	
p_abolish_list([Pred|Preds]) :-
	p_abolish_sub(Pred),!,p_abolish_list(Preds).
p_abolish_list([]):-!.

%%%  
%%%  p_abolish/2
%%%  

p_abolish(Name,Arity) :- p_abolish_sub(Name/Arity).

%%%
%%%  p_listing/{0,1,2}: print *Sentence* database.
%%%

%% p_listing/0: print all sentences.

p_listing :-
	message('{Sentences in PRISM sentence database:}'),nl,!,
	( setof((Name,Arity,M,Head,Body),
	        (N,Ps)^'*Sentence*'(N,normal,[Name/Arity,M,Ps,Head,Body]),
			Clauses),!,
%	  message('{Senteces in PRISM sentence database:}'),!,
	  print_clauses(Clauses)
	; true ),!.

%% p_listing/1

p_listing(Preds) :-
	message('{Sentences in PRISM sentence database:}'),nl,!,
	( is_list(Preds),!,p_listing_list(Preds)
	; p_listing_sub(Preds) ),!.

p_listing_list([Pred|Preds]) :-
	p_listing_sub(Pred),nl,!,
	p_listing_list(Preds).
p_listing_list([]) :-!.

% modified by hagi on Oct/1/1997
p_listing_sub(Pred) :-
  	( is_pred_spec(Pred,Name,Arity),!,
	  ( '*member*'(Pred,[target/2,target/3,data/1,data/2,fixed/1,fixed/2,values/2]),!,
	    message("{PRISM ERROR: p_listing(~w) -- can't show control declaration directly. Use show_decl/0.}",[Pred]),!,fail
	  ; p_listing_pred(Name,Arity) )
	; atom(Pred),!,
	  p_listing_pred(Pred)
	; message("{PRISM ERROR: p_listing(~w) -- ~w must be the form of Name or Name/Arity , where Name is a predicate name, and Arity is non-negative.}",[Pred,Pred]),!,
	  fail ),!.
	
%% body part of listing/{0,1,2}

p_listing_pred(Name) :-
	( setof((Name,Arity,M,Head,Body),
	        (N,Ps)^'*Sentence*'(N,normal,[Name/Arity,M,Ps,Head,Body]),
			Clauses),!,
	  print_clauses(Clauses)
	; true ),!.

p_listing_pred(Name,Arity) :-
	( setof((Name,Arity,M,Head,Body),
	        (N,Ps)^'*Sentence*'(N,normal,[Name/Arity,M,Ps,Head,Body]),
			Clauses),!,
	  print_clauses(Clauses)
	; true ),!.
	
print_clauses([(Name,Arity,_,Head,Body),
	           (NextN,NextA,M,NextH,NextB)|Clauses]) :-
	print_clause(Name,Arity,NextN,NextA,Head,Body),!,
	print_clauses([(NextN,NextA,M,NextH,NextB)|Clauses]).
print_clauses([(_,_,_,Head,Body)]) :-
	print_clause(Head,Body),!.
print_clauses([]) :-!.	

print_clause(Name,Arity,NextN,NextA,Head,Body) :-
	print_clause(Head,Body),!,
	( Name=NextN,Arity=NextA,!
	; nl ),!.

print_clause(Head,Body) :-
	numbervars(Head,0,C),!,
	( Body=true,!,
	  format("~w.~n",[Head])
	; format("~w :-~n",[Head]),!,
	  numbervars(Body,C,_),!,
	  print_listing_body(Body) ),!.

print_listing_body(Body) :-
	'*write_spaces*'(4),!,
	print_listing_body(1,4,Body),
	format(".~n",[]).

% N: indent width (the number of spaces).
% S: Status
print_listing_body(S,N,Body) :-
	( Body=(C,Cs),!,
	  print_listing_body(1,N,C),!,
	  format(",~n",[]),!,
	  '*write_spaces*'(N),!,
	  print_listing_body(1,N,Cs)
     ; Body=(D;Ds),!,
	   N2 is N+2,
	  ( S=1,!,
	    format("( ",[]),!,
		print_listing_body(2,N2,D),!,nl,
		'*write_spaces*'(N),write('; '),
		( Ds=(_;_),!,
		  print_listing_body(2,N,Ds)
	    ; print_listing_body(2,N2,Ds) ),
		nl,'*write_spaces*'(N),!,format(")",[])
	  ; S=2,!,
	    print_listing_body(2,N2,D),!,nl,
		'*write_spaces*'(N),write('; '),
		( Ds=(_;_),!,
		  print_listing_body(2,N,Ds)
	    ; print_listing_body(2,N2,Ds) ) )
	; write(Body) ).

'*write_spaces*'(0) :- !.
'*write_spaces*'(N) :- write(' '),N1 is N-1,!,'*write_spaces*'(N1).
