%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%                                                             
%%%      exec.pl: built-in commands for execution subsystem     
%%%                                                             
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%  Copyright (C) 1998
%%%    Taisuke Sato, Yoshitaka Kameya, Yasushi Hagiwara, Nobuhisa Ueda,
%%%      Dept. of Computer Science, Tokyo Institute of Technology.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%                                                             
%%%  Executable commands:                                       
%%%   (1) Sampling Execution                                    
%%%   (2) Answer with Probability                               
%%%   (3) Answer with Formula                                   
%%%   (4) Miscellaneous commands:                               
%%%      - switch handling commands (defined in sw_handling.pl) 
%%%      - sampling commands                                    
%%%      - ...                                                  
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%% (1) Sampling Execution
%%% <ex.>
%%%     | ?- sample(bloodtype(X)).
%%%  
%%%     X = a ?
%%%
%%% Also available for Utility program.
%%% <ex.>
%%% go(Loc,Dir) :-
%%%     ( is_wall(Loc),
%%%       sample(coin(X)),
%%%       ( X = head,!,Dir=right
%%%       ; Dir=left )
%%%     ; Dir=forward ).

sample(msw(G_id,N,R)) :-
	retractall('*Sampled_MSW*'(_,_,_)),!,sample_msw(G_id,N,R),!.
sample(msw(G_id,R)) :-
	retractall('*Sampled_MSW*'(_,_,_)),!,sample_msw(G_id,nil,R),!.
sample(bsw(G_id,N,R)) :-
	retractall('*Sampled_MSW*'(_,_,_)),!,sample_msw(G_id,N,R),!.
sample(bsw(G_id,R)) :-
	retractall('*Sampled_MSW*'(_,_,_)),!,sample_msw(G_id,nil,R),!.

% sample/1 modified by kame on Dec/14/1997.

sample((C,Conj)) :- !,
	message("{PRISM ERROR: sample(~w) -- conjunction is not allowed for sampling.}",[(C,Conj)]),!,fail.

sample((D;Disj)) :- !,
	message("{PRISM ERROR: sample(~w) -- disjunction is not allowd for sampling.}",[(D,Disj)]),!,fail.

sample(Term) :-
	functor(Term,F,ArgN),!,
	( '*Prob_Pred*'(F,ArgN,_,SF),!,
	  retractall('*Sampled_MSW*'(_,_,_)),
	  Term =.. [F|Args],!,
	  STerm =.. [SF|Args],!,
	  call(STerm)
    ; message("{PRISM ERROR: sample(~w) -- predicate ~w/~w must be probabilistic.}",[Term,F,ArgN]),!,fail ),!.

sample_msw(G_id,N,R) :-
	( var_check((G_id,N),yes),!,
	  message("{PRISM ERROR: sample(msw(~w,~w,_)) -- ~w and ~w must be instanciated.}",[G_id,N,G_id,N]),!,fail
	; true ),!,
	( clause('*Sampled_MSW*'(G_id,N,X),true),!,
	  R=X
	; sample_msw(G_id,X),!,assertz('*Sampled_MSW*'(G_id,N,X)),!,
	  R=X ).

% [NOTE] G_id must be instanciated in advance.
% sample_msw/2, sample_msw1/5 modified by hagi on Jan/5/1998
sample_msw(G_id,X) :-
	( clause('*Switch*'(G_id,_,_,_,_,undef),true),!,
	  message("{PRISM ERROR: Parameter of switch ~w is not assigned yet. Run show_sw/0 for confirmation, or run EM learning or set_sw/{1,2} for setup.}",
	          [G_id]),!,fail
	; clause('*Switch*'(G_id,_,_,_,Values,Pbs),true),!,
	  sum_list(Pbs,Sum),
	  random_float(Sum,R),
	  sample_msw1(Pbs,R,Values,X)
	; message("{PRISM ERROR: sample(msw(~w,_,_)) -- No such Switch ~w.}",[G_id,G_id]),  % bug-fixed by kame on Dec/13/1997.
	  !,fail ),!.

sample_msw1(Pbs,R,Vs,X) :- sample_msw1(0.0,Pbs,R,Vs,X),!.

sample_msw1(CPb,[Pb|Pbs],R,[V|Vs],X) :-
	CPb1 is CPb+Pb,!,
	( R < CPb1,!,X=V
	; Pbs=[],!,X=V                   % bug fixed by kame on Feb/23/1998
	; sample_msw1(CPb1,Pbs,R,Vs,X) ).

%%% (2) Answer with Probability
%%% <ex.>
%%%     | ?- prob((bloodtype(a);bloodtype(b)),P).
%%%
%%%     X = 0.6 ?
%%%
%%% Also available for meta-programming of PRISM 
%%% <ex.>
%%% do(Loc,Act) :- do Act at the location Loc
%%% wumpus(Loc) :- wumpus(a kind of monster) is at the location Loc
%%% move(L,Act,L') :- move by Act from L to L'
%%%
%%% do(Loc,Act) :- 
%%%     move(Loc,forward,NewLoc1),
%%%     move(Loc,backward,NewLoc2),
%%%     prob(wumpus(NewLoc1),P1), 
%%%     prob(wumpus(NewLoc2),P2),
%%%     ( P1 > P2, Act=forward   % move less dengerous place
%%%     ; Act=backward ). 

prob(Body,Prob) :-
	collect_pred(Body,Preds),      % borrowed from parsing routine
	( prob_pred_check(Preds,yes),!
	; message("{PRISM ERROR: prob(~w,_) -- ~w cannot have non-probabilistic atom.}",
	          [Body,Body]),!,fail ),!,
	( prob_calculate(Body,Prob)    % modified by kame on Dec/13/1997.
    ; Prob is 0.0 ),!.

prob(Body) :-
	collect_pred(Body,Preds),!,    % borrowed from parsing routine
	( prob_pred_check(Preds,yes),!
	; message("{PRISM ERROR: prob(~w) -- ~w cannot have non-probabilistic atom.}",
	          [Body,Body]), fail ),
	( prob_calculate(Body,Prob),!,print_prob(Body,Prob)
    ; nl,print_term(0,Body),nl,!,
	  message('is false with probability 1.') ),!.

% messages modified by kame on Mar/1/1998
cprob(Body,Cond,Prob) :-
	collect_pred(Body,BPreds),
	collect_pred(Cond,CPreds),!,
	( prob_pred_check(BPreds,yes),!
	; message("{PRISM ERROR: cprob(~w,_,_) -- ~w cannot have non-probablisitic atom.}",[Body,Body]),!,fail ),!,
	( prob_pred_check(CPreds,yes),!
	; message("{PRISM ERROR: cprob(_,~w,_) -- ~w cannot have non-probablisitic atom.}",[Cond,Body]),!,fail ),!,
	( var_check(Cond,no),!
	; message("{PRISM ERROR: cprob(_,~w,_) -- ~w must be ground.}",
	          [Cond,Cond]),!,fail ),!,
	( prob_calculate(Cond,CProb),!
	; message("{PRISM ERROR: Condition ~w can not be explained -- can't calculate conditional probability.}",[Cond]),!,fail ),!,
	( prob_calculate((Body,Cond),BProb),!
	; BProb is 0.0,!,
	  message("{PRISM WARNING: ~w can't be explained (when ~w is true).}",
	          [Body,Cond]) ),!,
	( CProb is 0.0,!,
	  message("{PRISM ERROR: The probability of condition ~w is 0.0 -- can't calculate conditional probability.}",[Cond]),!,fail
	; Prob is BProb/CProb ),!.

cprob(Body,Cond) :-
	collect_pred(Body,BPreds),
	collect_pred(Cond,CPreds),!,
	( prob_pred_check(BPreds,yes),!
	; message("{PRISM ERROR: cprob(~w,_) -- ~w cannot have non-probablisitic atom.}",
	          [Body,Body]),!,fail ),!,
	( prob_pred_check(CPreds,yes),!
	; message("{PRISM ERROR: cprob(_,~w) -- ~w cannot have non-probablisitic atom.}",
	          [Cond,Cond]),!,fail ),!,
	( var_check(Cond,no),!
	; message("{PRISM ERROR: cprob(_,~w) -- ~w must be ground.}",[Cond,Cond]),!,fail ),!,
	( prob_calculate(Cond,CProb),!,
	  ( CProb is 0.0,!,
	    message("{PRISM WARNING: The probability of condition ~w is 0.0 -- can't calculate conditional probability.}",[Cond])
	  ; ( prob_calculate((Body,Cond),BProb),!,
	      Prob is BProb/CProb,!,
		  print_cprob(Body,Cond,Prob)
	    ; nl,print_term(Body),!,
		  format("~nis false with probability 1~n(when ",[]),
		  print_term(Cond),!,
		  format(" is true).~n",[]) ))
    ; message("{PRISM WARNING: Condition ~w cannot be explained -- can't calculate conditinal probability.}",[Cond]) ).

% prob_calculate/2: the essential part of prob/1-2 or cprob/2-3
%   [NOTE] export_prob_calculate/1, count_prob_ans/0, calc_Pdb/1
%          are all C interface predicates.

prob_calculate(Body,Prob) :-
	init_prob_calculate,!,
	get_expl_formula(Body,Codes),!, % get expls and keep them in TRIE structure.
	export_prob_calculate(Codes),!, % tell infos about expls to C routine.
	count_prob_ans,!,               % build explanation table.
	calc_Pdb(Prob),!.               % C interface predicate
	
% initialization and preparation for prob_calculate/2
init_prob_calculate :-
	retractall('*Occ_Prob_SW*'(_,_,_)),!,
	retractall('*tmp_goal*'(_,_)),!,
	retractall('*exists_prob_ans*'),!,
	init_prob_trie,!,           % \
	init_prob_table,!,          %  C interface predicate
	prepare_prob_trie.          % /

% get_expl_formula:
%   get all explanations for Body (by Prolog)
%   and keep them in trie (by C routine).
% [NOTE] A similar technique for collection of explanation is
%        used in gen_ans_loop/2.
%
% modified by kame on Dec/13/1997.
%  - introduced *exists_prob_ans*, cancel_prob.

get_expl_formula(Body,Codes) :-
	% make_prob_expl_body modified by kame on Dec/15/1997.
	make_prob_expl_body((s0,_),tail,X,Y,Body,EBody),!,
	assertz(('*tmp_goal*'(X,Y) :- EBody)),!,
	( '*tmp_goal*'(Ans0,[]),     % finds an explanation.
                                 % (added by kame on Nov/20/1997)
	  sort(Ans0,Ans1),           % deletes dupl. conjs and sort them.
      deldup(Ans0,Ans2),         % deletes dupl. conjs but does not sort them.
                                 % (added by kame on Dec/13/1997)
	  non_contradictory_with_no_warning(Ans1),
                                 % checks if Ans1 is contradictory.
      assertz('*exists_prob_ans*'),  % records that at least one answer exists.
	  assert_prob_G_id(Ans1),    % assertz G_ids occuring in Ans1,
                                 %   together with integer code of each G_id.
	  encode_prob_ans(Ans2,Ans), % [NOTE] Ans2 reflects the computation
                                 %   process of '*tmp_goal*'/2, rather than
                                 %   Ans1, which is sorted.
                                 % (modified by kame on Dec/13/1997)	  
      export_prob_ans(Ans),      %        
	  fail                       % [NOTE] `fail-driven' loop -- search next Ans0.
	; true ),!,
	( clause('*exists_prob_ans*',true),!  % added by kame on Nov/20/1997.
    ; cancel_prob,!,                      % C interface predicate
	  fail ),!,
	( setof(Code,
	        (G_id,T_ids)^clause('*Occ_Prob_SW*'(G_id,Code,T_ids),true),
			Codes),!
	; Codes=[] ).

% assert_prob_G_id/1: assertz occuring switches.

assert_prob_G_id([C|Conj]) :-
	( C=bsw(G_id,T,V) ; C=msw(G_id,T,V) ),
	( var_check((G_id,T,V),yes),!,
	  % modified by kame on Nov/20/1997.
	  message("{PRISM WARNING: msw(~w,~w,~w) exists in explanation -- ~w, ~w and ~w must be instanciated.}",[G_id,T,V,G_id,T,V]),!,
	  fail
	; true ),!,
	( clause('*Switch*'(G_id,Code,_,_,_,_),true),!
	; msw_values(G_id,Size,Values),
	  unique_code(Code),!,
	  assertz('*Switch*'(G_id,Code,unfixed,Size,Values,undef)) ),!,
	( retract('*Occ_Prob_SW*'(G_id,GC,T_ids)),!,
	  insert_in_order(T,T_ids,T_ids1),         % borrowed from misc.pl
	  assertz('*Occ_Prob_SW*'(G_id,GC,T_ids1))
	; assertz('*Occ_Prob_SW*'(G_id,Code,[T])) ),!,
	assert_prob_G_id(Conj).
assert_prob_G_id([]).

count_prob_ans :-
	count_prob_ans(R),!,
	( export_error(R),!
	; message('{PRISM INTERNAL ERROR: count_prob_ans failed.}'),!,fail ).

encode_prob_ans([A|Ans],[(GC,TC,VC)|Encoded_Ans]) :-
	( A=bsw(G_id,T,V) ; A=msw(G_id,T,V) ),!,
	clause('*Switch*'(G_id,GC,_,_,Values,_),true),!,
	clause('*Occ_Prob_SW*'(G_id,GC,T_ids),true),!,
	( nth_member(T,T_ids,TC)  
	; message("{PRISM INTERNAL ERROR: encode_prob_ans/2 -- ~w does not occur in ~w.}",[T,T_ids]),!,fail ),
	( nth_member(V,Values,VC),!
	; message("{PRISM ERROR: msw(~w,~w,~w) does not take value ~w.}",[G_id,T,V,V]),!,fail ),!,
	encode_prob_ans(Ans,Encoded_Ans).
encode_prob_ans([],[]).

% print_prob/2: pretty printing part of prob/1
print_prob(Body,Prob) :-
	nl,write('The probability of '),
	print_term(Body),!,
	format(" is ~n~6f.~n",[Prob]),!.

% print_cprob/3: pretty printing part of cprob/2
print_cprob(Body,Cond,CProb) :-
	nl,write('The conditinal probability of '),
	print_term(Body),!,nl,
	write('(when '),
	print_term(Cond),!,
	format(" is true) is~n~6f.~n",[CProb]),!.

% prob_pred_check: check if there exists non-prob. predicate
% R is 'yes' or 'no'.
prob_pred_check([(Pred,ArgN)|Preds],R) :-
	( '*Prob_Pred*'(Pred,ArgN,_,_),!,
	  prob_pred_check(Preds,R)
	; R = no ),!.
prob_pred_check([],yes) :- !.
	
%%% (3) Answer with probabilistic formula
%%%
%%% (3-a) Using probf/1 for pretty print
%%% <ex.>
%%%     | ?- probf((bloodtype(a);bloodtype(b))).
%%%
%%%     bloodtype(a) v bloodtype(b)
%%%     is explained by
%%%       bsw(1,dad,1) & bsw(1,mum,1) 
%%%     v bsw(1,dad,1) & ... 
%%%     v ...
%%%
%%% (3-b) Using probf/2 for meta-programming
%%% <ex.>
%%%     | ?- probf((bloodtype(a);bloodtype(b)),Ans).
%%%
%%%     Ans = [[bsw(1,dad,1),bsw(1,mum,1),...],[bsw(1,dad,1),..],...]

probf(Body,Ans) :-
	init_probf,!,
	collect_pred(Body,Preds),!, % borrowed from parsing routine
	prob_pred_check(Preds,R),!,
	( R=yes,!,
	  % make_prob_expl_body/6 modified by kame on Dec/13/1997.
	  make_prob_expl_body((s0,_),tail,X,Y,Body,EBody),!,
	  assertz(('*tmp_goal*'(X,Y) :- EBody)),!,
	  ( '*tmp_goal*'(Ans,[]),
	    sort(Ans,Ans1),               
		deldup(Ans,Ans2),             % added by kame on Dec/13/1997
	    non_contradictory_with_no_warning(Ans1),
		assertz('*Probf_Ans*'(Ans2)), % assertz not-sorted one.
		fail                        % (modified by kame on Dec/13/1997)
	  ; true ),!
	; message("{PRISM ERROR: probf(~w,_) -- ~w must be probabilistic.}",
	          [Body,Body]),!,           % bug fixed by hagi on Feb/24/1998
	  fail ),!,
	findall(A,'*Probf_Ans*'(A),Ans),!.

init_probf :-
	retractall('*is_probf_ans*'),!,
	retractall('*Probf_Ans*'(_)),!,
	retractall('*tmp_goal*'(_,_)).

% modified by kame on Dec/13/1997.
probf(Body) :-
	init_probf,!,
	collect_pred(Body,Preds),!, % borrowed from parsing routine
	prob_pred_check(Preds,R),!,
	( R=yes,!,
	  nl,print_term(0,Body),nl,!,
	  make_prob_expl_body((s0,_),tail,X,Y,Body,EBody),!,
          % make_prob_expl_body/6 modified by kame on Dec/13/1997.
	  assertz(('*tmp_goal*'(X,Y) :- EBody)),!,
	  ( '*tmp_goal*'(Ans,[]),
	    sort(Ans,Ans1),
		deldup(Ans,Ans2), % added by kame on Dec/13/1997
	    non_contradictory_with_no_warning(Ans1),
		probf_sub(Ans2),  % prints not-sorted one.
                          % (modified by kame on Dec/13/1997)
		fail
	  ; true ),!
	; message("{PRISM ERROR: probf(~w) -- ~w must be probabilistic.}",
	          [Body]),!,
	  fail ),!,
	( clause('*is_probf_ans*',true),!
	; message('is false with probability 1.') ),!.

probf_sub(Ans) :-
	( clause('*is_probf_ans*',true),!,
	  print_probf('v ',Ans)
	; assertz('*is_probf_ans*'),!,message('is explained by'),
	  print_probf('  ',Ans) ),!.

print_probf(Indent,Ans) :-
	write(Indent),print_probf2(6,Ans),nl,!.

print_probf2(M,[C|Cs]) :-
	write(C),!,
	( Cs=[],!,true
	; M=1,!,nl,write('& '),print_probf2(6,Cs)
	; M1 is M-1,write(' & '),print_probf2(M1,Cs)),!.

print_term(Body) :- print_term(0,Body),!.
print_term(S,Body) :-
	( Body=(C,Cs),!,
	  print_term(1,C),write(' & '),print_term(1,Cs)
	; Body=(D;Ds),!,
	  ( S=1,!,write('( '),
	    print_term(D),write(' v '),print_term(0,Ds),
		write(' )')
  	; print_term(D),write(' v '),print_term(0,Ds) )
	; write(Body) ),!.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Tools for investigation of P_DB (probably used in debug phase):
%%% - answers disjointness or independence between each of
%%%   probabilistic atoms (or formulas).

%%% disjoint(A,B) -- answers if formula A and B are disjoint.
%%% <ex.> disjoint(bloodtype(a),bloodtype(b)).
%%% [NOTE] A and B must be probabilistic.

%disjoint(A,B) :-
%	probf(A,AnsA),probf(B,AnsB),
%	( A=[],B=[],!,
%	  message(disjoint

%%% Are all explanations of Goal disjoint?
disjoint_expls(Goal) :-
	time(get_expl_formula(Goal,Ans)),time(disjoint_ans(Ans)).

%%% list version of disjoint_ans/2
disjoint_ans([Ans|RestAns]) :-
	disjoint_ans1(Ans,RestAns),disjoint_ans(RestAns).
disjoint_ans([]).

disjoint_ans1(Ans,[RA|RestAns]) :-
	disjoint_ans(Ans,RA),
	disjoint_ans1(Ans,RestAns).
disjoint_ans1(_,[]).
	
%get_expl_formula_disjoint(Body,Id_pairs,Ans) :-
%	retractall('*tmp_goal*'(_,_)),
%	retractall('*tmp_Id_pairs*'(_,_)),
%	make_prob_expl_body(X,Y,('*tmp_goal*',0),Body,EBody), % borrowed from
%	assertz(('*tmp_goal*'(X,Y) :- EBody)),                %   make_expl
%	( setof(A,
%	        B^('*tmp_goal*'(B,[]),
%	           sort(B,A),
%			   non_contradictory(A),
%			   assert_id_pairs_disjoint(A)),
%			Ans)
%	; Ans=[] ),
%	setof((G_id,N),'*tmp_Id_pairs*'(G_id,N),Id_pairs).

%assert_id_pairs_disjoint([bsw(G_id,N,_)|Ans]) :-
%	( '*tmp_Id_pairs*'(G_id,N),!
%	; assertz('*tmp_Id_pairs*'(G_id,N)) ),!,
%	assert_id_pairs_disjoint(Ans).
%assert_id_pairs_disjoint([]).

%%% disjoint_ans(Ans1,Ans2)
%%%  -- answers if answer Ans1 and Ans2 are disjoint.
%%%  <ex.>
%%% disjoint_ans([bsw(1,n,1),bsw(2,m,0),...],[bsw(2,m,1),bsw(2,m,1),..]).
%%%                          ^^^^^^^^^                   ^^^^^^^^^
%%% (Since underlined BSWs are disjoint, Ans1 and Ans2 are disjoint.)
%%%
%%% [NOTE] - Ans1 and Ans2 must be already sorted.
%%%        - All BSWs in Ans1 and Ans2 must be instanciated.
%%%        - disjoint_ans/2 is more effective than
%%%          call(('*append*'(Ans1,Ans2,Ans),\+contradictory(Ans))).

disjoint_ans([bsw(G_id1,N1,R1)|Ans1],[bsw(G_id2,N2,R2)|Ans2]) :-
	( (G_id1,N1) = (G_id2,N2), !,
	  ( R1 = R2, !, disjoint_ans(Ans1,Ans2)
  	; true ) % succeed if disjoint BSWs are found.
	; (G_id1,N1) @< (G_id2,N2), !,
	  disjoint_ans(Ans1,[bsw(G_id2,N2,R2)|Ans2])
	; disjoint_ans([bsw(G_id1,N1,R1)|Ans1],Ans2) ).

disjoint_ans(_,[]) :- fail.
disjoint_ans([],_) :- fail.
disjoint_ans([],[]) :- fail.

%%% disjoint(Formulas)
%%% -- answers which formulas in Formulas are disjoint.
%%% <ex.> disjoint([bloodtype(a),bloodtype(b),bloodtype(o)]).
%%% [NOTE]
%%% - Formulas must contain 2 or more formulas.
%%% - Each formula in Formulas must be probablistic.
%%% - Formulas = [A,B,C] for example, disjoint([A,B,C]) is simply
%%%   an alias to probf((A,B,C)).

%%% independent(A,B) -- answers if formula A and B are independent.

%%%% two Dice routines
%%%%  - not included in translator
%% for multinominal distribution
% dice_multi modified by hagi on Nov/20/1997
% dice_multi modified by hagi on Jan/5/1998

dice_multi(As,Px,X) :- dice0(As,Px,X,dice_multi).
dice_uniform(As,X) :- dice0(As,X,dice_uniform).

%%% dice/{2-3}: mixture of dice_multi/3 and dice_uniform/2
%%% created by hagin on Feb/24/1998

dice(As,Px,X) :- dice0(As,Px,X,dice).
dice(As,X) :- dice0(As,X,dice).

dice0(As,Px,X,Pred) :-
	expand_dice(As,ExAs,Pred),
	dice1(ExAs,Px,X,Pred).

dice0(As,X,Pred) :-
	expand_dice(As,ExAs,Pred),
	length(ExAs,Size),
	make_ratio(Size,Px),
	dice1(ExAs,Px,X,Pred).

dice1(As,Px,X,Pred) :-
	dice_pbplus_check(As,Px,Ps,Sum,Pred),
	random_float(Sum,R),
	dice2(R,0.0,As,Ps,X).

dice2(R,Tmp,[A|As],[P|Ps],X) :-
	Tmp1 is Tmp+P,!,
	( R < Tmp1,!,X=A
	; dice2(R,Tmp1,As,Ps,X)).

dice_pbplus_check(As,Px,Ps,Sum,Pred) :-
	length(As,Size),!,
	( ( plus_to_list_w_sum(Px,Ps,Sum),!
	  ; sum_list(Px,Sum),Ps=Px),
	  ( abs(1-Sum)<0.000001,!,
	    ( length(Ps,Size),!
	    ; message("{PRISM ERROR: ~w(~w,~w,_) -- ~w does not match with ~w.}",
	        [Pred,As,Px,Px,As]),!,fail)
	  ; message("{PRISM ERROR: ~w(_,~w,_) -- ~w must be the form of Pb1+Pb2+..+PbN or [Pb1,Pb2,..,PbN] which equals 1.0.}",
	        [Pred,Px,Px]),!,fail)
	; ratio_to_list(Px,Ps),!,
	  sum_list(Ps,Sum),!,
	  ( length(Ps,Size),!
	  ; message("{PRISM ERROR: ~w(~w,~w,_) -- ~w does not match with ~w.}",
	        [Pred,As,Px,Px,As]),!,fail)
	; message("{PRISM ERROR: ~w(_,~w,_) -- ~w must be the form of Pb1+Pb2+..+PbN which equals 1.0 or R1:R2:..:RN, and each Pbi or Ri must be a non-negative number.}",
	        [Pred,Px,Px]),!,fail),!.

% expand_dice{,1,2} created by hagi on Nov/20/1997

expand_dice(As,ExAs,Pred) :- expand_dice(As,ExAs,[],Pred),!.

expand_dice([A|As],ExAs0,ExAs1,Pred) :-
	expand_dice1(A,ExAs0,ExAs2,Pred),!,
	expand_dice(As,ExAs2,ExAs1,Pred).

expand_dice([],ExAs,ExAs,_) :- !.

expand_dice1(A,ExAs0,ExAs1,Pred) :-
	( A = L-R,!,
	  (integer(L),integer(R),!,
	    ( L =< R,!,expand_dice2(L,L,R,ExAs0,ExAs1)
	    ; message("{PRISM ERROR: ~w(Candidates,_): Candidates contains ~w-~w -- Right-hand side must not be smaller than left-hand side.}",
		[Pred,L,R]),!,fail)
	  ; message("PRISM ERROR: ~w(Candidates,_): Candidates contains ~w-~w -- Each of arguments of '-' must be a integer.}",
		[Pred,L,R]),!,fail)
	; ExAs0=[A|ExAs1]).

expand_dice2(C,L,R,ExAs0,ExAs1) :-
	( C=R,!,ExAs0=[R|ExAs1]
	; ExAs0 = [C|ExAs2],
	  C1 is C+1,!,
	  expand_dice2(C1,L,R,ExAs2,ExAs1)).
