%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%%  em_stable.pl: Prolog part of EM routine (built in learning routine).
%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%  Copyright (C) 1998
%%%    Taisuke Sato, Yoshitaka Kameya, Yasushi Hagiwara, Nobuhisa Ueda,
%%%      Dept. of Computer Science, Tokyo Institute of Technology.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%                                          %%
%%  EM algorithm for Positive Observations  %%
%%                                          %%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Define global variables:
%
% assertz(('*goal*'(Args,Ans):- Bdy)).. Ans =[bsw(G_id,T,ON),...]
% assertz('*Ans*'(N,Ans2))	... Ans2 =[[G_id,#ON,#OFF],...] for Nth goal
% assertz('*P_DB*'(N,P_DB))     ... keep P_{DB} for Nth goal
% assertz('*dice_BSW*'(G_id,Pb))... G_id's Pb for data generation
% assertz('*Pb_BSW*'(G_id,Pb))	... temporary store for G_id's Pb
% assertz('*Id_BSW*'(G_id))	... keep the list of G_id's, auxiliary.
% assertz('*iteration*'(Count))	... count iterations

%-------------- Top loop ---------------

% Learning parameters from positive goals by using the EM algorithm. E.g.,
% :- em_msw(['*goal*'([a,b,b],Ans1),'*goal*'([a,b,b,b,b],Ans2)]) returns
% estimated values of probabilistic parameters in a BSW-program
% generating those goals.
% '*goal*'(Args,Ans) must be designed problem-wise.
% 'Ans' must be of the form: Ans =[{b,m}sw(G_id,T,ON),...]
%
% goal(_,_) renamed to '*goal*'(_,_) by kame on Feb/27/1998

define_goal(('*goal*'(Args,Ans):- Bdy)):-
	( retract(('*goal*'(_,_):-_)),fail ; true ),!,
	assertz(('*goal*'(Args,Ans):- Bdy)),!.

% Initialization both databases of Prolog and C.
initialize :-
	retract_global_vars,!,init_iteration,!,
	init_trie,!,
	init_expl_table,!,
	init_args_code,!,init_tcode,!.

retract_global_vars :-
	retractall('*Occ_SW*'(_,_,_)),!,
	retractall('*Target*'(_,_)),!,      % added by kame on Feb/27/1998
	retractall('*Ans_Args*'(_,_)),!,
	retractall('*Ans*'(_,_,_)),!,       % modified by kame on Feb/27/1998
	retractall('*P_DB*'(_,_)),!,
	retractall('*iteration*'(_)),!,
	retractall('*Log_like*'(_,_)),!.

init_iteration :- assertz('*iteration*'(0)),!.

% Reset *Unique_Args_Code* in advance of EM learning.
% [NOTE(1)] There is no problem in reset of *Unique_Args* since the
%           explanation table held in C routine is discarded together.
% [NOTE(2)] By the definition of unique_args_code/1, it isn't necessary
%           to do "assertz('*Unique_Args*'(0))". 
%           (_But_ to make sure, we do it.)
init_args_code :-
	( retract('*Unique_Args*'(_)) ; true),!,
	assertz('*Unique_Args*'(0)),!.

init_tcode :-
	retractall('*Unique_T_id*'(_,_)),!.

% save global variables into file
% modified by hagi on Feb/24/1998.
save_status :- save_status('EM_status'),!. % default file is `EMstate'.
save_status(File) :-
	( tell(File),!,
	  listing(['*Switch*','*Occ_SW*','*Target*','*Ans_Args*',
	         '*Ans*','*P_DB*','*iteration*','*Log_like*']),!,
	  told,!
	; message("{PRISM ERROR: Cannot write ~w.}",[File]),!,fail ),
	message("{EM learning status saved in ~w.}",[File]),!.

% restore global variables from file
% modified by hagi on Feb/24/1998.
restore_status :- restore_status('EM_status'),!. % default file is `EMstate'.
restore_status(File) :-	
	retract_global_vars,!,
	retractall('*Switch*'(_,_,_,_,_,_)),!,
	( see(File),!,restore_status1,!,seen,!
	; message("{PRISM ERROR: Cannot read ~w.}",[File]),!,fail ),
	message("{Learning status is restored from ~w.}",[File]),!,
	message("{Each switch's parameter overwritten as follows:}",[]),!,
	show_sw,!.
restore_status1 :-
	read(A),!,
	( A=end_of_file,!
	; assertz(A),!,restore_status1 ),!.

% modified by hagi on Oct/1/1997,Nov/20/1997
em_msw(_,[]) :- !,
	message('{PRISM ERROR: No appropriate goals found -- confirm your target declaration.}'),!,fail.

em_msw(Model,Goals):-
	% Goals must be of the form ['*goal*'([b,b],_),'*goal*'([b,b,a],_)]
	% em_msw maximizes the likelihood of '*goal*([b,b],_)&*goal*([b,b,a],_)'
	logs_opening,!,
	initialize,!,      % clean-up explanations (trie) & EM-table.
                       % reset the unique args_code and t_code counter.
	                   % clean-up global vars (*Occ_SW*,*Ans_Args*,*Ans*,..)
	assert_target_pred(Model),!,  % added by kame on Feb/27/1998
	if_logs_off(message('{Searching explanations...}')),!,
	time(gen_ans(Model,Goals,Codes),Time),!,
                       % generate Ans=[bsw(Id,T,ON),..] for Ith Goal,
                       %   and export them to C routine.
                       % assertz('*Ans_Args*'(I,Args)),
                       % assertz('*Ans*'(Args,ArgsCode,Counts)),
                       % assertz('*Occ_SW*'(G_id,Code,T_ids)),
                       % get all G_ids relevant to Goals and sort them.
	if_logs_off(message("{All explanations found (~w msec). Switches in explanations:}",[Time])),!,
	if_logs_off(print_codes(Codes)),!,
	( setof(Code,
	        (G_id,Size,Values,Pbs,T_ids)^(clause('*Switch*'(G_id,Code,
                                          fixed,Size,Values,Pbs),true),
				  clause('*Occ_SW*'(G_id,Code,T_ids),true)),
	        FixedCodes),!
	; FixedCodes=[]),!,
	export_EM(Codes,FixedCodes),!,
	if_logs_off((write('{Building explanation table...'),ttyflush)),!,
	count_ans,!,
	if_logs_off(message('done.}')),!,
	logs_ending,!,
	message('{EM learning started. Initialized switches:}'),!,
	em_loop(Codes),!.

count_ans :- 
	count_ans(R),!,
	( export_error(R),!
	; message('{PRISM INTERNAL ERROR: count_ans failed.}'),!,fail ),!.

show_c_status :- show_trie,!,c_sizes,!,c_goals,!,c_sw,!,show_table,!.

% assert_loglike/1, show_loglike/0, get_loglike/1
%  -- added by kame on Feb/24/1998.

assert_loglike(Loglike) :- assertz('*Log_like*'(final,Loglike)).

show_loglike :- 
	( clause('*Log_like*'(final,Loglike),true),!,
	  format("Loglike=~6f~n",[Loglike])
    ; message("{PRISM ERROR: loglike is undefined -- Maybe EM learning is not done yet.}",[Loglike]),!,fail ).

get_loglike(Loglike) :-
	( clause('*Log_like*'(final,Loglike),true),!
    ; message("{PRISM ERROR: loglike is undefined -- Maybe EM learning is not done yet.}",[Loglike]),!,fail ).

silent_em_loop(Codes) :-
	init_pbs,!,
	import_pbs(Codes),!,print_codes(Codes),!,
	c_EM_loop(I),!,
	import_loglike(Log_like),!,
	import_pbs(Codes),!,
	converged_em_loop(I,Codes,Log_like),!.

converged_em_loop(M,Codes,Loglike) :-
	nth(M,Mth),!,
	message("{EM learning converged at ~w iteration.}",[Mth]),!,
	message("[~w] Loglike=~6f",[M,Loglike]),!,
	print_codes(Codes),!.

display_em_loop(M) :-
	( 0 is M mod 50,!,write(M),ttyflush
	; 0 is M mod 5,!,display_dot
	; true ),!.

gen_ans(Model,Goals,Codes) :-
	init_gen_ans,!,
	gen_ans_loop(Model,1,no,no,Goals),!,
	( setof(Code,
	        (G_id,T_ids)^clause('*Occ_SW*'(G_id,Code,T_ids),true),
			Codes),!
	; Codes=[] ),!.

%%------- Initialization: set up ans_list & G_id list ----------
%% generate logical explanations for I_th goal: '*goal*'(Args,_)
%% store only non_contradictory Ans.

%%% gen_ans:
%%%   constructs TRIE structure of explanations by using C inteface
%%%   predicate export_ans_sw/4.
% gen_ans, gen_ans_loop modified by hagi on Oct/1/1997

%%% added by kame on Dec/13/1997.
init_gen_ans :- 
	prepare_trie(R),!,export_error(R),!,retractall('*exists_ans*'(_)),!.

%% gen_ans_loop(Model,I,IGT,IGP,Goals):
%%   generates answers (minimum support sets) for Ith goal in model Model.
%%
%% [NOTE] 3rd/4th argument IGT and IGP is yes/no flag. IGT is `yes' if
%%        there is some goal  which  have at least one answer  for all
%%        through  the previous (1st,..,"I-1"th) goals.   On the other
%%        hand, IGP is `yes' if the last ("I-1"th) goal has one or more
%%        answers.
%%
%%   *exists_ans*, cancel_trie  are introduced kame on Dec/13/1997.

gen_ans_loop(Model,I,IGT,IGP,['*goal*'(Args,_)|Goals]) :-
	assertz('*Ans_Args*'(I,Args)),!,
	( retract('*Ans*'(Args,AC,Count)),!,  % \
	  Count1 is Count+1,                  %  modified by kame on Feb/271/1998.
	  assertz('*Ans*'(Args,AC,Count1)),!, % /
      if_logs_off(display_dot),!,
      IGT=IGT1,!,
	  IGP1=yes
	; ( '*goal*'(Args,Ans0),     % generates a logical explanation for Args.
	    sort(Ans0,Ans1),         % deletes duplicating conjuncts and sort them.
	    non_contradictory(I,Ans1), % checks if Ans1 is contradictory.
        deldup(Ans0,Ans2),       % deletes dupl. conjs (but doesn't sort them).
                                 % (added by kame on Dec/9/1997)
	    assert_G_id(Model,Ans1), % assertz G_ids and T_ids occurring in Ans1
                                 % and assigns a unique code to each of them.
	    prepare_ans,             % preparation of export_ans.
                                 % (added by kame on Dec/12/1997)
        encode_ans(Ans2,Ans),% [NOTE] Ans2 reflects the computation process
                             %        rather than Ans1, which is sorted one.
                             %  (modified by kame on Dec/9/1997)
	    export_ans(Ans),     % builds a trie structure by C routine.
	    fail                 % [NOTE] search next Ans with backtrack.
      ; true ),!,
      gen_ans_message(Model,I,IGT,IGT1,IGP,IGP1,Args) ),!,
	I1 is I+1,!,
	gen_ans_loop(Model,I1,IGT1,IGP1,Goals).

%gen_ans_loop(Model,_,yes,[]):- nl,count_goals(Model).
gen_ans_loop(_,_,yes,_,[]):- nl,!.
gen_ans_loop(_,_,no,_,[]) :-
	cancel_trie,!,               % C interface predicate
	message('{PRISM ERROR: No goals can be explained. Anything wrong with your program?}'),!,
	fail.

%% gen_ans_message/7: post-processing for fail-driven loop.
%%
%%   added by kame on Dec/13/1997
%%   modified for *Ans*/3 by kame on Feb/27/1998

gen_ans_message(Model,I,IGT,IGT1,IGP,IGP1,Args) :-
	( retract('*exists_ans*'(ArgsCode)),!, % If there exists 
	  IGT1 = yes,!,IGP1 = yes,!,           %   at least one answer,...
	  assertz('*Ans*'(Args,ArgsCode,1)),!,
	  if_logs_off(display_o)
	; IGT1 = IGT,!,IGP1 = no,!,
	  nth(I,Ith),!,
	  ( IGP = no,! ; nl ),!,
	  build_target_atom(Model,Args,TAtom),!,
	  message("{PRISM WARNING: There is no explanations for ~w goal `~w' -- ignored.}",[Ith,TAtom])),!.

%% prepare_ans/0 added by kame on Dec/13/1997.
prepare_ans :-
	( clause('*exists_ans*'(_),true),!
    ; unique_args_code(ArgsCode),!,
	  assertz('*exists_ans*'(ArgsCode)),!,
	  prepare_expl(ArgsCode,R),!,      % C interface predicate
	  export_error(R) ),!.

%%% encode_ans: 
%%% <ex.>
%%%   Suppose that code No. of G_id 'f(x,0)', 'f(y,1)' are  3, 4, and
%%%   code No. of T_id '2nd', '3rd' are 3, 6, respectively,  and that
%%%   switch 'f(x,0)' and 'f(y,1)' takes {a,b,c} and {a,b,c,d} as its
%%%   value respectively. Then,
%%%   
%%%
%%%   | ?- encode_ans([msw(f(x,0),2nd,b),msw(f(y,1),3rd,d)], Encoded_Ans).
%%%
%%%   Encoded_Ans = [(3,3,1),(4,6,3)]?

encode_ans([A|Ans],[(GC,TC,VC)|Encoded_Ans]) :-
	( A=bsw(G_id,T,V) ; A=msw(G_id,T,V) ),!,
	clause('*Switch*'(G_id,GC,_,_,Values,_),true),!,
	clause('*Occ_SW*'(G_id,GC,T_ids),true),!,
	( nth_member(T,T_ids,TC),!
	; message("{PRISM INTERNAL ERROR: encode_ans/2 -- ~w does not occur in ~w.}",[T,T_ids]),!,fail ),!,
	( nth_member(V,Values,VC),!
	; message("{PRISM ERROR: msw(~w,~w,~w) does not take value ~w.}",[G_id,T,V,V]),!,fail ),!,
	encode_ans(Ans,Encoded_Ans),!.
encode_ans([],[]) :- !.

%%% all_expls_are_disjoint:

all_expls_are_disjoint :-
	( clause('*Disjoint*',true),! ; assertz('*Disjoint*') ),!,
	message('{All explanations are disjoint.}'),!.
are_all_expls_disjoint :- '*Disjoint*',!.

%%% use_silent_em_routine:

use_silent_em_routine :-
	( clause('*Silent*',true),! ; assertz('*Silent*')),!,
	message('{Silent EM routine will be used.}'),!.
not_use_silent_em_routine :-
	( retract('*Silent*'),! ; true ),!,
    message('{PRISM WARNING: Silent EM routine is already not used.}'),!.

%% assert_G_id:
% record G_id and Id_pair occurred in the explanation for Goals.
% [NOTE] G_id,T,V must be instanciated.
% modified by hagi on Oct/1/1997
% modified by kame on Mar/2/1997

assert_G_id(Model,[C|Conj]):-
	( C=bsw(G_id,T,V) ; C=msw(G_id,T,V) ),!,
	( clause('*Switch*'(G_id,Code,_,_,_,_),true),!
	; msw_values(G_id,Size,Values),!, % borrowed from sw_handling.pl
	  unique_code(Code),!,
	  assertz('*Switch*'(G_id,Code,unfixed,Size,Values,undef)) ),!,
	( retract('*Occ_SW*'(G_id,GC,T_ids)),!,
	  insert_in_order(T,T_ids,T_ids1),!,
	  assertz('*Occ_SW*'(G_id,GC,T_ids1))
	; assertz('*Occ_SW*'(G_id,Code,[T])),!
%	  contains_check(Model,G_id) ),!,
	  ),!,
	assert_G_id(Model,Conj),!.
assert_G_id(_,[]) :- !.

% Check if Ans contains uninstanciated or conflicting switches.
% [NOTE] Ans must be sorted in advance.
% modified by kame on Mar/2/1998

non_contradictory(I,[C|Cj]):-
	( ( C=msw(G_id,T,V) ; C=bsw(G_id,T,V) ),!
	; nth(I,Ith),!,
	  message("{PRISM INTERNAL ERROR: ~w found in the explanation for ~w goal.}",[C,Ith]),
	  !,fail ),!,
	( var_check((G_id,T,V),yes),!,
	  nth(I,Ith),!,
	  message("{PRISM WARNING: Uninstanciated switch ~w found in the explanation for ~w goal.}",[C,Ith]),!,fail
    ; true ),!,
	non_contradictory(I,(G_id,T,V),Cj).

non_contradictory(I,[]) :-
	nth(I,Ith),!,
	message("{PRISM WARNING: No switches is in the explanation for ~w goal.}",
	        [Ith]),!,fail.
	
non_contradictory(I,(G_id,T,V),[C|Conj]) :-
	( ( C=msw(G_id1,T1,V1) ; C=bsw(G_id1,T1,V1) ),!
    ; nth(I,Ith),!,
	  message("{PRISM INTERNAL ERROR: ~w found in the explanation for ~w goal.}",[C,Ith]),!,fail ),!,
	( var_check((G_id,T,V),yes),!,
	  nth(I,Ith),!,
	  message("{PRISM WARNING: Uninstanciated switch ~w found in the explanation for ~w goal -- ignored.}",[C,Ith]),!,fail
    ; true ),!,
	( G_id==G_id1, T==T1, V\==V1, !,
	  nth(I,Ith),!,
	  message("{PRISM WARNING: Conflicting switches ~w and ~w were found in the explanation for ~w goal -- ignored.",[msw(G_id,T,V),C,Ith]),!,fail
	; true ),!,
	non_contradictory(I,(G_id1,T1,V1),Conj).
non_contradictory(_,_,[]).

% no-warning version of non_contradictory/2

non_contradictory_with_no_warning([C|Cj]):-
	( ( C=msw(G_id,T,V) ; C=bsw(G_id,T,V) ),!
	; message("{PRISM INTERNAL ERROR: ~w found in the explanation.}",[C]),
	  !,fail ),!,
	( var_check((G_id,T,V),yes),!,
	  message("{PRISM WARNING: Uninstanciated switch ~w found -- ignored.}",
	          [C]),!,
	  fail
    ; true ),!,
	non_contradictory_with_no_warning((G_id,T,V),Cj).

non_contradictory_with_no_warning([]) :- fail.

non_contradictory_with_no_warning((G_id,T,V),[C|Conj]) :-
	( ( C=msw(G_id1,T1,V1) ; C=bsw(G_id1,T1,V1) ),!
    ; message("{PRISM INTERNAL ERROR: ~w found in the explanation.}",[C]),!,
	  fail ),!,
	( var_check((G_id,T,V),yes),!,
  	  message("{PRISM WARNING: Uninstanciated switch ~w found -- ignored.}",
	          [C]),!,
	  fail
    ; true ),!,
	( G_id==G_id1, T==T1, V\==V1, !,fail
	; true ),!,
	non_contradictory_with_no_warning((G_id1,T1,V1),Conj).
non_contradictory_with_no_warning(_,[]).

% get_goals(GPatt,GFreq): GPatt appears GFreq times in teacher data.
%
%   added by kame on Feb/27/1998

get_goals(GPatt,GFreq) :-
	clause('*Target*'(F,ArgN),true),!,
	clause('*Ans*'(Args,_,GFreq),true),
	build_target_atom(F,ArgN,Args,GPatt).

% show_goals/0-1 added by kame on Feb/27/1998

show_goals :-
	( clause('*Target*'(F,ArgN),true),!,
	  setof((GFreq,GPatt),
	      (F,ArgN,Args,AC,C)^('*Ans*'(Args,AC,C),
		                      build_target_atom(F,ArgN,Args,GPatt),
							  GFreq is 0-C),
	      GFPs)
	; message('{PRISM ERROR: Maybe learning is not done yet.}'),!,fail ),!,
	count_total_goals(GFPs,Total),!,
    show_goals(Total,GFPs),!,
	format("TOTAL ~w~n",[Total]).

show_goals(Total,[(GF,GPatt)|GFPs]) :-
	GFreq is 0-GF,!,
	GFreq1 is GFreq/Total,!,
	format("GOAL ~w -- ~w (~3f)~n",[GPatt,GFreq,GFreq1]),!,
	show_goals(Total,GFPs).
show_goals(_,[]).

count_total_goals(GFPs,N) :-
	count_total_goals(GFPs,0,N1),N is 0-N1.

count_total_goals([(GF,_)|GFPs],N0,N) :-
	N1 is N0+GF,!,
	count_total_goals(GFPs,N1,N).
count_total_goals([],N,N).


%  build_target_atom/3-4:
%  <ex.>
%    Suppose that a predicate foo/3 is the target predicate of model
%    `m1'. Then we have "build_target_atom(m1,[x,y,z],foo(x,y,z))."
%    And we have "build_target_atom(foo,3,[x,y,z],foo(x,y,z))."
%
%    created by kame on Dec/13/1997.
%    build_target_atom/4 added by kame on Feb/27/1998

build_target_atom(Model,Args,TargetAtom) :-
	'*Sentence*'(_,target,[Model,F,ArgN]),!,
	( ArgN==1,!,TargetAtom =.. [F,Args] ; TargetAtom =.. [F|Args] ),!.

build_target_atom(F,ArgN,Args,TargetAtom) :-
	( ArgN==1,!,TargetAtom =.. [F,Args] ; TargetAtom =.. [F|Args] ),!.

%  assert_target_pred(M): assert the target predicate of model M
%
%  added by kame on Feb/27/1998.

assert_target_pred(Model) :-
	( '*Sentence*'(_,target,[Model,F,ArgN]),!,
      assertz('*Target*'(F,ArgN))
    ; message("{PRISM ERROR: No such model ~w.}",[Model]),!,fail ),!.
