/*
Copyright (C) 1997 $B9b66(B $B?-(B (TAKAHASHI Shin)
*/

#include <stdio.h>
#include <math.h>
#include <sys/file.h>
#include "eq_solver.h"
/* prototypes */
#include "node.h"
#include "solve.h"

Node	*eq_tab[MAXEQ];		/* equations 		*/
int	eq_idx = 0;
Term	*r_tab[MAXRC];		/* rigid constraints 	*/
int	r_idx = 0;
Term	*p_tab[MAXPC];		/* pliable constraints 	*/
int	p_idx = 0;
/*	eq_idx = r_idx + p_idx	*/

static	Term	*termroot;
/* 2 vars are determined in ObjView.m. These are initial value */
double	center_x=300.0;
double	center_y=300.0;

Pict	*pictroot,*curpict;

int	errflag;

void eqs()
{
	int	i;
	void	distribute(Node *);
	void	normalize(Node *, int);
	void	seteq(Node *);
	void	average(void);
	void	substitute(void);
	void	lsm(void);
	void	backward(void);
	void	answer(void);
	void	eqinit(void);
	void	prtree(Node *);
	void	prlist(Term *);

	for (i=0; i<eq_idx; i++) {
		/*prtree(eq_tab[i]);*/
		distribute(eq_tab[i]);
		/*prtree(eq_tab[i]);*/
		termroot = (Term*)NULL;
		normalize(eq_tab[i],1);
		seteq(eq_tab[i]);			
		/*prlist(termroot);*/
	}

	fprintf(stderr,"%d variables & %d(%d+%d) constraints\n",
					var_count,eq_idx,r_idx,p_idx);
	
	if ((i=r_idx + p_idx) < var_count)
		if (i >= var_count - 2)
			average();	/* two more constraints added */
		else
			error(7);

	substitute();	
	if (p_idx)
		lsm();
	backward();	
	answer();
	eqinit();
}

void eqinit()
{
	eq_idx = 0;
	r_idx = 0;
	p_idx = 0;
	var_count = 0;
}

void output()
{
	int	i, typ;
	float	val;
	Pict	*p;
	int	count;

	count = 0;
	fprintf(stdout, "debug output\n");
	for (i=0; i<MAXVARS; i++) {
		if (var_tab[i].busy == EMPTY) 
			continue;
		val = var_tab[i].value;
		fprintf(stdout,"s/ %s/ %f/g\n",var_tab[i].name, val);
		
	}
}

/*----------------------------------------------*/
/*	DISTRIBUTIVE TRANSFORMATION		*/
/*	    *			     +		*/
/*	   / \                     /   \	*/
/*	  +   C        -->        *     *	*/
/*	 / \			 / \   / \	*/
/*	A   B 			A   C B   C	*/
/*						*/
/*	    *			     +		*/
/*	   / \                     /   \	*/
/*	  A   +        -->        *     *	*/
/*	     / \		 / \   / \	*/
/*	    B   C 		A   B A   C	*/
/*----------------------------------------------*/
void distribute(p)
Node	*p;
{
	void	check_dis(Node *);

	if (p->type == LEAF)
		return;
	if (p->u.n.left)
		distribute(p->u.n.left);
	if (p->u.n.right)
		distribute(p->u.n.right);
	check_dis(p);
}

void check_dis(p)
Node	*p;
{
	int	top,leaf;
	Node	*trans;

	if ((top=p->u.n.op) != MUL && top != DIV)
			return;
	if ((leaf=p->u.n.left->u.n.op) == ADD || leaf == SUB) {
		p->u.n.left->u.n.op = top;
		trans = p->u.n.left->u.n.right;
		p->u.n.left->u.n.right = p->u.n.right;
		p->u.n.right = mknode(top, trans, cpnode(p->u.n.right));
		p->u.n.op = leaf;	
		check_dis(p->u.n.left);
		check_dis(p->u.n.right);
	}
	else if ((leaf=p->u.n.right->u.n.op) == ADD || leaf == SUB) {
		p->u.n.right->u.n.op = top;
		trans = p->u.n.right->u.n.left;
		p->u.n.right->u.n.left = p->u.n.left;
		p->u.n.left = mknode(top, cpnode(p->u.n.left), trans);
		p->u.n.op = leaf;	
		check_dis(p->u.n.left);
		check_dis(p->u.n.right);
	}
}

void normalize(p, sign)
Node	*p;
int	sign;
{
	void	calc_item(Node *, int);

	if (p == (Node*)NULL) return;
	
	if (p->type == LEAF)
		calc_item(p, sign);
	else  	/* NODE */
	    switch (p->u.n.op) {
		case RIGID:
		case PLIABLE:
		case SUB:
			normalize(p->u.n.left, sign);
			normalize(p->u.n.right, -sign);
			break;
		case ADD:
			normalize(p->u.n.left, sign);
			normalize(p->u.n.right, sign);
			break;
		default:
			calc_item(p, sign);
			break;
	    }
}

#define CONSTANT	MAXVARS+1
static	int	cur_var;
static	double	cur_cof;

#define PI	(double)3.141592653589793

double	coef(p)
Node	*p;
{	
	if (p->type == LEAF) {
		if (p->u.l.kind == VAR)
			if (cur_var == CONSTANT) {
				cur_var = p->u.l.val; 
				return((double)1.0);
			}
			else{
				error(6); /* not linear */
				return -1;
			}
		else /* CONST */
			return((double)p->u.l.val);
	}
	else  /* NODE */
	    switch (p->u.n.op) {
		case MUL:	return(coef(p->u.n.left)*coef(p->u.n.right));
		case DIV:	return(coef(p->u.n.left)/coef(p->u.n.right));
		case SIN:	return(sin(coef(p->u.n.left)*
					atan((double)1.0)/45.0));
		case COS:	return(cos(coef(p->u.n.left)*
					atan((double)1.0)/45.0));
		case TAN:	return(tan(coef(p->u.n.left)*
					atan((double)1.0)/45.0));
	    }
	return (0.0);
}

void calc_item(p, sign)
Node	*p;
int	sign;
{
	void	termsearch(void);

	cur_var = CONSTANT;
	cur_cof = coef(p) * sign;
	if (cur_var != CONSTANT && fabs(cur_cof) < EPS)
		return;
	termsearch();
}

void termsearch()
{
	Term	**pt;
	int	var;

	for (pt = &termroot; *pt; pt = &((*pt)->next)) 
		if ((var=(*pt)->var) < cur_var)
			continue;
		else if (var == cur_var) {
			(*pt)->cof += cur_cof;
			return;
		}
		else
			break;
	(*pt) = mkterm(cur_var, cur_cof, *pt);
}

void substitute()
{
	int	i,j,var;
	int	nochange;
	int	shifts;
	int	subst(Term *, Term **);
	void	shifteq(int);

/*	prlists();	*/	
	shifts = 0;
	for (i=0; i<r_idx; ) {
		if (! r_tab[i])
			{ i++; continue; }
		nochange = 0;
		for (j=i+1; j<r_idx; j++)
			if (r_tab[j])
				nochange |= subst(r_tab[i], &r_tab[j]);
		for (j=0; j<p_idx; j++)
			if (r_tab[j])
				nochange |= subst(r_tab[i], &p_tab[j]);
		if (nochange == 1) {	/* at least 1 time called */
			shifteq(i);
			if (++shifts == r_idx-i)
				break;
			else
				continue;
		}
		shifts = 0;
		i++;
	}

	for (i=0; i<r_idx; i++) 
		if (r_tab[i] && (var=r_tab[i]->var) != CONSTANT)
			var_tab[var].state = DEPEND;
}

void prlists()
{
	int	i;
	void	prlist(Term *);

	printf("----------------------------------------\n");
	for (i=0; i<r_idx; i++) 
		prlist(r_tab[i]);
}

void shifteq(i)
int	i;
{
	Term	*sv;

	sv = r_tab[i];
	for (; i<r_idx-1; i++)
		r_tab[i] = r_tab[i+1];
	r_tab[i] = sv;
}

int	subst(s, pt)
Term	*s, **pt;
{ 
	double	rate;

	while ((*pt)->var < s->var)
		if ((*pt)->next)
			pt = &((*pt)->next);
		else
			return(1);
	if ((*pt)->var > s->var)
		return(1);	/* has no that variable */
	rate = -((*pt)->cof/s->cof);
	*pt = (*pt)->next;
	s = s->next;
	for ( ; s; )
		if (*pt && (*pt)->var < s->var)
			pt = &((*pt)->next);
		else if (*pt && (*pt)->var == s->var) {
			(*pt)->cof += (s->cof * rate);
			if (fabs((*pt)->cof) < EPS) 	/* reduction */
				*pt = (*pt)->next;
			else
				pt = &((*pt)->next);
			s = s->next;
		}	
		else {
			(*pt) = mkterm(s->var, s->cof * rate, *pt);
			s = s->next;
			pt = &((*pt)->next);
		}
	return(2);
}

double	calc_val(t)
Term	*t;
{
	double	d=0.0;

	while (t) {
		if (t->var == CONSTANT)
			d += t->cof;
		else if (var_tab[t->var].state != FIXED){
			error(8);
			return -1.0; /* dummy */
		}
		else 
			d += (var_tab[t->var].value * t->cof);
		t = t->next;
	}
	return(d);
}

void backward()
{
	int	i,var;
	Term	*t;

	for (i=r_idx-1; i >= 0; i--) {
		if (!(t=r_tab[i]))
			continue;		
		if ((var=t->var) == CONSTANT)
			if (fabs(t->cof) > EPS){
				error(8);
				return;
			}
			else	/* redundant constraints */
				continue;
		if (var_tab[var].state == FIXED){
			error(8);	
			return;
		}
		var_tab[var].value = (-1)*calc_val(t->next)/t->cof;	
		var_tab[var].state = FIXED;
	}
}

void lsm()
{
	int	i,j,var;
	int	lsms=0;
	Term	*t;

	for (i=0; i<MAXVARS; i++)
		if (var_tab[i].busy == ACTIVE && var_tab[i].state == FREE) 
			var_tab[i].element = lsms++;
	size = lsms;	/* the number of free variables */
	if (size > MAX_N){
		error(9);
		return;
	}
	for (i=0; i<p_idx; i++) {
		t = p_tab[i];
		while (t) {
			if ((var=t->var) == CONSTANT) {
				j = size;
				matrix[i][j] = -(t->cof);
			}
			else {
				j = var_tab[var].element;
				matrix[i][j] = t->cof;
			}
			t = t->next;
		}
	}
	solve();	/* LSM */
	for (i=0; i<MAXVARS; i++)
		if (var_tab[i].busy == ACTIVE && var_tab[i].state == FREE) {
			var_tab[i].value = ans[var_tab[i].element];
			var_tab[i].state = FIXED;
		}
}

#define max(x, y) (x >= y ? x : y)
#define min(x, y) (x <= y ? x : y)
void answer()
{
	int	i, typ;
	double	lx, rx, ty, by;

	/*	calculating bounding box 	*/
	lx = rx = center_x;
	ty = by = center_y;
	for (i=0; i<MAXVARS; i++)
		if (var_tab[i].busy == ACTIVE) {
			if (var_tab[i].state == FREE){
				error(8);
				return;
			}
		/*
			printf("s/ %s/ %d/g\n",var_tab[i].name,
				(int)(var_tab[i].value + 0.5));
		*/
			if ((typ=var_tab[i].type) == LX) 
				lx = min(lx, var_tab[i].value);	
			else if (typ == RX)
				rx = max(rx, var_tab[i].value);	
			else if (typ == BY) 
				by = min(by, var_tab[i].value);	
			else if (typ == TY)
				ty = max(ty, var_tab[i].value);	
		}
	for (i=0; i<MAXVARS; i++)
		if (var_tab[i].busy == ACTIVE) {
			typ = var_tab[i].type;
			if (typ == LX || typ == RX || typ == CX)
				var_tab[i].value -= lx;	
			else if (typ == BY || typ == TY || typ == CY)
				var_tab[i].value -= by;
			var_tab[i].busy = DEAD;
		}
	curpict->xsiz = (int)(rx - lx + 0.5);
	curpict->ysiz = (int)(ty - by + 0.5);
	if (curpict == pictroot) {
		curpict->xoff = (int)(lx + 0.5);
		curpict->yoff = (int)(by + 0.5);
	}
}

void push(p)
Node	*p;
{
	if (eq_idx >= MAXEQ){
		error(3);
		return;
	}
	eq_tab[eq_idx++] = p;
}

void seteq(p)
Node	*p;
{
	if (p->u.n.op == RIGID) {
		if (r_idx >= MAXRC){
			error(4);
			return;
		}
		r_tab[r_idx++] = termroot;
	}
	else { /* PLIABLE */
		if (p_idx >= MAXPC){
			error(5);
			return;
		}
		p_tab[p_idx++] = termroot;
	}
}

/*----------------------------------------------------------------------*/
/*	The following two constraints are added;			*/
/*	o1.cx + o2.cx + ... + on.cx = n * center_x,			*/
/*	o1.cy + o2.cy + ... + on.cy = n * center_y,			*/
/*	where center_x and center_y are pre-defined constants.		*/
/*----------------------------------------------------------------------*/
void average()
{
	Term	*xroot, *yroot;
	int	i, typ, count;

	xroot = yroot = NULL;
	for (i=MAXVARS-1; i>=0; i--) 
		if (var_tab[i].busy == ACTIVE) {
			if ((typ=var_tab[i].type) == CX)
				xroot = mkterm(i,(double)1.0,xroot);
			else if (typ == CY)
				yroot = mkterm(i,(double)1.0,yroot);
			else
				continue;
		}
	if (r_idx >= MAXRC-1){
		error(4);
		return;
	}
	r_tab[r_idx++] = xroot;
	r_tab[r_idx++] = yroot;
	for (count=1; (xroot->next)!=NULL; count++)
			xroot = xroot->next;;
	xroot->next = mkterm(CONSTANT,-count*center_x,NULL);
	for (count=1; (yroot->next)!=NULL; count++)
			yroot = yroot->next;;
	yroot->next = mkterm(CONSTANT,-count*center_y,NULL);
}

/*----------------------------------------------------------------------*/
/*	for debugging							*/
/*----------------------------------------------------------------------*/
void prtree(p)
Node	*p;
{
	void	prtreen(int, Node *);

	prtreen(0,p);
}

void prtreen(n,p)
int	n;
Node	*p;
{
	int	i;

	if (p->type == LEAF) {
		if (p->u.l.kind == CONST)
			printf("%d\n", p->u.l.val);
		else
			printf("%s\n", var_tab[p->u.l.val].name);
	}
	else {
		switch (p->u.n.op) {
			case RIGID: 	printf("=    "); break;
			case PLIABLE: 	printf("#    "); break;
			case ADD: 	printf("+    "); break;
			case SUB: 	printf("-    "); break;
			case MUL: 	printf("*    "); break;
			case DIV: 	printf("/    "); break;
			case SIN: 	printf("sin  "); break;
			case COS: 	printf("cos  "); break;
			case TAN: 	printf("tan  "); break;
		}
		if (p->u.n.left)
			prtreen(n+1, p->u.n.left);
		else
			printf("\n");
		if (p->u.n.right) {
			for (i=0; i<=n; i++)
				printf("     ");
			prtreen(n+1, p->u.n.right);
		}
	}
}

void prlist(t)
Term	*t;
{
	while (t) {
		printf("%f %s",t->cof, var_tab[t->var].name);
		if (t=t->next)
			printf(" + ");
	}
	printf(" = 0\n");
}

void eqs_varinit()
{
	int	i, j;

	void free_node(Node *node);
	void free_term(Term *term);

	var_count = 0;
	for(i = 0; i<MAXVARS; i++){
		var_tab[i].busy = 0;
		var_tab[i].name[0] = (char)0;
		var_tab[i].state = 0;
		var_tab[i].element = 0;
		var_tab[i].type = 0;
		var_tab[i].value = 0.0;
		var_tab[i].pict = NULL;
	}
	for(i = 0; i<eq_idx; i++)
		free_node(eq_tab[i]);
	eq_idx = 0;
	for(i = 0; i<r_idx; i++)
		free_term(r_tab[i]);
	r_idx = 0;
	for(i = 0; i<p_idx; i++)
		free_term(p_tab[i]);
	p_idx = 0;

	/*for(i=0;i<MAX_N;i++){
	   for(j=0;j<MAX_N+1;j++){
	      matrix[i][j] = (double)0.0;
	      transm[i][j] = (double)0.0;
	   }
	   ans[i] = (double)0.0;
	}*/
	size = 0;

	free(pictroot);
}

void free_node(node)
Node *node;
{
  if(node == NULL) return;
  if(node->type == LEAF){
     free(node);
     return;
  }else{
     free_node(node->u.n.left);
     free_node(node->u.n.right);
     free(node);
     return;
  }
}

void free_term(term)
Term *term;
{
  if(term == NULL){
	return;
  }else{
	free_term(term->next);
	free(term);
	return;
  }
}
