%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%                                                             %%%
%%%         PRISM: A Language for Statistical Modeling          %%%
%%%                                                             %%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%                                                             %%%
%%%           Created by Sato Lab., TIT on 28 Oct 1996          %%%
%%%     Last modified for IFS (ICOT Free Software) Project      %%%
%%%                    TIT on 25 Apr 1997                       %%%
%%%                                                             %%%
%%% Copyright (C) 1997 Sato Taisuke and KAMEYA Yoshitaka,       %%% 
%%%   Dept. of Computer Science, Tokyo Institute of Technology  %%%
%%%                                                             %%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Translator Part:
%%%    translates User's PRISM program to specialized program
%%%    for learning and execution phase.
%%% the task:
%%%    1. to parse PRISM program
%%%    2. to find the probabilistic predicates
%%%    3. to build expl_routines (used for learning and execution)
%%%    4. to build sample_routines (for execution)
%%%    5. to build learn_routines (for learning)

%%% pconsult/1 and pcompile/1:
%%%   PRISM-version of consult/1 and compile/1, respectively.

pconsult(Files) :-
	format("{translating ~w...}",[Files]),nl,
	avoid_abolish_warning,
	clean,!,
	parse(Files),!,
	make_routines,!,
	query,!,
	format("{~w translated to learning & execution programs.}",[Files]),nl,
	message('{Check translated program by show_routines/0}').

pcompile(Files) :-
	( pconsult(Files) ; message('{Translation failed.}'),fail ),
	( Files = [File|_] ; File = Files ),
	name(File,FILE),name('.psm',PSM),append(FILE,PSM,PRISMFILE),
	name(PRISMfile,PRISMFILE),
	print_routines(PRISMfile),
	format('{wrote BS routines into ~w}',[PRISMfile]),nl,
	clean,!,
	compile(PRISMfile).

make_routines :- make_learn,!,make_expl,!,make_sample.

%%%%%%%%%%%%%%%%%%%%%%%%%% temporal aliases %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
bscompile(Files) :- pcompile(Files).
bsconsult(Files) :- pconsult(Files).

clean :- clean_parse,clean_routines.
show :- show_parse,show_routines.
print_routines(File) :-
	tell(File),
	show_routines,show_normal_clauses,
	listing('*Prob_Pred*'),listing('*Sentence*'),
	told.	
message(M) :- write(M),nl.

avoid_abolish_warning :-
	assertz('*Expl*'),assertz('*Sample*').

%%%  Parsing User's PRISM-program
%%%   - parsing AND(,)/OR(;) structure
%%%   - find "calls" relationship
%%%     (ex. for clause "A:-B,C" say "A calls B" and "A calls C")

%%% show the result of parsing
show_parse :-
	listing('*Sentence*'),listing('*Num_of_sentences*'),
	listing('*Calls*'),	listing('*Prob_Pred*').

%%% clean up
clean_parse  :- clean_parse1,clean_parse2.
clean_parse1 :-
	retractall('*Sentence*'(_,_,_)),retractall('*Prob_Pred*'(_,_)).
clean_parse2 :-
	retractall('*Num_of_sentences*'(_,_)),retractall('*Calls*'(_,_)).

%%% query/0: execute queries
query :- 
	( setof([N,Goal],'*Sentence*'(N,query,Goal),NGoals0),
	  sort(NGoals0,NGoals),
	  query(NGoals)
    ; true ).   % do nothing if there are no queries.
query([[_,Goal]|NGoals]) :- call(Goal),query(NGoals).
query([]).

%%% parse/1: main parsing routine
%   allows file list
% | ?- parse([blood1,blood2,...]).
parse(Files) :-
	( atom(Files),!,parse_file(Files,1,_,_)
    ; parse_list(Files,1)),!,
	find_prob_pred.

parse_list([In|Ins],N) :-	
	parse_file(In,N,N1,Error),
	( Error=yes,! ; parse_list(Ins,N1)). % stop with erroneous signal
parse_list([],_):-!.
	
% parse for a file
parse_file(Infile,N0,N,Error) :-
	see(Infile),parse_sentence(Infile,N0,N,Error),seen.

% parse for a sentence
parse_sentence(File,N0,N,E) :-
	read(Sentence),!,
	( Sentence = end_of_file,!,                % end with success
	  N1 is N0-1,
	  assertz(('*Num_of_sentences*'(File,N1):-!)),
	  N=N0, E=no
    ; ( Sentence = target(Target,ArgN),!,      % target decl.
	    assertz('*Sentence*'(N0,target,[Target,ArgN])),
		E0=no
	  ; Sentence = data(Datafile),!,           % teacher data decl.
	    assertz('*Sentence*'(N0,data,Datafile)),
		E0=no
	  ; Sentence = (Head :- Body),!,           % normal Prolog
	    parse_body(N0,Head,Body),              % (difinite clause)
		E0=no
	  ; Sentence = (:- Goal),!,
	    assertz('*Sentence*'(N0,query,Goal)),
		E0=no
	  ; parse_body(N0,Sentence,true),          % normal Prolog
		E0=no                                  % (unit clause)
      ; write('Erroneous input: '),            % erroneous input
	    write(File),write(': sentence #'),
		write(N0), nl, clean, E0=yes),
	  ( E0=yes,!,E=yes,N=N0 ; N1 is N0+1, parse_sentence(File,N1,N,E) )).

%%%% Parse_body:
%%%% - parse for a normal clause (body)
%%%% - assert the result
%%%% - find 'calls' relationship

parse_body(N,Head,Body) :- parse_body0(N,Head,Body).

%% accumulate body-predicates in Preds
parse_body0(N,Head,Body) :-
	parse_body1(Body,BL,Preds0,[]),
	assertz('*Sentence*'(N,normal,[Head,BL])),
	functor(Head,F,ArgN),
	sort(Preds0,Preds),
	assert_calls(F,ArgN,Preds).

%% assert 'calls' relationship
assert_calls(F,N,[P|Ps]) :-
	( clause('*Calls*'([F,N],P),true),!   % already exists
    ; assertz('*Calls*'([F,N],P))),
	assert_calls(F,N,Ps).
assert_calls(_,_,[]):-!.


%%% parse_body1/4: convert conjuction/disjuction into *AND*/*OR* term
% Preds0 and Preds1: differencial list 
parse_body1(Body,Bterm,Preds0,Preds1) :-
	( Body = (Ch,Ct),!,                   %%%% if Body is conjuction
	  parse_body1(Ch,Chterm,Preds0,P),
	  parse_body1(Ct,Ctterm,P,Preds1),
	  Ctterm =.. [F|Args],
	  ( F = '*AND*',!,                    % post-processing:
	    Bterm =.. ['*AND*',Chterm|Args]   %  in the case
      ; Bterm = '*AND*'(Chterm,Ctterm))   %    *AND* isn't only binary
    ; Body = (Dh;Dt),!,                   %%%% if Body is disjunction
	  parse_body1(Dh,Dhterm,Preds0,P),
	  parse_body1(Dt,Dtterm,P,Preds1),
	  Dtterm =.. [F|Args],
	  ( F = '*OR*',!,                     % post-processing:
	    Bterm =.. ['*OR*',Dhterm|Args]    % in the case
	  ; Bterm = '*OR*'(Dhterm,Dtterm))    %   *OR* isn't only binary
    ; Bterm = Body,
	  functor(Body,F,ArgN),
	  Preds0 = [[F,ArgN]|Preds1] ).

%%% find_prob_pred/0: find probabilistic predicates
find_prob_pred :-
	assertz('*Prob_Pred*'(bsw,3)),find_prob_pred0.

find_prob_pred0 :-
	( setof([PF,PargN],'*Prob_Pred*'(PF,PargN),PPs)
    ; PPs = [] ),
	find_prob_pred1(PPs,PPs1), % use diff-list
	append(PPs,PPs1,PPs2),
	sort(PPs2,PPs3),
	( PPs = PPs3,!
    ; add_prob_pred(PPs3),find_prob_pred0 ).

find_prob_pred1([[PF,PargN]|PPs],PPs0) :-
	( setof([QF,QargN],'*Calls*'([QF,QargN],[PF,PargN]),QQs)
    ; QQs = [] ),
	append(QQs,PPs1,PPs0),
	find_prob_pred1(PPs,PPs1).
find_prob_pred1([],[]) :-!.

add_prob_pred([[PF,PargN]|PPs]) :-
	( clause('*Prob_Pred*'(PF,PargN),true),!  % already exists
    ; assertz('*Prob_Pred*'(PF,PargN))),
	add_prob_pred(PPs).
add_prob_pred([]):-!.

append([],A,A).
append([A|X],Y,[A|Z]):-!,append(X,Y,Z).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%   make_routines: make 3 routines
%%%%   - learn_routine (learn)
%%%%   - expl_routine  (*Expl*)
%%%%   - sample_routine(*Sample*)

%%% print the result of the translation
show_routines :-
	listing('learn'),listing('*Expl*'),listing('*Sample*').

show_normal_clauses :-
	findall(F/ArgN,
	        ('*Sentence*'(_,normal,[Head,_]),
			 functor(Head,F,ArgN),
			 \+('*Prob_Pred*'(F,ArgN))),
			FAs0),
	sort(FAs0,FAs),listing(FAs).

%%% clean up
clean_routines :-
	retractall(learn),
	retractall(learn(_)),
	abolish('*Expl*'),abolish('*Sample*').

clean_normal_clauses :-
	findall(F/ArgN,
	        ('*Sentence*'(_,normal,[Head,_]),
			 functor(Head,F,ArgN),
			 \+('*Prob_Pred*'(F,ArgN))),
			FAs0),
	sort(FAs0,FAs),abolish(FAs).

%%%% make_learn/0: make learn_routine(top-level command on learning phase)
make_learn :-
	( '*Sentence*'(_,target,[Target,ArgN]),!,
	  new_arg(ArgN,Arg),
	  append(Arg,[Ans,[]],Arg0),
	  Expl =.. ['*Expl*',Target|Arg0],
	  ( Arg = [Arg1],! ; Arg = Arg1 ),  % if arity = 1
	  D = (define_goal((goal(Arg1,Ans) :- Expl))),
	  E = em_bsw(Goals),
	  ( '*Sentence*'(_,data,user),!,
	    message('{learn/1 is available for EM learning.}'),
		G = mk_goals(user,Tdata,Goals),
	    assertz((learn(Tdata) :- (D,G,!,E)))
      ; '*Sentence*'(_,data,File),!,
	    message('{learn/0 is available for EM learning.}'),
	    G = mk_goals(File,File,Goals),
	    M = mk_goals_message(File),
	    assertz((learn :- (D,G,!,M,E)))
	  ; message('{PRISM ERROR: missing deta declaration.}'),
	    fail )
	; message('{Neither learn/0 nor learn/1 is available.}')).
	                   
mk_goals_message(File) :- format("{Extract goals from ~w}",[File]),nl.

% generate a list with length ArgN
new_arg(ArgN,Arg) :- new_arg0(0,ArgN,Arg).

new_arg0(N,N,[]) :- !.
new_arg0(M,N,Arg) :- Arg=[_|Arg1],M1 is M+1,new_arg0(M1,N,Arg1).

%%% mk_goals/3: get teacher data and make goals
%%%  if data file File="user", mk_goals/3 converts teacher data to goals.
%%%  otherwise, it extracts teacher data from File and convert them to
%%%  goals.
mk_goals(user,Tdata,Goals) :- !,mk_goals1(0,Tdata,Goals).
mk_goals(_,File,Goals) :- see(File),mk_goals0(0,File,Goals),seen.

mk_goals0(N,File,Goals) :-
	read(G),
	( G = end_of_file,!,Goals=[]
    ; '*Sentence*'(_,target,[F,ArgN]),
	  ( functor(G,F,ArgN),!,
	    G =.. [F|Args],
		( Args = [Args1] ; Args = Args1 ),
		Goals = [goal(Args1,_)|Goals0]
	  ; format("{PRISM WARNING: teacher data ~w incomplete in line ~w, skipped.}",[File,N]),
	    nl,
		Goals = Goals0 ),
	  N1 is N+1,
	  mk_goals0(N1,File,Goals0) ).

mk_goals1(N,[T|Tdata],Goals) :-
	'*Sentence*'(_,target,[F,ArgN]),!,
	( functor(T,F,ArgN),!,
	  T =.. [F|Args],
	  ( Args = [Args1] ; Args = Args1 ),
	  Goals = [goal(Args1,_)|Goals0]
    ; format("{PRISM WARNING: ~wth teacher data incomplete, skipped.}",[N]),
	  nl,
	  Goals = Goals0 ),
	N1 is N+1,
	mk_goals1(N1,Tdata,Goals0).
mk_goals1(_,[],[]).

%%%% two Dice routines
%%%%  - not included in translator
%% for multinominal distribution
dice_multi(As,Ps,X) :-
	random_float(1.0,R),!,
	dice_multi0(R,0.0,As,Ps,X).

dice_multi0(R,Tmp,[A|As],[P|Ps],X) :-
	Tmp1 is Tmp+P,
	( R < Tmp1,!,X=A
    ; As = [], !,X=A
	; dice_multi0(R,Tmp1,As,Ps,X)).

%% for uniform distribution
dice_uniform(As,X) :-
	length(As,N),
	P is 1.0/N,
	random_float(1.0,R),!,
	dice_uniform0(R,0.0,As,P,X).

dice_uniform0(R,Tmp,[A|As],P,X) :-
	Tmp1 is Tmp+P,
	( R < Tmp1,!,X=A
    ; As = [], !,X=A
    ; dice_uniform0(R,Tmp1,As,P,X)).

%%%% make_expl/0:
%%%%   make expl_routine (the routine to get explanations for a goal)
make_expl :-
	%% collect normal clauses
	findall([N,Head,Body],
	        '*Sentence*'(N,normal,[Head,Body]),
			Clauses0),!,
	sort(Clauses0,Clauses),
	make_expl0(Clauses).

make_expl0([[_,Head,Body]|Clauses]):-
	functor(Head,F,ArgN),
	( '*Prob_Pred*'(F,ArgN),!,
	  make_prob_expl(Head,Body)      
	; make_nonprob_expl(Head,Body)),!,
	make_expl0(Clauses).
make_expl0([]):-!.

%%% for Probabilistic clause
%%% --- add diff-list for each prob. predicate
%%%     (it becomes *Expl* pred.)

% Body is the compound-term of *AND* and *OR*
make_prob_expl(Head,Body) :-
	Head =.. [F|Arg0],
	append(Arg0,[X,Y],Arg),
	Head0 =.. ['*Expl*',F|Arg],
	make_dnf(Body,DNF),!,
	get_dnf_list(DNF,DNFL),!,
	make_prob_expl_body(X,Y,DNFL,Body0),
	( clause(Head0,Body0),!,true ; assertz((Head0 :- Body0)) ).
	
make_prob_expl_body(X,Y,[Conj|DNFL],Body) :-
	make_prob_expl_body0(X,Y,Conj,C0),
	( DNFL = [],!,Body=C0
    ; Body = (C0;B0), make_prob_expl_body(X,Y,DNFL,B0)).

% Conj is the form (_,_,...)
make_prob_expl_body0(X,Y,[C|Cs],Conj) :-
	C =.. [F|Arg0],
	functor(C,F,ArgN),
	( '*Prob_Pred*'(F,ArgN),!,
	  ( Cs=[],!,
	    ( (F,ArgN)=(bsw,3),!,
		  Conj = (X=[C|Y])
	    ; append(Arg0,[X,Y],Arg),
		  Conj =.. ['*Expl*',F|Arg] )
      ; ( (F,ArgN)=(bsw,3),!,
	      C0 = (X=[C|Z])
	    ; append(Arg0,[X,Z],Arg),
		  C0 =.. ['*Expl*',F|Arg] ),
		Conj = (C0,Conj0),
		make_prob_expl_body0(Z,Y,Cs,Conj0))
	; ( Cs=[],!,
	    Conj = C, X = Y     %%% [NOTE!] close the stream
	  ; Conj = (C,Conj0),
	    make_prob_expl_body0(X,Y,Cs,Conj0))).

%%% for Non-Probabilistic clause 
%%% --- only restration of body

make_nonprob_expl(Head,Body) :-
	restore_body(Body,B),
	( clause(Head,B),!,true ; assertz((Head :- B)) ).

% restration of body
% <ex.>
% *AND*(*OR*(a,*AND*(b,*OR*(d,x),f),g),*OR*(h,k))
%   -> ((a;b,(d;x),f;g),(h;k))

restore_body(Body,B) :-
	( Body =.. ['*AND*'|Conj],!,
	  restore_body_conj(Conj,B)
    ; Body =.. ['*OR*'|Disj],!,
	  restore_body_disj(Disj,B)
    ; B = Body ).

restore_body_conj([C|Cs],B) :-
	restore_body(C,C0),
	( Cs = [],!,B = C0
    ; B = (C0,B0),restore_body_conj(Cs,B0)).

restore_body_disj([D|Ds],B) :-
	restore_body(D,D0),
	( Ds = [],!,B = D0
    ; B = (D0;B0),restore_body_disj(Ds,B0)).

%%%% Conv:  Body --> DNF (both are the compound-term of *AND* and *OR*)
%%% <ex.>
%%% Body = ((a;b,(d;x),f;g),(h;k))
%%%      = *AND*(*OR*(a,*AND*(b,*OR*(d,x),f),g),*OR*(h,k))
%%% DNF = (a,h;a,k;b,d,f,h;b,d,f,k;b,x,f,h;b,x,f,k;g,h;g,k)
%%%     = *OR*(*AND*(a,h),*AND*(a,k),*AND*(b,d,f,h),...)

make_dnf(Body,DNF) :-
	Body =.. [F|Arg],
	( F='*AND*',!, make_dnf_conj(Arg,DNF)
    ; F='*OR*' ,!, make_dnf_disj(Arg,DNF)
    ; DNF = Body ).

make_dnf_conj(Arg,DNF) :-
	divide_and_conquer(Arg,[C|Cs]),
	C =.. [F|Arg0],
	( F='*OR*',!,make_dnf_conj0(Cs,Arg0,Conj)
    ; make_dnf_conj0(Cs,[C],Conj)),
	DNF =.. ['*OR*'|Conj].

%% Disj* must be the form [a,'*AND*'(a,b),'*AND*'(x,y),y,z,..]
make_dnf_conj0([C|Cs],Disj0,Disj) :-
	C =.. [F|Arg],
	( F='*OR*',!,
	  unfold(Disj0,Arg,Disj1)
    ; add_conj(Disj0,C,Disj1)),
	make_dnf_conj0(Cs,Disj1,Disj).
make_dnf_conj0([],Disj,Disj):-!.

unfold(B,[A|As],Disj) :-
	add_conj(B,A,Disj0),
	append(Disj0,Disj1,Disj),
	unfold(B,As,Disj1).
unfold(_,[],[]):-!.

add_conj([B|Bs],C,[D|Ds]) :-
	B =.. [F|Arg0],
	C =.. [G|Arg1],
	( F='*AND*',G='*AND*',!,append(Arg0,Arg1,Arg)
    ; F='*AND*',!,append(Arg0,[C],Arg)
	; G='*AND*',!,Arg=[B|Arg1]
    ; Arg=[B,C]),
	D =.. ['*AND*'|Arg],
	add_conj(Bs,C,Ds).
add_conj([],_,[]):-!.

make_dnf_disj(Arg,DNF) :-
	divide_and_conquer(Arg,Disj),
	make_dnf_disj0(Disj,Disj1),
	DNF =.. ['*OR*'|Disj1].
	  
make_dnf_disj0([D|Ds],Disj) :-
	D =.. [F|Arg],
	( F='*OR*',!,append(Arg,Disj0,Disj)
    ; Disj=[D|Disj0] ),
	make_dnf_disj0(Ds,Disj0).
make_dnf_disj0([],[]):-!.

divide_and_conquer([A|As],[B|Bs]) :-
	make_dnf(A,B),
	divide_and_conquer(As,Bs).
divide_and_conquer([],[]):-!.

%%%  Get DNF-list
%%%       DNF=*OR*(q,*AND*(a,h),*AND*(a,k),*AND*(b,d,f,h),...)
%%%  DNF-list=[[q],[a,h],[a,k],[b,d,f,h],...]

get_dnf_list(DNF,DNFL):-
	( DNF =.. ['*OR*'|Disj],!,get_dnf_list0(Disj,DNFL)
    ; DNFL = [[DNF]]).

get_dnf_list0([D|Ds],[Conj|DNFL]):-
	( D =.. ['*AND*'|Conj],! ; Conj=[D] ),
	get_dnf_list0(Ds,DNFL).
get_dnf_list0([],[]):-!.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%  Three exection commands:
%%%   (1) Sampling Execution
%%%   (2) Answer with Probability
%%%   (3) Answer with Formula
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%% (1) Sampling Execution
%%% <ex.>
%%%     | ?- sample(bloodtype(X)).
%%%  
%%%     X = a ?
%%%
%%% Also available for meta-programming of PRISM
%%% <ex.>
%%% go(Loc,Dir) :-
%%%     ( is_wall(Loc),
%%%       ( sample(coin(head)), Dir=right
%%%       ; Dir=left )
%%%     ; Dir=forward ).

sample(Term) :-
	retractall('*Sampled_BSW*'(_,_,_)),
	Term =.. TermL,
	STerm =.. ['*Sample*'|TermL],
	call(STerm).

sample_bsw(G,N,R) :-
	( clause('*Sampled_BSW*'(G,N,X),true),!,R=X
	; sample_bsw(G,X),assertz('*Sampled_BSW*'(G,N,X)),!,
	  R=X ).

sample_bsw(G_id,X) :-
	random_float(1.0,R),!,
	clause('*Pb_BSW*'(G_id,Pb),true),
	( Pb>R, X=1 ; X=0 ).

%%% make sample_routines

make_sample :-
	%% collect normal Clauses
	findall([N,Head,Body],
	        '*Sentence*'(N,normal,[Head,Body]),
		    Clauses0),
	sort(Clauses0,Clauses),
	make_sample_routines0(Clauses).

make_sample_routines0([[_,Head,Body]|Cs]):-
	make_sample_routines1(Body,Sbody0),
	restore_body(Sbody0,Sbody),   %% bollowed from make_expl translator
	Head =.. [F|Arg],             %% ( *AND*/*OR* term -> and(,)/or(;) )
	functor(Head,F,ArgN),
	( '*Prob_Pred*'(F,ArgN),!,
	  SHead =.. ['*Sample*',F|Arg]
    ; SHead = Head ),
	( clause(SHead,Sbody),!,true ; assertz((SHead :- Sbody)) ),
	make_sample_routines0(Cs).
make_sample_routines0([]):-!.

%%% Body is the compoundterm of *AND* and *OR*
make_sample_routines1(Body,Sbody) :-
	Body =.. [F|Arg],
	functor(Body,F,ArgN),
	( F='*AND*',!,
	  make_sample_routines2(Arg,Sarg),
	  Sbody =.. [F|Sarg]
    ; F='*OR*',!,
	  make_sample_routines2(Arg,Sarg),
	  Sbody =.. [F|Sarg]
    ; F= bsw,!,
	  Sbody =.. [sample_bsw|Arg]
    ; '*Prob_Pred*'(F,ArgN),!,
	  Sbody =.. ['*Sample*',F|Arg]
    ; Sbody = Body ).

make_sample_routines2([A|As],[SA|SAs]) :-
	make_sample_routines1(A,SA),!,
	make_sample_routines2(As,SAs).
make_sample_routines2([],[]):-!.

%%% (2) Answer with Probability
%%% <ex.>
%%%     | ?- prob((bloodtype(a);bloodtype(b)),P).
%%%
%%%     X = 0.6 ?
%%%
%%% Also available for meta-programming of PRISM 
%%% <ex.>
%%% do(Loc,Act) :- do Act at the location Loc
%%% wumpus(Loc) :- wumpus(a kind of monster) is at the location Loc
%%% move(L,Act,L') :- move by Act from L to L'
%%%
%%% do(Loc,Act) :- 
%%%     move(Loc,forward,NewLoc1),
%%%     move(Loc,backward,NewLoc2),
%%%     prob(wumpus(NewLoc1),P1), 
%%%     prob(wumpus(NewLoc2),P2),
%%%     ( P1 > P2, Act=forward   % move less dengerous place
%%%     ; Act=backward ). 

prob(Body,Prob) :-
	parse_body1(Body,Bterm,Preds,[]),!, % borrowed from parsing routine
	( prob_pred_check(Preds,yes),!
    ; message('{PRISM ERROR: prob(Formula,Prob) -- Formula cannot have non-probabilistic atom.}'), fail ),!,
	prob_calculate(Bterm,Prob).

prob(Body) :-
	parse_body1(Body,Bterm,Preds,[]),!, % borrowed from parsing routine
	( prob_pred_check(Preds,yes),!
    ; message('{PRISM ERROR: prob(Formula) -- Formula cannot have non-probabilistic atom.}'), fail ),!,
	( prob_calculate(Bterm,Prob), print_prob(Bterm,Prob)
    ; print_prob_false(Bterm) ).

%cprob(Body,Cond,Prob) :-
%	prob(Cond,CProb),
%	prob((Body,Cond),BProb),
%	Prob is BProb/CProb.

%cprob(Body,Cond) :-
%	parse_body1(Body,Bterm,_,[]),
%	parse_body1(Cond,Cterm,_,[]),
%	prob(Cond,CProb),
%	prob((Body,Cond),BProb),
%	Prob is BProb/CProb,
%	print_cprob(Bterm,Cterm,Prob).

cprob(Body,Cond,Prob) :-
	parse_body1(Body,Bterm,BPreds,[]),
	parse_body1(Cond,Cterm,CPreds,[]),!,
	( prob_pred_check(BPreds,yes),!
    ; message('{PRISM ERROR: cprob(Formula,Cond,CProb) -- Formula cannot have non-probablisitic atom.}'),fail ),
	( prob_pred_check(CPreds,yes),!
    ; message('{PRISM ERROR: cprob(Formula,Cond,CProb) -- Cond cannot have non-probablisitic atom.}'),fail ),
	( cond_var_check(Cterm,yes),!
    ; message('{PRISM ERROR: cprob(Formula,Cond,CProb) -- Cond must be ground.}'),
	  fail ),!,
	( prob_calculate(Cterm,CProb),!
    ; message('{PRISM ERROR: cprob(Formula,Cond) -- Cond is false with probability 1.}'), fail ),
    ( prob_calculate('*AND*'(Bterm,Cterm),BProb) ; fail),
	Prob is BProb/CProb.

cprob(Body,Cond) :-
	parse_body1(Body,Bterm,BPreds,[]),
	parse_body1(Cond,Cterm,CPreds,[]),!,
	( prob_pred_check(BPreds,yes),!
    ; message('{PRISM ERROR: cprob(Formula,Cond) -- Formula cannot have non-probablisitic atom.}'),fail ),
	( prob_pred_check(CPreds,yes),!
    ; message('{PRISM ERROR: cprob(Formula,Cond) -- Cond cannot have non-probablisitic atom.}'),fail ),
	( cond_var_check(Cterm,yes),!
    ; message('{PRISM ERROR: cprob(Formula,Cond) -- Cond must be ground.}'),
	  fail ),!,
	( prob_calculate(Cterm,CProb),!
    ; message('{PRISM ERROR: cprob(Formula,Cond) -- Cond is false with probability 1.}'), fail ),!,
	( prob_calculate('*AND*'(Bterm,Cterm),BProb),!
    ; BProb = 0.0 ),
	Prob is BProb/CProb,
	print_cprob(Bterm,Cterm,Prob).

% prob_calculate/2: the essential part of prob/1-2 or cprob/2-3
prob_calculate(Bterm,Prob) :-
	  make_dnf(Bterm,DNF),            % borrowed from make_expl routine
	  get_dnf_list(DNF,DNFL0),        % borrowed from make_expl routine
	  list_sort(DNFL0,DNFL1), % delete duplicating conjunct
	  sort(DNFL1,DNFL),       % delete duplicating disjunct
	  get_expl_formula(DNFL,Ans0),
	  ( Ans0 = [],!,fail
	  ; sort(Ans0,Ans), % delete duplicating disjunct
	    setof(X,clause('*Id_BSW*'(X),true),G_ids), % borrowed from
	    count_DB(G_ids,Ans,Ans_P,Ans_N),           % EM routine
	    denominator(Ans_P,Ans_N,Prob)).            %
	
% print_prob/2: pretty printing part of prob/1
print_prob(Bterm,Prob) :-
	nl,write('The probability of '),
	print_term(Bterm),
	write(' is '),nl,format("~w.",[Prob]),nl.

print_prob_false(Bterm) :-
	nl,print_term(Bterm),
	write(' is '),nl,message('is false with probability 1').

% print_cprob/3: pretty printing part of cprob/2
print_cprob(Bterm,Cterm,CProb) :-
	nl,write('The conditinal probability of '),
	print_term(Bterm),nl,
	write('(when '),
	print_term(Cterm),write(' is true) is'),nl,
	format("~w.",[CProb]),nl.

% prob_pred_check: check if there exists non-prob. predicate
% R is 'yes' or 'no'.
prob_pred_check([[Pred,ArgN]|Preds],R) :-
	( '*Prob_Pred*'(Pred,ArgN),
	  prob_pred_check(Preds,R)
	; R = no ).
prob_pred_check([],yes).
	
% cond_var_check: check if there exists variable in condition part.
% R is 'yes' or 'no'.
cond_var_check(Cterm,R) :-
	( var(Cterm),!,R=no
    ; Cterm =.. [_|Args],
	  cond_var_check1(Args,R) ).

cond_var_check1([Cterm|Cterms],R) :-
	cond_var_check(Cterm,R1),
	( R1=no,!,R=no ; cond_var_check1(Cterms,R) ).
cond_var_check1([],yes).

% get_expl_formula:
% <ex.>
% get DNF of BSW which explains given A1&..&An v B1&..&Bn v...  

get_expl_formula([Conj|DNFL],Ans) :-
	retractall('*tmp_goal*'(_,_)),
	build_expl_routine(Conj),
	findall(A,
            ('*tmp_goal*'(A0,[]),sort(A0,A),non_contradictory(A)),
            Ans1),
	append(Ans1,Ans2,Ans),
	get_expl_formula(DNFL,Ans2).
get_expl_formula([],[]).

% build_expl_routine:
%   build expl_routine (named '*tmp_goal*') corresponding to 
%   given conjunction(list) and assert it
% <ex.>
% | ?- build_expl_routine([bloodtype(a),bloodtype(b)]).
% yes
% | ?- listing('*tmp_goal*').
% 
% '*tmp_goal*'(A, B) :-
%         '*Expl*'(bloodtype, a, A, C),
%         '*Expl*'(bloodtype, b, C, B).
% yes

build_expl_routine(Conj) :-
	build_expl_routine1(Conj,Var0,Var1,Goal0),
	list_to_paren(Goal0,Goal),
	assertz(('*tmp_goal*'(Var0,Var1):-Goal)).

build_expl_routine1([C|Cs],Ans0,Ans1,[NewC|Goal]) :-
	C=..Clist,
	append(['*Expl*'|Clist],[Ans0,Ans2],NewClist),
	NewC=..NewClist,
	build_expl_routine1(Cs,Ans2,Ans1,Goal).
build_expl_routine1([],Ans,Ans,[]).

%%% (3) Answer with probabilistic formula
%%%
%%% (3-a) Using probf/1 for the interactive query
%%% <ex.>
%%%     | ?- probf((bloodtype(a);bloodtype(b))).
%%%
%%%     bloodtype(a) v bloodtype(b)
%%%     is explained by
%%%       bsw(1,dad,1) & bsw(1,mum,1) 
%%%     v bsw(1,dad,1) & ... 
%%%     v ...
%%%
%%% (3-b) Using probf/2 for meta-programming
%%% <ex.>
%%%     | ?- probf((bloodtype(a);bloodtype(b)),Ans).
%%%
%%%     Ans = [[bsw(1,dad,1),bsw(1,mum,1),...],[bsw(1,dad,1),..],...]
%%%

probf(Body) :-
	parse_body1(Body,Bterm,Preds,[]), % borrowed from parsing routine
	prob_pred_check(Preds,R),!,
	( R=yes,!,
	  make_dnf(Bterm,DNF),            % borrowed from make_expl routine
	  get_dnf_list(DNF,DNFL0),        % borrowed from make_expl routine
	  list_sort(DNFL0,DNFL1), % delete duplicating conjunct
	  sort(DNFL1,DNFL),       % delete duplicating disjunct
	  get_expl_formula(DNFL,Ans0),
	  sort(Ans0,Ans), % delete duplicating disjunct
	  print_probf(Bterm,Ans)
    ; message('{PRISM ERROR: probf(Formula) -- Formula cannot have non-probabilistic atom.}'),
	  fail ).

probf(Body,Ans) :-
	parse_body1(Body,Bterm,Preds,[]), % borrowed from parsing routine
	prob_pred_check(Preds,R),!,
	( R=yes,!,
	  make_dnf(Bterm,DNF),            % borrowed from make_expl routine
	  get_dnf_list(DNF,DNFL0),        % borrowed from make_expl routine
	  list_sort(DNFL0,DNFL1), % delete duplicating conjunct
	  sort(DNFL1,DNFL),       % delete duplicating disjunct
	  get_expl_formula(DNFL,Ans0),
	  sort(Ans0,Ans) % delete duplicating disjunct
    ; message('{PRISM ERROR: probf(Formula,DNF) -- Formula cannot have non-probabilistic atom.}'),
	  fail ).

print_probf(Bterm,Ans) :-
	nl,print_term(0,Bterm),nl,!,
	( Ans=[],!,message('is false with probability 1.')
    ; message('is explained by'),
	  print_probf1(1,Ans) ).

print_probf1(N,[A|As]) :-
	( N>1,!,write('v ') ; write('  ') ),
	print_probf2(6,A), % num of columns is 6
	nl,N1 is N+1,
	print_probf1(N1,As).
print_probf1(_,[]):-!.

print_probf2(M,[C|Cs]) :-
	write(C),
	( Cs=[],!,true
    ; M=1,!,nl,write('& '),print_probf2(6,Cs)
    ; M1 is M-1,write(' & '),print_probf2(M1,Cs)).

print_term(Bterm) :- print_term(0,Bterm).
print_term(S,Bterm) :-
	Bterm =.. [F|Arg],
	( F='*AND*',!,
	  print_conjterm(Arg)
    ; F='*OR*',!,
	  ( S=1,!,write('( '),print_disjterm(Arg),write(' )')
      ; print_disjterm(Arg))
    ; print(Bterm) ).

print_conjterm([C|Cs]) :-
	print_term(1,C),
	( Cs=[],!,true
    ; write(' & '),print_conjterm(Cs)).

print_disjterm([D|Ds]) :-
	print_term(0,D),
	( Ds=[],!,true
    ; write(' v '),print_disjterm(Ds)).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% 
%%% Miscellaneous User's commands
%%%

%%% set_bsw/1,set_bsw/2: initialize the prob. of BSW
%%% (a) | ?- set_bsw(1,0.6),set_bsw(2,0.3).
%%%     yes
%%% (b) | ?- set_bsw([1:0.6,2:0.3]).
%%%     yes
%%% in both cases,
%%%     | ?- setof((G_id,Pb),clause('*Pb_BSW*'(G_id,Pb),true),Xs).
%%%     Xs=[(1,0.6),(2,0.3)]?
%%%     yes

set_bsw(G_id,Pb) :-
	write('{Init: G_id '),write(G_id),write(' <- '),
	write(Pb),write('}'),nl,
	( retract('*Pb_BSW*'(G_id,_)) ; true ),
	assertz('*Pb_BSW*'(G_id,Pb)),!,
	( clause('*Id_BSW*'(G_id),true),! ; assertz('*Id_BSW*'(G_id)) ).

set_bsw([G_id:Pb|GPs]) :-!,set_bsw(G_id,Pb),set_bsw(GPs).
set_bsw([]):-!.
set_bsw(_) :- 
	write('{PRISM ERROR: init/1 -- argument must be the list of G_id:P where G_id is group ID, P is the probability assigned to BSWs whose group ID is G_id.}'),nl,fail.

%%% alias
init_bsw(G_id,Pb) :- set_bsw(G_id,Pb).
init_bsw(GPs) :- set_bsw(GPs).

%%% show_bsw/0: print the prob. of BSW
show_bsw :-	setof((G_id,Pb),'*Pb_BSW*'(G_id,Pb),GPs),show_bsw(GPs).
show_bsw([(G_id,Pb)|GPs]) :-
	format("G_id ~w : ~w",[G_id,Pb]),nl,show_bsw(GPs).
show_bsw([]).

%%% save_bsw/0, save_bsw/1: save the prob. of BSW to file
%%% (a) | ?- save_bsw(any_file_as_you_like).
%%% (b) | ?- save_bsw.
%%% in the case (a), saved to the file 'any_file_as_you_like'
%%% in the case (b), saved to the default file 'prismBSW'

save_bsw(File) :- tell(File),listing('*Pb_BSW*'),listing('*Id_BSW*'),told.
save_bsw :- save_bsw(saved_BSW).

%%% restore_bsw/0, restore_bsw/1: restore the prob. of BSW from file

restore_bsw :- restore_bsw(saved_BSW).
restore_bsw(File) :-
	( (clause('*Pb_BSW*'(_,_),true) ; clause('*Id_BSW*'(_,_),true)),!,
	  write('Are you sure you want to overwrite BSW parameters? (y./n.)-> '),
	  read(R)
    ; R=n ),
	( R=y,!,retractall('*Pb_BSW*'(_,_)),retractall('*Id_BSW*'(_,_)),
	  see(File),restore_bsw1(File),seen
	; message('ignored.') ).

restore_bsw1(File) :-
	read(Sentence),!,
	( Sentence=end_of_file,!
    ; ( Sentence='*Pb_BSW*'(_,_) ; Sentence='*Id_BSW*'(_) ),!,
	  assertz(Sentence),restore_bsw1(File)
    ; format("{PRISM WARNING: ~w sentence ~w ignored.}",[File,Sentence]),
	  nl,restore_bsw1(File) ).

% show_prob_pred/0: show probabilistic predicates
show_prob_pred :-
	setof(Predname/Arity,'*Prob_Pred*'(Predname,Arity),PAs),
	message('Probabilistic predicates are:'),nl,
	show_prob_pred1(PAs).

show_prob_pred1([PA|PAs]) :-
	format('    ~w',[PA]),nl,show_prob_pred1(PAs).
show_prob_pred1([]).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%%   Data handling routines:
%%%

%%% list_to_paren: convert list to conjunction
%%% <ex.> list_to_paren([a,b,c],(a,b,c)).
list_to_paren([X|Xs],(X,Ys)) :-	list_to_paren(Xs,Ys).
list_to_paren([X],(X)).

%%% append (list version)
list_append([],[]) :- !.
list_append([L|Ls],X) :- append(L,Y,X),list_append(Ls,Y).

%%% is this list?
is_list([]).
is_list([_|X]) :- is_list(X).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%%   EM routine (built-in learning routine)
%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% An EM learning algorithm for BSW programs run on SICStus2.1
% currently, only deal with positive goals.
% created for Quintus Prolog Release 2.5 by T.Sato, July 22th, 1994
% modified for Sicstus Prolog by T.Sato, August 6th, 1995
% modified for PRISM (Programming In Statistical Modeling)
%     by Y.Kameya, October 28th, 1996

:- dynamic foreign/3, foreign_file/2.

foreign(set_seed,c,set_seed(+integer)).
	% set_seed(99999) etc, initialization for random C routines
foreign(random_float,c,random_float(+float,[-float])).
	% random_float(r,R): return a random real <R> <= <r>
foreign(random_int,c,random_int(+integer,[-integer])).
	% random_int(r,I): return a random integer <I> <= <r>

% Assert which file those C functions are on
foreign_file('random.o',[set_seed, random_float, random_int]).

% Load those C functions (this is placed above assertions)
:- load_foreign_files(['random.o'],[]),
       abolish([foreign/3,foreign_file/2]).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%                                          %%
%%  EM algorithm for Positive Observations  %%
%%                                          %%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Define global variables:
%
% assertz((goal(Args,Ans):- Bdy)).. Ans =[bsw(G_id,T,ON),...]
% assertz('*Ans*'(N,Ans2))	... Ans2 =[[G_id,#ON,#OFF],...] for Nth goal
% assertz('*P_DB*'(N,P_DB))     ... keep P_{DB} for Nth goal
% assertz('*dice_BSW*'(G_id,Pb))... G_id's Pb for data generation
% assertz('*Pb_BSW*'(G_id,Pb))	... temporary store for G_id's Pb
% assertz('*Id_BSW*'(G_id))	... keep the list of G_id's, auxiliary.
% assertz('*iteration*'(Count))	... count iterations

%-------------- Top loop ---------------

% Learning parameters from positive goals by using the EM algorithm. E.g.,
% :- em_bsw([goal([a,b,b],Ans1),goal([a,b,b,b,b],Ans2)]) returns
% estimated values of probabilistic parameters in a BSW-program
% generating those goals.
% goal(Args,Ans) must be designed problem-wise.
% 'Ans' must be of the form: Ans =[bsw(G_id,T,ON),...]

define_goal((goal(Args,Ans):- Bdy)):-
	( retract((goal(_,_):-_)), fail ; true ),!,
	assertz((goal(Args,Ans):- Bdy)).

initialize :- retract_global_vars, init_iteration.

retract_global_vars :-
	retractall('*Id_BSW*'(_)),
	retractall('*Id_pair_BSW*'(_,_)),
	retractall('*Id_size*'(_,_)),
	retractall('*Ans_Args*'(_,_)),
	retractall('*Ans*'(_,_,_,_)),
	retractall('*Pooled_Av*'(_,_)),
	retractall('*Pooled_P_DB*'(_,_)),
 	retractall('*Pb_BSW*'(_,_)),
	retractall('*P_DB*'(_,_)),
	retractall('*iteration*'(_)),
	retractall('*Log_like*'(_,_)),
	retractall('*Goals*'(_)).

init_iteration :- assertz('*iteration*'(0)).

% save global variables into file
save_status :- save_status('EM_status'). % default file name is EMstate
save_status(File) :-
	tell(File),
	listing(['*Id_BSW*','*Id_pair_BSW*','*Id_size*','*Ans_Args*',
	         '*Ans*','*Pb_BSW*','*P_DB*','*iteration*',
		 '*Log_like*','*Goals*','*Pooled_Av*','*Pooled_P_DB*']),
	told,
	format("{learning status is saved in ~w.}",[File]),nl.

% restore global variables from file
restore_status :- restore_status('EM_status'). % default file name is EMstate
restore_status(File) :-	
	retract_global_vars,see(File),restore_status1,seen,
	format("{learning status is restored from ~w.}",[File]),nl.
restore_status1 :- read(A),!,( A=end_of_file ; assertz(A),!,restore_status1 ).

em_bsw(Goals):-
	% Goals must be of the form [goal([b,b],_),goal([b,b,a],_)]
	% em_bsw maximizes the likelihood of 'goal([b,b],_)&goal([b,b,a],_)'
	initialize,!,
	message('{building explanations...}'),
	gen_ans(1,1,Goals),!, % generate Ans=[bsw(Id,T,ON),..] for Ith Goal and
                          % assertz('*Ans_Args*'(I,Args)),
                          % assertz('*Ans*'(Args,Ans_P,Ans_N,Ans_C)),
                          % assertz('*Id_BSW*'(Id))
	goal_length(Goals,M), % Goals are numbered from 1 to M. They are
                          % referred to by their numbers hereafter.
	                      % assertz('*Goal*'(M))
	( setof(X,'*Id_BSW*'(X),Y),
      message('{built explanations for all the goals.}')
    ; message('{PRISM ERROR: no explanations for all the goals.}'),
      fail ),
	sort(Y,G_ids),
		 	          % get all G_ids relevant to Goals and sort them
	ini_Pb(G_ids),	  % assign random Pb to each G_id and
			          % assertz('*Pb_BSW*'(G_id,Pb))
	setof(pb_bsw(G_id,Pb),clause('*Pb_BSW*'(G_id,Pb),true),Pbs),
	print_loop(Pbs),
	renew_P_DB(M),    % compute P_{DB}(B_N=Y_N|Theta) for Nth Goal and
			          % assertz('*P_DB*'(N,P_DB)) (1=<N=<M)
	clause('*P_DB*'(log_total,Log_like),true),
	print_log_like(Log_like),
	ini_Pb_sum(G_ids,Ini),	% Ini=[total(G_id,0.0,0.0),...]
	Epsilon = 0.000001, 	% judged to be 0 if below Epsilon
	em_loop(M,G_ids,Ini,Epsilon).

restart :-
	clause('*iteration*'(N),true),
	clause('*Goals*'(M),true),
	format('{restart em_learning with ~wth iteration}',[N]),nl,
	setof(X,'*Id_BSW*'(X),Y),sort(Y,G_ids),
	setof(pb_bsw(G_id,Pb),clause('*Pb_BSW*'(G_id,Pb),true),Pbs),
	print_loop(Pbs),
	renew_P_DB(M),
	clause('*P_DB*'(log_total,Log_like),true),
	print_log_like(Log_like),
	ini_Pb_sum(G_ids,Ini),
	Epsilon = 0.000001,
	em_loop(M,G_ids,Ini,Epsilon).

goal_length(Goals,M) :-	length(Goals,M),assertz('*Goals*'(M)),!.

em_loop(M,G_ids,Ini,Epsilon):-
	clause('*P_DB*'(log_total,Old_Log_like),true),
	retractall('*Pooled_Av*'(_,_)),
	retractall('*Pooled_P_DB*'(_,_)),
	sum_on_off(M,G_ids,Ini,Total),   % most computationally loaded part
				         % Total=[total(G_id,Av_on,Av_off)..]
	renew_Pb(Total,New_Pbs,Epsilon), % renew Pb for each G_id and
                                     % assertz '*Pb_BSW*'(G_id,New_Pb)
	print_loop(New_Pbs),		     % New_Pbs=[pb_bsw(G_id,New_Pb_ON),..]
	renew_P_DB(M),	% renew P_DB= P_{DB}(B_N=Y_N|Theta) with New_Pbs
			        % for N_th Goal and assertz '*P_DB*'(N,P_DB)
	clause('*P_DB*'(log_total,Log_like),true),
	print_log_like(Log_like),
	D is (Log_like - Old_Log_like),
	( ( D<Epsilon ; Log_like>0 ),!,print_converge
    ; em_loop(M,G_ids,Ini,Epsilon) ).

% assertz '*P_DB*'(N,P_{DB}(B_N=Y_N|Theta)) (N>0) and
% '*P_DB*'(log_total,Log_like) where
% Log_like =log(P_{DB}(B1=Y1|Theta)*...*P_{DB}(B_M=Y_M|Theta))
% modified by Y.K.
renew_P_DB(M):- renew_P_DB_1(M,0.0),!.
renew_P_DB_1(I,Log_P1):-
	0<I,
	clause('*Ans_Args*'(I,Args),true),
	( clause('*Pooled_P_DB*'(Args,P_DB),true),!
    ; clause('*Ans*'(Args,Ans_P,Ans_N,_),true),
	  denominator(Ans_P,Ans_N,P_DB),
	  assertz('*Pooled_P_DB*'(Args,P_DB)) ),
	% P_DB =0.0 either if Ans_L=[] or P_DB is too small, see
	% | ?-  0.0 =:=1.0e-44.
	% yes
	% | ?-  0.0 =:=1.0e-43.
	% no
	% We will skip any computation concerning P_DB for I_th Goal
	( P_DB > 0.0, Log_P_DB is log(P_DB) % C routine
    ; Log_P_DB = 0.0 ),
	( retract('*P_DB*'(I,_)) ; true ),
	assertz('*P_DB*'(I,P_DB)),
	Log_P3 is (Log_P1 + Log_P_DB),
	I1 is I-1,!,
	renew_P_DB_1(I1,Log_P3).

renew_P_DB_1(0,Log_like):-
	( retract('*P_DB*'(log_total,_)) ; true ),!,
	assertz('*P_DB*'(log_total,Log_like)).

% Assign the initial prob. to each bsw.
ini_Pb([G_id|X]):-
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%	prob_bsw(G_id,1,Pb),         % C rountine
	random_float(0.8,P1), Pb is P1+0.1,
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
	assertz('*Pb_BSW*'(G_id,Pb)),!,
	ini_Pb(X).
ini_Pb([]).

% Just prepare initial data
ini_Pb_sum([G_id|X],[total(G_id,0.0,0.0)|Y]):-!,ini_Pb_sum(X,Y).
ini_Pb_sum([],[]).

%--------- Renewing probabilities ------------------
% First caluculate Av_on/off for each G_id where
% Av_on/off= Sum_{Xi} P_{DB}(Ai=Xi|Bi=Yi,Theta)*|Ai =_{G_id} Xi|_{1}/{0},
% and return [pb_bsw(G_id,Pb_ON),...] where Pb_ON =Av_on/(Av_on+Av_off)
renew_Pb([total(G_id,Av_on,Av_off)|X],[pb_bsw(G_id,Pb_ON)|Y],Epsilon):-
	Av_on_off is (Av_on + Av_off),
	( Av_on_off <Epsilon,
			% Av_on_off too small. G_id cannot increase
			% the likelihood anymore, So keep the old value.
	  clause('*Pb_BSW*'(G_id,Pb_ON),true)
	; Pb_ON is Av_on/Av_on_off,
	  ( retract('*Pb_BSW*'(G_id,_)) ; true ),
	  assertz('*Pb_BSW*'(G_id,Pb_ON)) ),!,
	renew_Pb(X,Y,Epsilon).
renew_Pb([],[],_).

sum_on_off(I,G_ids,T1,Total):-
	I>0,            % we are processing Ith goal B_I
	( clause('*P_DB*'(I,P_DB),true) ; P_DB = 0.0 ),
			% P_DB=Sum_{Xi} P_{DB}(Ai=Xi|B_I=Y_I,Theta)
	( P_DB > 0.0,
	  clause('*Ans_Args*'(I,Args),true),
	  ( clause('*Pooled_Av*'(Args,Av_L),true),!
      ; clause('*Ans*'(Args,_,_,Ans_C),true),
	    av_list(G_ids,Ans_C,P_DB,Av_L),
		assertz('*Pooled_Av*'(Args,Av_L)) ),
      add_av(Av_L,T1,T2)	% calculate Av_L for B_I where
                            % Av_L =[on_off(G_id,Av_on,Av_off),..]
    ; T2=T1 ),              % P_DB = 0.0 for I_th Goal. We will skip 
	                        % any computation concerning I_th Goal
	N1 is I-1,!,	        % T2 =[total(G_id,Av_on,Av_off),..]
	sum_on_off(N1,G_ids,T2,Total).
sum_on_off(0,_,Total,Total).

% subroutine for sum_on_off
add_av( [on_off(G_id,Av_on,Av_off)|X],
	[total(G_id,ON,OFF)|Y],
	[total(G_id,ON1,OFF1)|Z] ):-
	ON1 is (ON + Av_on),
	OFF1 is (OFF + Av_off),!,
	add_av(X,Y,Z).
add_av([],[],[]).

%%------- Initialization: set up ans_list & G_id list ----------
%% generate logical explanations for I_th goal: goal(Args,_)
%% store only non_contradictory Ans.

%%% gen_ans:
%%%
%%% [NOTE] the probability of disjunction is calculated
%%% recursively by the following formula:
%%% P(C0 v C1 v...v Cn) = P(C0)+P(C1 v ... v Cn)-P(C0 & (C1 v...v Cn))
%%% where Cj is the conjunction of bsw/3

gen_ans(I,J,[goal(Args,_)|R]) :-
	assertz('*Ans_Args*'(I,Args)),
	( clause('*Ans*'(Args,Ans_P,Ans_N,Ans_C),true),!,
      J1 is J+1
    ; findall(A,
	          (goal(Args,A0),   % generate a logical explanation
               sort(A0,A),      % delete duplicating conjunct
               % bsw(G_id,T,ON) & bsw(G_id,T,ON) = bsw(G_id,T,ON)
 		       non_contradictory(A), % check if A is contradictrory
			   assert_G_id(A)),      % also modified by Y.K.
			  Ans0),
 	  % Ans0=[[bsw(G_id0,T,ON),...],[bsw(G_id1,T,ON),...],...]
      %     = (bsw(G_id0,T,ON)&...)v(bsw(G_id1,T,ON)&...)v...
	  ( Ans0 = [],!,
	    format("{PRISM ERROR: no exaplanation for ~wth goal.}",
		       [I]),fail
      ; setof(X,clause('*Id_BSW*'(X),true),G_ids),
	    sort(Ans0,Ans),  % delete duplicating disjunct
                         % ...v Ci v Ci v... = ...v Ci v...
	    id_size(G_ids),  % get G_id size and assert it
        count_DB(G_ids,Ans,Ans_P0,Ans_N0),
	    difference(Ans_P0,Ans_N0,Ans_P,Ans_N),
        % CONVERT: DNF --> Disjoint Sum (with complement)
	    disjoint_sum(Ans,DSum), 
	    count_F(G_ids,DSum,Ans_C),
	    assertz('*Ans*'(Args,Ans_P,Ans_N,Ans_C))),
	    % 'P' stands for Positive, and 'N' stands for Negative
        % P(C0 v...v Cn) = P(C0)+P(C1 v ... v Cn)-P(C0 & (C1 v...v Cn))
        %                  ------ positive ------ ------ negative -----
      J1=J ),
	display_dot,
	I1 is I+1,!,
	gen_ans(I1,J1,R).
gen_ans(_,_,[]):-nl,!.

display_dot :- put(46),ttyflush.

%%% id_size: get each G_id's size (num of T's) and assert it
%%% [NOTE] Pairs must be sorted.
id_size([G_id|G_ids]) :-
	( clause('*Id_size*'(G_id,_),true),!
	; setof(T,clause('*Id_pair_BSW*'(G_id,T),true),Ts),
	  length(Ts,N),
	  assertz('*Id_size*'(G_id,N)) ),!,
	id_size(G_ids).
id_size([]).

%%% count_DB: created by Y.Kameya
%%% P(C0 v C1 v...v Cn) = P(C0)+P(C1 v ... v Cn)-P(C0 & (C1 v...v Cn))
% Ans=[[bsw(G_id0,T,ON),...],[bsw(G_id1,T,ON),...],...]
%    = (bsw(G_id0,T,ON)&...)v(bsw(G_id1,T,ON)&...)v...
% Ans_S=[[[Gid,#ON,#OFF],...],[[Gid,#ON,#OFF],...],...]
% Ans_I=[[[Gid,#ON,#OFF],...],[[Gid,#ON,#OFF],...],...]

count_DB(G_id,[A|Ans],Ans_S,Ans_I) :-
	( contradictory(A),!, % A is already sorted
	  Ans_S=Ans_S2, Ans_I=Ans_I2
    ; count_ans(G_id,A,CountS),
	  unfold_ans(A,Ans,Ans1),
      count_DB(G_id,Ans1,Ans_S1,Ans_I1), % calculate C0 & (C1 v...v Cn)
	  append([CountS|Ans_I1],Ans_S2,Ans_S),
	  append(Ans_S1,Ans_I2,Ans_I)),
	count_DB(G_id,Ans,Ans_S2,Ans_I2).    % calculate C1 v...v Cn
count_DB(_,[],[],[]):-!.

%% Count ON/OFF occurrences of G_id in Ans1 =[bsw(G_id,T,ON),...]
%% count_ans: borrowed from original count/3 and modified
count_ans([G_id|X],Ans1,Z):-
	count_ans1(Ans1,G_id,0,N1,0,N0),!,
	( N1=0,N0=0,!,Z=Y
    ; Z=[(G_id,N1,N0)|Y] ),
	count_ans(X,Ans1,Y).
count_ans([],_,[]):-!.
count_ans1([bsw(G_id,_,ON)|L],G_id,X1,N1,X0,N0):-!,
	( ON == 1, Y1 is X1+1, Y0=X0
	; ON == 0, Y1=X1, Y0 is X0+1 ),!,
	count_ans1(L,G_id,Y1,N1,Y0,N0).
count_ans1([_|L],G_id,X1,N1,X0,N0):-!,count_ans1(L,G_id,X1,N1,X0,N0).
count_ans1([],_,N1,N1,N0,N0):-!.

%%% unfold_ans: created by Y.K.
%%% P(C0 & (C1 v...v Cn)) = P(C0&C1 v C0&C2 v...v C0&Cn)
unfold_ans(A,[Ans0|Ans1],[Ans3|Ans]) :-
	append(A,Ans0,Ans2),sort(Ans2,Ans3), % delete duplicating conjunct
	unfold_ans(A,Ans1,Ans).
unfold_ans(_,[],[]):-!.

% count_F: does not fill up complement
count_F(G_id,[A|Ans],[AC|Ans_C]) :-
	count_ans(G_id,A,AC),count_F(G_id,Ans,Ans_C).
count_F(_,[],[]):-!.

%%% cmpl: complement disjoint bsw/3
% {A,B,C,D}: possible BSW -- bsw(G_id,T,1)
% ~bsw(G_id,T,1)=bsw(G_id,T,0)
% A&B v A&C = A&B&C&D  + A&B&C&~D + A&B&~C&D + A&B&~C&~D 
%           + A&~B&C&D + A&~B&C&~D
% (since each disjunct is disjoint)
% <ex.>
% cmpl([(1,d),(1,m),(2,m)],
%      [[bsw(1,d,1),bsw(1,m,0)],[bsw(1,d,1),bsw(2,m,0)]],
%      [[bsw(1,d,1),bsw(1,m,0),bsw(2,m,0)],
%       [bsw(1,d,1),bsw(1,m,0),bsw(2,m,1)],
%       [bsw(1,d,1),bsw(1,m,1),bsw(2,m,0)]]).

cmpl(Pairs,Ans,Ans_C) :-
	cmpl1(Pairs,Ans,Ans_C1),sort(Ans_C1,Ans_C).

% subroutines for cmpl/3
cmpl1(Pairs,[A|Ans],Ans_C) :-
	cmpl_check(Pairs,A,C_Pairs), % extract complement-pairs
	add_cmpl(C_Pairs,A,Ans_C1),
	list_sort(Ans_C1,Ans_C2),
	append(Ans_C2,Ans_C3,Ans_C),
	cmpl1(Pairs,Ans,Ans_C3).
cmpl1(_,[],[]):-!.

%% cmpl_check: extract complement-pairs
% <ex.>
% cmpl_check([(1,d),(1,m),(2,m)],[bsw(1,d,1),bsw(1,m,0)],[(2,m)]).
% (a pair (2,m) is not occurred in [bsw(1,d,1),bsw(1,m,0)].

cmpl_check([(G_id,T)|Pairs],A,C_Pairs) :-
	( cmpl_check1(G_id,T,A),!,
	  C_Pairs=C_Pairs1
    ; C_Pairs=[(G_id,T)|C_Pairs1] ),
	cmpl_check(Pairs,A,C_Pairs1).
cmpl_check([],_,[]):-!.

% This routine looks like 'member' ;-p
cmpl_check1(G_id,T,[bsw(G_id,T,_)|_]):-!.
cmpl_check1(G_id,T,[_|As]) :- !, cmpl_check1(G_id,T,As).

%% add_cmpl: add complement
% <ex.>
% add_cmpl([(2,d),(2,m)],[bsw(1,d,1),bsw(1,m,0)],
%          [[bsw(1,d,1),bsw(1,m,0),bsw(2,d,1),bsw(2,m,1)],
%           [bsw(1,d,1),bsw(1,m,0),bsw(2,d,1),bsw(2,m,0)],
%           [bsw(1,d,1),bsw(1,m,0),bsw(2,d,0),bsw(2,m,1)],
%           [bsw(1,d,1),bsw(1,m,0),bsw(2,d,0),bsw(2,m,0)]]).

add_cmpl(C_Pairs,A,Ans_C) :- add_cmpl1(C_Pairs,[A],Ans_C).

add_cmpl1([(G_id,T)|C_Pairs],A,Ans_C) :-
	add_cmpl2(1,G_id,T,A,Ans_C1), % on
	add_cmpl2(0,G_id,T,A,Ans_C2), % off
	append(Ans_C1,Ans_C2,Ans_C3),
	add_cmpl1(C_Pairs,Ans_C3,Ans_C).
add_cmpl1([],A,A):-!.

add_cmpl2(ONOFF,G_id,T,[A|As],[[bsw(G_id,T,ONOFF)|A]|Ans_C]) :-
	add_cmpl2(ONOFF,G_id,T,As,Ans_C).
add_cmpl2(_,_,_,[],[]):-!.


% Check if Ans1 contains both bsw(G_id,T,1) and bsw(G_id,T,0) etc
% [NOTE] Ans1 must be sorted in advance.
non_contradictory(Ans1):- \+ contradictory(Ans1).

contradictory([bsw(G_id,T1,ON1),bsw(G_id,T2,ON2)|_]):-
    T1==T2,
    ON1\==ON2,!,
%	write('Contradictory BSWs: '),
%	write([bsw(G_id,T1,ON1),bsw(G_id,T2,ON2)]),nl,
    true.
contradictory([_|L]):- contradictory(L).

%% record which G_id appeared.
%assert_G_id([bsw(G_id,_,_)|X],[G_id|Y]):-
%	( clause('*Id_BSW*'(G_id),true)
%	; assertz('*Id_BSW*'(G_id))
%	),!,
%	assert_G_id(X,Y).
%assert_G_id([],[]).

%% assert_G_id: modified by Y.K.
% record which G_id and Id_pair appeared.
assert_G_id([bsw(G_id,T,_)|X]):-
	( clause('*Id_BSW*'(G_id),true) ; assertz('*Id_BSW*'(G_id)) ),
	( clause('*Id_pair_BSW*'(G_id,T),true)
    ; assertz('*Id_pair_BSW*'(G_id,T)) ),!,
	assert_G_id(X).
assert_G_id([]).

%----------  Average list for G_id w.r.t Ith goal ----------
% modified by Y.K.
% Compute Av_L for some goal B_I where
% Av_L = [on_off(G_id,Av_on,Av_off),on_off(2,1.0,0.0),...]
% Av_on/off= Sum_{Xi}P_{DB}(Ai=Xi|B_I=Y_I,Theta)*|Ai =_{G_id} Xi|_{1}/{0}
% Ans_L = [[(G_id,#ON,#OFF),...],...]
av_list([G_id|X],[Ans],_,[on_off(G_id,N1,N0)|Y]):-
	num_of(Ans,G_id,N1,N0),!,
	av_list(X,[Ans],_,Y).

av_list([G_id|X],Ans_C,P_DB,[on_off(G_id,Av_on,Av_off)|Y]):-
	enumerator(G_id,Ans_C,Sum_1,Sum_0),
	Av_on is Sum_1/P_DB,
	Av_off is Sum_0/P_DB,!,
	av_list(X,Ans_C,P_DB,Y).
av_list([],_,_,[]).

% Ans_C = [[(G_id,#ON,#OFF),...],...] for some goal B_I
% Sum_1/0 =Sum_{Xi} P_F(Ai=Xi|Theta)*|Ai =_{G_id} Xi|_{1}/{0}
% P_DB =P_{DB}(B_I=Y_I|Theta)
enumerator(G_id,Ans_C,Sum_1,Sum_0):-!,
	enumerator1(G_id,Ans_C,0.0,Sum_1,0.0,Sum_0).

enumerator1(G_id,[Ans|Ans_C],X1,Sum_1,X0,Sum_0):-
	pb_ans_times(Ans,Pb),
	num_of(Ans,G_id,N1,N0),
	clause('*Pb_BSW*'(G_id,PbON),true),
	PbOFF is 1-PbON,
	clause('*Id_size*'(G_id,Size),true),
	N is Size-(N1+N0),
	enumerator2(N1,N0,N,PbON,PbOFF,ON,OFF),!,
	Y1 is Pb*ON + X1,
	Y0 is Pb*OFF+ X0,
	enumerator1(G_id,Ans_C,Y1,Sum_1,Y0,Sum_0).
enumerator1(_,[],Sum_1,Sum_1,Sum_0,Sum_0).

enumerator2(N1,N0,N,PbON,PbOFF,ON,OFF) :-
	ON is N1 + N*PbON, OFF is N0 + N*PbOFF.
%        $B",(B
%       reduced  
%        $B",(B
%enumerator2(N1,N0,N,PbON,PbOFF,ON,OFF) :-
%	enumerator3(0,1.0,N1,N0,N,PbON,PbOFF,0.0,ON,0.0,OFF).

%enumerator3(K,COMB,N1,N0,N,PbON,PbOFF,ON1,ON,OFF1,OFF) :-
%	D is N-K,
%	ON2  is ON1 + COMB*exp(PbON,D)*exp(PbOFF,K)*(N1+D),
%	OFF2 is OFF1+ COMB*exp(PbOFF,D)*exp(PbON,K)*(N0+D),
%	( K = N,!,ON=ON2,OFF=OFF2
%	; K1 is K+1,
%	  COMB1 is COMB*D/(K+1),
%	  enumerator3(K1,COMB1,N1,N0,N,PbON,PbOFF,ON2,ON,OFF2,OFF)).

% Retrieve ON/OFF occurrences of G_id in Ans
num_of([(G_id,Num_ON,Num_OFF)|_],G_id,Num_ON,Num_OFF):-!.
num_of([_|L],G_id,N1,N0):- !,num_of(L,G_id,N1,N0).
num_of([],_,0,0).

% Calculate the conditional probability P_{DB}(B_I=Y_I|Theta),
% P_DB = Sum_{Xi} of P_F(Ai=Xi|Theta), given B_I=Y_I
%      = P_{DB}(B_I=Y_I|Theta),
% Ans = [[G_id,#ON,#OFF],...] <->  Ai=Xi, P_F =P_{F}(Ai=Xi|Theta)
% Ans_L = [[[G_id,#ON,#OFF],...],...] for B_I
denominator(Ans_P,Ans_N,P_DB):-
	sum_pb_ans_times_1(Ans_P,0.0,P_DB_P),
	sum_pb_ans_times_1(Ans_N,0.0,P_DB_N),
	P_DB is P_DB_P - P_DB_N.

sum_pb_ans_times(Ans_P,Ans_N,P_DB):-
	sum_pb_ans_times_1(Ans_P,0.0,P_DB_P),
	sum_pb_ans_times_1(Ans_N,0.0,P_DB_N),
	P_DB is P_DB_P - P_DB_N.

sum_pb_ans_times_1([Ans|X],P1,P_DB):-
	pb_ans_times(Ans,P2),
	P3 is P1+P2,!,
	sum_pb_ans_times_1(X,P3,P_DB).
sum_pb_ans_times_1([],P_DB,P_DB).

% P_F =P_{F}(Ai=Xi|Theta), Ans = ans. for Ai=Xi, 
pb_ans_times(Ans,P_F):- pb_ans_times_1(Ans,1.0,P_F).
pb_ans_times_1([(G_id,Num_ON,Num_OFF)|X],P1,P_F):-
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%	prob_bsw(G_id,ON,Pb),		% C routine
	clause('*Pb_BSW*'(G_id,Pb),true),
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
	pb_ans_times_2(Num_ON,Num_OFF,Pb,P1,P2),!,
	pb_ans_times_1(X,P2,P_F).
pb_ans_times_1([],P_F,P_F).

%pb_ans_times
pb_ans_times_2(1,0,Pb,P1,P2):-!,P2 is P1*Pb.
pb_ans_times_2(0,1,Pb,P1,P2):-!,P2 is P1*(1.0-Pb).
pb_ans_times_2(N1,N0,Pb,P1,P2):-
	N1>0, Y is P1*Pb, Z is N1-1,!,
	pb_ans_times_2(Z,N0,Pb,Y,P2).
pb_ans_times_2(0,N0,Pb,P1,P2):-
	N0>0, Y is P1*(1.0-Pb), Z is N0-1,!,
	pb_ans_times_2(0,Z,Pb,Y,P2).

% disjoint_sum:
%   convert DNF -- D= C1 v C2 v ... v Cn (Ci is conjunction of bsw)
%   to Disjoint Sum
%   D = C1+C2+..+(Ci v Cj)+..+(Ck v..v Cl)+..+Cn (divide)
%   (Ci and Cj are not disjoint)
%     = C1+C2+..+(C'1+C'2+..)+..+(C''1+C''+2..)+..+Cn
%                ^^^^^^^^^^^^    ^^^^^^^^^^^^^^(by complementing)

disjoint_sum([A|Ans],DSum) :-
	disjoint_sum1([A],[],Ans,DSum1),sort(DSum1,DSum2),% divide
	dsum_cmpl(DSum2,DSum3),sort(DSum3,DSum).          % complementing
disjoint_sum([],_).

% subroutines for disjoint_sum
disjoint_sum1([],[Ds0],[A|Ans],[Ds|DSum]) :- !,
	sort(Ds0,Ds),
	disjoint_sum1([A],[],Ans,DSum).
disjoint_sum1([],Ds0,[A|Ans],[non_disjoint(Ds)|DSum]) :-
	sort(Ds0,Ds),
	disjoint_sum1([A],[],Ans,DSum).
disjoint_sum1([C|Cs],Ds,Ans,DSum) :-
	disjoint_sum2(C,Ans,Cs1,Ans1),
	append(Cs,Cs1,Cs2),
	disjoint_sum1(Cs2,[C|Ds],Ans1,DSum).
disjoint_sum1([],[Ds],[],[Ds]):-!.
disjoint_sum1([],Ds,[],[non_disjoint(Ds)]).

disjoint_sum2(C,[A|Ans],Cs1,Ans1) :-
	append(C,A,Conj0),
	sort(Conj0,Conj),!,
	( contradictory(Conj),!,Cs1=Cs2,Ans1=[A|Ans2]
    ; Cs1=[A|Cs2],Ans1=Ans2 ),
	disjoint_sum2(C,Ans,Cs2,Ans2).
disjoint_sum2(_,[],[],[]).

dsum_cmpl([non_disjoint(Ds)|DSum0],DSum) :-!,
	get_Id_pairs(Ds,Pairs),
	cmpl(Pairs,Ds,DSum1),
	append(DSum1,DSum2,DSum),!,
	dsum_cmpl(DSum0,DSum2).
dsum_cmpl([Ds|DSum0],[Ds|DSum]) :- dsum_cmpl(DSum0,DSum).
dsum_cmpl([],[]).

% get_Id_pairs(Ds,Pairs): get ID-pairs Pairs from Ds
% <ex.> get_Id_pairs([[bsw(1,d,0),bsw(2,d,1)],[bsw(1,d,1),bsw(2,m,1)]],
%                    [(1,d),(2,d),(2,m)]).
get_Id_pairs(Ds,Pairs) :- 
	list_append(Ds,Ds1),
	sort(Ds1,Ds2),
	get_Id_pairs1(Ds2,Pairs).

% [NOTE] Ds must be sorted.
get_Id_pairs1([bsw(G_id,T,_)|Ds],[(G_id,T)|Pairs]) :-
	get_Id_pairs2(G_id,T,Ds,Pairs).
get_Id_pairs1([],[]).

get_Id_pairs2(G_id,T,[bsw(G_id,T,_)|Ds],Pairs) :- !,
	get_Id_pairs2(G_id,T,Ds,Pairs).
get_Id_pairs2(_,_,[bsw(G_id,T,_)|Ds],[(G_id,T)|Pairs]) :-
	get_Id_pairs2(G_id,T,Ds,Pairs).
get_Id_pairs2(_,_,[],[]).

%--------------- Miscellaneous ------------------
clear_Id_BSW :-( retract('*Id_BSW*'(_)),fail ; true ).
clear_Ans :-
	( retract('*Ans*'(_,_,_,_)),fail ; true ),
	( retract('*Ans_Args*'(_,_,_)),fail ; true ).
clear_Pb_BSW :-( retract('*Pb_BSW*'(_,_)),fail ; true ).
clear_P_DB :-( retract('*P_DB*'(_,_)),fail ; true ).

print_log_like(Log_like):-
	clause('*iteration*'(Count),true),
	Count1 is Count+1,
	retract('*iteration*'(_)),assertz('*iteration*'(Count1)),
	format("[~w] Log_like= ~6f",[Count,Log_like]),nl,
	assert('*Log_like*'(Count,Log_like)).

print_converge :- write(' -- converged!'),nl.

% display loglike graph using "gnuplot"
display_graph :- print_graph,plsys(shell('gnuplot loglike.gnu')).

% default file is 'loglike.plt'
print_graph :- 	print_graph('loglike.plt').
print_graph(File) :- 
	( clause('*Log_like*'(_,_),true),!,
	  tell(File),print_graph1(0),told
    ; write('{PRISM ERROR: [print_graph] There exists no learning data.}'),
	  nl,fail ).
print_graph1(N) :-
	( clause('*Log_like*'(N,Log_like),true),!,
	  write(N),write(' '),write(Log_like),nl,
	  N1 is N+1,
	  print_graph1(N1)
    ; true ).

print_loop(X):- Columns = 3, print_loop1(X,Columns,Columns).
print_loop1([],_,_):-nl,!.
print_loop1(X,N,Columns):- N<1,nl,!,print_loop1(X,Columns,Columns).
print_loop1([pb_bsw(G_id,New_Pb)|Y],N,Columns):-
	0<N,N1 is N-1,
	format("BSW ~w: ~6f   ",[G_id,New_Pb]),
	print_loop1(Y,N1,Columns).

rev(X,Y):- rev1(X,[],Y).
rev1([H|X],Z,Y):- !,rev1(X,[H|Z],Y).
rev1([],Z,Z).

flatten([A|As],Bs) :-!,
	append(A,Bs1,Bs),flatten(As,Bs1).
flatten([],[]).

% difference(A,B,C,D): C is C-A&B, D is D-A&B
% [NOTE] each of A,B,C,D is multi-set
% <ex1.> difference([a,b,c],[b,d],[a,c],[d]).
% <ex2.> difference([a,b,b,c],[b,d],[a,b,c],[d]).

difference(As,[B|Bs],Xs,Ys) :-
	delete(B,As,As1),
	( As=As1,!, % B isn't in As
	  Ys=[B|Ys1]
    ; Ys=Ys1 ),
	difference(As1,Bs,Xs,Ys1).
difference(As,[],As,[]).

% delete(X,Ys,Zs): delete *one* X from Ys (the rest is Zs)
delete(X,[X|Ys],Ys) :- !.
delete(X,[Y|Ys],[Y|Zs]) :- delete(X,Ys,Zs).
delete(_,[],[]).

% list version of sort/2
list_sort([A|B],[A1|C]) :- sort(A,A1),list_sort(B,C).
list_sort([],[]) :-!.

time(G):-
	statistics(runtime,_),
	call(G),
	statistics(runtime,[_,Time]),write('[ '),
	write(Time),write(' msec ]'),ttynl.

time(G,Time):-
	statistics(runtime,_),call(G),
	statistics(runtime,[_,Time]).

:-  Seed=1234567890,
	set_seed(Seed),
	write('{random_seed is set to '),
	write(Seed),write('.}'),nl.

