// Copyright (C) 1997  Toyoda Masashi (toyoda@is.titech.ac.jp)

#include <amulet.h>
#include INTER_ADVANCED__H 
#include FORMULA_ADVANCED__H

#include <strstream.h>

#include "ToyWidgets.h"
#include "Port.h"
#include "Message.h"
#include "Process.h"
#include "Rule.h"
#include "Pattern.h"
#include "Binder.h"

// ------------------------------------------------------------
// Common Slots
// ------------------------------------------------------------

// Models --------
DEFINE_SLOT(Name);
DEFINE_SLOT(ModuleList);
DEFINE_SLOT(ModuleNameList);
DEFINE_SLOT(Process);
DEFINE_SLOT(ProcessList);
DEFINE_SLOT(ProcessNameList);
DEFINE_SLOT(PatternList);
DEFINE_SLOT(PatternNameList);
DEFINE_SLOT(ProcPatternList);
DEFINE_SLOT(ProcPatternNameList);
DEFINE_SLOT(Hole);
DEFINE_SLOT(HoleList);
DEFINE_SLOT(HoleNameList);
DEFINE_SLOT(RuleList);
DEFINE_SLOT(PortList);
DEFINE_SLOT(PortNameList);
DEFINE_SLOT(InnerPortList);
DEFINE_SLOT(InnerPortNameList);
DEFINE_SLOT(GuardList);
DEFINE_SLOT(ValueList);
DEFINE_SLOT(NameList);
DEFINE_SLOT(MainRule);

// Processes
DEFINE_SLOT(Module);
DEFINE_SLOT(ModuleName);
DEFINE_SLOT(RuleVisible);

// Ports
DEFINE_SLOT(Mode);
DEFINE_SLOT(Type);
DEFINE_SLOT(Dir);
DEFINE_SLOT(Age);
DEFINE_SLOT(State);
DEFINE_SLOT(Special);
DEFINE_SLOT(Values);
DEFINE_SLOT(Binders);
DEFINE_SLOT(InputBinders);
DEFINE_SLOT(OutputBinders);
DEFINE_SLOT(HoleBinders);
DEFINE_SLOT(InnerBinders);


// Binders --------
DEFINE_SLOT(SrcPort);
DEFINE_SLOT(DestPort);

// Views --------
DEFINE_SLOT(Parent);
DEFINE_SLOT(Model);
DEFINE_SLOT(ToCopyModel);
DEFINE_SLOT(FramePart);
DEFINE_SLOT(ContentsPart);
DEFINE_SLOT(NamePart);
DEFINE_SLOT(MarkerPart);
DEFINE_SLOT(IconPart);
DEFINE_SLOT(BindersPart);

DEFINE_SLOT(MenuPopper);
DEFINE_SLOT(NameHeightMax);

// Rename Method
DEFINE_SLOT(NamingMethod);
// Add Methods
DEFINE_SLOT(AddMethod);
DEFINE_SLOT(RemoveMethod);
// Copying Methods
DEFINE_SLOT(NewMethod);
DEFINE_SLOT(CopyMethod);
DEFINE_SLOT(CopyAllMethod);
// Drag & Dropping Methods
DEFINE_SLOT(DropPoints);
DEFINE_SLOT(DropMethod);
DEFINE_SLOT(CheckDropMethod);

// For Translator --------
DEFINE_SLOT(Id);
DEFINE_SLOT(NextId);
DEFINE_SLOT(InnerName);
DEFINE_SLOT(InnerNextName);
DEFINE_SLOT(GenOrder);

// ------------------------------------------------------------
// Utility Functions
// ------------------------------------------------------------

// Path Utility
// A path is a list such as ( type, idx, type, idx, type, idx, ... )
// -----------------------------------------------------------------------

// Path types
#define PathProcess   1
#define PathHole      2
#define PathPort      3
#define PathValue     4
#define PathBinder    5


Am_Object Path_To_Obj(Am_Object root, Am_Value_List path)
{
  Am_Object node = root;
  for (path.Start(); !path.Last(); path.Next()) {
    int type = path.Get();  path.Next();
    int idx  = path.Get();
    Am_Slot_Key list_slot = 0;
    if      (type == PathProcess) list_slot = ProcessList;
    else if (type == PathHole)    list_slot = HoleList;
    else if (type == PathPort)    list_slot = PortList;
    else if (type == PathValue)   list_slot = ValueList;
    else if (type == PathBinder)  list_slot = Binders;
    else return node;
    if (!node.Valid()) return 0;
    Am_Value_List obj_list = node.Get(list_slot);
    node = Object_List_Get(obj_list, idx);
  }
  return node;
}

Am_Value_List Obj_To_Path(Am_Object obj, Am_Object root)
{
  Am_Value_List path = Am_Value_List();
  if (obj == root) return path;
  Am_Object pivot = obj;
  Am_Object parent = obj.Get(Parent);
  while (pivot != root) {
    int type = 0;
    int idx  = 0;
    if (pivot.Is_Instance_Of(ProcessProto) || pivot.Is_Instance_Of(PatternProto)) {
      type = PathProcess;
      idx = Object_List_Index(parent.Get(ProcessList), pivot);
    }
    else if (pivot.Is_Instance_Of(HoleProto)) {
      type = PathHole;
      idx = Object_List_Index(parent.Get(HoleList), pivot);
    }
    else if (parent.Is_Instance_Of(PortProto) || parent.Is_Instance_Of(MessageProto)) {
      type = PathValue;
      idx = Object_List_Index(parent.Get(ValueList), pivot);
    }
    else if (pivot.Is_Instance_Of(PortProto) && !parent.Is_Instance_Of(PortProto)) {
      type = PathPort;
      idx = Object_List_Index(parent.Get(PortList), pivot);
    }
    else if (pivot.Is_Instance_Of(BinderProto)) {
      type = PathBinder;
      idx = Object_List_Index(parent.Get_Part(BindersPart).Get(Am_GRAPHICAL_PARTS), pivot);
    }
    else { cerr << "Obj_To_Path: no such path, " << pivot << parent << endl; }
    path.Add(idx, Am_HEAD);
    path.Add(type, Am_HEAD);
    //    cout << "add " << idx << " " << type << " by " << pivot << parent << endl;
    pivot = parent;
    parent = parent.Get(Parent);
  }
  return path;
}

// Search an object at a coordinate
// --------------------------------
Am_Object Content_Instance_Of(Am_Object group, int x, int y, Am_Object ref, Am_Value_List protos)
{
  //  cout << "start content instance of" << endl;
  Am_Object found = Am_No_Object;
  for (Am_Object current = group; current.Get_Slot_Type(ContentsPart); ) {
    Am_Object parts_group = current.Get(ContentsPart);
    current = Am_Point_In_Part (parts_group, x, y, ref, false, true);

    if (current.Valid()) {
      for (protos.Start(); !protos.Last(); protos.Next()) {
	Am_Object proto = protos.Get();
	//	cout << "found" << current << endl;
	if (current.Is_Instance_Of(proto)) found = current;
      }
    }
  }
  if (found.Valid()) {
    //   cout << "returning" << found << endl;
    return found;
  }

  return Am_No_Object;
}

// name list utilities
// -------------------
bool Member_Name(Am_Object obj, Am_Slot_Key list_key, char* name)
{
  Am_Value_List name_list = obj.Get(list_key);
  name_list.Start();
  bool result = name_list.Member(name);
  name_list.Start();
  return result;
}

Am_String Next_Name(Am_Object obj, Am_Slot_Key list_key, char* base)
// base $B$r(Bprefix$B$H$7$?%W%m%;%9Cf$G%f%K!<%/$JL>A0$rJV$9!%(B
{
  char name[1000];
  ostrstream oname(name, 1000);
  Am_Value_List name_list = obj.Get(list_key);
  name_list.Start();
  if (!name_list.Member(base)) {
    oname << base << '\0';
  } else {
    int i = 0; do {
      i = i + 1;
      oname.seekp(0, ios::beg);
      oname << base << i << '\0';
    } while (name_list.Start(), name_list.Member(name));
  }
  name_list.Start();
  return (Am_String)name;
}

/*
void Add_Name(Am_Object obj, Am_Slot_Key list_key, char* name)
{
  obj.Make_Unique(list_key);
  Am_Value_List name_list = obj.Get(list_key);
  if (name_list.Empty()) {
    obj.Set(list_key, name_list.Add(name));
  } else {
    name_list.Add(name, Am_TAIL, false);
    obj.Note_Changed(list_key);
  }
}

void Remove_Name(Am_Object obj, Am_Slot_Key list_key, char* name)
{
  obj.Make_Unique(list_key);
  Am_Value_List name_list = obj.Get(list_key);
  name_list.Start();
  if (name_list.Member(name)) {
    name_list.Delete(false);
    obj.Note_Changed(list_key);
  } else {
    cerr << "No " << name << " in " << obj << endl;
  }
  name_list.Start();
}
*/ 

// $BL>A0$+$i%]!<%H$rC5$9!#(B
// ---------------------
Am_Object Proc_Get_Port(Am_Object process, char* name)
{
  Am_Value_List ports = process.Get(PortList);
  for (ports.Start(); !ports.Last(); ports.Next()) {
    Am_Object pt = ports.Get();
    Am_String port_name = (Am_String)pt.Get_Object(Model).Get(Name);
    if (port_name == name) {
      return pt;
    }
  }
  return 0;
}

// Model$B$N%9%m%C%H$NJ8;zNs$,0lCW$9$k%*%V%8%'%/%H$rC5$9!#(B
Am_Object Object_List_Get_By_Model_Key(Am_Value_List list, Am_Slot_Key key, Am_String s)
{
  for (list.Start(); !list.Last(); list.Next()) {
    Am_Object item = (Am_Object)list.Get();
    Am_Object model = item.Get_Object(Model);
    if (model.Valid()) {
      if ((Am_String)model.Get(key) == s) {
	return item;
      }
    }
  }
  return 0;
}  

// $B?F%k!<%k$^$?$O%Q%?!<%s$rC5$9!#(B
bool Is_Network(Am_Object obj)
{
  return obj.Is_Instance_Of(RuleProto) ||
         obj.Is_Instance_Of(PatternProto) ||
         obj.Is_Instance_Of(ReplPatternProto);
}

Am_Object Parent_Rule(Am_Object obj)
{
  Am_Object parent = obj;
  if (parent.Get_Slot_Type(Parent) == Am_NONE) return 0;
  while((parent = parent.Get(Parent)).Valid()) {
    if (Is_Network(parent)) return parent;
    if (parent.Get_Slot_Type(Parent) == Am_NONE) return 0;
  }
  return 0;
}

Am_Object Parent_Rule_In_Form(Am_Constraint_Context& cc, Am_Object& obj)
{
  Am_Object parent = obj;
  if (parent.Get_Slot_Type(Parent) == Am_NONE ||
      parent.Get_Slot_Type(Parent) == Am_UNINIT) return 0;
  while((parent.GV(Parent)).Valid()) {
    //    cout << parent << parent.GV(Parent) << endl;
    parent = parent.GV(Parent);
    if (Is_Network(parent)) return parent;
    if (parent.Get_Slot_Type(Parent) == Am_NONE ||
	parent.Get_Slot_Type(Parent) == Am_UNINIT) return 0;
  }
  return 0;
}

Am_Object Top_Pattern(Am_Object obj)
{
  Am_Object pivot = obj;
  Am_Object parent = obj.Get(Parent);
  while(parent.Valid()) {
    if (pivot.Is_Instance_Of(PatternProto) && !parent.Is_Instance_Of(PatternProto)) return pivot;
    pivot = parent;
    parent = parent.Get(Parent);
  }
  if (pivot.Is_Instance_Of(PatternProto)) return pivot;
  return 0;
}

// Editable Text $B$+$i%$%s%?%i%/%?!<$r>C5n$9$k!#(B
void Remove_Text_Edit_Interactor(Am_Object text_obj)
{
  Am_Object inter = text_obj.Get_Part(TextPart).Get_Part(Am_INTERACTOR);
  inter.Remove_From_Owner();
  inter.Destroy();
}

// $B%*%V%8%'%/%H$NL>A0$NJQ99$rIT2D$K$9$k!#(B
void Disable_Name_Edit(Am_Object obj)
{
  Am_Object name = obj.Get_Part(NamePart);
  Remove_Text_Edit_Interactor(name);
}

// Menu$B$NJQ99!#(B
void Change_Menu(Am_Object obj, Am_Object popper)
{
  Am_Object old_popper = obj.Get_Part(MenuPopper);
  obj.Remove_Part(MenuPopper);
  old_popper.Destroy();
  obj.Add_Part(MenuPopper, popper);
}

// ------------------------------------------------------------
// Methods
// ------------------------------------------------------------

Am_Object DropPointFeedback = 0;

Am_Define_Method_Type(Copy_Method, Am_Object, (Am_Object));
Am_Define_Method_Type_Impl(Copy_Method);

Am_Define_Method_Type(Bin_Check_Method, bool, (Am_Object, Am_Object));
Am_Define_Method_Type_Impl(Bin_Check_Method);

Am_Define_Method_Type(Drop_Method, bool, (Am_Object, Am_Object, Am_Object));
Am_Define_Method_Type_Impl(Drop_Method);

Am_Define_Method_Type(Naming_Method, bool, (Am_Object, Am_String));
Am_Define_Method_Type_Impl(Naming_Method);


void set_commands_for_move(Am_Object widget, Am_Object inter,
			   Am_Value object_modified, Am_Value old_value,
			   Am_Value new_value, int final_left, int final_top) {
  Am_Object inter_command, widget_command;
  inter_command = inter.Get_Part(Am_COMMAND);
  if (inter_command.Valid()) {
    inter_command.Set(Am_OLD_VALUE, old_value);
    inter_command.Set(Am_VALUE, new_value);
    inter_command.Set(Am_OBJECT_MODIFIED, object_modified);
    inter_command.Set(Am_LEFT, final_left);
    inter_command.Set(Am_TOP, final_top);
  }
  widget_command = widget.Get_Part(Am_MOVE_GROW_COMMAND);
  if (widget_command.Valid()) {
    widget_command.Set(Am_OLD_VALUE, old_value);
    widget_command.Set(Am_VALUE, new_value);
    widget_command.Set(Am_OBJECT_MODIFIED, object_modified);
    widget_command.Set(Am_GROWING, false);
  }
}

Am_Define_Method(Am_Current_Location_Method, void, Drag_And_Drop_Do,
		 (Am_Object inter, Am_Object object_modified,
		  Am_Inter_Location data)) {
  Am_Object window = inter.Get(Am_WINDOW);
  int xoff = inter.Get(Am_X_OFFSET);
  int yoff = inter.Get(Am_Y_OFFSET);

  bool as_line;
  Am_Object ref;
  int x, y, w, h;
  data.Get_Location(as_line, ref, x, y, w, h);
  //  cout << x << ", " << y << ", " << w << ", " << h << ", " << ref << endl;
  Am_Translate_Coordinates(ref, x, y, window, x, y);
  //  cout << x << ", " << y << endl;

  Am_Value_List droppts = object_modified.Get(DropPoints);
  Am_Object target = Content_Instance_Of(window, x + xoff, y + yoff, window, droppts);
  bool success = false;
  if (target.Valid()) {
    Am_Object feedback;
    feedback = inter.Get(Am_FEEDBACK_OBJECT);
    if (feedback.Valid ()) 
      feedback.Set(Am_VISIBLE, false);
    else if (object_modified.Valid ()) {
      bool growing = inter.Get(Am_GROWING);
      Am_Modify_Object_Pos(object_modified, data, growing, inter);
    }
    DropPointFeedback.Set(Am_VISIBLE, false);

    Am_Object widget = inter.Get_Owner();
    Am_Value objs_value = (Am_Value)object_modified;
    Am_Value old_values_value = inter.Get(Am_OLD_VALUE);
    Am_Value new_values_value = (Am_Value)data;
    int final_left = object_modified.Get(Am_LEFT);
    int final_top = object_modified.Get(Am_TOP);
    set_commands_for_move(widget, inter, objs_value, old_values_value,
			  new_values_value, final_left, final_top);

    // drop $B$9$k%*%V%8%'%/%H$N(BCheckDropMethod$B$G%A%'%C%/$7!"(B
    // drop $B$9$k%*%V%8%'%/%H$N(BDropMethod$B$r<B9T$9$k!#(B
    Bin_Check_Method check_method = object_modified.Get(CheckDropMethod);
    Drop_Method drop_method = object_modified.Get(DropMethod);
    if (check_method.Valid() && drop_method.Valid()) {
      success = check_method.Call(object_modified, target);
      if (success) {
	drop_method.Call(object_modified, target, inter.Get(Am_COMMAND));
      }
    }
  }
  if (!success) {
    Am_Current_Location_Method orig_do = Am_Selection_Widget
      .Get_Part(Am_MOVE_INTERACTOR).Get(Am_DO_METHOD);
    orig_do.Call(inter, object_modified, data);
  }
}


/*
Am_Define_Method(Am_Object_Method, void, DoDragAndDrop, (Am_Object command_obj))
{
  Am_Object inter = command_obj.Get_Owner();
  Am_Object window = inter.Get(Am_WINDOW);
  int xoff = inter.Get(Am_X_OFFSET);
  int yoff = inter.Get(Am_Y_OFFSET);

  Am_Inter_Location final_loc = command_obj.Get(Am_VALUE);
  bool as_line;
  Am_Object ref;
  int x, y, w, h;
  final_loc.Get_Location(as_line, ref, x, y, w, h);
  cout << x << ", " << y << ", " << w << ", " << h << ", " << ref << endl;
  Am_Translate_Coordinates(ref, x, y, window, x, y);
  cout << x << ", " << y << endl;

  Am_Object source = command_obj.Get(Am_OBJECT_MODIFIED);
  Am_Value_List droppts = source.Get(DropPoints);
  Am_Object target = Content_Instance_Of(window, x + xoff, y + yoff, window, droppts);
  if (target.Valid()) {
    Drop_Method drop_method = source.Get(DropMethod);
    if (drop_method.Valid()) drop_method.Call(source, target, command_obj);
  }
}
*/

Am_Define_Method(Am_Object_Method, void, Drag_Interim_Do, (Am_Object command_obj))
{
  Am_Object inter = command_obj.Get_Owner();
  Am_Object window = inter.Get(Am_WINDOW);
  int xoff = inter.Get(Am_X_OFFSET);
  int yoff = inter.Get(Am_Y_OFFSET);

  Am_Inter_Location interim_loc = inter.Get(Am_INTERIM_VALUE);
  bool as_line;
  Am_Object ref;
  int x, y, w, h;
  interim_loc.Get_Location(as_line, ref, x, y, w, h);
  Am_Translate_Coordinates(ref, x, y, window, x, y);

  Am_Object source = inter.Get(Am_OBJECT_MODIFIED);
  Am_Value_List droppts = source.Get(DropPoints);
  Bin_Check_Method check_method = source.Get(CheckDropMethod);
  if (!check_method.Valid()) return;
  Am_Object target = Content_Instance_Of(window, x + xoff, y + yoff, window, droppts);

  if (!DropPointFeedback.Get_Owner().Valid())
    window.Add_Part(DropPointFeedback);
  if (target.Valid() && check_method.Call(source, target)) {
    Am_Object owner = target.Get_Owner();
    int left = target.Get(Am_LEFT);
    int top = target.Get(Am_TOP);
    int width = target.Get(Am_WIDTH);
    int height = target.Get(Am_HEIGHT);
    Am_Translate_Coordinates(owner, left, top, window, left, top);
    DropPointFeedback.Set(Am_VISIBLE, true)
      .Set(Am_LEFT, left)
      .Set(Am_TOP, top)
      .Set(Am_WIDTH, width)
      .Set(Am_HEIGHT, height);
  } else {
    DropPointFeedback.Set(Am_VISIBLE, false);
  }
}

Am_Define_Method(Am_Object_Method, void, Name_Edit_Do, (Am_Object command_obj))
{
  Am_Object text_obj = command_obj.Get_Owner().Get_Owner();
  Am_Object owner = text_obj.Get_Owner().Get_Owner();
  if (owner.Get_Slot_Type(Model) != Am_NONE) {
    Am_String new_name = command_obj.Get(Am_VALUE);
    Naming_Method m = owner.Get(NamingMethod);
    if (m.Call(owner, new_name)) {
      owner.Get_Object(Model).Set(Name, new_name);
    } else {
      Am_Abort_Interactor(command_obj.Get_Owner());
    }
  }
}

Am_Define_Method(Am_Object_Method, void, New_Object_Do, (Am_Object command_obj))
{
  Am_Object menu = command_obj.Get_Owner().Get_Owner(); // $B%a%K%e!<(B

  Am_Object target = menu.Get(TargetObj); // $BDI2C@h(B
  Am_Object group = target.Get_Part(ContentsPart);
  int x = menu.Get(XinTarget);  // $B:BI8(B
  int y = menu.Get(YinTarget);
  Am_Translate_Coordinates(target, x, y, group, x, y);

  Am_Object obj = command_obj.Get(Am_LABEL); // $BDI2C$9$k%*%V%8%'%/%H(B
  Am_Object new_obj = ((Copy_Method)obj.Get(NewMethod)).Call(obj);
  new_obj
    .Set(Am_LEFT, x - (int)new_obj.Get(Am_WIDTH) / 2) // $B%+!<%=%k$N0LCV$rCf?4$K(B
    .Set(Am_TOP, y - (int)new_obj.Get(Am_HEIGHT) / 2)
    ;
  if (!((Bin_Check_Method)target.Get(AddMethod)).Call(target, new_obj))
    new_obj.Destroy();
}

Am_Define_Method(Am_Object_Method, void, Delete_Object_Do, (Am_Object command_obj))
{
  Am_Object menu = command_obj.Get_Owner().Get_Owner();
  Am_Object target = menu.Get(TargetObj);
  Am_Object parent = target.Get(Parent);
  ((Bin_Check_Method)parent.Get(RemoveMethod)).Call(parent, target);
}


// ------------------------------------------------------------
// Formulas
// ------------------------------------------------------------

// $B%]!<%H$N$_;D$7$F;D$j$r>C$9$?$a$N(BVisible 
// ---------------------------------------
Am_Define_Formula(bool, ContentsVisibleForm)
{
  Am_Object owner = self.GV_Owner();
  if (!owner.Valid()) return true;
  int w = owner.GV(Am_WIDTH);
  int h = owner.GV(Am_HEIGHT);
  if (w > 100 || h > 100) return true;
  int rw = Get_Required_Width(owner);
  int rh = Get_Required_Height(owner);
  if (w * 2 < rw || h * 2 < rh) return false;
  return true;
}

Am_Define_Formula(bool, SelfVisibleForm)
{
  if ((int)self.GV(Am_WIDTH) <= 100 ||
      (int)self.GV(Am_HEIGHT) <= 100)
    return false;
  return true;
}

// Povl$B$N3,AX$K$*$1$k?F%*%V%8%'%/%H(B
// --------------------------------
Am_Define_Object_Formula(ParentForm)
{
  Am_Object owner_contents = self.GV_Owner();
  if (!owner_contents.Valid()) return 0;
  Am_Object owner = owner_contents.GV_Owner();
  if (!owner.Valid()) return 0;
  return owner;
}

// zoom$B$N%W%m%Q%2!<%7%g%sMQ(B
// ------------------------
Am_Define_Object_Formula(PropagateForm)
{
  Am_Object owner = self.GV_Owner();
  if (!owner.Valid()) return 0;
  return owner;
}

Am_Define_Formula(int, NumberThresholdForm)
{
  //  return (int)self.GV_Object(propagateParent).GV(numberThreshold);
  Am_Object parent = self.GV(propagateParent);
  if (!parent.Valid()) return 0;
  //  if (parent.Get_Slot_Type(numberThreshold) == Am_NONE) return 50;
  //  if (parent.Get_Slot_Type(numberThreshold) == Am_UNINIT) return 50;
  //  if (!parent.GV(number_threshold).Valid()) return number_threshold;
  return (int)parent.GV(numberThreshold);
}


// NamePart$B$N9b$5(B
// --------------
Am_Define_Formula(int, NameHeightForm)
{
  Am_Object owner = self.GV_Owner();
  int max = self.GV(NameHeightMax);
  if (owner.Valid()) {
    max = imin(max, (int)owner.GV(Am_HEIGHT));
    Am_Object gowner = owner.GV_Owner();
    if (gowner.Valid()) {
      max = imin(max, (int)gowner.GV(Am_HEIGHT) - (int)owner.GV(Am_HEIGHT));
    }
  }
  return max;
}

// $B%a%K%e!<$N%?%$%H%kMQ(B
// --------------------
Am_Define_String_Formula(TargetNameForm)
{
  Am_Object item = self.GV_Owner();  if (!item.Valid()) return (Am_String)"";
  Am_Object menu = item.GV_Owner();  if (!menu.Valid()) return (Am_String)"";
  Am_Object target = menu.GV_Object(TargetObj);  if (!target.Valid()) return (Am_String)"";
  if (!target.Get_Prototype().Valid()) return (Am_String)"";
  if (target.GV_Part(NamePart).Valid()) {
    Am_Object text = target.GV_Part(NamePart).GV_Part(TextPart);
    if (text.Valid()) return text.GV(Am_TEXT);
  }
  return (Am_String)"";
}

// Model$B$N(BName$B$r;2>H$9$k%F%-%9%H%*%V%8%'%/%HMQ(B
// -------------------------------------------
Am_Define_String_Formula(ModelNameForm)
{
  Am_Object text = self.GV_Owner();  if (!text.Valid()) return (Am_String)"";
  Am_Object port = text.GV_Owner();  if (!port.Valid()) return (Am_String)"";
  Am_Object model = port.GV(Model);  if (!model.Valid()) return (Am_String)"";
  return model.GV(Name);
}

// contents part list
// ------------------
Am_Define_Value_List_Formula(ContentsListForm)
{
  return (Am_Value_List)self.GV_Part(ContentsPart).GV(Am_GRAPHICAL_PARTS);
}

// Instance List
// -------------
static void instance_list_procedure (Am_Constraint_Context& cc, Am_Object& self, Am_Value& value) {
  Am_Object* proto = (Am_Object*)cc.Get_Data ();

  Am_Value_List list = Am_Value_List();
  Am_Value_List parts = ContentsListForm(cc, self);
  for (parts.Start(); !parts.Last(); parts.Next()) {
    Am_Object p = parts.Get();
    if (p.Is_Instance_Of(*proto)) list.Add(p);
  }
  value = list;
}

Am_Constraint* InstanceListForm (Am_Object& proto)
{
  Am_Formula_Advanced* formula = (Am_Formula_Advanced*)(Am_Constraint*)
      Am_Formula (instance_list_procedure, "InstanceListForm");
  formula->Set_Data (&proto, 0);
  return formula;
}


// Name List
// ---------
static void name_list_procedure (Am_Constraint_Context& cc, Am_Object& self, Am_Value& value) {
  Am_Slot_Key list_key = (Am_Slot_Key)cc.Get_Data ();

  Am_Value_List names = Am_Value_List();
  Am_Value_List parts = self.GV(list_key);
  for (parts.Start(); !parts.Last(); parts.Next()) {
    Am_Object p = parts.Get();
    names.Add((Am_String)p.GV_Object(Model).GV(Name));
  }
  value = names;
}

Am_Constraint* NameListForm (Am_Slot_Key list_key)
{
  Am_Formula_Advanced* formula = (Am_Formula_Advanced*)(Am_Constraint*)
      Am_Formula (name_list_procedure, "NameListForm");
  formula->Set_Data ((void*)list_key, 0);
  return formula;
}

// Sequence Layout
// ---------------
/*
Am_Define_Formula(int, SequenceLayoutForm)
{
  int totalw = 0;
  int totalh = 0;
  int maxw = 0;
  int maxh = 0;

  Am_Value_List parts = self.GV(Am_GRAPHICAL_PARTS);
  int length = parts.Length();
  if (length <= 0) return 1;

  for (parts.Start(); !parts.Last(); parts.Next()) {
    Am_Object p = parts.Get();
    int w = p.GV(Am_WIDTH);
    int h = p.GV(Am_HEIGHT);
    totalw += w;
    totalh += h;
    maxw = imax(maxw, w);
    maxh = imax(maxh, h);
  }
  int dir = self.GV(Dir);
  if (dir == LeftRight || dir == RightLeft) {
    space = self.GV(Am_H_SPACING);
    self.Set(Am_WIDTH, totalw + space * (length - 1))
      .Set(Am_HEIGHT, maxh);
    inc_left = true;
  } else {
    space = self.GV(Am_V_SPACING);
    self.Set(Am_WIDTH, maxw)
      .Set(Am_HEIGHT, totalh += space * (length - 1));
    inc_left = false;
  }
  int left = 0;
  int top = 0;
  bool forward = true;
  if (dir == LeftRight || dir == TopBottom) {parts.Start(); forward = true;}
  if (dir == RightLeft || dir == BottomTop) {parts.End(); forward = false;}
  for ( ; !parts.Last() && !parts.First ; ) {
    Am_Object pt = parts.Get();
    int w = pt.Get(Am_WIDTH);
    int h = pt.Get(Am_HEIGHT);
    if (inc_left) top = (maxh - h) / 2;
    else          left = (maxw - w) / 2;
    pt.Set(Am_LEFT, left)
      .Set(Am_TOP,  top)
      ;
    if (inc_left) left += w + space; else top += h + space; 
    if (forward) parts.Next(); else parts.Prev();
  }
}



Am_Define_Formula(int, SequenceLayoutForm)
{
  int width = self.GV(Am_WIDTH);
  int height = self.GV(Am_HEIGHT);

  Am_Value_List parts = self.GV(Am_GRAPHICAL_PARTS);
  int length = parts.Length();
  if (length <= 0) return 1;

  int dir = self.GV(Dir);
  int space = 0;
  int w = width;
  int h = height;
  bool inc_left = true;
  if (dir == LeftRight || dir == RightLeft) {
    space = self.GV(Am_H_SPACING);
    width = width - space * (length - 1);
    w = width / length;
    inc_left = true;
  } else {
    space = self.GV(Am_V_SPACING);
    height = height - space * (length - 1);
    h = height / length;
    inc_left = false;
  }
  int left = 0;
  int top = 0;
  bool forward = true;
  if (dir == LeftRight || dir == TopBottom) {parts.Start(); forward = true;}
  if (dir == RightLeft || dir == BottomTop) {parts.End(); forward = false;}
  for ( ; !parts.Last() && !parts.First ; ) {
    Am_Object pt = parts.Get();
    pt.Set(Am_LEFT, left)
      .Set(Am_TOP,  top)
      .Set(Am_WIDTH,  w)
      .Set(Am_HEIGHT, h)
      ;
    if (inc_left) left += w + space; else top += h + space; 
    if (forward) parts.Next(); else parts.Prev();
  }
}
*/

// ------------------------------------------------------------
// Demons
// ------------------------------------------------------------

void povl_copy_demon(Am_Object self)
{
  Am_Object_Advanced proto_adv = (Am_Object_Advanced&)Am_Group;
  Am_Object_Demon* proto_create = proto_adv.Get_Demons().Get_Object_Demon(Am_COPY_OBJ);
  if (proto_create) proto_create(self);

  if ((bool)self.Get(ToCopyModel) == true && self.Get_Slot_Type(Model) != Am_NONE) {
    Am_Object copied_model = self.Get_Object(Model).Copy();
    self.Set(Model, copied_model);
  }
}

void sequence_povl_copy_demon(Am_Object self)
{
  Am_Object_Advanced proto_adv = (Am_Object_Advanced&)SequenceAlignZoomGroup;
  Am_Object_Demon* proto_create = proto_adv.Get_Demons().Get_Object_Demon(Am_COPY_OBJ);
  if (proto_create) proto_create(self);

  if ((bool)self.Get(ToCopyModel) == true && self.Get_Slot_Type(Model) != Am_NONE) {
    Am_Object copied_model = self.Get_Object(Model).Copy();
    self.Set(Model, copied_model);
  }
}


// ------------------------------------------------------------
// Widgets 
// ------------------------------------------------------------

Am_Object PovlSelectionWidget = 0;

Am_Object EditableZoomingText = 0;

Am_Object NewObjectCmd = 0;
Am_Object RemoveObjectCmd = 0;

Am_Object PovlGroup = 0;
Am_Object SequencePovlGroup = 0;
Am_Object ValueProto = 0;

void InitializeWidgets (void)
{
  PovlSelectionWidget = AlignZoomSelectionWidget.Create()
    .Get_Part(Am_MOVE_INTERACTOR)
      .Set(Am_DO_METHOD, Drag_And_Drop_Do)
      .Get_Part(Am_COMMAND)
        .Set(Am_INTERIM_DO_METHOD, Drag_Interim_Do)
      .Get_Owner()
    .Get_Owner()
    ;

  DropPointFeedback = Am_Rectangle.Create()
    .Set(Am_LINE_STYLE, Am_No_Style)
    .Set(Am_FILL_STYLE, Am_Light_Gray_Stipple)
    ;

  EditableZoomingText = ZoomingText.Create()
    .Get_Part(TextPart)
      .Set_Single_Constraint_Mode(Am_TEXT, false)
      .Add_Part(Am_INTERACTOR, Am_Text_Edit_Interactor.Create()
		.Set(Am_START_WHEN, "double_left_down")
		.Set(Am_PRIORITY, 3)
		.Get_Part(Am_COMMAND)
		  .Set(Am_DO_METHOD, Name_Edit_Do)
		.Get_Owner()
		)
    .Get_Owner()
    ;

  NewObjectCmd = MenuCommand.Create()
    .Set(Am_DO_METHOD, New_Object_Do)
    ;

  RemoveObjectCmd = MenuCommand.Create()
    .Set(Am_DO_METHOD, Delete_Object_Do)
    ;

  PovlGroup = Am_Group.Create("PovlGroup")
    .Set(ToCopyModel, true)
    .Set(numberThreshold, 30)
    .Set(Am_ACTIVE, true)
    .Set_Inherit_Rule(Am_LEFT, Am_COPY)
    .Set_Inherit_Rule(Am_TOP, Am_COPY)
    .Set_Inherit_Rule(Am_WIDTH, Am_COPY)
    .Set_Inherit_Rule(Am_HEIGHT, Am_COPY)
    ;
  Am_Object_Advanced obj_adv = (Am_Object_Advanced&)PovlGroup;
  Am_Demon_Set obj_demon_set(obj_adv.Get_Demons().Copy());
  obj_demon_set.Set_Object_Demon(Am_COPY_OBJ, povl_copy_demon);
  obj_adv.Set_Demons(obj_demon_set);

  SequencePovlGroup = SequenceAlignZoomGroup.Create("SequencePovlGroup")
    .Set(ToCopyModel, true)
    .Set_Inherit_Rule(Am_LEFT, Am_COPY)
    .Set_Inherit_Rule(Am_TOP, Am_COPY)
    .Set_Inherit_Rule(Am_WIDTH, Am_COPY)
    .Set_Inherit_Rule(Am_HEIGHT, Am_COPY)
    ;
  obj_adv = (Am_Object_Advanced&)SequencePovlGroup;
  Am_Demon_Set obj_demon_set2(obj_adv.Get_Demons().Copy());
  obj_demon_set2.Set_Object_Demon(Am_COPY_OBJ, sequence_povl_copy_demon);
  obj_adv.Set_Demons(obj_demon_set2);

  ValueProto = PovlGroup.Create("ValueProto");
}
