// Copyright (C) 1997  Toyoda Masashi (toyoda@is.titech.ac.jp)

#include <fstream.h>
#include <strstream.h>

#include <amulet.h>
#include FORMULA_ADVANCED__H
#include INTER_ADVANCED__H
#include WIDGETS_ADVANCED__H

#include "ToyWidgets.h"
#include "PovlWidgets.h"
#include "Resources.h"
#include "Module.h"
#include "Process.h"
#include "Rule.h"
#include "Pattern.h"
#include "Atom.h"
#include "Message.h"
#include "Guard.h"
#include "Port.h"
#include "Binder.h"

// ---- Tags ----
// a b c d e  g h i  l m n o p q r s t v x

#define Beg_Module 'M'
#define End_Module 'm'

#define Beg_Procs  'P'
#define End_Procs  'p'
#define Beg_Proc   'C'
#define End_Proc   'c'
#define Beg_Patts  'X'
#define End_Patts  'x'
#define Beg_Rules  'R'
#define End_Rules  'r'
#define Beg_Rule   'L'
#define End_Rule   'l'
#define Beg_Patt   'E'
#define End_Patt   'e'
#define Beg_Goals  'O'
#define End_Goals  'o'
#define Beg_Goal   'N'
#define End_Goal   'n'
#define Beg_Holes  'H'
#define End_Holes  'h'
#define Beg_Hole   'I'
#define End_Hole   'i'
#define Beg_Gurds  'G'
#define End_Gurds  'g'
#define Beg_Gurd   'F'
#define End_Gurd   'f'
#define Beg_Bndrs  'B'
#define End_Bndrs  'b'
#define Beg_Bndr   'D'
#define End_Bndr   'd'


#define Beg_Ports  'Q'
#define End_Ports  'q'
#define Beg_Port   'T'
#define End_Port   't'
#define Beg_Vals   'V'
#define End_Vals   'v'
#define Beg_Stru   'S'
#define End_Stru   's'
#define Beg_Atom   'A'
#define End_Atom   'a'

// ---- Global ----
Am_Object Process_table[1024];

// ---- Utility ----
Am_Object New_obj(Am_Object proto)
{
  Copy_Method new_method = proto.Get(NewMethod);
  if (new_method.Valid()) 
    return new_method.Call(proto);
  else
    return 0;
}

void Add_obj(Am_Object dest, Am_Object src)
{
  Bin_Check_Method add_method = dest.Get(AddMethod);
  if (add_method.Valid())
    add_method.Call(dest, src);
}

int Save_position(Am_Object obj, ofstream& ofile)
{
  ofile << " " << origX(obj) << " " << origY(obj) << " "
	<< origW(obj) << " " << origH(obj) << " ";
  return 1;
}

int Load_position(Am_Object obj, ifstream& ifile)
{
  int left, top, height, width;
  ifile >> left >> top >> width >> height;
  obj.Get(Am_LEFT); // demon$B$r8F$S=P$9$?$a!)!)!)(B
  obj.Set(Am_LEFT, left)
    .Set(Am_TOP, top)
    .Set(Am_WIDTH, width)
    .Set(Am_HEIGHT, height)
    ;
  return 1;
}

void Set_zooming(Am_Object obj, bool active)
{
  Am_Do_Events();
  obj.Get_Part(ContentsPart).Set(Am_ACTIVE, active);
}

// -------- Error --------
void Err(char *string, char* name)
{
  Am_Show_Alert_Dialog(Am_Value_List().Add("Error:").Add(string).Add(name));
  //  cerr << "Error: " << string << endl;
}

// -----------------------------------
int Save_module(Am_Object Window, ofstream& ofile);
int Load_module(Am_Object window, ifstream& ifile);
int   Save_proc_part(Am_Value_List procs, ofstream& ofile);
int   Load_proc_part(Am_Object module, ifstream& ifile);
int     Save_process(Am_Object process, ofstream& ofile);
int     Load_process(Am_Object proc_agg, ifstream& ifile);
int   Save_patterns_part(Am_Value_List patterns, ofstream& ofile);
int   Load_patterns_part(Am_Object module, ifstream& ifile);
int   Save_rules_part(Am_Value_List rules, ofstream& ofile);
int   Load_rules_part(Am_Object module, ifstream& ifile);
int     Save_rule(Am_Object rule, ofstream& ofile);
int     Load_rule(Am_Object module, ifstream& ifile);
int     Load_really_rule(Am_Object module, ifstream& ifile);
int       Save_goals_part(Am_Object rule, ofstream& ofile);
int       Load_goals_part(Am_Object rule, ifstream& ifile);
int         Save_goal(Am_Object goal, ofstream& ofile);
int         Load_goal(Am_Object rule, ifstream& ifile);
int         Load_really_goal(Am_Object rule, ifstream& ifile);
int         Save_binder_part(Am_Object rule, ofstream& ofile);
int       Load_binder_part(Am_Object rule, ifstream& ifile);
int       Save_binder(Am_Object rule, Am_Object binder, ofstream& ofile);
int       Load_binder(Am_Object rule, ifstream& ifile);
int       Save_guard_part(Am_Object rule, ofstream& ofile);
int       Load_guard_part(Am_Object rule, ifstream& ifile);
int       Save_guard(Am_Object guard, ofstream& ofile);
int       Load_guard(Am_Object rule, ifstream& ifile);
int     Save_pattern(Am_Object pattern, ofstream& ofile);
int     Load_pattern(Am_Object patt_agg, ifstream& ifile);
int       Save_holes_part(Am_Value_List holes, ofstream& ofile);
int       Load_holes_part(Am_Object patt, ifstream& ifile);
int       Save_hole(Am_Object hole, ofstream& ofile);
int       Load_hole(Am_Object patt, ifstream& ifile);

int Save_port_part(Am_Value_List ports, ofstream& ofile);
int Load_port_part(Am_Object proc_agg, ifstream& ifile);
int Load_exist_port_part(Am_Object process, ifstream& ifile);
int Save_port(Am_Object port, ofstream& ofile);
int Load_port(Am_Object port, ifstream& ifile);
int Load_exist_port(Am_Object port, ifstream& ifile);

int Save_value_part(Am_Value_List values, ofstream& ofile);
int Load_value_part(Am_Object port, ifstream& ifile);
int Save_value(Am_Object value, ofstream& ofile);
int Load_value(Am_Object port, ifstream& ifile);

int Save_atom(Am_Object value, ofstream& ofile);
int Load_atom(Am_Object port, ifstream& ifile);
int Save_structure(Am_Object value, ofstream& ofile);
int Load_structure(Am_Object port, ifstream& ifile);

// -------- Module --------
// Beg_Module Name
// ... processes ...
// ... rules ...
// End_Module

int Save_module(Am_Object module)
{
  char name[1000];
  ostrstream oname(name, 1000);

  oname << module.MGet(Name) << ".klg" << ends;
  ofstream ofile;
  ofile.open(name, ios::out);
  if (!ofile) {
    cerr << "can't save to file: " << name << endl;
    return 0;
  }
  ofile << Beg_Module << " " << module.MGet(Name) << endl;
  Save_proc_part(module.Get(ProcessList), ofile); 
  Save_patterns_part(module.Get(PatternList), ofile);
  Save_rules_part(module.Get(RuleList), ofile); 
  ofile << End_Module << endl;
  return 1;
}

int Load_module(Am_Object module, ifstream& ifile)
{
  char tag;
  char name[1000];

  ifile >> tag; if (tag != Beg_Module) {Err("Beg_Module", ""); return 0;}
  ifile >> name;
  if (!CALL2(module, Naming_Method, NamingMethod, name)) {
    Err(name, "Already exists.");
    // $B$G$b%3%T!<$H$+$5$l$F$k$HBgJQ$J$s$@$h$J!#%3%T!<$H$O$J$s$@!)(B
    return 0;
  }
  Set_zooming(module, false);
  Load_proc_part (module, ifile);
  Load_patterns_part(module, ifile);
  Load_rules_part(module, ifile);
  Set_zooming(module, true);
  ifile >> tag; if (tag != End_Module) {Err("End_Module", name); return 0;}
}

// -------- Processes --------
// Beg_Procs
// ... process ...
// End_Procs
int Save_proc_part(Am_Value_List procs, ofstream& ofile)
{
  ofile << Beg_Procs << endl;
  int id = 1;
  for (procs.Start(); !procs.Last(); procs.Next(), ++id) {
    Am_Object proc = procs.Get();
    Save_process(proc.Set(Id, id), ofile);
  }
  ofile << End_Procs << endl;
}

int Load_proc_part(Am_Object module, ifstream& ifile)
{
  Am_String mname = module.MGet(Name);
  char tag;
  ifile >> tag; if (tag != Beg_Procs) {Err("Beg_Procs", mname); return 0;}
  while (Load_process(module, ifile));
  ifile >> tag; if (tag != End_Procs) {Err("End_Procs", mname); return 0;}
}

// -------- Process --------
// Beg_Proc Id Name position_info
// ... ports ... 
// End_Proc
int Save_process(Am_Object process, ofstream& ofile)
{
  int id         = process.Get(Id);
  Am_String name = process.MGet(Name);

  ofile << Beg_Proc << " ";
  ofile << id << " " << name;
  Save_position(process, ofile);
  ofile << endl;
  Save_port_part(process.Get(PortList), ofile);
  ofile << End_Proc << endl;
}

int Load_process(Am_Object module, ifstream& ifile)
{
  char tag;
  int id;
  char name[1000];

  ifile >> tag; if (tag != Beg_Proc) {ifile.putback(tag); return 0;}
  ifile >> id >> name;
  { cerr << "Load process: " << name << " id: " << id << endl; }

  Am_Object new_proc = New_obj(ProcessProto);
  new_proc.MSet(Name, name);
  Load_position(new_proc, ifile);

  Set_zooming(new_proc, false);
  Load_port_part(new_proc, ifile);
  Set_zooming(new_proc, true);

  Process_table[id] = new_proc; // global table $B$KEPO?!%(B
  Add_obj(module, new_proc);

  ifile >> tag; if (tag != End_Proc) {Err("End_Proc", name); return 0;}
  return 1;
}

// -------- Patterns Part --------
// Beg_Patts
// ... patterns ...
// End_Patts
int Save_patterns_part(Am_Value_List patterns, ofstream& ofile)
{
  ofile << Beg_Patts << endl;
  for (patterns.Start(); !patterns.Last(); patterns.Next()) {
    Am_Object patt = patterns.Get();
    Save_pattern(patt, ofile);
  }
  ofile << End_Patts << endl;

}

int Load_patterns_part(Am_Object module, ifstream& ifile)
{
  Am_String mname = module.MGet(Name);
  char tag;
  ifile >> tag; if (tag != Beg_Patts) {Err("Beg_Patts", mname); return 0;}
  while (Load_pattern(module, ifile));
  ifile >> tag; if (tag != End_Patts) {Err("End_Patts", mname); return 0;}
}

// -------- Rules Part --------
// Beg_Rules
// ... rules ...
// End_Rules
int Save_rules_part(Am_Value_List rules, ofstream& ofile)
{
  ofile << Beg_Rules << endl;
  for (rules.Start(); !rules.Last(); rules.Next()) {
    Am_Object rule = rules.Get();
    Save_rule(rule, ofile);
  }
  ofile << End_Rules << endl;
}

int Load_rules_part(Am_Object module, ifstream& ifile)
{
  Am_String mname = module.MGet(Name);
  char tag;
  ifile >> tag; if (tag != Beg_Rules) {Err("Beg_Rules", mname); return 0;}
  while (Load_rule(module, ifile));
  ifile >> tag; if (tag != End_Rules) {Err("End_Rules", mname); return 0;}
}


// -------- Rule --------
// Beg_Rule Proc Type position_info
// ... ports ...
// ... goals ...
// ... binders...
// End_Rule

int Save_rule(Am_Object rule, ofstream& ofile)
{
  ofile << Beg_Rule << " ";
  int proc = 0;
  if (!rule.Is_Instance_Of(MainRuleProto)) proc = rule.Get_Object(Process).Get(Id);
  int type = rule.Get(Type);
  ofile << proc << " " << type;
  Save_position(rule, ofile);
  ofile << endl;
  Save_port_part(rule.Get(PortList), ofile);
  Save_goals_part(rule, ofile);
  Save_binder_part(rule, ofile);
  Save_guard_part(rule, ofile);
  ofile << End_Rule << endl;
}

int Load_rule(Am_Object module, ifstream& ifile)
{
  char tag;
  ifile >> tag; if (tag != Beg_Rule) {ifile.putback(tag); return 0;}
  int proc, type;
  ifile >> proc >> type;
  cerr << "Load rule, Proc id: " << proc << " type: " << type << endl;
  Am_Object rule;
  if   (type == TransitionRule) rule = TransitionRuleProto;
  else if (type == NetworkRule) rule = NetworkRuleProto;
  else return 0;

  Am_Object p = Process_table[proc];
  if (proc != 0) rule = Rule_Create(p, rule);
  else           rule = module.Get_Part(ContentsPart).Get_Part(MainRule);

  Load_position(rule, ifile);

  Set_zooming(rule, false);  
  Load_exist_port_part(rule, ifile);
  Load_goals_part(rule, ifile);
  Load_binder_part(rule, ifile);
  Load_guard_part(rule, ifile);
  Set_zooming(rule, true);  
  if (proc != 0) Add_obj(module, rule);
  ifile >> tag; if (tag != End_Rule) {Err("End_Rule", ""); return 0;}
  return 1;
}

// -------- Goals Part --------
// Beg_Goals
// ... goals ...
// End_Goals

int Save_goals_part(Am_Object rule, ofstream& ofile)
{
  ofile << Beg_Goals << endl;
  Am_Value_List goals = rule.Get(ProcessList);
  for (goals.Start(); !goals.Last(); goals.Next()) {
    Am_Object goal = goals.Get();
    if (goal.Is_Instance_Of(ProcessProto)) Save_goal(goal, ofile);
    if (goal.Is_Instance_Of(PatternProto)) Save_pattern(goal, ofile);
  }
  ofile << End_Goals << endl;
}

int Load_goals_part(Am_Object rule, ifstream& ifile)
{
  Am_String rname = rule.MGet(Name);
  char tag;
  ifile >> tag; if (tag != Beg_Goals) {Err("Beg_Goals", rname); return 0;}
  while (Load_goal(rule, ifile));
  ifile >> tag; if (tag != End_Goals) {Err("End_Goals", rname); return 0;}
}

// -------- Goal --------
// Beg_Goal Id Hole Name ModuleName position_info
// ... ports ...
// End_Goal
int Save_goal(Am_Object goal, ofstream& ofile)
{
  ofile << Beg_Goal << " ";
  int type = goal.Get(Type);
  int id = 0; // InterGoal$B$N$H$-(B0
  Am_String module_name;
  if (type == ProcessGoal) {
    Am_Object proc = goal.Get_Prototype();
    id = proc.Get(Id);
    module_name = "module";
  } else {
    id = 0;
    module_name = goal.MGet(Module); // InterGoal$B$N$H$-$O(Bmodule$BL>$,F~$C$F$k!#(B
  }
  int hole = 0; // Pattern$B$N$H$-$NBP1~!%(B
  Am_Object hl = goal.Get(Hole);
  if (hl.Valid()) {
    hole = Object_List_Index(goal.Get_Object(Parent).Get(HoleList), hl);
  }
  ofile << id << " " << hole << " " << goal.MGet(Name) << " " << module_name;
  Save_position(goal, ofile);
  ofile << endl;
  Save_port_part(goal.Get(PortList), ofile);
  ofile << End_Goal << endl;
}

int Load_goal(Am_Object rule, ifstream& ifile)
{
  char tag;
  ifile >> tag;
  if (tag == Beg_Goal) return Load_really_goal(rule, ifile);
  if (tag == Beg_Patt) {ifile.putback(tag); return Load_pattern(rule, ifile);}
  ifile.putback(tag);
  return 0;
}

int Load_really_goal(Am_Object rule, ifstream& ifile)
{
  char tag;
  // tag check omitted by Load_goal()
  int id, hole;
  char name[1000];
  char mname[1000];
  ifile >> id >> hole >> name >> mname;

  Am_Object proc = Process_table[id];
  Am_Object goal;
  if (id == 0) {
    goal = Process_Inter_Module_Goal_New(name, mname);
    rule.Get_Part(ContentsPart).Add_Part(goal); // fake
    Load_position(goal, ifile);
    Set_zooming(goal, false);
    Load_port_part(goal, ifile);
    Set_zooming(goal, true);
    goal.Remove_From_Owner();
  } else {
    goal = Process_Goal_Create(proc);
    Load_position(goal, ifile);
  }

  if (hole != 0) {
    Am_Object h = Object_List_Get(rule.Get(HoleList), hole);
    Single_Hole_Replace(h, goal);
  } else {
    Add_obj(rule, goal);
  }

  if (id != 0) {
    Set_zooming(goal, false);
    Load_exist_port_part(goal, ifile);
    Set_zooming(goal, true);
  }

  ifile >> tag; if (tag != End_Goal) {Err("End_Goal", name); return 0;}
  return 1;
}

// -------- Binder Part --------
// Beg_Bndrs
// ... binders ...
// End_Bndrs
int Save_binder_part(Am_Object rule, ofstream& ofile)
{
  ofile << Beg_Bndrs << endl;
  Am_Value_List binders = rule.Get_Object(BindersPart).Get(Am_GRAPHICAL_PARTS);
  for (binders.Start(); !binders.Last(); binders.Next()) {
    Am_Object binder = binders.Get();
    Save_binder(rule, binder, ofile);
  }
  ofile << End_Bndrs << endl;
}

int Load_binder_part(Am_Object rule, ifstream& ifile)
{
  char tag;
  ifile >> tag; if (tag != Beg_Bndrs) return 0;
  while (Load_binder(rule, ifile));
  ifile >> tag; if (tag != End_Bndrs) return 0;
}


// -------- Binder --------
// Beg_Bndr
// Pos1.1 Pos1.2 ... -1
// Pos2.1 Pos2.2 ... -1
// End_Bndr

int Save_binder(Am_Object rule, Am_Object binder, ofstream& ofile)
{
  ofile << Beg_Bndr << endl;
  Am_Object pair[2];
  pair[0] = binder.Get(SrcPort);
  pair[1] = binder.Get(DestPort);
  int i;
  for (i = 0; i < 2; ++i) {
    Am_Value_List path = Obj_To_Path(pair[i], rule);
    for (path.Start(); !path.Last(); path.Next()) {
      int pos = path.Get();
      ofile << pos << " ";
    }
    ofile << -1 << endl;
  }
  ofile << End_Bndr << endl;
}

int Load_binder(Am_Object rule, ifstream& ifile)
{
  char tag;
  ifile >> tag; if (tag != Beg_Bndr) {ifile.putback(tag); return 0;}
  int pos;
  Am_Value_List paths[2] = {Am_Value_List(), Am_Value_List()};
  int i;
  for (i = 0; i < 2; ++i) {
    while (ifile >> pos, pos >= 0) {
      paths[i].Add(pos);
    }
  }
  Am_Object src_port = Path_To_Obj(rule, paths[0]);
  Am_Object des_port = Path_To_Obj(rule, paths[1]);
  Am_Object new_line = Single_Binding_Line_Create(src_port, des_port);
  //  cout << "create binder: " << src_port << des_port << new_line << endl;
  ifile >> tag; if (tag != End_Bndr) return 0;
}

  
// -------- Guard Part --------
// Beg_Gurds
// ... gurads ...
// End_Gurds

int Save_guard_part(Am_Object rule, ofstream& ofile)
{
  ofile << Beg_Gurds << endl;
  Am_Value_List guards = rule.Get(GuardList);
  for (guards.Start(); !guards.Last(); guards.Next()) {
    Am_Object guard = guards.Get();
    Save_guard(guard, ofile);
  }
  ofile << End_Gurds << endl;
}

int Load_guard_part(Am_Object rule, ifstream& ifile)
{
  char tag;
  ifile >> tag; if (tag != Beg_Gurds) return 0;
  while (Load_guard(rule, ifile));
  ifile >> tag; if (tag != End_Gurds) return 0;
}

// -------- Guard --------
// Beg_Gurd position_info Contents
// End_Gurd

int Save_guard(Am_Object guard, ofstream& ofile)
{
  ofile << Beg_Gurd << " ";
  Save_position(guard, ofile);
  ofile << guard.MGet(Name) << endl;
  ofile << End_Gurd << endl;
}

int Load_guard(Am_Object rule, ifstream& ifile)
{
  char tag;
  ifile >> tag; if (tag != Beg_Gurd) {ifile.putback(tag); return 0;}
  Am_Object new_guard = New_obj(GuardProto);
  Load_position(new_guard, ifile);
  char Contents[1000];
  ifile >> ws;
  ifile.getline(Contents, 1000);
  new_guard.MSet(Name, Contents);
  Add_obj(rule, new_guard);
  ifile >> tag; if (tag != End_Gurd) return 0;
  return 1;
}

// -------- Pattern --------
// Beg_Patt Type Name Dir Hole position_with_icon_info
// ... ports ...
// ... holes ...
// ... goals ...
// ... binders ...
// End_Patt

int Save_pattern(Am_Object pattern, ofstream& ofile)
{
  ofile << Beg_Patt << " ";
  Am_String name = pattern.MGet(Name);
  int type = pattern.MGet(Type);
  int dir  = pattern.Get(Dir);

  int hole = 0; // Pattern$B$N$H$-$NBP1~!%(B
  Am_Object hl = pattern.Get(Hole);
  if (hl.Valid()) {
    hole = Object_List_Index(pattern.Get_Object(Parent).Get(HoleList), hl);
  }
  ofile << type << " " << name << " " << dir << " " << " " << hole;
  Save_position(pattern, ofile); ofile << endl;

  Save_port_part(pattern.Get(PortList), ofile);
  Save_holes_part(pattern.Get(HoleList), ofile);
  Save_goals_part(pattern, ofile);
  Save_binder_part(pattern, ofile);

  ofile << End_Patt << endl;
}

int Load_pattern(Am_Object rule, ifstream& ifile)
{
  char tag;
  ifile >> tag; if (tag != Beg_Patt) {ifile.putback(tag); return 0;}
  int type, dir, hole;
  char name[1000];
  ifile >> type >> name >> dir >> hole;
  cout << "Load pattern: " << name << " type: " << type << " hole: " << hole << endl;
  
  Am_Object new_patt;
  if (type == NormalPattern) new_patt = New_obj(PatternProto);
  if (type == ReplPattern)  new_patt = New_obj(ReplPatternProto).Set(Dir, dir);
  new_patt.Get_Object(Model).Set(Name, name);

  Load_position(new_patt, ifile);

  Set_zooming(new_patt, false);
  Load_port_part(new_patt, ifile);
  Load_holes_part(new_patt, ifile);
  Load_goals_part(new_patt, ifile);
  Load_binder_part(new_patt, ifile);
  Set_zooming(new_patt, true);

  if (hole != 0) {
    Am_Object dest = Object_List_Get(rule.Get(HoleList), hole);
    Single_Hole_Replace(dest, new_patt);
  } else {
    Add_obj(rule, new_patt);
  }
  ifile >> tag; if (tag != End_Patt) return 0;
  cout << "end Load pattern " << endl;
  return 1;
}

// -------- Hole Part --------
// Beg_Holes
// ... holes ...
// End_Holes

int Save_holes_part(Am_Value_List holes, ofstream& ofile)
{
  ofile << Beg_Holes << endl;
  for (holes.Start(); !holes.Last(); holes.Next()) {
    Am_Object hole = holes.Get();
    Save_hole(hole, ofile);
  }
  ofile << End_Holes << endl;
}

int Load_holes_part(Am_Object patt, ifstream& ifile)
{
  char tag;
  ifile >> tag; if (tag != Beg_Holes) return 0;
  while (Load_hole(patt, ifile));
  ifile >> tag; if (tag != End_Holes) return 0;
}

// -------- Hole --------
// Beg_Hole Name position_info
// ... ports ...
// End_Hole
// $B?F$OC/$+!)(B

int Save_hole(Am_Object hole, ofstream& ofile)
{
  ofile << Beg_Hole << " ";
  Am_String name = hole.MGet(Name);
  ofile << name;
  Save_position(hole, ofile); ofile << endl;
  Save_port_part(hole.Get(PortList), ofile);
  ofile << End_Hole << endl;
}

int Load_hole(Am_Object patt, ifstream& ifile)
{
  char tag;
  ifile >> tag; if (tag != Beg_Hole) {ifile.putback(tag); return 0;}
  char name[1000];
  ifile >> name;
  Am_Object hole;
  if (patt.Is_Instance_Of(ReplPatternProto)) {
    Am_Value_List hole_list = patt.Get(HoleList);  hole_list.Start();
    hole = hole_list.Get();
  } else {
    hole = New_obj(HoleProto);
  } 
  hole.Get_Object(Model).Set(Name, name);
  Load_position(hole, ifile);
  if (!patt.Is_Instance_Of(ReplPatternProto)) Add_obj(patt, hole);
  Am_Value_List ports = hole.Get(PortList);

  Set_zooming(hole, false);
  if (ports.Empty()) Load_port_part(hole, ifile); // $B$-$?$M!<!#(B
  else Load_exist_port_part(hole, ifile);
  Set_zooming(hole, true);

  ifile >> tag; if (tag != End_Hole) {Err("End_Hole", name); return 0;}
  return 1;
}


// -------- Port Part --------
// Beg_Ports
// ... ports ...
// End_Ports

int Save_port_part(Am_Value_List ports, ofstream& ofile)
{
  ofile << Beg_Ports << endl;
  for (ports.Start(); !ports.Last(); ports.Next()) {
    Am_Object port = ports.Get();
    Save_port(port, ofile);
  }
  ofile << End_Ports << endl;
}

int Load_port_part(Am_Object process, ifstream& ifile)
{
  char tag;
  ifile >> tag; if (tag != Beg_Ports) return 0;
  while (Load_port(process, ifile));
  ifile >> tag; if (tag != End_Ports) return 0;
}

int Load_exist_port_part(Am_Object process, ifstream& ifile)
{
  char tag;
  ifile >> tag; if (tag != Beg_Ports) return 0;

  Am_Value_List ports = process.Get(PortList);
  for (ports.Start(); !ports.Last(); ports.Next()) {
    Am_Object port = (Am_Object)ports.Get();
    if (!Load_exist_port(port, ifile)) return 0;
  }

  ifile >> tag; if (tag != End_Ports) return 0;
}

// -------- Port --------
// Beg_Port Name Type Age Mode Open Dir Spec position_info
// ... Values ...
// End_Port

int Save_port(Am_Object port, ofstream& ofile)
{
  ofile << Beg_Port  << " ";
  Am_String name = port.MGet(Name);
  int type = port.MGet(Type);
  int age  = port.Get(Age);
  int mode = port.Get(Mode);
  int state = port.Get(State);
  int dir  = port.Get(Dir);
  int Spec = port.MGet(Special);
  ofile << name <<" "<< type <<" "<< age <<" "<< mode <<" "<< state <<" "<< dir <<" "<< Spec;
  Save_position(port, ofile);
  ofile << endl;
  Save_value_part(port.Get(ValueList), ofile);
  ofile << End_Port << endl;
}

int Load_port(Am_Object process, ifstream& ifile)
{
  char tag;
  ifile >> tag; if (tag != Beg_Port) {ifile.putback(tag); return 0;}
  int type, age, mode, state, dir, Spec;
  char name[1000];
  ifile >> name >> type >> age >> mode >> state >> dir >> Spec;
  Am_Object new_port;
  if (Spec == PortNormal) {
    if (mode == PortInput  && type == PortSingleton) new_port = New_obj(InSingletonProto);
    if (mode == PortOutput && type == PortSingleton) new_port = New_obj(OutSingletonProto);
    if (mode == PortInput  && type == PortStream) new_port = New_obj(InStreamProto);
    if (mode == PortOutput && type == PortStream) new_port = New_obj(OutStreamProto);
  } else {
    if (Spec == PortPatSingleton && mode == PortInput) new_port = New_obj(SpecialSingletonProto);
    if (Spec == PortPatSingleton && mode == PortOutput) new_port = New_obj(SpecialSingletonProto);
    if (Spec == PortPatStream && mode == PortInput) new_port = New_obj(SpecialStreamProto);
    if (Spec == PortPatStream && mode == PortOutput)new_port = New_obj(SpecialStreamProto);
    if (Spec == PortMap && mode == PortOutput) new_port = New_obj(MapInProto);
    if (Spec == PortMap && mode == PortInput) new_port = New_obj(MapOutProto);
    if (Spec == PortBroadcast && type == PortSingleton) new_port = New_obj(BroadSingletonProto);
    if (Spec == PortBroadcast && type == PortStream) new_port = New_obj(BroadStreamProto);
    if (Spec == PortMerge) new_port = New_obj(MergeProto);
  }
  new_port
    //    .Set(Age,  age)
    .Set(Mode, mode)
    .Set(State, state)
    .Set(Dir,  dir)
    .Get_Object(Model)
      .Set(Name, name)
    ;
  //  if (process.Is_Instance_Of(RuleProto) && age == PortOld) new_port.Set(Mode, 1 - mode);
  Load_position(new_port, ifile);
  Add_obj(process, new_port);
  Load_value_part(new_port, ifile);
  ifile >> tag; if (tag != End_Port) return 0;
  return 1;
}

int Load_exist_port(Am_Object port, ifstream& ifile)
{
  char tag;
  ifile >> tag; if (tag != Beg_Port) {ifile.putback(tag); return 0;}
  int type, age, mode, state, dir, Spec;
  char name[1000];
  ifile >> name >> type >> age >> mode >> state >> dir >> Spec;
  port.Set(State, state)
      .Set(Dir,  dir)
    ;
  Am_Object owner = port.Get_Owner();
  port.Remove_From_Owner(); // $B1x$$!#(B
  Load_position(port, ifile);
  owner.Add_Part(port); // $B1x$$!#(B
  Load_value_part(port, ifile);
  ifile >> tag; if (tag != End_Port) return 0;
  return 1;
}

int Load_val_port(Am_Object port, ifstream& ifile)
{
  char tag;
//  ifile >> tag; if (tag != Beg_Port) {ifile.putback(tag); return 0;}
  int type, age, mode, state, dir, Spec;
  char name[1000];
  ifile >> name >> type >> age >> mode >> state >> dir >> Spec;

  Am_Object rule = Parent_Rule(port);
  if (!rule.Valid()) return 0;
  Am_Object old_port = 0;
  Am_Value_List inner_ports = (Am_Value_List)rule.Get(InnerPortList);
  old_port = Object_List_Get_By_Model_Key(inner_ports, Name, name);

  Am_Object new_port;
  if (old_port.Valid()) {
    new_port = old_port.Create(); // ***
    Load_position(new_port, ifile);
    port.Get_Part(ContentsPart).Add_Part(new_port);
  } else if (Spec == PortNormal) {
    if (mode == PortInput  && type == PortSingleton) new_port = New_obj(InSingletonProto);
    if (mode == PortOutput && type == PortSingleton) new_port = New_obj(OutSingletonProto);
    if (mode == PortInput  && type == PortStream) new_port = New_obj(InStreamProto);
    if (mode == PortOutput && type == PortStream) new_port = New_obj(OutStreamProto);
    new_port
      .Set(Dir,  dir)
      .Set(Mode, mode)
      .Set(Age,  age)
      .Set(State, state)
      .Get_Object(Model)
        .Set(Name, name)
      ;
    Load_position(new_port, ifile);
    Add_obj(port, new_port);
  } else return 0;
  Load_value_part(new_port, ifile);
  ifile >> tag; if (tag != End_Port) return 0;
  return 1;
}

// -------- Values Part --------
// Beg_Vals
// ... values ...
// End_Vals

int Save_value_part(Am_Value_List vals, ofstream& ofile)
{
  ofile << Beg_Vals << endl;
  for (vals.Start(); !vals.Last(); vals.Next()) {
    Am_Object val = vals.Get();
    Save_value(val, ofile);
  }
  ofile << End_Vals << endl;
}

int Load_value_part(Am_Object port, ifstream& ifile)
{
  char tag;
  ifile >> tag; if (tag != Beg_Vals) return 0;
  while (Load_value(port, ifile));
  ifile >> tag; if (tag != End_Vals) return 0;
}

// -------- Value --------

int Save_value(Am_Object value, ofstream& ofile)
{
  if (value.Is_Instance_Of(AtomProto)) return Save_atom(value, ofile);
  if (value.Is_Instance_Of(MessageProto)) return Save_structure(value, ofile);
  if (value.Is_Instance_Of(PortProto)) return Save_port(value, ofile);
  return 0;
}

int Load_value(Am_Object port, ifstream& ifile)
{
  char tag;
  ifile >> tag;
  if (tag == Beg_Atom) return Load_atom(port, ifile);
  if (tag == Beg_Stru) return Load_structure(port, ifile);
  if (tag == Beg_Port) return Load_val_port(port, ifile);
  ifile.putback(tag);
  return 0;
}

// -------- Atom --------
// Beg_Atom Contents
// End_Atom
int Save_atom(Am_Object value, ofstream& ofile)
{
  ofile << Beg_Atom << " ";
  ofile << (Am_String)value.Get_Object(Model).Get(Name) << endl;
  ofile << End_Atom << endl;
}

int Load_atom(Am_Object port, ifstream& ifile)
{
  char tag;
  char Contents[1000];
  ifile >> ws;
  ifile.getline(Contents, 1000);
  Am_Object new_atom = New_obj(AtomProto);
  new_atom.Get_Object(Model)
    .Set(Name, Contents);
  Add_obj(port, new_atom);
  ifile >> tag; if (tag != End_Atom) return 0;
  return 1;
}

// -------- Structure --------
// Beg_Stru Name
// ... values ...
// End_Stru

int Save_structure(Am_Object value, ofstream& ofile)
{
  ofile << Beg_Stru << " ";
  ofile << (Am_String)value.Get_Object(Model).Get(Name) << endl;
  Save_value_part(value.Get(ValueList), ofile);
  ofile << End_Stru << endl;
}

int Load_structure(Am_Object port, ifstream& ifile)
{
  char tag;
  char name[1000];
  ifile >> name;
  Am_Object new_str = New_obj(MessageProto);
  new_str.Get_Object(Model)
    .Set(Name, name);
  Add_obj(port, new_str);
  Load_value_part(new_str, ifile);
  ifile >> tag; if (tag != End_Stru) return 0;
  return 1;
}

