/* Copyright (C) 1997 Itoh Hidenori */
/******************************************************************************/
/*                                                                            */
/*  Domain independent utilitiy predicates                                    */
/*                                                                            */
/******************************************************************************/

/******************************************************************************/
/*                                                                            */
/*  call       : runtime(+Goal)                                               */
/*                                                                            */
/*  arguments  : Goal = A goal in prolog syntax.                              */
/*                                                                            */
/******************************************************************************/
/* It measures runtime to be spent to execute a goal.                         */
/******************************************************************************/
runtime(Goal) :-
    statistics( runtime, [T0|_]),
    call(Goal),
    statistics( runtime, [T1|_]),
    T is T1 - T0,
    format( 'Runtime: ~3d sec.~n', [T]).

/******************************************************************************/
/*                                                                            */
/*  call       : difference(+List1, +List2, -Difference)                      */
/*                                                                            */
/*  arguments  : List1      = a list,                                         */
/*               List2      = a list,                                         */
/*               Difference = a list which contains element containd in List1 */
/*                            but not containd in List2.                      */
/*                                                                            */
/******************************************************************************/
difference([], _, []):-!.
difference([H|T], List2, Difference):-!,
    ( member(H, List2) -> (
	  difference(T, List2, Difference)
      );(
	  difference(T, List2, D),
	  Difference = [H|D]
    )).

/******************************************************************************/
/*                                                                            */
/*  call       : make_list_to_n(+N, -List)                                    */
/*                                                                            */
/*  arguments  : N    = a positive integer,                                   */
/*               List = a list [1, 2, ... , N].                               */
/*                                                                            */
/******************************************************************************/
make_list_to_n(N, L):-!,
    make_list_to_n1(N, 1, L).
make_list_to_n1(N, N, [N]):-!.
make_list_to_n1(N, M, [M|L]):-!,
    M1 is M+1,
    make_list_to_n1(N, M1, L).

/******************************************************************************/
/*                                                                            */
/*  call       : make_list_from_n(+N, -List)                                  */
/*                                                                            */
/*  arguments  : N    = a positive integer,                                   */
/*               List = a list [N, N-1, ... , 1].                             */
/*                                                                            */
/******************************************************************************/
make_list_from_n(0, []):-!.
make_list_from_n(X, [X|L]):-
    X1 is X - 1,!,
    make_list_from_n(X1, L).

/******************************************************************************/
/*                                                                            */
/*  call       : make_a_n_tuple_of_integer_upto_m(+N, +M, -List)              */
/*                                                                            */
/*  arguments  : N    = a positive integer,                                   */
/*               M    = a positive integer,                                   */
/*               List = a list [i1, i2, ..., iN] of integers,                 */
/*                      where 1=<i1, i2, ... , iN=<M.                         */
/*                                                                            */
/******************************************************************************/
/* It give one of lists satisfying the above.                                 */
/******************************************************************************/
make_a_n_tuple_of_integer_upto_m(0, _, []):-!.
make_a_n_tuple_of_integer_upto_m(N, M, [X|T1]):-
    N1 is N - 1,
    from_1_to_n(X, M),
    make_a_n_tuple_of_integer_upto_m(N1, M, T1).
from_1_to_n(X, N):-
    N>0,!,
    (X=N ; (N1 is N-1,
	    from_1_to_n(X, N1))).

/******************************************************************************/
/*                                                                            */
/*  call       : append_with_check(+List1,+ List2, -New)                      */
/*                                                                            */
/*  arguments  : List1, List2 = lists to be appended.                         */
/*               New          = a resulted list.                              */
/*                                                                            */
/******************************************************************************/
/* It appends two list and remove duplicates from the list.                   */
/******************************************************************************/
append_with_check(List1, List2, New):-!,
    append(List1, List2, List3),
    remove_duplicates(List3, New).

/******************************************************************************/
/*                                                                            */
/*  call       : make_list_difference(+List, -Differencial)                   */
/*                                                                            */
/*  arguments  : List         = a list.                                       */
/*               Differencial = a resulted differential list.                 */
/*                                                                            */
/******************************************************************************/
/* It changes a list into a differential list.                                */
/******************************************************************************/
make_list_difference([], L-L):-!.
make_list_difference([X|R], [X|L1]-L2):-!,
    make_list_difference(R, L1-L2).

/******************************************************************************/
/*                                                                            */
/*  call       : nthnew(+Num, +Element, -List)                                */
/*               nthnew(+Num, -Element, +List)                                */
/*                                                                            */
/*  arguments  : Num     = an integer.                                        */
/*               Element = an object.                                         */
/*               List    = a list.                                            */
/*                                                                            */
/******************************************************************************/
/* It extracts an element indexed by the number form list, or make a list     */
/* that has  the element in the position indexed by the number.               */
/******************************************************************************/
nthnew(1, X, [X|_]):-!.
nthnew(1, _, [_|_]):-!,fail.
nthnew(N, X, [_|R]):-
    N1 is N-1,
    nthnew(N1, X, R).
    
/******************************************************************************/
/*                                                                            */
/*  call       : project(+Tuples, -Arity, -Reduced)                           */
/*                                                                            */
/*  arguments  : Tuples  = a list of tuples.                                  */
/*               Arity   = an integer.                                        */
/*               Reduced = a list of reduced(projected) tuples.               */
/*                                                                            */
/******************************************************************************/
/* It makes a list of tuples each of which has first N(=Arity) elements of    */
/* each of original tuples.                                                   */
/******************************************************************************/
project([], _Arity, []):-!.
project([Tuple|Tuples], Arity, Reduced):-!,
    project(Tuples, Arity, Reduced1),
    first_n(Tuple, Arity, Shorten),
    (member(Shorten, Reduced1) ->
        Reduced = Reduced1 ; Reduced = [Shorten|Reduced1]).

/******************************************************************************/
/*                                                                            */
/*  call       : first_n(+List, +N, -Shorten)                                 */
/*                                                                            */
/*  arguments  : List = a list.                                               */
/*               N    = an integer.                                           */
/*               Shorten = a list shortened.                                  */
/*                                                                            */
/******************************************************************************/
/* It shortens a list to make first N elements of the original list.          */
/******************************************************************************/
first_n(_List, 0, []):-!.
first_n([X|List], N, [X|Short_list]):-!,
    N1 is N-1,
    first_n(List, N1, Short_list).

/******************************************************************************/
/*                                                                            */
/*  call       : subset(+Set1, +Set2)                                         */
/*                                                                            */
/*  arguments  : Set1 = a set.                                                */
/*               Set2 = a set which may include set1.                         */
/*                                                                            */
/******************************************************************************/
/* It checks whether Set1 is a subset of Set2.                                */
/******************************************************************************/
subset([], _):-!.
subset([Element|Set1], Set2):-
    member(Element, Set2) -> subset(Set1, Set2).

/******************************************************************************/
/*                                                                            */
/*  call       : pick_arguments(+Tuple, +Args, -Picked)                       */
/*                                                                            */
/*  arguments  : Tuple  = a tuple.                                            */
/*               Args   = a list of integer.                                  */
/*               Picked = a tuple.                                            */
/*                                                                            */
/******************************************************************************/
/* It makes a tuple(Picked) that consists of element indexed by the list from */
/* another tuple(Tuple).                                                      */
/******************************************************************************/
pick_arguments(_Tuple, [], []):-!.
pick_arguments(Tuple, [Arg|Args], [X|Picked]):-
    nthnew(Arg, X, Tuple),!,
    pick_arguments(Tuple, Args, Picked).

/******************************************************************************/
/*                                                                            */
/*  call       : membernew(+Element, +List)                                   */
/*                                                                            */
/*  arguments  : Element = an object.                                         */
/*               List    = a list.                                            */
/*                                                                            */
/******************************************************************************/
/* It is the same as 'member' in the library except that 'membernew' can      */
/* properly treat nolimitlist.                                                */
/******************************************************************************/
membernew(_, Y):-var(Y),!,fail.
membernew(X, [Y|_]):-X=Y,!.
membernew(X, [_|Y]):-!,membernew(X,Y).    

/******************************************************************************/
/*                                                                            */
/*  call       : memberdiff(+Element, +List)                                  */
/*                                                                            */
/*  arguments  : Element = an object.                                         */
/*               List    = a differencial list.                               */
/*                                                                            */
/******************************************************************************/
/* It is a version of 'member' for differencial list.                         */ 
/******************************************************************************/
memberdiff(_, L1-L2):-L1==L2,!,fail.
memberdiff(X, [X|_]-_):-!,true.
memberdiff(X, [_|L1]-L2):-!,memberdiff(X, L1-L2).

/******************************************************************************/
/*                                                                            */
/*  call       : insert_last(+Element, +(-)List)                              */
/*                                                                            */
/*  arguments  : Element = an object.                                         */
/*               List    = a differencial list.                               */
/*                                                                            */
/******************************************************************************/
/* It inserts an object to the last of a differencial list.                   */
/******************************************************************************/
insert_last(A, [X|_]):-var(X),!,X=A.
insert_last(A, [_|Y]):-insert_last(A,Y).


/******************************************************************************/
/*                                                                            */
/*  call       : make_nolimit_all(+ListOfList1, -ListOfList2)                 */
/*                                                                            */
/*  arguments  : ListOfList1 = a list of lists.                               */
/*               ListOfList2 = a list of nolimit lists.                       */
/*                                                                            */
/******************************************************************************/
/* It makes all of lists in the list1 nolimit.                                */
/******************************************************************************/
make_nolimit_all([],[]):-!.
make_nolimit_all([P|Rest], [P_nolimit|Rest_nolimit]):-!,
    make_list_nolimit(P,P_nolimit),
    make_nolimit_all(Rest, Rest_nolimit).

/******************************************************************************/
/*                                                                            */
/*  call       : make_list_nolimit(+List1, -List2)                            */
/*                                                                            */
/*  arguments  : List1 = a list.                                              */
/*               List2 = a nolimit list,                                      */
/*                       i.e. a list whose last element is open.              */
/*                                                                            */
/******************************************************************************/
/* It makes a list nolimit.                                                   */
/******************************************************************************/
make_list_nolimit([], _):-!.
make_list_nolimit([X|R], [X|R1]):-
    make_list_nolimit(R,R1),!.

/******************************************************************************/
/*                                                                            */
/*  call       : make_list_limit(+List1, -List2)                              */
/*                                                                            */
/*  arguments  : List1 = a nolimit list.                                      */
/*               List2 = a list.                                              */
/*                                                                            */
/******************************************************************************/
/* It makes a list limit.                                                     */
/******************************************************************************/
make_list_limit([F|_], []):-
    var(F),!.
make_list_limit([F|R], [F|R1]):-
    !,make_list_limit(R,R1).

