/* Copyright (C) 1997 Itoh Hidenori */
/******************************************************************************/
/*                                                                            */
/*  pre_and_post.pl --- predicates for pre-processing and post-processing.    */
/*                                                                            */
/******************************************************************************/

/******************************************************************************/
/*                                                                            */
/*  call       : print_title                                                  */
/*                                                                            */
/******************************************************************************/
/* It prints the a title before a session.                                    */
/******************************************************************************/
print_title:-
    format("~s  ~s~n~s ~s~n~n", 
	   ["F O I L - I, v e r. 0.8(July, 1996)",
            "by N.Inuzuka(inuzuka@ics.nitech.ac.jp)",
	    "Dept. Artificial Intelligence and Computer Science,",
	    "Nagoya Institute of Technology, Japan"]).

/******************************************************************************/
/*                                                                            */
/*  call       : initializing(+Sample_file_name)                              */
/*                                                                            */
/*  arguments  : Sample_file_name = a name of sample file                     */
/*                                                                            */
/******************************************************************************/
/* It reads information from a sample file, deletes a previous setting,       */
/*      calculates tuples for hojo relations, etc.                            */
/******************************************************************************/
initialize(Sample_file):-!,
    literal_count_initial,                  % <-- initilizing the counter that
    delete_information,                     %    conunts clauses visited during
    read_sample(Sample_file),!,             %    a learning process.
    assert(covered_sample([])),
    assert(literals_sofar([])),
    calculate_hojo_tuples,
    calculate_builtin_tuples,
    clause(plus(Original_plus),_),
    assert_known(Original_plus).

/******************************************************************************/
/*                                                                            */
/*  call       : initialize_in_learning(+Sample)                              */
/*                                                                            */
/*  arguments  : Sample = a sample given, i.e. pair (Pos, Neg) of positive    */
/*                        and negative samples which are lists of examples.   */
/*                                                                            */
/******************************************************************************/
/* It initializes a memory to store intermediate clauses.                     */
/******************************************************************************/
initialize_in_learning(Sample):-!,
    target(_,Arity,_,Type),
    abolish(intermediate_clauses/1),
    abolish(determinate_predicate/1),
    clause(literals_sofar(Sofar), _),
    assert(intermediate_clauses([0-(Sample, Arity, Type, [], [], Sofar)])).

/******************************************************************************/
/*                                                                            */
/*  call       : read_sample(+Sample_file_name)                               */
/*                                                                            */
/*  arguments  : Sample_file_name = a name of sample file                     */
/*                                                                            */
/******************************************************************************/
/* It reads a sample file and get information about a target predicate,       */
/* hojo predicate, and plus, minus, and unknown sample. The informatin is     */
/* asserted.                                                                  */
/******************************************************************************/
read_sample(Sample_file):-
    open(Sample_file, read, Sample),
    repeat (
	read(Sample, A_data),
        process_sample(A_data)
    ) until A_data == end_of_file,
    close(Sample).

/******************************************************************************/
/*                                                                            */
/*  call       : process_sample(+Term)                                        */
/*                                                                            */
/*  arguments  : Term = a term that is read from a sample file                */
/*                                                                            */
/******************************************************************************/
/* It processes a term read from a sample file. The term is precessed         */
/* accorging to the fanctor.                                                  */
/******************************************************************************/
process_sample(end_of_file):- !.
process_sample(plus(Plus)):-!,
    assert(plus(Plus)),
    make_nolimit_all(Plus, Plus_nolimit),
    assert(uncovered_plus(Plus_nolimit)).
process_sample(minus(Minus)):-!,
    assert(negative(Minus)),
    make_nolimit_all(Minus, Minus_nolimit),
    assert(minus(Minus_nolimit)).
process_sample(A_data):- !,assert(A_data).

/******************************************************************************/
/*                                                                            */
/*  call       : delete_information                                           */
/*                                                                            */
/******************************************************************************/
/* It deletes all information of the previous session.                        */
/******************************************************************************/
delete_information:-!,
    abolish(domain/2),
    abolish(target/4),
    abolish(plus/1),
    abolish(negative/1),
    abolish(minus/1),
    abolish(uncovered_plus/1),
    abolish(hojo_tuples/6),
    abolish(intermediate_clauses/1),
    abolish(covered_sample/1),
    abolish(result/1),
    abolish(literals_sofar/1),
    abolish(determinate_predicate/1),
    abolish(result/1),
    abolish(this_depth/1),
    abolish(this_call/1),
    abolish(this_clause/1),
    abolish(literal_sofar/1),
    abolish(foili_exist_base_clause/0),
    abolish(target_head/1),
    abolish(target_head_string/1),
    abolish(foili_known/1),
    abolish(candidate_literal/1),
    abolish(clause_for_merge/2),
    abolish(foili_learning_failure/0),
    abolish(recursion/1),
    abolish(base/1).
