/* Copyright (C) 1997 Itoh Hidenori */
/******************************************************************************/
/*                                                                            */
/*  Miscellaneous predicates (domain dependent utilities)                     */
/*                                                                            */
/******************************************************************************/

/******************************************************************************/
/*                                                                            */
/*  call       : add_clause(+Best)                                            */
/*                                                                            */
/*  arguments  : Best = a list of intermediate clauses with their evaluations */
/*                      to be added to the candidate list.                    */
/*                                                                            */
/******************************************************************************/
/* It adds candidate literals to the memory of intermediate clauses, in which */
/* clauses are sorted by their evaluation values.                             */
/******************************************************************************/
add_clause(New_clauses):-
    clause(intermediate_clauses(Clauses),_),!,
    abolish(intermediate_clauses/1),
    append(Clauses, New_clauses, Clauses_plus),
    keysort(Clauses_plus, Clauses_sorted1),
    reverse(Clauses_sorted1, Clauses_sorted),
    choose_best_ten(Clauses_sorted, Best_ten),
    assert(intermediate_clauses(Best_ten)),
    show_intermediate(Best_ten).

/******************************************************************************/
/*                                                                            */
/*  call       : show_intermediate(+Clauses)                                  */
/*                                                                            */
/*  arguments  : Clauses : a list of clauses                                  */
/*                                                                            */
/******************************************************************************/
/*  It shows clauses with their information.                                  */
/******************************************************************************/
show_intermediate(Clauses):-!,
    format("~nCandidate(s):~n", []),
    format("  Eval.:(Pos.,Neg.) clause~n", []),
    show_intermediate1(Clauses),
    format("~n", []).
show_intermediate1([]):-!.
show_intermediate1([EV-((Plus, Minus),_,_,L,_,_)|Rest]):-!,
    length(Plus, Lp), length(Minus, Lm),
    format("  ~3f:(~d,~d) ~p~n", [EV,Lp,Lm,L]),
    show_intermediate1(Rest).    

/******************************************************************************/
/*                                                                            */
/*  call       : get_the_best_clause(-Best)                                   */
/*                                                                            */
/*  arguments  : Best = the intermediate clause with the best evaluation      */
/*                                                                            */
/******************************************************************************/
/* It gives a clause with the best evaluation.                                */
/* Too long clauses are discarded.                                            */
/******************************************************************************/
get_the_best_clause(Best):-!,
    (clause(intermediate_clauses([Best1|Rest]), _) ->
     (
	 abolish(intermediate_clauses/1), 
	 assert(intermediate_clauses(Rest)),
	 Best1 = _-(_,_,_,Literals,_,_),
	 length(Literals, Length),
	 (Length > 5 -> get_the_best_clause(Best) ; _-Best=Best1)
     ); fail
    ).

/******************************************************************************/
/*                                                                            */
/*  call       : pick_found_clauses(+Clauses, -Found)                         */
/*                                                                            */
/*  arguments  : Clauses = a list of clauses processed and evaluated,         */
/*               Found   = a list of consistent clauses from the clauses.     */
/*                                                                            */
/******************************************************************************/
/* It gives a list of consistent clauses, i.e. clauses that coveres no        */
/* negative examples and a positive examples.                                 */
/******************************************************************************/
pick_found_clauses([], []).
pick_found_clauses([_-C|Clauses], Found):-
    (C=(([_|_],[]),_,_,_,_,_) -> Found=[C|Found1]; Found=Found1),
    pick_found_clauses(Clauses, Found1).

/******************************************************************************/
/*                                                                            */
/*  call       : choose_best_ten(+Clauses, -Best_ten)                         */
/*                                                                            */
/*  arguments  : Clauses  = a list of clauses processed and evaluated,        */
/*               Best_ten = a list of at most ten best clauses.               */
/*                                                                            */
/******************************************************************************/
/* It gives a list of the best evaluated ten clauses from given clauses.      */
/* If the number of given clause is less than eleven it gives all clauses     */
/******************************************************************************/
choose_best_ten(Clauses, Best_ten):-!,
    keysort(Clauses, Sortedr),
    reverse(Sortedr, Sorted),
    length(Sorted, L),
    ( L<11 -> (
	  Best_ten = Sorted 
      ) ; (
	  Sorted = [X1, X2, X3, X4, X5, X6, X7, X8, X9, X10| _],
	  Best_ten = [X1, X2, X3, X4, X5, X6, X7, X8, X9, X10]
      )
    ).

/******************************************************************************/
/*                                                                            */
/*  call       : check_to_be_solved(+Examples, -Remain)                       */
/*                                                                            */
/*  arguments  : Examples = a set of positive examples,                       */
/*               Remain   = a set of positive examples that are not covered   */
/*                          by clauses induced so far.                        */
/*                                                                            */
/******************************************************************************/
/* It tries to derive a target relation with given examples using clauses     */
/* induced so far(by "try_to_solve"), and gives a set examples that is        */
/* not direved.                                                               */
/******************************************************************************/
check_to_be_solved(Examples, Remain):-!,
    (clause(target_head(Head),_) ; assert_target_head(Head)),
    abolish(this_clause/1),
    try_to_solve(Head, Examples,  Solved),
    difference(Examples, Solved, Remain).

/******************************************************************************/
/*                                                                            */
/*  call       : try_to_solve(+Head, +Examples, -Solved)                      */
/*                                                                            */
/*  arguments  : Goal     = an uninstantiated goal literal,                   */
/*               Tuples   = a list of tuples,                                 */
/*               Solved   = a list of tuples with which the goal succeeds.    */
/*                                                                            */
/******************************************************************************/
/* It tries to solve the goal instanciated  with each tuple. Tuples with      */
/* which the goal succeeded are gatherd into the Solved.                      */
/******************************************************************************/
try_to_solve(_Goal, [], []):-!.
try_to_solve(Goal, [Tuple|Tuples], Solved):-!,
    try_to_solve(Goal, Tuples, Solved1),
    copy_term(Goal, Goal_copy),
    Goal_copy=..[_|Args],
    matching_to_tuple1(Args, Tuple),
    abolish(this_depth/1),
    abolish(this_call/1),
    assert(this_depth(-1),_),
    assert(this_call(Goal_copy),_),
    (check_goal(Goal_copy, _) -> Solved=[Goal|Solved1] ;
     Solved=Solved1).

/******************************************************************************/
/*                                                                            */
/*  call       : process_found_clauses(+Found_clauses, -Complete, -Uncovered) */
/*                                                                            */
/*  arguments  : Found_clauses = a list of clauses induced in a search,       */
/*               Complete      = a list of clauses that is not redundant,     */
/*               Uncovered     = a list of positive examples not covered by   */
/*                                the found clauses.                          */
/*                                                                            */
/******************************************************************************/
/* It gives a list of non-redundant clauses and a liat of examples not        */
/* covered by the clauses and the clauses induced so far.                     */
/******************************************************************************/
process_found_clauses([], [], Uncovered):-!,
    uncovered(Uncovered).
process_found_clauses(Found_clauses, Complete, Uncovered):-!,
    remove_redundancy(Found_clauses, Non_redundant),
    calculate_uncovered(Non_redundant, Uncovered),
    remove_nonuse_determinate(Non_redundant, Complete).

/******************************************************************************/
/*                                                                            */
/*  call       : calculate_covered(+Covered_tuples, -Covered_exam)            */
/*                                                                            */
/*  arguments  : Covered_tuples = a set of tuples,                            */
/*               Covered_exam   = a set of examples that.                     */
/*                                                                            */
/******************************************************************************/
/* It makes a set of tuples of only corresponding values with variables       */
/* in a targe relations from a given set of tuples that also includes         */
/* values of variables ocured in bodies of clauses.                           */
/******************************************************************************/
calculate_covered(Covered_tuples, Covered_exam):-
    target(_,Arity,_,_),
    project(Covered_tuples, Arity, Covered_exam1),
    remove_duplicates(Covered_exam1, Covered_exam).

/******************************************************************************/
/*                                                                            */
/*  call       : calculate_uncovered(+Clauses, -Uncovered)                    */
/*                                                                            */
/*  arguments  : Clauses   = a set of clauses processed,                      */
/*               Uncovered = a set of positive examples not covered by the    */
/*                           clauses and by clauses induced so far.           */
/*                                                                            */
/******************************************************************************/
calculate_uncovered(Clauses, Uncovered):-!,
    uncovered(Uncovered_now),
    calculate_uncovered1(Clauses, Uncovered_now, Uncovered).

calculate_uncovered1([], Uncovered, Uncovered):-!.
calculate_uncovered1([(_,_,_,_,Covered,_)|Clauses],
		     Uncovered_now, Uncovered):-!,
    difference(Uncovered_now, Covered, Uncovered1),
    calculate_uncovered1(Clauses, Uncovered1, Uncovered).

/******************************************************************************/
/*                                                                            */
/*  call       : save_uncovered(+Uncovered)                                   */
/*                                                                            */
/*  arguments  : Uncovered = a list of positive examples.                     */
/*                                                                            */
/******************************************************************************/
/* It stores a given set of positive examples in the form:                    */
/*   uncovered_plus([p1, p2, ... ]).                                          */
/******************************************************************************/
save_uncovered(Uncovered):-!,
    abolish(uncovered_plus/1),
    assert(uncovered_plus(Uncovered)).

/******************************************************************************/
/*                                                                            */
/*  call       : inform_uncovered(+Uncovered)                                 */
/*                                                                            */
/*  arguments  : Uncovered = a list of positive examples.                     */
/*                                                                            */
/******************************************************************************/
/* It writes the number of the positive examples to the terminal.             */
/******************************************************************************/
inform_uncovered(Uncovered):-
    length(Uncovered, LU),
    format("~d positive examples remain uncovered~n~n", [LU]),
    format("Uncovered positives are:~n  ~p~n", Uncovered).

uncovered(Uncovered):-
    clause(uncovered_plus(Uncovered), _).

/******************************************************************************/
/*                                                                            */
/*  call       : process_initial_sofar(+Clause, +Sofar)                       */
/*                                                                            */
/*  arguments  : Clause = a partial clause,                                   */
/*               Sofar  = a sofar set found during a search with the partial  */
/*                        clause.                                             */
/*                                                                            */
/******************************************************************************/
/* If the clause is a bodyless clause the sofar set is kept as an initial     */
/* sofar set with a sofar set that is already found.                          */
/******************************************************************************/
process_initial_sofar((_,_,_,[_|_],_,_), _Sofar):-!.
process_initial_sofar((_,_,_,   [],_,_),  Sofar):-!,
    clause(literals_sofar(Old), _),
    abolish(literal_sofar/1),
    append(Old, Sofar, New),
    assert(literal_sofar(New)).

/******************************************************************************/
/*                                                                            */
/*  call       : store_and_show_result(+Clauses_coded)                        */
/*                                                                            */
/*  arguments  : Clauses_coded = a list of clauses in the sysntax of partial  */
/*                               clause.                                      */
/*                                                                            */
/******************************************************************************/
/* It transforms clauses in the partial clause syntax into the prolog syntax, */
/* stores them, and prints them.                                              */
/******************************************************************************/
store_and_show_result(Clauses_coded):-!,
    transform_to_clauses(Clauses_coded, Clauses),
    store_result(Clauses),
    print_result(Clauses),
    abolish(foili_exist_base_clause/0),
    assert(foili_exist_base_clause).

/******************************************************************************/
/*                                                                            */
/*  call       : store_result(+Clauses)                                       */
/*                                                                            */
/*  arguments  : Clauses = a list of clauses in the sysntax of Prolog.        */
/*                                                                            */
/******************************************************************************/
/* Stores the clauses with the functor "result".                              */
/******************************************************************************/
store_result([]):-!.
store_result([Clause|Rest]):-!,
    assert(result(Clause)),
    store_result(Rest).

/******************************************************************************/
/*                                                                            */
/*  call       : print_result(+Clauses)                                       */
/*                                                                            */
/*  arguments  : Clauses = a list of clauses in the sysntax of Prolog.        */
/*                                                                            */
/******************************************************************************/
/* Prints the clauses.                                                        */
/******************************************************************************/
print_result(Clauses):-!,
    format("Clause(s):~n",[]),
    copy_term(Clauses, Clauses_copy),
    print_result1(Clauses_copy),
    format("is/are induced.~n~n",[]).
print_result1([]):-!.
print_result1([Clause|Clauses]):-!,
    numbervars(Clause, 0, _),
    format("  ~p.~n", [Clause]),
    print_result1(Clauses).

/******************************************************************************/
/*                                                                            */
/*  call       : show_result                                                  */
/*                                                                            */
/******************************************************************************/
/* Prints clauses induced in a search.                                        */
/******************************************************************************/
show_result:-!,
    format("~nResulting clauses are:~n",[]),
    show_result1.
show_result1:-
    clause(result(R),_),
    numbervars(R,0,_),
    format("  ~p~n", [R]),
    fail.
show_result1:-
    literal_counter(C),
    format("~n~d literals visited~n", [C]).

/******************************************************************************/
/*                                                                            */
/*  call          : make_tuples(+Domain_names, -Tuples)                       */
/*                                                                            */
/*  arguments     : Domain_names = a list of domain names,                    */
/*                  Tuples       = a set of all tuples.                       */
/*                                                                            */
/*  subpredicates : distribute_elements, add_elements.                        */
/*                                                                            */
/******************************************************************************/
/* Gives all possible tuples that have values from corresponding domains      */
/* of the Domain_names.                                                       */
/******************************************************************************/
make_tuples([], [[]]):-!.
make_tuples([Domain_name|Domain_names], Tuples):-
    make_tuples(Domain_names, T1),
    ( clause(domain(Domain_name, Domain),_) ;
      clause(expr(domain(Domain_name, Domain)),_) ),
    distribute_elements(T1, Domain, Tuples).

distribute_elements(_, [], []):-!.
distribute_elements(T1, [F|R], T):-!,
    distribute_elements(T1, R, T2),
    add_element(T1, F, T3),
    append(T3, T2, T).

add_element([], _, []):-!.
add_element([H|T], Element, [[Element|H]|R1]):-
    !,add_element(T, Element, R1).

/******************************************************************************/
/*                                                                            */
/*  call       : get_sample(-Sample)                                          */
/*                                                                            */
/*  arguments  : Sample = a sample, i.e. a pair of sets of positive and       */
/*                        negative examples.                                  */
/*                                                                            */
/******************************************************************************/
/* Gives a current sample, that is, a set of uncovered positive examples      */
/* and all negative examples.                                                 */
/******************************************************************************/
get_sample((Plus, Minus)):-!,
    uncovered(Plus),
    clause(minus(Minus),_).

/******************************************************************************/
/*  information retrieval predicates                                          */
/*                                                                            */
/*  call       : target(-Pred, -Arity, -Mode, -Type)                          */
/*               hojo(+Pred, -Arity, -Mode, -Type, -Defs, -Condition)         */
/*                                                                            */
/******************************************************************************/
/* It gives information from the database.                                    */
/******************************************************************************/
target(Pred, Arity, Mode, Type):-
    clause(target(Pred, Arity, Mode, Type), _).

hojo(Pred, Arity, Mode, Type, Defs, Condition):-
    clause(hojo_tuples(Pred, Arity, Mode, Type, Defs, Condition), _).

/******************************************************************************/
/*  information asserting predicates                                          */
/*                                                                            */
/*  call          : assert_known(+Plus)                                       */
/*                  assert_target_head_string(+Head_string)                   */
/*                  assert_target_head(+Head)                                 */
/*                                                                            */
/*  subpredicates : construct_a_head = to construct a head literal,           */
/*                                     which is in the "construc_clausee.pl". */
/*                                                                            */
/******************************************************************************/
/* It stores information to the database.                                     */
/******************************************************************************/
assert_known([]):-!.
assert_known([P|Plus]):-
    assert(foili_known(P)),
    assert_known(Plus).    

assert_target_head_string(H):-
    target(Target, Arity,_,_),
    construct_a_head((Target,Arity), H),
    assert(target_head_string(H)).

assert_target_head(Head):-
    (clause(target_head_string(H),_) ; assert_target_head_string(H)),
    append(H, [46,10], HeadS),
    read_from_chars(HeadS, Head),
    assert(target_head(Head)).

/******************************************************************************/
/*  literal counter predicates                                                */
/*                                                                            */
/*  call       : literal_count_initial                                        */
/*               literal_count_inc                                            */
/*               literal_counter(-C)                                          */
/*                                                                            */
/******************************************************************************/
/* They initilize, increment and show the counter.                            */
/******************************************************************************/
literal_count_initial:-
    abolish(foil_counter/1),
    assert(foil_counter(0)).
literal_count_inc:-
    clause(foil_counter(C),_),
    abolish(foil_counter/1),
    C1 is C+1,
    assert(foil_counter(C1)).
literal_counter(C):-
    clause(foil_counter(C),_).
