/* Copyright (C) 1997 Itoh Hidenori */
/******************************************************************************/
/*                                                                            */
/*  call          : try_merge_clauses(+Clauses,   -Merged,                    */
/*                                    +Old_Arity, +Old_Sample)                */
/*                                                                            */
/*  arguments     : Clauses    = clauses each of which includs a literal      */
/*                               found.                                       */
/*                  Merged     = clauses made by merging clauses above.       */
/*                  Old_Arity  = the number of variables before               */
/*                               the learning loop.                           */
/*                  Old_Sample = sample before the learning loop.             */
/*                                                                            */
/*  subpredicates : gather_merged, assert_for_merge                           */
/*                                                                            */
/******************************************************************************/
/* It makes a clause by merging leterals found in a learning loop.            */
/******************************************************************************/
try_merge_clauses(Clauses, _Merged, Old_Arity, Old_Sample):-
    abolish(clause_for_merge/2),
    assert_for_merge(Clauses),
    target(Target, T_Ay, _,_),
    length(Clauses, Total),
    choose_merge_pair(Total, Pair, Old_Arity),
    make_merge(Pair, Old_Arity, Old_Sample, Target, T_Ay),
    fail.

try_merge_clauses(_,Merged,_,_):-
    gather_merged(Merged).

gather_merged([M|Merged]):-
    retract(merged_clause(M)),!,
    gather_merged(Merged).
gather_merged([]):-!.

assert_for_merge(Clauses):-!,assert_for_merge(1, Clauses).
assert_for_merge(_, []):-!.
assert_for_merge(N, [C|Clauses]):-
    assert(clause_for_merge(N,C)),!,
    N1 is N+1,
    assert_for_merge(N1, Clauses).

/******************************************************************************/
/*                                                                            */
/*  call          : make_merge(+Pair, +Old_Arity, +Old_Sample, +Target, T_Ay) */
/*                                                                            */
/*  arguments     : Pair       = two clauses that may be merged.              */
/*                  Old_Arity  = the number of variables before               */
/*                               this learning loop.                          */
/*                  Old_Sample = the sample before this learning loop.        */
/*                  Target     = the name of the target predicate.            */
/*                  T_Ay       = the arity of the target predicate.           */
/*                                                                            */
/******************************************************************************/
/* It makes a clause by merging two clauses after checking effect of the      */
/* merge. Merged clause is asserted into the database.                        */
/******************************************************************************/
make_merge(Pair, Old_Arity, Old_Sample, Target, T_Ay):-!,
    (check_merge_effect(Pair, C, Old_Arity, Old_Sample, Target, T_Ay) ->
       assert(merged_clause(C)) ; true).

/******************************************************************************/
/*                                                                            */
/*  call          : choose_merge_pair(+Total, -(C1, C2), +Old_Arity)          */
/*                                                                            */
/*  arguments     : Total     = the total number of hypothsis clauses         */
/*                               which are checked to be marged.              */
/*                  (C1, C2)  = two clauses that may be merged.               */
/*                  Old_Arity = the sample before this learning loop.         */
/*                                                                            */
/******************************************************************************/
/* It choose a pair of clauses which may be able to be merged.                */
/******************************************************************************/
choose_merge_pair(Total, (C1, C2), Old_Arity):-
    from_1_to_n(N1, Total), 
    from_1_to_n(N2, Total), 
    N2 > N1,
    clause(clause_for_merge(N1,C1),_),
    clause(clause_for_merge(N2,C2),_),
    C1=_-(([_|_],[_|_]),_,_,[(_, _, Args1)|_],_,_),
    C2=_-(([_|_],[_|_]),_,_,[(_, _, Args2)|_],_,_),
    check_merginable(Args1, Args2, Old_Arity).

/******************************************************************************/
/*                                                                            */
/*  call          : check_merginable(+Args1, +Args2, +Old_Arity)              */
/*                                                                            */
/*  arguments     : Args1     = the list of arguments of a clause.            */
/*                  Args2     = the list of arguments of another clause.      */
/*                  Old_Arity = the sample before this learning loop.         */
/*                                                                            */
/******************************************************************************/
/* Check whether a clause and another one can be merged, that is, whether     */
/* arguments of the clauses are exclusive.                                    */
/******************************************************************************/
check_merginable(Args1, Args2, Old_Arity):-!,
    check_merginable1(Args1, Args2, Old_Arity),
    check_merginable1(Args2, Args1, Old_Arity).
check_merginable1([], _, _):-!.
check_merginable1([A|Args], B, Old_Arity):-!,
    (member(A, B) -> A > Old_Arity ; true),
    check_merginable1(Args, B, Old_Arity).

/******************************************************************************/
/*                                                                            */
/*  call          : check_merge_effect(+(C1,C2),    -Merged, +Old_Ay,         */
/*                                     +Old_Sample, +Target, +Target_Ay)      */
/*                                                                            */
/*  arguments     : (C1, C2)   = two clauses that may be merged.              */
/*                  Merged     = a clause made by merging the two clauses.    */
/*                  Old_Ay     = the number of variables before this          */
/*                               learning loop.                               */
/*                  Old_Sample = the sample before this learning loop.        */
/*                  Target     = the name of the target predicate.            */
/*                  Target_Ay  = the arity of the target predicate.           */
/*                                                                            */
/******************************************************************************/
/* It checks effect of merging the two clauses, that is, checks whether the   */
/* merged clause is consistent with sample, and it makes a clause by merging  */
/* if there is effect.                                                        */
/******************************************************************************/
check_merge_effect((C1,C2), Merged, Old_Ay, Old_Sample, Target, Target_Ay):-
    C1=_-((P1,M1),Ay1,_,[L1|L],_,_), L1=(Sg1, Pred1, Ag1),
    C2=_-((P2,M2),Ay2,_,[L2|L],_,_), L2=(Sg2, Pred2, Ag2),!,
    (
	((Pred1==Target, Pred2\==Target)-> (
	     shift_arguments(Ag2, Ag1, Old_Ay, Modified_Ag1, _),
	     Modified_L1=(Sg1, Target, Modified_Ag1),
	     expand_sample([L2|L], Modified_L1, (P2,M2), (P,M),_,Ay2,_,_)
	 )) ;
	((Pred2==Target, Pred1\==Target) -> (
	     shift_arguments(Ag1, Ag2, Old_Ay, Modified_Ag2, _),
	     Modified_L2=(Sg2, Target, Modified_Ag2),
	     expand_sample([L1|L], Modified_L2, (P1,M1), (P,M),_,Ay1,_,_),!
	 )) ; 
	((Pred1\==Target, Pred2\==Target) -> (
	     make_list_to_n(Ay1, Args1),
	     make_list_to_n(Ay2, Args2),
	     shift_arguments(Args1, Args2, Old_Ay, New_Args2, New_Ay),
	     int_tuples(P1, New_Args2, P2, P, _),
	     (P=[]->M=[];int_tuples(M1, New_Args2, M2, M, _))
	 ))
    ),
    length(M, LenM),length(M1, LenM1),length(M2, LenM2),!,
    length(P, NP),
    ((P=[_|_], LenM < min(LenM1, LenM2)) -> (
	 format("(~p) and (~p) can be merged. ~d pos.and ~d neg.example(s) covered.~n",
		[L1, L2, NP, LenM]),
	 make_merged_clause(C1, C2, (P,M), Merged, Old_Ay, New_Ay,
			    Target_Ay, Old_Sample)
     );false
    ).

/******************************************************************************/
/*                                                                            */
/*  call          : shift_arguments(+A1,  +A2,  +Old_Arity,                   */
/*                                  -New_Args2, -New_Arity)                   */
/*                                                                            */
/*  arguments     : A1, A2    = argument lists of lietral1,2, respectively.   */
/*                  Old_Arity = the number of variables before this learning  */
/*                               loop.                                        */
/*                  New_Args2 = new argument list of lietral1,2, respectively.*/
/*                  New_Arity = the number of variables after marging.        */
/*                                                                            */
/*  subpredicates : shift_arguments1                                          */
/*                                                                            */
/******************************************************************************/
/* It shifts argument identification numbers of literal2 to new numbers in    */
/* order to avoid to use the same aruguments with literal1.                   */
/******************************************************************************/
shift_arguments(A1, A2, Old_Arity, New_Args2, New_Arity):-!,
    max_list(A1, Max1),
    Max is max(Max1, Old_Arity),
    shift_arguments1(A2, Old_Arity, Max, New_Args2),
    max_list(New_Args2, Max2),
    New_Arity is max(Max1, Max2).
shift_arguments1([], _Old, _Max, []).
shift_arguments1([A2|Rest], Old, Max, [N|New_Args2]):-
    (A2 =< Old -> N = A2 ; N is Max + (A2 - Old)),
    shift_arguments1(Rest, Old, Max, New_Args2).

/******************************************************************************/
/*                                                                            */
/*  call          : make_merged_clause(+C1, +C2, -(P,N), -Merged, +Old_Ay,    */
/*                                     -New_Ay,  +Target_Ay,  +Old_Sample)    */
/*                                                                            */
/*  arguments     : C1, C2     = clauses including literal1,2, respectively.  */
/*                  (P,N)      = new sample covered by the merged clause.     */
/*                  Merged     = a clause made by merging C1 and C2.          */
/*                  Old_Ay     = the number of variables before this learning */
/*                               loop.                                        */
/*                  New_Ay     = the number of variavles of merged clause.    */
/*                  Target_Ay  = the arity of target predicate.               */
/*                  Old_Sample = old sample.                                  */
/*                                                                            */
/******************************************************************************/
/* It merges two clauses.                                                     */
/******************************************************************************/
make_merged_clause(C1, C2, (P,N), Merged, Old_Ay, New_Ay,
		   Target_Ay, Old_Sample):-
    % EV-(Sample, Arity, Type, Literals, Covered, Literals_sofar)
    C1=_-(_,_,Ty1, [(S1, Pr1, Ag1)|Ls],_,Sf1),
    C2=_-(_,_,Ty2, [(S2, Pr2, Ag2)|Ls],_,_),!,
    shift_arguments(Ag1, Ag2, Old_Ay, New_ag2, _New_Ay),
    merge_type(Ty1, Ty2, New_Ay, Ag1, Ag2, New_ag2, Ty),
    project(P, Target_Ay, Cv),
    length(Cv, NCv),
    calculate_evaluation(
	Old_Sample, (P,N),
	(_, New_Ay, Ty, [(S2, Pr2, New_ag2), (S1, Pr1, Ag1)|Ls],_,_),
	NCv, Ev),
    Merged= Ev-((P,N),New_Ay,Ty,[(S2,Pr2,New_ag2),(S1,Pr1,Ag1)|Ls],Cv,Sf1).

/******************************************************************************/
/*                                                                            */
/*  call          : merge_type(+Ty1, +Ty2, +New_Ay, +Ag1, +Ag2, +New_ag2, -Ty)*/
/*                                                                            */
/*  arguments     : Ty1, Ty2 = type lists of clause1,2, respectively.         */
/*                  New_Ay   = the number of variables of merged clause.      */
/*                  Ag1, Ag2 = argument lists of literal1,2, respectively.    */
/*                  New_ag2  = new shifted argument list of literal2.         */
/*                  Ty       = new type list of merged clause.                */
/*                                                                            */
/*  subpredicates : merge_type1, merge_type2                                  */
/*                                                                            */
/******************************************************************************/
/* It merges two type lists.                                                  */
/******************************************************************************/
merge_type(Ty1, Ty2, New_Ay, _Ag1, Ag2, New_ag2, Ty):-
    length(Ty, New_Ay),
    merge_type1(Ty1, Ty),
    merge_type2(Ag2, New_ag2, Ty2, Ty).
merge_type1([],_):-!.
merge_type1([X|Y],[X|Z]):-!,
    merge_type1(Y,Z).
merge_type2([],[],_,_):-!.
merge_type2([A|Args],[B|Brgs], Old, New):-
    nthnew(A, X, Old),
    nthnew(B, X, New),
    merge_type2(Args, Brgs, Old, New).
