/* Copyright (C) 1997 Itoh Hidenori */
/******************************************************************************/
/*                                                                            */
/*  Predicates to generate candidate literals to construct clauses.           */
/*                                                                            */
/******************************************************************************/

/******************************************************************************/
/*                                                                            */
/*  call          : generate_candidate_literals(+Best, +Sofar, +N_Vars)       */
/*                                                                            */
/*  arguments     : Best   = an intermediate clause with the best evaluation. */
/*                  Sofar  = literals known not to be candidates.             */
/*                  N_Vars = the number of variables before this learning.    */
/*                                                                            */
/*  subpredicates :                                                           */
/*    choose_a_predicate_with_sign = to choose a predicate with a sign for it.*/
/*                  make_arguments = to make an argument list for the literal.*/
/*           check_to_be_candidate = to chech the argument list to be regular.*/
/*                                                                            */
/******************************************************************************/
/* It generates candidate literals to add the intermediate clauses.           */
/* The generated literals are kept in the detabase with the name              */
/*  'candidate_literal'.                                                      */
/******************************************************************************/
generate_candidate_literals(Best, Sofar, N_Vars):-
    Best=(_,N_Vars,_,_,_,Sofar),
    abolish(candidte_literal/1),!,
    generate_candidate_literals1(Best, Sofar, N_Vars).

generate_candidate_literals1(
    (_, N_Vars, Old_type, Literals, _, Sofar), Sofar, N_Vars):-
    choose_a_predicate_with_sign(Literals, Pred, Sign, 
				 Arity, Mode, Type, Condition),
    make_arguments(Arity, N_Vars, Args),
    check_to_be_candidate((Sign, Pred, Args), Literals,
			  N_Vars, NN_Vars, Old_type, New_type, Mode, Type,
			  Condition, Sofar),
    assert(candidate_literal(
	       (_, NN_Vars, New_type, [(Sign, Pred, Args)|Literals], _, Sofar))),
    fail.
generate_candidate_literals1(_,_,_):-!,true.

/******************************************************************************/
/*                                                                            */
/*  call      : choose_a_predicate_with_sign(+Literals, -Pred, -Sign, -Arity, */
/*                                           -Mode,     -Type, -Condition)    */
/*                                                                            */
/*  arguments : Literals  = literals before this learning loop.               */
/*              Pred      = a predicate name chosen.                          */
/*              Sign      = a sign chosen.                                    */
/*              Arity     = an arity for the predicate.                       */
/*              Mode      = a mode list for the predicate.                    */
/*              Type      = a type list for the predicate.                    */
/*              Condition = a condition for the predicate.                    */
/*                                                                            */
/******************************************************************************/
/* It choose a predicate with a sign for it to make a literal.                */
/******************************************************************************/
choose_a_predicate_with_sign([_|_], Pred, p, Arity, Mode, Type, []):-
    clause(foili_exist_base_clause,_),
    target(Pred, Arity, Mode, Type).
choose_a_predicate_with_sign(_,     Pred, Sign, Arity, Mode, Type, Condition):-
    hojo_tuples(Pred, Arity, Mode, Type, _, Condition),
    \+clause(determinate_predicate(Pred),_),
    Sign = p.
%   (Sign = p ; Sign = n).

/******************************************************************************/
/*                                                                            */
/*  call          : make_arguments(+Arity, +N_Vars, -Args)                    */
/*                                                                            */
/*  arguments     : Arity  = the arity of the predicate to make a new literal.*/
/*                  N_Vars = the number of variables used in the clause       */
/*                           before this learning loop.                       */
/*                  Args   = an argument list generated.                      */
/*                                                                            */
/******************************************************************************/
/* It generates a candidate of argument lists for a new literal.              */
/******************************************************************************/
make_arguments(Arity, N_Vars, Args):-
    N is N_Vars + Arity - 1,
    make_a_n_tuple_of_integer_upto_m(Arity, N, Args).

/******************************************************************************/
/*                                                                            */
/*  call          : check_to_be_candidate(+Literal, +Literals,                */
/*                               +N_Vars, -NN_Vars, +Old_type,  -New_type,    */
/*                               +Mode,   +Type,    +Condition, +Sofar)       */
/*                                                                            */
/*  arguments     : Literal    = a literal to be checked.                     */
/*                  Literals   = literal to which another literal is added.   */
/*                  N_Vars     = the number of variables before this loop.    */
/*                  NN_Vars    = the number of variables after this loop.     */
/*                  Old_type   = a type list of variables before this loop.   */
/*                  New_type   = a type list of variables after this loop.    */
/*                  Mode       = a mode list of the literal to be added.      */
/*                  Type       = a type list of the literal to be added.      */
/*                  Condition  = conditions of the literal to be added.       */
/*                  Sofar      = a sofar set.                                 */
/*                                                                            */
/*  subpredicates : check_mode = to check consistency of mode.                */
/*                  check_type = to check consistency of type.                */
/*            check_duplicates = not to allow the same literal even if        */
/*                               it has a different appearance.               */
/*             check_arguments = to check arguments to be canonical.          */
/*     check_illegal_recursion = not to allow abnormal recursive definitions. */
/* avoid_permutation_with_recursion = not to allow recusive definitions       */
/*                               with an argument list of permutation of a    */
/*                               head arguments.                              */
/*             check_condition = to check conditions of a predicate           */
/*                 check_sofar = to check the literal not to be included in   */
/*                               the sofar sat.                               */
/*                                                                            */
/******************************************************************************/
/* It checks if a literal can be added to an intermediate clause.             */
/******************************************************************************/
check_to_be_candidate((Sign, Pred, Args),  Literals,
              N_Vars, NN_Vars,   Old_type, New_type, Mode,
              Type,   Condition, Sofar):-!,
    check_mode(Mode, N_Vars, Args),
    check_type(Type, N_Vars, Args, Old_type, New_type),
%   non_determinate(Mode),
    check_duplicates((Sign, Pred, Args), Literals, N_Vars),
    check_arguments(Args, N_Vars, NN_Vars),
    check_illegal_recursion(Args, Pred),
    avoid_permutation_with_recursion((Sign, Pred, Args)),
    check_condition(Args, Condition),
    check_sofar((Sign, Pred, Args), Sofar).
    % Fail if it has been already known that Literal give no positive examples.

/******************************************************************************/
/*                                                                            */
/*  call          : check_mode(+Mode, +N_vars, +Args)                         */
/*                                                                            */
/*  arguments     : Mode   = a mode list of the literal checked.              */
/*                  N_vars = the number of variables before this literal.     */
/*                  Args   = an arugument list of the literal checked.        */
/*                                                                            */
/******************************************************************************/
/* It checks consistency of arguments of literal with the mode of the literal.*/
/******************************************************************************/
check_mode(Mode, N_vars, Args):-
    !,check_mode1(N_vars, Mode, Args).
check_mode1(_, [], []):-!.
check_mode1(N_vars, [F_mode|R_mode], [F_arg|R_args]):-
    (F_arg > N_vars -> 
        (F_mode = '-' ; F_mode = '=') ; true), !,
    check_mode1(N_vars, R_mode, R_args).

/******************************************************************************/
/*                                                                            */
/*  call          : check_type(+Type, +N_vars, +Args, +Old_type, -New_type)   */
/*                                                                            */
/*  arguments     : Type     = a type list of a predicate constituting the    */
/*                             literal checked.                               */
/*                  N_vars   = the number of variables before this literal.   */
/*                  Args     = an argument list of this literal.              */
/*                  Old_type = a type list of clause before adding this lit.  */
/*                  New_type = a type list of clause after adding this lit.   */
/*                                                                            */
/******************************************************************************/
/* It checks consistency of type of the literal with the type of clause,      */
/* and generates a new type list of clause.                                   */
/******************************************************************************/
check_type(Type, N_vars, Args, Old_type, New_type):-!,
    make_list_nolimit(Old_type, New_type1),
    check_type1(N_vars, Args, Type, Old_type, New_type1),
    make_list_limit(New_type1, New_type).
check_type1(_, [], [], _, _):-!.
check_type1(N_vars, [F_arg|R_args], [F_type|R_type], Old_type, New_type):-!,
    ( F_arg > N_vars -> 
        true ; nthnew(F_arg, F_type, Old_type) ),
    nthnew(F_arg, F_type, New_type),
    check_type1(N_vars, R_args, R_type, Old_type, New_type).

/******************************************************************************/
/*                                                                            */
/*  call          : check_duplicates(+Literal, +Literals, +N_Vars)            */
/*                                                                            */
/*  arguments     : Literal  = a literal to be checked.                       */
/*                  Literals = literals constituting the clause.              */
/*                  N_vars   = the number of variables before this literal.   */
/*                                                                            */
/*  subpredicates : q_same_literal = to check that a literal is the same as   */
/*                                   another one essentially.                 */
/*                  homomorphic    = to check that there is a homomorphism    */
/*                                   from a argument list to another one.     */
/*        make_a_partial_arguments = to make a uninstanciated argument list   */
/*                                   to chech the homomorhism.                */
/*                                                                            */
/******************************************************************************/
/* It checks the literal not to be a duplication of another literal in the    */
/* clause.                                                                    */
/******************************************************************************/
check_duplicates(_Literal, [], _N_Vars):-!.
check_duplicates(Literal, [L|_Literals], N_Vars):-
    q_same_literal(L, Literal, N_Vars), !, false.
check_duplicates(Literal, [_L|Literals], N_Vars):-
    !, check_duplicates(Literal, Literals, N_Vars).

q_same_literal((Sign1, Pred1, Args1), (Sign2, Pred2, Args2), N_Vars):-
    (((Sign1==p ; Sign1==d), (Sign2==p ; Sign2==d))
    ; (Sign1==n, Sign2==n)),
    Pred1 == Pred2,!,
    homomorphic(Args1, Args2, N_Vars).

homomorphic(Args1, Args2, N_Vars):-!,
    make_a_partial_arguments(Args2, Args1, PArgs1, N_Vars, []),!,
    Args2 = PArgs1.

make_a_partial_arguments([], [], [], _N_Vars, _):-!.
make_a_partial_arguments([A2|Args2], [A1|Args1], [P1|PArgs1], N_Vars, Pairs):-
    A2>N_Vars, !,
    ((member((A1, P1), Pairs),
      make_a_partial_arguments(Args2, Args1, PArgs1, N_Vars, Pairs)) ;
     (make_a_partial_arguments(Args2, Args1, PArgs1, N_Vars, [(A1,P1)|Pairs]))).
make_a_partial_arguments([_|Args2], [A1|Args1], [A1|PArgs1], N_Vars, Pairs):-
    !,make_a_partial_arguments(Args2, Args1, PArgs1, N_Vars, Pairs).

    
/******************************************************************************/
/*                                                                            */
/*  call          : check_arguments(+Args, +N_Vars, -NN_Vars)                 */
/*                                                                            */
/*  arguments     : Args    = an argument list of the literal checked.        */
/*                  N_vars  = the number of variables before this literal.    */
/*                  NN_Vars = the number of variables after this literal.     */
/*                                                                            */
/******************************************************************************/
/* It checks the argument list of literal to be canonical.                    */
/******************************************************************************/
check_arguments(Args, N_Vars, NN_Vars):-
    !,
    min_list(Args, Min_arg), Min_arg =< N_Vars, %  One of args is appeared
                                                %  already.
    max_list(Args, Max_arg),                    %
    make_list_from_n(Max_arg, M_List),          %  M_List = 1..Max_arg
    make_list_from_n(N_Vars,  V_List),          %  V_List = 1..N_Vars
/* remove duplicate permutation of extra variables */
    difference(Args, V_List, Ex_vars),          %  Extra var's = Args - V_List
    sort(Ex_vars, Ex_vars),                     %  Extra var's should be sorted.
/* remove skipping extra variables */
    difference(M_List, V_List, Temp),           %  "(M_List-V_List)-Args=[]"
    difference(Temp, Args, []),!,               %  means no skip extra var's.
    NN_Vars is max(Max_arg, N_Vars).

/******************************************************************************/
/*                                                                            */
/*  call          : check_illegal_recursion(+Args, +Pred)                     */
/*                                                                            */
/*  arguments     : Args = an argument list of the literal checked.           */
/*                  Pred = a predicate name of the literal.                   */
/*                                                                            */
/******************************************************************************/
/* It checks a recursive literal to be a normal call. That is, variables with */
/* input mode should also appeare in the head literal with the same order.    */
/******************************************************************************/
check_illegal_recursion(Args, Pred):-!,
    (clause(target(Pred, _, Mode, _),_) -> (
	check_illegal_recursion1(1, Args, Mode)
     );true).
check_illegal_recursion1(_, _, []):-!,false.
check_illegal_recursion1(Count, [Count|Args], [+|Mode]):-!,
    Count1 is Count+1,
    check_illegal_recursion1(Count1, Args, Mode).
check_illegal_recursion1(Count, [_|Args], [-|Mode]):-!,
    Count1 is Count+1,
    check_illegal_recursion1(Count1, Args, Mode).
check_illegal_recursion1(_Count, [_Not_eq_count|_Args], [+|_Mode]):-!,true.

/******************************************************************************/
/*                                                                            */
/*  call          : avoid_permutation_with_recursion(+Literal)                */
/*                                                                            */
/*  arguments     : Literal = the literal checked.                            */
/*                                                                            */
/******************************************************************************/
/* It  checks  not  to  be  a recusive literal  with an argument list of      */
/* permutation of  a head arguments.                                          */
/******************************************************************************/
avoid_permutation_with_recursion((_, Pred, Args)):-!,
    (target(Pred, Arity,_,_) -> (
	 make_list_from_n(Arity, To_arity),
	 difference(Args, To_arity, [_|_])
     ); true).

/******************************************************************************/
/*                                                                            */
/*  call          : check_condition(+Args, +Condition)                        */
/*                                                                            */
/*  arguments     : Args      = an argument list.                             */
/*                  Condition = a list of conditions.                         */
/*                                                                            */
/*  the following conditions are possible:                                    */
/*      proper_sorted : to request that argument ID should be sorted without  */
/*                      duplications.                                         */
/*      apart         : to request that argument ID should not be duplicated. */
/*                                                                            */
/******************************************************************************/
/* It checks coditions of literals.                                           */
/******************************************************************************/
check_condition(_Args, []):-!.
check_condition(Args, [Condition|Rest]):-!,
    C=..[Condition, Args],
    call(C),
    check_condition(Args, Rest).

/******************************************************************************/
/*                                                                            */
/*  call          : check_sofar(+Literal, +Sofar)                             */
/*                                                                            */
/*  arguments     : Literal = a literal                                       */
/*                  Sofar   = a sorfar set                                    */
/*                                                                            */
/******************************************************************************/
/* It checks the literal not to be included in the sofar set.                 */
/******************************************************************************/
check_sofar(Literal, Sofar):-!,
    (member(Literal, Sofar) -> fail ; true).
