/* Copyright (C) 1997 Itoh Hidenori */
/******************************************************************************/
/*                                                                            */
/*  F O I L - I (First Order Inductive Learner from Incomplete samples)       */
/*                                                       /SICStus Prolog 2.1  */
/*                                                                            */
/*  V. 1.1     by Nobuhiro Inuzuka (inuzuka@ics.nitech.ac.jp)                 */
/*                ITOH's Hidenoriratory                                           */
/*                Department of Intelligence and Computer Science             */
/*                Nagoya Institute of Technology                              */
/*                                                                            */
/*             August 1996                                                    */
/*                                                                            */
/******************************************************************************/
/*                                                                            */
/* 0.1 April 1996 the first implementation                                    */
/*                                                                            */
/* 0.2 June       keeps a literal which does not cover any positive examples  */
/*                to avid calculation for the literal again.                  */
/*                                                                            */
/* 0.3 June       adds determinate literals to hypothesis at first, and tries */
/*                to merge fine candidates which are independent each other.  */
/*                                                                            */
/* 0.4 June       deletes nonuse determinate literals from results.           */
/*                                                                            */
/* 0.5 July       avoids candidate literals that have a target predicate and  */
/*                whose arguments are just permutation of the arguments of a  */
/*                head literal.checks satisfaction of examples with recursive */
/*                clauses using both two methods, calculation of recursion    */
/*                and memberships in given target relation.                   */
/*                merges clauses also in case that negative examples are      */
/*                reduced by the merge.                                       */
/*                                                                            */
/* 0.6 July       differential lists are used to keeps tuples(Known) that are */
/*                known to satisfy a target relation.                         */
/*                the merge routine asserts clauses instead to keep them.     */
/*                                                                            */
/* 0.7 July       Known tuples are asserted. Candidate literals are asserted  */
/*                as well. Several points are refined to make speed up.       */
/*                Negative literals are available.                            */
/*                                                                            */
/* 0.8 July       Main routines are refined.                                  */
/*                                                                            */
/* 1.0 October    Files are devided and documented.                           */
/*                                                                            */
/******************************************************************************/
/*                                                                            */
/* Reference                                                                  */
/*   Nobuhiro Inuzuka, Masakage Kamo, Naohiro Ishii,                          */
/*   Hirohisa Seki and Hidenori Itoh                                          */
/*   "Top-down Induction of Logic Programs from Incomplete Samples"           */
/*   Proc. ILP96                                                              */
/*                                                                            */
/******************************************************************************/

/******************************************************************************/
/* ===============                                                            */
/* Data Structures                                                            */
/* ===============                                                            */
/*                                                                            */
/* partial clauses                                                            */
/*                                                                            */
/*   A partial clause is a clause under construction by Foil-i process.       */
/*   A partial clause is kept by the following syntax:                        */
/*     EV-(Sample, Arity, Type, Literals, Covered, Sofar),                    */
/*   where EV is an evaluation value of this clause,                          */
/*         Sample is a pair (P,N) of sets of positive and negative examples,  */
/*         Arity is the number of variables occured in the clause,            */
/*         Type is a list of corresponding types to variables in the clause,  */
/*         Literals is a list of body literals,                               */
/*         Covered is a set of covered positive examples by the clause        */
/*         and Sofar set is a sofar set of the clause.                        */
/*   In a Foil-i process a lits of partial clauses is maintained by a clause  */
/*   with the following syntax:                                               */
/*     intermediate_clauses([partial_clause1, partial_clause2, ... ]).        */
/*                                                                            */
/* examples                                                                   */
/* type                                                                       */
/* literal                                                                    */
/*                                                                            */
/* sofar set                                                                  */
/*                                                                            */
/*   Sofar is a set of literals that were found not to cover any positive     */
/*   examples during a search for a partial clause. Literals in the set are   */
/*   not necessary to be checked again in the search for literals to be       */
/*   added to the same partial clause. Note that a sofar set is valid only    */
/*   during a search process and a search for another partial clause has      */
/*   another sofar set.                                                       */
/*                                                                            */
/******************************************************************************/

% Global informations (saved in databese)
%
%   uncovered_plus(_)
%   covered_sample([])),
%   intermediate_clauses([EV-(Sample, Arity, Type, Literals,
%                             Covered, Literals_sofar)|_]),
%   this_clause(_),
%   this_call(_),
%   this_depth(_),
%   result(_),
%   hojo_tuples(Pred, Arity, Mode, Type, Hojo_tuples, Condition),

:-module(foili, [foili/1]).

:-use_module(library(lists)).
:-use_module(library(charsio)).

:-compile('declarations.pl').

:-compile('pre_and_post.pl').
:-compile('determinate.pl').
:-compile('evaluate.pl').
:-compile('generate_candidates.pl').
:-compile('merge.pl').
:-compile('redundancy.pl').
:-compile('hojo.pl').
:-compile('construct_clauses.pl').
:-compile('utilities.pl').
:-compile('misc.pl').
:-compile('builtin.pl').

/******************************************************************************/
/*                                                                            */
/*  call       : foili(+Sample_File_Name)                                     */
/*                                                                            */
/*  arguments  : Sample_File_Name = a name of sample file                     */
/*                                                                            */
/******************************************************************************/
/* The Main predicate (Starting Point).                                       */
/* It read a sample file and starts learning                                  */
/******************************************************************************/
foili(Sample_file):-!,
    runtime(foili_learning(Sample_file)).

foili_learning(Sample_file):-
    print_title,
    format("file: ~p processing~n", [Sample_file]),
    initialize(Sample_file),!,
    learn.

/******************************************************************************/
/*                                                                            */
/*  call       : learn                                                        */
/*                                                                            */
/******************************************************************************/
/* Learns logical definitions. It loops until a complete and sound            */
/* definition are found or learning process falis.                            */
/******************************************************************************/
learn:-
    get_sample(Initial_sample),
    initialize_in_learning(Initial_sample),!,
    (find_clause(Remain) ->
     (Remain == [] -> (
	  show_result
      ) ; learn) ; fail).
/******************************************************************************/
/*                                                                            */
/*  call       : find_clause(-Remain)                                         */
/*                                                                            */
/*  arguments  : Remain = a list of remaining positive examples not to be     */
/*                        covered by a clause found in the process of this    */
/*                        call.                                               */
/*                                                                            */
/******************************************************************************/
/* Finds a sound clause and returns a set of remaining positive examples      */
/* that are not covered by the clause. Fails if any clause can not be found.  */
/* First search_clause is called to find a clause and coverage positive       */
/* examples not to be covered by the found clause are checked with other      */
/* clauses.                                                                   */
/******************************************************************************/
find_clause(Remain):-
    search_clause(Uncovered, Found),!,
    store_and_show_result(Found),
    check_to_be_solved(Uncovered, Remain),
    save_uncovered(Remain),
    inform_uncovered(Remain).
find_clause(_):-!,fail.

/******************************************************************************/
/*                                                                            */
/*  call       : search_clause(-Uncovered, -Consistent_clauses)               */
/*                                                                            */
/*  arguments  : Uncovered          = a list of uncovered positive examples   */
/*                                    by Consistent_clauses.                  */
/*               Consistent_clauses = A found clause in the process of this   */
/*                                    call.                                   */
/*                                                                            */
/******************************************************************************/
/* Searches a clause that coveres some positive examples remaining to be      */
/* covered and no negative examples. First it add all determinate literals    */
/* to the best-evaliated clauses.  Second generates all possible candidate    */
/* literals, and third it evaluate them.  If some literals make the clause    */
/* consistent with negative examples the clause is output with the literal.   */
/* Otherwise the best-evaluated 10 literals are stored and search process     */
/* are continued. During the search this process tries to merge clauses to    */
/* get consistent clauses.                                                    */
/******************************************************************************/
search_clause(Uncovered, Consistent_clauses):-
    get_the_best_clause(Best),
    add_determinate_literals(Best, Best_ad),
    Best_ad=(Sample,Arity,_,_,_,Sofar),
    generate_candidate_literals(Best_ad, Sofar, Old_N_Vars),
    eval_candidates(Sample, Useful_clauses, Sofar, New_Sofar, Old_N_Vars),
    pick_found_clauses(Useful_clauses, Found_clauses1),
    (Found_clauses1 = [_|_] -> (
	 process_found_clauses(Found_clauses1, Consistent_clauses1, Uncovered1)
     ) ; (
	 Consistent_clauses1=[],
	 uncovered(Uncovered1)
     )),
    (Uncovered1 = [_|_] -> (
         try_merge_clauses(Useful_clauses, Merged, Arity, Sample),
	 pick_found_clauses(Merged, Found_clauses2),
	 append(Merged, Useful_clauses, Intermediate_clauses),
	 append(Consistent_clauses1, Found_clauses2, Found_clauses3),
	 process_found_clauses(Found_clauses3, Consistent_clauses2, Uncovered2)
     ) ; (
	 Intermediate_clauses=Useful_clauses,
	 Consistent_clauses2=Consistent_clauses1,
	 Uncovered2=Uncovered1
     )),
    add_clause(Intermediate_clauses),
    save_uncovered(Uncovered2),
    process_initial_sofar(Best, New_Sofar),!,
    (Consistent_clauses2 = [] -> 
     search_clause(Uncovered, Consistent_clauses) ;
    (Consistent_clauses=Consistent_clauses2, Uncovered=Uncovered2)).
search_clause(_,_):-
    format("~nNo (more) clause found.  Learning failed!~n", []),!,fail.

:-format("~n",[]).
:-format("F O I L - I ver.1,  August 1996~n",[]).
:-format("by N.Inuzuka(inuzuka@ics.nitech.ac.jp)~n",[]).
:-format("Dept. Intelligence and Computer Sience~n",[]).
:-format("Nagoya Institute of Technology~n~n",[]).
