/* Copyright (C) 1997 Itoh Hidenori */
/******************************************************************************/
/*                                                                            */
/*  Predicates to evaluate clauses and to expand samples                      */
/*                                                                            */
/******************************************************************************/

/******************************************************************************/
/*                                                                            */
/*  call          : eval_candidates(+Sample,    -Useful_clauses, +Sofar,      */
/*                                  -New_sofar, +Old_N_Vars)                  */
/*                                                                            */
/*  arguments     : Sample         = sample of this learning loop.            */
/*                  Useful_clauses = clauses eveluated to be useful.          */
/*                  Sofar          = old sofar set.                           */
/*                  New_sofar      = new sofar set.                           */
/*                  Old_N_Vars     = the number of variables before this      */
/*                                   learning loop.                           */
/*                                                                            */
/*  subpredicates : eval_candidate, eval_candidate1                           */
/*                                                                            */
/******************************************************************************/
/* It evaluates clauses, which are kept in the databese with                  */
/* 'candidate_literal', to gather useful clauses.                             */
/******************************************************************************/
eval_candidates(Sample, Useful_clauses, Sofar, New_sofar, Old_N_Vars):-
    abolish(sofar_eval_candidate/1),
    abolish(useful_eval_candidate/1),
    assert(sofar_eval_candidate(Sofar)),
    assert(useful_eval_candidate(Sofar)),
    eval_candidate(Sample, Old_N_Vars),
    abolish(candidate_literal/1),
    retract(sofar_eval_candidate(New_sofar)),
    retract(useful_eval_candidate(Useful_clauses)).

eval_candidate(Sample, Old_N_Vars):-
    clause(candidate_literal(Candidate),_),
    abolish(foili_known_here/1),
    eval_candidate1(Sample, Candidate, Old_N_Vars),
    fail.

eval_candidate(_,_):-!,true.

eval_candidate1(Sample, Candidate, Old_N_Vars):-!,
    clause(sofar_eval_candidate(Sofar),_),
    Candidate=(Sample_new,_,_,_,Covered,_),
    eval_and_process_a_candidate(Candidate, Sample, Sample_new, Covered,
				 Ev, Additional_sofar, Old_N_Vars),
    (Additional_sofar=[] -> true ; 
     (abolish(sofar_eval_candidate/1),
      assert(sofar_eval_candidate([Additional_sofar|Sofar])))),
    (Sample_new=([], _)  -> true ;
     (clause(useful_eval_candidate(Useful),_),
      abolish(useful_eval_candidate/1),
      assert(useful_eval_candidate([Ev-Candidate|Useful])))),!.


/******************************************************************************/
/*                                                                            */
/*  call       : eval_and_process_a_candidate(+Candidate,                     */
/*                  +Sample,     -Sample_new, -Covered,                       */
/*                  -Evaluation, -Ad_sofar,   +Old_N_Vars):-                  */
/*                                                                            */
/*  arguments  : Candidate  = a clause including a literal to be evaluated.   */
/*               Sample     = old sample.                                     */
/*               Sample_new = new sample.                                     */
/*               Covered    = set of plus example covered by this clause.     */
/*               Evaluation = an evaluation value of this clause.             */
/*               Ad_sofar   = additional sofar set which are added            */
/*                            in this evaluation.                             */
/*               Old_N_Vars = the number of variables before this evaluation. */
/*                                                                            */
/******************************************************************************/
/* It expands a sample by a clause and convey an evaluate value.              */
/******************************************************************************/
eval_and_process_a_candidate(Candidate, Sample, Sample_new,
			   Covered, Evaluation, Ad_sofar, Old_N_Vars):-
    literal_count_inc,
    Candidate = (_,_,_,[Literal|Literals],_,_),!,
    expand_sample(Literals, Literal, Sample, Sample_new,
		  Ad_sofar, Old_N_Vars,
		  Covered, Number_of_examples_expanded),
    Sample_new=(NPlus, NMinus),
    ( NPlus = [_|_] -> (
	  length(NPlus, NPL),length(NMinus, NML),
	  calculate_evaluation(Sample, Sample_new, Candidate,
			       Number_of_examples_expanded, Evaluation),
	  format("(~p) coveres ~d pos.and ~d neg.example(s).  Evaluation is ~3f~n",
		 [Literal, NPL, NML, Evaluation])
      );(
	  Evaluation is -99999
      )
    ).

/******************************************************************************/
/*                                                                            */
/*  call       : calculate_evaluation(+Sample_old, +Sample_new,               */
/*                                    +Candidate,  +N,  -Evaluation)          */
/*                                                                            */
/*  arguments  : Sample_old = old sample.                                     */
/*               Sample_new = new sample.                                     */
/*               Candidate  = a clause including a literal to be evaluated.   */
/*               N          = the number of examples covered by the clause.   */
/*               Evaluation = the evaluation value.                           */
/*                                                                            */
/******************************************************************************/
/* It calculates an evaluation value of a clause.                             */
/******************************************************************************/
calculate_evaluation(Sample_old, Sample_new, Candidate, N, Evaluation):-
    Candidate = (_,NN_vars,_,Literals,_,_),!,
    information_value(Sample_old, Value1),
    information_value(Sample_new, Value2),
    Evaluation1 is N * (Value1 - Value2),
    length(Literals, Leng),
    Complexity is log(2, Leng+1)+log(2, NN_vars+1),
    Evaluation is Evaluation1 - Complexity.

/******************************************************************************/
/*                                                                            */
/*  call       : information_value(+Sample, -Value)                           */
/*                                                                            */
/*  arguments  : Sample = a sample.                                           */
/*               Value  = information-theoritic value of the sample.          */
/*                                                                            */
/******************************************************************************/
/* It calculates an information-theoretic value of the sample.                */
/******************************************************************************/
information_value((Plus, Minus), Value):-!,
    length(Plus,  NP),
    length(Minus, NM),
     X is NP/(NP+NM),
    (X > 0
     -> Value is -log(2, NP/(NP+NM)) ; Value is -999999).

/******************************************************************************/
/*                                                                            */
/*  call          : expand_sample(+Literals, +Literal, +Sample, -Sample_new,  */
/*                                -Ad_sofar, +Old_N_Vars, -Covered,           */
/*                                -Number_of_expanded):-                      */
/*                                                                            */
/*  arguments     : Literals           = literals constituting a clause       */
/*                                       before this learning loop.           */
/*                  Literal            = a literal to be added to the clause  */
/*                                       and by which sample is expanded.     */
/*                  Sample             = old sample.                          */
/*                  Sample_new         = new sample expanded by the literal.  */
/*                  Ad_sofar           = an additional sofar set.             */
/*                  Old_N_Vars         = the number of variables before this  */
/*                                       learning loop.                       */
/*                  Covered            = positive examples covered by the lit.*/
/*                  Number_of_expanded = the number of pos.examples covered.  */
/*                                                                            */
/*  subpredicates : make_add_sofar           = to make additional sofar.      */
/*                  make_anonymouns_new_vars = to make an argument list       */
/*                                             anonymous.                     */
/*                                                                            */
/******************************************************************************/
/* It expands sample by a literal.                                            */
/******************************************************************************/
expand_sample(Literals, Literal, (Plus, Minus), (Plus_new, Minus_new),
	      Ad_sofar, Old_N_Vars, Covered, Number_of_expanded):-
    (Literal =(p, Pred, _Args) ; Literal =(d, Pred, _Args)),
    target(Target,_,_,_),
    member((_, Target, _), [Literal|Literals]),!,
    reverse([Literal|Literals], LiteralsR),
    construct_a_clause(LiteralsR, This_clause_s),
    abolish(this_clause/1),
    read_from_chars(This_clause_s, This_clause),
    assert(this_clause(This_clause)),
    get_all_solutions(This_clause, LiteralsR, Plus, Plus_new,
		      Number_of_expanded),
    (Plus_new=[] -> (
	 Minus_new = [],
	 Covered=[],
	 make_add_sofar(Literal, Old_N_Vars, Ad_sofar)
     );(
	 get_all_solutions(This_clause, LiteralsR, Minus, Minus_new, _),
	 calculate_covered(Plus_new, Covered),
	 Ad_sofar = []
    )).

expand_sample(_Literals, Literal, (Plus, Minus), (Plus_new, Minus_new),
	      Ad_sofar, Old_N_Vars, Covered, Number_of_expanded):-
    (Literal =(p, Pred, Args) ; Literal =(d, Pred, Args)),
    hojo(Pred,_,_,_,Tuples,_),!,
    int_tuples(Plus,  Args, Tuples, Plus_new, Number_of_expanded),
    (Plus_new=[] -> (
	 Minus_new = [],
	 Covered=[],
	 make_add_sofar(Literal, Old_N_Vars, Ad_sofar)
     );(
	 int_tuples(Minus, Args, Tuples, Minus_new, _),
	 calculate_covered(Plus_new, Covered),
	 Ad_sofar = []
    )).

expand_sample(_Literals, Literal, (Plus, Minus), (Plus_new, Minus_new),
	      Ad_sofar, Old_N_Vars, Covered, Number_of_expanded):-
    Literal =(n, Pred, Args),
    hojo(Pred,_,_,_,Tuples,_),!,
    diff_tuples(Plus, Args, Tuples, Plus_new, Number_of_expanded),
    (Plus_new=[] -> (
	 Minus_new = [],
	 Covered=[],
	 make_add_sofar(Literal, Old_N_Vars, Ad_sofar)
     );(
	 diff_tuples(Minus, Args, Tuples, Minus_new, _),
	 calculate_covered(Plus_new, Covered),
	 Ad_sofar = []
    )).

make_add_sofar((S,P,Args), Old_N_Vars, (S,P,A_Args)):-
    !,make_anonymouns_new_vars(Args, A_Args, Old_N_Vars).

make_anonymouns_new_vars([], [], _Old_N_Vars):-!.
make_anonymouns_new_vars([Arg|Args], [_|A_Args], Old_N_Vars):-
    Arg > Old_N_Vars,!,
    make_anonymouns_new_vars(Args, A_Args, Old_N_Vars).
make_anonymouns_new_vars([Arg|Args], [Arg|A_Args], Old_N_Vars):-!,
    make_anonymouns_new_vars(Args, A_Args, Old_N_Vars).

/******************************************************************************/
/*                                                                            */
/*  call       : get_all_solutions(+This_clause, +Literals,                   */
/*                                 +Exams, -Passed, <---Pay attention!!       */
/*                                 -N)                  Read the following.   */
/*                                                                            */
/*  arguments  : This_clause = a clause to be used to process examples.       */
/*               Literals    = literals constituting the clause.              */
/*               Exams       = all exmaples with which the clause is          */
/*                             processed.                                     */
/*               Passed      = examples with which the clause is succeeded.   */
/*                             an example(a tuple) is expressed as a list     */
/*                             with uninstantiated tail, and a part of the    */
/*                             tail of Passed may be instantiated with a      */
/*                             value that let the clause succeed.             */
/*               N           = the number of examples that are succeeded.     */
/*                                                                            */
/******************************************************************************/
/* It tries to solve clauses with the examples. A variable binding for each   */
/* solution is kept in the Passed.                                            */
/******************************************************************************/
get_all_solutions(_,_, [], [], 0):-!.
get_all_solutions(This_clause, Literals, [Exam|Exams], Passed, N):-
    get_all_solutions(This_clause, Literals, Exams, Passed1, N1),
    copy_term(This_clause, This_clause_copy),
    copy_term(Exam, Test),
    maps_from_clause_to_tuple(Literals, This_clause_copy, Exam),
    This_clause_copy=(_:-Body),
    abolish(this_call/1),
    abolish(this_depth/1),
    assert(this_call([dumy]),_),
    assert(this_depth(0),_),
    (check_goal(Body, _Use_thisclause) -> (
	 Passed = [Test|Passed1],
	 N is N1+1
     );(
	 Passed=Passed1,
	 N = N1
    )).

/******************************************************************************/
/*                                                                            */
/*  call          : maps_from_clause_to_tuple(+Literals, +(-)Clause, +Tuple)  */
/*                                                                            */
/*  arguments     : Literals = literals constituting the clause               */
/*                  Clause   = a clause, variables in this clause are bound   */
/*                             with values in the tuple during process of     */
/*                             this predicate.                                */
/*                  Tuple    = a tuple                                        */
/*                                                                            */
/*  subpredicates : matching_to_tuple1 = matching of Head variables.          */
/*                  matching_to_tuple2 = matching of Body variables.          */
/*                  matching_to_tuple3 = matching of Body variables, too.     */
/*                                                                            */
/******************************************************************************/
/* It makes matching of variables in the clause with values in the tuple.     */
/******************************************************************************/
maps_from_clause_to_tuple(Literals, (Head:-Body), Tuple):-
    Head=..[_|Args],!,
    matching_to_tuple1(Args, Tuple),
    matching_to_tuple2(Body, Literals, Tuple).

matching_to_tuple1([],_):-!.
matching_to_tuple1([X|Args], [X|Tuple]):-!,
    matching_to_tuple1(Args, Tuple).

matching_to_tuple2(Body, [(_,_,NumArgs)|Literals], Tuple):-
    Body=..[',', First, Rest],!,
    First=..[_|RealArgs],
    matching_to_tuple3(RealArgs, NumArgs, Tuple),
    matching_to_tuple2(Rest, Literals, Tuple).
matching_to_tuple2(Body, [(_,_,NumArgs)], Tuple):-!,
    Body=..[_|RealArgs],
    matching_to_tuple3(RealArgs, NumArgs, Tuple).

matching_to_tuple3([], [], _Tuple):-!.
matching_to_tuple3([Real|RealArgs], [Num|NumArgs], Tuple):-!,
    (nthnew(Num, Real, Tuple) ; true),
    matching_to_tuple3(RealArgs, NumArgs, Tuple).
    
/******************************************************************************/
/*                                                                            */
/*  call          : check_goal(+Literal(s), +Use_thisclause)                  */
/*                                                                            */
/*  arguments     : Literal(s)     = (a) literal(s) to be checked.            */
/*                  Use_thisclause = a clause which is now under constituting.*/
/*                                                                            */
/*  subpredicates : check_goal/4                                              */
/*                                                                            */
/******************************************************************************/
/* It checks if the literal(s) are true using the Use_thisclause.             */
/******************************************************************************/
check_goal(Body, Use_thisclause):-
    Body=..[',', First, Rest],!,
    check_goal(First, Use_thisclause),!,
    check_goal(Rest, Use_thisclause).

check_goal(Atom, Use_thisclause):-
    Atom=..[Pred|Args],!,
    check_goal(Atom, Pred, Args, Use_thisclause).

check_goal(_Atom, Pred, Args, _Use_thisclause):-
    hojo(Pred,_,_,_,Tuples,_), !,
    member(Args, Tuples).

check_goal(Atom, _Pred, _Args, Use_thisclause):-!,
    retract(this_depth(This_depth)),
    ((
	 This_depth>=0,
	 Atom=..[_|Args],
	 ((clause(foili_known(Args),_)) ;
	  (clause(foili_known_here(Args),_),
	   Use_thisclause=yes
	  ))
     );(
	 retract(this_call(This_call)),
	 This_call \== Atom,
	 This_depth =< 10,
	 Next_depth is This_depth + 1,
	 assert(this_depth(Next_depth)),
	 assert(this_call(Atom)),
	 ((
	      clause(result((Atom:-Body)), _),
	      check_goal(Body, Use_thisclause),
	      Atom=..[_|Args],
	      (Use_thisclause==yes ->
	       assert(foili_known_here(Args)) ;
	       assert(foili_known(Args)))
	  );(
	      clause(this_clause((Atom:-Body)), _),
	      check_goal(Body, Use_thisclause),
	      Atom=..[_|Args],
	      Use_thisclause=yes,
	      assert(foili_known_here(Args))
	  )))
    ).

/******************************************************************************/
/*                                                                            */
/*  call          : make_a_goal(+(Sg, Pred, Args), +Exam, -Goal)              */
/*                                                                            */
/*  arguments     : (Sg, Pred, Args) = a literal in internal syntax.          */
/*                  Exam             = an example.                            */
/*                  Goal             = the literal instanciated with the      */
/*                                      value of the example in prolog syntax.*/
/*                                                                            */
/*  subpredicates : pick_arguments(Args, Exam, Real, Temp)                    */
/*                                                                            */
/******************************************************************************/
/* It constitutes ain instanciated literal in prolog syntax from a literal in */
/* internal syntax and an example.                                            */
/******************************************************************************/
make_a_goal((Sg, Pred, Args), Exam, Goal):-!,
    (Sg == p ; Sg == d),
    (target(Pred, Arity,_,_);hojo(Pred, Arity,_,_,_,_)),
    length(Real_arguments, Arity),
    max_list(Args, Max),
    length(Temp, Max),
    pick_arguments(Args, Exam, Real_arguments, Temp),
    Goal=..[Pred|Real_arguments].

pick_arguments([],_,_,_):-!.
pick_arguments([A|Args], Exam, [R|Reals],  Temp):-!,
    pick_arguments(Args, Exam, Reals, Temp),
    ( nthnew(A, E, Exam) -> R=E ; nthnew(A, R, Temp)).


/******************************************************************************/
/*                                                                            */
/*  call          : int_tuples(+OldEx, +Args, +Tuples, -NewEx, -N)            */
/*                                                                            */
/*  arguments     : OldEx  = a set of examples before expanded by the literal.*/
/*                  Args   = an argument list of the literal                  */
/*                  Tuples = an extensional definition of the predicate       */
/*                           in the literal.                                  */
/*                  NewEx  = a set of examples after expanded by the literal. */
/*                  N      = the number of old examples expanded.             */
/*                                                                            */
/*  subpredicates : all_from_tuples = to expand an example with tuples.       */
/*                  expand_a_datum  = to expand an example to have arguments  */
/*                                    specified.                              */
/*                  take_unifiable_and_rest                                   */
/*                                  = get all posible expansion of an example */
/*                                    with tuples.                            */
/*                                                                            */
/******************************************************************************/
/* It expands examples using a literal(only its argument list and its         */
/* extensional definition are given).                                         */
/******************************************************************************/
int_tuples([], _Args, _Tuples, [], 0).
int_tuples([X|R], Args, Tuples, New, N):-
    int_tuples(R, Args, Tuples, New1, N1),!,
    all_from_tuples(X, Args, Tuples, Result),
    ( Result = [] -> N=N1 ; N is N1 + 1),
    append(Result, New1, New).

all_from_tuples(Datum, Args, Tuples, Results):-
     copy_term(Datum, Result),
     expand_a_datum(Result, Args, Expanded),!,
     (take_unifiable_and_rest(Tuples, Expanded, Rest_Tuples) -> (
 	 Results = [Result|Results1],
 	 all_from_tuples(Datum, Args, Rest_Tuples, Results1)
      ) ; Results=[]
     ).

expand_a_datum(_, [], []):-!.
expand_a_datum(Datum, [A|Args], [E|Expanded]):-!,
    nthnew(A, E, Datum),
    expand_a_datum(Datum, Args, Expanded).

take_unifiable_and_rest([],_,_):-!,fail.
take_unifiable_and_rest([X|R], X, R):-!.
take_unifiable_and_rest([_|R], X, R1):-!,
    take_unifiable_and_rest(R, X, R1).

/******************************************************************************/
/*                                                                            */
/*  call          : diff_tuples(OldEx, Args, Tuples, NewEx, N)                */
/*                                                                            */
/*  arguments     : OldEx  = a set of examples before expanded by the literal.*/
/*                  Args   = an argument list of the literal                  */
/*                  Tuples = an extensional definition of the predicate       */
/*                           in the literal.                                  */
/*                  NewEx  = a set of examples after expanded by the literal. */
/*                  N      = the number of old examples expanded.             */
/*                                                                            */
/*  subpredicate  : not_unifiable_to_tuples                                   */
/*                                                                            */
/******************************************************************************/
/* It calculates defference of set of examples with a literal, that is,       */
/* expands examples by a negated literal.                                     */
/******************************************************************************/
diff_tuples([], _Args, _Tuples, [], 0).
diff_tuples([X|R], Args, Tuples, New, N):-
    diff_tuples(R, Args, Tuples, New1, N1),!,
    (not_unifiable_to_tuples(X, Args, Tuples, Result)
      -> (New = [Result|New1],N is N1+1) ;
         (New = New1, N = N1)).

not_unifiable_to_tuples(Datum, Args, Tuples, Expanded):-
    expand_a_datum(Datum, Args, Expanded),!,
    (take_unifiable_and_rest(Tuples, Expanded, _Rest_Tuples)
      -> fail ; true).

/******************************************************************************/
/*                                                                            */
/*  call      : calc_tuples_for_builtin(+Pred, +Tuples, +Args, +Definition,   */
/*                                      -New_tuples, -N                       */
/*                                                                            */
/*  arguments : Pred       = a name of predicate.                             */
/*              Tuples     = tuples to be checked.                            */
/*              Args       = an augument list.                                */
/*              Definition = a definition of the predicate.                   */
/*              New_tuples = tuples expanded.                                 */
/*              N          = the number of examples expanded.                 */
/*                                                                            */
/******************************************************************************/
/* It expands examples(tuples) using definitions of a built-in predicate.     */
/******************************************************************************/
calc_tuples_for_builtin(_Pred, [], _Args, _Definition, [], 0):-!.
calc_tuples_for_builtin(Pred, [Tuple|Tuples], Args, Definition, New_tuples, N):-
    calc_tuples_for_builtin(Pred, Tuples, Args, Definition, New_tuples1, N1),
    pick_arguments(Tuple, Args, Picked),
    Goal=..[Pred|Picked],
    copy_term(Definition, Copy),
    (Goal:-Body) = Copy,
    (call(Body) -> (
	 New_tuples = [Tuple|New_tuples1],
	 N is N1 + 1
     );(
	 New_tuples = New_tuples1,
	 N = N1
    )).
